#!/usr/bin/env bash
. $builddir/tests/test_common.sh

set -e -o pipefail

function test_normal {
    kickstart=$(mktemp)
    stderr=$(mktemp)
    expected_modified=$(mktemp)
    kickstart_modified=$(mktemp)

    sed "/This file was generated by OpenSCAP .* using:/d" "$srcdir/test_remediation_kickstart_expected.cfg" > "$expected_modified"
    sed -i "/rootpw .*/d" "$expected_modified"
    sed -i "s;TEST_DATA_STREAM_PATH;$srcdir/test_remediation_kickstart.ds.xml;" "$expected_modified"

    $OSCAP xccdf generate fix --fix-type kickstart --output "$kickstart" --profile common "$srcdir/test_remediation_kickstart.ds.xml"

    sed "/This file was generated by OpenSCAP .* using:/d" "$kickstart" > "$kickstart_modified"
    sed -i "/rootpw .*/d" "$kickstart_modified"

    diff -u "$expected_modified" "$kickstart_modified"

    rm -rf "$kickstart"
    rm -rf "$stderr"
    rm -rf "$expected_modified"
    rm -rf "$kickstart_modified"
}

function test_tailoring {
    kickstart=$(mktemp)
    stderr=$(mktemp)

    $OSCAP xccdf generate fix --tailoring-file "$srcdir/test_remediation_kickstart.tailoring.xml" --fix-type kickstart --output "$kickstart" --profile custom "$srcdir/test_remediation_kickstart.ds.xml"

    grep -q 'services --enabled=auditd,rsyslog' "$kickstart"
    ! grep -q 'openssh-server' "$kickstart"
    grep -q 'cat >/root/oscap_tailoring.xml <<END_OF_TAILORING' "$kickstart"
    grep -q 'ns0:Tailoring' "$kickstart"
    grep -q 'oscap xccdf eval --remediate --tailoring-file /root/oscap_tailoring.xml' "$kickstart"

    rm -rf "$kickstart"
    rm -rf "$stderr"
}


function test_results_oriented {
    kickstart=$(mktemp)
    stderr=$(mktemp)

    $OSCAP xccdf generate fix --fix-type kickstart --output "$kickstart" --result-id xccdf_org.open-scap_testresult_xccdf_org.ssgproject.content_profile_ospp "$srcdir/test_remediation_kickstart.ds.xml" 2> "$stderr" || ret=$?

    [ $ret = 1 ]
    grep -q "It isn't possible to generate results-oriented Kickstarts." $stderr

    rm -rf "$kickstart"
    rm -rf "$stderr"
}

function test_error_service {
    kickstart=$(mktemp)
    stderr=$(mktemp)

    $OSCAP xccdf generate fix --fix-type kickstart --output "$kickstart" --profile broken_service "$srcdir/test_remediation_kickstart_invalid.ds.xml" 2> "$stderr"

    grep -q "Unsupported 'service' command keyword 'slow' in command: 'service slow down'" "$stderr"

    rm -rf "$kickstart"
    rm -rf "$stderr"
}

function test_error_package {
    kickstart=$(mktemp)
    stderr=$(mktemp)

    $OSCAP xccdf generate fix --fix-type kickstart --output "$kickstart" --profile broken_package "$srcdir/test_remediation_kickstart_invalid.ds.xml" 2> "$stderr"

    grep -q "Unsupported 'package' command keyword 'build' in command:'package build sources'" "$stderr"

    rm -rf "$kickstart"
    rm -rf "$stderr"
}

function test_error_logvol {
    kickstart=$(mktemp)
    stderr=$(mktemp)

    $OSCAP xccdf generate fix --fix-type kickstart --output "$kickstart" --profile broken_logvol "$srcdir/test_remediation_kickstart_invalid.ds.xml" 2> "$stderr"

    grep -q "Unexpected string 'crypto' in command: 'logvol /var 158 crypto'" "$stderr"

    rm -rf "$kickstart"
    rm -rf "$stderr"
}

function test_error_unknown {
    kickstart=$(mktemp)
    stderr=$(mktemp)

    $OSCAP xccdf generate fix --fix-type kickstart --output "$kickstart" --profile unknown_command "$srcdir/test_remediation_kickstart_invalid.ds.xml" 2> "$stderr"

    grep -q "Unsupported command keyword 'unknown' in command: 'unknown command'" "$stderr"

    rm -rf "$kickstart"
    rm -rf "$stderr"
}

test_normal
test_tailoring
test_results_oriented
test_error_service
test_error_package
test_error_logvol
test_error_unknown
