/*
 * cowbell
 * Copyright (c) 2005 Brad Taylor
 *
 * This file is part of cowbell.
 *
 * cowbell is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * cowbell is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with cowbell; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using System;
using System.Text;
using System.Collections;

namespace Cowbell.Base
{
	public class CliParser
	{
		/* public fields */
		public string AppName;
		public string Description;
		public string Usage;

		/* public properties */
		public string[] this[string key] {
			get {
				if (Keys.ContainsKey (key)) {
					string[] values = ((CliOption)CliOptions[(int)Keys[key]]).Values;
					if (values == null) {
						return new string[0];
					}
					return values;
				}

				return null;
			}
		}

		public string[] Parameters {
			get {
				return (string[])Params.ToArray (typeof (string));
			}
		}

		/* public methods */
		public CliParser (string application_name, string description, string usage)
		{
			AppName = application_name;
			Description = description;
			Usage = usage;

			Options = new Hashtable ();
			Keys = new Hashtable ();
			CliOptions = new ArrayList ();
			Params = new ArrayList ();
		}

		public void Parse (string[] args)
		{
			string arg, str;

			for (int i = 0; i < args.Length; i++)
			{
				arg = args[i];
				if (arg.StartsWith ("--")) {
					str = arg.Substring (2);

					if (!Options.ContainsKey (str)) {
						throw new Exception (String.Format (Catalog.GetString ("{0}: invalid option -- {1}"), AppName, str));
					}

					ArrayList values = new ArrayList ();
					int pos = (int)Options[str];
					CliOption opt = ((CliOption)CliOptions[pos]);

					Keys.Add (str, pos);
					while (args.Length > i + 1 && !args[i + 1].StartsWith ("-"))
					{
						if (!opt.ValueParam) {
							Params.Add (args[i + 1]);
						}
						
						values.Add (args[i + 1]);
						i++;
					}

					opt.Values = (string[])values.ToArray (typeof (string));
				} else if (arg.StartsWith ("-") || arg.StartsWith ("/")) {
					arg = arg.Substring (1);
					for (int j = 0; j < arg.Length; j++)
					{
						str = arg.Substring (j, 1);

						if (!Options.ContainsKey (str)) {
							throw new Exception (String.Format (Catalog.GetString ("{0}: invalid option -- {1}"), AppName, str));
						}

						ArrayList values = new ArrayList ();
						int pos = (int)Options[str];
						CliOption opt = ((CliOption)CliOptions[pos]);

						Keys.Add (str, pos);
						while (args.Length > i + 1 && !args[i + 1].StartsWith ("-"))
						{
							if (!opt.ValueParam) {
								Params.Add (args[i + 1]);
							}
							
							values.Add (args[i + 1]);
							i++;
						}

						opt.Values = (string[])values.ToArray (typeof (string));
					}
				} else {
					Params.Add (arg);
				}
			}
		}

		public void AddOption (string key, string description, bool valueparam)
		{
			AddOption (new string[] { key }, description, valueparam);
		}

		public void AddOption (string[] keys, string description, bool valueparam)
		{
			int pos = CliOptions.Add (new CliOption (keys,
			                                         description,
			                                         valueparam));

			foreach (string key in keys)
			{
				Options.Add (key, pos);
			}
		}

		public string GetHelp ()
		{
			StringBuilder help = new StringBuilder ();
			
			help.AppendFormat (Catalog.GetString ("Usage: {0} {1}\n"), AppName, Usage);
			help.Append (Description + "\n");
			help.Append ("\n");

			int maxlen = 0;
			foreach (CliOption opt in CliOptions)
			{
				maxlen = Math.Max (opt.ToString ().Length, maxlen);
			}

			maxlen += 3;
			foreach (CliOption opt in CliOptions)
			{
				help.AppendFormat (" {0,-" + maxlen + "}{1}\n", opt.ToString (), opt.Description);
			}

			return help.ToString ();
		}

		/* private structs */
		private struct CliOption {
			public string[] Keys;
			public string Description;
			public bool ValueParam;
			public string[] Values;

			public CliOption (string[] k, string d, bool vp)
			{
				Keys = k;
				Description = d;
				ValueParam = vp;
				Values = null;
			}

			public override string ToString ()
			{
				string temp = "";
				for (int i = 0; i < Keys.Length; i++)
				{
					if (Keys[i].Length > 1) {
						temp += "--" + Keys[i];
					} else {
						temp += "-" + Keys[i];
					}

					if (i < Keys.Length - 1) {
						temp += ", ";
					}
				}
				return temp;
			}
		}
		
		/* private fields */
		Hashtable Options;
		Hashtable Keys;
		ArrayList CliOptions;
		ArrayList Params;
	}
}
