//
// immer: immutable data structures for C++
// Copyright (C) 2016, 2017, 2018 Juan Pedro Bolivar Puente
//
// This software is distributed under the Boost Software License, Version 1.0.
// See accompanying file LICENSE or copy at http://boost.org/LICENSE_1_0.txt
//

#include "extra/fuzzer/fuzzer_input.hpp"
#include <array>
#include <catch2/catch_test_macros.hpp>
#include <immer/flex_vector.hpp>
#include <iostream>

#define IMMER_FUZZED_TRACE_ENABLE 0

#if IMMER_FUZZED_TRACE_ENABLE
#define IMMER_FUZZED_TRACE(...) std::cout << __VA_ARGS__ << std::endl;
#else
#define IMMER_FUZZED_TRACE(...)
#endif

namespace {

template <std::size_t VarCount = 2, unsigned Bits = 2>
int run_input(const std::uint8_t* data, std::size_t size)
{
    using vector_t =
        immer::flex_vector<int, immer::default_memory_policy, Bits, Bits>;
    using size_t = std::uint8_t;

    auto vars = std::array<vector_t, VarCount>{};

#if IMMER_FUZZED_TRACE_ENABLE
    std::cout << "/// new test run" << std::endl;
    for (auto i = 0u; i < VarCount; ++i)
        std::cout << "auto var" << i << " = vector_t{};" << std::endl;
#endif

    auto is_valid_var   = [&](auto idx) { return idx >= 0 && idx < VarCount; };
    auto is_valid_index = [](auto& v) {
        return [&](auto idx) { return idx >= 0 && idx < v.size(); };
    };
    auto is_valid_size = [](auto& v) {
        return [&](auto idx) { return idx >= 0 && idx <= v.size(); };
    };
    auto can_concat = [](auto&& v1, auto&& v2) {
        using size_type = decltype(v1.size());
        return v2.size() < (std::numeric_limits<size_type>::max() - v1.size());
    };
    auto can_insert = [](auto&& v1) {
        using size_type = decltype(v1.size());
        return v1.size() < std::numeric_limits<size_type>::max();
    };

    return fuzzer_input{data, size}.run([&](auto& in) {
        enum ops
        {
            op_push_back,
            op_update,
            op_take,
            op_drop,
            op_concat,
            op_push_back_move,
            op_update_move,
        };
        auto src = read<std::uint8_t>(in, is_valid_var);
        auto dst = read<std::uint8_t>(in, is_valid_var);
        switch (read<char>(in)) {
        case op_push_back:
            if (can_insert(vars[src])) {
                IMMER_FUZZED_TRACE("var" << +dst << " = var" << +src
                                         << ".push_back(42);");
                vars[dst] = vars[src].push_back(42);
            }
            break;
        case op_update: {
            auto idx = read<size_t>(in, is_valid_index(vars[src]));
            IMMER_FUZZED_TRACE("var" << +dst << " = var" << +src << ".update("
                                     << +idx
                                     << ", [] (auto x) { return x + 1; });");
            vars[dst] = vars[src].update(idx, [](auto x) { return x + 1; });
            break;
        }
        case op_take: {
            auto idx = read<size_t>(in, is_valid_size(vars[src]));
            IMMER_FUZZED_TRACE("var" << +dst << " = var" << +src << ".take("
                                     << +idx << ");");
            vars[dst] = vars[src].take(idx);
            break;
        }
        case op_drop: {
            auto idx = read<size_t>(in, is_valid_size(vars[src]));
            IMMER_FUZZED_TRACE("var" << +dst << " = var" << +src << ".take("
                                     << +idx << ");");
            vars[dst] = vars[src].drop(idx);
            break;
        }
        case op_concat: {
            auto src2 = read<std::uint8_t>(in, is_valid_var);
            if (can_concat(vars[src], vars[src2])) {
                IMMER_FUZZED_TRACE("var" << +dst << " = var" << +src << " + var"
                                         << +src2 << ";");
                vars[dst] = vars[src] + vars[src2];
            }
            break;
        }
        case op_push_back_move: {
            if (can_insert(vars[src])) {
                IMMER_FUZZED_TRACE("var" << +dst << " = std::move(var" << +src
                                         << ").push_back(21);");
                vars[dst] = std::move(vars[src]).push_back(21);
            }
            break;
        }
        case op_update_move: {
            auto idx = read<size_t>(in, is_valid_index(vars[src]));
            IMMER_FUZZED_TRACE("var" << +dst << " = std::move(var" << +src
                                     << ").update(" << +idx
                                     << ", [] (auto x) { return x + 1; });");
            vars[dst] =
                std::move(vars[src]).update(idx, [](auto x) { return x + 1; });
            break;
        }
        default:
            break;
        };
        return true;
    });
}

} // anonymous namespace

TEST_CASE("bug: memory leak because of move update")
{
    // There was a problem caused with shared "sizes buffer" in
    // relaxed nodes.  In particular, the ensure_mutable_relaxed(...)
    // function was not decremeting the old sizes buffer. That is why
    // the last transient push_back (which uses mutable operations)
    // causes some of the relaxed buffers that are created during the
    // previous concatenations, and that start to be shared from the
    // update() onwards, to later be leaked.
    SECTION("simplified")
    {
        using vector_t =
            immer::flex_vector<int, immer::default_memory_policy, 2, 2>;
        auto var0 = vector_t{};
        auto var1 = vector_t{};
        var0      = var0.push_back(42);
        var0      = var0.push_back(42);
        var0      = var0.push_back(42);
        var0      = var0 + var0;
        var1      = var0.push_back(42);
        var0      = var0 + var1;
        var1      = var0.push_back(42);
        var0      = var0 + var0;
        var0      = var1 + var0;
        var0      = var1.update(5, [](auto x) { return x + 1; });
        var0      = std::move(var0).push_back(21);
    }

#if !IMMER_DISABLE_FUZZER_DUE_TO_GCC_BUG
    SECTION("")
    {
        constexpr std::uint8_t input[] = {
            0xff, 0x0,  0xff, 0x0, 0x0,  0x0,  0x0,  0x0,  0x0,  0x0, 0x0,
            0x40, 0x0,  0x0,  0x4, 0x0,  0x6d, 0x6d, 0x0,  0x1,  0x0, 0x4,
            0x6d, 0x6d, 0x6d, 0x0, 0x0,  0x4,  0x1,  0x6d, 0x6d, 0x0, 0x1,
            0x0,  0x0,  0x0,  0x4, 0x28, 0x0,  0xfc, 0x1,  0x0,  0x4, 0x0,
            0x0,  0x0,  0xfc, 0x1, 0x0,  0x1,  0x5,  0x0,  0x0,  0x1, 0x5,
            0x0,  0x0,  0x5,  0x0, 0x0,  0xff, 0xff, 0xff, 0x27,
        };
        CHECK(run_input(input, sizeof(input)) == 0);
    }
#endif
}

TEST_CASE("non-bug: crash")
{
    // This is an interesting finding that is left here for
    // documentation.  This test actually should not run... the
    // problem is that when we build too large vectors via
    // concatenation, we can sometimes "overflow the shift".  This is
    // a degenerate case that we won't fix, but this helped adding
    // appropriate assertions to the code.
    //
    // To prevent this in further fuzzing, the can_concat check has
    // been made stricter.
    return;

    SECTION("simplified")
    {
        using vector_t =
            immer::flex_vector<int, immer::default_memory_policy, 2, 2>;
        auto var4 = vector_t{};
        var4      = var4.push_back(42);
        var4      = var4.push_back(42);
        var4      = var4.push_back(42);
        var4      = var4.push_back(42);
        var4      = var4.push_back(42);
        auto var0 = var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var0 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4 + var4;
        var4      = var4.update(4, [](auto x) { return x + 1; });
    }

#ifndef IMMER_DISABLE_FUZZER_DUE_TO_GCC_BUG
    SECTION("")
    {
        constexpr std::uint8_t input[] = {
            0x00, 0x00, 0x00, 0x23, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x00,
            0x00, 0x00, 0x00, 0x04, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
            0x04, 0x00, 0x04, 0x04, 0x04, 0x00, 0x00, 0x00, 0x00, 0x04, 0x04,
            0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
            0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x00, 0x2a, 0x00,
            0x00, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
            0x04, 0x04, 0x04, 0x04, 0xfc, 0xf9, 0x04, 0x04, 0x04, 0x04, 0x04,
            0x04, 0x04, 0x04, 0x04, 0x05, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x23, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
            0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
            0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x05, 0x04, 0x04, 0x04,
            0x04, 0x04, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x23, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x04, 0x04, 0x04, 0x04,
            0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
            0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x05, 0x04,
            0x04, 0x04, 0x04, 0x04, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x23,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x04, 0x04,
            0x04, 0x04, 0x04, 0x04, 0x00, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
            0x04, 0x04, 0x04, 0x04, 0x04, 0xd5, 0x04, 0x04, 0x04, 0x04, 0x04,
            0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x05, 0x04, 0x04,
            0x04, 0x04, 0x04, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x23, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x04, 0x04, 0x04,
            0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
            0x04, 0x04, 0x04, 0x00, 0x01, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
            0x04, 0x04, 0x04, 0x04, 0x04, 0x05, 0x04, 0x04, 0x04, 0x04, 0x04,
            0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x23, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x00,
            0x01, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
            0x04, 0x05, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x23, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3a,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x21, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x04,
            0x04, 0x04, 0x04, 0x00, 0x04, 0x04, 0x00, 0x00, 0x04, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
            0xff, 0xff, 0x13, 0x13, 0x13, 0x13, 0x13, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x29, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03,
            0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x21, 0x00, 0x10, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
            0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
            0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x05, 0x04, 0x04, 0x04, 0x04,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x3a, 0x00, 0x02, 0x00, 0x00, 0x00,
            0x04, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x23, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x04, 0x04, 0x04, 0x04,
            0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
            0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x05, 0x04,
            0x04, 0x04, 0x04, 0x04, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x23,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff,
            0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x04, 0x04,
            0x04, 0x04, 0x04, 0x04, 0x00, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
            0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
            0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x05, 0x04, 0x04,
            0x04, 0x04, 0x04, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x23, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x04, 0x04, 0x04,
            0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x00, 0x01,
            0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
            0x05, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x23, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04,
            0x04, 0x04, 0x04, 0x04, 0x04, 0x00, 0x01, 0x04, 0x04, 0x04, 0x04,
            0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x05, 0x04, 0x04, 0x04,
            0x04, 0x04, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x23, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x3a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x21, 0x04, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x04, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x23, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x04, 0x04, 0x04,
            0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
            0x00,
        };
        CHECK(run_input<8>(input, sizeof(input)) == 0);
    }
#endif
}
