"""Experimental data for the sloshing tank problems.

The data is extracted from plots in the following references:

 "ODD M. FALTINSEN, OLAV F. ROGNEBAKKE, IVAN A.
LUKOVSKY, and ALEXANDER N. TIMOKHA. Multidimensional
modal analysis of nonlinear sloshing in a rectangular tank with finite
water depth. Journal of Fluid Mechanics, 407:201234, 2000."
DOI: https://doi.org/10.1017/S0022112099007569

The case is from "Hakan Akyildiz, N. Erdem Unal,
Sloshing in a three-dimensional rectangular tank:
Numerical simulation and experimental validation,
Ocean Engineering, Volume 33, Issue 16,
2006, Pages 2135-2149, ISSN 0029-8018"
DOI: https://doi.org/10.1016/j.oceaneng.2005.11.001.

and

"Bouscasse, B., Antuono, M., Colagrossi, A., & Lugni, C. (2013).
Numerical and Experimental Investigation of Nonlinear Shallow Water Sloshing,
International Journal of Nonlinear Sciences and Numerical Simulation,
14(2), 123-138"
DOI: https://doi.org/10.1515/ijnsns-2012-0100

"""

import numpy as np
from io import StringIO

# Data extracted from Figure 7, Faltinsen et al., 2000
faltinsen = u"""
0.101	-0.0027
0.2485	-0.0134
0.3577	-0.0238
0.4981	-0.0312
0.6386	-0.0218
0.7166	-0.0083
0.779	0.0087
0.8258	0.018
0.8727	0.0308
0.9195	0.0393
0.9663	0.052
1.0287	0.0633
1.1629	0.0754
1.2784	0.0677
1.3408	0.052
1.3876	0.0384
1.4188	0.0248
1.45	0.0121
1.4812	-0.0032
1.5124	-0.0163
1.5436	-0.0261
1.5592	-0.0409
1.6061	-0.0528
1.6841	-0.0681
1.7621	-0.0796
1.9253	-0.088
2.043	-0.077
2.0898	-0.0651
2.1054	-0.0558
2.1366	-0.0465
2.1678	-0.0286
2.2094	-0.0105
2.2302	0.0138
2.2302	0.0036
2.2614	0.0303
2.277	0.0452
2.3083	0.0601
2.3239	0.0712
2.3551	0.0851
2.3863	0.1052
2.4175	0.1148
2.4643	0.1322
2.5735	0.1421
2.6828	0.1241
2.7296	0.1114
2.7608	0.0961
2.792	0.0825
2.8232	0.0681
2.8411	0.0588
2.8544	0.0494
2.8856	0.0376
2.9035	0.0258
2.9168	0.0134
2.948	0.0028
2.9792	-0.0121
3.0151	-0.0278
3.0417	-0.0414
3.0729	-0.0566
3.1041	-0.069
3.1509	-0.0804
3.2671	-0.0925
3.4006	-0.0774
3.4318	-0.0643
3.4786	-0.0511
3.5254	-0.0371
3.5722	-0.021
3.619	-0.0064
3.6725	0.0086
3.697	0.0206
3.7282	0.0299
3.7595	0.0418
3.7907	0.0537
3.8219	0.0625
3.8687	0.0753
3.8999	0.0866
3.9374	0.0992
4.0403	0.1093
4.134	0.097
4.1808	0.0791
4.212	0.0668
4.2588	0.0473
4.29	0.0371
4.3212	0.0223
4.368	0.0079
4.4304	-0.0087
4.4929	-0.0261
4.5709	-0.0405
4.7113	-0.0454
4.8518	-0.0337
4.9454	-0.0218
5.039	-0.0077
5.1326	0.0033
5.2949	0.0126
5.4619	0.0102
5.58	0.0033
5.7568	0.0033
5.9129	0.0129
6.0065	0.0231
6.1313	0.0343
6.2998	0.0398
6.4278	0.0325
6.4902	0.0223
6.5526	0.0096
6.615	-0.0045
6.6775	-0.021
6.7399	-0.0384
6.8023	-0.0494
6.8803	-0.0606
7.0045	-0.0668
7.0208	-0.0647
7.1456	-0.052
7.208	-0.0405
7.2704	-0.0247
7.3172	-0.0103
7.3485	0.0002
7.3953	0.0149
7.4109	0.0238
7.4577	0.0378
7.4889	0.0562
7.5201	0.0722
7.5825	0.0967
7.6137	0.1114
7.7074	0.1248
7.7854	0.1131
7.8166	0.1029
7.8478	0.0888
7.8946	0.0748
7.9414	0.0622
7.9726	0.05
8.0038	0.0422
8.035	0.0279
8.0663	0.0146
8.0975	0.0058
8.0975	-0.01
8.1287	-0.0226
8.1599	-0.0408
8.1911	-0.0583
8.2223	-0.066
8.3003	-0.0825
8.472	-0.0765
8.55	-0.0634
8.6124	-0.0477
8.6696	-0.0306
8.706	-0.0163
8.7372	-0.0036
8.7684	0.0104
8.7841	0.0201
8.8153	0.0322
8.8465	0.0452
8.8933	0.0668
8.9245	0.0825
8.9713	0.0919
8.9869	0.1038
9.0181	0.1165
9.0735	0.1302
9.1586	0.1414
9.2678	0.1292
9.2834	0.1173
9.3146	0.108
9.3458	0.0885
9.3614	0.0779
9.377	0.0626
9.4082	0.052
9.4082	0.037
9.4394	0.0207
9.4706	-0.0034
9.5019	-0.0159
9.5175	-0.0303
9.5799	-0.0558
9.6891	-0.0685
9.892	-0.0639
"""

# Data extracted from Figure 5, Akyildiz and Unal, 2006
# Probe 2
au_pitch_2 = u"""
0.036	3.226
0.418	3.491
0.81	3.591
1.176	3.456
1.548	3.266
1.928	3.052
2.307	2.797
2.679	2.901
3.051	3.17
3.422	3.378
3.794	3.561
4.166	3.578
4.567	3.37
4.925	3.075
5.311	2.91
5.683	2.823
6.055	3.057
6.427	3.283
6.817	3.504
7.19	3.591
7.563	3.478
7.929	3.231
8.308	2.983
8.687	2.866
9.059	2.936
9.431	3.161
9.803	3.404
"""

# Data extracted from Figure 5, Akyildiz and Unal, 2006
# Probe 8
au_pitch_8 = u"""
0.339	0.24
0.71	0.241
1.081	0.241
1.452	0.308
1.837	0.516
2.208	0.741
2.579	0.758
2.94	0.504
3.307	0.294
3.706	0.25
4.077	0.251
4.448	0.246
4.819	0.471
5.204	0.658
5.575	0.768
5.943	0.661
6.313	0.372
6.702	0.249
7.073	0.25
7.444	0.256
7.815	0.327
8.2	    0.618
8.571	0.742
8.942	0.69
9.327	0.532
9.698	0.259
"""

# Data extracted from Figure 19, Bouscasse et al., 2013
bouscasse_5 = """
8.547   -0.009
8.589   -0.011
8.631   -0.011
8.674   -0.009
8.716   -0.009
8.769   -0.007
8.783   -0.002
8.795   0.005
8.825   0.014
8.839   0.023
8.853   0.030
8.864   0.035
8.867   0.042
8.874   0.050
8.884   0.056
8.938   0.042
8.958   0.036
8.991   0.038
9.009   0.033
9.039   0.028
9.075   0.024
9.110   0.020
9.134   0.014
9.166   0.010
9.201   0.007
9.233   0.002
9.282   -0.002
9.310   -0.005
9.345   -0.005
9.391   -0.006
9.439   -0.008
9.484   -0.009
9.521   -0.008
9.560   -0.011
9.608   -0.011
9.655   -0.011
9.691   -0.009
9.736   -0.008
9.769   -0.004
9.777   0.003
9.789   0.008
9.810   0.015
9.837   0.026
9.848   0.034
9.858   0.041
9.864   0.048
9.864   0.056
9.876   0.060
9.893   0.052
9.905   0.047
9.914   0.040
9.957   0.033
9.978   0.037
10.013  0.031
10.059  0.023
10.098  0.019
10.154  0.012
10.181  0.005
10.217  0.003
10.255  -0.001
10.297  -0.002
10.332  -0.005
10.377  -0.007
10.414  -0.007
10.464  -0.008
10.506  -0.009
10.548  -0.011
10.587  -0.011
10.632  -0.011
10.668  -0.009
10.712  -0.009
10.748  -0.006
10.757  0.000
10.766  0.008
10.789  0.012
10.804  0.019
10.822  0.025
10.834  0.032
10.846  0.039
10.849  0.047
10.854  0.053
10.869  0.059
10.882  0.051
10.900  0.044
10.945  0.034
10.956  0.034
10.998  0.033

"""


def get_faltinsen_data():
    """Returns H v/s t for the case with hw=0.6 and T=1.3
    """
    # H vs t
    t, H = np.loadtxt(StringIO(faltinsen), unpack=True)
    return t, H


def get_au_pitch_data():
    """Returns P v/s t for the case of unbaffled tank, roll pitch = 2 rad/s
       pitch angle = 4 deg, transducer location no: 2 and 8. Fig. 5. from
       Akyildiz H., Erdem Unal N., 2006
    """
    # P vs t
    t2, p2 = np.loadtxt(StringIO(au_pitch_2), unpack=True)
    t8, p8 = np.loadtxt(StringIO(au_pitch_8), unpack=True)
    return t2, p2, t8, p8


def get_bouscasse_data():
    """Returns H v/s t for Series 5 case from Bouscasse et al., 2013
    """
    # H vs t
    t, H = np.loadtxt(StringIO(bouscasse_5), unpack=True)
    return t, H
