/*
Copyright 2018 Google Inc. All Rights Reserved.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS-IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

#include "hrtf_assets.h"

namespace sadie {

std::unique_ptr<std::string> HrtfAssets::GetFile(
    const std::string& filename) const {
  AssetDataMap::const_iterator map_entry_itr = kAssetMap.find(filename);
  if (map_entry_itr == kAssetMap.end()) {
    return nullptr;
  }
  const char* data =
      reinterpret_cast<const char*>(map_entry_itr->second.data());
  const size_t data_size = map_entry_itr->second.size();
  return std::unique_ptr<std::string>(new std::string(data, data_size));
}

const HrtfAssets::AssetDataMap HrtfAssets::kAssetMap = {
    {"WAV/Subject_002/SH/sh_hrir_order_1.wav",
     {0x52, 0x49, 0x46, 0x46, 0x24, 0x8, 0x0, 0x0, 0x57, 0x41, 0x56, 0x45, 0x66,
      0x6d, 0x74, 0x20, 0x10, 0x0, 0x0, 0x0, 0x1, 0x0, 0x4, 0x0, 0x80, 0xbb,
      0x0, 0x0, 0x0, 0xdc, 0x5, 0x0, 0x8, 0x0, 0x10, 0x0, 0x64, 0x61, 0x74,
      0x61, 0x0, 0x8, 0x0, 0x0, 0xfe, 0xff, 0x2, 0x0, 0x0, 0x0, 0x0, 0x0, 0xf6,
      0xff, 0x10, 0x0, 0xfd, 0xff, 0x2, 0x0, 0xfb, 0xff, 0xb, 0x0, 0xff, 0xff,
      0x1, 0x0, 0xfe, 0xff, 0x7, 0x0, 0xff, 0xff, 0x1, 0x0, 0xfe, 0xff, 0xc,
      0x0, 0xff, 0xff, 0x1, 0x0, 0xff, 0xff, 0x1, 0x0, 0xfd, 0xff, 0x0, 0x0,
      0xfb, 0xff, 0x1, 0x0, 0x2, 0x0, 0xfe, 0xff, 0x8, 0x0, 0x4, 0x0, 0xf9,
      0xff, 0x0, 0x0, 0xf8, 0xff, 0xfa, 0xff, 0x2, 0x0, 0xfc, 0xff, 0x18, 0x0,
      0x10, 0x0, 0x5, 0x0, 0x4, 0x0, 0xee, 0xff, 0xe7, 0xff, 0xf3, 0xff, 0x0,
      0x0, 0xfb, 0xff, 0xf4, 0xff, 0x1a, 0x0, 0xf9, 0xff, 0xcc, 0xff, 0xc0,
      0xff, 0xd5, 0xff, 0x4, 0x0, 0xfd, 0xfd, 0xf8, 0xfd, 0x1d, 0x0, 0xce, 0xff,
      0x1d, 0xfc, 0x12, 0xfc, 0x1a, 0x0, 0x41, 0x0, 0xf1, 0x1, 0xf1, 0x1, 0x4e,
      0xfe, 0x87, 0x0, 0xb4, 0xd, 0xa5, 0xd, 0xd0, 0xfc, 0x75, 0xfe, 0x37, 0x1,
      0x25, 0x1, 0xe7, 0x7, 0x5b, 0xfe, 0x49, 0xf5, 0x3e, 0xf5, 0x14, 0xa, 0x6f,
      0x7, 0x14, 0xfd, 0x3, 0xfd, 0x2c, 0xee, 0xa4, 0x2, 0xc0, 0xf5, 0xb5, 0xf5,
      0x3b, 0xfc, 0xfd, 0xf0, 0x68, 0xfd, 0x6b, 0xfd, 0x91, 0xd, 0xd0, 0x0,
      0xb8, 0xf4, 0xce, 0xf4, 0x61, 0xf3, 0xe2, 0x4, 0xf3, 0xe6, 0x3a, 0xe7,
      0x88, 0x5, 0x5e, 0xf7, 0xdc, 0xfd, 0x4f, 0xfe, 0x94, 0x8, 0xcd, 0x6, 0x6a,
      0x2, 0xf7, 0x2, 0x6, 0xf8, 0x7a, 0xfe, 0x2b, 0x1c, 0xfd, 0x1c, 0x7c, 0xfb,
      0xad, 0xfd, 0x6e, 0x18, 0xa1, 0x19, 0x9, 0xfd, 0x7, 0x10, 0x6, 0xfd, 0xe3,
      0xfe, 0x49, 0x15, 0x5d, 0x6, 0x26, 0xc, 0xcd, 0xd, 0xa, 0x1, 0x4a, 0x1,
      0xa0, 0x3, 0x3b, 0x4, 0x51, 0xeb, 0x6a, 0xfd, 0x41, 0x5, 0x0, 0x6, 0x34,
      0xff, 0x26, 0xf8, 0x24, 0x5, 0xac, 0x4, 0xba, 0x4, 0x2, 0xfa, 0xb3, 0x3,
      0x97, 0x1, 0xe2, 0x5, 0x50, 0x0, 0xd9, 0x9, 0xb2, 0xb, 0x88, 0x3, 0xe6,
      0x0, 0x2d, 0x0, 0xb6, 0x4, 0x11, 0x6, 0x31, 0xfd, 0x78, 0x1, 0xe6, 0x6,
      0xa5, 0xf9, 0x2c, 0x1, 0xb5, 0xf8, 0x0, 0x3, 0x8e, 0xf6, 0xba, 0xfc, 0x68,
      0xf9, 0x39, 0x1, 0x6e, 0x0, 0xff, 0xff, 0xcf, 0xfa, 0xc7, 0x0, 0xf, 0xfc,
      0x26, 0x3, 0xac, 0xfb, 0xc, 0x4, 0x32, 0x5, 0xe6, 0xfd, 0x98, 0x4, 0x82,
      0x2, 0x1d, 0x2, 0x28, 0x1, 0x4, 0x6, 0xdc, 0xfa, 0x3a, 0xfd, 0xf7, 0x1,
      0xf9, 0x5, 0x6a, 0xfe, 0x7, 0x0, 0x2d, 0x0, 0x98, 0xff, 0x68, 0xf8, 0xd0,
      0xfe, 0x91, 0x0, 0x3c, 0x4, 0xf3, 0xf7, 0x34, 0x3, 0x95, 0x3, 0x81, 0x1,
      0x22, 0xfa, 0x87, 0x1, 0x6c, 0xff, 0x31, 0x0, 0xa1, 0xfd, 0xc2, 0x1, 0xf1,
      0xfc, 0x1f, 0x3, 0x5b, 0xfd, 0x9d, 0x0, 0x43, 0xff, 0xfb, 0xfe, 0xd2,
      0xfb, 0x3, 0xff, 0x2f, 0xfd, 0xc2, 0x1, 0x8e, 0xfc, 0xee, 0xff, 0x98,
      0xff, 0x1d, 0xff, 0x10, 0xf8, 0x23, 0xff, 0x66, 0x1, 0x20, 0x0, 0xcb,
      0xfc, 0x23, 0x1, 0x37, 0x2, 0xf7, 0xfd, 0xce, 0xfb, 0xfd, 0x0, 0x62, 0xff,
      0x3e, 0xfe, 0x5d, 0xfe, 0xae, 0x0, 0xe0, 0xfc, 0xc2, 0x0, 0x7b, 0xff,
      0x96, 0x0, 0x20, 0xff, 0xe2, 0xfe, 0xed, 0xfc, 0xc, 0x0, 0x21, 0xff, 0xc5,
      0x0, 0x31, 0xfd, 0xcb, 0xfe, 0x7d, 0x0, 0x91, 0xfe, 0x80, 0xfa, 0x7b,
      0xff, 0xce, 0x0, 0x7d, 0xfe, 0xc, 0xfe, 0x93, 0x1, 0xdc, 0xff, 0xbe, 0xfd,
      0x33, 0xfd, 0xd7, 0x0, 0x5f, 0xff, 0xc4, 0xfe, 0xbc, 0xfe, 0x5f, 0x0,
      0x8c, 0xff, 0x99, 0xfe, 0xc3, 0xfd, 0x4f, 0x0, 0x7, 0xff, 0xba, 0xfe,
      0x2d, 0xfd, 0xd, 0x0, 0x47, 0xff, 0x62, 0x0, 0x34, 0xfe, 0xad, 0xfe, 0x23,
      0x1, 0xee, 0xfd, 0xd0, 0xfb, 0x3d, 0xff, 0x96, 0x0, 0x7d, 0xfe, 0x40,
      0xfe, 0x9, 0x0, 0xfe, 0x0, 0xee, 0xfd, 0x6c, 0xfd, 0x1e, 0x0, 0xe9, 0x0,
      0x66, 0xfe, 0xf1, 0xfe, 0xef, 0x0, 0x51, 0xff, 0x2, 0xff, 0x53, 0xfe,
      0x4b, 0xff, 0x4e, 0xff, 0xa5, 0xff, 0xf1, 0xfd, 0x41, 0xff, 0xa, 0x0,
      0xc4, 0xfe, 0xeb, 0xfd, 0xe0, 0xff, 0xc7, 0xff, 0xe9, 0xfd, 0x7c, 0xfd,
      0x83, 0xff, 0x7b, 0x0, 0x7d, 0xfe, 0x45, 0xff, 0x4d, 0x0, 0x83, 0x0, 0x47,
      0xfd, 0x1b, 0xfe, 0xca, 0x0, 0xd1, 0xfe, 0x2b, 0xff, 0xab, 0xff, 0xfd,
      0xff, 0xe2, 0xff, 0xe0, 0xfe, 0x4b, 0xfe, 0x11, 0x0, 0xed, 0xff, 0xc2,
      0xfe, 0x7d, 0xfe, 0xb1, 0x0, 0xaa, 0xff, 0xdc, 0xfe, 0x98, 0xfe, 0x92,
      0xff, 0xb2, 0x0, 0x3a, 0xfe, 0xd7, 0xfd, 0xba, 0xff, 0xe8, 0xff, 0x44,
      0xfe, 0xf4, 0xfe, 0x8, 0x0, 0x73, 0xff, 0x46, 0xfe, 0x8c, 0xfe, 0xd7,
      0xff, 0xf1, 0xff, 0xea, 0xfe, 0x80, 0xff, 0x4b, 0x0, 0xa6, 0xff, 0x60,
      0xfe, 0xf7, 0xfe, 0xed, 0xff, 0x61, 0xff, 0x8f, 0xff, 0xca, 0xff, 0xc1,
      0xff, 0x2a, 0x0, 0x6c, 0xfe, 0xe6, 0xfe, 0x0, 0x0, 0xc8, 0xff, 0x28, 0xfe,
      0xb, 0xff, 0xb5, 0xff, 0x3, 0x0, 0xde, 0xfe, 0xd8, 0xff, 0x60, 0xff, 0x91,
      0x0, 0x19, 0xfe, 0x1e, 0xff, 0x4e, 0x0, 0x97, 0xff, 0xdb, 0xfe, 0x17, 0x0,
      0x16, 0x0, 0xfc, 0xff, 0xcf, 0xfe, 0x5c, 0xff, 0xaa, 0xff, 0x4d, 0x0,
      0xb0, 0xfe, 0xc5, 0xff, 0x54, 0x0, 0x98, 0xff, 0x61, 0xfe, 0xae, 0xff,
      0xab, 0xff, 0x4, 0x0, 0xc6, 0xfe, 0xce, 0xff, 0xc3, 0xff, 0x3c, 0x0, 0x1d,
      0xfe, 0xf2, 0xff, 0x2d, 0x0, 0x8d, 0xff, 0x83, 0xfe, 0x34, 0x0, 0xcc,
      0xff, 0x29, 0x0, 0xfd, 0xfe, 0x4d, 0x0, 0xee, 0xff, 0x28, 0x0, 0x2b, 0xfe,
      0xb0, 0xff, 0x12, 0x0, 0x99, 0xff, 0x33, 0xff, 0x84, 0x0, 0xa7, 0xff,
      0x55, 0x0, 0xa1, 0xfe, 0xc5, 0xff, 0xc2, 0xff, 0xf4, 0xff, 0x81, 0xfe,
      0x38, 0x0, 0x29, 0x0, 0xac, 0xff, 0xe3, 0xfe, 0x76, 0x0, 0xb3, 0xff, 0x52,
      0x0, 0xa6, 0xfe, 0x25, 0x0, 0x1f, 0x0, 0xee, 0xff, 0x93, 0xfe, 0x6a, 0x0,
      0x36, 0x0, 0xbc, 0xff, 0xe0, 0xfe, 0x3f, 0x0, 0x9e, 0xff, 0x6b, 0x0, 0xbe,
      0xfe, 0x31, 0x0, 0x19, 0x0, 0xf3, 0xff, 0x58, 0xfe, 0x2, 0x0, 0xf8, 0xff,
      0xd5, 0xff, 0xa, 0xff, 0x6d, 0x0, 0xcc, 0xff, 0x6d, 0x0, 0x70, 0xfe, 0xfb,
      0xff, 0x41, 0x0, 0xbf, 0xff, 0xcd, 0xfe, 0x62, 0x0, 0xe, 0x0, 0xf4, 0xff,
      0x20, 0xff, 0x3a, 0x0, 0xe4, 0xff, 0x4c, 0x0, 0x92, 0xfe, 0xe0, 0xff,
      0x41, 0x0, 0xcc, 0xff, 0xa, 0xff, 0x68, 0x0, 0xb, 0x0, 0x31, 0x0, 0x6,
      0xff, 0xa, 0x0, 0xfc, 0xff, 0x49, 0x0, 0xca, 0xfe, 0x4c, 0x0, 0x6a, 0x0,
      0xb9, 0xff, 0xff, 0xfe, 0x6d, 0x0, 0x6, 0x0, 0xb, 0x0, 0x21, 0xff, 0x4e,
      0x0, 0xfc, 0xff, 0x10, 0x0, 0xa0, 0xfe, 0x1e, 0x0, 0x3b, 0x0, 0x8c, 0xff,
      0x1c, 0xff, 0x5a, 0x0, 0xdf, 0xff, 0x22, 0x0, 0x13, 0xff, 0x24, 0x0, 0x5,
      0x0, 0xff, 0xff, 0xcf, 0xfe, 0x11, 0x0, 0x24, 0x0, 0xb7, 0xff, 0x62, 0xff,
      0x60, 0x0, 0xcb, 0xff, 0x55, 0x0, 0x8, 0xff, 0xdd, 0xff, 0x14, 0x0, 0xfa,
      0xff, 0x26, 0xff, 0x2b, 0x0, 0x35, 0x0, 0xe8, 0xff, 0x62, 0xff, 0x20, 0x0,
      0xd0, 0xff, 0x5c, 0x0, 0x25, 0xff, 0xf3, 0xff, 0x1e, 0x0, 0xf9, 0xff,
      0x24, 0xff, 0x2b, 0x0, 0x16, 0x0, 0xea, 0xff, 0x5e, 0xff, 0x1b, 0x0, 0xbe,
      0xff, 0x37, 0x0, 0x11, 0xff, 0xfe, 0xff, 0x18, 0x0, 0xc3, 0xff, 0x23,
      0xff, 0x28, 0x0, 0x9, 0x0, 0xe8, 0xff, 0x6b, 0xff, 0x2a, 0x0, 0xe1, 0xff,
      0x27, 0x0, 0x0, 0xff, 0xf5, 0xff, 0x22, 0x0, 0xb1, 0xff, 0x60, 0xff, 0x56,
      0x0, 0xf2, 0xff, 0xe, 0x0, 0x59, 0xff, 0x9, 0x0, 0xde, 0xff, 0x20, 0x0,
      0x21, 0xff, 0x2, 0x0, 0x24, 0x0, 0xc4, 0xff, 0x76, 0xff, 0x3e, 0x0, 0xe7,
      0xff, 0x27, 0x0, 0x5c, 0xff, 0xf0, 0xff, 0xec, 0xff, 0x16, 0x0, 0x3d,
      0xff, 0x16, 0x0, 0x2d, 0x0, 0xc9, 0xff, 0x76, 0xff, 0x29, 0x0, 0xd4, 0xff,
      0x27, 0x0, 0x5f, 0xff, 0xfd, 0xff, 0xf6, 0xff, 0xfb, 0xff, 0x3f, 0xff,
      0x13, 0x0, 0x25, 0x0, 0xd0, 0xff, 0x95, 0xff, 0x35, 0x0, 0xcf, 0xff, 0x35,
      0x0, 0x4b, 0xff, 0xe7, 0xff, 0x4, 0x0, 0xdc, 0xff, 0x54, 0xff, 0x18, 0x0,
      0x15, 0x0, 0xe4, 0xff, 0x8a, 0xff, 0x1e, 0x0, 0xc9, 0xff, 0x37, 0x0, 0x4e,
      0xff, 0xf4, 0xff, 0xb, 0x0, 0xd2, 0xff, 0x83, 0xff, 0x41, 0x0, 0x15, 0x0,
      0xf4, 0xff, 0x78, 0xff, 0xfb, 0xff, 0xe6, 0xff, 0x30, 0x0, 0x58, 0xff,
      0xf6, 0xff, 0x9, 0x0, 0xd7, 0xff, 0x75, 0xff, 0x1c, 0x0, 0xed, 0xff, 0xf6,
      0xff, 0x7c, 0xff, 0xf3, 0xff, 0xf6, 0xff, 0x24, 0x0, 0x57, 0xff, 0xf7,
      0xff, 0xa, 0x0, 0xd9, 0xff, 0x67, 0xff, 0x1, 0x0, 0xe3, 0xff, 0x3, 0x0,
      0x61, 0xff, 0xdb, 0xff, 0xa, 0x0, 0x16, 0x0, 0x4d, 0xff, 0xf1, 0xff, 0xff,
      0xff, 0xdb, 0xff, 0x93, 0xff, 0x23, 0x0, 0xd8, 0xff, 0x11, 0x0, 0x88,
      0xff, 0x3, 0x0, 0xb, 0x0, 0x19, 0x0, 0x7f, 0xff, 0x21, 0x0, 0xe, 0x0,
      0xfa, 0xff, 0xa3, 0xff, 0x24, 0x0, 0xee, 0xff, 0x1b, 0x0, 0x99, 0xff,
      0x15, 0x0, 0xf8, 0xff, 0xe, 0x0, 0x8a, 0xff, 0x27, 0x0, 0xf0, 0xff, 0xf0,
      0xff, 0xaf, 0xff, 0x1e, 0x0, 0xf8, 0xff, 0x12, 0x0, 0xab, 0xff, 0x23, 0x0,
      0x4, 0x0, 0x5, 0x0, 0x9b, 0xff, 0x2e, 0x0, 0xf5, 0xff, 0xf5, 0xff, 0xcc,
      0xff, 0x38, 0x0, 0x4, 0x0, 0x1a, 0x0, 0xa2, 0xff, 0x24, 0x0, 0xf8, 0xff,
      0x0, 0x0, 0xa9, 0xff, 0x3b, 0x0, 0xe7, 0xff, 0xff, 0xff, 0xc4, 0xff, 0x33,
      0x0, 0x0, 0x0, 0x19, 0x0, 0x98, 0xff, 0x25, 0x0, 0xf5, 0xff, 0xff, 0xff,
      0xb4, 0xff, 0x45, 0x0, 0xed, 0xff, 0x5, 0x0, 0xc1, 0xff, 0x2a, 0x0, 0x5,
      0x0, 0x15, 0x0, 0xae, 0xff, 0x27, 0x0, 0xef, 0xff, 0x0, 0x0, 0xc8, 0xff,
      0x39, 0x0, 0xea, 0xff, 0xb, 0x0, 0xcb, 0xff, 0x23, 0x0, 0x7, 0x0, 0x15,
      0x0, 0xb3, 0xff, 0x1a, 0x0, 0xec, 0xff, 0x1, 0x0, 0xd5, 0xff, 0x2a, 0x0,
      0xf4, 0xff, 0x12, 0x0, 0xc9, 0xff, 0x16, 0x0, 0x6, 0x0, 0xb, 0x0, 0xbc,
      0xff, 0x14, 0x0, 0xeb, 0xff, 0xfe, 0xff, 0xe3, 0xff, 0x27, 0x0, 0x1, 0x0,
      0x12, 0x0, 0xc5, 0xff, 0xb, 0x0, 0x6, 0x0, 0x5, 0x0, 0xc9, 0xff, 0x14,
      0x0, 0xef, 0xff, 0x2, 0x0, 0xe1, 0xff, 0x19, 0x0, 0x3, 0x0, 0x16, 0x0,
      0xc1, 0xff, 0x6, 0x0, 0x2, 0x0, 0x2, 0x0, 0xce, 0xff, 0x14, 0x0, 0xef,
      0xff, 0x0, 0x0, 0xe1, 0xff, 0x16, 0x0, 0x6, 0x0, 0x13, 0x0, 0xc3, 0xff,
      0x7, 0x0, 0x1, 0x0, 0xfd, 0xff, 0xd4, 0xff, 0x14, 0x0, 0xee, 0xff, 0x4,
      0x0, 0xe0, 0xff, 0x10, 0x0, 0x6, 0x0, 0x10, 0x0, 0xc1, 0xff, 0x1, 0x0,
      0xfa, 0xff, 0xf8, 0xff, 0xdb, 0xff, 0x14, 0x0, 0xf3, 0xff, 0x5, 0x0, 0xdc,
      0xff, 0x9, 0x0, 0x9, 0x0, 0xb, 0x0, 0xc3, 0xff, 0x2, 0x0, 0xf8, 0xff,
      0xf7, 0xff, 0xe1, 0xff, 0x12, 0x0, 0xf8, 0xff, 0x8, 0x0, 0xda, 0xff, 0x4,
      0x0, 0x7, 0x0, 0xa, 0x0, 0xc8, 0xff, 0x2, 0x0, 0xf4, 0xff, 0xf9, 0xff,
      0xe4, 0xff, 0xd, 0x0, 0xf9, 0xff, 0xe, 0x0, 0xd5, 0xff, 0xfe, 0xff, 0x7,
      0x0, 0x8, 0x0, 0xcb, 0xff, 0x3, 0x0, 0xf4, 0xff, 0xfb, 0xff, 0xe6, 0xff,
      0xc, 0x0, 0xfd, 0xff, 0xe, 0x0, 0xd2, 0xff, 0xfe, 0xff, 0x3, 0x0, 0x3,
      0x0, 0xd1, 0xff, 0x6, 0x0, 0xf1, 0xff, 0xfd, 0xff, 0xe5, 0xff, 0x9, 0x0,
      0xff, 0xff, 0xf, 0x0, 0xce, 0xff, 0xfc, 0xff, 0xff, 0xff, 0x1, 0x0, 0xd4,
      0xff, 0x7, 0x0, 0xf3, 0xff, 0xff, 0xff, 0xe4, 0xff, 0x7, 0x0, 0x1, 0x0,
      0xd, 0x0, 0xce, 0xff, 0xfc, 0xff, 0xfd, 0xff, 0xfe, 0xff, 0xda, 0xff, 0x9,
      0x0, 0xf5, 0xff, 0x2, 0x0, 0xe2, 0xff, 0x4, 0x0, 0x2, 0x0, 0xd, 0x0, 0xcd,
      0xff, 0xfc, 0xff, 0xfb, 0xff, 0xfc, 0xff, 0xdf, 0xff, 0x8, 0x0, 0xf6,
      0xff, 0x4, 0x0, 0xdf, 0xff, 0xff, 0xff, 0x3, 0x0, 0xa, 0x0, 0xd0, 0xff,
      0xfd, 0xff, 0xf9, 0xff, 0xfb, 0xff, 0xe2, 0xff, 0x7, 0x0, 0xf9, 0xff, 0x6,
      0x0, 0xdd, 0xff, 0xfe, 0xff, 0x3, 0x0, 0x7, 0x0, 0xd3, 0xff, 0xff, 0xff,
      0xf8, 0xff, 0xfb, 0xff, 0xe4, 0xff, 0x6, 0x0, 0xfb, 0xff, 0x7, 0x0, 0xdb,
      0xff, 0xfc, 0xff, 0x1, 0x0, 0x4, 0x0, 0xd5, 0xff, 0x0, 0x0, 0xf8, 0xff,
      0xfc, 0xff, 0xe7, 0xff, 0x5, 0x0, 0xfd, 0xff, 0x9, 0x0, 0xd9, 0xff, 0xfb,
      0xff, 0x1, 0x0, 0x2, 0x0, 0xda, 0xff, 0x2, 0x0, 0xf7, 0xff, 0xfd, 0xff,
      0xe6, 0xff, 0x3, 0x0, 0xff, 0xff, 0x9, 0x0, 0xd8, 0xff, 0xfb, 0xff, 0xff,
      0xff, 0xff, 0xff, 0xde, 0xff, 0x4, 0x0, 0xf8, 0xff, 0xff, 0xff, 0xe6,
      0xff, 0x2, 0x0, 0x0, 0x0, 0x8, 0x0, 0xd8, 0xff, 0xfd, 0xff, 0xfd, 0xff,
      0xfe, 0xff, 0xe1, 0xff, 0x5, 0x0, 0xf9, 0xff, 0x0, 0x0, 0xe5, 0xff, 0x1,
      0x0, 0x0, 0x0, 0x7, 0x0, 0xd9, 0xff, 0xfe, 0xff, 0xfc, 0xff, 0xfd, 0xff,
      0xe4, 0xff, 0x6, 0x0, 0xfa, 0xff, 0x2, 0x0, 0xe3, 0xff, 0x0, 0x0, 0x0,
      0x0, 0x6, 0x0, 0xdb, 0xff, 0x0, 0x0, 0xfb, 0xff, 0xfc, 0xff, 0xe7, 0xff,
      0x6, 0x0, 0xfc, 0xff, 0x4, 0x0, 0xe1, 0xff, 0xff, 0xff, 0x0, 0x0, 0x4,
      0x0, 0xdd, 0xff, 0x2, 0x0, 0xfa, 0xff, 0xfd, 0xff, 0xe7, 0xff, 0x5, 0x0,
      0xfc, 0xff, 0x5, 0x0, 0xe0, 0xff, 0xff, 0xff, 0xff, 0xff, 0x2, 0x0, 0xdf,
      0xff, 0x3, 0x0, 0xfa, 0xff, 0xfe, 0xff, 0xe8, 0xff, 0x5, 0x0, 0xfe, 0xff,
      0x5, 0x0, 0xe0, 0xff, 0xff, 0xff, 0xfe, 0xff, 0x1, 0x0, 0xe1, 0xff, 0x4,
      0x0, 0xfb, 0xff, 0xff, 0xff, 0xe8, 0xff, 0x4, 0x0, 0xff, 0xff, 0x6, 0x0,
      0xde, 0xff, 0xff, 0xff, 0xfe, 0xff, 0x0, 0x0, 0xe1, 0xff, 0x5, 0x0, 0xfb,
      0xff, 0x1, 0x0, 0xe2, 0xff, 0x3, 0x0, 0xff, 0xff, 0x4, 0x0, 0xdb, 0xff,
      0x1, 0x0, 0xfd, 0xff, 0x0, 0x0, 0xdd, 0xff, 0x4, 0x0, 0xfd, 0xff, 0x1,
      0x0, 0xdc, 0xff, 0x2, 0x0, 0xfe, 0xff, 0x3, 0x0, 0xd9, 0xff, 0x2, 0x0,
      0xfd, 0xff, 0x0, 0x0, 0xdb, 0xff, 0x3, 0x0, 0xfd, 0xff, 0x2, 0x0, 0xdb,
      0xff, 0x2, 0x0, 0xfe, 0xff, 0x2, 0x0, 0xdb, 0xff, 0x2, 0x0, 0xfd, 0xff,
      0x1, 0x0, 0xdd, 0xff, 0x2, 0x0, 0xfd, 0xff, 0x2, 0x0, 0xde, 0xff, 0x2,
      0x0, 0xfe, 0xff, 0x2, 0x0, 0xe1, 0xff, 0x1, 0x0, 0xfe, 0xff, 0x1, 0x0,
      0xe3, 0xff, 0x1, 0x0, 0xfe, 0xff, 0x1, 0x0}},
    {"WAV/Subject_002/SH/sh_hrir_order_2.wav",
     {0x52, 0x49, 0x46, 0x46, 0x24, 0x12, 0x0, 0x0, 0x57, 0x41, 0x56, 0x45,
      0x66, 0x6d, 0x74, 0x20, 0x10, 0x0, 0x0, 0x0, 0x1, 0x0, 0x9, 0x0, 0x80,
      0xbb, 0x0, 0x0, 0x0, 0x2f, 0xd, 0x0, 0x12, 0x0, 0x10, 0x0, 0x64, 0x61,
      0x74, 0x61, 0x0, 0x12, 0x0, 0x0, 0xfe, 0xff, 0x4, 0x0, 0xff, 0xff, 0xff,
      0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xf3,
      0xff, 0x19, 0x0, 0xfd, 0xff, 0xfe, 0xff, 0xfe, 0xff, 0x0, 0x0, 0xff, 0xff,
      0xfe, 0xff, 0xfa, 0xff, 0xfa, 0xff, 0xc, 0x0, 0xfe, 0xff, 0xff, 0xff,
      0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc, 0xff, 0xfe, 0xff,
      0x3, 0x0, 0x0, 0x0, 0xff, 0xff, 0xff, 0xff, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
      0xff, 0xff, 0xfe, 0xff, 0xa, 0x0, 0xff, 0xff, 0x1, 0x0, 0x0, 0x0, 0x0,
      0x0, 0xfe, 0xff, 0xfe, 0xff, 0xfd, 0xff, 0x4, 0x0, 0xfd, 0xff, 0x1, 0x0,
      0xfd, 0xff, 0xfe, 0xff, 0x1, 0x0, 0x0, 0x0, 0x1, 0x0, 0x1, 0x0, 0x3, 0x0,
      0x3, 0x0, 0x0, 0x0, 0x1, 0x0, 0x1, 0x0, 0x0, 0x0, 0xfc, 0xff, 0xfd, 0xff,
      0xfe, 0xff, 0x4, 0x0, 0xec, 0xff, 0x1, 0x0, 0x5, 0x0, 0x7, 0x0, 0xff,
      0xff, 0x9, 0x0, 0x9, 0x0, 0x9, 0x0, 0xfc, 0xff, 0xf2, 0xff, 0x5, 0x0,
      0xf7, 0xff, 0xf5, 0xff, 0x5, 0x0, 0xfd, 0xff, 0xfa, 0xff, 0x4, 0x0, 0xf3,
      0xff, 0xc7, 0xff, 0xfa, 0xff, 0x1d, 0x0, 0x23, 0x0, 0xf2, 0xff, 0xf, 0x0,
      0x7, 0x0, 0x19, 0x0, 0x47, 0xff, 0x74, 0xfe, 0xd9, 0xff, 0xfd, 0xff, 0x9,
      0x0, 0xdb, 0xff, 0xa4, 0x0, 0x65, 0x0, 0xb6, 0x0, 0x86, 0xfe, 0xff, 0xfc,
      0x13, 0x0, 0xdc, 0xff, 0xd0, 0xff, 0x18, 0x0, 0x19, 0x1, 0x93, 0x0, 0x5b,
      0x1, 0xc0, 0x1, 0x6b, 0x3, 0xa3, 0x0, 0x4f, 0xff, 0x1b, 0xff, 0xbb, 0x0,
      0x8e, 0xfe, 0x2f, 0xff, 0x56, 0xfe, 0x3, 0x6, 0x37, 0xc, 0xa1, 0xff, 0x47,
      0x0, 0x63, 0x0, 0x96, 0xff, 0x95, 0xfa, 0x9a, 0xfc, 0x1f, 0xfa, 0xa9,
      0xfd, 0xff, 0xfb, 0xa3, 0xff, 0xfe, 0x0, 0x3d, 0x1, 0x70, 0xff, 0x74, 0x0,
      0x71, 0xff, 0x67, 0x1, 0x6e, 0xf7, 0x70, 0xef, 0xa3, 0xff, 0xbe, 0xff,
      0xd7, 0xff, 0xb4, 0xff, 0xf7, 0x5, 0xd9, 0x2, 0x68, 0x7, 0x93, 0x2, 0x3a,
      0x2, 0x67, 0xfb, 0xe2, 0xff, 0xd9, 0x0, 0xd6, 0xfb, 0xcf, 0x2, 0x5c, 0x4,
      0x94, 0x0, 0x7e, 0x2, 0xbb, 0x2, 0x6f, 0x3, 0xf7, 0xfc, 0x1, 0xfc, 0xc8,
      0x3, 0x23, 0x2, 0xd3, 0x3, 0xed, 0xff, 0x74, 0xfc, 0x11, 0xfb, 0xb0, 0xc,
      0x90, 0x9, 0x42, 0x8, 0x25, 0x9, 0xd, 0xff, 0xdf, 0xfd, 0x11, 0x1, 0x54,
      0xfb, 0xb1, 0xf7, 0x22, 0xfa, 0xfa, 0x2, 0xe5, 0x4, 0xfe, 0xf9, 0xc2, 0x2,
      0x3b, 0x1, 0xfb, 0x2, 0x65, 0xf4, 0x1a, 0xee, 0xc5, 0xf4, 0x5d, 0xe7,
      0x45, 0xe6, 0x84, 0xfc, 0xa1, 0x2, 0x61, 0xfc, 0x4a, 0x4, 0xcd, 0xfc,
      0x45, 0xfd, 0x38, 0x6, 0x17, 0x6, 0x21, 0x3, 0xbf, 0x5, 0x55, 0xfd, 0x37,
      0xfc, 0x9e, 0xfe, 0x9a, 0x7, 0xea, 0xa, 0xcd, 0xfd, 0xb7, 0xd, 0xb8, 0xc,
      0x47, 0xfa, 0xde, 0xf8, 0x67, 0xff, 0x8a, 0xfe, 0x49, 0x8, 0x89, 0x13,
      0x93, 0xfb, 0xc9, 0xf6, 0x9e, 0xfc, 0x89, 0xfb, 0x1a, 0xed, 0xf0, 0xec,
      0xb0, 0xf4, 0x20, 0x5, 0xde, 0xd, 0x7a, 0xb, 0x5f, 0xff, 0x9d, 0x6, 0x50,
      0x6, 0x31, 0xff, 0xc9, 0xfb, 0xc9, 0xf3, 0x41, 0x1, 0x8b, 0xfd, 0x9f, 0x0,
      0x43, 0x4, 0xf6, 0xf9, 0xae, 0xfe, 0x72, 0x1, 0xc0, 0x5, 0x91, 0x4, 0xcd,
      0x6, 0xff, 0x10, 0x81, 0xf2, 0x62, 0xa, 0x50, 0xa, 0xd9, 0xf7, 0x86, 0xf8,
      0xdd, 0x7, 0x5d, 0xf8, 0x71, 0x0, 0xa1, 0xb, 0x1b, 0xfd, 0x37, 0xf7, 0x65,
      0x7, 0xff, 0xff, 0x19, 0xe, 0x1e, 0x16, 0x62, 0xf1, 0x64, 0x4, 0xd4, 0x3,
      0xb1, 0xe, 0x35, 0x0, 0x71, 0xf8, 0xe8, 0x9, 0xba, 0x3, 0x6b, 0x3, 0x3d,
      0x0, 0xfb, 0x8, 0x1d, 0x14, 0xda, 0xa, 0x44, 0xf, 0xad, 0x4, 0xec, 0x9,
      0xb3, 0xfb, 0xe2, 0x1, 0x1, 0xf6, 0xb1, 0xf9, 0xf9, 0x2, 0xc0, 0xf8, 0x4e,
      0xf2, 0x27, 0xf5, 0x86, 0xff, 0xee, 0xff, 0xdc, 0xff, 0xf5, 0xf4, 0xc8,
      0x5, 0x26, 0x4, 0x7d, 0xf0, 0x1a, 0xf6, 0xce, 0xf6, 0xda, 0xf8, 0xce,
      0xfd, 0x2d, 0xfd, 0xc, 0x3, 0xd1, 0x9, 0x20, 0x4, 0x0, 0xf9, 0x11, 0x5,
      0x9, 0x0, 0x1c, 0xfc, 0xa7, 0x2, 0x89, 0x1, 0x9c, 0x6, 0x65, 0x1, 0xd1,
      0xf3, 0x1, 0xc, 0xcc, 0xc, 0xb0, 0x4, 0x61, 0x1, 0xd8, 0xfe, 0x2b, 0xf8,
      0x93, 0x10, 0xd8, 0x8, 0xcd, 0x0, 0xca, 0x8, 0xaa, 0x0, 0x17, 0x5, 0xd,
      0x0, 0xa9, 0x3, 0xa5, 0xff, 0xd0, 0xc, 0x27, 0x8, 0xb0, 0x5, 0x73, 0x0,
      0xa8, 0xfa, 0xbf, 0x0, 0x5c, 0x0, 0x2b, 0x2, 0x13, 0xfd, 0x99, 0x7, 0xb8,
      0x2, 0x51, 0x2, 0xa1, 0x0, 0x65, 0x1, 0x6, 0x3, 0x6d, 0x3, 0x3f, 0xfd,
      0x74, 0xfd, 0x8e, 0x7, 0xb2, 0xfb, 0x49, 0xfc, 0xdb, 0xf8, 0x4, 0xf9,
      0x16, 0xf9, 0x91, 0xfe, 0x49, 0xfe, 0xe, 0xff, 0xb9, 0x1, 0xff, 0xfb,
      0x2b, 0xfb, 0xae, 0xfc, 0x1f, 0xff, 0x4c, 0xfe, 0x17, 0xfd, 0xf8, 0xfd,
      0xc5, 0xfb, 0xd7, 0x3, 0x5f, 0xfb, 0xe9, 0xfc, 0xb8, 0x0, 0xf4, 0xfa,
      0x3a, 0x2, 0xf, 0xfe, 0xa9, 0x2, 0x7d, 0xfd, 0xba, 0x2, 0x68, 0xfe, 0x16,
      0xfa, 0xf, 0x2, 0x14, 0x0, 0x1b, 0x1, 0xe5, 0xff, 0xe2, 0x1, 0xce, 0xfa,
      0x1f, 0x5, 0xd, 0x5, 0x5c, 0x1, 0xd, 0x7, 0x37, 0x3, 0x91, 0x4, 0x2a, 0x2,
      0xbb, 0x2, 0x5c, 0x1, 0x71, 0x5, 0x25, 0x0, 0xc9, 0xfc, 0x21, 0x5, 0xf2,
      0xfc, 0x5d, 0xfe, 0xad, 0x0, 0x48, 0x3, 0x14, 0x2, 0x74, 0x3, 0x9d, 0x1,
      0xe7, 0xfe, 0x13, 0xff, 0x74, 0x3, 0x3f, 0xfc, 0xa6, 0xff, 0xf7, 0x2,
      0x17, 0x6, 0x6e, 0x1, 0xc4, 0xfe, 0x66, 0xfe, 0x99, 0xfb, 0x74, 0xff,
      0xd9, 0xfa, 0x11, 0xfc, 0x7, 0x3, 0xd8, 0x4, 0x3c, 0xfe, 0xcd, 0xfb, 0xd5,
      0xf6, 0x5c, 0x0, 0xb2, 0x2, 0x58, 0xff, 0x61, 0xfe, 0x2e, 0x2, 0x76, 0xff,
      0xd8, 0x1, 0xc6, 0xfc, 0xcc, 0xfc, 0xfa, 0xfe, 0x42, 0xfe, 0xfd, 0x1,
      0x6d, 0x0, 0xf6, 0x2, 0x61, 0xff, 0x18, 0x0, 0xed, 0xfe, 0x12, 0xfa, 0xad,
      0x0, 0x3a, 0xff, 0x2e, 0x2, 0xe9, 0x0, 0xac, 0x0, 0x55, 0xfc, 0x9c, 0x2,
      0x6a, 0x4, 0x10, 0xfb, 0x64, 0x2, 0x88, 0x1, 0x43, 0x1, 0xa3, 0x2, 0x44,
      0x0, 0x8e, 0x1, 0xe0, 0x0, 0x49, 0x2, 0xdd, 0xfa, 0x1e, 0xfd, 0x35, 0xff,
      0xa3, 0xfd, 0xb4, 0xff, 0xbd, 0xfe, 0xe8, 0x0, 0xff, 0xfd, 0xf6, 0xff,
      0xb0, 0xf9, 0xed, 0xff, 0xa8, 0x4, 0xed, 0xff, 0xdd, 0xff, 0x5b, 0xff,
      0x6c, 0x2, 0xc9, 0xfe, 0x70, 0xfc, 0x90, 0xfb, 0x3, 0xfe, 0xe6, 0xfc,
      0xe9, 0xfd, 0xf2, 0xfd, 0xec, 0x0, 0xc4, 0x1, 0x13, 0xfc, 0xdb, 0xfd,
      0xe7, 0xf9, 0x4f, 0xfe, 0x2f, 0x0, 0xae, 0xff, 0x55, 0xff, 0x7b, 0xff,
      0x3c, 0xfe, 0xc0, 0xff, 0x1a, 0xfe, 0x2f, 0xfc, 0xe5, 0x0, 0xe4, 0xfe,
      0xa7, 0x1, 0x85, 0x1, 0x3a, 0x0, 0x4f, 0xfe, 0xe9, 0xfd, 0x12, 0xfe, 0xfc,
      0xfa, 0x8a, 0x1, 0x3b, 0xfe, 0xee, 0x1, 0x7e, 0x0, 0x1d, 0x0, 0x7d, 0xfd,
      0x9, 0x0, 0x51, 0x1, 0xa0, 0xfc, 0xcb, 0x1, 0x18, 0x1, 0x73, 0x2, 0x58,
      0x1, 0xfe, 0xff, 0x4b, 0x1, 0x7b, 0x1, 0xd7, 0xfe, 0x2, 0xfe, 0xe4, 0xfe,
      0x28, 0xfe, 0xa1, 0xff, 0x20, 0xff, 0x79, 0xff, 0x83, 0x0, 0xe9, 0xfe,
      0x4b, 0xff, 0x95, 0xfd, 0xa2, 0xff, 0x78, 0x2, 0xcf, 0xfe, 0xb, 0x0, 0xc8,
      0xff, 0xf4, 0x1, 0x8d, 0xff, 0x1a, 0xfd, 0xbf, 0xfd, 0x14, 0xfe, 0xa1,
      0xfe, 0xa4, 0xfd, 0xb, 0x1, 0x74, 0x0, 0x5c, 0x0, 0x3a, 0xfd, 0x23, 0xfe,
      0x99, 0xfd, 0x1a, 0xff, 0xf8, 0xff, 0xee, 0xff, 0xb1, 0x0, 0xd, 0x0, 0xed,
      0xfe, 0xed, 0xfe, 0xe3, 0xfe, 0x57, 0xfe, 0x37, 0x0, 0x22, 0xff, 0x6e,
      0x0, 0x8d, 0x0, 0x25, 0x0, 0xc9, 0xff, 0xcd, 0xfe, 0xbf, 0xfd, 0x81, 0xfd,
      0x54, 0x0, 0xe7, 0xff, 0x2, 0x1, 0x83, 0x0, 0xff, 0xff, 0xbd, 0xfe, 0x70,
      0xff, 0x91, 0xff, 0xe4, 0xfd, 0xc5, 0x0, 0x45, 0x2, 0x4e, 0x1, 0xcc, 0x0,
      0xdb, 0xff, 0x88, 0x0, 0x28, 0xff, 0x39, 0xfe, 0x22, 0xfd, 0x48, 0xff,
      0x83, 0x0, 0x9e, 0xff, 0x95, 0xff, 0x69, 0xff, 0x5d, 0xff, 0xaa, 0xfe,
      0x30, 0xff, 0x45, 0xfe, 0xbb, 0xff, 0x2a, 0x2, 0xfe, 0xff, 0xaf, 0xff,
      0xa4, 0xff, 0xdd, 0x0, 0xf9, 0xfe, 0xe7, 0xfd, 0x50, 0xfe, 0x58, 0xfe,
      0xc1, 0xfe, 0xe7, 0xfe, 0x65, 0xff, 0xbb, 0xff, 0xc3, 0xff, 0xc2, 0xfd,
      0x2f, 0xfe, 0xab, 0xfd, 0x22, 0x0, 0xf6, 0xff, 0xa0, 0x0, 0xeb, 0xff, 0xc,
      0x0, 0x7b, 0xff, 0x9a, 0xff, 0x9a, 0xfe, 0xe8, 0xfd, 0x2e, 0x1, 0x49,
      0xff, 0xe5, 0x0, 0x61, 0x0, 0xfd, 0xff, 0xb2, 0xff, 0xfd, 0xfe, 0x79,
      0xfe, 0xc0, 0xfd, 0xc4, 0x0, 0x50, 0x0, 0xe2, 0x0, 0xa4, 0x0, 0xaf, 0xff,
      0xff, 0xfe, 0xde, 0xff, 0xa4, 0xff, 0xc2, 0xfe, 0x40, 0x0, 0x8e, 0x0, 0xd,
      0x1, 0x26, 0x0, 0xd4, 0xff, 0x90, 0x0, 0x68, 0xff, 0xb8, 0xfe, 0x49, 0xfe,
      0x5, 0xff, 0xa4, 0xff, 0xb8, 0xff, 0xa1, 0xff, 0x64, 0xff, 0x99, 0xff,
      0xeb, 0xfe, 0xd0, 0xfe, 0xe, 0xff, 0x75, 0xff, 0x62, 0x1, 0xc6, 0xff,
      0x1d, 0x0, 0xca, 0xff, 0x9d, 0x0, 0x4f, 0xff, 0xd8, 0xfd, 0xb2, 0xfe,
      0x4a, 0xfe, 0x28, 0xff, 0x6b, 0xff, 0xdf, 0xff, 0xd1, 0xff, 0x9a, 0xff,
      0x2, 0xff, 0x98, 0xfe, 0xf6, 0xfe, 0x8f, 0xff, 0x47, 0x0, 0x6c, 0x0, 0x96,
      0x0, 0xf, 0x0, 0xeb, 0xff, 0x11, 0x0, 0x7d, 0xfe, 0xa4, 0xff, 0x3a, 0x0,
      0x8b, 0xff, 0x61, 0x0, 0x14, 0x0, 0x10, 0x0, 0xe2, 0xff, 0x75, 0xff, 0xbb,
      0xfe, 0x10, 0x0, 0x96, 0x0, 0xd, 0x1, 0x79, 0x0, 0x3d, 0x0, 0xac, 0xff,
      0x7c, 0xff, 0x74, 0x0, 0xfb, 0xfe, 0x73, 0x0, 0x57, 0x0, 0xbf, 0x0, 0x59,
      0x0, 0xd4, 0xff, 0xf8, 0xff, 0x6e, 0x0, 0xe5, 0xff, 0x61, 0xfe, 0x67,
      0xff, 0x22, 0xff, 0x2c, 0x0, 0xe0, 0xff, 0x61, 0xff, 0xeb, 0xff, 0xb7,
      0xff, 0x4c, 0x0, 0xae, 0xfe, 0xf, 0x0, 0x9e, 0xff, 0xf7, 0x0, 0xe9, 0xff,
      0xf0, 0xff, 0x36, 0x0, 0x67, 0x0, 0x77, 0x0, 0xce, 0xfd, 0xb2, 0xff, 0x46,
      0xff, 0x62, 0xff, 0xb3, 0xff, 0x91, 0xff, 0x1e, 0x0, 0x9b, 0xff, 0xe7,
      0xff, 0xa2, 0xfe, 0xff, 0xff, 0x18, 0x0, 0x52, 0x0, 0x84, 0x0, 0x31, 0x0,
      0x24, 0x0, 0x15, 0x0, 0xc2, 0x0, 0x68, 0xfe, 0x22, 0x0, 0x34, 0x0, 0x28,
      0xff, 0x6f, 0x0, 0x11, 0x0, 0x46, 0x0, 0xce, 0xff, 0x42, 0x0, 0xc, 0xff,
      0x15, 0x0, 0x16, 0x0, 0x60, 0x0, 0x91, 0x0, 0x5f, 0x0, 0xa, 0x0, 0xf7,
      0xff, 0xce, 0x0, 0x2c, 0xff, 0xb, 0x0, 0x88, 0xff, 0xe4, 0xff, 0x14, 0x0,
      0x26, 0x0, 0x1a, 0x0, 0x7a, 0x0, 0x4f, 0x0, 0x8d, 0xfe, 0x96, 0xff, 0x4c,
      0xff, 0xbc, 0xff, 0xd6, 0xff, 0xc, 0x0, 0x1a, 0x0, 0xec, 0xff, 0x77, 0x0,
      0xd6, 0xfe, 0x32, 0x0, 0xce, 0xff, 0x3f, 0x0, 0xf3, 0xff, 0x17, 0x0, 0xa,
      0x0, 0x47, 0x0, 0x28, 0x0, 0x53, 0xfe, 0xc1, 0xff, 0xac, 0xff, 0x82, 0xff,
      0xbc, 0xff, 0xbd, 0xff, 0xf, 0x0, 0x76, 0xff, 0x33, 0x0, 0xb8, 0xfe, 0x3b,
      0x0, 0x2c, 0x0, 0x44, 0x0, 0x33, 0x0, 0x50, 0x0, 0x3f, 0x0, 0x0, 0x0,
      0xaa, 0x0, 0x71, 0xfe, 0x55, 0x0, 0xeb, 0xff, 0x88, 0xff, 0x2b, 0x0, 0xf5,
      0xff, 0x5, 0x0, 0xb6, 0xff, 0x2f, 0x0, 0xfd, 0xfe, 0x6c, 0x0, 0x26, 0x0,
      0x6b, 0x0, 0x31, 0x0, 0x3f, 0x0, 0x12, 0x0, 0x0, 0x0, 0xae, 0x0, 0xda,
      0xfe, 0x73, 0x0, 0xcb, 0xff, 0xb4, 0xff, 0xc9, 0xff, 0x1d, 0x0, 0x3b, 0x0,
      0x14, 0x0, 0x12, 0x0, 0xe9, 0xfe, 0x2c, 0x0, 0xc5, 0xff, 0x2a, 0x0, 0xef,
      0xff, 0xb, 0x0, 0x2a, 0x0, 0xa3, 0xff, 0x56, 0x0, 0xe4, 0xfe, 0x21, 0x0,
      0xeb, 0xff, 0x15, 0x0, 0xe5, 0xff, 0x17, 0x0, 0x3e, 0x0, 0x1, 0x0, 0x28,
      0x0, 0x5d, 0xfe, 0xfe, 0xff, 0xae, 0xff, 0x99, 0xff, 0xd1, 0xff, 0xf1,
      0xff, 0x17, 0x0, 0x9c, 0xff, 0x2d, 0x0, 0xee, 0xfe, 0x9a, 0x0, 0x12, 0x0,
      0x32, 0x0, 0x11, 0x0, 0x1d, 0x0, 0x24, 0x0, 0x3, 0x0, 0x4c, 0x0, 0x82,
      0xfe, 0x40, 0x0, 0xed, 0xff, 0x8b, 0xff, 0xef, 0xff, 0xe8, 0xff, 0x7, 0x0,
      0xf1, 0xff, 0xfd, 0xff, 0xde, 0xfe, 0x71, 0x0, 0x4b, 0x0, 0x4e, 0x0, 0x10,
      0x0, 0x51, 0x0, 0xf5, 0xff, 0x60, 0x0, 0x3b, 0x0, 0xd0, 0xfe, 0x32, 0x0,
      0xfb, 0xff, 0xd1, 0xff, 0xd9, 0xff, 0x16, 0x0, 0x1, 0x0, 0x25, 0x0, 0xdb,
      0xff, 0xac, 0xfe, 0x9, 0x0, 0xf3, 0xff, 0x16, 0x0, 0x11, 0x0, 0x32, 0x0,
      0xf0, 0xff, 0x36, 0x0, 0x24, 0x0, 0xa8, 0xfe, 0x6a, 0x0, 0xde, 0xff, 0xcc,
      0xff, 0xf2, 0xff, 0x12, 0x0, 0xf1, 0xff, 0x5c, 0x0, 0xf0, 0xff, 0xb9,
      0xfe, 0x71, 0x0, 0xbb, 0xff, 0x3, 0x0, 0xf9, 0xff, 0xec, 0xff, 0xee, 0xff,
      0xf8, 0xff, 0x2d, 0x0, 0xde, 0xfe, 0x98, 0x0, 0x32, 0x0, 0x2c, 0x0, 0x10,
      0x0, 0x2e, 0x0, 0x3, 0x0, 0x64, 0x0, 0x31, 0x0, 0xbc, 0xfe, 0x37, 0x0,
      0x19, 0x0, 0xb2, 0xff, 0xef, 0xff, 0xff, 0xff, 0xef, 0xff, 0xfd, 0xff,
      0x25, 0x0, 0x35, 0xff, 0x5e, 0x0, 0x4c, 0x0, 0x3c, 0x0, 0x1d, 0x0, 0x2f,
      0x0, 0x1d, 0x0, 0x24, 0x0, 0x6c, 0x0, 0xf5, 0xfe, 0x20, 0x0, 0x1c, 0x0,
      0xa3, 0xff, 0xf4, 0xff, 0xf9, 0xff, 0x5a, 0x0, 0xf0, 0xff, 0x33, 0x0,
      0x35, 0xff, 0x59, 0x0, 0xf5, 0xff, 0x1c, 0x0, 0x22, 0x0, 0x6, 0x0, 0x45,
      0x0, 0xfb, 0xff, 0x54, 0x0, 0x0, 0xff, 0x61, 0x0, 0xfc, 0xff, 0xcb, 0xff,
      0x12, 0x0, 0x5, 0x0, 0x16, 0x0, 0x48, 0x0, 0x10, 0x0, 0xcc, 0xfe, 0x3d,
      0x0, 0x3d, 0x0, 0xe8, 0xff, 0x1a, 0x0, 0x1a, 0x0, 0xf7, 0xff, 0x59, 0x0,
      0x28, 0x0, 0xf9, 0xfe, 0x6d, 0x0, 0x23, 0x0, 0xde, 0xff, 0xff, 0xff, 0xd,
      0x0, 0x7, 0x0, 0x45, 0x0, 0xf5, 0xff, 0xdb, 0xfe, 0x20, 0x0, 0xee, 0xff,
      0x9a, 0xff, 0xf9, 0xff, 0xf4, 0xff, 0x9, 0x0, 0xf4, 0xff, 0xf6, 0xff,
      0xfa, 0xfe, 0x51, 0x0, 0x30, 0x0, 0xc1, 0xff, 0x1b, 0x0, 0x1e, 0x0, 0x12,
      0x0, 0x43, 0x0, 0xfa, 0xff, 0xe1, 0xfe, 0x34, 0x0, 0xfb, 0xff, 0x7f, 0xff,
      0xf1, 0xff, 0xd7, 0xff, 0xee, 0xff, 0x3, 0x0, 0xee, 0xff, 0x12, 0xff,
      0x41, 0x0, 0x29, 0x0, 0x2e, 0x0, 0x18, 0x0, 0x2, 0x0, 0xc0, 0xff, 0x55,
      0x0, 0x2e, 0x0, 0xf1, 0xfe, 0x14, 0x0, 0x2c, 0x0, 0xf6, 0xff, 0xf1, 0xff,
      0xf1, 0xff, 0xb9, 0xff, 0x5f, 0x0, 0x12, 0x0, 0x4d, 0xff, 0x39, 0x0, 0x11,
      0x0, 0x66, 0x0, 0x9, 0x0, 0xf7, 0xff, 0xb0, 0xff, 0x21, 0x0, 0x54, 0x0,
      0x4a, 0xff, 0x43, 0x0, 0x23, 0x0, 0x4d, 0x0, 0x1b, 0x0, 0xb, 0x0, 0xe5,
      0xff, 0x2c, 0x0, 0x44, 0x0, 0x33, 0xff, 0x4, 0x0, 0x38, 0x0, 0x2a, 0x0,
      0x11, 0x0, 0xef, 0xff, 0xf8, 0xff, 0x16, 0x0, 0x65, 0x0, 0x5e, 0xff, 0x52,
      0x0, 0x12, 0x0, 0x5d, 0x0, 0x1c, 0x0, 0xf4, 0xff, 0xc3, 0xff, 0x56, 0x0,
      0x46, 0x0, 0x19, 0xff, 0x28, 0x0, 0xe, 0x0, 0xd, 0x0, 0xa, 0x0, 0xf6,
      0xff, 0xca, 0xff, 0x2b, 0x0, 0x38, 0x0, 0x4d, 0xff, 0x45, 0x0, 0x21, 0x0,
      0x31, 0x0, 0xf8, 0xff, 0x5, 0x0, 0xe5, 0xff, 0x1d, 0x0, 0x60, 0x0, 0x30,
      0xff, 0x32, 0x0, 0xc7, 0xff, 0xf1, 0xff, 0xfc, 0xff, 0xed, 0xff, 0xe8,
      0xff, 0xf8, 0xff, 0x2e, 0x0, 0x41, 0xff, 0x26, 0x0, 0x2d, 0x0, 0x2, 0x0,
      0x3a, 0x0, 0x20, 0x0, 0x21, 0x0, 0x10, 0x0, 0x45, 0x0, 0x39, 0xff, 0x1a,
      0x0, 0x1d, 0x0, 0xcb, 0xff, 0x6, 0x0, 0xf5, 0xff, 0x20, 0x0, 0x1f, 0x0,
      0x21, 0x0, 0x44, 0xff, 0x12, 0x0, 0xff, 0xff, 0x1d, 0x0, 0xf, 0x0, 0xde,
      0xff, 0xfd, 0xff, 0x24, 0x0, 0x1f, 0x0, 0x47, 0xff, 0x21, 0x0, 0x3c, 0x0,
      0x1, 0x0, 0xe, 0x0, 0xfd, 0xff, 0x1d, 0x0, 0x40, 0x0, 0x0, 0x0, 0x3e,
      0xff, 0x4, 0x0, 0x4, 0x0, 0xda, 0xff, 0xda, 0xff, 0xd7, 0xff, 0x18, 0x0,
      0x5, 0x0, 0x9, 0x0, 0x72, 0xff, 0x31, 0x0, 0x22, 0x0, 0x7, 0x0, 0xf7,
      0xff, 0xed, 0xff, 0x22, 0x0, 0x5, 0x0, 0x10, 0x0, 0x28, 0xff, 0xf4, 0xff,
      0x12, 0x0, 0xc4, 0xff, 0xf7, 0xff, 0xf5, 0xff, 0x28, 0x0, 0xf7, 0xff, 0x3,
      0x0, 0x4e, 0xff, 0x1b, 0x0, 0xeb, 0xff, 0x15, 0x0, 0xf6, 0xff, 0xf3, 0xff,
      0x9, 0x0, 0xb, 0x0, 0x12, 0x0, 0x57, 0xff, 0x3e, 0x0, 0xfe, 0xff, 0xe3,
      0xff, 0xf9, 0xff, 0xf7, 0xff, 0xa, 0x0, 0x3, 0x0, 0xec, 0xff, 0x45, 0xff,
      0x26, 0x0, 0xc, 0x0, 0xf5, 0xff, 0x2, 0x0, 0x9, 0x0, 0x7, 0x0, 0x9, 0x0,
      0x10, 0x0, 0x5a, 0xff, 0x41, 0x0, 0xde, 0xff, 0xea, 0xff, 0xe7, 0xff,
      0xfa, 0xff, 0xf6, 0xff, 0xf9, 0xff, 0x6, 0x0, 0x55, 0xff, 0x35, 0x0, 0xf8,
      0xff, 0xf5, 0xff, 0xe, 0x0, 0x3, 0x0, 0x6, 0x0, 0xea, 0xff, 0x12, 0x0,
      0x54, 0xff, 0x1e, 0x0, 0x13, 0x0, 0xf8, 0xff, 0xb, 0x0, 0xfd, 0xff, 0x25,
      0x0, 0x7, 0x0, 0x22, 0x0, 0x5c, 0xff, 0x24, 0x0, 0xee, 0xff, 0xe4, 0xff,
      0xee, 0xff, 0xe5, 0xff, 0x13, 0x0, 0xe8, 0xff, 0x11, 0x0, 0x6d, 0xff,
      0x41, 0x0, 0x1c, 0x0, 0xd, 0x0, 0x1e, 0x0, 0x1, 0x0, 0x13, 0x0, 0xe, 0x0,
      0x3, 0x0, 0x50, 0xff, 0xc, 0x0, 0x9, 0x0, 0xc7, 0xff, 0xf7, 0xff, 0xea,
      0xff, 0x1a, 0x0, 0xff, 0xff, 0x4, 0x0, 0x7c, 0xff, 0x2a, 0x0, 0xfe, 0xff,
      0x9, 0x0, 0xfe, 0xff, 0xf0, 0xff, 0xa, 0x0, 0xf3, 0xff, 0xf, 0x0, 0x6b,
      0xff, 0x27, 0x0, 0x17, 0x0, 0x1, 0x0, 0x16, 0x0, 0xfe, 0xff, 0x19, 0x0,
      0xff, 0xff, 0xf8, 0xff, 0x62, 0xff, 0x4, 0x0, 0xfc, 0xff, 0xf9, 0xff,
      0xfb, 0xff, 0xf8, 0xff, 0x18, 0x0, 0xee, 0xff, 0x1d, 0x0, 0x7d, 0xff,
      0x30, 0x0, 0xf8, 0xff, 0xb, 0x0, 0x2, 0x0, 0xf9, 0xff, 0xd, 0x0, 0xf6,
      0xff, 0xb, 0x0, 0x65, 0xff, 0x1f, 0x0, 0x16, 0x0, 0x1, 0x0, 0x13, 0x0,
      0x2, 0x0, 0x12, 0x0, 0x3, 0x0, 0x8, 0x0, 0x82, 0xff, 0x1b, 0x0, 0x5, 0x0,
      0xfd, 0xff, 0xf5, 0xff, 0xf9, 0xff, 0x7, 0x0, 0xf8, 0xff, 0xe, 0x0, 0x88,
      0xff, 0x1a, 0x0, 0xf9, 0xff, 0x1, 0x0, 0x7, 0x0, 0xf8, 0xff, 0x4, 0x0,
      0xe4, 0xff, 0xff, 0xff, 0x6f, 0xff, 0xfc, 0xff, 0x16, 0x0, 0x1d, 0x0,
      0x22, 0x0, 0x8, 0x0, 0x17, 0x0, 0x4, 0x0, 0x1c, 0x0, 0x7d, 0xff, 0x0, 0x0,
      0xee, 0xff, 0xe4, 0xff, 0xf6, 0xff, 0xe7, 0xff, 0xa, 0x0, 0xea, 0xff, 0xf,
      0x0, 0x95, 0xff, 0x20, 0x0, 0x0, 0x0, 0x9, 0x0, 0x1a, 0x0, 0xff, 0xff,
      0x5, 0x0, 0xfb, 0xff, 0x6, 0x0, 0x6e, 0xff, 0xe4, 0xff, 0x26, 0x0, 0xff,
      0xff, 0x20, 0x0, 0x5, 0x0, 0x15, 0x0, 0x7, 0x0, 0x12, 0x0, 0x8e, 0xff,
      0xeb, 0xff, 0xf7, 0xff, 0x0, 0x0, 0x7, 0x0, 0xe3, 0xff, 0x8, 0x0, 0xec,
      0xff, 0x17, 0x0, 0x8c, 0xff, 0xf7, 0xff, 0xf9, 0xff, 0xf3, 0xff, 0x0, 0x0,
      0xf6, 0xff, 0x12, 0x0, 0x0, 0x0, 0x3, 0x0, 0x67, 0xff, 0xbe, 0xff, 0x1a,
      0x0, 0xef, 0xff, 0x1, 0x0, 0x5, 0x0, 0x13, 0x0, 0xf1, 0xff, 0x18, 0x0,
      0xa4, 0xff, 0x9, 0x0, 0xeb, 0xff, 0x1f, 0x0, 0x15, 0x0, 0xe7, 0xff, 0xfa,
      0xff, 0xf5, 0xff, 0x12, 0x0, 0x90, 0xff, 0x11, 0x0, 0xeb, 0xff, 0xe6,
      0xff, 0x3, 0x0, 0xee, 0xff, 0xf4, 0xff, 0xed, 0xff, 0xfa, 0xff, 0x89,
      0xff, 0xfa, 0xff, 0x21, 0x0, 0xff, 0xff, 0x10, 0x0, 0x10, 0x0, 0xff, 0xff,
      0xec, 0xff, 0x10, 0x0, 0xab, 0xff, 0x16, 0x0, 0xe7, 0xff, 0xf, 0x0, 0x11,
      0x0, 0xe6, 0xff, 0xfa, 0xff, 0xf2, 0xff, 0xd, 0x0, 0x9f, 0xff, 0x19, 0x0,
      0xe7, 0xff, 0x0, 0x0, 0xc, 0x0, 0xf1, 0xff, 0xef, 0xff, 0xf7, 0xff, 0xa,
      0x0, 0x94, 0xff, 0x16, 0x0, 0x6, 0x0, 0xf9, 0xff, 0x1a, 0x0, 0x0, 0x0,
      0xa, 0x0, 0x7, 0x0, 0xf7, 0xff, 0xb6, 0xff, 0x2a, 0x0, 0xea, 0xff, 0x9,
      0x0, 0x7, 0x0, 0xee, 0xff, 0xa, 0x0, 0x9, 0x0, 0x6, 0x0, 0xaa, 0xff, 0x3d,
      0x0, 0x6, 0x0, 0x6, 0x0, 0x7, 0x0, 0xa, 0x0, 0xf4, 0xff, 0x4, 0x0, 0xf6,
      0xff, 0x96, 0xff, 0x2b, 0x0, 0x1, 0x0, 0xeb, 0xff, 0x12, 0x0, 0x1, 0x0,
      0xa, 0x0, 0x0, 0x0, 0xdf, 0xff, 0xc1, 0xff, 0x3e, 0x0, 0xd7, 0xff, 0xf5,
      0xff, 0xef, 0xff, 0xf0, 0xff, 0x6, 0x0, 0x8, 0x0, 0xfe, 0xff, 0xb6, 0xff,
      0x3a, 0x0, 0xf2, 0xff, 0xf8, 0xff, 0xf7, 0xff, 0x6, 0x0, 0xf2, 0xff, 0xf6,
      0xff, 0xf2, 0xff, 0xb7, 0xff, 0x28, 0x0, 0x3, 0x0, 0x17, 0x0, 0x11, 0x0,
      0x9, 0x0, 0xa, 0x0, 0x3, 0x0, 0xfa, 0xff, 0xcf, 0xff, 0x2e, 0x0, 0xe9,
      0xff, 0xfc, 0xff, 0xe8, 0xff, 0xf7, 0xff, 0x4, 0x0, 0x2, 0x0, 0x14, 0x0,
      0xce, 0xff, 0x3a, 0x0, 0x3, 0x0, 0xa, 0x0, 0x7, 0x0, 0xe, 0x0, 0xfa, 0xff,
      0xf6, 0xff, 0x4, 0x0, 0xc7, 0xff, 0x28, 0x0, 0xfb, 0xff, 0x10, 0x0, 0xa,
      0x0, 0x6, 0x0, 0xe, 0x0, 0x2, 0x0, 0x4, 0x0, 0xcf, 0xff, 0x24, 0x0, 0xde,
      0xff, 0xf6, 0xff, 0xe1, 0xff, 0xf8, 0xff, 0xf8, 0xff, 0xf1, 0xff, 0x1b,
      0x0, 0xcc, 0xff, 0x3a, 0x0, 0xfc, 0xff, 0xc, 0x0, 0x6, 0x0, 0xe, 0x0,
      0xfb, 0xff, 0xf9, 0xff, 0x1, 0x0, 0xbf, 0xff, 0x1b, 0x0, 0xee, 0xff, 0xfc,
      0xff, 0x0, 0x0, 0xfa, 0xff, 0xc, 0x0, 0xf6, 0xff, 0xc, 0x0, 0xd5, 0xff,
      0x2a, 0x0, 0xe6, 0xff, 0xf9, 0xff, 0xf0, 0xff, 0xfd, 0xff, 0xf8, 0xff,
      0xf0, 0xff, 0x20, 0x0, 0xc9, 0xff, 0x2f, 0x0, 0x1, 0x0, 0xfc, 0xff, 0x14,
      0x0, 0xa, 0x0, 0x4, 0x0, 0xf5, 0xff, 0x2, 0x0, 0xcd, 0xff, 0x17, 0x0,
      0xfa, 0xff, 0xfe, 0xff, 0x7, 0x0, 0xf9, 0xff, 0xc, 0x0, 0xf9, 0xff, 0x1d,
      0x0, 0xd5, 0xff, 0x28, 0x0, 0xee, 0xff, 0xf1, 0xff, 0xef, 0xff, 0xfd,
      0xff, 0xff, 0xff, 0xfb, 0xff, 0x18, 0x0, 0xcb, 0xff, 0x25, 0x0, 0x4, 0x0,
      0x6, 0x0, 0x11, 0x0, 0x2, 0x0, 0xa, 0x0, 0xfe, 0xff, 0x2, 0x0, 0xd1, 0xff,
      0x13, 0x0, 0xf9, 0xff, 0x0, 0x0, 0xfe, 0xff, 0xf2, 0xff, 0xf, 0x0, 0x1,
      0x0, 0x1c, 0x0, 0xcd, 0xff, 0x1b, 0x0, 0xf2, 0xff, 0xf0, 0xff, 0xf2, 0xff,
      0xfc, 0xff, 0x2, 0x0, 0xf5, 0xff, 0x12, 0x0, 0xcf, 0xff, 0x1e, 0x0, 0x9,
      0x0, 0x11, 0x0, 0x15, 0x0, 0x3, 0x0, 0x10, 0x0, 0x1, 0x0, 0x7, 0x0, 0xd4,
      0xff, 0x5, 0x0, 0xf6, 0xff, 0xf8, 0xff, 0xf0, 0xff, 0xf1, 0xff, 0xf, 0x0,
      0xff, 0xff, 0x1e, 0x0, 0xda, 0xff, 0x19, 0x0, 0xfa, 0xff, 0x4, 0x0, 0xf5,
      0xff, 0x2, 0x0, 0x3, 0x0, 0xff, 0xff, 0xf, 0x0, 0xd2, 0xff, 0x13, 0x0,
      0x7, 0x0, 0x10, 0x0, 0xb, 0x0, 0xfe, 0xff, 0x14, 0x0, 0x7, 0x0, 0x4, 0x0,
      0xd6, 0xff, 0x1, 0x0, 0xf7, 0xff, 0xfb, 0xff, 0xea, 0xff, 0xf2, 0xff, 0xb,
      0x0, 0x0, 0x0, 0x1e, 0x0, 0xd5, 0xff, 0x19, 0x0, 0xfa, 0xff, 0x2, 0x0,
      0xf9, 0xff, 0x5, 0x0, 0x6, 0x0, 0x3, 0x0, 0x4, 0x0, 0xcc, 0xff, 0x7, 0x0,
      0x0, 0x0, 0x9, 0x0, 0x6, 0x0, 0xfc, 0xff, 0x10, 0x0, 0x2, 0x0, 0x6, 0x0,
      0xd7, 0xff, 0x2, 0x0, 0xf7, 0xff, 0xfb, 0xff, 0xeb, 0xff, 0xf8, 0xff, 0x7,
      0x0, 0x0, 0x0, 0x19, 0x0, 0xd2, 0xff, 0x12, 0x0, 0xfe, 0xff, 0xfd, 0xff,
      0xff, 0xff, 0x6, 0x0, 0x8, 0x0, 0x1, 0x0, 0x1, 0x0, 0xd4, 0xff, 0x5, 0x0,
      0x5, 0x0, 0xc, 0x0, 0x9, 0x0, 0xfb, 0xff, 0xf, 0x0, 0x5, 0x0, 0xe, 0x0,
      0xd6, 0xff, 0x3, 0x0, 0xf6, 0xff, 0xf5, 0xff, 0xe8, 0xff, 0xf8, 0xff, 0x5,
      0x0, 0x3, 0x0, 0x13, 0x0, 0xd3, 0xff, 0xf, 0x0, 0x2, 0x0, 0x7, 0x0, 0x1,
      0x0, 0x8, 0x0, 0x7, 0x0, 0x4, 0x0, 0xfe, 0xff, 0xd4, 0xff, 0x0, 0x0, 0x0,
      0x0, 0x7, 0x0, 0xff, 0xff, 0xfa, 0xff, 0xe, 0x0, 0x7, 0x0, 0xa, 0x0, 0xd8,
      0xff, 0x2, 0x0, 0xf6, 0xff, 0xf4, 0xff, 0xe8, 0xff, 0xfb, 0xff, 0x3, 0x0,
      0x1, 0x0, 0xe, 0x0, 0xd5, 0xff, 0xc, 0x0, 0x6, 0x0, 0xa, 0x0, 0x8, 0x0,
      0x8, 0x0, 0x9, 0x0, 0x5, 0x0, 0xfb, 0xff, 0xd5, 0xff, 0xf9, 0xff, 0xfe,
      0xff, 0x1, 0x0, 0xfb, 0xff, 0xf8, 0xff, 0xa, 0x0, 0x3, 0x0, 0xb, 0x0,
      0xda, 0xff, 0x5, 0x0, 0xf8, 0xff, 0xfa, 0xff, 0xf0, 0xff, 0x1, 0x0, 0xff,
      0xff, 0x0, 0x0, 0x9, 0x0, 0xd2, 0xff, 0x8, 0x0, 0x4, 0x0, 0x8, 0x0, 0x9,
      0x0, 0x7, 0x0, 0x9, 0x0, 0x3, 0x0, 0xf9, 0xff, 0xd7, 0xff, 0xfb, 0xff,
      0xf9, 0xff, 0x0, 0x0, 0xf8, 0xff, 0xf9, 0xff, 0x5, 0x0, 0x1, 0x0, 0xb,
      0x0, 0xd8, 0xff, 0x5, 0x0, 0xf7, 0xff, 0xf7, 0xff, 0xf4, 0xff, 0x2, 0x0,
      0xff, 0xff, 0xff, 0xff, 0x1, 0x0, 0xd3, 0xff, 0x3, 0x0, 0x5, 0x0, 0x7,
      0x0, 0xb, 0x0, 0x5, 0x0, 0x8, 0x0, 0x1, 0x0, 0xf9, 0xff, 0xda, 0xff, 0xf9,
      0xff, 0xfb, 0xff, 0xfe, 0xff, 0xf6, 0xff, 0xf9, 0xff, 0x3, 0x0, 0x0, 0x0,
      0xa, 0x0, 0xd8, 0xff, 0x4, 0x0, 0xfb, 0xff, 0xf9, 0xff, 0xf8, 0xff, 0x4,
      0x0, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xd6, 0xff, 0x1, 0x0, 0x5, 0x0,
      0x9, 0x0, 0xb, 0x0, 0x2, 0x0, 0x7, 0x0, 0x2, 0x0, 0xfa, 0xff, 0xda, 0xff,
      0xf9, 0xff, 0xf8, 0xff, 0xfa, 0xff, 0xf3, 0xff, 0xf9, 0xff, 0x1, 0x0, 0x0,
      0x0, 0x7, 0x0, 0xd9, 0xff, 0x5, 0x0, 0xfd, 0xff, 0xfe, 0xff, 0xfe, 0xff,
      0x6, 0x0, 0xff, 0xff, 0xff, 0xff, 0xf9, 0xff, 0xd7, 0xff, 0xfc, 0xff, 0x2,
      0x0, 0x6, 0x0, 0x8, 0x0, 0xff, 0xff, 0x7, 0x0, 0x2, 0x0, 0xfa, 0xff, 0xdc,
      0xff, 0xfa, 0xff, 0xf8, 0xff, 0xf9, 0xff, 0xf2, 0xff, 0xfb, 0xff, 0xfe,
      0xff, 0xff, 0xff, 0x5, 0x0, 0xd8, 0xff, 0x6, 0x0, 0x1, 0x0, 0x1, 0x0, 0x1,
      0x0, 0x6, 0x0, 0x1, 0x0, 0x0, 0x0, 0xf7, 0xff, 0xd9, 0xff, 0xfc, 0xff,
      0x2, 0x0, 0x5, 0x0, 0x5, 0x0, 0xfd, 0xff, 0x6, 0x0, 0x1, 0x0, 0xfd, 0xff,
      0xdd, 0xff, 0xfe, 0xff, 0xf9, 0xff, 0xfa, 0xff, 0xf3, 0xff, 0xfc, 0xff,
      0xfe, 0xff, 0xff, 0xff, 0x4, 0x0, 0xd8, 0xff, 0x5, 0x0, 0x1, 0x0, 0x2,
      0x0, 0x5, 0x0, 0x5, 0x0, 0x2, 0x0, 0x0, 0x0, 0xf6, 0xff, 0xda, 0xff, 0xfd,
      0xff, 0x0, 0x0, 0x4, 0x0, 0x2, 0x0, 0xfb, 0xff, 0x4, 0x0, 0x0, 0x0, 0x0,
      0x0, 0xdd, 0xff, 0xff, 0xff, 0xf9, 0xff, 0xf9, 0xff, 0xf4, 0xff, 0xfd,
      0xff, 0xfe, 0xff, 0xff, 0xff, 0x1, 0x0, 0xd9, 0xff, 0x5, 0x0, 0x2, 0x0,
      0x4, 0x0, 0x7, 0x0, 0x4, 0x0, 0x3, 0x0, 0x0, 0x0, 0xf7, 0xff, 0xdc, 0xff,
      0xfc, 0xff, 0xfe, 0xff, 0x1, 0x0, 0xff, 0xff, 0xfb, 0xff, 0x3, 0x0, 0x0,
      0x0, 0x2, 0x0, 0xde, 0xff, 0x2, 0x0, 0xfa, 0xff, 0xfa, 0xff, 0xf7, 0xff,
      0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x0, 0x0, 0xda, 0xff, 0x3, 0x0, 0x2,
      0x0, 0x4, 0x0, 0x8, 0x0, 0x3, 0x0, 0x3, 0x0, 0x0, 0x0, 0xf8, 0xff, 0xde,
      0xff, 0xfc, 0xff, 0xfc, 0xff, 0xff, 0xff, 0xfc, 0xff, 0xfa, 0xff, 0x1,
      0x0, 0x0, 0x0, 0x3, 0x0, 0xde, 0xff, 0x3, 0x0, 0xfb, 0xff, 0xfc, 0xff,
      0xfa, 0xff, 0x1, 0x0, 0xfd, 0xff, 0xfe, 0xff, 0xfe, 0xff, 0xdb, 0xff, 0x2,
      0x0, 0x2, 0x0, 0x4, 0x0, 0x8, 0x0, 0x1, 0x0, 0x4, 0x0, 0x0, 0x0, 0xfa,
      0xff, 0xdf, 0xff, 0xfd, 0xff, 0xfb, 0xff, 0xfe, 0xff, 0xfa, 0xff, 0xfa,
      0xff, 0x0, 0x0, 0xfe, 0xff, 0x4, 0x0, 0xde, 0xff, 0x4, 0x0, 0xfd, 0xff,
      0xfd, 0xff, 0xfd, 0xff, 0x2, 0x0, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff,
      0xdd, 0xff, 0x1, 0x0, 0x2, 0x0, 0x5, 0x0, 0x7, 0x0, 0x0, 0x0, 0x4, 0x0,
      0xff, 0xff, 0xfc, 0xff, 0xe1, 0xff, 0xff, 0xff, 0xfa, 0xff, 0xfd, 0xff,
      0xf9, 0xff, 0xfb, 0xff, 0xff, 0xff, 0xfe, 0xff, 0x4, 0x0, 0xdf, 0xff, 0x5,
      0x0, 0xff, 0xff, 0xff, 0xff, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0xfe, 0xff,
      0xfb, 0xff, 0xdf, 0xff, 0x0, 0x0, 0x0, 0x0, 0x4, 0x0, 0x5, 0x0, 0xfe,
      0xff, 0x4, 0x0, 0xff, 0xff, 0xfe, 0xff, 0xe2, 0xff, 0x0, 0x0, 0xfa, 0xff,
      0xfc, 0xff, 0xf8, 0xff, 0xfc, 0xff, 0xff, 0xff, 0xfe, 0xff, 0x3, 0x0,
      0xdf, 0xff, 0x5, 0x0, 0xff, 0xff, 0x0, 0x0, 0x2, 0x0, 0x2, 0x0, 0x1, 0x0,
      0xfe, 0xff, 0xfb, 0xff, 0xdf, 0xff, 0x0, 0x0, 0xff, 0xff, 0x2, 0x0, 0x2,
      0x0, 0xfd, 0xff, 0x4, 0x0, 0xff, 0xff, 0xff, 0xff, 0xe3, 0xff, 0x2, 0x0,
      0xfb, 0xff, 0xfc, 0xff, 0xf9, 0xff, 0xfd, 0xff, 0xfe, 0xff, 0xfe, 0xff,
      0x1, 0x0, 0xdf, 0xff, 0x4, 0x0, 0x0, 0x0, 0x1, 0x0, 0x3, 0x0, 0x3, 0x0,
      0x2, 0x0, 0xff, 0xff, 0xfb, 0xff, 0xe1, 0xff, 0x0, 0x0, 0xfe, 0xff, 0x1,
      0x0, 0x0, 0x0, 0xfc, 0xff, 0x3, 0x0, 0xff, 0xff, 0x0, 0x0, 0xe3, 0xff,
      0x3, 0x0, 0xfb, 0xff, 0xfc, 0xff, 0xfa, 0xff, 0xfe, 0xff, 0xfe, 0xff,
      0xfe, 0xff, 0x0, 0x0, 0xe0, 0xff, 0x4, 0x0, 0x0, 0x0, 0x2, 0x0, 0x4, 0x0,
      0x2, 0x0, 0x2, 0x0, 0xff, 0xff, 0xfb, 0xff, 0xe2, 0xff, 0xff, 0xff, 0xfd,
      0xff, 0xff, 0xff, 0xfe, 0xff, 0xfc, 0xff, 0x1, 0x0, 0xff, 0xff, 0x2, 0x0,
      0xe3, 0xff, 0x3, 0x0, 0xfc, 0xff, 0xfd, 0xff, 0xfb, 0xff, 0x0, 0x0, 0xfe,
      0xff, 0xfe, 0xff, 0xff, 0xff, 0xe1, 0xff, 0x3, 0x0, 0x1, 0x0, 0x2, 0x0,
      0x4, 0x0, 0x1, 0x0, 0x3, 0x0, 0xff, 0xff, 0xfc, 0xff, 0xe4, 0xff, 0x0,
      0x0, 0xfc, 0xff, 0xfe, 0xff, 0xfc, 0xff, 0xfc, 0xff, 0x1, 0x0, 0xff, 0xff,
      0x2, 0x0, 0xe4, 0xff, 0x5, 0x0, 0xfd, 0xff, 0xfd, 0xff, 0xfd, 0xff, 0x1,
      0x0, 0xff, 0xff, 0xfe, 0xff, 0xfe, 0xff, 0xe2, 0xff, 0x2, 0x0, 0x0, 0x0,
      0x2, 0x0, 0x4, 0x0, 0x0, 0x0, 0x3, 0x0, 0xff, 0xff, 0xfd, 0xff, 0xe5,
      0xff, 0x1, 0x0, 0xfc, 0xff, 0xfd, 0xff, 0xfb, 0xff, 0xfd, 0xff, 0x0, 0x0,
      0xff, 0xff, 0x2, 0x0, 0xe4, 0xff, 0x5, 0x0, 0xfe, 0xff, 0xfe, 0xff, 0xff,
      0xff, 0x1, 0x0, 0x0, 0x0, 0xfe, 0xff, 0xfd, 0xff, 0xe2, 0xff, 0x1, 0x0,
      0xff, 0xff, 0x1, 0x0, 0x3, 0x0, 0xff, 0xff, 0x4, 0x0, 0xff, 0xff, 0xfe,
      0xff, 0xe5, 0xff, 0x1, 0x0, 0xfc, 0xff, 0xfc, 0xff, 0xfb, 0xff, 0xfd,
      0xff, 0x0, 0x0, 0xfe, 0xff, 0x1, 0x0, 0xe3, 0xff, 0x4, 0x0, 0xff, 0xff,
      0xff, 0xff, 0x0, 0x0, 0x2, 0x0, 0x1, 0x0, 0xff, 0xff, 0xfd, 0xff, 0xe2,
      0xff, 0x1, 0x0, 0xff, 0xff, 0x0, 0x0, 0x1, 0x0, 0xfe, 0xff, 0x2, 0x0,
      0xff, 0xff, 0xff, 0xff, 0xe3, 0xff, 0x2, 0x0, 0xfc, 0xff, 0xfd, 0xff,
      0xfb, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0x1, 0x0, 0xe0, 0xff, 0x3,
      0x0, 0xff, 0xff, 0x0, 0x0, 0x1, 0x0, 0x1, 0x0, 0x1, 0x0, 0xff, 0xff, 0xfd,
      0xff, 0xdf, 0xff, 0x0, 0x0, 0xfe, 0xff, 0x0, 0x0, 0x0, 0x0, 0xfe, 0xff,
      0x2, 0x0, 0xff, 0xff, 0x0, 0x0, 0xe0, 0xff, 0x2, 0x0, 0xfd, 0xff, 0xfe,
      0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x0, 0x0, 0xde,
      0xff, 0x2, 0x0, 0xff, 0xff, 0x0, 0x0, 0x1, 0x0, 0x0, 0x0, 0x1, 0x0, 0x0,
      0x0, 0xfe, 0xff, 0xe0, 0xff, 0x0, 0x0, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff,
      0xfe, 0xff, 0x1, 0x0, 0x0, 0x0, 0x1, 0x0, 0xe1, 0xff, 0x1, 0x0, 0xfe,
      0xff, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0x0, 0x0, 0xff, 0xff, 0x0, 0x0,
      0xe2, 0xff, 0x0, 0x0, 0xff, 0xff, 0x0, 0x0, 0x0, 0x0, 0xff, 0xff, 0x1,
      0x0, 0xff, 0xff, 0xff, 0xff, 0xe3, 0xff, 0x0, 0x0, 0xfe, 0xff, 0xff, 0xff,
      0xff, 0xff, 0xff, 0xff, 0x1, 0x0, 0x0, 0x0, 0x0, 0x0, 0xe6, 0xff, 0x0,
      0x0, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x0, 0x0, 0xff, 0xff,
      0x0, 0x0, 0xe8, 0xff, 0x0, 0x0, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
      0xff, 0x1, 0x0, 0x0, 0x0, 0x0, 0x0, 0xec, 0xff, 0x0, 0x0, 0xff, 0xff,
      0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x1, 0x0, 0x0, 0x0, 0x0, 0x0, 0xee,
      0xff, 0x0, 0x0, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x0, 0x0,
      0x0, 0x0, 0x0, 0x0}},
    {"WAV/Subject_002/SH/sh_hrir_order_3.wav",
     {0x52, 0x49, 0x46, 0x46, 0x24, 0x20, 0x0, 0x0, 0x57, 0x41, 0x56, 0x45,
      0x66, 0x6d, 0x74, 0x20, 0x10, 0x0, 0x0, 0x0, 0x1, 0x0, 0x10, 0x0, 0x80,
      0xbb, 0x0, 0x0, 0x0, 0x70, 0x17, 0x0, 0x20, 0x0, 0x10, 0x0, 0x64, 0x61,
      0x74, 0x61, 0x0, 0x20, 0x0, 0x0, 0xfe, 0xff, 0x4, 0x0, 0xff, 0xff, 0xff,
      0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xff,
      0xff, 0x0, 0x0, 0xff, 0xff, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0xff, 0xff,
      0xf3, 0xff, 0x18, 0x0, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 0xff,
      0xfc, 0xff, 0xfd, 0xff, 0xf9, 0xff, 0xff, 0xff, 0x0, 0x0, 0xff, 0xff, 0x0,
      0x0, 0x1, 0x0, 0x0, 0x0, 0xff, 0xff, 0xfd, 0xff, 0xa, 0x0, 0x0, 0x0, 0x0,
      0x0, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xfb, 0xff, 0xfe,
      0xff, 0x0, 0x0, 0xff, 0xff, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0xff, 0xff, 0x2,
      0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0xff, 0xff, 0xff, 0xff, 0x0,
      0x0, 0xff, 0xff, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0xff, 0xff, 0xff, 0xff,
      0x0, 0x0, 0x0, 0x0, 0xff, 0xff, 0x2, 0x0, 0x1, 0x0, 0x0, 0x0, 0xff, 0xff,
      0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
      0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x0, 0x0, 0x6, 0x0, 0xf3, 0xff,
      0x0, 0x0, 0xff, 0xff, 0xff, 0xff, 0x0, 0x0, 0x2, 0x0, 0x2, 0x0, 0x5, 0x0,
      0x1, 0x0, 0x0, 0x0, 0x1, 0x0, 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0x0,
      0x0, 0x0, 0x0, 0xf1, 0xff, 0xfd, 0xff, 0x0, 0x0, 0x0, 0x0, 0xfc, 0xff,
      0x2, 0x0, 0xff, 0xff, 0x8, 0x0, 0x2, 0x0, 0xfe, 0xff, 0xff, 0xff, 0xff,
      0xff, 0xfd, 0xff, 0x2, 0x0, 0x0, 0x0, 0xa, 0x0, 0xef, 0xff, 0x4, 0x0,
      0xfe, 0xff, 0xfd, 0xff, 0x8, 0x0, 0x8, 0x0, 0x4, 0x0, 0x7, 0x0, 0x2, 0x0,
      0x0, 0x0, 0x6, 0x0, 0xff, 0xff, 0xff, 0xff, 0xf9, 0xff, 0x1, 0x0, 0xde,
      0xff, 0x7b, 0xff, 0xf9, 0xff, 0x2, 0x0, 0x3, 0x0, 0xf7, 0xff, 0x41, 0x0,
      0xff, 0xff, 0x79, 0x0, 0x4c, 0x0, 0xfd, 0xff, 0x38, 0x0, 0xff, 0xff, 0xfb,
      0xff, 0x6, 0x0, 0xff, 0xff, 0xb2, 0xff, 0xd3, 0xfe, 0x9, 0x0, 0x0, 0x0,
      0xff, 0xff, 0xe, 0x0, 0x9c, 0x0, 0x5, 0x0, 0xe, 0x1, 0xa3, 0x0, 0x1, 0x0,
      0x82, 0x0, 0xfc, 0xff, 0xff, 0xff, 0xf8, 0xff, 0x2, 0x0, 0x51, 0x0, 0xf0,
      0x0, 0xfa, 0xff, 0xfe, 0xff, 0x1, 0x0, 0xfa, 0xff, 0x6f, 0xff, 0x7, 0x0,
      0x1, 0xff, 0x42, 0xff, 0x3, 0x0, 0x6f, 0xff, 0x1, 0x0, 0xfd, 0xff, 0xff,
      0xff, 0x0, 0x0, 0x75, 0x1, 0x13, 0x5, 0xf7, 0xff, 0xc, 0x0, 0xc, 0x0,
      0xf3, 0xff, 0x1, 0xfd, 0xf8, 0xff, 0xd5, 0xfa, 0x42, 0xfc, 0xf7, 0xff,
      0x13, 0xfd, 0xfc, 0xff, 0xf5, 0xff, 0xf, 0x0, 0xfe, 0xff, 0x7b, 0xfe,
      0x93, 0xfc, 0x1d, 0x0, 0xfb, 0xff, 0xfe, 0xff, 0x2f, 0x0, 0x5e, 0x1, 0x19,
      0x0, 0x21, 0x2, 0xb0, 0x0, 0x10, 0x0, 0xad, 0x0, 0x1, 0x0, 0x2, 0x0, 0xd5,
      0xff, 0x1, 0x0, 0x3f, 0xfc, 0xb4, 0xf5, 0xfa, 0xff, 0xa5, 0xff, 0x7e,
      0xff, 0xf4, 0xff, 0xa2, 0x4, 0x14, 0x0, 0xb6, 0x8, 0x40, 0x5, 0xd, 0x0,
      0xb3, 0x3, 0xf9, 0xff, 0x1d, 0x0, 0x13, 0x0, 0x3d, 0x0, 0xbb, 0x3, 0xc0,
      0x7, 0xc2, 0xfe, 0x8d, 0xff, 0x61, 0xff, 0x50, 0xfe, 0x99, 0xfc, 0x7, 0x0,
      0x6d, 0xfb, 0x9e, 0xff, 0xfe, 0xff, 0xf0, 0xfe, 0x28, 0x0, 0x59, 0x0, 0x1,
      0x1, 0x47, 0x0, 0xcd, 0x4, 0x66, 0x8, 0xf8, 0xfe, 0xf8, 0x0, 0x60, 0x1,
      0xb1, 0xfe, 0x9d, 0xfe, 0x5, 0xff, 0xf2, 0xfb, 0x4c, 0x0, 0x2e, 0xff,
      0x47, 0x1, 0x71, 0x0, 0x95, 0xff, 0x36, 0x0, 0x5e, 0xff, 0x15, 0xff, 0xd8,
      0xfb, 0x77, 0x5, 0x4f, 0x1, 0xd9, 0x1, 0xb5, 0x7, 0xf9, 0x4, 0xd, 0x0,
      0x59, 0x3, 0x5c, 0x0, 0xd, 0x0, 0x89, 0x3, 0xaa, 0xff, 0x98, 0xfe, 0x91,
      0xfa, 0x20, 0xff, 0xbd, 0xfa, 0xf9, 0xf3, 0x0, 0x2, 0x2d, 0x0, 0x67, 0x0,
      0x54, 0x2, 0xaa, 0x2, 0xf6, 0x1, 0x4f, 0x9, 0x47, 0x4, 0xa0, 0x1, 0x6d,
      0x0, 0xec, 0xfd, 0x68, 0xff, 0x78, 0x1, 0xc2, 0xff, 0xfc, 0xf7, 0x41,
      0xef, 0x68, 0xf9, 0x8, 0xff, 0x21, 0xfe, 0x5d, 0xf6, 0x5d, 0x2, 0xd6,
      0xff, 0x8e, 0xb, 0xa0, 0x4, 0xe0, 0xff, 0xe, 0xff, 0x64, 0xff, 0xb5, 0x3,
      0x7d, 0x8, 0x36, 0x1, 0xf0, 0x1, 0x9c, 0xb, 0x12, 0xfe, 0xe8, 0xfc, 0x14,
      0xfc, 0x8c, 0xfe, 0x6f, 0xfb, 0xdd, 0xff, 0x25, 0xef, 0xf, 0xf1, 0xcf,
      0xff, 0x68, 0xf9, 0x51, 0x4, 0x1, 0x2, 0x53, 0xfa, 0xad, 0x1, 0x74, 0xff,
      0xbf, 0x6, 0xdd, 0xff, 0x56, 0x0, 0x70, 0x2, 0x3d, 0x0, 0xdd, 0xfa, 0x4c,
      0xff, 0x2f, 0xf3, 0xac, 0xf5, 0x59, 0xff, 0x48, 0xf9, 0x24, 0x0, 0xe0,
      0xf8, 0x45, 0xff, 0x0, 0xfe, 0x32, 0xf9, 0x2e, 0xf4, 0x9b, 0x3, 0x4f, 0x1,
      0xd5, 0xff, 0xe5, 0x4, 0xfa, 0x2, 0xc0, 0xff, 0x9a, 0x4, 0x8f, 0xfe, 0xcb,
      0xff, 0x5f, 0xff, 0x22, 0xfc, 0x9, 0x1, 0xce, 0x1, 0xf9, 0x0, 0x0, 0x2,
      0x2, 0x3, 0xc5, 0xff, 0x8b, 0xfb, 0xa3, 0xf4, 0x3a, 0x1, 0x9d, 0xfa, 0xa7,
      0xfe, 0x2d, 0x0, 0x3f, 0xfb, 0x11, 0xff, 0xb6, 0xfa, 0xd2, 0x2, 0x9a, 0x6,
      0x5f, 0xfa, 0x3d, 0x7, 0x70, 0x3, 0xe0, 0x6, 0x29, 0xfd, 0x27, 0xfd, 0xc,
      0x2, 0xbb, 0xfa, 0xe9, 0xf5, 0x5b, 0xfd, 0x42, 0xfe, 0x3f, 0xfe, 0x3a,
      0xfd, 0x45, 0xf6, 0xc0, 0xff, 0xf9, 0xfc, 0x7f, 0x5, 0x49, 0xfc, 0xa0,
      0x7, 0xd, 0x11, 0x87, 0x9, 0x4f, 0x5, 0xb, 0xf, 0xd9, 0x5, 0x1d, 0x3,
      0xc2, 0x4, 0xa, 0xf0, 0x8, 0x0, 0x60, 0x3, 0x94, 0x5, 0x69, 0xfe, 0xfc,
      0xfd, 0x1a, 0x2, 0xe4, 0xf5, 0xe3, 0xd, 0xd9, 0x10, 0x9a, 0x0, 0x6f, 0xd,
      0xf1, 0x2, 0x40, 0xfd, 0xb1, 0x2, 0xef, 0x0, 0x6f, 0xff, 0x2e, 0xa, 0x67,
      0x2, 0xb, 0xc, 0x9, 0x4, 0xa9, 0xfc, 0xe3, 0xfe, 0xb4, 0x4, 0x7, 0xd,
      0xf1, 0x15, 0xed, 0xf4, 0x46, 0xfe, 0x4c, 0xfe, 0x59, 0xfd, 0x9b, 0xf6,
      0x6c, 0xfe, 0x1e, 0xfc, 0x5, 0xb, 0x58, 0xfe, 0x3c, 0x6, 0x8f, 0xfe, 0xcb,
      0x0, 0xa, 0xfc, 0x9, 0x0, 0x58, 0x1, 0x5f, 0x10, 0x61, 0xfc, 0xb4, 0xf4,
      0x8a, 0x8, 0xe7, 0x2, 0xf6, 0xf9, 0x63, 0x0, 0x42, 0xeb, 0x5e, 0x6, 0xf6,
      0xfe, 0x71, 0x5, 0xce, 0xfa, 0xf3, 0x3, 0x8, 0xfe, 0xd7, 0xf2, 0xf0, 0xfd,
      0xb3, 0xfa, 0xf6, 0x8, 0xb9, 0xb, 0xfb, 0x5, 0xa, 0x6, 0x90, 0xff, 0x1d,
      0x1, 0x69, 0x5, 0x9f, 0x2, 0x2a, 0x2, 0xa4, 0xfd, 0xd1, 0xfc, 0xca, 0xfc,
      0x21, 0x4, 0xf3, 0x1, 0x41, 0x2, 0x1f, 0x7, 0x19, 0x0, 0xe6, 0x3, 0xc2,
      0xf9, 0x69, 0x0, 0xc5, 0xfe, 0x9d, 0x3, 0x29, 0xf9, 0x51, 0xfc, 0x1d, 0x3,
      0x96, 0xfe, 0xf0, 0x1, 0x72, 0x0, 0x73, 0xfe, 0x7a, 0x7, 0x4a, 0xfe, 0x32,
      0x6, 0xa2, 0xf3, 0xfd, 0xf4, 0x50, 0xf7, 0xba, 0xf0, 0xb6, 0x0, 0x8a,
      0xfa, 0xcd, 0xf1, 0x37, 0xfe, 0x9b, 0xfc, 0x55, 0x3, 0x5e, 0x5, 0xfa, 0x1,
      0x54, 0x4, 0x1c, 0x1, 0xd1, 0x3, 0x4a, 0x1, 0x93, 0x1, 0xc, 0xfb, 0x38,
      0xf8, 0x8f, 0x2, 0x6c, 0xff, 0xc4, 0x0, 0x19, 0x8, 0xb3, 0x1, 0xb0, 0xfe,
      0x98, 0x2, 0xbf, 0xff, 0xae, 0x2, 0xa1, 0xfe, 0xd3, 0x4, 0x85, 0x4, 0x95,
      0xff, 0x86, 0x9, 0x31, 0xfd, 0xf9, 0xf9, 0x8, 0x9, 0x15, 0xfe, 0x36, 0x2,
      0x9e, 0xc, 0x9, 0x3, 0x88, 0xfd, 0x4a, 0x0, 0xe1, 0xfd, 0x49, 0xfe, 0xc4,
      0xfc, 0x6c, 0x5, 0x8, 0xa, 0x58, 0xfb, 0xa7, 0x6, 0x69, 0x9, 0x7c, 0x0,
      0x71, 0x3, 0xbf, 0x5, 0xd8, 0xf6, 0x1b, 0x18, 0x9e, 0x1, 0x13, 0xfe, 0x22,
      0xff, 0x60, 0x3, 0x2d, 0xfd, 0xa2, 0xff, 0x91, 0x4, 0x6, 0x8, 0xd7, 0xfb,
      0x24, 0x4, 0x10, 0x4, 0x91, 0x3, 0xdc, 0x0, 0x21, 0xb, 0xf6, 0x0, 0xa1,
      0x14, 0xbb, 0x2, 0xb4, 0x3, 0xf0, 0x2, 0x6b, 0x3, 0x26, 0x5, 0x5b, 0x0,
      0x51, 0x0, 0xf7, 0xfc, 0x8f, 0xf9, 0xba, 0x5, 0xb6, 0xfb, 0xf, 0x1, 0x9f,
      0xf9, 0x4e, 0x4, 0x50, 0x8, 0xf7, 0x7, 0xc9, 0x0, 0xff, 0x0, 0xd0, 0xfe,
      0x65, 0x1, 0xb8, 0xff, 0x0, 0xa, 0xc9, 0xfc, 0x77, 0xff, 0x97, 0xff, 0xef,
      0xfa, 0x6e, 0x0, 0x48, 0x1, 0xdc, 0xfb, 0x83, 0xff, 0x14, 0x0, 0x11, 0x3,
      0xa9, 0xfd, 0xb3, 0x2, 0xbe, 0xff, 0x5f, 0x3, 0xdf, 0xfd, 0x44, 0xff,
      0xc6, 0xfc, 0xe9, 0xf9, 0xa, 0xfb, 0xec, 0xf1, 0x7a, 0xfd, 0x1d, 0x1,
      0x75, 0xfb, 0x9f, 0x2, 0x26, 0xfc, 0x67, 0x1, 0xd9, 0xff, 0x77, 0x0, 0x27,
      0x1, 0x7e, 0x0, 0xe3, 0xff, 0xf2, 0xfa, 0x7a, 0x0, 0x1c, 0xfc, 0x14, 0xfd,
      0x1d, 0x1, 0x89, 0xfd, 0x1c, 0xfd, 0x70, 0x3, 0x1d, 0x1, 0xff, 0x1, 0x3e,
      0x3, 0xbf, 0x0, 0xfd, 0xfd, 0xb8, 0xfd, 0x6f, 0xfe, 0x10, 0x0, 0x23, 0xfe,
      0x3c, 0x1, 0x2f, 0xfe, 0x78, 0xff, 0xea, 0x2, 0xe7, 0xfc, 0x38, 0xff,
      0xde, 0x4, 0x8d, 0x2, 0x81, 0xfb, 0x19, 0x2, 0x50, 0x1, 0xb6, 0xfe, 0xc9,
      0x0, 0x9a, 0x0, 0x27, 0x2, 0x78, 0xfe, 0xa0, 0xfc, 0x31, 0xff, 0xe7, 0xfc,
      0x58, 0x4, 0x40, 0x1, 0xe2, 0x0, 0xca, 0x2, 0xb1, 0x6, 0x8f, 0xfd, 0xea,
      0x6, 0xcd, 0x1, 0xda, 0xff, 0xff, 0xff, 0x4a, 0xff, 0x59, 0xff, 0x31, 0x3,
      0x4c, 0xff, 0x57, 0x1, 0x3d, 0xfb, 0xfe, 0x6, 0xc8, 0x0, 0xe9, 0xfe, 0x72,
      0x3, 0xbf, 0x2, 0x89, 0x6, 0x2e, 0x6, 0x14, 0x0, 0x66, 0x2, 0xa0, 0xfe,
      0x68, 0xff, 0xae, 0xfe, 0x28, 0x2, 0x27, 0xfd, 0xc7, 0xff, 0x51, 0xf7,
      0x46, 0xfc, 0x12, 0x0, 0xd8, 0x1, 0x8a, 0xfd, 0xa4, 0x2, 0xa0, 0xfe, 0xc5,
      0x1, 0x32, 0xfb, 0x14, 0x1, 0x4b, 0x0, 0xe1, 0x0, 0xd, 0x0, 0x97, 0x0,
      0xf0, 0xfd, 0xfa, 0x0, 0x4f, 0xfc, 0xc2, 0xfa, 0xf8, 0x1, 0x26, 0x2, 0x6b,
      0xfd, 0x47, 0xff, 0x71, 0x3, 0x13, 0xfd, 0x8a, 0xfa, 0x66, 0x1, 0x76, 0x0,
      0x59, 0xfe, 0x79, 0xff, 0x90, 0xfe, 0x60, 0x1, 0x77, 0xfc, 0x71, 0xf8,
      0xcb, 0xfc, 0x53, 0xff, 0xb7, 0xfe, 0x8a, 0xfc, 0x8e, 0xfb, 0x29, 0x2,
      0x3d, 0xfb, 0xb, 0xfc, 0x91, 0xfe, 0xe0, 0xfe, 0xb2, 0xfc, 0xe0, 0xfe,
      0x54, 0xff, 0x88, 0xff, 0x82, 0x1, 0x1f, 0xf9, 0xdc, 0xff, 0x22, 0x2,
      0x73, 0xfc, 0xca, 0x1, 0xff, 0xff, 0xaa, 0xfb, 0x69, 0xfd, 0x20, 0xfd,
      0xf4, 0xfe, 0xbc, 0xfd, 0xc4, 0xfd, 0x22, 0x0, 0x48, 0xfe, 0x51, 0x1,
      0xe1, 0x0, 0x61, 0xf9, 0xf0, 0x1, 0x27, 0xfc, 0x82, 0xfe, 0x54, 0x3, 0xe6,
      0xff, 0xce, 0xfe, 0x52, 0xfd, 0x15, 0x0, 0xf9, 0xff, 0x94, 0xfd, 0x4f,
      0xfe, 0xbb, 0x0, 0xe0, 0xff, 0x2a, 0x0, 0x44, 0x0, 0x72, 0xf9, 0x1f, 0x3,
      0x69, 0xfd, 0xc1, 0x1, 0x4a, 0x0, 0x97, 0x2, 0x81, 0x0, 0xeb, 0x3, 0xb3,
      0xff, 0x7b, 0xff, 0xf2, 0xfe, 0xcb, 0xff, 0xb5, 0xff, 0x3d, 0x1, 0x14,
      0x1, 0x33, 0x1, 0x62, 0xfc, 0x82, 0x2, 0x3, 0x1, 0x10, 0x0, 0x9e, 0xff,
      0xad, 0x2, 0x23, 0x2, 0xd6, 0x1, 0xdd, 0xfe, 0xc2, 0x1, 0xc6, 0xfe, 0x85,
      0x0, 0xe, 0xff, 0x9a, 0xff, 0xee, 0xff, 0x4c, 0xfd, 0x77, 0xfa, 0x1, 0xfd,
      0xcc, 0x0, 0x52, 0x1, 0x24, 0xfd, 0x5, 0x0, 0x24, 0x0, 0x6e, 0xfe, 0xc5,
      0xfd, 0x4c, 0x0, 0x92, 0xfe, 0xd6, 0xff, 0xbb, 0xfe, 0xe, 0x0, 0x76, 0x0,
      0x20, 0xff, 0x57, 0xfe, 0xdd, 0xfc, 0xbe, 0x2, 0xc4, 0x1, 0xed, 0xfe,
      0xd0, 0x0, 0x47, 0x1, 0xfe, 0xfd, 0x98, 0xfd, 0x4c, 0x0, 0xbf, 0xff, 0xb,
      0xff, 0x20, 0xff, 0xe3, 0xfd, 0xf2, 0x1, 0xf7, 0xfb, 0xb, 0xfd, 0xca,
      0xfd, 0x44, 0xff, 0xb7, 0xfe, 0xc0, 0xff, 0xdf, 0xff, 0xfd, 0xfd, 0x97,
      0xfc, 0xa1, 0xfe, 0xb1, 0xfe, 0xd5, 0xfd, 0x84, 0xfe, 0xc4, 0x0, 0x66,
      0xfe, 0xef, 0xfe, 0x2a, 0xff, 0x98, 0xfe, 0x43, 0x0, 0xb2, 0x0, 0xbd,
      0xfe, 0x17, 0x2, 0x86, 0x0, 0x65, 0xfe, 0x17, 0x0, 0xb4, 0xff, 0xa0, 0xfe,
      0xf5, 0xfe, 0xf8, 0xfe, 0x5a, 0x1, 0x56, 0xfe, 0x7f, 0x0, 0x1b, 0xfe,
      0xc4, 0xfc, 0xf6, 0x2, 0x94, 0xfc, 0x14, 0xff, 0x5, 0x1, 0x45, 0x0, 0x26,
      0x1, 0x59, 0xfe, 0x56, 0x1, 0x82, 0xff, 0xd5, 0xff, 0x7c, 0xff, 0xb4,
      0xff, 0x6f, 0x0, 0x48, 0xff, 0xfb, 0xfd, 0x42, 0xfc, 0xaf, 0x1, 0x6f,
      0xfe, 0x42, 0x0, 0x36, 0x0, 0x2d, 0x1, 0x41, 0x0, 0x5c, 0x1, 0x30, 0x1,
      0x27, 0x0, 0x45, 0x0, 0x57, 0x0, 0x55, 0xff, 0x60, 0x0, 0xcf, 0xff, 0xdf,
      0xff, 0xda, 0xff, 0xa5, 0x0, 0x28, 0x1, 0x9d, 0x1, 0x88, 0xff, 0x4e, 0x0,
      0x3f, 0x2, 0x1d, 0x1, 0x9a, 0x0, 0x67, 0x1, 0xad, 0x0, 0xb4, 0xff, 0x28,
      0xff, 0xaa, 0xff, 0x70, 0xff, 0x3f, 0xfd, 0x0, 0xfe, 0x20, 0xfe, 0xcc,
      0xff, 0xd1, 0x1, 0xb6, 0xfd, 0xa, 0x0, 0x48, 0x0, 0x44, 0xfe, 0x9d, 0xff,
      0x8, 0x0, 0x22, 0x0, 0x8a, 0xff, 0x8f, 0xff, 0xfa, 0x0, 0x72, 0xff, 0x91,
      0xff, 0x92, 0xfe, 0x91, 0xfc, 0x86, 0x2, 0xe1, 0x0, 0xb0, 0xff, 0xca,
      0xff, 0x35, 0x0, 0x18, 0xfe, 0x1a, 0x0, 0xbf, 0x0, 0x3e, 0x1, 0xde, 0xff,
      0x24, 0x0, 0xb6, 0xfe, 0x32, 0x1, 0x8d, 0xfd, 0xba, 0xfc, 0xd5, 0xfe,
      0xc2, 0xfe, 0xbc, 0xff, 0x9c, 0xff, 0x26, 0xfd, 0xbd, 0xfe, 0xa2, 0xfa,
      0x6a, 0x1, 0x68, 0xff, 0x6, 0x0, 0x32, 0xff, 0x66, 0x0, 0x1e, 0x0, 0x3b,
      0xff, 0xfd, 0xfe, 0xac, 0xfd, 0x71, 0x1, 0x40, 0xff, 0x71, 0xff, 0x4e,
      0x2, 0xf2, 0xfe, 0x97, 0xff, 0x78, 0xfd, 0x9, 0x2, 0x61, 0xff, 0x75, 0x0,
      0x94, 0xff, 0xae, 0x0, 0xa0, 0xff, 0xa0, 0x0, 0xba, 0xfe, 0xd0, 0xfc,
      0x7a, 0x1, 0x96, 0xfe, 0xd, 0xff, 0x5b, 0x1, 0x5c, 0xff, 0x47, 0x0, 0x96,
      0xfe, 0x5b, 0x2, 0xc5, 0xff, 0xae, 0x0, 0xab, 0xff, 0xd5, 0xff, 0x6, 0x0,
      0xa1, 0xff, 0x9c, 0xfe, 0xc1, 0xfc, 0xd5, 0x0, 0x8, 0x0, 0xa5, 0xff, 0x32,
      0x0, 0x53, 0x0, 0x41, 0x0, 0x97, 0xfe, 0x11, 0x2, 0xad, 0xff, 0x27, 0x1,
      0x13, 0x0, 0x9b, 0xff, 0x94, 0x0, 0xd3, 0xff, 0xfb, 0xfe, 0x58, 0xfd,
      0xcf, 0xff, 0xc9, 0x1, 0xfd, 0xff, 0xe1, 0xff, 0x4f, 0x0, 0xe9, 0x1, 0x98,
      0xfe, 0x6c, 0x1, 0xa9, 0x0, 0x84, 0x1, 0xf1, 0xff, 0xf5, 0xfe, 0x0, 0x0,
      0x82, 0x0, 0x55, 0xfe, 0xc3, 0xfd, 0x3b, 0xfe, 0xd9, 0x0, 0x2a, 0x0, 0x1a,
      0xfe, 0x39, 0xff, 0xce, 0xff, 0xe4, 0xfd, 0x17, 0x0, 0xfd, 0xff, 0xe9,
      0x0, 0xc5, 0xff, 0xa0, 0xff, 0xbc, 0x0, 0x6, 0x0, 0x82, 0xff, 0x75, 0xff,
      0x67, 0xfe, 0xd3, 0x1, 0xd5, 0xff, 0xe, 0x0, 0x5a, 0xff, 0x4d, 0x0, 0x96,
      0xfe, 0x1, 0x0, 0x41, 0x0, 0xf7, 0x0, 0x10, 0x0, 0x28, 0x0, 0x90, 0xff,
      0x72, 0x0, 0x73, 0xfd, 0xc4, 0xfd, 0x72, 0xff, 0xc3, 0xff, 0x9d, 0xff,
      0x77, 0x0, 0xe7, 0xfe, 0xe6, 0xfe, 0x6f, 0xfe, 0xe, 0x0, 0x56, 0xff, 0xab,
      0xff, 0xad, 0xff, 0xfe, 0xff, 0x47, 0x0, 0x91, 0xff, 0xc9, 0xfe, 0x7b,
      0xff, 0x3e, 0x0, 0x8b, 0x0, 0x6, 0x0, 0x31, 0x1, 0xcf, 0xff, 0x93, 0xff,
      0xe2, 0xff, 0xfd, 0xff, 0x29, 0xff, 0x22, 0x0, 0x9e, 0xff, 0x3b, 0x0, 0x1,
      0x0, 0xe0, 0xff, 0x18, 0xfe, 0x52, 0xfe, 0xb7, 0x0, 0x7b, 0xff, 0x9e,
      0xff, 0x49, 0x0, 0xe3, 0xff, 0x41, 0x0, 0x1b, 0x0, 0xe, 0x0, 0x47, 0xff,
      0x5, 0x0, 0x87, 0xff, 0x81, 0xff, 0x44, 0x0, 0x2b, 0xff, 0xc6, 0xfe, 0x20,
      0xff, 0x6e, 0x0, 0xe2, 0xff, 0xfc, 0xff, 0x21, 0x0, 0xa6, 0x0, 0x43, 0x0,
      0x42, 0x0, 0xab, 0xff, 0xf0, 0xff, 0x64, 0x0, 0xba, 0xff, 0x89, 0xff,
      0x35, 0x0, 0x37, 0x0, 0x2f, 0xff, 0xd8, 0xff, 0x5a, 0xff, 0x3e, 0x0, 0x70,
      0x0, 0x81, 0xff, 0xc3, 0xff, 0x36, 0x1, 0xb2, 0xff, 0x7c, 0xff, 0x3f, 0x0,
      0x86, 0x0, 0xc2, 0xff, 0x6a, 0xff, 0xed, 0xff, 0x14, 0x0, 0x76, 0xfe,
      0xd4, 0xfe, 0xee, 0xfe, 0x22, 0x0, 0x8e, 0x0, 0x12, 0xff, 0xe6, 0xff,
      0xab, 0xff, 0xf2, 0xff, 0x3e, 0xff, 0xde, 0xff, 0x2d, 0x0, 0xe0, 0xff,
      0xf3, 0xff, 0x25, 0x0, 0x24, 0x0, 0x9a, 0xfe, 0xea, 0xff, 0x36, 0xff,
      0x77, 0x0, 0x8c, 0x0, 0xb0, 0x0, 0x36, 0x0, 0x4f, 0x0, 0xb4, 0xff, 0x4e,
      0xff, 0x34, 0x0, 0xd, 0x0, 0xaf, 0xff, 0x1f, 0x0, 0xc4, 0xff, 0xcf, 0x0,
      0xae, 0xfd, 0x9d, 0xff, 0xa3, 0xff, 0x6d, 0xff, 0x93, 0x0, 0xe9, 0xff,
      0x29, 0xff, 0x5b, 0xff, 0x3f, 0xff, 0x6e, 0xff, 0x69, 0xff, 0x77, 0xff,
      0xb2, 0xff, 0x42, 0x0, 0x3f, 0x0, 0xe1, 0xff, 0xb2, 0xfe, 0x5c, 0x0, 0x92,
      0x0, 0x4e, 0x0, 0xd, 0x0, 0xc2, 0x0, 0x59, 0x0, 0x4, 0x0, 0x9a, 0x0, 0xd9,
      0xff, 0xb, 0x0, 0x1, 0x0, 0xef, 0xff, 0x59, 0x0, 0xc6, 0xff, 0x73, 0x0,
      0xea, 0xfd, 0x43, 0xff, 0xd5, 0x0, 0x22, 0xff, 0xb9, 0xff, 0x65, 0x0,
      0x1d, 0x0, 0x87, 0x0, 0x5b, 0x0, 0xf8, 0xff, 0xfd, 0xff, 0x59, 0xff, 0xb4,
      0xff, 0xcb, 0xff, 0x6b, 0x0, 0x35, 0x0, 0x37, 0xff, 0xe9, 0xff, 0x25, 0x0,
      0xa1, 0x0, 0x92, 0x0, 0x2, 0x0, 0x34, 0x0, 0xd, 0x0, 0x9c, 0x0, 0x0, 0x0,
      0xe, 0x0, 0xd8, 0xff, 0x34, 0x0, 0x28, 0x0, 0x53, 0x0, 0x87, 0x0, 0xcc,
      0xfe, 0x71, 0xff, 0xc4, 0xff, 0x60, 0x0, 0x3, 0x0, 0xf5, 0xff, 0x20, 0x0,
      0x37, 0x0, 0xfc, 0xff, 0xe2, 0xff, 0x5b, 0x0, 0xc7, 0xff, 0xf6, 0xff,
      0xc9, 0xff, 0x2c, 0x0, 0x2f, 0x0, 0x9e, 0xfe, 0xdb, 0xff, 0xae, 0xff, 0x7,
      0x0, 0xdf, 0xff, 0x8e, 0xff, 0x1, 0x0, 0xce, 0xff, 0x23, 0x0, 0x8b, 0xff,
      0x1a, 0x0, 0x7b, 0xff, 0xdf, 0xff, 0xe, 0x0, 0x22, 0x0, 0x4c, 0x0, 0xe4,
      0xfe, 0x7c, 0x0, 0x8f, 0xff, 0x99, 0x0, 0x5a, 0x0, 0x34, 0x0, 0xee, 0xff,
      0xef, 0xff, 0x39, 0x0, 0xc3, 0xff, 0x3, 0x0, 0xac, 0xff, 0x39, 0x0, 0x35,
      0x0, 0xb2, 0xff, 0x13, 0x0, 0xfe, 0xfd, 0x35, 0xff, 0xef, 0xff, 0x6f,
      0xff, 0xfe, 0xff, 0x44, 0x0, 0xe6, 0xff, 0x66, 0xff, 0x2a, 0x0, 0xcc,
      0xff, 0xce, 0xff, 0x7a, 0xff, 0x2d, 0x0, 0x11, 0x0, 0xc, 0x0, 0xe0, 0xff,
      0xf2, 0xfe, 0x48, 0x0, 0x4c, 0x0, 0xce, 0xff, 0x1d, 0x0, 0x7e, 0x0, 0x3c,
      0x0, 0x20, 0x0, 0xd3, 0x0, 0xf4, 0xff, 0x3f, 0x0, 0xdd, 0xff, 0x32, 0x0,
      0xb, 0x0, 0x17, 0x0, 0x58, 0x0, 0x81, 0xfe, 0xc0, 0xff, 0x43, 0x0, 0x64,
      0xff, 0x3c, 0x0, 0x4, 0x0, 0xd0, 0xff, 0xc4, 0xff, 0x90, 0x0, 0x1a, 0x0,
      0xc9, 0xff, 0x95, 0xff, 0x1f, 0x0, 0xe7, 0xff, 0x6f, 0x0, 0x92, 0xff,
      0x15, 0xff, 0x35, 0x0, 0xdf, 0xff, 0x6f, 0x0, 0xa, 0x0, 0x7d, 0x0, 0x48,
      0x0, 0x62, 0x0, 0xe1, 0x0, 0x22, 0x0, 0x20, 0x0, 0xf9, 0xff, 0x36, 0x0,
      0x2b, 0x0, 0xed, 0xff, 0x55, 0x0, 0x94, 0xfe, 0x4e, 0x0, 0x7b, 0xff, 0xe8,
      0xff, 0xfa, 0xff, 0x20, 0x0, 0xd6, 0xff, 0x4b, 0x0, 0x2c, 0x0, 0xb, 0x0,
      0xf8, 0xff, 0xab, 0xff, 0x23, 0x0, 0x9, 0x0, 0xf6, 0xff, 0x12, 0x0, 0xe9,
      0xfe, 0x87, 0x0, 0xaa, 0xff, 0x15, 0x0, 0x11, 0x0, 0xf4, 0xff, 0x1b, 0x0,
      0x57, 0xff, 0x5d, 0x0, 0xfe, 0xff, 0xdc, 0xff, 0xef, 0xff, 0x41, 0x0,
      0x47, 0x0, 0xed, 0xff, 0x7, 0x0, 0x9b, 0xfe, 0x5d, 0x0, 0xec, 0xff, 0x17,
      0x0, 0xe6, 0xff, 0x71, 0x0, 0x31, 0x0, 0xd7, 0xff, 0x4d, 0x0, 0xdb, 0xff,
      0x44, 0x0, 0xea, 0xff, 0x18, 0x0, 0x23, 0x0, 0xe1, 0xff, 0x24, 0x0, 0x54,
      0xfe, 0x32, 0x0, 0xef, 0xff, 0xba, 0xff, 0x20, 0x0, 0x4, 0x0, 0xb2, 0xff,
      0x84, 0xff, 0x2d, 0x0, 0xc9, 0xff, 0xf0, 0xff, 0xbe, 0xff, 0x2e, 0x0,
      0x59, 0x0, 0xf3, 0xff, 0xe4, 0xff, 0xfe, 0xfe, 0x93, 0x0, 0xe0, 0xff,
      0x65, 0x0, 0x7, 0x0, 0x65, 0x0, 0x13, 0x0, 0xfa, 0xff, 0x92, 0x0, 0xd,
      0x0, 0x29, 0x0, 0x35, 0x0, 0x54, 0x0, 0x77, 0x0, 0xa1, 0xff, 0x8, 0x0,
      0x5b, 0xfe, 0xe8, 0xff, 0x12, 0x0, 0x95, 0xff, 0xb7, 0xff, 0x3d, 0x0, 0x2,
      0x0, 0xc, 0x0, 0x67, 0x0, 0xe0, 0xff, 0x15, 0x0, 0xc8, 0xff, 0x0, 0x0,
      0x1f, 0x0, 0x9, 0x0, 0xf2, 0xff, 0x21, 0xff, 0xc5, 0x0, 0x1, 0x0, 0x4c,
      0x0, 0x2e, 0x0, 0xa, 0x0, 0xb, 0x0, 0x1f, 0x0, 0x8c, 0x0, 0xfb, 0xff,
      0x2f, 0x0, 0x8, 0x0, 0x21, 0x0, 0x20, 0x0, 0xfa, 0xff, 0x2e, 0x0, 0xc0,
      0xfe, 0x49, 0x0, 0xcc, 0xff, 0xee, 0xff, 0x23, 0x0, 0xc3, 0xff, 0x19, 0x0,
      0x8d, 0xff, 0x3f, 0x0, 0x3, 0x0, 0x1b, 0x0, 0xd6, 0xff, 0x1c, 0x0, 0xf8,
      0xff, 0x10, 0x0, 0xc9, 0xff, 0xf4, 0xfe, 0x8b, 0x0, 0xfb, 0xff, 0xe3,
      0xff, 0xfa, 0xff, 0xfd, 0xff, 0x56, 0x0, 0xc2, 0xff, 0x81, 0x0, 0xe9,
      0xff, 0x3b, 0x0, 0xf7, 0xff, 0x8, 0x0, 0xe4, 0xff, 0xf3, 0xff, 0x42, 0x0,
      0xbd, 0xfe, 0xa4, 0x0, 0xe0, 0xff, 0xf6, 0xff, 0x16, 0x0, 0xed, 0xff,
      0xdb, 0xff, 0xe7, 0xff, 0xd, 0x0, 0xc, 0x0, 0xed, 0xff, 0xf5, 0xff, 0x24,
      0x0, 0xee, 0xff, 0xe5, 0xff, 0xe6, 0xff, 0xb4, 0xfe, 0x24, 0x0, 0xe2,
      0xff, 0xf8, 0xff, 0xf5, 0xff, 0xe0, 0xff, 0xf1, 0xff, 0xa3, 0xff, 0x10,
      0x0, 0xa, 0x0, 0xde, 0xff, 0x7, 0x0, 0x3d, 0x0, 0x41, 0x0, 0xd2, 0xff,
      0xdf, 0xff, 0xcf, 0xfe, 0x54, 0x0, 0x22, 0x0, 0xfe, 0xff, 0xec, 0xff,
      0x3b, 0x0, 0x3, 0x0, 0x65, 0x0, 0x36, 0x0, 0x20, 0x0, 0x20, 0x0, 0x1e,
      0x0, 0x17, 0x0, 0xf9, 0xff, 0xdc, 0xff, 0x29, 0x0, 0xb0, 0xfe, 0x53, 0x0,
      0x1a, 0x0, 0xc5, 0xff, 0x2d, 0x0, 0xba, 0xff, 0xcb, 0xff, 0xc9, 0xff, 0xe,
      0x0, 0x28, 0x0, 0xc1, 0xff, 0xf3, 0xff, 0x16, 0x0, 0xe9, 0xff, 0x16, 0x0,
      0xb8, 0xff, 0x2d, 0xff, 0x75, 0x0, 0xfa, 0xff, 0x3e, 0x0, 0x11, 0x0, 0x7,
      0x0, 0x36, 0x0, 0x1e, 0x0, 0x48, 0x0, 0x1e, 0x0, 0x31, 0x0, 0x2d, 0x0,
      0x1a, 0x0, 0xe5, 0xff, 0xec, 0xff, 0xf, 0x0, 0x85, 0xfe, 0x14, 0x0, 0xe7,
      0xff, 0xbc, 0xff, 0xf7, 0xff, 0x5, 0x0, 0xf8, 0xff, 0x56, 0x0, 0xe2, 0xff,
      0xf7, 0xff, 0x10, 0x0, 0xd6, 0xff, 0x0, 0x0, 0xc5, 0xff, 0x16, 0x0, 0xed,
      0xff, 0xf, 0xff, 0xa7, 0x0, 0xd0, 0xff, 0x43, 0x0, 0x2f, 0x0, 0xdb, 0xff,
      0xf8, 0xff, 0x37, 0x0, 0xe7, 0xff, 0xfa, 0xff, 0xed, 0xff, 0x1c, 0x0,
      0x34, 0x0, 0xf, 0x0, 0xed, 0xff, 0xee, 0xff, 0xbe, 0xfe, 0x3a, 0x0, 0xea,
      0xff, 0xed, 0xff, 0xf2, 0xff, 0x12, 0x0, 0xe2, 0xff, 0x51, 0x0, 0xdc,
      0xff, 0xe9, 0xff, 0x3, 0x0, 0xfb, 0xff, 0x14, 0x0, 0xe4, 0xff, 0xf0, 0xff,
      0xd8, 0xff, 0xcf, 0xfe, 0x5a, 0x0, 0x39, 0x0, 0xd3, 0xff, 0x13, 0x0, 0x1a,
      0x0, 0xd2, 0xff, 0x49, 0x0, 0x10, 0x0, 0xe7, 0xff, 0xf0, 0xff, 0x3, 0x0,
      0x5, 0x0, 0xeb, 0xff, 0xe7, 0xff, 0x6, 0x0, 0xb, 0xff, 0x7f, 0x0, 0x35,
      0x0, 0x22, 0x0, 0x34, 0x0, 0x24, 0x0, 0xd8, 0xff, 0x27, 0x0, 0x8, 0x0,
      0x14, 0x0, 0xe2, 0xff, 0x3b, 0x0, 0x25, 0x0, 0xe8, 0xff, 0xe7, 0xff, 0xd8,
      0xff, 0xd8, 0xfe, 0x23, 0x0, 0x3f, 0x0, 0xf3, 0xff, 0xe9, 0xff, 0x3a, 0x0,
      0x1c, 0x0, 0x1, 0x0, 0x2a, 0x0, 0xec, 0xff, 0xf7, 0xff, 0x17, 0x0, 0x17,
      0x0, 0xe7, 0xff, 0xf6, 0xff, 0xf3, 0xff, 0x21, 0xff, 0xae, 0x0, 0x3e, 0x0,
      0x23, 0x0, 0x2, 0x0, 0x1f, 0x0, 0xc, 0x0, 0x58, 0x0, 0x23, 0x0, 0xdc,
      0xff, 0x9, 0x0, 0x7, 0x0, 0x1c, 0x0, 0xd1, 0xff, 0xfe, 0xff, 0x24, 0x0,
      0xf3, 0xfe, 0x64, 0x0, 0xf0, 0xff, 0x7, 0x0, 0x12, 0x0, 0xd9, 0xff, 0xd8,
      0xff, 0xe1, 0xff, 0xdf, 0xff, 0xdd, 0xff, 0xcf, 0xff, 0xe5, 0xff, 0x40,
      0x0, 0xf0, 0xff, 0x13, 0x0, 0xb9, 0xff, 0x1c, 0xff, 0x44, 0x0, 0x6, 0x0,
      0x20, 0x0, 0xde, 0xff, 0x35, 0x0, 0xc, 0x0, 0x4c, 0x0, 0x17, 0x0, 0xd0,
      0xff, 0x17, 0x0, 0x7, 0x0, 0x27, 0x0, 0x8, 0x0, 0xe6, 0xff, 0x1a, 0x0,
      0xcb, 0xfe, 0x31, 0x0, 0xc, 0x0, 0xb7, 0xff, 0xf8, 0xff, 0xfa, 0xff, 0x98,
      0xff, 0x36, 0x0, 0xda, 0xff, 0xda, 0xff, 0xcf, 0xff, 0xd3, 0xff, 0x9, 0x0,
      0xed, 0xff, 0xc, 0x0, 0xd9, 0xff, 0x1e, 0xff, 0x4e, 0x0, 0xf6, 0xff, 0x1c,
      0x0, 0xe, 0x0, 0xf7, 0xff, 0xdf, 0xff, 0xdc, 0xff, 0xd, 0x0, 0xe6, 0xff,
      0xd0, 0xff, 0x1e, 0x0, 0x27, 0x0, 0x28, 0x0, 0xea, 0xff, 0xdc, 0xff, 0xec,
      0xfe, 0xa, 0x0, 0x37, 0x0, 0xdf, 0xff, 0xc0, 0xff, 0x37, 0x0, 0x14, 0x0,
      0x36, 0x0, 0x1, 0x0, 0xd6, 0xff, 0xfd, 0xff, 0xff, 0xff, 0x7, 0x0, 0xf1,
      0xff, 0xa, 0x0, 0x5, 0x0, 0xc, 0xff, 0x30, 0x0, 0x46, 0x0, 0xcd, 0xff,
      0xe3, 0xff, 0xcf, 0xff, 0xef, 0xff, 0xfa, 0xff, 0xf4, 0xff, 0xe6, 0xff,
      0xc9, 0xff, 0xde, 0xff, 0x21, 0x0, 0xf3, 0xff, 0x2b, 0x0, 0xde, 0xff,
      0x42, 0xff, 0x37, 0x0, 0xf, 0x0, 0x20, 0x0, 0xf4, 0xff, 0xe5, 0xff, 0x2,
      0x0, 0x8, 0x0, 0x0, 0x0, 0xf0, 0xff, 0xfb, 0xff, 0x2, 0x0, 0x1b, 0x0, 0x8,
      0x0, 0xb, 0x0, 0xdc, 0xff, 0xe8, 0xfe, 0xe6, 0xff, 0x2, 0x0, 0xdf, 0xff,
      0xd1, 0xff, 0xff, 0xff, 0xd4, 0xff, 0x28, 0x0, 0x19, 0x0, 0xdd, 0xff,
      0xff, 0xff, 0xd1, 0xff, 0xe7, 0xff, 0xf9, 0xff, 0x15, 0x0, 0x0, 0x0, 0x51,
      0xff, 0x41, 0x0, 0x19, 0x0, 0x43, 0x0, 0x13, 0x0, 0xdb, 0xff, 0xca, 0xff,
      0x36, 0x0, 0x47, 0x0, 0x6, 0x0, 0xeb, 0xff, 0xff, 0xff, 0xe, 0x0, 0x3,
      0x0, 0xe, 0x0, 0x1f, 0x0, 0x2d, 0xff, 0xfa, 0xff, 0x21, 0x0, 0x23, 0x0,
      0xe, 0x0, 0xf3, 0xff, 0xde, 0xff, 0xf6, 0xff, 0x4b, 0x0, 0x6, 0x0, 0xf1,
      0xff, 0xfb, 0xff, 0xe, 0x0, 0xf0, 0xff, 0x1d, 0x0, 0x8, 0x0, 0x45, 0xff,
      0x2e, 0x0, 0x3e, 0x0, 0x2a, 0x0, 0x1, 0x0, 0x19, 0x0, 0xf6, 0xff, 0x28,
      0x0, 0x7f, 0x0, 0xfe, 0xff, 0x21, 0x0, 0x2, 0x0, 0xff, 0xff, 0xcc, 0xff,
      0xc, 0x0, 0x5e, 0x0, 0x44, 0xff, 0x47, 0x0, 0x25, 0x0, 0x1e, 0x0, 0x34,
      0x0, 0xe1, 0xff, 0xc6, 0xff, 0xfd, 0xff, 0x34, 0x0, 0x12, 0x0, 0xf7, 0xff,
      0xfb, 0xff, 0x1a, 0x0, 0xcd, 0xff, 0x1a, 0x0, 0x21, 0x0, 0x53, 0xff, 0x11,
      0x0, 0xfc, 0xff, 0x2a, 0x0, 0x11, 0x0, 0xf9, 0xff, 0xed, 0xff, 0xe7, 0xff,
      0x48, 0x0, 0x4, 0x0, 0xd, 0x0, 0x16, 0x0, 0x23, 0x0, 0xfd, 0xff, 0xf2,
      0xff, 0x33, 0x0, 0x37, 0xff, 0x13, 0x0, 0x18, 0x0, 0xfa, 0xff, 0xf4, 0xff,
      0x2, 0x0, 0xe7, 0xff, 0x35, 0x0, 0x37, 0x0, 0xfd, 0xff, 0x10, 0x0, 0xf2,
      0xff, 0x10, 0x0, 0xd8, 0xff, 0xf3, 0xff, 0x49, 0x0, 0x4c, 0xff, 0x35, 0x0,
      0xe, 0x0, 0x4, 0x0, 0x1c, 0x0, 0xdc, 0xff, 0xeb, 0xff, 0xe0, 0xff, 0x2a,
      0x0, 0xfd, 0xff, 0xda, 0xff, 0xfc, 0xff, 0x33, 0x0, 0x2, 0x0, 0x4, 0x0,
      0xa, 0x0, 0x6d, 0xff, 0x1d, 0x0, 0x16, 0x0, 0x22, 0x0, 0xf8, 0xff, 0x1b,
      0x0, 0x2e, 0x0, 0xe, 0x0, 0x5b, 0x0, 0xf0, 0xff, 0x1b, 0x0, 0x19, 0x0,
      0x2d, 0x0, 0xb, 0x0, 0xf5, 0xff, 0x2f, 0x0, 0x2e, 0xff, 0x8, 0x0, 0x1c,
      0x0, 0xc4, 0xff, 0xf5, 0xff, 0xf7, 0xff, 0xf9, 0xff, 0xb, 0x0, 0x25, 0x0,
      0xea, 0xff, 0x1, 0x0, 0xdc, 0xff, 0xe, 0x0, 0xed, 0xff, 0x1b, 0x0, 0x9,
      0x0, 0x68, 0xff, 0x45, 0x0, 0xed, 0xff, 0xe, 0x0, 0x18, 0x0, 0xec, 0xff,
      0x7, 0x0, 0xef, 0xff, 0x16, 0x0, 0xf5, 0xff, 0xf4, 0xff, 0x18, 0x0, 0x26,
      0x0, 0x19, 0x0, 0xf6, 0xff, 0xec, 0xff, 0x26, 0xff, 0xf5, 0xff, 0xe2,
      0xff, 0xe4, 0xff, 0xeb, 0xff, 0x10, 0x0, 0x12, 0x0, 0xf, 0x0, 0x10, 0x0,
      0xe3, 0xff, 0x4, 0x0, 0xfe, 0xff, 0x5, 0x0, 0xa, 0x0, 0xed, 0xff, 0xf3,
      0xff, 0x41, 0xff, 0x2d, 0x0, 0xf6, 0xff, 0xf2, 0xff, 0x0, 0x0, 0xfc, 0xff,
      0x6, 0x0, 0x2a, 0x0, 0x24, 0x0, 0xeb, 0xff, 0xf2, 0xff, 0x3, 0x0, 0x0,
      0x0, 0xe, 0x0, 0x2, 0x0, 0x10, 0x0, 0x50, 0xff, 0x2b, 0x0, 0xf4, 0xff,
      0x6, 0x0, 0x12, 0x0, 0xf7, 0xff, 0x25, 0x0, 0xf7, 0xff, 0x3, 0x0, 0xf8,
      0xff, 0xe6, 0xff, 0x18, 0x0, 0x22, 0x0, 0x1a, 0x0, 0x0, 0x0, 0xe6, 0xff,
      0x40, 0xff, 0xb, 0x0, 0xf4, 0xff, 0xee, 0xff, 0xea, 0xff, 0x16, 0x0, 0x32,
      0x0, 0xfb, 0xff, 0x11, 0x0, 0xe5, 0xff, 0x9, 0x0, 0x0, 0x0, 0x9, 0x0, 0xf,
      0x0, 0xf9, 0xff, 0x5, 0x0, 0x55, 0xff, 0x3e, 0x0, 0xfc, 0xff, 0xee, 0xff,
      0xc, 0x0, 0x9, 0x0, 0x5, 0x0, 0x1b, 0x0, 0xd, 0x0, 0xf4, 0xff, 0x0, 0x0,
      0x3, 0x0, 0x2, 0x0, 0x0, 0x0, 0xb, 0x0, 0x5, 0x0, 0x5b, 0xff, 0x13, 0x0,
      0xea, 0xff, 0xed, 0xff, 0x18, 0x0, 0xfd, 0xff, 0x16, 0x0, 0xce, 0xff,
      0xfa, 0xff, 0xf7, 0xff, 0xef, 0xff, 0xa, 0x0, 0x1d, 0x0, 0x13, 0x0, 0x1,
      0x0, 0xe3, 0xff, 0x61, 0xff, 0x5, 0x0, 0xfc, 0xff, 0x2, 0x0, 0xec, 0xff,
      0x21, 0x0, 0x30, 0x0, 0x1c, 0x0, 0x1f, 0x0, 0xed, 0xff, 0x18, 0x0, 0xf,
      0x0, 0x5, 0x0, 0x5, 0x0, 0xf9, 0xff, 0x25, 0x0, 0x4a, 0xff, 0x1b, 0x0,
      0x14, 0x0, 0xe2, 0xff, 0x10, 0x0, 0xf1, 0xff, 0x8, 0x0, 0x14, 0x0, 0xf,
      0x0, 0xf8, 0xff, 0xec, 0xff, 0xf8, 0xff, 0x2, 0x0, 0x8, 0x0, 0x15, 0x0,
      0xf5, 0xff, 0x7e, 0xff, 0x1e, 0x0, 0xfc, 0xff, 0x15, 0x0, 0x19, 0x0, 0xf7,
      0xff, 0x2c, 0x0, 0xe9, 0xff, 0x23, 0x0, 0xf9, 0xff, 0x7, 0x0, 0x13, 0x0,
      0xb, 0x0, 0x23, 0x0, 0xf6, 0xff, 0xfe, 0xff, 0x59, 0xff, 0x2, 0x0, 0x3,
      0x0, 0xe3, 0xff, 0xfc, 0xff, 0xa, 0x0, 0x17, 0x0, 0x8, 0x0, 0x20, 0x0,
      0xf7, 0xff, 0x1b, 0x0, 0xf0, 0xff, 0xef, 0xff, 0xff, 0xff, 0x2, 0x0, 0x11,
      0x0, 0x77, 0xff, 0x3c, 0x0, 0x8, 0x0, 0xf6, 0xff, 0x1f, 0x0, 0xe5, 0xff,
      0x0, 0x0, 0xf1, 0xff, 0x19, 0x0, 0x1, 0x0, 0xf5, 0xff, 0xfc, 0xff, 0x0,
      0x0, 0xf, 0x0, 0x1, 0x0, 0xf3, 0xff, 0x7d, 0xff, 0x11, 0x0, 0xf8, 0xff,
      0xc, 0x0, 0x2, 0x0, 0xfc, 0xff, 0x18, 0x0, 0xec, 0xff, 0x1a, 0x0, 0xf8,
      0xff, 0x4, 0x0, 0xc, 0x0, 0x5, 0x0, 0xf, 0x0, 0xf0, 0xff, 0xf5, 0xff,
      0x68, 0xff, 0xfe, 0xff, 0xa, 0x0, 0xe6, 0xff, 0xf1, 0xff, 0x4, 0x0, 0x9,
      0x0, 0x9, 0x0, 0x1a, 0x0, 0xf5, 0xff, 0x0, 0x0, 0xf4, 0xff, 0xf2, 0xff,
      0xfd, 0xff, 0x0, 0x0, 0x6, 0x0, 0x84, 0xff, 0x20, 0x0, 0xd, 0x0, 0xb, 0x0,
      0x13, 0x0, 0xeb, 0xff, 0x5, 0x0, 0x8, 0x0, 0x7, 0x0, 0x3, 0x0, 0xec, 0xff,
      0x10, 0x0, 0x6, 0x0, 0xb, 0x0, 0xfa, 0xff, 0xed, 0xff, 0x6f, 0xff, 0xf4,
      0xff, 0xf6, 0xff, 0xff, 0xff, 0xfa, 0xff, 0xff, 0xff, 0x15, 0x0, 0xe9,
      0xff, 0x8, 0x0, 0xf4, 0xff, 0xfe, 0xff, 0x6, 0x0, 0x6, 0x0, 0x6, 0x0,
      0xef, 0xff, 0xe8, 0xff, 0x79, 0xff, 0x17, 0x0, 0x1, 0x0, 0x0, 0x0, 0x2,
      0x0, 0x4, 0x0, 0x2, 0x0, 0x1c, 0x0, 0x18, 0x0, 0xf8, 0xff, 0x9, 0x0, 0xfe,
      0xff, 0xf0, 0xff, 0xf2, 0xff, 0xf8, 0xff, 0xc, 0x0, 0x73, 0xff, 0x17, 0x0,
      0xfd, 0xff, 0xf9, 0xff, 0x1d, 0x0, 0xe8, 0xff, 0xf5, 0xff, 0xf3, 0xff,
      0x0, 0x0, 0xfa, 0xff, 0xe9, 0xff, 0x3, 0x0, 0x6, 0x0, 0xfc, 0xff, 0xfc,
      0xff, 0xe6, 0xff, 0x8a, 0xff, 0xf, 0x0, 0xf7, 0xff, 0x9, 0x0, 0x5, 0x0,
      0xf, 0x0, 0x11, 0x0, 0xf0, 0xff, 0x21, 0x0, 0xe9, 0xff, 0xb, 0x0, 0xb,
      0x0, 0x8, 0x0, 0xfa, 0xff, 0xf0, 0xff, 0x8, 0x0, 0x79, 0xff, 0x15, 0x0,
      0x6, 0x0, 0xee, 0xff, 0x7, 0x0, 0x1, 0x0, 0xed, 0xff, 0x18, 0x0, 0xe, 0x0,
      0xf2, 0xff, 0xfe, 0xff, 0xf4, 0xff, 0xf1, 0xff, 0xea, 0xff, 0x9, 0x0, 0x6,
      0x0, 0x89, 0xff, 0x18, 0x0, 0xfa, 0xff, 0x3, 0x0, 0x13, 0x0, 0xec, 0xff,
      0xfd, 0xff, 0xe7, 0xff, 0x7, 0x0, 0xf5, 0xff, 0xe9, 0xff, 0x7, 0x0, 0xf,
      0x0, 0x8, 0x0, 0xf8, 0xff, 0xf0, 0xff, 0x83, 0xff, 0xf1, 0xff, 0xf9, 0xff,
      0x6, 0x0, 0xf3, 0xff, 0x9, 0x0, 0x1c, 0x0, 0xfc, 0xff, 0x22, 0x0, 0xf5,
      0xff, 0xd, 0x0, 0x5, 0x0, 0x6, 0x0, 0xf8, 0xff, 0xf0, 0xff, 0x14, 0x0,
      0x78, 0xff, 0x11, 0x0, 0x1, 0x0, 0xed, 0xff, 0x9, 0x0, 0xe9, 0xff, 0xf4,
      0xff, 0xc, 0x0, 0x6, 0x0, 0xf8, 0xff, 0xf6, 0xff, 0xef, 0xff, 0xfd, 0xff,
      0xf6, 0xff, 0xa, 0x0, 0x3, 0x0, 0x9a, 0xff, 0x27, 0x0, 0xfb, 0xff, 0x18,
      0x0, 0x15, 0x0, 0xeb, 0xff, 0x10, 0x0, 0xf1, 0xff, 0x6, 0x0, 0xf5, 0xff,
      0xf8, 0xff, 0xd, 0x0, 0x1a, 0x0, 0x8, 0x0, 0xf8, 0xff, 0x0, 0x0, 0x7d,
      0xff, 0xf0, 0xff, 0x8, 0x0, 0xf8, 0xff, 0xf9, 0xff, 0x10, 0x0, 0x1f, 0x0,
      0xf8, 0xff, 0x1a, 0x0, 0xf7, 0xff, 0x8, 0x0, 0xfc, 0xff, 0x7, 0x0, 0xf5,
      0xff, 0xf9, 0xff, 0xe, 0x0, 0x8e, 0xff, 0x1f, 0x0, 0xf, 0x0, 0xff, 0xff,
      0x15, 0x0, 0xfe, 0xff, 0xfa, 0xff, 0xf, 0x0, 0xa, 0x0, 0x5, 0x0, 0xf6,
      0xff, 0xfe, 0xff, 0xfe, 0xff, 0xfe, 0xff, 0x7, 0x0, 0x0, 0x0, 0x88, 0xff,
      0xf5, 0xff, 0xf7, 0xff, 0xd, 0x0, 0xe, 0x0, 0xed, 0xff, 0xf, 0x0, 0xe8,
      0xff, 0x12, 0x0, 0x7, 0x0, 0xf5, 0xff, 0x3, 0x0, 0x10, 0x0, 0xa, 0x0, 0x5,
      0x0, 0xf5, 0xff, 0x8a, 0xff, 0xfb, 0xff, 0xf9, 0xff, 0x1, 0x0, 0xf8, 0xff,
      0xa, 0x0, 0x11, 0x0, 0x2, 0x0, 0x19, 0x0, 0xf0, 0xff, 0xb, 0x0, 0xf5,
      0xff, 0x8, 0x0, 0xff, 0xff, 0xf6, 0xff, 0x13, 0x0, 0x8d, 0xff, 0x25, 0x0,
      0x2, 0x0, 0xfc, 0xff, 0x15, 0x0, 0xf5, 0xff, 0xe9, 0xff, 0xc, 0x0, 0xeb,
      0xff, 0xf4, 0xff, 0xf4, 0xff, 0xf0, 0xff, 0x5, 0x0, 0xf6, 0xff, 0x9, 0x0,
      0xf2, 0xff, 0x8c, 0xff, 0xe9, 0xff, 0xf9, 0xff, 0x12, 0x0, 0x5, 0x0, 0xfb,
      0xff, 0x17, 0x0, 0xe5, 0xff, 0x1b, 0x0, 0xfd, 0xff, 0xfc, 0xff, 0x0, 0x0,
      0x10, 0x0, 0xa, 0x0, 0x4, 0x0, 0xf6, 0xff, 0x90, 0xff, 0x0, 0x0, 0xff,
      0xff, 0xfd, 0xff, 0xe7, 0xff, 0x13, 0x0, 0x4, 0x0, 0xe, 0x0, 0x1c, 0x0,
      0xec, 0xff, 0x9, 0x0, 0xf3, 0xff, 0x4, 0x0, 0x5, 0x0, 0xf7, 0xff, 0x1d,
      0x0, 0x96, 0xff, 0x24, 0x0, 0xfa, 0xff, 0xee, 0xff, 0xd, 0x0, 0xed, 0xff,
      0xd1, 0xff, 0xf7, 0xff, 0xe7, 0xff, 0xef, 0xff, 0xe7, 0xff, 0xe4, 0xff,
      0x8, 0x0, 0x2, 0x0, 0xd, 0x0, 0xef, 0xff, 0xa3, 0xff, 0xfe, 0xff, 0xfe,
      0xff, 0x16, 0x0, 0x17, 0x0, 0x9, 0x0, 0xc, 0x0, 0xef, 0xff, 0xb, 0x0,
      0xfa, 0xff, 0x6, 0x0, 0x9, 0x0, 0x9, 0x0, 0xa, 0x0, 0x2, 0x0, 0xf9, 0xff,
      0xa0, 0xff, 0x19, 0x0, 0x9, 0x0, 0xff, 0xff, 0x1, 0x0, 0x9, 0x0, 0xf5,
      0xff, 0xb, 0x0, 0xfe, 0xff, 0xf4, 0xff, 0x5, 0x0, 0xf9, 0xff, 0x3, 0x0,
      0xff, 0xff, 0xf9, 0xff, 0x12, 0x0, 0xc2, 0xff, 0x4b, 0x0, 0xf1, 0xff, 0xb,
      0x0, 0x16, 0x0, 0xe3, 0xff, 0xd4, 0xff, 0xf5, 0xff, 0xf4, 0xff, 0x2, 0x0,
      0xf0, 0xff, 0xfa, 0xff, 0xd, 0x0, 0x2, 0x0, 0x4, 0x0, 0xf3, 0xff, 0xa8,
      0xff, 0xe, 0x0, 0xf3, 0xff, 0x1, 0x0, 0x16, 0x0, 0x7, 0x0, 0x4, 0x0, 0xef,
      0xff, 0xf6, 0xff, 0xb, 0x0, 0x5, 0x0, 0xd, 0x0, 0xff, 0xff, 0x7, 0x0, 0x2,
      0x0, 0xe7, 0xff, 0xb6, 0xff, 0x2d, 0x0, 0xe, 0x0, 0x7, 0x0, 0xa, 0x0, 0xd,
      0x0, 0xe9, 0xff, 0xd, 0x0, 0xe8, 0xff, 0xff, 0xff, 0x3, 0x0, 0xff, 0xff,
      0xfe, 0xff, 0x3, 0x0, 0xfa, 0xff, 0x3, 0x0, 0xc6, 0xff, 0x3c, 0x0, 0xf3,
      0xff, 0x14, 0x0, 0x11, 0x0, 0xee, 0xff, 0xd6, 0xff, 0xf8, 0xff, 0xe3,
      0xff, 0x8, 0x0, 0xf4, 0xff, 0x2, 0x0, 0xa, 0x0, 0x5, 0x0, 0x8, 0x0, 0xf4,
      0xff, 0xab, 0xff, 0x17, 0x0, 0xe7, 0xff, 0xed, 0xff, 0x4, 0x0, 0xfe, 0xff,
      0xa, 0x0, 0xec, 0xff, 0xd8, 0xff, 0xfe, 0xff, 0x2, 0x0, 0x10, 0x0, 0x5,
      0x0, 0x15, 0x0, 0xff, 0xff, 0xf1, 0xff, 0xc0, 0xff, 0x47, 0x0, 0x1, 0x0,
      0xf9, 0xff, 0x0, 0x0, 0xfe, 0xff, 0xe6, 0xff, 0xc, 0x0, 0xe6, 0xff, 0xf7,
      0xff, 0xfe, 0xff, 0x2, 0x0, 0x4, 0x0, 0xa, 0x0, 0xfc, 0xff, 0x8, 0x0,
      0xc3, 0xff, 0x38, 0x0, 0xf7, 0xff, 0x2, 0x0, 0x3, 0x0, 0xf7, 0xff, 0xe0,
      0xff, 0xed, 0xff, 0x4, 0x0, 0xfe, 0xff, 0xf0, 0xff, 0xfb, 0xff, 0x6, 0x0,
      0x7, 0x0, 0x5, 0x0, 0xf3, 0xff, 0xc5, 0xff, 0x2e, 0x0, 0xfb, 0xff, 0x1,
      0x0, 0xfb, 0xff, 0x10, 0x0, 0x14, 0x0, 0xef, 0xff, 0xb, 0x0, 0xf0, 0xff,
      0x5, 0x0, 0xa, 0x0, 0x7, 0x0, 0xf, 0x0, 0xf4, 0xff, 0x2, 0x0, 0xd1, 0xff,
      0x48, 0x0, 0x7, 0x0, 0xc, 0x0, 0x1, 0x0, 0xfc, 0xff, 0xef, 0xff, 0xfe,
      0xff, 0x1a, 0x0, 0xf5, 0xff, 0xf5, 0xff, 0xf7, 0xff, 0xa, 0x0, 0x6, 0x0,
      0x1, 0x0, 0x6, 0x0, 0xd2, 0xff, 0x37, 0x0, 0x1, 0x0, 0xd, 0x0, 0x4, 0x0,
      0xf9, 0xff, 0xfe, 0xff, 0xf6, 0xff, 0x23, 0x0, 0xfc, 0xff, 0xf7, 0xff,
      0xfe, 0xff, 0xc, 0x0, 0xf, 0x0, 0x8, 0x0, 0xf7, 0xff, 0xbf, 0xff, 0x31,
      0x0, 0xfc, 0xff, 0xf9, 0xff, 0xfb, 0xff, 0x3, 0x0, 0x18, 0x0, 0xf8, 0xff,
      0x10, 0x0, 0xeb, 0xff, 0x6, 0x0, 0xfe, 0xff, 0xc, 0x0, 0x9, 0x0, 0xfb,
      0xff, 0x0, 0x0, 0xd1, 0xff, 0x4b, 0x0, 0xf1, 0xff, 0xe, 0x0, 0x6, 0x0,
      0xea, 0xff, 0xf5, 0xff, 0xf3, 0xff, 0x2c, 0x0, 0xf0, 0xff, 0xf6, 0xff,
      0xed, 0xff, 0x10, 0x0, 0x9, 0x0, 0xfb, 0xff, 0x4, 0x0, 0xc6, 0xff, 0x30,
      0x0, 0xf5, 0xff, 0x4, 0x0, 0x0, 0x0, 0xfa, 0xff, 0xb, 0x0, 0xfa, 0xff,
      0x24, 0x0, 0xf8, 0xff, 0xfc, 0xff, 0xf9, 0xff, 0x8, 0x0, 0xf, 0x0, 0xff,
      0xff, 0xf8, 0xff, 0xbc, 0xff, 0x2e, 0x0, 0xfe, 0xff, 0xf5, 0xff, 0xf7,
      0xff, 0x6, 0x0, 0x12, 0x0, 0xf0, 0xff, 0x17, 0x0, 0xee, 0xff, 0x0, 0x0,
      0xf8, 0xff, 0xc, 0x0, 0x8, 0x0, 0xfe, 0xff, 0xff, 0xff, 0xdd, 0xff, 0x46,
      0x0, 0xfa, 0xff, 0x16, 0x0, 0x9, 0x0, 0xf5, 0xff, 0xfa, 0xff, 0xf1, 0xff,
      0x31, 0x0, 0xfd, 0xff, 0xfa, 0xff, 0xf7, 0xff, 0xe, 0x0, 0x9, 0x0, 0x2,
      0x0, 0x2, 0x0, 0xc1, 0xff, 0x15, 0x0, 0x4, 0x0, 0xf9, 0xff, 0xfd, 0xff,
      0x4, 0x0, 0x16, 0x0, 0xf9, 0xff, 0x21, 0x0, 0x1, 0x0, 0x1, 0x0, 0xfa,
      0xff, 0x5, 0x0, 0x9, 0x0, 0xb, 0x0, 0xf2, 0xff, 0xd0, 0xff, 0x26, 0x0,
      0x2, 0x0, 0x1, 0x0, 0xfd, 0xff, 0x8, 0x0, 0x12, 0x0, 0xf6, 0xff, 0x1c,
      0x0, 0xf0, 0xff, 0x7, 0x0, 0xfa, 0xff, 0xf, 0x0, 0x5, 0x0, 0xfe, 0xff,
      0x2, 0x0, 0xdb, 0xff, 0x2c, 0x0, 0xef, 0xff, 0x10, 0x0, 0xe, 0x0, 0xf0,
      0xff, 0xf7, 0xff, 0xf7, 0xff, 0x22, 0x0, 0xfd, 0xff, 0xfc, 0xff, 0xf9,
      0xff, 0xb, 0x0, 0x7, 0x0, 0x5, 0x0, 0xfb, 0xff, 0xc4, 0xff, 0x3, 0x0, 0x1,
      0x0, 0xfc, 0xff, 0xfa, 0xff, 0x2, 0x0, 0x11, 0x0, 0x5, 0x0, 0x13, 0x0,
      0xfb, 0xff, 0x4, 0x0, 0xff, 0xff, 0x1, 0x0, 0x6, 0x0, 0x8, 0x0, 0xf6,
      0xff, 0xd7, 0xff, 0x19, 0x0, 0x3, 0x0, 0x5, 0x0, 0xfa, 0xff, 0x1, 0x0,
      0x2, 0x0, 0xf9, 0xff, 0xe, 0x0, 0xef, 0xff, 0x3, 0x0, 0xfa, 0xff, 0x8,
      0x0, 0x1, 0x0, 0xfb, 0xff, 0x3, 0x0, 0xde, 0xff, 0x12, 0x0, 0xf5, 0xff,
      0xa, 0x0, 0x8, 0x0, 0xef, 0xff, 0xf7, 0xff, 0xf2, 0xff, 0x15, 0x0, 0x1,
      0x0, 0xf8, 0xff, 0xf7, 0xff, 0x4, 0x0, 0x6, 0x0, 0x4, 0x0, 0xf9, 0xff,
      0xcd, 0xff, 0xf6, 0xff, 0xc, 0x0, 0xf6, 0xff, 0xf8, 0xff, 0x5, 0x0, 0x14,
      0x0, 0x9, 0x0, 0xa, 0x0, 0xfb, 0xff, 0x6, 0x0, 0xff, 0xff, 0xff, 0xff,
      0x2, 0x0, 0x0, 0x0, 0xfc, 0xff, 0xdc, 0xff, 0xf, 0x0, 0x3, 0x0, 0x5, 0x0,
      0xff, 0xff, 0xf5, 0xff, 0xff, 0xff, 0xf6, 0xff, 0xf, 0x0, 0xf3, 0xff,
      0xfe, 0xff, 0xf5, 0xff, 0x9, 0x0, 0x0, 0x0, 0xf7, 0xff, 0x1, 0x0, 0xe1,
      0xff, 0xb, 0x0, 0xf9, 0xff, 0xe, 0x0, 0x7, 0x0, 0xee, 0xff, 0xfe, 0xff,
      0xfb, 0xff, 0x15, 0x0, 0x1, 0x0, 0xfb, 0xff, 0xf9, 0xff, 0x2, 0x0, 0x4,
      0x0, 0x1, 0x0, 0xfb, 0xff, 0xc8, 0xff, 0xf1, 0xff, 0xa, 0x0, 0xf5, 0xff,
      0xf8, 0xff, 0x2, 0x0, 0x12, 0x0, 0x9, 0x0, 0x4, 0x0, 0xf7, 0xff, 0x4, 0x0,
      0xfd, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xfe, 0xff, 0xfc, 0xff, 0xe1, 0xff,
      0x14, 0x0, 0xfb, 0xff, 0xd, 0x0, 0x2, 0x0, 0xf3, 0xff, 0xfd, 0xff, 0xf1,
      0xff, 0xf, 0x0, 0xf5, 0xff, 0xfb, 0xff, 0xf5, 0xff, 0x8, 0x0, 0x2, 0x0,
      0xf5, 0xff, 0x4, 0x0, 0xd8, 0xff, 0x2, 0x0, 0xfa, 0xff, 0x4, 0x0, 0x4,
      0x0, 0xf6, 0xff, 0x0, 0x0, 0xfb, 0xff, 0x8, 0x0, 0x3, 0x0, 0xfb, 0xff,
      0xf8, 0xff, 0xff, 0xff, 0x6, 0x0, 0xff, 0xff, 0xf7, 0xff, 0xc9, 0xff,
      0xf3, 0xff, 0x8, 0x0, 0xf0, 0xff, 0xf9, 0xff, 0x4, 0x0, 0x11, 0x0, 0x2,
      0x0, 0x1, 0x0, 0xf8, 0xff, 0x3, 0x0, 0xfd, 0xff, 0x1, 0x0, 0x1, 0x0, 0xfa,
      0xff, 0xff, 0xff, 0xe3, 0xff, 0x14, 0x0, 0xfa, 0xff, 0xa, 0x0, 0x6, 0x0,
      0xf6, 0xff, 0xf9, 0xff, 0xf3, 0xff, 0xd, 0x0, 0xfb, 0xff, 0xfc, 0xff,
      0xf8, 0xff, 0x9, 0x0, 0x2, 0x0, 0xf7, 0xff, 0x3, 0x0, 0xd2, 0xff, 0xfb,
      0xff, 0xfb, 0xff, 0xfc, 0xff, 0x4, 0x0, 0xfc, 0xff, 0x3, 0x0, 0xfd, 0xff,
      0x8, 0x0, 0x6, 0x0, 0xfe, 0xff, 0xfd, 0xff, 0xfe, 0xff, 0x4, 0x0, 0x3,
      0x0, 0xf5, 0xff, 0xd1, 0xff, 0xfc, 0xff, 0x9, 0x0, 0xf8, 0xff, 0xfd, 0xff,
      0x5, 0x0, 0xf, 0x0, 0x2, 0x0, 0x5, 0x0, 0xf7, 0xff, 0x6, 0x0, 0xff, 0xff,
      0x3, 0x0, 0xff, 0xff, 0xfa, 0xff, 0x3, 0x0, 0xe2, 0xff, 0x11, 0x0, 0xf7,
      0xff, 0xa, 0x0, 0x9, 0x0, 0xf2, 0xff, 0xf7, 0xff, 0xf0, 0xff, 0x9, 0x0,
      0xff, 0xff, 0xfc, 0xff, 0xf7, 0xff, 0x9, 0x0, 0x1, 0x0, 0xfa, 0xff, 0xff,
      0xff, 0xd2, 0xff, 0xf5, 0xff, 0xfb, 0xff, 0xfc, 0xff, 0x3, 0x0, 0xfe,
      0xff, 0x4, 0x0, 0xff, 0xff, 0x0, 0x0, 0x5, 0x0, 0x2, 0x0, 0x0, 0x0, 0xfe,
      0xff, 0x3, 0x0, 0x2, 0x0, 0xf8, 0xff, 0xd2, 0xff, 0xfd, 0xff, 0x7, 0x0,
      0xf7, 0xff, 0xfd, 0xff, 0x4, 0x0, 0x5, 0x0, 0xfe, 0xff, 0xfc, 0xff, 0xf9,
      0xff, 0x5, 0x0, 0xfd, 0xff, 0x3, 0x0, 0xfe, 0xff, 0xf9, 0xff, 0x3, 0x0,
      0xe3, 0xff, 0xd, 0x0, 0xf5, 0xff, 0xa, 0x0, 0x8, 0x0, 0xf3, 0xff, 0xf4,
      0xff, 0xf0, 0xff, 0x7, 0x0, 0x4, 0x0, 0xfa, 0xff, 0xf9, 0xff, 0x7, 0x0,
      0x3, 0x0, 0xfc, 0xff, 0xfe, 0xff, 0xd1, 0xff, 0xf1, 0xff, 0x3, 0x0, 0xf9,
      0xff, 0xfd, 0xff, 0x3, 0x0, 0x6, 0x0, 0x3, 0x0, 0xfd, 0xff, 0x5, 0x0, 0x3,
      0x0, 0x2, 0x0, 0xfd, 0xff, 0x1, 0x0, 0x2, 0x0, 0xf8, 0xff, 0xd7, 0xff,
      0xfe, 0xff, 0x7, 0x0, 0xfc, 0xff, 0xfc, 0xff, 0x0, 0x0, 0x3, 0x0, 0xfc,
      0xff, 0xfd, 0xff, 0xfc, 0xff, 0x2, 0x0, 0xfd, 0xff, 0x5, 0x0, 0xfe, 0xff,
      0xf9, 0xff, 0x3, 0x0, 0xe4, 0xff, 0x8, 0x0, 0xf7, 0xff, 0xd, 0x0, 0x6,
      0x0, 0xf3, 0xff, 0xf6, 0xff, 0xf7, 0xff, 0x5, 0x0, 0x6, 0x0, 0xfb, 0xff,
      0xfc, 0xff, 0x5, 0x0, 0x3, 0x0, 0xfe, 0xff, 0xfd, 0xff, 0xcf, 0xff, 0xee,
      0xff, 0x3, 0x0, 0xf7, 0xff, 0xfb, 0xff, 0x2, 0x0, 0x7, 0x0, 0x6, 0x0,
      0xf8, 0xff, 0x3, 0x0, 0x3, 0x0, 0x2, 0x0, 0xfe, 0xff, 0x1, 0x0, 0x1, 0x0,
      0xf8, 0xff, 0xdc, 0xff, 0x3, 0x0, 0x1, 0x0, 0x4, 0x0, 0xfe, 0xff, 0xfd,
      0xff, 0xfe, 0xff, 0xfb, 0xff, 0xfe, 0xff, 0xfb, 0xff, 0x0, 0x0, 0xfd,
      0xff, 0x7, 0x0, 0xff, 0xff, 0xf9, 0xff, 0x5, 0x0, 0xe0, 0xff, 0x3, 0x0,
      0xf6, 0xff, 0x9, 0x0, 0x5, 0x0, 0xf5, 0xff, 0xf6, 0xff, 0xf8, 0xff, 0x0,
      0x0, 0x7, 0x0, 0xfa, 0xff, 0xfd, 0xff, 0x3, 0x0, 0x3, 0x0, 0xff, 0xff,
      0xf9, 0xff, 0xd0, 0xff, 0xef, 0xff, 0x6, 0x0, 0xf7, 0xff, 0xf9, 0xff, 0x3,
      0x0, 0x8, 0x0, 0x5, 0x0, 0xf7, 0xff, 0x1, 0x0, 0x2, 0x0, 0x3, 0x0, 0xfe,
      0xff, 0x1, 0x0, 0x0, 0x0, 0xfa, 0xff, 0xdf, 0xff, 0x5, 0x0, 0x1, 0x0, 0x5,
      0x0, 0x0, 0x0, 0xfc, 0xff, 0xfb, 0xff, 0xf9, 0xff, 0xff, 0xff, 0xfd, 0xff,
      0xfe, 0xff, 0xfd, 0xff, 0x6, 0x0, 0xff, 0xff, 0xf8, 0xff, 0x4, 0x0, 0xdf,
      0xff, 0xff, 0xff, 0xf7, 0xff, 0x5, 0x0, 0x4, 0x0, 0xf7, 0xff, 0xf9, 0xff,
      0xfb, 0xff, 0xff, 0xff, 0x8, 0x0, 0xfb, 0xff, 0x0, 0x0, 0x2, 0x0, 0x3,
      0x0, 0x1, 0x0, 0xf8, 0xff, 0xd2, 0xff, 0xf2, 0xff, 0x5, 0x0, 0xf7, 0xff,
      0xfa, 0xff, 0x6, 0x0, 0x7, 0x0, 0x7, 0x0, 0xf7, 0xff, 0xfe, 0xff, 0x3,
      0x0, 0x5, 0x0, 0xff, 0xff, 0xff, 0xff, 0x0, 0x0, 0xfd, 0xff, 0xe0, 0xff,
      0x8, 0x0, 0xfb, 0xff, 0x7, 0x0, 0x3, 0x0, 0xfb, 0xff, 0xf9, 0xff, 0xf7,
      0xff, 0x0, 0x0, 0xfe, 0xff, 0xfc, 0xff, 0xff, 0xff, 0x7, 0x0, 0x0, 0x0,
      0xfb, 0xff, 0x3, 0x0, 0xdc, 0xff, 0xfd, 0xff, 0xf7, 0xff, 0x3, 0x0, 0x4,
      0x0, 0xfb, 0xff, 0xfb, 0xff, 0xfe, 0xff, 0xfc, 0xff, 0x5, 0x0, 0xfc, 0xff,
      0x3, 0x0, 0x0, 0x0, 0x3, 0x0, 0x3, 0x0, 0xf8, 0xff, 0xd2, 0xff, 0xf6,
      0xff, 0x5, 0x0, 0xf7, 0xff, 0xfb, 0xff, 0x6, 0x0, 0x5, 0x0, 0x5, 0x0,
      0xf7, 0xff, 0xfc, 0xff, 0x2, 0x0, 0x4, 0x0, 0x0, 0x0, 0xff, 0xff, 0xfe,
      0xff, 0xfe, 0xff, 0xe2, 0xff, 0xa, 0x0, 0xfa, 0xff, 0x9, 0x0, 0x4, 0x0,
      0xfa, 0xff, 0xf7, 0xff, 0xf7, 0xff, 0x2, 0x0, 0xff, 0xff, 0xfb, 0xff,
      0xfe, 0xff, 0x6, 0x0, 0x1, 0x0, 0xfc, 0xff, 0x3, 0x0, 0xda, 0xff, 0xfa,
      0xff, 0xfb, 0xff, 0xff, 0xff, 0x2, 0x0, 0xff, 0xff, 0xfe, 0xff, 0x1, 0x0,
      0xfb, 0xff, 0x5, 0x0, 0xfe, 0xff, 0x3, 0x0, 0xfe, 0xff, 0x3, 0x0, 0x4,
      0x0, 0xf8, 0xff, 0xd6, 0xff, 0xf9, 0xff, 0x5, 0x0, 0xfa, 0xff, 0xfc, 0xff,
      0x4, 0x0, 0x4, 0x0, 0x2, 0x0, 0xfa, 0xff, 0xfb, 0xff, 0x2, 0x0, 0x3, 0x0,
      0x1, 0x0, 0xff, 0xff, 0xfe, 0xff, 0x1, 0x0, 0xe4, 0xff, 0xa, 0x0, 0xf9,
      0xff, 0x9, 0x0, 0x5, 0x0, 0xf9, 0xff, 0xf7, 0xff, 0xf8, 0xff, 0x2, 0x0,
      0x0, 0x0, 0xfb, 0xff, 0xfe, 0xff, 0x5, 0x0, 0x1, 0x0, 0xfe, 0xff, 0x1,
      0x0, 0xd8, 0xff, 0xf8, 0xff, 0xfd, 0xff, 0xfd, 0xff, 0x0, 0x0, 0xff, 0xff,
      0x1, 0x0, 0x2, 0x0, 0xfa, 0xff, 0x3, 0x0, 0xff, 0xff, 0x3, 0x0, 0xfe,
      0xff, 0x2, 0x0, 0x3, 0x0, 0xf8, 0xff, 0xd9, 0xff, 0xfd, 0xff, 0x4, 0x0,
      0xfd, 0xff, 0xfd, 0xff, 0x2, 0x0, 0x2, 0x0, 0x0, 0x0, 0xfc, 0xff, 0xfa,
      0xff, 0x1, 0x0, 0x1, 0x0, 0x2, 0x0, 0xff, 0xff, 0xfd, 0xff, 0x2, 0x0,
      0xe4, 0xff, 0x8, 0x0, 0xf8, 0xff, 0x7, 0x0, 0x5, 0x0, 0xf8, 0xff, 0xf8,
      0xff, 0xf7, 0xff, 0x1, 0x0, 0x1, 0x0, 0xfb, 0xff, 0xfd, 0xff, 0x4, 0x0,
      0x2, 0x0, 0xff, 0xff, 0xff, 0xff, 0xd8, 0xff, 0xf7, 0xff, 0x0, 0x0, 0xfb,
      0xff, 0xff, 0xff, 0x1, 0x0, 0x2, 0x0, 0x3, 0x0, 0xfa, 0xff, 0x1, 0x0, 0x0,
      0x0, 0x2, 0x0, 0xfe, 0xff, 0x1, 0x0, 0x4, 0x0, 0xfa, 0xff, 0xdc, 0xff,
      0x0, 0x0, 0x3, 0x0, 0xff, 0xff, 0xfe, 0xff, 0x1, 0x0, 0x0, 0x0, 0xfe,
      0xff, 0xfe, 0xff, 0xfb, 0xff, 0x0, 0x0, 0xff, 0xff, 0x3, 0x0, 0xff, 0xff,
      0xfc, 0xff, 0x3, 0x0, 0xe4, 0xff, 0x7, 0x0, 0xf8, 0xff, 0x7, 0x0, 0x5,
      0x0, 0xf9, 0xff, 0xf9, 0xff, 0xf9, 0xff, 0x2, 0x0, 0x2, 0x0, 0xfb, 0xff,
      0xfd, 0xff, 0x3, 0x0, 0x3, 0x0, 0x0, 0x0, 0xfe, 0xff, 0xd8, 0xff, 0xf7,
      0xff, 0x1, 0x0, 0xf9, 0xff, 0xfe, 0xff, 0x2, 0x0, 0x4, 0x0, 0x4, 0x0,
      0xfa, 0xff, 0xff, 0xff, 0x1, 0x0, 0x2, 0x0, 0xfe, 0xff, 0x1, 0x0, 0x2,
      0x0, 0xfb, 0xff, 0xe0, 0xff, 0x4, 0x0, 0x1, 0x0, 0x1, 0x0, 0x0, 0x0, 0xff,
      0xff, 0xfe, 0xff, 0xfc, 0xff, 0x0, 0x0, 0xfb, 0xff, 0xff, 0xff, 0xfe,
      0xff, 0x3, 0x0, 0xff, 0xff, 0xfc, 0xff, 0x3, 0x0, 0xe3, 0xff, 0x4, 0x0,
      0xf9, 0xff, 0x5, 0x0, 0x5, 0x0, 0xfa, 0xff, 0xfa, 0xff, 0xfb, 0xff, 0x1,
      0x0, 0x2, 0x0, 0xfc, 0xff, 0xfe, 0xff, 0x2, 0x0, 0x3, 0x0, 0x1, 0x0, 0xfd,
      0xff, 0xd9, 0xff, 0xf8, 0xff, 0x2, 0x0, 0xf9, 0xff, 0xfd, 0xff, 0x3, 0x0,
      0x4, 0x0, 0x3, 0x0, 0xfb, 0xff, 0xfe, 0xff, 0x2, 0x0, 0x2, 0x0, 0xff,
      0xff, 0x1, 0x0, 0x1, 0x0, 0xfd, 0xff, 0xe2, 0xff, 0x6, 0x0, 0xff, 0xff,
      0x3, 0x0, 0x1, 0x0, 0xfd, 0xff, 0xfc, 0xff, 0xfb, 0xff, 0x1, 0x0, 0xfc,
      0xff, 0xfe, 0xff, 0xfd, 0xff, 0x4, 0x0, 0x0, 0x0, 0xfd, 0xff, 0x3, 0x0,
      0xe2, 0xff, 0x3, 0x0, 0xfa, 0xff, 0x3, 0x0, 0x4, 0x0, 0xfb, 0xff, 0xfc,
      0xff, 0xfc, 0xff, 0x0, 0x0, 0x3, 0x0, 0xfd, 0xff, 0xff, 0xff, 0x1, 0x0,
      0x4, 0x0, 0x2, 0x0, 0xfc, 0xff, 0xda, 0xff, 0xfa, 0xff, 0x3, 0x0, 0xfa,
      0xff, 0xfd, 0xff, 0x3, 0x0, 0x4, 0x0, 0x2, 0x0, 0xfc, 0xff, 0xfd, 0xff,
      0x2, 0x0, 0x1, 0x0, 0xff, 0xff, 0x0, 0x0, 0x0, 0x0, 0xfe, 0xff, 0xe5,
      0xff, 0x8, 0x0, 0xfe, 0xff, 0x4, 0x0, 0x2, 0x0, 0xfc, 0xff, 0xfc, 0xff,
      0xf9, 0xff, 0x3, 0x0, 0xfd, 0xff, 0xfe, 0xff, 0xfd, 0xff, 0x4, 0x0, 0x1,
      0x0, 0xfd, 0xff, 0x3, 0x0, 0xe1, 0xff, 0x1, 0x0, 0xfb, 0xff, 0x1, 0x0,
      0x3, 0x0, 0xfc, 0xff, 0xfe, 0xff, 0xfe, 0xff, 0x0, 0x0, 0x2, 0x0, 0xfe,
      0xff, 0x0, 0x0, 0x0, 0x0, 0x3, 0x0, 0x2, 0x0, 0xfc, 0xff, 0xdc, 0xff,
      0xfd, 0xff, 0x3, 0x0, 0xfb, 0xff, 0xfd, 0xff, 0x3, 0x0, 0x3, 0x0, 0x1,
      0x0, 0xfd, 0xff, 0xfc, 0xff, 0x2, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0xff,
      0xff, 0x0, 0x0, 0xe6, 0xff, 0x9, 0x0, 0xfc, 0xff, 0x5, 0x0, 0x3, 0x0,
      0xfb, 0xff, 0xfb, 0xff, 0xf9, 0xff, 0x3, 0x0, 0xff, 0xff, 0xfe, 0xff,
      0xfd, 0xff, 0x4, 0x0, 0x1, 0x0, 0xfd, 0xff, 0x1, 0x0, 0xe0, 0xff, 0xfe,
      0xff, 0xfd, 0xff, 0xff, 0xff, 0x1, 0x0, 0xfe, 0xff, 0xff, 0xff, 0xff,
      0xff, 0xff, 0xff, 0x2, 0x0, 0xff, 0xff, 0x0, 0x0, 0xff, 0xff, 0x3, 0x0,
      0x2, 0x0, 0xfb, 0xff, 0xdf, 0xff, 0xff, 0xff, 0x3, 0x0, 0xfc, 0xff, 0xfe,
      0xff, 0x2, 0x0, 0x2, 0x0, 0x0, 0x0, 0xfe, 0xff, 0xfc, 0xff, 0x1, 0x0, 0x0,
      0x0, 0x1, 0x0, 0x0, 0x0, 0xfe, 0xff, 0x1, 0x0, 0xe6, 0xff, 0x8, 0x0, 0xfb,
      0xff, 0x5, 0x0, 0x3, 0x0, 0xfa, 0xff, 0xfb, 0xff, 0xf9, 0xff, 0x3, 0x0,
      0x0, 0x0, 0xfd, 0xff, 0xfd, 0xff, 0x3, 0x0, 0x2, 0x0, 0xff, 0xff, 0x0,
      0x0, 0xe0, 0xff, 0xfe, 0xff, 0xfe, 0xff, 0xfd, 0xff, 0x1, 0x0, 0xff, 0xff,
      0x1, 0x0, 0x0, 0x0, 0xfe, 0xff, 0x1, 0x0, 0x0, 0x0, 0x1, 0x0, 0xff, 0xff,
      0x2, 0x0, 0x1, 0x0, 0xfb, 0xff, 0xe1, 0xff, 0x2, 0x0, 0x2, 0x0, 0xfe,
      0xff, 0xfe, 0xff, 0x0, 0x0, 0x1, 0x0, 0xff, 0xff, 0xff, 0xff, 0xfd, 0xff,
      0x1, 0x0, 0xff, 0xff, 0x2, 0x0, 0x0, 0x0, 0xfe, 0xff, 0x1, 0x0, 0xe7,
      0xff, 0x7, 0x0, 0xfb, 0xff, 0x4, 0x0, 0x4, 0x0, 0xfa, 0xff, 0xfb, 0xff,
      0xfa, 0xff, 0x2, 0x0, 0x1, 0x0, 0xfd, 0xff, 0xfe, 0xff, 0x3, 0x0, 0x2,
      0x0, 0xff, 0xff, 0xff, 0xff, 0xe0, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xfc,
      0xff, 0x0, 0x0, 0x1, 0x0, 0x2, 0x0, 0x1, 0x0, 0xfd, 0xff, 0x0, 0x0, 0x1,
      0x0, 0x1, 0x0, 0xff, 0xff, 0x2, 0x0, 0x1, 0x0, 0xfc, 0xff, 0xe3, 0xff,
      0x3, 0x0, 0x1, 0x0, 0xff, 0xff, 0xff, 0xff, 0x0, 0x0, 0xff, 0xff, 0xfd,
      0xff, 0x0, 0x0, 0xfd, 0xff, 0x0, 0x0, 0xff, 0xff, 0x2, 0x0, 0x0, 0x0,
      0xfe, 0xff, 0x1, 0x0, 0xe7, 0xff, 0x6, 0x0, 0xfb, 0xff, 0x3, 0x0, 0x3,
      0x0, 0xfb, 0xff, 0xfc, 0xff, 0xfb, 0xff, 0x2, 0x0, 0x2, 0x0, 0xfe, 0xff,
      0xfe, 0xff, 0x2, 0x0, 0x2, 0x0, 0xff, 0xff, 0xfe, 0xff, 0xe0, 0xff, 0xfd,
      0xff, 0x1, 0x0, 0xfb, 0xff, 0xff, 0xff, 0x1, 0x0, 0x3, 0x0, 0x1, 0x0,
      0xfc, 0xff, 0xff, 0xff, 0x1, 0x0, 0x1, 0x0, 0x0, 0x0, 0x1, 0x0, 0x0, 0x0,
      0xfd, 0xff, 0xe5, 0xff, 0x6, 0x0, 0x0, 0x0, 0x1, 0x0, 0x0, 0x0, 0xff,
      0xff, 0xff, 0xff, 0xfd, 0xff, 0x1, 0x0, 0xfd, 0xff, 0xff, 0xff, 0xfe,
      0xff, 0x3, 0x0, 0x0, 0x0, 0xfd, 0xff, 0x1, 0x0, 0xe6, 0xff, 0x4, 0x0,
      0xfb, 0xff, 0x2, 0x0, 0x3, 0x0, 0xfc, 0xff, 0xfd, 0xff, 0xfd, 0xff, 0x1,
      0x0, 0x2, 0x0, 0xfe, 0xff, 0xff, 0xff, 0x1, 0x0, 0x2, 0x0, 0x0, 0x0, 0xfd,
      0xff, 0xe0, 0xff, 0xfe, 0xff, 0x2, 0x0, 0xfb, 0xff, 0xfe, 0xff, 0x2, 0x0,
      0x2, 0x0, 0x1, 0x0, 0xfd, 0xff, 0xfe, 0xff, 0x1, 0x0, 0x1, 0x0, 0x0, 0x0,
      0x1, 0x0, 0x0, 0x0, 0xfd, 0xff, 0xe6, 0xff, 0x6, 0x0, 0xff, 0xff, 0x2,
      0x0, 0x1, 0x0, 0xfd, 0xff, 0xfd, 0xff, 0xfd, 0xff, 0x1, 0x0, 0xfe, 0xff,
      0xff, 0xff, 0xfe, 0xff, 0x3, 0x0, 0x0, 0x0, 0xfe, 0xff, 0x1, 0x0, 0xe5,
      0xff, 0x2, 0x0, 0xfd, 0xff, 0x0, 0x0, 0x2, 0x0, 0xfd, 0xff, 0xfe, 0xff,
      0xfd, 0xff, 0x0, 0x0, 0x1, 0x0, 0xfe, 0xff, 0xff, 0xff, 0x1, 0x0, 0x2,
      0x0, 0x1, 0x0, 0xfc, 0xff, 0xe2, 0xff, 0xff, 0xff, 0x2, 0x0, 0xfc, 0xff,
      0xfe, 0xff, 0x1, 0x0, 0x3, 0x0, 0x1, 0x0, 0xfd, 0xff, 0xfe, 0xff, 0x1,
      0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0xff, 0xff, 0xfe, 0xff, 0xe7, 0xff,
      0x6, 0x0, 0xfe, 0xff, 0x2, 0x0, 0x1, 0x0, 0xfd, 0xff, 0xfd, 0xff, 0xfc,
      0xff, 0x2, 0x0, 0xfe, 0xff, 0xfe, 0xff, 0xfe, 0xff, 0x2, 0x0, 0x1, 0x0,
      0xfe, 0xff, 0x0, 0x0, 0xe4, 0xff, 0x1, 0x0, 0xfd, 0xff, 0x0, 0x0, 0x1,
      0x0, 0xfe, 0xff, 0x0, 0x0, 0xff, 0xff, 0xff, 0xff, 0x1, 0x0, 0xff, 0xff,
      0x0, 0x0, 0x0, 0x0, 0x2, 0x0, 0x1, 0x0, 0xfc, 0xff, 0xe1, 0xff, 0x0, 0x0,
      0x2, 0x0, 0xfd, 0xff, 0xfe, 0xff, 0x0, 0x0, 0x1, 0x0, 0x0, 0x0, 0xfe,
      0xff, 0xfd, 0xff, 0x0, 0x0, 0x0, 0x0, 0x1, 0x0, 0x0, 0x0, 0xff, 0xff,
      0xff, 0xff, 0xe5, 0xff, 0x5, 0x0, 0xfd, 0xff, 0x3, 0x0, 0x2, 0x0, 0xfc,
      0xff, 0xfd, 0xff, 0xfc, 0xff, 0x2, 0x0, 0xff, 0xff, 0xfe, 0xff, 0xfe,
      0xff, 0x2, 0x0, 0x1, 0x0, 0xff, 0xff, 0xff, 0xff, 0xe1, 0xff, 0xff, 0xff,
      0xff, 0xff, 0xff, 0xff, 0x0, 0x0, 0xfe, 0xff, 0x0, 0x0, 0x0, 0x0, 0xff,
      0xff, 0x0, 0x0, 0xff, 0xff, 0x0, 0x0, 0x0, 0x0, 0x1, 0x0, 0x1, 0x0, 0xfd,
      0xff, 0xe1, 0xff, 0x1, 0x0, 0x1, 0x0, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
      0x0, 0x0, 0xff, 0xff, 0x0, 0x0, 0xfe, 0xff, 0x0, 0x0, 0xff, 0xff, 0x1,
      0x0, 0x0, 0x0, 0xff, 0xff, 0x0, 0x0, 0xe4, 0xff, 0x3, 0x0, 0xfe, 0xff,
      0x2, 0x0, 0x1, 0x0, 0xfd, 0xff, 0xfd, 0xff, 0xfd, 0xff, 0x2, 0x0, 0x0,
      0x0, 0xfe, 0xff, 0xfe, 0xff, 0x1, 0x0, 0x1, 0x0, 0xff, 0xff, 0xff, 0xff,
      0xe1, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x0, 0x0, 0xff, 0xff, 0x0,
      0x0, 0xff, 0xff, 0x0, 0x0, 0x0, 0x0, 0xff, 0xff, 0x0, 0x0, 0x0, 0x0, 0x1,
      0x0, 0x0, 0x0, 0xfe, 0xff, 0xe4, 0xff, 0x1, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
      0x0, 0xff, 0xff, 0x0, 0x0, 0xfe, 0xff, 0x0, 0x0, 0xfe, 0xff, 0x0, 0x0,
      0xff, 0xff, 0x1, 0x0, 0x0, 0x0, 0xff, 0xff, 0x0, 0x0, 0xe6, 0xff, 0x1,
      0x0, 0xfe, 0xff, 0x0, 0x0, 0x1, 0x0, 0xfe, 0xff, 0xff, 0xff, 0xfe, 0xff,
      0x1, 0x0, 0x0, 0x0, 0xff, 0xff, 0xff, 0xff, 0x1, 0x0, 0x1, 0x0, 0x0, 0x0,
      0xff, 0xff, 0xe7, 0xff, 0xff, 0xff, 0x0, 0x0, 0xff, 0xff, 0x0, 0x0, 0xff,
      0xff, 0x0, 0x0, 0xff, 0xff, 0x0, 0x0, 0xff, 0xff, 0x0, 0x0, 0x0, 0x0, 0x0,
      0x0, 0x1, 0x0, 0x0, 0x0, 0xff, 0xff, 0xea, 0xff, 0x0, 0x0, 0xff, 0xff,
      0x0, 0x0, 0x0, 0x0, 0xfe, 0xff, 0x0, 0x0, 0xff, 0xff, 0x0, 0x0, 0xff,
      0xff, 0xff, 0xff, 0xff, 0xff, 0x1, 0x0, 0x0, 0x0, 0xff, 0xff, 0xff, 0xff,
      0xed, 0xff, 0x0, 0x0, 0xff, 0xff, 0x0, 0x0, 0x0, 0x0, 0xfe, 0xff, 0xff,
      0xff, 0xff, 0xff, 0x0, 0x0, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x1, 0x0,
      0x1, 0x0, 0x0, 0x0, 0xff, 0xff, 0xef, 0xff, 0xff, 0xff, 0xff, 0xff, 0x0,
      0x0, 0x0, 0x0, 0xff, 0xff, 0x0, 0x0, 0xff, 0xff, 0x0, 0x0, 0xff, 0xff,
      0xff, 0xff, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0xff, 0xff, 0xf3,
      0xff, 0x0, 0x0, 0xff, 0xff, 0x0, 0x0, 0x0, 0x0, 0xff, 0xff, 0x0, 0x0,
      0xff, 0xff, 0x0, 0x0, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x0, 0x0, 0x0,
      0x0, 0x0, 0x0, 0xff, 0xff, 0xf5, 0xff, 0xff, 0xff, 0xff, 0xff, 0x0, 0x0,
      0x0, 0x0, 0xff, 0xff, 0x0, 0x0, 0xff, 0xff, 0x0, 0x0, 0xff, 0xff, 0xff,
      0xff, 0xff, 0xff, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0xff, 0xff}}};

}  // namespace sadie
