# MANIFEST:
#  class Rubber (a supplementary class for a multi-D array class)
#  class NMDArray
#  class Range (extension for NMDArray)

require "NArray"

class Rubber
# a supplementary class for a multi-D array class
# To mark a rubber dimension; no ocntent
end
RUB=Rubber.new     # In practice, will be the only instance of Rubber
NDMAX=7            # Max rank
DSKP=[] ; for i in 0..NDMAX-1; DSKP=DSKP+[Rubber.new]; end

class NMDArray

# NMDArray -- Numeric Multi-Dimensional Array
# 1999/09/08-  T. Horinouchi
# *last modified* : 1999/10/04 (Horinouchi)
#
# sort of inherits (but not technically) NArray
#
#
# class methods
#
#   NMDArray.new(l1,l2,...) initialization (shape only; values are yet to 
#                      be set).   l? is the lengths of ?-th dimension;
#                      The # of the arguments determines the # of dimensions
#
#   NMDArray.indgen(l1,l2,..) similar to NMDArray.new, but linearly 
#                      varing values are set ([0,1,2,...] as a 1D array)
#
# methods
#
#   [],[]=     : subset making and subset alternation, respectively
#                Example 1 (when 3D): [1..2,2,-4] [10..-9]
#                    The # of arguments must be either equal to the rank of
#                    the array (the former case) or one (the latter case).
#                    If the former, the rank of the result of [] is the same
#                    as that of the original array. Use "trim" or "trim!" to
#                    trim the dimensions of length 1.
#                    If the latter, the array is treated as if it were 1D.
#                    Then, the result of [] becomes 1D.
#                Example 2: [{0..-1,2},{-1..0,-1},0]
#                    A step can be specified by using a hash of length 1.
#                    Note that {0..-1,2} is equivalent to {0..-1=>2}.
#                    Specify a range as the key (here, 0..-1), and a step
#                    as the value (here, 2). Negative steps are accepted.
#                    In that case, range.first must be larger than range.last.
#
#   dup        : duplication (shallow)
#   clone      : same as dup
#   dclone     : duplication (deep clone)
#   ddup       : aliased to dclone
#
#   init_another : initialize another object (with the same shape by default)
#
#   prt        : print the content orderly
#
#   each_set!  : Elementwise operation (data values are substituted).
#                Can be applied to selected elements by using if conditioning
#   each_set   : Same as each_element but returns a new object instead of
#                changing the original object.
#
#   transpose  : transpose (dimension order is reversed unless specified)
#
#   rebin      : re-bin the array (without changing the 1D order inside)
#
#   trim trim!    :    eliminate the dimensions of length 1.
#                      trim creates another object; trim! transforms "self".
#                      (Example: a 3*1*2*1 4D array becomes 2D of 3*2)
#
#   max  min      : as you imagine
#   length  size      : total # of elements (length as a 1D array)
#   + - * / **        : numeric operators
#   abs               : absolute values
#   span              : as in NArray (1D numeric array)
#
#   to_a       : into NArray (apply to_a again to get an Array)
#
#   import1d   : import a 1D array. (both Array and NArray are accepted)
#
#   shape      : shape of the array (lengths of dimensions)
#
#   ndims  rank     : rank (# of dimensions) of the array
#
#   sin cos tan exp log log10 sqrt ldexp atan2   :  from Math module
#
#   dif : difference between succesive elements in a dimension (length shrinks)
#
#   cdif : centered difference with upstream and downstream difference at the 
#         ends -- shape preserved
#
#   zcen : average of succesive two elements (length shrinks)
#
#   cycdif : cyclic difference -- shape preserved
#
#   concat : concatenation along a dimension
#
#   l_extend : extension to the "left": dupilicate the first elements
#              (alond a dimension (or dimensions), and so forth)
#   r_extend : extension to the "right": dupilicate the first elements
#   cyc_extend : cyclic extention: duplicate the first elements after the last
#   icyc_extend : inverse cyclic extention: last elements before the first

  ## class methods ##

  def initialize(*lens)
    @lens=lens.dup            # lengths of dimensions
    @nd=@lens.length      #   # of dimension
    @ntot=1 ; for i in 0..@nd-1 ; @ntot*=@lens[i]; end
    @dat=NArray.new(@ntot)
  end

  def NMDArray.indgen(*lens)
    # create a multi-dimensional array filled with with [0.0,1.0,2.0,...,@ntot]
    # --- the linearly values are set one dimensionally
    out=self.new(*lens)
    out.instance_eval{@dat =  NArray.indgen(self.length,0.0)}
    return out
  end

  ## methods ##

#  def dclone
#    out=self.dup
#    return out
#  end

  def dclone
    out=self.dup
    copy_list=['@lens','@dat']
    copy_list.each{ |v|
      out.instance_eval("#{v} = #{v}.clone")
    }
    return out
  end

  alias ddup dclone

  def prt
    if @nd > 1 then
      print("[")
      for n in 0..@ntot-1
	if n-(n/@lens[0])*@lens[0] != @lens[0]-1 then
	  div=", "
	else
	  div=",\n"
	end
	ll=@lens[0]*@lens[1]
	if (n-(n/ll)*ll == ll-1) then
	  div=",\n\n"
	end
	div='' if n == @ntot-1
	print(@dat[n],div)
      end
      print("]\n")
    else
      p @dat
    end
  end

  def each_set!(&block)
    # Elementwise operation. Each element is replaced with the value
    # evaluated by the last statement in the block. This data substitution
    # is suppressed when the value is nil.
    # Note that the operation can be applied to selected elements by using
    # if conditioning (see examples).
    # This is a destructive method (with "!"); the object is modified
    # directly. Use each_set (without "!") to create another object to
    # return the result, leaving the original one intact.
    #
    # Usage: array.each_set!{block}
    # Example: array.each_set!{|x| x*3}     # multiply 3 to all elements
    #          array.each_set!{|x| x*3 if x>0}    #same but only when positive
    #          array.each_set!{|x| s=sin(x); x+s if(cos(x)>0.0)}
    #                       # last value (x+s) is substituted if the condition
    #                       # is satisfied. Here, s is a local variable
    #                       # only in the block ({} introduces a new scope).

    @dat.each_index do
      |i|
      eval = block.call(@dat[i])
      @dat[i]=eval if (eval != nil)
    end
  end

  def each_set(&block)
    out=self.dclone
    out.each_set!(&block)
    return out
  end

  def transpose(order=nil)
    # transpose
    # By default, dimension order is reversed.
    # The order of the dimension can be specified by ORDER:
    #
    # optional input:
    #   order    must be an array that is a reorder of [0..(@nd-1)]
    #            (@nd is the dimension of the array)
    #            Example: [0,2,1] for a 3D array

    if (order==nil) then
      order=(0..@nd-1).to_a.reverse
    else
      if order.length != @nd then
	raise(RuntimeError,'the transpose directory array must be an reorder of [0,1,..,(# od dims)]')
      end
    end

    newlens=@lens.indices(*order)
    newdat=NArray.new(@ntot)   # initialization

    cl=[1]*@nd; for i in 1..@nd-1; cl[i]*=cl[i-1]*@lens[i-1]; end
    nl=[1]*@nd; for i in 1..@nd-1; nl[i]*=nl[i-1]*newlens[i-1]; end

    for n0 in 0..@ntot-1
      n0a=NArray.const(@nd,n0) ; t=n0a/cl
      dc = t - (t/@lens)* @lens
      n1=(dc.indices(*order)*nl).sum
      newdat[n1]=@dat[n0]
    end

    out=self.init_another(order)
    out.setv(newdat)
    return out
  end

  def rebin(*lens)
    # re-bin the array (without changing the 1D order inside)
    #
    # Example: a[3,2,4] -> a[6,4] ; a[4] -> a[4,2] (repeated twice)
    #
    # USAGE: array.rebin(D1,D2,D3,..), or array.rebin([D1,D2,D3,..])
    #        where D? is the lenth of the ?-th dimension.
    #        D1*D2*..*Dn must be an multiple of the total lengh of the
    #        original array (if the multiple is not 1, the contents are
    #        repeated)

    if lens.length == 1 && lens[0].is_a?(Array) then
      lens=lens[0]
    end

    ntot=NArray[*lens].product
    raise(RuntimeError,"dimension lengths you specified have a zero")if ntot==0

    if (ntot-(ntot/@ntot)*@ntot) != 0 then
      raise(RuntimeError,"# of elements must be a multiple of that of the original array")
    end

    out=self.type.new(*lens)

    if (ntot/@ntot) == 1 then
      out.setv(@dat)
    else
      out.setv(@dat.repeat(ntot/@ntot))
    end
    return out
  end

  def insert_dim(*dim_lens)
    # insert dimension(s) to expand the array.
    #
    # Usage: insert_dim([dim,len],[dim,len],..)
    #        The number of the arguments is the number of dimensions you 
    #        want to add. dim is the dimension in the resultant array;
    #        len is its length.
    # Example: 
    #    Suppose that ar is a 3D array with the shape of [10,20,30].
    #    Then, array.dimlens([2,6],[3,7]) returns a %d array with 
    #    the shape [10,6,7,20,30]. dimlens([3,7],[2,6]) yeilds the same result.

    dim_lens=dim_lens.sort    # sort by the first elements of subarrays

    nchg=dim_lens.length
    dim_set=-1 # a initializaton

    nd_=@nd
    dat_=@dat.dup
    ntot_=@ntot
    lens_=@lens.dup

    for i in 0..nchg-1
      dim_len=dim_lens[i]
      raise(RuntimeError,"argument(s) must be array(s)") if !dim_len.is_a?(Array)
      dim=dim_len[0]
      if (dim == dim_set)
	raise(RuntimeError,"#{dim}-th dim is specified twice or more")
      end
      dim_set=dim
      len=dim_len[1]
      raise(RuntimeError,"dimension too large") if (dim>nd_)

      if(dim==0); n0=1; else; n0=NArray[*lens_[0..dim-1]].product; end
      if(dim==nd_); n2=1; else; n2=NArray[*lens_[dim..nd_-1]].product; end
      dat=NArray.new(0)
      for d in 0..n2-1
	dt=NArray[*dat_[(d*n0)..((d+1)*n0-1)]]
	dat=dat.merge(dt.repeat(len))
      end

      # renewal
      if(dim==0)
	lens_=[len]+lens_
      elsif(dim==nd_)
	lens_=lens_+[len]
      else
	lens_=lens_[0..dim-1]+[len]+[dim..nd_-1]
      end
      nd_=nd_+1
      ntot_=ntot_*len
      dat_=dat
    end

    #out=self.dup
    out=self.init_another
    out.instance_eval{@lens = lens_}
    out.instance_eval{@nd = nd_}
    out.instance_eval{@ntot = ntot_}
    out.instance_eval{@dat = dat_}
    return out
  end

  def max; @dat.max; end
  def min; @dat.min; end

  def length; @dat.length; end
  def size; @dat.size; end
  def span(*v); @dat.span(*v); end
  def +(a);r=self.dclone;r.setv(@dat+(a.is_a?(NMDArray)? a.to_a: a));return r;end
  def -(a);r=self.dclone;r.setv(@dat-(a.is_a?(NMDArray)? a.to_a: a));return r;end
  def *(a);r=self.dclone;r.setv(@dat*(a.is_a?(NMDArray)? a.to_a: a));return r;end
  def /(a);r=self.dclone;r.setv(@dat/(a.is_a?(NMDArray)? a.to_a: a));return r;end
  def **(a);r=self.dclone;r.setv(@dat**a);return r;end
  def +@; self.dclone; end
  def -@;r=self.dclone;r.setv(-@dat);return r;end
  def abs;r=self.dclone;r.setv(@dat.abs);return r;end
  def sin;r=self.dclone;r.setv(@dat.sin);return r;end
  def cos;r=self.dclone;r.setv(@dat.cos);return r;end
  def tan;r=self.dclone;r.setv(@dat.tan);return r;end
  def exp;r=self.dclone;r.setv(@dat.exp);return r;end
  def log;r=self.dclone;r.setv(@dat.log);return r;end
  def log10;r=self.dclone;r.setv(@dat.log10);return r;end
  def sqrt;r=self.dclone;r.setv(@dat.sqrt);return r;end
  def ldexp(exp);r=self.dclone;r.setv(@dat.ldexp(exp));return r;end
  def atan2(y);r=self.dclone;r.setv(@dat.atan2(y.to_a));return r;end

  def to_a; @dat.dup; end
  def shape; @lens.dup; end
  def ndims; @nd; end
  alias rank ndims


  def trim!
    @lens.delete(1)
    @nd=@lens.length
  end

  def trim
    out=self.dclone
    out.trim!
    return out
  end

  def import1d(a)
    # Import a 1D array. (both Array and NArray are accepted)
    # trim if too long; tail unchanged if too short
    if !a.is_a?(NArray) then raise(RuntimeError,"Not a NArray"); end
    if a.length <= @dat.length then
      @dat[0..a.length-1]=a
    else
      @dat[0..-1]=a[0..@dat.length-1]
    end
  end


  def [](*idx)
    # multi-dimensional subset (get).
    # NOTICE: if the number of the arguments is 1, the array is treated 
    # as if it were 1D.
    ni=idx.length
    if ni==0 then raise(RuntimeError,"argument(s) is(are) needed"); end
    if ni == 1 && (idx[0].is_a?(Range) || idx[0].is_a?(Numeric)) && @nd!=1 then
      # Short cut for 1D specification
      ii=indxar(*idx)
      il=ii[0].length
      out=self.type.new(il)
      out.setv(@dat[idx[0]])
      return out
    else
      # real multi-D treatment
      ii=indxar(*idx)
      nd=ii.length
      il=[] ; for i in 0..nd-1; il=il+ii[i].length; end
      out=self.init_another((0..nd-1).to_a,il)
      out.setv(@dat.indices(*indx1d(ii)))       # indx is a private method
      return out
    end
  end

  def []=(*idx)
    # multi-dimensional subset (set).
    # NOTICE: if the number of the arguments is 1, the array is treated 
    # as if it were 1D.
    rhs=idx.pop     # idx=idx[0..-2] and rhs=idx[-1]
    ni=idx.length
    if ni==0 then raise(RuntimeError,"argument(s) is(are) needed"); end

    idxar=indx(*idx)
    if (idxar == nil) then
      raise(RuntimeError,"Invalid substitution -- nil subset specification")
    end
    if rhs.is_a?(Array) then
      if idxar.length < rhs.length then
	raise(RuntimeError,"the array at rhs is too short")
      end
      for i in 0..idxar.length-1; @dat[idxar[i]]=rhs[i]; end
    elsif rhs.is_a?(Numeric) 
      for i in 0..idxar.length-1; @dat[idxar[i]]=rhs; end
    else
      raise(RuntimeError,"invalid type: "+rhs.type.to_s)
    end
  end

  def dif(*dims)
    # pairwise difference between succesive elements 
    # with respect to dim[0],dim[1],..-th dimension(s).
    # the lenght of the dimension has a length one less than the original one.

    raise(RuntimeError,'dimension(s) must be specified') if dims.length == 0

    fst=true
    for dim in dims
      if !dim.is_a?(Fixnum)
	raise(RuntimeError,'dimension must be specified as integer')
      end
      if fst then
	out=self[DSKP[dim],1..-1,RUB]-self[DSKP[dim],0..-2,RUB]
	fst=false
      else
	out=out[DSKP[dim],1..-1,RUB]-out[DSKP[dim],0..-2,RUB]
      end
    end

    return out
  end

  def zcen(*dims)
    # pairwise average of succesive two elements 
    # with respect to dim[0],dim[1],..-th dimension(s).
    # the lenght of the dimension has a length one less than the original one.

    raise(RuntimeError,'dimension(s) must be specified') if dims.length == 0

    fst=true
    for dim in dims
      if !dim.is_a?(Fixnum)
	raise(RuntimeError,'dimension must be specified as integer')
      end
      if fst then
	out=(self[DSKP[dim],1..-1,RUB]+self[DSKP[dim],0..-2,RUB])/2
	fst=false
      else
	out=(out[DSKP[dim],1..-1,RUB]+out[DSKP[dim],0..-2,RUB])/2
      end
    end

    return out
  end

  def cdif(*dims)
    # centered difference between every other elements except for at the ends
    # with respect to dim[0],dim[1],..-th dimension(s). length preserved.
    # i.e., [a[1]-a[0],a[2]-a[0],a[3]-a[1],...,a[-1]-a[-3],a[-1]-a[-2]] etc.

    raise(RuntimeError,'dimension(s) must be specified') if dims.length == 0

    out=self.l_extend(*dims).r_extend(*dims)

    for dim in dims
      if !dim.is_a?(Fixnum)
	raise(RuntimeError,'dimension must be specified as integer')
      end
      out=out[DSKP[dim],2..-1,RUB]-out[DSKP[dim],0..-3,RUB]
    end

    return out
  end

  def cycdif(*dims)
    # cyclic difference.
    # with respect to dim[0],dim[1],..-th dimension(s). length preserved.
    # i.e., [a[1]-a[0],a[2]-a[1],...,a[-1]-a[-2],a[0]-a[-1]] etc.

    raise(RuntimeError,'dimension(s) must be specified') if dims.length == 0

    out=self.cyc_extend(*dims)

    for dim in dims
      if !dim.is_a?(Fixnum)
	raise(RuntimeError,'dimension must be specified as integer')
      end
      out=out[DSKP[dim],1..-1,RUB]-out[DSKP[dim],0..-2,RUB]
    end

    return out
  end

  def concat(ar,dim)
    # concatenation along a dimension
    #
    #  ar: array to be concatenated to self
    #      ar''s shape must agree with that of self except for the dim-th
    #      dimension
    #  dim: dimension to concatenate (0,1,2, or, @nd-1)
    alens=ar.shape
    adat=ar.to_a
    raise(RuntimeError,"negative dimension") if (dim < 0)
    tmps=@lens.dup ; tmps.delete_at(dim)
    tmpa=alens.dup ; tmpa.delete_at(dim)
    if( tmps != tmpa ) then
      raise(RuntimeError,"incompatible shapes")
    end
    if(dim==0); n0=1; else; n0=NArray[*@lens[0..dim-1]].product; end
    n1s=@lens[dim]
    n1a=alens[dim]
    n1t=n1s+n1a
    if(dim==@nd-1); n2=1; else; n2=NArray[*@lens[dim+1..@nd-1]].product; end

    lens=@lens.dup
    lens[dim]=n1t

    dat=NArray.new(n0*n1t*n2)
    for d in 0..n2-1
      dat[(d*n0*n1t)..(d*n0*n1t+n0*n1s-1)]=@dat[(d*n0*n1s)..((d+1)*n0*n1s-1)]
      dat[(d*n0*n1t+n0*n1s)..((d+1)*n0*n1t-1)]=adat[(d*n0*n1a)..((d+1)*n0*n1a-1)]
    end

    out=self.dup
    out.instance_eval{@lens=lens}
    out.instance_eval{@ntot=dat.length}
    out.instance_eval{@dat=dat}
    return out
  end

  def l_extend(*dims)
    # extension to the "left": dupilicate the first elements
    # along a dimension (or dimensions)
    # i.e., [a[0],a[0],a[1],a[2],..], etc.
    fst=true
    for dim in dims
      if fst then
	out=self[DSKP[dim],0,RUB].concat(self,dim)
	fst=false
      else
	out=out[DSKP[dim],0,RUB].concat(out,dim)
      end
    end
    return out
  end

  def r_extend(*dims)
    # extension to the "right": dupilicate the last elements
    # along a dimension (or dimensions)
    # i.e., [a[0],a[1],..,a[-2],a[-1],a[-1]], etc.
    fst=true
    for dim in dims
      if fst then
	out=self.concat(self[DSKP[dim],-1,RUB],dim)
	fst=false
      else
	out=out.concat(out[DSKP[dim],-1,RUB],dim)
      end
    end
    return out
  end

  def cyc_extend(*dims)
    # cyclic extention: duplicate the first elements after the last
    # along a dimension (or dimensions)
    # i.e., [a[0],a[1],..,a[-2],a[-1],a[0]], etc.
    fst=true
    for dim in dims
      if fst then
	out=self.concat(self[DSKP[dim],0,RUB],dim)
	fst=false
      else
	out=out.concat(out[DSKP[dim],0,RUB],dim)
      end
    end
    return out
  end

  def cyc_extend_coord(dim,mod)
    # cyclic extention with a mod added -- for coordinate values
    out=self.concat(self[DSKP[dim],0,RUB]+mod,dim)
    return out
  end

  def icyc_extend(*dims)
    # inverse cyclic extention: duplicate the last elements before the last
    # along a dimension (or dimensions)
    # i.e., [a[-1],a[0],a[1],..,a[-2],a[-1]], etc.
    fst=true
    for dim in dims
      if fst then
	out=self[DSKP[dim],-1,RUB].concat(self,dim)
	fst=false
      else
	out=out[DSKP[dim],-1,RUB].concat(out,dim)
      end
    end
    return out
  end

  ######### PRIVATE & PROTECTED METHODS ########

  protected

  def setv(a)
    #(PRIVATE)
    # Import a 1D NAarray for internal usage
    # -- no validity check; assume the same length
    @dat=a.dup
  end

  def init_another(dimmap=nil, lens=nil)
    # initialize another object (with the same shape by default, or the shape 
    # specified by dimmap or by dimmap and lens). 
    # Note that lens is an Array (not init_another(*lens))

    if (dimmap==nil) then
      lens=@lens
    else
      if(lens==nil)
	lens=@lens.indices(*dimmap)
      else
	if (dimmap.length != lens.length) then
	  raise(RuntimeError,"lengths of dimmap and lens do not agree")
	end
	# use the given lens.
	# Then why do we have to care the consistency between dimmap and lens?
	# That's because it's needed in a important descendant
      end
    end

    nd=lens.length
    ntot=1 ; for i in 0..nd-1 ; ntot*=lens[i]; end
    out=self.dup
    out.instance_eval{@lens = lens.dup}
    out.instance_eval{@nd = nd}
    out.instance_eval{@ntot = ntot}
    out.instance_eval{@dat = NArray.new(ntot)}
    return out
  end


  private


  def indxar(*idx)
    #(PRIVATE)
    # indices => vector indices
    # Example: [0..-1,{1..3,2}] of a 3x4 array -> [[0,1,2],[1,3]]

    ni=idx.length

    # rubber dimension interpretation 

    rubfound=nil
    idx.each{|i| rubfound=true if i.type == Rubber}

    if rubfound then

      # (skip a speficified number of dimensions)
      oid=[] ;  idx.each{|i| oid=oid+i.id}

      for i in 1..NDMAX-1
	while ( (irub=idx.index(DSKP[i])) != nil )
	  idx[irub] = 0..-1
	  ncn=i-1
	  case irub
	  when 0
	    for i in 0..ncn-1; idx.unshift(0..-1); end
	  when ni-1
	    for i in 0..ncn-1; idx.push(0..-1); end
	  else
	    idx = idx[0..irub-1] + [0..-1]*ncn + idx[irub..-1]
	  end
	  ni=ni+ncn
	end
      end

      i=0
      while ( (irub=idx.index(DSKP[i])) != nil )
	idx.delete_at(irub)
	ni=ni-1
      end

      # real rubber dimension
      irub=idx.index(RUB)
      if irub != nil then
	# has rubber dimension(s)
	if (irub != idx.rindex(RUB)) then    # Two of more -> ambiguous
	  raise(RuntimeError,"Two or more rubber dimension exists") 
	end
	if ni<=@nd then
	  idx[irub] = 0..-1
	  ncn=@nd-ni     # how many dimensions are contracted
	  if ncn > 0 then
	    case irub
	    when 0
	      for i in 0..ncn-1; idx.unshift(0..-1); end
	    when ni-1
	      for i in 0..ncn-1; idx.push(0..-1); end
	    else
	      idx = idx[0..irub-1] + [0..-1]*ncn + idx[irub..-1]
	    end
	  end
	  ni=@nd
	else
	  idx.delete_at(irub)
	  ni=ni-1
	end
      end
      
    end

    ##

    if ni != @nd && ni != 1 then 
      raise(RuntimeError,"# of arguments ("+ni.to_s+") do not agree with dimension # ("+@nd.to_s+")")
    end

    for i in 0..ni-1
      len= ( ni != 1 ? @lens[i] : @ntot )
      if idx[i].is_a?(Range) then
	idx[i]=idx[i].to_idx(len)
      elsif idx[i].is_a?(Numeric) then
	idx[i]=idx[i] % len    # for negative values
	idx[i]=idx[i].to_a
      elsif idx[i].is_a?(Hash) then
	w=idx[i].to_a
	range=w[0][0]
	step=w[0][1]
	raise(RuntimeError,"Not a range") if (!range.is_a?(Range))
	raise(RuntimeError,"Not a Fixnum") if (!step.is_a?(Fixnum))
	idx[i]=range.to_idx(len,step)
      end
    end
    return idx
  end

  def indx1d(idx)
    #(PRIVATE)
    # vector indices (see indxar) -> 1D indices

    ni=idx.length
    nds=Array.new(ni)
    for i in 0..ni-1; nds[i]=idx[i].length; end    # nds: length of each dim
    ncf=nds.dup ; for i in 1..ni-1; ncf[i]*=ncf[i-1]; end
    if (ncf.min <= 0) then
      return nil
    end
    tot=ncf[-1]
    ncb=Array.new(ni) ; for i in 0..ni-1; ncb[i]=tot/ncf[i]; end

    index=NArray.new(tot); index.fill(0)

    cl=@lens.dup; for i in 1..@nd-1; cl[i]*=cl[i-1]; end  # cumulative lengths

    for d in 0..ni-1
      if d == 0 then
	idxd=idx[d]
      else
	idxd=Array.new(0)
	for i in 0..(nds[d]-1); idxd=idxd+[cl[d-1]*idx[d][i]]*ncf[d-1]; end
      end
      index=index+idxd*ncb[d]
    end

    return index
  end

  def indx(*idx)
    #(PRIVATE)
    # indices of a multi-D array -> indices of its equivalent 1D array
    # Example: [0..1,0..1] of a 3x4 array -> [1,2,5,6]
    indx1d(indxar(*idx))
  end

end

class Range
# extension for NMDArray

  def to_idx(len,step=1)
    # to_a with a negative value alternation.
    #
    # len : length of the dimension
    #
    # first -> first+len if (first < 0);  last -> last+len if (last < 0)
    f=self.first
    l=self.last
    f=f%len if (f<0)
    l=l%len if (l<0)
    l=l-1 if (self.exclude_end?)    # then, one can handle only with f..l
    if (step == 0) then; raise(RuntimeError,"step==0"); end
    if (step != 1) then
      shift=f
      f=0 ; l=(l-shift)/step
    end
    rg=f..l
    if(step == 1) then
      return rg.to_a
    else
      return (NArray.from_a(rg.to_a)*step + shift).to_a
    end
  end

end


####### test for development ########

if __FILE__ == $0

  # < set up for the following tests >

  a=NMDArray.indgen(4,3,2)

  # < test: multi-D vs 1-D treatment >

  #p '+++++'
  #p a[0]
  #p a[-3..-1]
  #p a[-2..-1,0..2,-1]
  #a[-2..-1,0..2,-1]=555
  #p a.to_a
  #a[-3..-1]=[331,332,333]
  #p a.to_a
  #a[1..3,0,0]=[881,882,883]
  #p a.to_a

  # < test: range with step >

  #a[{-1..0,-2},{0..2,2},0]=999
  #p a.to_a

  # < test: rubber dimension >

  #a[0,RUB]=777
  #p a.to_a
  #a[RUB,0,0]=88
  #p a.to_a

  # < test: some methods >

  #p a.to_a.type
  #p a.to_a.to_a.type

  #b=a[0,0..2,0]
  #p b
  #b.trim!
  #p b

  # < test: cloning (duplication) >
  
  #b=a.dup      # shallow copy
  #c=a.dclone   # deep copy
  #p b.to_a
  #a[{-1..0,-2},{0..2,2},0]=99
  #p b.to_a
  #p c.to_a

  # < test: mathematical operations >

  #b=a+a
  #p b.to_a
  #p b.shape
  #b=(-a).abs**2
  #p b.to_a
  
  #p=NArray.indgen(24,0.0,PI/23)
  #a.import1d(p)
  #p a.sin.to_a

  # test insert dim
  a=NMDArray.indgen(3)
  b=a.insert_dim([1,5],[2,2])
  a.prt
  p b
  b.prt
end
