!= 簡単雲モデル - 1 ビン
!
!= simple cloud model with 1 bin
!
! Authors::   Yoshiyuki O. Takahashi
! Version::   $Id: cloud_mars_h2o.f90,v 1.2 2015/01/29 12:06:42 yot Exp $
! Tag Name::  $Name:  $
! Copyright:: Copyright (C) GFD Dennou Club, 2008. All rights reserved.
! License::   See COPYRIGHT[link:../../../COPYRIGHT]
!

module cloud_simple_1bin
  !
  != 簡単雲モデル - 1 ビン
  !
  != simple cloud model with 1 bin
  !
  ! <b>Note that Japanese and English are described in parallel.</b>
  !
  ! 
  !
  ! In this module, H2O cloud on Mars is calculated by use of a simple
  ! cloud model.
  !
  !== Procedures List
  !
!!$  ! RadiationFluxDennouAGCM :: 放射フラックスの計算
!!$  ! ------------            :: ------------
!!$  ! RadiationFluxDennouAGCM :: Calculate radiation flux
  !
  !== NAMELIST
  !
  ! NAMELIST#cloud_simple_nml
  !

  ! モジュール引用 ; USE statements

  !
  ! Kind type parameter
  !
  use dc_types, only: DP, &      ! Double precision.
    &                 STRING, &  ! Strings.
    &                 TOKEN      ! Keywords.

  ! メッセージ出力
  ! Message output
  !
  use dc_message, only: MessageNotify

  ! 格子点設定
  ! Grid points settings
  !
  use gridset, only: imax, & ! 経度格子点数.
                             ! Number of grid points in longitude
    &                jmax, & ! 緯度格子点数.
                             ! Number of grid points in latitude
    &                kmax    ! 鉛直層数.
                             ! Number of vertical level

  implicit none

  private


  ! 公開手続き
  ! Public procedure
  !
  public :: CloudSimple1BinWithIce
  public :: CloudSimple1BinInit



  ! 公開変数
  ! Public variables
  !


  ! 非公開変数
  ! Private variables
  !
  integer,              save :: NThreads
  integer, allocatable, save :: a_ls(:)
  integer, allocatable, save :: a_le(:)


  real(DP), save :: CloudLiqRad
  real(DP), save :: CloudSolRad
  real(DP), save :: CloudNumRatio
  real(DP), save :: CloudNuclRad
  real(DP), save :: CloudNuclDen

    logical, save :: cloud_simple_1bin_inited = .false.
                              ! 初期設定フラグ.
                              ! Initialization flag

  character(*), parameter:: module_name = 'cloud_simple_1bin'
                              ! モジュールの名称.
                              ! Module name
  character(*), parameter:: version = &
    & '$Name:  $' // &
    & '$Id: cloud_mars_h2o.f90,v 1.2 2015/01/29 12:06:42 yot Exp $'
                              ! モジュールのバージョン
                              ! Module version

  !--------------------------------------------------------------------------------------

contains

  !--------------------------------------------------------------------------------------

  subroutine CloudSimple1BinWithIce(               &
    & xyr_Press, xyz_Press, xyr_Height,               & ! (in)
    & xyz_DQH2OLiqDtCum, xyz_DQH2OLiqDtLSC,           & ! (in)
    & xyz_DQH2OSolDtCum, xyz_DQH2OSolDtLSC,           & ! (in)
    & xyz_Temp,                                       & ! (inout)
    & xyz_QH2OVap, xyz_QH2OLiq, xyz_QH2OSol,          & ! (inout)
    & xy_Rain, xy_Snow                                & ! (out)
    & )

    !
    !
    !
    ! simple cloud model with 1 bin
    !

    ! モジュール引用 ; USE statements
    !

    ! 時刻管理
    ! Time control
    !
    use timeset, only: &
      & DelTime, &            ! $ \Delta t $
      & TimeN, &              ! ステップ $ t $ の時刻. Time of step $ t $.
      & TimesetClockStart, TimesetClockStop

    ! ヒストリデータ出力
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoPut

    ! 物理定数設定
    ! Physical constants settings
    !
    use constants, only: &
      & Grav, &               ! $ g $ [m s-2].
                              ! 重力加速度.
                              ! Gravitational acceleration
      & CpDry, &
                              ! $ C_p $ [J kg-1 K-1].
                              ! 乾燥大気の定圧比熱.
                              ! Specific heat of air at constant pressure
      & LatentHeat,       &
                              ! $ L $ [J kg-1] .
                              ! 凝結の潜熱.
                              ! Latent heat of condensation
      & LatentHeatFusion
                              ! $ L $ [J kg-1] .
                              ! 融解の潜熱.
                              ! Latent heat of fusion

    ! 温度の半整数σレベルの補間, 気圧と高度の算出
    ! Interpolate temperature on half sigma level,
    ! and calculate pressure and height
    !
    use auxiliary, only: AuxVars

    ! 質量の補正
    ! Mass fixer
    !
    use mass_fixer, only: MassFixerColumn

    ! 雲関系ルーチン
    ! Cloud-related routines
    !
    use cloud_utils, only : CloudUtils1BinCalcPartProp1D

    ! 大規模凝結 (非対流性凝結)
    ! Large scale condensation
    !
    use lscond, only: LScaleCond1Grid


    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(in   ) :: xyr_Press        ( 0:imax-1, 1:jmax, 0:kmax )
    real(DP), intent(in   ) :: xyz_Press        ( 0:imax-1, 1:jmax, 1:kmax )
    real(DP), intent(in   ) :: xyr_Height       ( 0:imax-1, 1:jmax, 0:kmax )
    real(DP), intent(in   ) :: xyz_DQH2OLiqDtCum( 0:imax-1, 1:jmax, 1:kmax )
    real(DP), intent(in   ) :: xyz_DQH2OLiqDtLSC( 0:imax-1, 1:jmax, 1:kmax )
    real(DP), intent(in   ) :: xyz_DQH2OSolDtCum( 0:imax-1, 1:jmax, 1:kmax )
    real(DP), intent(in   ) :: xyz_DQH2OSolDtLSC( 0:imax-1, 1:jmax, 1:kmax )
    real(DP), intent(inout) :: xyz_Temp         ( 0:imax-1, 1:jmax, 1:kmax )
    real(DP), intent(inout) :: xyz_QH2OVap      ( 0:imax-1, 1:jmax, 1:kmax )
    real(DP), intent(inout) :: xyz_QH2OLiq      ( 0:imax-1, 1:jmax, 1:kmax )
    real(DP), intent(inout) :: xyz_QH2OSol      ( 0:imax-1, 1:jmax, 1:kmax )
    real(DP), intent(out  ) :: xy_Rain          ( 0:imax-1, 1:jmax )
    real(DP), intent(out  ) :: xy_Snow          ( 0:imax-1, 1:jmax )


    ! 作業変数
    ! Work variables
    !
    real(DP) :: TimeStep

    real(DP) :: xyz_LiqPartDen  (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_LiqPartRad  (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_SolPartDen  (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_SolPartRad  (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_DelAtmMass  (0:imax-1, 1:jmax, 1:kmax)


    real(DP):: xyz_TempB           (0:imax-1, 1:jmax, 1:kmax)
                                   ! 調節前の温度.
                                   ! Temperature before adjustment
    real(DP):: xyz_QH2OVapB        (0:imax-1, 1:jmax, 1:kmax)
    real(DP):: xyz_QH2OLiqB        (0:imax-1, 1:jmax, 1:kmax)
    real(DP):: xyz_QH2OSolB        (0:imax-1, 1:jmax, 1:kmax)

    real(DP):: xyz_DTempDt         (0:imax-1, 1:jmax, 1:kmax)
                                   ! 温度変化率.
                                   ! Temperature tendency

    real(DP) :: CondMaterialDen
    real(DP) :: CloudRad

    real(DP):: r_Press           (0:kmax)
    real(DP):: z_Press           (1:kmax)
    real(DP):: z_Temp            (1:kmax)
    real(DP):: r_Height          (0:kmax)

    real(DP):: z_DelAtmMass      (1:kmax)

    real(DP) :: z_QH2OVap   (1:kmax)
    real(DP) :: z_QH2OLiq   (1:kmax)
    real(DP) :: z_QH2OSol   (1:kmax)
    real(DP) :: z_LiqPartDen(1:kmax)
    real(DP) :: z_LiqPartRad(1:kmax)
    real(DP) :: z_SolPartDen(1:kmax)
    real(DP) :: z_SolPartRad(1:kmax)
    real(DP) :: z_DelNuclNum(1:kmax)

    real(DP) :: r_LiqGravSedMassFlux  (0:kmax)
    real(DP) :: r_SolGravSedMassFlux  (0:kmax)
    real(DP) :: xyr_LiqGravSedMassFlux(0:imax-1, 1:jmax, 0:kmax)
    real(DP) :: xyr_SolGravSedMassFlux(0:imax-1, 1:jmax, 0:kmax)

    real(DP) :: DQH2OLiqDt
    real(DP) :: DQH2OSolDt

    real(DP) :: xy_TotalEneB(0:imax-1, 1:jmax)
    real(DP) :: xy_TotalEneA(0:imax-1, 1:jmax)
    real(DP) :: TotalEneB
    real(DP) :: TotalEneA
    real(DP) :: xy_H2OMassB(0:imax-1, 1:jmax)
    real(DP) :: xy_H2OMassA(0:imax-1, 1:jmax)
    real(DP) :: H2OMassB
    real(DP) :: H2OMassA
    real(DP) :: Ratio

    integer:: iThread

    integer:: i              ! 経度方向に回る DO ループ用作業変数
                             ! Work variables for DO loop in longitude
    integer:: j              ! 緯度方向に回る DO ループ用作業変数
                             ! Work variables for DO loop in latitude
    integer:: k              ! 鉛直方向に回る DO ループ用作業変数
                             ! Work variables for DO loop in vertical direction
    integer:: l


    ! 実行文 ; Executable statement
    !

    ! 初期化
    ! Initialization
    !
    if ( .not. cloud_simple_1bin_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    ! 計算時間計測開始
    ! Start measurement of computation time
    !
    call TimesetClockStart( module_name )


    ! add source
    !
    xyz_QH2OLiq = xyz_QH2OLiq &
      & + ( xyz_DQH2OLiqDtCum + xyz_DQH2OLiqDtLSC ) * 2.0_DP * DelTime
    xyz_QH2OSol = xyz_QH2OSol &
      & + ( xyz_DQH2OSolDtCum + xyz_DQH2OSolDtLSC ) * 2.0_DP * DelTime


    xy_Rain = 0.0_DP
    xy_Snow = 0.0_DP


    ! Store variables
    !
    xyz_TempB    = xyz_Temp
    xyz_QH2OVapB = xyz_QH2OVap
    xyz_QH2OLiqB = xyz_QH2OLiq
    xyz_QH2OSolB = xyz_QH2OSol


    TimeStep = 2.0_DP * DelTime

    CondMaterialDen =  1.0e3_DP
    CloudRad        = -1.0_DP

    !$OMP PARALLEL DEFAULT(PRIVATE) &
    !$OMP SHARED( &
    !$OMP         NThreads, TimeStep, a_ls, a_le, &
    !$OMP         imax, jmax, kmax, &
    !$OMP         Grav, &
    !$OMP         CloudNumRatio, &
    !$OMP         CondMaterialDen, CloudRad, CloudNuclRad, CloudNuclDen, &
    !$OMP         xyr_Press, xyz_Press, xyr_Height, &
    !$OMP         xyz_DelAtmMass, &
    !$OMP         xyz_Temp, xyz_QH2OVap, xyz_QH2OLiq, xyz_QH2OSol, &
    !$OMP         xyz_LiqPartRad, xyz_LiqPartDen, &
    !$OMP         xyz_SolPartRad, xyz_SolPartDen, &
    !$OMP         xyr_LiqGravSedMassFlux, xyr_SolGravSedMassFlux &
    !$OMP       )

    !$OMP DO
    do iThread = 0, NThreads-1

      do l = a_ls(iThread), a_le(iThread)

        i = mod( (l-1), imax ) + 1 - 1
        j = int( (l-1) / imax ) + 1

        do k = 0, kmax
          r_Press (k) = xyr_Press (i,j,k)
          r_Height(k) = xyr_Height(i,j,k)
        end do
        do k = 1, kmax
          z_Press     (k) = xyz_Press     (i,j,k)
          z_Temp      (k) = xyz_Temp      (i,j,k)
          z_QH2OVap   (k) = xyz_QH2OVap   (i,j,k)
          z_QH2OLiq   (k) = xyz_QH2OLiq   (i,j,k)
          z_QH2OSol   (k) = xyz_QH2OSol   (i,j,k)
        end do


        do k = 1, kmax
          z_DelAtmMass(k) = ( r_Press(k-1) - r_Press(k) ) / Grav
        end do

        ! saturation adjustment
        do k = 1, kmax
          call LScaleCond1Grid(  &
            & z_Temp   (k), & ! (inout)
            & z_QH2OVap(k), & ! (inout)
            & z_QH2OLiq(k), & ! (inout)
            & z_QH2OSol(k), & ! (inout)
            & z_Press  (k), & ! (in)
            & DQH2OLiqDt, DQH2OSolDt  &  ! (out)
            & )
        end do

        z_DelNuclNum    = z_DelAtmMass * CloudNumRatio
        !
        call CloudUtils1BinCalcPartProp1D(             &
          & CondMaterialDen, CloudRad,                 & ! (in )
          & CloudNuclRad, CloudNuclDen,                & ! (in )
          & z_DelAtmMass, z_QH2OLiq, z_DelNuclNum,     & ! (in )
          & z_LiqPartRad, z_LiqPartDen                 & ! (out)
          & )
        call CloudUtils1BinCalcPartProp1D(             &
          & CondMaterialDen, CloudRad,                 & ! (in )
          & CloudNuclRad, CloudNuclDen,                & ! (in )
          & z_DelAtmMass, z_QH2OSol, z_DelNuclNum,     & ! (in )
          & z_SolPartRad, z_SolPartDen                 & ! (out)
          & )


        ! 重力沈降過程
        ! Gravitational sedimentation process
        !
        call CloudSimple1BinCalcSedMassFlux1D( &
          & TimeStep,                                     & ! (in   )
          & r_Press, z_Press, r_Height,                   & ! (in   )
          & z_LiqPartDen, z_LiqPartRad,                   & ! (in   )
          & z_SolPartDen, z_SolPartRad,                   & ! (in   )
          & z_Temp,                                       & ! (inout)
          & z_QH2OVap, z_QH2OLiq, z_QH2OSol,              & ! (inout)
          & r_LiqGravSedMassFlux, r_SolGravSedMassFlux    & ! (out  )
          & )

        do k = 1, kmax
          xyz_DelAtmMass(i,j,k) = z_DelAtmMass(k)
        end do
        do k = 1, kmax
          xyz_LiqPartDen(i,j,k) = z_LiqPartDen(k)
          xyz_LiqPartRad(i,j,k) = z_LiqPartRad(k)
          xyz_SolPartDen(i,j,k) = z_SolPartDen(k)
          xyz_SolPartRad(i,j,k) = z_SolPartRad(k)
        end do
        do k = 1, kmax
          xyz_Temp      (i,j,k) = z_Temp      (k)
          xyz_QH2OVap   (i,j,k) = z_QH2OVap   (k)
          xyz_QH2OLiq   (i,j,k) = z_QH2OLiq   (k)
          xyz_QH2OSol   (i,j,k) = z_QH2OSol   (k)
        end do
        do k = 0, kmax
          xyr_LiqGravSedMassFlux(i,j,k) = r_LiqGravSedMassFlux(k)
          xyr_SolGravSedMassFlux(i,j,k) = r_SolGravSedMassFlux(k)
        end do

      end do
    end do
    !$OMP END DO
    !$OMP END PARALLEL


    xy_Rain = - xyr_LiqGravSedMassFlux(:,:,0)
    xy_Snow = - xyr_SolGravSedMassFlux(:,:,0)



    ! check
    xy_TotalEneB = 0.0_DP
    xy_TotalEneA = 0.0_DP
    do k = kmax, 1, -1
      xy_TotalEneB = xy_TotalEneB                        &
        & + (   CpDry            * xyz_TempB   (:,:,k)   &
        &     + LatentHeat       * xyz_QH2OVapB(:,:,k)   &
        &     - LatentHeatFusion * xyz_QH2OSolB(:,:,k) ) &
        &   * xyz_DelAtmMass(:,:,k)
      xy_TotalEneA = xy_TotalEneA                        &
        & + (   CpDry            * xyz_Temp    (:,:,k)   &
        &     + LatentHeat       * xyz_QH2OVap (:,:,k)   &
        &     - LatentHeatFusion * xyz_QH2OSol (:,:,k) ) &
        &   * xyz_DelAtmMass(:,:,k)
    end do
    xy_TotalEneA = xy_TotalEneA &
      & - LatentHeatFusion * xy_Snow * TimeStep
    do j = 1, jmax
      do i = 0, imax-1
        TotalEneB = xy_TotalEneB(i,j)
        TotalEneA = xy_TotalEneA(i,j)
        Ratio = abs( TotalEneB - TotalEneA ) / TotalEneB
        if ( Ratio > 1.0e-10_DP ) then
          call MessageNotify( 'M', module_name, &
            & 'Phase change does not conserve energy, %f, at (%d,%d).', &
            & d = (/ Ratio /), i = (/ i, j /) )
        end if
      end do
    end do
    !
    xy_H2OMassB = 0.0_DP
    xy_H2OMassA = 0.0_DP
    do k = kmax, 1, -1
      xy_H2OMassB = xy_H2OMassB       &
        & + (   xyz_QH2OVapB(:,:,k)   &
        &     + xyz_QH2OLiqB(:,:,k)   &
        &     + xyz_QH2OSolB(:,:,k) ) &
        &   * xyz_DelAtmMass(:,:,k)
      xy_H2OMassA = xy_H2OMassA       &
        & + (   xyz_QH2OVap (:,:,k)   &
        &     + xyz_QH2OLiq (:,:,k)   &
        &     + xyz_QH2OSol (:,:,k) ) &
        &   * xyz_DelAtmMass(:,:,k)
    end do
    xy_H2OMassA = xy_H2OMassA &
      & + xy_Rain * TimeStep  &
      & + xy_Snow * TimeStep
    do j = 1, jmax
      do i = 0, imax-1
        H2OMassB = xy_H2OMassB(i,j)
        H2OMassA = xy_H2OMassA(i,j)
        if ( H2OMassB > 0.0_DP ) then
          Ratio = abs( H2OMassB - H2OMassA ) / H2OMassB
        else
          Ratio = 0.0_DP
        end if
        if ( Ratio > 1.0e-10_DP ) then
          call MessageNotify( 'M', module_name, &
            & 'Phase change does not conserve mass, %f, at (%d,%d).', &
            & d = (/ Ratio /), i = (/ i, j /) )
        end if
      end do
    end do


    ! ヒストリデータ出力
    ! History data output
    !
!!$    xyz_DTempDt = ( xyz_Temp - xyz_TempB ) / ( 2.0_DP * DelTime )
!!$    call HistoryAutoPut( TimeN, 'DTempDtMajCompPhaseChange', xyz_DTempDt )

    call HistoryAutoPut( TimeN, 'LiqCloudRadiusForGravSed', xyz_LiqPartRad )
    call HistoryAutoPut( TimeN, 'SolCloudRadiusForGravSed', xyz_SolPartRad )

    do j = 1, jmax
      do i = 0, imax-1
        do k = 0, kmax
          r_Press (k) = xyr_Press (i,j,k)
        end do
        do k = 1, kmax
          z_DelAtmMass(k) = ( r_Press(k-1) - r_Press(k) ) / Grav
        end do
        do k = 1, kmax
          z_QH2OLiq   (k) = xyz_QH2OLiq   (i,j,k)
          z_QH2OSol   (k) = xyz_QH2OSol   (i,j,k)
        end do
        z_DelNuclNum    = z_DelAtmMass * CloudNumRatio
        !
        call CloudUtils1BinCalcPartProp1D(             &
          & CondMaterialDen, CloudRad,                 & ! (in )
          & CloudNuclRad, CloudNuclDen,                & ! (in )
          & z_DelAtmMass, z_QH2OLiq, z_DelNuclNum,     & ! (in )
          & z_LiqPartRad, z_LiqPartDen                 & ! (out)
          & )
        call CloudUtils1BinCalcPartProp1D(             &
          & CondMaterialDen, CloudRad,                 & ! (in )
          & CloudNuclRad, CloudNuclDen,                & ! (in )
          & z_DelAtmMass, z_QH2OSol, z_DelNuclNum,     & ! (in )
          & z_SolPartRad, z_SolPartDen                 & ! (out)
          & )
        do k = 1, kmax
          xyz_LiqPartDen(i,j,k) = z_LiqPartDen(k)
          xyz_LiqPartRad(i,j,k) = z_LiqPartRad(k)
          xyz_SolPartDen(i,j,k) = z_SolPartDen(k)
          xyz_SolPartRad(i,j,k) = z_SolPartRad(k)
        end do
      end do
    end do
    call HistoryAutoPut( TimeN, 'LiqCloudRadius', xyz_LiqPartRad )
    call HistoryAutoPut( TimeN, 'SolCloudRadius', xyz_SolPartRad )


    ! 計算時間計測一時停止
    ! Pause measurement of computation time
    !
    call TimesetClockStop( module_name )

  end subroutine CloudSimple1BinWithIce

  !--------------------------------------------------------------------------------------

  subroutine BKSerial_20190921_CloudSimple1BinWithIce(               &
    & xyr_Press, xyz_Press, xyr_Height,               & ! (in)
    & xyz_DQH2OLiqDtCum, xyz_DQH2OLiqDtLSC,           & ! (in)
    & xyz_DQH2OSolDtCum, xyz_DQH2OSolDtLSC,           & ! (in)
    & xyz_Temp,                                       & ! (inout)
    & xyz_QH2OVap, xyz_QH2OLiq, xyz_QH2OSol,          & ! (inout)
    & xy_Rain, xy_Snow                                & ! (out)
    & )

    !
    !
    !
    ! simple cloud model with 1 bin
    !

    ! モジュール引用 ; USE statements
    !

    ! 時刻管理
    ! Time control
    !
    use timeset, only: &
      & DelTime, &            ! $ \Delta t $
      & TimeN, &              ! ステップ $ t $ の時刻. Time of step $ t $.
      & TimesetClockStart, TimesetClockStop

    ! ヒストリデータ出力
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoPut

    ! 物理定数設定
    ! Physical constants settings
    !
    use constants, only: &
      & Grav, &               ! $ g $ [m s-2].
      ! 重力加速度.
      ! Gravitational acceleration
      & CpDry, &
      ! $ C_p $ [J kg-1 K-1].
      ! 乾燥大気の定圧比熱.
      ! Specific heat of air at constant pressure
      & LatentHeat,       &
      ! $ L $ [J kg-1] .
      ! 凝結の潜熱.
      ! Latent heat of condensation
      & LatentHeatFusion
      ! $ L $ [J kg-1] .
      ! 融解の潜熱.
      ! Latent heat of fusion

    ! 温度の半整数σレベルの補間, 気圧と高度の算出
    ! Interpolate temperature on half sigma level,
    ! and calculate pressure and height
    !
    use auxiliary, only: AuxVars

    ! 質量の補正
    ! Mass fixer
    !
    use mass_fixer, only: MassFixerColumn

    ! 雲関系ルーチン
    ! Cloud-related routines
    !
    use cloud_utils, only : CloudUtils1BinCalcPartProp

    ! 大規模凝結 (非対流性凝結)
    ! Large scale condensation
    !
    use lscond, only: LScaleCond1Grid


    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(in   ) :: xyr_Press        ( 0:imax-1, 1:jmax, 0:kmax )
    real(DP), intent(in   ) :: xyz_Press        ( 0:imax-1, 1:jmax, 1:kmax )
    real(DP), intent(in   ) :: xyr_Height       ( 0:imax-1, 1:jmax, 0:kmax )
    real(DP), intent(in   ) :: xyz_DQH2OLiqDtCum( 0:imax-1, 1:jmax, 1:kmax )
    real(DP), intent(in   ) :: xyz_DQH2OLiqDtLSC( 0:imax-1, 1:jmax, 1:kmax )
    real(DP), intent(in   ) :: xyz_DQH2OSolDtCum( 0:imax-1, 1:jmax, 1:kmax )
    real(DP), intent(in   ) :: xyz_DQH2OSolDtLSC( 0:imax-1, 1:jmax, 1:kmax )
    real(DP), intent(inout) :: xyz_Temp         ( 0:imax-1, 1:jmax, 1:kmax )
    real(DP), intent(inout) :: xyz_QH2OVap      ( 0:imax-1, 1:jmax, 1:kmax )
    real(DP), intent(inout) :: xyz_QH2OLiq      ( 0:imax-1, 1:jmax, 1:kmax )
    real(DP), intent(inout) :: xyz_QH2OSol      ( 0:imax-1, 1:jmax, 1:kmax )
    real(DP), intent(out  ) :: xy_Rain          ( 0:imax-1, 1:jmax )
    real(DP), intent(out  ) :: xy_Snow          ( 0:imax-1, 1:jmax )


    ! 作業変数
    ! Work variables
    !
    real(DP) :: TimeStep

    real(DP) :: xyz_LiqPartDen  (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_LiqPartRad  (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_SolPartDen  (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_SolPartRad  (0:imax-1, 1:jmax, 1:kmax)


    real(DP):: xyz_TempB           (0:imax-1, 1:jmax, 1:kmax)
    ! 調節前の温度.
    ! Temperature before adjustment
    real(DP):: xyz_QH2OVapB        (0:imax-1, 1:jmax, 1:kmax)
    real(DP):: xyz_QH2OLiqB        (0:imax-1, 1:jmax, 1:kmax)
    real(DP):: xyz_QH2OSolB        (0:imax-1, 1:jmax, 1:kmax)

    real(DP):: xyz_DelAtmMass      (0:imax-1, 1:jmax, 1:kmax)

    real(DP):: xyz_DTempDt         (0:imax-1, 1:jmax, 1:kmax)
    ! 温度変化率.
    ! Temperature tendency

    real(DP) :: CondMaterialDen
    real(DP) :: CloudRad
    real(DP) :: CloudNuclRad
    real(DP) :: CloudNuclDen
    real(DP) :: xyz_DelNuclNum (0:imax-1, 1:jmax, 1:kmax)

    real(DP):: r_Press           (0:kmax)
    real(DP):: z_Press           (1:kmax)
    real(DP):: z_Temp            (1:kmax)
    real(DP):: r_Height          (0:kmax)

    real(DP) :: z_QH2OVap   (1:kmax)
    real(DP) :: z_QH2OLiq   (1:kmax)
    real(DP) :: z_QH2OSol   (1:kmax)
    real(DP) :: z_LiqPartDen(1:kmax)
    real(DP) :: z_LiqPartRad(1:kmax)
    real(DP) :: z_SolPartDen(1:kmax)
    real(DP) :: z_SolPartRad(1:kmax)

    real(DP) :: r_LiqGravSedMassFlux  (0:kmax)
    real(DP) :: r_SolGravSedMassFlux  (0:kmax)
    real(DP) :: xyr_LiqGravSedMassFlux(0:imax-1, 1:jmax, 0:kmax)
    real(DP) :: xyr_SolGravSedMassFlux(0:imax-1, 1:jmax, 0:kmax)

    real(DP) :: DQH2OLiqDt
    real(DP) :: DQH2OSolDt

    real(DP) :: xy_TotalEneB(0:imax-1, 1:jmax)
    real(DP) :: xy_TotalEneA(0:imax-1, 1:jmax)
    real(DP) :: TotalEneB
    real(DP) :: TotalEneA
    real(DP) :: xy_H2OMassB(0:imax-1, 1:jmax)
    real(DP) :: xy_H2OMassA(0:imax-1, 1:jmax)
    real(DP) :: H2OMassB
    real(DP) :: H2OMassA
    real(DP) :: Ratio

    integer:: i               ! 経度方向に回る DO ループ用作業変数
    ! Work variables for DO loop in longitude
    integer:: j               ! 緯度方向に回る DO ループ用作業変数
    ! Work variables for DO loop in latitude
    integer:: k               ! 鉛直方向に回る DO ループ用作業変数
    ! Work variables for DO loop in vertical direction


    ! 実行文 ; Executable statement
    !

    ! 初期化
    ! Initialization
    !
    if ( .not. cloud_simple_1bin_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    ! 計算時間計測開始
    ! Start measurement of computation time
    !
    call TimesetClockStart( module_name )


    ! add source
    !
    xyz_QH2OLiq = xyz_QH2OLiq &
      & + ( xyz_DQH2OLiqDtCum + xyz_DQH2OLiqDtLSC ) * 2.0_DP * DelTime
    xyz_QH2OSol = xyz_QH2OSol &
      & + ( xyz_DQH2OSolDtCum + xyz_DQH2OSolDtLSC ) * 2.0_DP * DelTime


    xy_Rain = 0.0_DP
    xy_Snow = 0.0_DP


    ! Store variables
    !
    xyz_TempB    = xyz_Temp
    xyz_QH2OVapB = xyz_QH2OVap
    xyz_QH2OLiqB = xyz_QH2OLiq
    xyz_QH2OSolB = xyz_QH2OSol


    TimeStep = 2.0_DP * DelTime


    do k = 1, kmax
      xyz_DelAtmMass(:,:,k) = ( xyr_Press(:,:,k-1) - xyr_Press(:,:,k) ) / Grav
    end do

    ! saturation adjustment
    do k = 1, kmax
      do j = 1, jmax
        do i = 0, imax-1
          call LScaleCond1Grid(  &
            & xyz_Temp   (i,j,k), & ! (inout)
            & xyz_QH2OVap(i,j,k), & ! (inout)
            & xyz_QH2OLiq(i,j,k), & ! (inout)
            & xyz_QH2OSol(i,j,k), & ! (inout)
            & xyz_Press  (i,j,k), & ! (in)
            & DQH2OLiqDt, DQH2OSolDt  &  ! (out)
            & )
        end do
      end do
    end do


    CondMaterialDen =  1.0e3_DP
    CloudRad        = -1.0_DP
    CloudNuclRad    =  1.0e-6_DP  ! dummy
    CloudNuclDen    =  3.0e3_DP   ! dummy
    xyz_DelNuclNum  = xyz_DelAtmMass * CloudNumRatio
    !
    call CloudUtils1BinCalcPartProp( &
      & CondMaterialDen, CloudRad,                 & ! (in )
      & CloudNuclRad, CloudNuclDen,                & ! (in )
      & xyr_Press,                                 & ! (in )
      & xyz_DelNuclNum,                            & ! (in )
      & xyz_QH2OLiq,                               & ! (in )
      & xyz_LiqPartRad, xyz_LiqPartDen             & ! (out)
      & )
    call CloudUtils1BinCalcPartProp( &
      & CondMaterialDen, CloudRad,                 & ! (in )
      & CloudNuclRad, CloudNuclDen,                & ! (in )
      & xyr_Press,                                 & ! (in )
      & xyz_DelNuclNum,                            & ! (in )
      & xyz_QH2OSol,                               & ! (in )
      & xyz_SolPartRad, xyz_SolPartDen             & ! (out)
      & )


    do j = 1, jmax
      do i = 0, imax-1

        do k = 0, kmax
          r_Press (k) = xyr_Press (i,j,k)
          r_Height(k) = xyr_Height(i,j,k)
        end do
        do k = 1, kmax
          z_Press     (k) = xyz_Press     (i,j,k)
          z_LiqPartDen(k) = xyz_LiqPartDen(i,j,k)
          z_LiqPartRad(k) = xyz_LiqPartRad(i,j,k)
          z_SolPartDen(k) = xyz_SolPartDen(i,j,k)
          z_SolPartRad(k) = xyz_SolPartRad(i,j,k)
          z_Temp      (k) = xyz_Temp      (i,j,k)
          z_QH2OVap   (k) = xyz_QH2OVap   (i,j,k)
          z_QH2OLiq   (k) = xyz_QH2OLiq   (i,j,k)
          z_QH2OSol   (k) = xyz_QH2OSol   (i,j,k)
        end do

        ! 重力沈降過程
        ! Gravitational sedimentation process
        !
        call CloudSimple1BinCalcSedMassFlux1D( &
          & TimeStep,                                     & ! (in   )
          & r_Press, z_Press, r_Height,                   & ! (in   )
          & z_LiqPartDen, z_LiqPartRad,                   & ! (in   )
          & z_SolPartDen, z_SolPartRad,                   & ! (in   )
          & z_Temp,                                       & ! (inout)
          & z_QH2OVap, z_QH2OLiq, z_QH2OSol,              & ! (inout)
          & r_LiqGravSedMassFlux, r_SolGravSedMassFlux    & ! (out  )
          & )

        do k = 1, kmax
          xyz_Temp      (i,j,k) = z_Temp      (k)
          xyz_QH2OVap   (i,j,k) = z_QH2OVap   (k)
          xyz_QH2OLiq   (i,j,k) = z_QH2OLiq   (k)
          xyz_QH2OSol   (i,j,k) = z_QH2OSol   (k)
        end do
        do k = 0, kmax
          xyr_LiqGravSedMassFlux(i,j,k) = r_LiqGravSedMassFlux(k)
          xyr_SolGravSedMassFlux(i,j,k) = r_SolGravSedMassFlux(k)
        end do

      end do
    end do

    xy_Rain = - xyr_LiqGravSedMassFlux(:,:,0)
    xy_Snow = - xyr_SolGravSedMassFlux(:,:,0)



    ! check
    xy_TotalEneB = 0.0_DP
    xy_TotalEneA = 0.0_DP
    do k = kmax, 1, -1
      xy_TotalEneB = xy_TotalEneB                        &
        & + (   CpDry            * xyz_TempB   (:,:,k)   &
        &     + LatentHeat       * xyz_QH2OVapB(:,:,k)   &
        &     - LatentHeatFusion * xyz_QH2OSolB(:,:,k) ) &
        &   * xyz_DelAtmMass(:,:,k)
      xy_TotalEneA = xy_TotalEneA                        &
        & + (   CpDry            * xyz_Temp    (:,:,k)   &
        &     + LatentHeat       * xyz_QH2OVap (:,:,k)   &
        &     - LatentHeatFusion * xyz_QH2OSol (:,:,k) ) &
        &   * xyz_DelAtmMass(:,:,k)
    end do
    xy_TotalEneA = xy_TotalEneA &
      & - LatentHeatFusion * xy_Snow * TimeStep
    do j = 1, jmax
      do i = 0, imax-1
        TotalEneB = xy_TotalEneB(i,j)
        TotalEneA = xy_TotalEneA(i,j)
        Ratio = abs( TotalEneB - TotalEneA ) / TotalEneB
        if ( Ratio > 1.0e-10_DP ) then
          call MessageNotify( 'M', module_name, &
            & 'Phase change does not conserve energy, %f, at (%d,%d).', &
            & d = (/ Ratio /), i = (/ i, j /) )
        end if
      end do
    end do
    !
    xy_H2OMassB = 0.0_DP
    xy_H2OMassA = 0.0_DP
    do k = kmax, 1, -1
      xy_H2OMassB = xy_H2OMassB       &
        & + (   xyz_QH2OVapB(:,:,k)   &
        &     + xyz_QH2OLiqB(:,:,k)   &
        &     + xyz_QH2OSolB(:,:,k) ) &
        &   * xyz_DelAtmMass(:,:,k)
      xy_H2OMassA = xy_H2OMassA       &
        & + (   xyz_QH2OVap (:,:,k)   &
        &     + xyz_QH2OLiq (:,:,k)   &
        &     + xyz_QH2OSol (:,:,k) ) &
        &   * xyz_DelAtmMass(:,:,k)
    end do
    xy_H2OMassA = xy_H2OMassA &
      & + xy_Rain * TimeStep  &
      & + xy_Snow * TimeStep
    do j = 1, jmax
      do i = 0, imax-1
        H2OMassB = xy_H2OMassB(i,j)
        H2OMassA = xy_H2OMassA(i,j)
        if ( H2OMassB > 0.0_DP ) then
          Ratio = abs( H2OMassB - H2OMassA ) / H2OMassB
        else
          Ratio = 0.0_DP
        end if
        if ( Ratio > 1.0e-10_DP ) then
          call MessageNotify( 'M', module_name, &
            & 'Phase change does not conserve mass, %f, at (%d,%d).', &
            & d = (/ Ratio /), i = (/ i, j /) )
        end if
      end do
    end do


    ! ヒストリデータ出力
    ! History data output
    !
!!$    xyz_DTempDt = ( xyz_Temp - xyz_TempB ) / ( 2.0_DP * DelTime )
!!$    call HistoryAutoPut( TimeN, 'DTempDtMajCompPhaseChange', xyz_DTempDt )

    call HistoryAutoPut( TimeN, 'LiqCloudRadius', xyz_LiqPartRad )
    call HistoryAutoPut( TimeN, 'SolCloudRadius', xyz_SolPartRad )


    ! 計算時間計測一時停止
    ! Pause measurement of computation time
    !
    call TimesetClockStop( module_name )

  end subroutine BKSerial_20190921_CloudSimple1BinWithIce

  !------------------------------------------------------------------------------------

  subroutine CloudSimple1BinCalcSedMassFlux1D( &
    & TimeStep,                                     & ! (in   )
    & r_Press, z_Press, r_Height,                   & ! (in   )
    & z_LiqPartDen, z_LiqPartRad,                   & ! (in   )
    & z_SolPartDen, z_SolPartRad,                   & ! (in   )
    & z_Temp,                                       & ! (inout)
    & z_QH2OVap, z_QH2OLiq, z_QH2OSol,              & ! (inout)
    & r_LiqGravSedMassFlux, r_SolGravSedMassFlux    & ! (out  )
    & )


    ! 物理定数設定
    ! Physical constants settings
    !
    use constants, only: &
      & Grav                  ! $ g $ [m s-2].
                              ! 重力加速度.
                              ! Gravitational acceleration

    ! 重力沈降過程
    ! Gravitational sedimentation process
    !
    use grav_sed, only : GravSedCalcSedVel1D

    ! 大規模凝結 (非対流性凝結)
    ! Large scale condensation
    !
    use lscond, only: LScaleCond1Grid


    real(DP)    , intent(in   ) :: TimeStep
    real(DP)    , intent(in   ) :: r_Press   (0:kmax)
    real(DP)    , intent(in   ) :: z_Press   (1:kmax)
    real(DP)    , intent(in   ) :: r_Height  (0:kmax)
    real(DP)    , intent(in   ) :: z_LiqPartDen (1:kmax)
    real(DP)    , intent(in   ) :: z_LiqPartRad (1:kmax)
    real(DP)    , intent(in   ) :: z_SolPartDen (1:kmax)
    real(DP)    , intent(in   ) :: z_SolPartRad (1:kmax)
    real(DP)    , intent(inout) :: z_Temp    (1:kmax)
    real(DP)    , intent(inout) :: z_QH2OVap (1:kmax)
    real(DP)    , intent(inout) :: z_QH2OLiq (1:kmax)
    real(DP)    , intent(inout) :: z_QH2OSol (1:kmax)
    real(DP)    , intent(out  ) :: r_LiqGravSedMassFlux(0:kmax)
    real(DP)    , intent(out  ) :: r_SolGravSedMassFlux(0:kmax)


    !
    ! local variables
    !
    real(DP) :: r_PartDen(0:kmax)
    real(DP) :: r_PartRad(0:kmax)

    real(DP) :: z_DelAtmMass    (1:kmax)
    real(DP) :: z_DelLiqMass    (1:kmax)
    real(DP) :: z_DelSolMass    (1:kmax)
    real(DP) :: FallingLiqMass
    real(DP) :: FallingSolMass
    real(DP) :: FallingLiqMassB
    real(DP) :: FallingSolMassB
    real(DP) :: z_LiqMassSink  (1:kmax)
    real(DP) :: z_SolMassSink  (1:kmax)

    real(DP) :: r_LiqSedVel        (0:kmax)
    real(DP) :: r_LiqGravSedMass   (0:kmax)
    real(DP) :: r_SolSedVel        (0:kmax)
    real(DP) :: r_SolGravSedMass   (0:kmax)

    real(DP) :: APHeight
    real(DP) :: DPHeight

    real(DP) :: MassOfPartialLayer
    real(DP) :: MassConsumed

    real(DP) :: Press1
    real(DP) :: Press2
    real(DP) :: Press

    real(DP) :: QH2OLiq
    real(DP) :: QH2OSol
    real(DP) :: DQH2OLiqDt
    real(DP) :: DQH2OSolDt
    logical  :: FlagExitLoopLiq
    logical  :: FlagExitLoopSol

    integer  :: k
    integer  :: kk
    integer  :: kkk


    ! 実行文 ; Executable statement
    !

    ! 初期化確認
    ! Initialization check
    !
    if ( .not. cloud_simple_1bin_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    ! Calculation of mass in each layer and layer thickness in unit of meter
    !   Layer thickness is calculated by using mass of a layer.
    do k = 1, kmax
      z_DelAtmMass(k) = ( r_Press(k-1) - r_Press(k) ) / Grav
    end do

    z_DelLiqMass = z_QH2OLiq * z_DelAtmMass
    z_DelSolMass = z_QH2OSol * z_DelAtmMass

    !
    ! calculation of sedimentation terminal velocity
    !
    r_PartDen(0:kmax-1) = z_LiqPartDen
    r_PartDen(kmax)     = z_LiqPartDen(kmax)
    r_PartRad(0:kmax-1) = z_LiqPartRad
    r_PartRad(kmax)     = z_LiqPartRad(kmax)
    call GravSedCalcSedVel1D(                  &
      & kmax+1, r_Press, r_PartDen, r_PartRad, & ! (in )
      & r_LiqSedVel                            & ! (out)
      & )
    r_PartDen(0:kmax-1) = z_SolPartDen
    r_PartDen(kmax)     = z_SolPartDen(kmax)
    r_PartRad(0:kmax-1) = z_SolPartRad
    r_PartRad(kmax)     = z_SolPartRad(kmax)
    call GravSedCalcSedVel1D(                  &
      & kmax+1, r_Press, r_PartDen, r_PartRad, & ! (in )
      & r_SolSedVel                            & ! (out)
      & )


    ! arbitrary upper (lower) limit
!        r_SedVel = sign( 1.0_DP, r_SedVel ) * max( abs(r_SedVel), 1.0e-20_DP )


    ! Evaluation of gravitational sedimentation mass

    !   Initialization
    r_LiqGravSedMass   = 0.0_DP
    z_LiqMassSink      = 0.0_DP
    r_SolGravSedMass   = 0.0_DP
    z_SolMassSink      = 0.0_DP

    ! kk : layer index
    do kk = 1, kmax

      ! Initialization
      FallingLiqMass = z_DelLiqMass(kk)
      FallingSolMass = z_DelSolMass(kk)

      FlagExitLoopLiq = .false.
      FlagExitLoopSol = .false.
      ! k : level index
      loop_level_k : do k = kk-1, 0, -1

        FallingLiqMassB = FallingLiqMass
        FallingSolMassB = FallingSolMass

        ! mass is sublimed in (kkk)th = (k+1)th layer during sedimentation
        !   kkk : layer index
        kkk = k+1

        ! saturation adjustment
!!$        QH2OLiq = z_QH2OLiq(k) + FallingLiqMass / z_DelAtmMass(k)
!!$        QH2OSol = z_QH2OSol(k) + FallingSolMass / z_DelAtmMass(k)
        QH2OLiq = FallingLiqMass / z_DelAtmMass(kkk)
        QH2OSol = FallingSolMass / z_DelAtmMass(kkk)
        !
        call LScaleCond1Grid(  &
          & z_Temp   (kkk), & ! (inout)
          & z_QH2OVap(kkk), & ! (inout)
          & QH2OLiq,        & ! (inout)
          & QH2OSol,        & ! (inout)
          & z_Press  (kkk), & ! (in)
          & DQH2OLiqDt, DQH2OSolDt  &  ! (out)
          & )
        !
        FallingLiqMass = QH2OLiq * z_DelAtmMass(kkk)
        FallingSolMass = QH2OSol * z_DelAtmMass(kkk)

        z_LiqMassSink(kkk) =     &
          &   z_LiqMassSink(kkk) &
          & + ( FallingLiqMassB - FallingLiqMass )
        z_SolMassSink(kkk) =     &
          &   z_SolMassSink(kkk) &
          & + ( FallingSolMassB - FallingSolMass )


        ! For Liquid
        ! check whether (kk-1)th level crosses kth level
        !   note : (kk-1)th level is a bottom of (kk)th layer
        !   note : APHeight : height of arrival point
        APHeight = r_Height(kk-1) + r_LiqSedVel(kk-1) * TimeStep
        if ( r_Height(k) <= APHeight ) then
          ! (kk-1)th level does not cross kth level
          FallingLiqMass = 0.0_DP
          FlagExitLoopLiq = .true.
        else
          ! (kk-1)th level crosses kth level

          ! check whether (kk)th level crosses kth level
          !   note : (kk)th level is a top of (kk)th layer
          !   Here, sedimentation velocity in (kk)th layer is assumed to be
          !   constant with a velocity at (kk-1)th level (bottom level of a
          !   layer).
          !   So, sedimentation velocity at (kk-1)th level is used.
          !   note : r_SedVel is negative
          !   note : APHeight : height of arrival point
          APHeight = r_Height(kk) + r_LiqSedVel(kk-1) * TimeStep
          if ( APHeight <= r_Height(k) ) then
            ! All mass in (kk)th layer cross kth level

            ! do nothing
          else
            ! Part of mass in (kk)th layer crosses kth level
            ! note : r_SedVel is negative

            ! note : DPHeight : height of departure point, i.e., height which
            !                   reaches kth level at end of a time step
            DPHeight = r_Height(k) - r_LiqSedVel(kk-1) * TimeStep

            ! evaluate pressure at the departure point
            !   Press1 : pressure at bottom of (kk)th layer
            !   Press2 : pressure at top    of (kk)th layer
            Press1 = r_Press(kk-1)
            if ( r_Press(kk) == 0.0_DP ) then
              Press2 = r_Press(kk-1) * 1.0e-1_DP
            else
              Press2 = r_Press(kk  )
            end if
            !   Press  : pressure at the departure point
            Press = &
              &   log( Press2 / Press1 )              &
              & / ( r_Height  (kk) - r_Height(kk-1) ) &
              & * ( DPHeight       - r_Height(kk-1) ) &
              & + log( Press1 )
            Press = exp( Press )
!            DelCompMass =              &
!              &   ( Press1 - Press  )  &
!              & / ( Press1 - Press2 )  &
!              & * z_DelCompMass(kk)
            ! MassOfPartialLayer : Mass between Press1 and Press, i.e., mass
            !                    : which crosses kth level
            MassOfPartialLayer =       &
              &   ( Press1 - Press  )  &
              & / ( Press1 - Press2 )  &
              & * z_DelLiqMass(kk)
            ! MassConsumed : Mass of (kk)th layer which is consumed (sublimed)
            ! in layers above
            MassConsumed = z_DelLiqMass(kk) - FallingLiqMass
            ! Concept:
            !   MassOfPartialLayer is mass of bottom part of (kk)th layer.
            !   Mass of bottom part of layer will be consumed (sublimed) in
            !   layers above, first (preferentially).
            FallingLiqMass = max( MassOfPartialLayer - MassConsumed, 0.0_DP )
          end if

        end if

        ! For Solid
        ! check whether (kk-1)th level crosses kth level
        !   note : (kk-1)th level is a bottom of (kk)th layer
        !   note : APHeight : height of arrival point
        APHeight = r_Height(kk-1) + r_SolSedVel(kk-1) * TimeStep
        if ( r_Height(k) <= APHeight ) then
          ! (kk-1)th level does not cross kth level
          FallingSolMass = 0.0_DP
          FlagExitLoopSol = .true.
        else
          ! (kk-1)th level crosses kth level

          ! check whether (kk)th level crosses kth level
          !   note : (kk)th level is a top of (kk)th layer
          !   Here, sedimentation velocity in (kk)th layer is assumed to be
          !   constant with a velocity at (kk-1)th level (bottom level of a
          !   layer).
          !   So, sedimentation velocity at (kk-1)th level is used.
          !   note : r_SedVel is negative
          !   note : APHeight : height of arrival point
          APHeight = r_Height(kk) + r_SolSedVel(kk-1) * TimeStep
          if ( APHeight <= r_Height(k) ) then
            ! All mass in (kk)th layer cross kth level

            ! do nothing
          else
            ! Part of mass in (kk)th layer crosses kth level
            ! note : r_SedVel is negative

            ! note : DPHeight : height of departure point, i.e., height which
            !                   reaches kth level at end of a time step
            DPHeight = r_Height(k) - r_SolSedVel(kk-1) * TimeStep

            ! evaluate pressure at the departure point
            !   Press1 : pressure at bottom of (kk)th layer
            !   Press2 : pressure at top    of (kk)th layer
            Press1 = r_Press(kk-1)
            if ( r_Press(kk) == 0.0_DP ) then
              Press2 = r_Press(kk-1) * 1.0e-1_DP
            else
              Press2 = r_Press(kk  )
            end if
            !   Press  : pressure at the departure point
            Press = &
              &   log( Press2 / Press1 )              &
              & / ( r_Height  (kk) - r_Height(kk-1) ) &
              & * ( DPHeight       - r_Height(kk-1) ) &
              & + log( Press1 )
            Press = exp( Press )
!            DelCompMass =              &
!              &   ( Press1 - Press  )  &
!              & / ( Press1 - Press2 )  &
!              & * z_DelCompMass(kk)
            ! MassOfPartialLayer : Mass between Press1 and Press, i.e., mass
            !                    : which crosses kth level
            MassOfPartialLayer =       &
              &   ( Press1 - Press  )  &
              & / ( Press1 - Press2 )  &
              & * z_DelSolMass(kk)
            ! MassConsumed : Mass of (kk)th layer which is consumed (sublimed)
            ! in layers above
            MassConsumed = z_DelSolMass(kk) - FallingSolMass
            ! Concept:
            !   MassOfPartialLayer is mass of bottom part of (kk)th layer.
            !   Mass of bottom part of layer will be consumed (sublimed) in
            !   layers above, first (preferentially).
            FallingSolMass = max( MassOfPartialLayer - MassConsumed, 0.0_DP )
          end if

        end if

        if ( FlagExitLoopLiq .and. FlagExitLoopSol ) then
          exit loop_level_k
        end if

        ! Gravitational sedimentation mass is increased by contribution of 
        ! (kk)th layer (remnant of (kk)th layer mass). 
        r_LiqGravSedMass(k) = r_LiqGravSedMass(k) + FallingLiqMass
        r_SolGravSedMass(k) = r_SolGravSedMass(k) + FallingSolMass

      end do loop_level_k

    end do


    do k = 1, kmax
      ! note: negative sign on r_GravSedMass means downward transport
      z_DelLiqMass(k) = z_DelLiqMass(k)                       &
        & - ( ( -r_LiqGravSedMass(k) ) - ( -r_LiqGravSedMass(k-1) ) ) &
        & - z_LiqMassSink(k)
      z_DelSolMass(k) = z_DelSolMass(k)                       &
        & - ( ( -r_SolGravSedMass(k) ) - ( -r_SolGravSedMass(k-1) ) ) &
        & - z_SolMassSink(k)
    end do

    z_QH2OLiq = z_DelLiqMass / z_DelAtmMass
    z_QH2OSol = z_DelSolMass / z_DelAtmMass

    ! note: negative sign on r_GravSedMass means downward transport
    r_LiqGravSedMassFlux = - r_LiqGravSedMass / TimeStep
    r_SolGravSedMassFlux = - r_SolGravSedMass / TimeStep


  end subroutine CloudSimple1BinCalcSedMassFlux1D

  !--------------------------------------------------------------------------------------

  subroutine CloudSimple1BinInit

    ! OpenMP
    !
    !$ use omp_lib

    ! ファイル入出力補助
    ! File I/O support
    !
    use dc_iounit, only: FileOpen

    ! NAMELIST ファイル入力に関するユーティリティ
    ! Utilities for NAMELIST file input
    !
    use namelist_util, only: namelist_filename, NmlutilMsg, NmlutilAryValid

    ! ヒストリデータ出力
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoAddVariable

    ! 座標データ設定
    ! Axes data settings
    !
    use axesset, only: &
      & AxnameX, &
      & AxnameY, &
      & AxnameZ, &
      & AxnameR, &
      & AxnameT

    ! 物理定数設定
    ! Physical constants settings
    !
    use constants, only : ConstantsInit

    ! 雲関系ルーチン
    ! Cloud-related routines
    !
    use cloud_utils, only : CloudUtilsInit

    ! 重力沈降過程
    ! Gravitational sedimentation process
    !
    use grav_sed, only : GravSedInit

    ! 短波入射 (太陽入射)
    ! Short wave (insolation) incoming
    !
    use rad_short_income, only : RadShortIncomeInit


    use set_Mars_dust, only : SetMarsDustInit

    ! 大規模凝結 (非対流性凝結)
    ! Large scale condensation (non-convective condensation)
    !
    use lscond, only : LScaleCondInit


    ! 宣言文 ; Declaration statements
    !

    integer:: unit_nml        ! NAMELIST ファイルオープン用装置番号.
                              ! Unit number for NAMELIST file open
    integer:: iostat_nml      ! NAMELIST 読み込み時の IOSTAT.
                              ! IOSTAT of NAMELIST read

    integer:: iThread


    ! NAMELIST 変数群
    ! NAMELIST group name
    !
    namelist /cloud_simple_1bin_nml/ &
      & CloudLiqRad,              &
      & CloudSolRad,              &
      & CloudNumRatio,            &
      & CloudNuclRad,             &
      & CloudNuclDen
          !
          ! デフォルト値については初期化手続 "cloud_simple_1bin#CloudSimple1BinInit"
          ! のソースコードを参照のこと.
          !
          ! Refer to source codes in the initialization procedure
          ! "cloud_simple_1bin#CloudSimple1BinInit" for the default values.
          !

    ! 実行文 ; Executable statement
    !

    if ( cloud_simple_1bin_inited ) return


    ! デフォルト値の設定
    ! Default values settings
    !
    CloudLiqRad      =  2.0d-6
    CloudSolRad      =  2.0d-6
    CloudNumRatio    = -1.0d0
    CloudNuclRad     =  0.0d0
    CloudNuclDen     =  0.0d0


    ! NAMELIST の読み込み
    ! NAMELIST is input
    !
    if ( trim(namelist_filename) /= '' ) then
      call FileOpen( unit_nml, &          ! (out)
        & namelist_filename, mode = 'r' ) ! (in)

      rewind( unit_nml )
      read( unit_nml,                     & ! (in)
        & nml = cloud_simple_1bin_nml,    & ! (out)
        & iostat = iostat_nml )             ! (out)
      close( unit_nml )

      call NmlutilMsg( iostat_nml, module_name ) ! (in)
    end if


    ! Preparation for OpenMP
    !
    NThreads = 1
    !$ NThreads  = omp_get_max_threads()
!    !$ write( 6, * ) "Number of processors : ", omp_get_num_procs()
!    !$ write( 6, * ) "Number of threads    : ", nthreads

    allocate( a_ls(0:NThreads-1) )
    allocate( a_le(0:NThreads-1) )

    do iThread = 0, NThreads-1
      if ( iThread == 0 ) then
        a_ls(iThread) = 1
      else
        a_ls(iThread) = a_le(iThread-1) + 1
      end if
      a_le(iThread) = a_ls(iThread) + imax*jmax / NThreads - 1
      if ( iThread + 1 <= mod( imax*jmax, NThreads ) ) then
        a_le(iThread) = a_le(iThread) + 1
      end if
    end do


    ! Initialization of modules used in this module
    !

    ! 物理定数設定
    ! Physical constants settings
    !
    call ConstantsInit

    ! 雲関系ルーチン
    ! Cloud-related routines
    !
    call CloudUtilsInit

    ! 重力沈降過程
    ! Gravitational sedimentation process
    !
    call GravSedInit

    ! 短波入射 (太陽入射)
    ! Short wave (insolation) incoming
    !
    call RadShortIncomeInit

    call SetMarsDustInit

    ! 大規模凝結 (非対流性凝結)
    ! Large scale condensation (non-convective condensation)
    !
    call LScaleCondInit


    ! ヒストリデータ出力のためのへの変数登録
    ! Register of variables for history data output
    !
    call HistoryAutoAddVariable( 'LiqCloudRadiusForGravSed', &
      & (/ AxnameX, AxnameY, AxnameZ, AxnameT /), &
      & 'water cloud radius for gravitational sedimentation', &
      & 'm' )
    call HistoryAutoAddVariable( 'SolCloudRadiusForGravSed', &
      & (/ AxnameX, AxnameY, AxnameZ, AxnameT /), &
      & 'ice cloud radius for gravitational sedimentation', &
      & 'm' )
    call HistoryAutoAddVariable( 'LiqCloudRadius', &
      & (/ AxnameX, AxnameY, AxnameZ, AxnameT /), &
      & 'water cloud radius', &
      & 'm' )
    call HistoryAutoAddVariable( 'SolCloudRadius', &
      & (/ AxnameX, AxnameY, AxnameZ, AxnameT /), &
      & 'ice cloud radius', &
      & 'm' )


    ! 印字 ; Print
    !
    call MessageNotify( 'M', module_name, '----- Initialization Messages -----' )
    call MessageNotify( 'M', module_name, ' NThreads         = %d', i = (/NThreads/) )
    call MessageNotify( 'M', module_name, ' CloudLiqRad      = %f', d = (/ CloudLiqRad /) )
    call MessageNotify( 'M', module_name, ' CloudSolRad      = %f', d = (/ CloudSolRad /) )
    call MessageNotify( 'M', module_name, ' CloudNumRatio    = %f', d = (/ CloudNumRatio /) )
    call MessageNotify( 'M', module_name, ' CloudNuclRad     = %f', d = (/ CloudNuclRad /) )
    call MessageNotify( 'M', module_name, ' CloudNuclDen     = %f', d = (/ CloudNuclDen /) )
    call MessageNotify( 'M', module_name, '-- version = %c', c1 = trim(version) )


    cloud_simple_1bin_inited = .true.

  end subroutine CloudSimple1BinInit

  !--------------------------------------------------------------------------------------

end module cloud_simple_1bin
