!= 下部境界フラックス
!
!= Lower boundary flux
!
! Authors::   Yoshiyuki O. Takahashi
! Version::   $Id: lb_flux_simple.f90,v 1.2 2014/05/07 09:39:22 murashin Exp $ 
! Tag Name::  $Name:  $
! Copyright:: Copyright (C) GFD Dennou Club, 2008-2009. All rights reserved.
! License::   See COPYRIGHT[link:../../../COPYRIGHT]
!

module lb_flux_simple
  !
  != 下部境界フラックス
  !
  != Lower boundary flux
  !
  ! <b>Note that Japanese and English are described in parallel.</b>
  !
  ! 
  !
  ! 
  !
  !== References
  !
!!$  ! Louis, J-F., M. Tiedtke, and J-F. Geleyn, 
!!$  ! A short history of the PBL parameterization at ECMWF, 
!!$  ! Workshop on Planetary Boundary Layer Parameterization, 59-80, ECMWF, Reading, U.K., 
!!$  ! 1982.
  !
  !== Procedures List
  !
!!$  ! SurfaceFlux       :: 地表面フラックスの計算
!!$  ! SurfaceFluxOutput :: 地表面フラックスの出力
!!$  ! ------------      :: ------------
!!$  ! SurfaceFlux       :: Calculate surface fluxes
!!$  ! SurfaceFluxOutput :: Output surface fluxes
  !
  !== NAMELIST
  !
  ! NAMELIST#lb_flux_simple_nml
  !

  ! モジュール引用 ; USE statements
  !

  ! 格子点設定
  ! Grid points settings
  !
  use gridset, only: imax, & ! 経度格子点数. 
                             ! Number of grid points in longitude
    &                jmax, & ! 緯度格子点数. 
                             ! Number of grid points in latitude
    &                kmax    ! 鉛直層数. 
                             ! Number of vertical level

  ! 組成に関わる配列の設定
  ! Settings of array for atmospheric composition
  !
  use composition, only: ncmax, IndexH2OVap

  ! 種別型パラメタ
  ! Kind type parameter
  !
  use dc_types, only: DP, &      ! 倍精度実数型. Double precision. 
    &                 STRING, &  ! 文字列.       Strings. 
    &                 TOKEN      ! キーワード.   Keywords.

  ! メッセージ出力
  ! Message output
  !
  use dc_message, only: MessageNotify

  ! 宣言文 ; Declaration statements
  !
  implicit none
  private

  ! 公開手続き
  ! Public procedure
  !
  public :: LBFluxSimple
  public :: LBFluxSimpleInit

  ! 公開変数
  ! Public variables
  !

  ! 非公開変数
  ! Private variables
  !
  logical, save :: lb_flux_simple_inited = .false.
                              ! 初期設定フラグ. 
                              ! Initialization flag

  integer, save      :: IDLBMomFluxMode
  integer, parameter :: IDLBMomFluxModeNoFlux       = 10
  integer, parameter :: IDLBMomFluxModeFixTimeConst = 11
  integer, save      :: IDLBHeatFluxMode
  integer, parameter :: IDLBHeatFluxModeFixFlux     = 20
  integer, parameter :: IDLBHeatFluxModeFixTemp     = 21
  integer, save      :: IDLBH2OVapFluxMode
  integer, parameter :: IDLBH2OVapFluxModeFixFlux   = 30
  integer, parameter :: IDLBH2OVapFluxModeFixMixRt  = 31


  real(DP), save:: FricTimeConstAtLB
                            ! 下部境界摩擦の係数 (s).
                            ! Time constant of surface friction (s).
  real(DP), save:: FricLowestLatAtLB
                            ! 下部境界摩擦が働く最低緯度 (degree).
                            ! Lowest latitude where the friction is applied (degree)
  real(DP), save:: HeatFluxAtLB
                            ! 下部境界での熱フラックス (W m-2).
                            ! Heat flux at the lower boundary (W m-2).
  real(DP), save:: TempAtLB
  real(DP), save:: H2OVapFluxAtLB
                            ! 下部境界での H2O 蒸気質量フラックス (W m-2).
                            ! 実際にはゼロに固定するために使う程度にしか使えないだろう.
                            ! H2O vapor flux at the lower boundary (kg m-2 s-1).
  real(DP), save:: QH2OVapAtLB
  real(DP), save:: DiffCoefHeatMass


  character(*), parameter:: module_name = 'lb_flux_simple'
                              ! モジュールの名称. 
                              ! Module name
  character(*), parameter:: version = &
    & '$Name:  $' // &
    & '$Id: lb_flux_simple.f90,v 1.2 2014/05/07 09:39:22 murashin Exp $'
                              ! モジュールのバージョン
                              ! Module version


contains

  !--------------------------------------------------------------------------------------

  subroutine LBFluxSimple( &
    & xyz_U, xyz_V, xyz_Temp, xyr_VirTemp, xyzf_QMix,             & ! (in)
    & xyr_Press, xy_SurfHeight, xyz_Height, xyz_Exner, xyr_Exner, & ! (in)
    & xyr_MomFluxX, xyr_MomFluxY, xyr_HeatFlux, xyrf_QMixFlux,    & ! (out)
    & xy_SurfVelTransCoef, xy_SurfTempTransCoef,                  & ! (out)
    & xy_SurfQVapTransCoef                                        & ! (out)
    & )
    !
    ! 温度, 比湿, 気圧から, 放射フラックスを計算します. 
    !
    ! Calculate radiation flux from temperature, specific humidity, and 
    ! air pressure. 
    !

    ! モジュール引用 ; USE statements
    !

    ! 物理・数学定数設定
    ! Physical and mathematical constants settings
    !
    use constants0, only: &
      & PI
                              ! $ \pi $ .
                              ! 円周率.  Circular constant

    ! 物理定数設定
    ! Physical constants settings
    !
    use constants, only: &
      & Grav, &               ! $ g $ [m s-2].
                              ! 重力加速度.
                              ! Gravitational acceleration
      & GasRDry, &
                              ! $ R $ [J kg-1 K-1]. 
                              ! 乾燥大気の気体定数. 
                              ! Gas constant of air
      & CpDry
                              ! $ C_p $ [J kg-1 K-1]. 
                              ! 乾燥大気の定圧比熱. 
                              ! Specific heat of air at constant pressure

    ! 飽和比湿の算出
    ! Evaluate saturation specific humidity
    !
    use saturate, only: xy_CalcQVapSat

    ! 座標データ設定
    ! Axes data settings
    !
    use axesset, only: &
      & y_Lat                 ! $ \varphi $ [rad.] . 緯度. Latitude

    ! 時刻管理
    ! Time control
    !
    use timeset, only: &
      & TimesetClockStart, TimesetClockStop

    ! デバッグ用ユーティリティ
    ! Utilities for debug
    !
    use dc_trace, only: DbgMessage, BeginSub, EndSub

    ! 宣言文 ; Declaration statements
    !

    real(DP), intent(in):: xyz_U (0:imax-1, 1:jmax, 1:kmax)
                              ! $ u $ . 東西風速. Eastward wind
    real(DP), intent(in):: xyz_V (0:imax-1, 1:jmax, 1:kmax)
                              ! $ v $ . 南北風速. Northward wind

    real(DP), intent(in):: xyz_Temp (0:imax-1, 1:jmax, 1:kmax)
                              ! $ T $ . 温度 (整数レベル). 
                              ! Temperature (full level)
    real(DP), intent(in):: xyr_VirTemp (0:imax-1, 1:jmax, 0:kmax)
                              ! $ T_v $ . 仮温度 (半整数レベル). 
                              ! Virtual temperature (half level)
    real(DP), intent(in):: xyzf_QMix(0:imax-1, 1:jmax, 1:kmax, 1:ncmax)
                              ! $ q $ .     比湿. Specific humidity
    real(DP), intent(in):: xyr_Press (0:imax-1, 1:jmax, 0:kmax)
                              ! $ p_s $ . 地表面気圧 (半整数レベル). 
                              ! Surface pressure (half level)
    real(DP), intent(in):: xy_SurfHeight(0:imax-1,1:jmax)
                              ! $ z_s $ . 地表面高度. 
                              ! Surface height. 
    real(DP), intent(in):: xyz_Height (0:imax-1, 1:jmax, 1:kmax)
                              ! 高度 (整数レベル). 
                              ! Height (full level)
    real(DP), intent(in):: xyz_Exner (0:imax-1, 1:jmax, 1:kmax)
                              ! Exner 関数 (整数レベル). 
                              ! Exner function (full level)
    real(DP), intent(in):: xyr_Exner (0:imax-1, 1:jmax, 0:kmax)
                              ! Exner 関数 (半整数レベル). 
                              ! Exner function (half level)
    real(DP), intent(out):: xyr_MomFluxX (0:imax-1, 1:jmax, 0:0)
                              ! 東西方向運動量フラックス. 
                              ! Eastward momentum flux
    real(DP), intent(out):: xyr_MomFluxY (0:imax-1, 1:jmax, 0:0)
                              ! 南北方向運動量フラックス. 
                              ! Northward momentum flux
    real(DP), intent(out):: xyr_HeatFlux (0:imax-1, 1:jmax, 0:0)
                              ! 熱フラックス. 
                              ! Heat flux
    real(DP), intent(out):: xyrf_QMixFlux(0:imax-1, 1:jmax, 0:0, 1:ncmax)
                              ! 比湿フラックス. 
                              ! Specific humidity flux
    real(DP), intent(out):: xy_SurfVelTransCoef (0:imax-1, 1:jmax)
                              ! 輸送係数：運動量. 
                              ! Diffusion coefficient: velocity
    real(DP), intent(out):: xy_SurfTempTransCoef (0:imax-1, 1:jmax)
                              ! 輸送係数：温度. 
                              ! Transfer coefficient: temperature
    real(DP), intent(out):: xy_SurfQVapTransCoef (0:imax-1, 1:jmax)
                              ! 輸送係数：水蒸気
                              ! Transfer coefficient: water vapor

    ! 作業変数
    ! Work variables
    !
    real(DP) :: xy_TempAtLB   (0:imax-1, 1:jmax)
    real(DP) :: xy_QH2OVapAtLB(0:imax-1, 1:jmax)

    integer:: j


    ! 実行文 ; Executable statement
    !

    ! 初期化確認
    ! Initialization check
    !
    if ( .not. lb_flux_simple_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    ! 計算時間計測開始
    ! Start measurement of computation time
    !
    call TimesetClockStart( module_name )


    ! 輸送係数の計算
    ! Calculate transfer coefficient
    !
    select case ( IDLBMomFluxMode )
    case ( IDLBMomFluxModeNoFlux )
      xy_SurfVelTransCoef = 0.0_DP
    case ( IDLBMomFluxModeFixTimeConst )
      if ( FricTimeConstAtLB <= 0.0_DP ) then
        call MessageNotify( 'E', module_name, 'FricTimeConstAtLB must be greater than or equal to zero.' )
      end if
      do j = 1, jmax
        if ( abs( y_Lat(j) ) >= FricLowestLatAtLB * PI / 180.0_DP ) then
          xy_SurfVelTransCoef(:,j) =                           &
            & - ( xyr_Press(:,j,1) - xyr_Press(:,j,0) ) / Grav &
            &   / FricTimeConstAtLB
        else
          xy_SurfVelTransCoef(:,j) = 0.0_DP
        end if
      end do
    end select

    select case ( IDLBHeatFluxMode )
    case ( IDLBHeatFluxModeFixFlux )
      xy_SurfTempTransCoef = HeatFluxAtLB
    case ( IDLBHeatFluxModeFixTemp )
      xy_SurfTempTransCoef =                                    &
        &   xyr_Press(:,:,0) / ( GasRDry * xyr_VirTemp(:,:,0) ) &
        & * DiffCoefHeatMass                                    &
        & / ( xyz_Height(:,:,1) - xy_SurfHeight )
    end select

    select case( IDLBH2OVapFluxMode )
    case ( IDLBH2OVapFluxModeFixFlux )
      xy_SurfQVapTransCoef = H2OVapFluxAtLB
    case ( IDLBH2OVapFluxModeFixMixRt )
      xy_SurfQVapTransCoef =                                    &
        &   xyr_Press(:,:,0) / ( GasRDry * xyr_VirTemp(:,:,0) ) &
        & * DiffCoefHeatMass                                    &
        & / ( xyz_Height(:,:,1) - xy_SurfHeight )
    end select


    ! 下部境界フラックスの計算
    ! Calculate fluxes at lower boundary
    !
    !   Momentum
    !
    xyr_MomFluxX(:,:,0) = - xy_SurfVelTransCoef * xyz_U(:,:,1)
    xyr_MomFluxY(:,:,0) = - xy_SurfVelTransCoef * xyz_V(:,:,1)

    !   Heat
    !
    select case ( IDLBHeatFluxMode )
    case ( IDLBHeatFluxModeFixFlux )
      xyr_HeatFlux = HeatFluxAtLB
    case ( IDLBHeatFluxModeFixTemp )
      xy_TempAtLB = TempAtLB
      xyr_HeatFlux(:,:,0) = - CpDry * xyr_Exner(:,:,0) * xy_SurfTempTransCoef &
        &                       * (   xyz_Temp(:,:,1) / xyz_Exner(:,:,1)      &
        &                           - xy_TempAtLB     / xyr_Exner(:,:,0) )
    end select

    !   Mass
    !
    select case( IDLBH2OVapFluxMode )
    case ( IDLBH2OVapFluxModeFixFlux )
      xyrf_QMixFlux(:,:,0,IndexH2OVap) = H2OVapFluxAtLB
    case ( IDLBH2OVapFluxModeFixMixRt )
      xy_QH2OVapAtLB = QH2OVapAtLB
      xyrf_QMixFlux(:,:,0,IndexH2OVap) =                      &
        & - xy_SurfQVapTransCoef                              &
        & * ( xyzf_QMix(:,:,1,IndexH2OVap) - xy_QH2OVapAtLB )
    end select
    !
    xyrf_QMixFlux(:,:,0,1:IndexH2OVap-1)     = 0.0_DP
    xyrf_QMixFlux(:,:,0,IndexH2OVap+1:ncmax) = 0.0_DP

    ! Surface flux of constituents except for water vapor is zero.
!!$    write( 6, * ) "MEMO: Surface flux of constituents except for water vapor is zero. (YOT, 2013/05/15)"


    ! ヒストリデータ出力
    ! History data output
    !

    ! 計算時間計測一時停止
    ! Pause measurement of computation time
    !
    call TimesetClockStop( module_name )

  end subroutine LBFluxSimple

  !--------------------------------------------------------------------------------------

  subroutine LBFluxSimpleInit
    !
    ! surface_flux_bulk モジュールの初期化を行います. 
    ! NAMELIST#surface_flux_bulk_nml の読み込みはこの手続きで行われます. 
    !
    ! "surface_flux_bulk" module is initialized. 
    ! "NAMELIST#surface_flux_bulk_nml" is loaded in this procedure. 
    !

    ! モジュール引用 ; USE statements
    !

    ! NAMELIST ファイル入力に関するユーティリティ
    ! Utilities for NAMELIST file input
    !
    use namelist_util, only: namelist_filename, NmlutilMsg, NmlutilAryValid

    ! ファイル入出力補助
    ! File I/O support
    !
    use dc_iounit, only: FileOpen

    ! 種別型パラメタ
    ! Kind type parameter
    !
    use dc_types, only: STDOUT ! 標準出力の装置番号. Unit number of standard output

    ! 文字列操作
    ! Character handling
    !
    use dc_string, only: StoA


    ! 日付および時刻の取り扱い
    ! Date and time handler
    !
    use dc_calendar, only: DCCalConvertByUnit

    ! ヒストリデータ出力
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoAddVariable

    ! 物理定数設定
    ! Physical constant settings
    !
    use constants, only: &
      & GasRDry, &
                              ! $ R $ [J kg-1 K-1]. 
                              ! 乾燥大気の気体定数. 
                              ! Gas constant of air
      & CpDry
                              ! $ C_p $ [J kg-1 K-1]. 
                              ! 乾燥大気の定圧比熱. 
                              ! Specific heat of air at constant pressure

    ! 宣言文 ; Declaration statements
    !
    real(DP)         :: FricTimeConstAtLBValue
    character(TOKEN) :: FricTimeConstAtLBUnit

    character(STRING) :: LBMomFluxMode
    character(STRING) :: LBHeatFluxMode
    character(STRING) :: LBH2OVapFluxMode

    integer:: unit_nml        ! NAMELIST ファイルオープン用装置番号. 
                              ! Unit number for NAMELIST file open
    integer:: iostat_nml      ! NAMELIST 読み込み時の IOSTAT. 
                              ! IOSTAT of NAMELIST read

    ! NAMELIST 変数群
    ! NAMELIST group name
    !
    namelist /lb_flux_simple_nml/                        &
      & LBMomFluxMode,                                   &
      & FricTimeConstAtLBValue, FricTimeConstAtLBUnit,   &
      & FricLowestLatAtLB,                               &
      & LBHeatFluxMode,                                  &
      & HeatFluxAtLB, TempAtLB,                          &
      & LBH2OVapFluxMode,                                &
      & H2OVapFluxAtLB, QH2OVapAtLB, DiffCoefHeatMass
          !
          ! デフォルト値については初期化手続 "lb_flux_simple#LBFluxSimpleInit" 
          ! のソースコードを参照のこと. 
          !
          ! Refer to source codes in the initialization procedure
          ! "lb_flux_simple#LBFluxSimpleInit" for the default values. 
          !

    ! 実行文 ; Executable statement
    !

    if ( lb_flux_simple_inited ) return


    ! デフォルト値の設定
    ! Default values settings
    !
    LBMomFluxMode            = 'FixTimeConst'
    FricTimeConstAtLBValue   = 20.0_DP   ! Schneider and Liou (2009)
    FricTimeConstAtLBUnit    = 'day'
    FricLowestLatAtLB        = 16.3_DP   ! Schneider and Liou (2009)

    LBHeatFluxMode           = 'FixFlux'
    HeatFluxAtLB             =  5.7_DP   ! Schneider and Liou (2009)
    TempAtLB                 =  160.0_DP * ( 30.0_DP / 0.6_DP )**( GasRDry / CpDry )
                                 ! Sugiyama et al. (2009), Nagare Multimedia
                                 ! Potential Temperature of 160 K with 
                                 ! reference pressure of 0.6 bars

    LBH2OVapFluxMode         = 'FixFlux'
    H2OVapFluxAtLB           =  0.0_DP
    QH2OVapAtLB              =  7.816e-4_DP
                                 ! Sugiyama et al. (2009), Nagare Multimedia
    DiffCoefHeatMass         = 800.0_DP
                                 ! Sugiyama et al. (2009), Nagare Multimedia

    ! NAMELIST の読み込み
    ! NAMELIST is input
    !
    if ( trim(namelist_filename) /= '' ) then
      call FileOpen( unit_nml, &          ! (out)
        & namelist_filename, mode = 'r' ) ! (in)

      rewind( unit_nml )
      read( unit_nml, &                ! (in)
        & nml = lb_flux_simple_nml, &  ! (out)
        & iostat = iostat_nml )        ! (out)
      close( unit_nml )

      call NmlutilMsg( iostat_nml, module_name ) ! (in)
    end if


    FricTimeConstAtLB = &
      & DCCalConvertByUnit( FricTimeConstAtLBValue, FricTimeConstAtLBUnit, 'sec' ) ! (in)


    ! Identification of calculation method for momentum flux
    !
    call MessageNotify( 'M', module_name, &
      & 'LBMomFluxMode=<%c>.', c1 = trim(LBMomFluxMode) )
    select case ( LBMomFluxMode )
    case ( 'NoFlux' )
      IDLBMomFluxMode = IDLBMomFluxModeNoFlux
    case ( 'FixTimeConst' )
      IDLBMomFluxMode = IDLBMomFluxModeFixTimeConst
    case default
      call MessageNotify( 'E', module_name, &
        & 'LBMomFluxMode=<%c> is not supported.', c1 = trim(LBMomFluxMode) )
    end select

    ! Identification of calculation method for heat flux
    !
    call MessageNotify( 'M', module_name, &
      & 'LBHeatFluxMode=<%c>.', c1 = trim(LBHeatFluxMode) )
    select case ( LBHeatFluxMode )
    case ( 'FixFlux' )
      IDLBHeatFluxMode = IDLBHeatFluxModeFixFlux
    case ( 'FixTemp' )
      IDLBHeatFluxMode = IDLBHeatFluxModeFixFlux
    case default
      call MessageNotify( 'E', module_name, &
        & 'LBHeatFluxMode=<%c> is not supported.', c1 = trim(LBHeatFluxMode) )
    end select

    ! Identification of calculation method for H2O vapor flux
    !
    call MessageNotify( 'M', module_name, &
      & 'LBH2OVapFluxMode=<%c>.', c1 = trim(LBH2OVapFluxMode) )
    select case ( LBH2OVapFluxMode )
    case ( 'FixFlux' )
      IDLBH2OVapFluxMode = IDLBH2OVapFluxModeFixFlux
    case ( 'FixMixRt' )
      IDLBH2OVapFluxMode = IDLBH2OVapFluxModeFixMixRt
    case default
      call MessageNotify( 'E', module_name, &
        & 'LBH2OVapFluxMode=<%c> is not supported.', c1 = trim(LBH2OVapFluxMode) )
    end select



    ! ヒストリデータ出力のためのへの変数登録
    ! Register of variables for history data output
    !
    call HistoryAutoAddVariable( 'TauX', &
      & (/ 'lon ', 'lat ', 'time' /), &
      & 'surface stress(x)  ', 'N m-2' )
    call HistoryAutoAddVariable( 'TauY', &
      & (/ 'lon ', 'lat ', 'time' /), &
      & 'surface stress(y)  ', 'N m-2' )
    call HistoryAutoAddVariable( 'Sens', &
      & (/ 'lon ', 'lat ', 'time' /), &
      & 'sensible heat flux', 'W m-2' )
    call HistoryAutoAddVariable( 'SurfH2OVapFlux', &
      & (/ 'lon ', 'lat ', 'time' /), &
      & 'surface H2O vapor flux  ', 'kg m-2 s-1' )
    call HistoryAutoAddVariable( 'Evap', &
      & (/ 'lon ', 'lat ', 'time' /), &
      & 'latent heat flux  ', 'W m-2' )

    call HistoryAutoAddVariable( 'TauXB', &
      & (/ 'lon ', 'lat ', 'time' /), &
      & 'surface stress(x)  ', 'N m-2' )
    call HistoryAutoAddVariable( 'TauYB', &
      & (/ 'lon ', 'lat ', 'time' /), &
      & 'surface stress(y)  ', 'N m-2' )
    call HistoryAutoAddVariable( 'SensB', &
      & (/ 'lon ', 'lat ', 'time' /), &
      & 'sensible heat flux', 'W m-2' )
    call HistoryAutoAddVariable( 'SurfH2OVapFluxB', &
      & (/ 'lon ', 'lat ', 'time' /), &
      & 'surface H2O vapor flux  ', 'kg m-2 s-1' )
    call HistoryAutoAddVariable( 'EvapB', &
      & (/ 'lon ', 'lat ', 'time' /), &
      & 'latent heat flux  ', 'W m-2' )

    call HistoryAutoAddVariable( 'TauXA', &
      & (/ 'lon ', 'lat ', 'time' /), &
      & 'surface stress(x)  ', 'N m-2' )
    call HistoryAutoAddVariable( 'TauYA', &
      & (/ 'lon ', 'lat ', 'time' /), &
      & 'surface stress(y)  ', 'N m-2' )
    call HistoryAutoAddVariable( 'SensA', &
      & (/ 'lon ', 'lat ', 'time' /), &
      & 'sensible heat flux', 'W m-2' )
    call HistoryAutoAddVariable( 'SurfH2OVapFluxA', &
      & (/ 'lon ', 'lat ', 'time' /), &
      & 'surface H2O vapor flux  ', 'kg m-2 s-1' )
    call HistoryAutoAddVariable( 'EvapA', &
      & (/ 'lon ', 'lat ', 'time' /), &
      & 'latent heat flux  ', 'W m-2' )

    call HistoryAutoAddVariable( 'SurfH2OVapFluxU', &
      & (/ 'lon ', 'lat ', 'time' /), &
      & 'surface H2O vapor flux  ', 'kg m-2 s-1' )
    call HistoryAutoAddVariable( 'EvapU', &
      & (/ 'lon ', 'lat ', 'time' /), &
      & 'latent heat flux  ', 'W m-2' )

    ! 印字 ; Print
    !
    call MessageNotify( 'M', module_name, '----- Initialization Messages -----' )
    call MessageNotify( 'M', module_name, 'LBMomFluxMode            = %c', c1 = trim( LBMomFluxMode ) )
    call MessageNotify( 'M', module_name, 'FricTimeConstAtLB        = %f', d = (/ FricTimeConstAtLB /) )
    call MessageNotify( 'M', module_name, 'FricLowestLatAtLB        = %f', d = (/ FricLowestLatAtLB /) )
    call MessageNotify( 'M', module_name, 'LBHeatFluxMode           = %c', c1 = trim( LBHeatFluxMode ) )
    call MessageNotify( 'M', module_name, 'HeatFluxAtLB             = %f', d = (/ HeatFluxAtLB /) )
    call MessageNotify( 'M', module_name, 'TempAtLB                 = %f', d = (/ TempAtLB /) )
    call MessageNotify( 'M', module_name, 'LBH2OVapFluxMode         = %c', c1 = trim( LBH2OVapFluxMode ) )
    call MessageNotify( 'M', module_name, 'H2OVapFluxAtLB           = %f', d = (/ H2OVapFluxAtLB /) )
    call MessageNotify( 'M', module_name, 'QH2OVapAtLB              = %f', d = (/ QH2OVapAtLB /) )
    call MessageNotify( 'M', module_name, 'DiffCoefHeatMass         = %f', d = (/ DiffCoefHeatMass /) )
    call MessageNotify( 'M', module_name, '-- version = %c', c1 = trim(version) )

    lb_flux_simple_inited = .true.

  end subroutine LBFluxSimpleInit

  !--------------------------------------------------------------------------------------

end module lb_flux_simple
