!= 
!
!= Trajectory calculation
!
! Authors::   Yoshiyuki O. TAKAHASHI
! Version::   $Id: dynamics_hspl_vas83.F90,v 1.83 2014/05/07 09:39:17 murashin Exp $
! Tag Name::  $Name:  $
! Copyright:: Copyright (C) GFD Dennou Club, 2008. All rights reserved.
! License::   See COPYRIGHT[link:../../../COPYRIGHT]
!

module trajectory
  !
  != 
  !
  != Trajectory calculation
  !
  ! <b>Note that Japanese and English are described in parallel.</b>
  !
  ! Trajectory of parcels are calculated. 
  !
  !== Procedures List
  !
!!$  ! DynamicsHSplVAS83         :: 力学計算
!!$  ! DynamicsHSplVAS83Init     :: 初期化
!!$  ! DynamicsHSplVAS83Finalize :: 終了処理 (モジュール内部の変数の割り付け解除)
!!$  ! ---------------------     :: ------------
!!$  ! DynamicsHSplVAS83         :: Calculate dynamics
!!$  ! DynamicsHSplVAS83Init     :: Initialization
!!$  ! DynamicsHSplVAS83Finalize :: Termination (deallocate variables in this module)
  !
  !== NAMELIST
  !
  ! NAMELIST#trajectory_nml
  !
  !== References
  !
!!$  ! * Arakawa, A., Suarez, M. J., 1983:
!!$  !   Vertical differencing of the primitive equations
!!$  !   in sigma coordinates.
!!$  !   <i>Mon. Wea. Rev.</i>, <b>111</b>, 34--35.

  ! モジュール引用 ; USE statements
  !


  ! 種別型パラメタ
  ! Kind type parameter
  !
  use dc_types, only: DP, &  ! 倍精度実数型. Double precision. 
    &                 TOKEN  ! キーワード.   Keywords. 

  ! メッセージ出力
  ! Message output
  !
  use dc_message, only: MessageNotify

  ! 宣言文 ; Declaration statements
  !
  implicit none
  private

  ! 公開手続き
  ! Public procedure
  !
  public :: TrajectoryCalc
  public :: TrajectoryInit

  ! 公開変数
  ! Public variables
  !

  ! 非公開変数
  ! Private variables
  !
  logical, save :: trajectory_inited = .false.
                              ! 初期設定フラグ. 
                              ! Initialization flag

!!$  integer , save      :: IDTimeIntegScheme
!!$                              ! ID for used time integration scheme
!!$
!!$  integer , parameter :: IDTimeIntegSchemeExplicit     = 0
!!$  integer , parameter :: IDTimeIntegSchemeSemiIMplicit = 1

!!$  real(DP), save:: DelTimeSave
!!$                              ! 前回の $ \Delta t $ [s]. 
!!$                              ! 陰解法のための係数の再生成の必要性の
!!$                              ! チェックに使用する. 
!!$                              ! 
!!$                              ! $ \Delta t $ [s] at previous step
!!$                              ! This is used to check necessity of 
!!$                              ! regeneration of coefficients for 
!!$                              ! implicit method. 

  integer , save       :: NParcel
  real(DP), allocatable:: a_ParcelLonN(:)
  real(DP), allocatable:: a_ParcelLatN(:)
  real(DP), allocatable:: a_ParcelSigN(:)
  real(DP), allocatable:: a_ParcelLonA(:)
  real(DP), allocatable:: a_ParcelLatA(:)
  real(DP), allocatable:: a_ParcelSigA(:)

  real(DP), allocatable:: a_UAtParcel     (:)
  real(DP), allocatable:: a_VAtParcel     (:)
  real(DP), allocatable:: a_SigDotAtParcel(:)

  integer , save       :: NIteration

  integer, save        :: FileNumber
  real(DP), save       :: OutputInterval

  character(*), parameter:: module_name = 'trajectory'
                              ! モジュールの名称. 
                              ! Module name
  character(*), parameter:: version = &
    & '$Name:  $' // &
    & '$Id: dynamics_hspl_vas83.F90,v 1.83 2014/05/07 09:39:17 murashin Exp $'
                              ! モジュールのバージョン
                              ! Module version

contains

  !--------------------------------------------------------------------------------------

  subroutine TrajectoryCalc(        &
    & xyz_UN, xyz_VN, xyz_SigDotN   & ! (in)
    & )
    !

    ! モジュール引用 ; USE statements
    !

    use constants0, only: &
      & PI

    use constants, only: &
      & RPlanet, &
                              ! $ a $ [m]. 
                              ! 惑星半径. 
                              ! Radius of planet
      & CpDry, &
                              ! $ C_p $ [J kg-1 K-1]. 
                              ! 乾燥大気の定圧比熱. 
                              ! Specific heat of air at constant pressure
      & Grav
                              ! $ g $ [m s-2]. 
                              ! 重力加速度. 
                              ! Gravitational acceleration

    ! 格子点設定
    ! Grid points settings
    !
    use gridset, only:       &
      &                imax, & ! 経度格子点数. 
                               ! Number of grid points in longitude
      &                jmax, & ! 緯度格子点数. 
                               ! Number of grid points in latitude
      &                kmax    ! 鉛直層数. 
                               ! Number of vertical level

    ! 時刻管理
    ! Time control
    !
    use timeset, only: &
      & DelTime, &            ! $ \Delta t $ [s]
      & TimeN, &              ! ステップ $ t $ の時刻. Time of step $ t $. 
      & RestartTime, &
      & TimesetClockStart, TimesetClockStop

    ! 座標データ設定
    ! Axes data settings
    !
    use axesset, only: &
      & x_Lon, &
      & y_Lat, &
      & r_Sigma, &
                              ! $ \sigma $ レベル (半整数).
                              ! Half $ \sigma $ level
      & z_Sigma               ! $ \sigma $ レベル (整数). 
                              ! Full $ \sigma $ level

    ! ヒストリデータ出力
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoPut

    ! 文字列操作
    ! Character handling
    !
    use dc_string, only: LChar

    ! 種別型パラメタ
    ! Kind type parameter
    !
    use dc_types, only: DP      ! 倍精度実数型. Double precision. 


    ! 宣言文 ; Declaration statements
    !
    implicit none

    real(DP), intent(in):: xyz_UN     (0:imax-1, 1:jmax, 1:kmax)
                              ! $ u (t) $ .     東西風速. Eastward wind
    real(DP), intent(in):: xyz_VN     (0:imax-1, 1:jmax, 1:kmax)
                              ! $ v (t) $ .     南北風速. Northward wind
    real(DP), intent(in):: xyz_SigDotN(0:imax-1, 1:jmax, 1:kmax)
                              ! Time derivative of sigma

    ! 作業変数
    ! Work variables
    !
    real(DP) :: x_Axis(0:1)
    real(DP) :: y_Axis(0:1)
    real(DP) :: z_Axis(0:1)
    real(DP) :: xyz_Val(0:1,0:1,0:1)
    real(DP) :: yz_Val (0:1,0:1)
    real(DP) :: z_Val  (0:1)
    real(DP) :: Val

    real(DP) :: TimeStep

    real(DP) :: a_ParcelLonSave(1:NParcel)
    real(DP) :: a_ParcelLatSave(1:NParcel)
    real(DP) :: a_ParcelSigSave(1:NParcel)

    integer:: l1
    integer:: l2
    integer:: l3

    integer:: i
    integer:: j

    integer:: k               ! 鉛直方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in vertical direction
    integer:: l
    integer:: m

    integer:: n               ! 組成方向に回る DO ループ用作業変数
                              ! Work variables for DO loop in dimension of constituents


    ! 実行文 ; Executable statement
    !

    ! 初期化確認
    ! Initialization check
    !
    if ( .not. trajectory_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    ! 計算時間計測開始
    ! Start measurement of computation time
    !
    call TimesetClockStart( module_name )



    ! Save position
    !
    a_ParcelLonSave = a_ParcelLonN
    a_ParcelLatSave = a_ParcelLatN
    a_ParcelSigSave = a_ParcelSigN


    loop_iteration : do m = 1, NIteration

      if ( m < NIteration ) then
        TimeStep = DelTime / 2.0_DP
      else
        TimeStep = DelTime
      end if

      loop_parcel : do l = 1, NParcel


        search_i : do i = 0+1, imax-1
          if ( x_Lon(i) > a_ParcelLonN(l) ) exit search_i
        end do search_i
        i = i - 1
        i = min( i, imax-1 )

        search_j : do j = 1+1, jmax
          if ( y_Lat(j) > a_ParcelLatN(l) ) exit search_j
        end do search_j
        j = j - 1
        j = min( j, jmax )

        search_k : do k = 1+1, kmax
          if ( z_Sigma(k) < a_ParcelSigN(l) ) exit search_k
        end do search_k
        k = k - 1
        k = min( k, kmax )

!!$      a_UAtParcel     (l) = xyz_UN     (i,j,k)
!!$      a_VAtParcel     (l) = xyz_VN     (i,j,k)
!!$      a_SigDotAtParcel(l) = xyz_SigDotN(i,j,k)


        do l1 = 0, 1
          x_Axis(l1) = x_Lon(i+l1)
        end do
        do l2 = 0, 1
          y_Axis(l2) = y_Lat(j+l2)
        end do
        do l3 = 0, 1
          z_Axis(l3) = z_Sigma(k+l3)
        end do

        loop_wind_direction : do n = 1, 3

          select case ( n )
          case ( 1 )
            do l3 = 0, 1
              do l2 = 0, 1
                do l1 = 0, 1
                  xyz_Val(l1,l2,l3) = xyz_UN     (i+l1,j+l2,k+l3)
                end do
              end do
            end do
          case ( 2 )
            do l3 = 0, 1
              do l2 = 0, 1
                do l1 = 0, 1
                  xyz_Val(l1,l2,l3) = xyz_VN     (i+l1,j+l2,k+l3)
                end do
              end do
            end do
          case ( 3 )
            do l3 = 0, 1
              do l2 = 0, 1
                do l1 = 0, 1
                  xyz_Val(l1,l2,l3) = xyz_SigDotN(i+l1,j+l2,k+l3)
                end do
              end do
            end do
          end select

          ! Interporation in longitude
          do l3 = 0, 1
            do l2 = 0, 1
              yz_Val(l2,l3) = &
                &   ( xyz_Val(1,l2,l3) - xyz_Val(0,l2,l3) ) &
                & / ( x_Axis (1)       - x_Axis (0)       ) &
                & * ( a_ParcelLonN(l)  - x_Axis (0)       ) &
                & + xyz_Val(0,l2,l3)
            end do
          end do
          ! Interporation in latitude
          do l3 = 0, 1
            z_Val(l3) = &
              &   ( yz_Val(1,l3)    - yz_Val(0,l3) ) &
              & / ( y_Axis(1)       - y_Axis(0)    ) &
              & * ( a_ParcelLatN(l) - y_Axis(0)    ) &
              & + yz_Val(0,l3)
          end do
          ! Interporation in sigma
          Val = &
            &   ( z_Val (1)       - z_Val (0)    ) &
            & / ( z_Axis(1)       - z_Axis(0)    ) &
            & * ( a_ParcelSigN(l) - z_Axis(0)    ) &
            & + z_Val(0)


          select case ( n )
          case ( 1 )
            a_UAtParcel     (l) = Val
          case ( 2 )
            a_VAtParcel     (l) = Val
          case ( 3 )
            a_SigDotAtParcel(l) = Val
          end select

        end do loop_wind_direction

      end do loop_parcel

      ! Time integration
      !
!!$    a_ParcelLonA = a_ParcelLonN &
      a_ParcelLonA = a_ParcelLonSave &
        & + a_UAtParcel      * TimeStep &
        &   * 2.0_DP * PI / ( 2.0_DP * PI * RPlanet * cos( a_ParcelLatN ) )
!!$    a_ParcelLatA = a_ParcelLatN &
      a_ParcelLatA = a_ParcelLatSave &
        & + a_VAtParcel      * TimeStep &
        &   * 2.0_DP * PI / ( 2.0_DP * PI * RPlanet )
!!$    a_ParcelSigA = a_ParcelSigN &
      a_ParcelSigA = a_ParcelSigSave &
        & + a_SigDotAtParcel * TimeStep

!!$    a_ParcelLonA = max( min( a_ParcelLonA, 2.0_DP * PI ), 0.0_DP )
      a_ParcelLonA = mod( a_ParcelLonA + 2.0_DP * PI, 2.0_DP * PI )
!!$    a_ParcelLatA = max( min( a_ParcelLatA, 2.0_DP * PI ), 0.0_DP )
      do l = 1, NParcel
        if ( a_ParcelLatA(l) > PI / 2.0_DP ) then
          a_ParcelLonA(l) = mod( a_ParcelLonA(l) + PI, 2.0_DP * PI )
          a_ParcelLatA(l) = PI / 2.0_DP - ( a_ParcelLatA(l) - PI / 2.0_DP )
        else if ( a_ParcelLatA(l) < - PI / 2.0_DP ) then
          a_ParcelLonA(l) = mod( a_ParcelLonA(l) + PI, 2.0_DP * PI )
          a_ParcelLatA(l) = - PI / 2.0_DP + ( - PI / 2.0_DP - a_ParcelLatA(l) )
        end if
      end do
      a_ParcelSigA = max( min( a_ParcelSigA, 1.0_DP ), 0.0_DP )


      ! Replace arrays
      !
      a_ParcelLonN = a_ParcelLonA
      a_ParcelLatN = a_ParcelLatA
      a_ParcelSigN = a_ParcelSigA


    end do loop_iteration


    ! Output
    !
    if ( ( TimeN - RestartTime ) >= FileNumber * OutputInterval ) then
      call MessageNotify( 'M', module_name, 'Output particle positions, %d', &
        & i = (/ FileNumber /) )
      call TrajectoryOutput
    end if



    ! 計算時間計測一時停止
    ! Pause measurement of computation time
    !
    call TimesetClockStop( module_name )

  end subroutine TrajectoryCalc

  !--------------------------------------------------------------------------------------

  subroutine TrajectoryOutput

    use constants0, only : &
      & PI


    ! Local variables
    !
    character(len=256) :: FileName
    integer            :: FU
    integer            :: ios

    integer            :: l


    ! Output
    !  A new file is created at each time step.
    !
    write( FileName, '(a,i6.6,a)' ) 'out/ParticleData', FileNumber, '.vtk'
    call findfu( FileName, ios, FU )
    open( FU, file = FileName, status = 'unknown' )
    !===== VTK format =====
    write( FU, '(a)' ) '# vtk DataFile Version 3.0'
    write( FU, '(a)' ) trim(FileName)
    write( FU, '(a)' ) 'ASCII '
    write( FU, '(a)' ) 'DATASET UNSTRUCTURED_GRID'
    write( FU, '(a,i6,a)') 'POINTS ', NParcel, ' float'
    do l = 1, NParcel
      write( FU, '(3(f10.4,1x))' ) &
        & a_ParcelLonN(l) * 180.0_DP / PI, &
        & a_ParcelLatN(l) * 180.0_DP / PI, &
        & a_ParcelSigN(l)
    end do
    write( FU, '(a,i4,a,i5)' ) 'CELLS ', NParcel, ' ', NParcel * ( 1 + 1 )
    do l = 1, NParcel
      write( FU, '(a,i6)') '1 ', l-1
    end do
    write( FU, '(a,i6)' ) 'CELL_TYPES ', NParcel
    do l = 1, NParcel
      write( FU, '(a)' ) '1'
    end do
    close( FU )


    FileNumber = FileNumber + 1


  end subroutine TrajectoryOutput

  !--------------------------------------------------------------------------------------

  subroutine findfu( fn, ios, fu, mode )

    character(len=*), intent(in )           :: fn
    integer         , intent(out)           :: ios, fu
    character(len=*), intent(in ), optional :: mode


    ! Local variables
    logical            :: od, ex
    integer, parameter :: fus = 11, fue = 99


    if( present( mode ) ) then
       if( mode .ne. "read" ) then
          write( 6, * ) 'Now, mode of "', trim( mode ), '" is not supported.'
          stop
       end if
    end if


    inquire( file = fn, opened = od, exist = ex )

    if( present( mode ) ) then
       if( ( mode .eq. "read" ) .and. ( .not. ex ) ) go to 903
    end if

    if( od ) goto 901


    do fu = fus, fue
       inquire( unit = fu, opened = od, iostat = ios )
       if( .not. od ) goto 101
    end do
    goto 902
101 continue

    ios = 0

    return


901 continue
    write( 6, * ) 'ERROR: In findfu: File ', trim( fn ), ' is open.'
    ios = 901
    fu = -1
    return

902 continue
    write( 6, * ) 'ERROR: In findfu: Unable to find a file unit.'
    ios = 902
    fu = -1
    return

903 continue
    write( 6, * ) 'ERROR: In findfu: File ', trim( fn ), ' does not exist.'
    ios = 903
    fu = -1
    return

  end subroutine findfu

  !--------------------------------------------------------------------------------------

  subroutine TrajectoryInit
    !
    ! 計算に必要なパラメタの設定や NAMELIST#trajectory_nml
    ! の読み込みを行います. 
    ! 
    ! Configure parameters for calculation, 
    ! and load "NAMELIST#trajectory_nml"
    !

    ! モジュール引用 ; USE statements
    !

    use constants0, only: &
      & PI

    ! 物理定数設定
    ! Physical constants settings
    !
    use constants, only: &
      & RPlanet, &
                              ! $ a $ [m]. 
                              ! 惑星半径. 
                              ! Radius of planet
      & Omega, &
                              ! $ \Omega $ [s-1]. 
                              ! 回転角速度. 
                              ! Angular velocity
      & GasRDry, &
                              ! $ R $ [J kg-1 K-1]. 
                              ! 乾燥大気の気体定数. 
                              ! Gas constant of air
      & CpDry
                              ! $ C_p $ [J kg-1 K-1]. 
                              ! 乾燥大気の定圧比熱. 
                              ! Specific heat of air at constant pressure

    ! 格子点設定
    ! Grid points settings
    !
    use gridset, only: nmax, & ! 最大全波数. 
                               ! Maximum truncated wavenumber
      &                lmax, & ! スペクトルデータの配列サイズ
                               ! Size of array for spectral data
      &                imax, & ! 経度格子点数. 
                               ! Number of grid points in longitude
      &                jmax, & ! 緯度格子点数. 
                               ! Number of grid points in latitude
      &                kmax    ! 鉛直層数. 
                               ! Number of vertical level

    ! 座標データ設定
    ! Axes data settings
    !
    use axesset, only: &
      & z_Sigma, &            ! $ \sigma $ レベル (整数). 
                              ! Full $ \sigma $ level
      & r_Sigma, &            ! $ \sigma $ レベル (半整数). 
                              ! Half $ \sigma $ level
      & z_DelSigma, &         ! $ \Delta \sigma $ (整数). 
                              ! $ \Delta \sigma $ (Full)
      & AxNameX, AxNameY, AxNameZ, AxNameR, AxNameT

    ! NAMELIST ファイル入力に関するユーティリティ
    ! Utilities for NAMELIST file input
    !
    use namelist_util, only: namelist_filename, NmlutilMsg

    ! ヒストリデータ出力
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoAddVariable

    ! ファイル入出力補助
    ! File I/O support
    !
    use dc_iounit, only: FileOpen

    ! 種別型パラメタ
    ! Kind type parameter
    !
    use dc_types, only: &
      & STDOUT, &             ! 標準出力の装置番号. Unit number of standard output
      & STRING                ! 文字列.       Strings. 

    ! 時刻管理
    ! Time control
    !
    use timeset, only: &
      & TimeN                 ! ステップ $ t $ の時刻. Time of step $ t $. 

    ! 宣言文 ; Declaration statements
    !
    implicit none


    integer:: unit_nml        ! NAMELIST ファイルオープン用装置番号. 
                              ! Unit number for NAMELIST file open
    integer:: iostat_nml      ! NAMELIST 読み込み時の IOSTAT. 
                              ! IOSTAT of NAMELIST read

    integer:: i
    integer:: k
    integer:: l

    ! NAMELIST 変数群
    ! NAMELIST group name
    !
    namelist /trajectory_nml/                                        &
      & NParcel, &
      & NIteration
          !
          ! デフォルト値については初期化手続 "trajectory#TrajectoryInit" 
          ! のソースコードを参照のこと. 
          !
          ! Refer to source codes in the initialization procedure
          ! "trajectory#TrajectoryInit" for the default values. 
          !

    ! 実行文 ; Executable statement
    !

    if ( trajectory_inited ) return


    ! デフォルト値の設定
    ! Default values settings
    !
    NParcel    = 18 * 9
    NIteration = 10

    ! NAMELIST の読み込み
    ! NAMELIST is input
    !
!!$    if ( trim(namelist_filename) /= '' ) then
!!$      call FileOpen( unit_nml, &          ! (out)
!!$        & namelist_filename, mode = 'r' ) ! (in)
!!$
!!$      rewind( unit_nml )
!!$      read( unit_nml, &                ! (in)
!!$        & nml = trajectory_nml,     &  ! (out)
!!$        & iostat = iostat_nml )        ! (out)
!!$      close( unit_nml )
!!$
!!$      call NmlutilMsg( iostat_nml, module_name ) ! (in)
!!$      if ( iostat_nml == 0 ) write( STDOUT, nml = trajectory_nml )
!!$    end if


    ! ヒストリデータ出力のためのへの変数登録
    ! Register of variables for history data output
    !
!!$    call HistoryAutoAddVariable( 'DUDtDyn', &
!!$      & (/ AxNameX, AxNameY, AxNameZ, AxNameT /), &
!!$      & 'dynamical tendency of zonal wind', 'm s-2' )


    ! Initialization of modules used in this module
    !

    allocate( a_ParcelLonN( NParcel ) )
    allocate( a_ParcelLatN( NParcel ) )
    allocate( a_ParcelSigN( NParcel ) )
    allocate( a_ParcelLonA( NParcel ) )
    allocate( a_ParcelLatA( NParcel ) )
    allocate( a_ParcelSigA( NParcel ) )

    allocate( a_UAtParcel     ( NParcel ) )
    allocate( a_VAtParcel     ( NParcel ) )
    allocate( a_SigDotAtParcel( NParcel ) )


    ! Initialize file number
    !
    FileNumber     = 0
    OutputInterval = 21600.0_DP

    ! Initial condition
    !
    do l = 1, NParcel
      i = mod( l, 18 )
      k = l / 18 + 1
      a_ParcelLonN( l ) =   10.0_DP * (i-1) * PI / 180.0_DP
      a_ParcelLatN( l ) =   45.0_DP * PI / 180.0_DP
      a_ParcelSigN( l ) =    0.9_DP - 0.1_DP * (k-1)
    end do
    !
    a_ParcelLonN = max( min( a_ParcelLonN, 2.0_DP * PI ), 0.0_DP )
    a_ParcelLatN = max( min( a_ParcelLatN, 2.0_DP * PI ), 0.0_DP )
    a_ParcelSigN = max( min( a_ParcelSigN, 1.0_DP ), 0.0_DP )


    ! 印字 ; Print
    !
    call MessageNotify( 'M', module_name, '----- Initialization Messages -----' )
    call MessageNotify( 'M', module_name, '  NParcel                     = %d', i = (/ NParcel /) )
    call MessageNotify( 'M', module_name, '  NIteration                  = %d', i = (/ NIteration /) )
    call MessageNotify( 'M', module_name, '  OutputInterval              = %f', d = (/ OutputInterval /) )
!!$    call MessageNotify( 'M', module_name, '  TimeIntegScheme             = %c', c1 = trim( TimeIntegScheme ) )
!!$    call MessageNotify( 'M', module_name, '  HDEFoldTime                 = %f [%c]', &
!!$      & d = (/ HDEFoldTimeValue /), c1 = trim(HDEFoldTimeUnit) )
!!$    call MessageNotify( 'M', module_name, '  FlagSpongeLayerforHeat      = %b', l = (/ FlagSpongeLayerforHeat /) )
!!$    call MessageNotify( 'M', module_name, '  SLEFoldTime                 = %f [%c]', &
!!$      & d = (/ SLEFoldTimeValue /), c1 = trim(SLEFoldTimeUnit) )
    call MessageNotify( 'M', module_name, '-- version = %c', c1 = trim(version) )

    trajectory_inited = .true.

  end subroutine TrajectoryInit

  !--------------------------------------------------------------------------------------

end module trajectory
