!= 火星 H2O 雲モデル
!
!= Mars H2O cloud model
!
! Authors::   Yoshiyuki O. Takahashi
! Version::   $Id: cloud_mars_h2o.f90,v 1.2 2015/01/29 12:06:42 yot Exp $
! Tag Name::  $Name:  $
! Copyright:: Copyright (C) GFD Dennou Club, 2008. All rights reserved.
! License::   See COPYRIGHT[link:../../../COPYRIGHT]
!

module cloud_mars_h2o
  !
  != 火星 H2O 雲モデル
  !
  != Mars H2O cloud model
  !
  ! <b>Note that Japanese and English are described in parallel.</b>
  !
  ! 
  !
  ! In this module, H2O cloud on Mars is calculated by use of a simple
  ! cloud model.
  !
  !== Procedures List
  !
!!$  ! RadiationFluxDennouAGCM :: 放射フラックスの計算
!!$  ! ------------            :: ------------
!!$  ! RadiationFluxDennouAGCM :: Calculate radiation flux
  !
  !== NAMELIST
  !
  ! NAMELIST#cloud_simple_nml
  !

  ! モジュール引用 ; USE statements

  !
  ! Kind type parameter
  !
  use dc_types, only: DP, &      ! Double precision.
    &                 STRING, &  ! Strings.
    &                 TOKEN      ! Keywords.

  ! メッセージ出力
  ! Message output
  !
  use dc_message, only: MessageNotify

  ! 格子点設定
  ! Grid points settings
  !
  use gridset, only: imax, & ! 経度格子点数.
                             ! Number of grid points in longitude
    &                jmax, & ! 緯度格子点数.
                             ! Number of grid points in latitude
    &                kmax    ! 鉛直層数.
                             ! Number of vertical level

  implicit none

  private


  ! 公開手続き
  ! Public procedure
  !
  public :: CloudMarsH2O
  public :: CloudMarsH2OCalcPartProp
  public :: CloudMarsH2OInit



  ! 公開変数
  ! Public variables
  !


  ! 非公開変数
  ! Private variables
  !
  real(DP), save :: CloudIceRad
  real(DP), save :: CloudIceNumRatio
  real(DP), save :: CloudNuclDustRad


  logical, save :: cloud_mars_h2o_inited = .false.
                              ! 初期設定フラグ.
                              ! Initialization flag

  character(*), parameter:: module_name = 'cloud_mars_h2o'
                              ! モジュールの名称.
                              ! Module name
  character(*), parameter:: version = &
    & '$Name:  $' // &
    & '$Id: cloud_mars_h2o.f90,v 1.2 2015/01/29 12:06:42 yot Exp $'
                              ! モジュールのバージョン
                              ! Module version

  !--------------------------------------------------------------------------------------

contains

  !--------------------------------------------------------------------------------------

  subroutine CloudMarsH2O(                            &
    & xyr_Press, xyz_Press, xyr_Height,               & ! (in)
    & xyz_DQH2OSolDtCum, xyz_DQH2OSolDtLSC,           & ! (in)
    & xyz_QH2OSol,                                    & ! (inout)
    & xy_Rain, xy_Snow                                & ! (out)
    & )

    ! USE statements
    !

    ! ヒストリデータ出力
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoPut

    ! 時刻管理
    ! Time control
    !
    use timeset, only: &
      & DelTime,       &     ! $ \Delta t $ [s]
      & TimeN                ! ステップ $ t $ の時刻. 
                             ! Time of step $ t $. 

    ! 重力沈降過程
    ! Gravitational sedimentation process
    !
    use grav_sed, only : GravSedAdjust


    real(DP), intent(in   ) :: xyr_Press        ( 0:imax-1, 1:jmax, 0:kmax )
    real(DP), intent(in   ) :: xyz_Press        ( 0:imax-1, 1:jmax, 1:kmax )
    real(DP), intent(in   ) :: xyr_Height       ( 0:imax-1, 1:jmax, 0:kmax )
    real(DP), intent(in   ) :: xyz_DQH2OSolDtCum( 0:imax-1, 1:jmax, 1:kmax )
    real(DP), intent(in   ) :: xyz_DQH2OSolDtLSC( 0:imax-1, 1:jmax, 1:kmax )
    real(DP), intent(inout) :: xyz_QH2OSol      ( 0:imax-1, 1:jmax, 1:kmax )
    real(DP), intent(out  ) :: xy_Rain          ( 0:imax-1, 1:jmax )
    real(DP), intent(out  ) :: xy_Snow          ( 0:imax-1, 1:jmax )


    real(DP) :: xyz_PartDen     (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_PartRad     (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyr_SnowMassFlux(0:imax-1, 1:jmax, 0:kmax)


    ! 実行文 ; Executable statement
    !

    ! 初期化確認
    ! Initialization check
    !
    if ( .not. cloud_mars_h2o_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    xyz_QH2OSol = xyz_QH2OSol &
      & + ( xyz_DQH2OSolDtCum + xyz_DQH2OSolDtLSC ) * 2.0_DP * DelTime


    xy_Rain = 0.0_DP
    xy_Snow = 0.0_DP

    call CloudMarsH2OCalcPartProp( &
      & xyr_Press, xyz_Press,            & ! (in )
      & xyz_QH2OSol,                     & ! (in )
      & xyz_PartRad, xyz_PartDen         & ! (out)
      & )

    ! 重力沈降過程
    ! Gravitational sedimentation process
    !
    call GravSedAdjust(                    &
      & xyr_Press, xyz_Press, xyr_Height,  & ! (in )
      & xyz_PartDen, xyz_PartRad,          & ! (in )
      & xyz_QH2OSol,                       & ! (out)
      & xyr_SnowMassFlux                   & ! (out) optional
      & )
    xy_Snow = xyr_SnowMassFlux(:,:,0)
    xy_Snow = - xy_Snow


    ! ヒストリデータ出力
    ! History data output
    !
    call HistoryAutoPut( TimeN, 'MarsH2OCloudRadius', xyz_PartRad )


  end subroutine CloudMarsH2O

  !------------------------------------------------------------------------------------

  subroutine CloudMarsH2OCalcPartProp( &
    & xyr_Press, xyz_Press,            & ! (in )
    & xyz_QMix,                        & ! (out)
    & xyz_PartRad, xyz_PartDen         & ! (out)
    & )

    ! ヒストリデータ出力
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoPut

    ! 物理・数学定数設定
    ! Physical and mathematical constants settings
    !
    use constants0, only: &
      & PI                    ! $ \pi $ .
                              ! 円周率.  Circular constant

    ! 物理定数設定
    ! Physical constants settings
    !
    use constants, only: &
      & Grav
                              ! $ g $ [m s-2].
                              ! 重力加速度.
                              ! Gravitational acceleration

    ! 短波入射 (太陽入射)
    ! Short wave (insolation) incoming
    !
    use rad_short_income, only : RadShortIncome

    use set_Mars_dust, only : &
      & SetMarsDustSetDOD067, &
      & SetMarsDustCalcDOD067

    real(DP)    , intent(in   ) :: xyr_Press  (0:imax-1, 1:jmax, 0:kmax)
    real(DP)    , intent(in   ) :: xyz_Press  (0:imax-1, 1:jmax, 1:kmax)
    real(DP)    , intent(in   ) :: xyz_QMix   (0:imax-1, 1:jmax, 1:kmax)
    real(DP)    , intent(out  ) :: xyz_PartRad(0:imax-1, 1:jmax, 1:kmax)
    real(DP)    , intent(out  ) :: xyz_PartDen(0:imax-1, 1:jmax, 1:kmax)


    !
    ! local variables
    !
    ! rhod : dust density
    ! mfp  : mean free path
    ! mvc  : modecular viscosity coefficient
    ! rdia : particle diameter
    !
    real(DP) :: PartDen

    real(DP), parameter :: AMU = 1.6605655e-27_DP

    real(DP) :: MeanMolMass
    real(DP) :: DustExtEff
    real(DP) :: DustR
    real(DP) :: DustDen
    real(DP) :: xyz_DelDustNum (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_DelIceNum  (0:imax-1, 1:jmax, 1:kmax)

    real(DP) :: PlanetLonFromVE
    real(DP) :: xyr_DOD067     (0:imax-1, 1:jmax, 0:kmax)

    integer  :: i
    integer  :: j
    integer  :: k


    ! 実行文 ; Executable statement
    !

    ! 初期化確認
    ! Initialization check
    !
    if ( .not. cloud_mars_h2o_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if


    !
    ! calculation of sedimentation terminal velocity
    !

    !
    ! The values below are obtained from Conrath (1975). 
    !
    PartDen         = 1.0e3_DP   ! ice density
    !

    if ( CloudIceRad > 0.0_DP ) then
      xyz_PartDen     = PartDen
      xyz_PartRad     = CloudIceRad
    else
      if ( CloudIceNumRatio < 0.0_DP ) then

        ! 短波入射の計算
        ! Calculate short wave (insolation) incoming radiation
        !
        call RadShortIncome(                       &
          & PlanetLonFromVE   = PlanetLonFromVE,   & ! (out) optional
          & FlagOutput = .false.                   & ! (in)  optional
          & )
        PlanetLonFromVE = PlanetLonFromVE * 180.0_DP / PI

        call SetMarsDustSetDOD067(                  &
          & PlanetLonFromVE, xyr_Press, xyz_Press,  & ! (in)
          & xyr_DOD067,                             & ! (out)
          & FlagOutput = .false.                    & ! (in)  optional
          & )

        DustExtEff   = 3.04_DP    ! Ockert-Bell et al. (1997)
        DustR        = CloudNuclDustRad
        DustDen      = 3.0e3_DP
        do k = 1, kmax
          xyz_DelDustNum(:,:,k) =                           &
            &   ( xyr_DOD067(:,:,k-1) - xyr_DOD067(:,:,k) ) &
            & / ( DustExtEff * PI * DustR**2 )
        end do
        xyz_DelIceNum = xyz_DelDustNum
      else
        DustR        = 0.0_DP
        DustDen      = 3.0e3_DP
        MeanMolMass  = 44.0_DP * AMU
        do k = 1, kmax
          xyz_DelIceNum(:,:,k) = &
            &   ( xyr_Press(:,:,k-1) - xyr_Press(:,:,k) ) / Grav &
            & / MeanMolMass &
            & * CloudIceNumRatio
        end do
      end if
      ! arbitrary minimum
      xyz_DelIceNum = max( xyz_DelIceNum, 1.0_DP )
      ! calculate radius
      do k = 1, kmax
        xyz_PartRad(:,:,k) =                                                &
          & (                                                               &
          &     xyz_QMix(:,:,k)                                             &
          &       * ( xyr_Press(:,:,k-1) - xyr_Press(:,:,k) ) / Grav        &
          &       / ( xyz_DelIceNum(:,:,k) * PartDen * 4.0_DP / 3.0_DP * PI ) &
          &   + DustR**3                                                &
          & )**(1.0_DP/3.0_DP)
      end do
      ! arbitrary minimum
      xyz_PartRad = max( xyz_PartRad, 1.0e-10_DP )
      !
      xyz_PartDen = &
        &   ( &
        &     DustR**3 * DustDen + ( xyz_PartRad**3 - DustR**3 ) * PartDen &
        &   ) &
        & / ( xyz_PartRad**3 )
    end if


  end subroutine CloudMarsH2OCalcPartProp

  !--------------------------------------------------------------------------------------

  subroutine CloudMarsH2OInit

    ! ファイル入出力補助
    ! File I/O support
    !
    use dc_iounit, only: FileOpen

    ! NAMELIST ファイル入力に関するユーティリティ
    ! Utilities for NAMELIST file input
    !
    use namelist_util, only: namelist_filename, NmlutilMsg, NmlutilAryValid

    ! ヒストリデータ出力
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoAddVariable

    ! 座標データ設定
    ! Axes data settings
    !
    use axesset, only: &
      & AxnameX, &
      & AxnameY, &
      & AxnameZ, &
      & AxnameR, &
      & AxnameT

    ! 重力沈降過程
    ! Gravitational sedimentation process
    !
    use grav_sed, only : GravSedInit

    ! 短波入射 (太陽入射)
    ! Short wave (insolation) incoming
    !
    use rad_short_income, only : RadShortIncomeInit


    use set_Mars_dust, only : SetMarsDustInit


    ! 宣言文 ; Declaration statements
    !

    integer:: unit_nml        ! NAMELIST ファイルオープン用装置番号.
                              ! Unit number for NAMELIST file open
    integer:: iostat_nml      ! NAMELIST 読み込み時の IOSTAT.
                              ! IOSTAT of NAMELIST read

    ! NAMELIST 変数群
    ! NAMELIST group name
    !
    namelist /cloud_mars_h2o_nml/ &
      & CloudIceRad,              &
      & CloudIceNumRatio,         &
      & CloudNuclDustRad
          !
          ! デフォルト値については初期化手続 "cloud_mars_h2o#CloudMarsH2OInit"
          ! のソースコードを参照のこと.
          !
          ! Refer to source codes in the initialization procedure
          ! "cloud_mars_h2o#CloudMarsH2OInit" for the default values.
          !

    ! 実行文 ; Executable statement
    !

    if ( cloud_mars_h2o_inited ) return


    ! デフォルト値の設定
    ! Default values settings
    !
    CloudIceRad      =  2.0d-6
    CloudIceNumRatio = -1.0d0
    CloudNuclDustRad =  1.0d-6


    ! NAMELIST の読み込み
    ! NAMELIST is input
    !
    if ( trim(namelist_filename) /= '' ) then
      call FileOpen( unit_nml, &          ! (out)
        & namelist_filename, mode = 'r' ) ! (in)

      rewind( unit_nml )
      read( unit_nml,                     & ! (in)
        & nml = cloud_mars_h2o_nml,       & ! (out)
        & iostat = iostat_nml )             ! (out)
      close( unit_nml )

      call NmlutilMsg( iostat_nml, module_name ) ! (in)
    end if



    ! Initialization of modules used in this module
    !

    ! 重力沈降過程
    ! Gravitational sedimentation process
    !
    call GravSedInit

    ! 短波入射 (太陽入射)
    ! Short wave (insolation) incoming
    !
    call RadShortIncomeInit

    call SetMarsDustInit


    ! ヒストリデータ出力のためのへの変数登録
    ! Register of variables for history data output
    !
    call HistoryAutoAddVariable( 'MarsH2OCloudRadius', &
      & (/ AxnameX, AxnameY, AxnameZ, AxnameT /), &
      & 'Mars H2O cloud radius', &
      & 'm' )


    ! 印字 ; Print
    !
    call MessageNotify( 'M', module_name, '----- Initialization Messages -----' )
    call MessageNotify( 'M', module_name, ' CloudIceRad      = %f', d = (/ CloudIceRad /) )
    call MessageNotify( 'M', module_name, ' CloudIceNumRatio = %f', d = (/ CloudIceNumRatio /) )
    call MessageNotify( 'M', module_name, ' CloudNuclDustRad = %f', d = (/ CloudNuclDustRad /) )
    call MessageNotify( 'M', module_name, '-- version = %c', c1 = trim(version) )


    cloud_mars_h2o_inited = .true.

  end subroutine CloudMarsH2OInit

  !--------------------------------------------------------------------------------------

end module cloud_mars_h2o
