module pbl_mym_condensation
  use pp_vardef
  implicit none

  private

  public :: pbl_mym_condensation_run

  real(r_size) :: rr2
  real(r_size) :: rrp
  real(r_size) :: temp_ice

  logical :: l_first = .true.


contains
  subroutine pbl_mym_condensation_run(temp, ptl, qw, prs, tsq, qsq, cov, &
    & vt, vq, cld, ql, q1)
    use pp_vardef
    use pbl_grid, only: nz
    use pp_phys_const, only: pi, epsilon, one_minus_epsilon, &
      & ttriple, tkelvn, hlatnt, hlf, rd, cp, rdvcp, &
      & e0cw, tetn1w, tetn2w, tetn3w, &
      & e0ci, tetn1i, tetn2i, tetn3i, &
      & c_virtual
    use pbl_const, only: pref
    use pbl_mym_option, only: levflag
    implicit none

    real(r_size), intent(in) :: temp(nz)
    real(r_size), intent(in) :: ptl(nz)
    real(r_size), intent(in) :: qw(nz)
    real(r_size), intent(in) :: prs(nz)
    real(r_size), intent(in) :: tsq(nz)
    real(r_size), intent(in) :: qsq(nz)
    real(r_size), intent(in) :: cov(nz)

    real(r_size), intent(out) :: vt(nz)
    real(r_size), intent(out) :: vq(nz)
    real(r_size), intent(out) :: cld(nz)
    real(r_size), intent(out) :: ql(nz)
    real(r_size), intent(out) :: q1(nz)


    integer(4) :: kz
    real(r_size) :: tl
    real(r_size) :: qsw
    real(r_size) :: qsi
    real(r_size) :: rice
    real(r_size) :: hl
    real(r_size) :: qsl
    real(r_size) :: dqsl
    real(r_size) :: r_exner
    real(r_size) :: t3sq
    real(r_size) :: r3sq
    real(r_size) :: c3sq
    real(r_size) :: alp_qsl
    real(r_size) :: eq1
    real(r_size) :: qll
    real(r_size) :: q2p
    real(r_size) :: pt_tmp
    real(r_size) :: qt
    real(r_size) :: rac

    real(r_size) :: hl_ovr_cp(nz)
    real(r_size) :: exner(nz)
    real(r_size) :: qmq(nz)
    real(r_size) :: alp(nz)
    real(r_size) :: bet(nz)
    real(r_size) :: sgm(nz)
    real(r_size) :: erf_arg(nz)
    real(r_size) :: erf_val(nz)

    real(r_size), parameter :: my_sgm_max_fct = 1.0d0
    real(r_size), parameter :: my_sgm_min_fct = 0.0d0


    if (l_first) then
      rr2 = 1.0 / sqrt(2.0)
      rrp = 1.0 / sqrt(2.0 * pi)
      temp_ice = tkelvn - 36.0
      l_first = .false.
    end if

    ! Here, qsw and qsi are saturated vapor pressure.
    ! Using the Teten's formula instead of the subroutine "qmix"
    ! because the saturated vapor pressure on liquid water is necessary
    ! even in sub-zero temperature.
    do kz = 1, nz
      exner(kz) = (prs(kz) / pref) ** rdvcp

      tl = ptl(kz) * exner(kz)

      qsw = e0cw * exp(tetn1w *                              &
        (tl  - tetn2w) / (tl - tetn3w) )
      qsi = e0ci * exp(tetn1i *                              &
        (tl - tetn2i)  / (tl - tetn3i) )

    ! convert to specific humidity
      qsw = epsilon * qsw / (prs(kz)                &
        - one_minus_epsilon * qsw)
      qsi = epsilon * qsi / (prs(kz)                &
        - one_minus_epsilon * qsi)

    ! calculate sgm
      if (tl >= ttriple) then
        rice = 0.0
      else if (tl < temp_ice) then
        rice = 1.0
      else
        rice = (ttriple - tl) / (ttriple - temp_ice)
      end if

      hl = (1.0 - rice) * hlatnt + rice * hlf
      qsl = (1.0 - rice) * qsw + rice * qsi

      hl_ovr_cp(kz) = hl / cp

      dqsl = qsl * epsilon * hl / (rd * tl **2)

      qmq(kz) = qw(kz) - qsl
      alp(kz) = 1.0 /(1.0 + dqsl * hl_ovr_cp(kz))
      bet(kz) = dqsl * exner(kz)

      t3sq = max(tsq(kz), 0.0_r_size)
      r3sq = max(qsq(kz), 0.0_r_size)
      c3sq = cov(kz)
      c3sq = sign(min(abs(c3sq), sqrt(t3sq * r3sq)), c3sq)
      
      r3sq = r3sq + bet(kz) ** 2 * t3sq                          &
        -2.0 * bet(kz) * c3sq
      alp_qsl = min(alp(kz) * qsl, qw(kz))
      sgm(kz) = max(                                             &
        min(0.5_r_size * alp(kz) * sqrt(max(r3sq, 0.0_r_size)),  &
        my_sgm_max_fct * alp_qsl),                      &
        my_sgm_min_fct * alp_qsl, 1.e-10_r_size)
    end do

    if (levflag /= 3) then
      sgm(1) = sgm(2)
    end if

    !

    ! Preparation to calculate values of the err function
    do kz = 1, nz
      q1(kz) = 0.5 * alp(kz) * qmq(kz) / sgm(kz)
      erf_arg(kz) = q1(kz) * rr2
    end do

    call pbl_mym_errfunc(nz, erf_arg, erf_val)

    ! Calculate the buoyancy parameters vt and vq
    do kz = 1, nz
      cld(kz) = 0.5 * (1.0 + erf_val(kz))
      if(abs(q1(kz)) > 10.0 ) then
        eq1 = 0.0
      else
        eq1  = rrp * exp(- 0.5 * q1(kz) ** 2)
      end if
      ! qll = ql / (2 * sgm)
      qll  = max(cld(kz) * q1(kz) + eq1, 0.0_r_size)

      if(qw(kz) < 1.e-10) then
        ql(kz) = 0.0
      else
        ql(kz) = max(2.0_r_size * sgm(kz) * qll, 0.0_r_size)
      end if
      ! To avoid negative QV (for safety)
      ql(kz) = min(ql(kz), qw(kz) * 0.5_r_size)

      r_exner = 1.0 / exner(kz)
      q2p  = hl_ovr_cp(kz) * r_exner
      pt_tmp = temp(kz) * r_exner
      qt = 1.0 + c_virtual * qw(kz) - (1.0 + c_virtual) * ql(kz)
      rac = alp(kz) * (cld(kz) - qll * eq1) * &
        &            (q2p * qt - (1.0 + c_virtual) * pt_tmp)
      
      vt (kz) = qt - rac * bet(kz)
      vq (kz) = c_virtual * pt_tmp + rac
    end do

    return
  end subroutine pbl_mym_condensation_run
!
  subroutine pbl_mym_errfunc(nn, x, y)
    use pp_vardef
    use pp_phys_const, only: pi

    implicit none

    integer(4), intent(in) :: nn       ! size of array
    real(r_size), intent(in)    :: x(nn)    ! input array
    real(r_size), intent(out)   :: y(nn)    ! output array


    ! Local Variables
    integer(4)             :: i        ! Loop index

    real(r_size), save :: c01
                                ! expansion coefficient of x
    real(r_size), save :: c03
                                ! expansion coefficient of x**3
    real(r_size), save :: c05
                                ! expansion coefficient of x**5
    real(r_size), save :: c07
                                ! expansion coefficient of x**7
    real(r_size), save :: c09
                                ! expansion coefficient of x**9
    real(r_size), save :: c11
                                ! expansion coefficient of x**11
    real(r_size), save :: c13
                                ! expansion coefficient of x**13
    real(r_size), save :: factor
                               ! common factor to all the coefficients

    real(r_size) :: x02
                                ! x powered by 2
    real(r_size) :: x04
                                ! x powered by 4
    real(r_size) :: x06
                                ! x powered by 6
    real(r_size) :: x08
                                ! x powered by 8
    real(r_size) :: x10
                                ! x powered by 10
    real(r_size) :: x12
                                ! x powered by 12

    logical, save       :: first = .true.
                                ! flag to indication first run

    real(r_size), parameter :: erfmax = 1.0
                    ! upper limit of the value to avoid it outside domain

    if (first) then
      factor =  2.0 / sqrt(pi)
      c01 = factor * 1.0
      c03 = factor * 1.0 /    3.0
      c05 = factor * 1.0 /   10.0
      c07 = factor * 1.0 /   42.0
      c09 = factor * 1.0 /  216.0
      c11 = factor * 1.0 / 1320.0
      c13 = factor * 1.0 / 9360.0
      first = .false.
    end if
    do i = 1, nn
      x02 = x(i) * x(i)
      x04 = x02 * x02
      x06 = x04 * x02
      x08 = x06 * x02
      x10 = x08 * x02
      x12 = x10 * x02
      y(i) = x(i) * (                                                     &
        + c01                                                         &
        - c03 * x02                                                   &
        + c05 * x04                                                   &
        - c07 * x06                                                   &
        + c09 * x08                                                   &
        - c11 * x10                                                   &
        + c13 * x12)
      if(x(i) > 0) then
        y(i) = min(y(i), erfmax)
      else
        y(i) = max(y(i), -erfmax)
      end if
    end do
    return
  end subroutine pbl_mym_errfunc

end module pbl_mym_condensation



