!= ϳز (Arakawa and Suarez (1983))
!
!= Dynamical Core (Arakawa and Suarez (1983))
!
! Authors::   Yasuhiro MORIKAWA
! Version::   $Id: dyn_as83.f90,v 1.25 2007/09/09 15:23:50 morikawa Exp $
! Tag Name::  $Name: dcpam4-20070909 $
! Copyright:: Copyright (C) GFD Dennou Club, 2007. All rights reserved.
! License::   See COPYRIGHT[link:../../../COPYRIGHT]
!

module dyn_as83
  !
  != ϳإ (Arakawa and Suarez (1983))
  !
  != Dynamical Core (Arakawa and Suarez (1983))
  !
  ! <b>Note that Japanese and English are described in parallel.</b>
  !
  ! ϳإ⥸塼Ǥ. ľΥˤϰʲμˡѤƤޤ.
  !
  ! This is a dynamical core module. Used vertical discretization
  ! method is as follows.
  !
  ! * Arakawa and Suarez (1983)
  !
  ! ʬˡˤϥ꡼ץեåѤƤޤ. 
  ! ǥեȤǤ, $ \Delta t $ 礭Ȥ뤿, ȹ 
  ! ߥץꥷåˡŬѤƤޤ. 
  ! ȹ򥨥ץꥷåˡˤäƲ򤯤ȤǽǤ. 
  ! ܤ Create  NAMELIST#dyn_as83_nml 򻲾ȤƤ.
  !
  ! Leap-frog scheme is used as time integration. 
  ! By default, semi-implicit scheme is applied to gravitational terms 
  ! for extension of $ \Delta t $ . 
  ! Explicit scheme can be applied to gravitational terms. 
  ! For details, see "Create" or "NAMELIST#dyn_as83_nml". 
  !
  !== Procedures List
  !
  ! Create          :: DYNAS83 ѿν
  ! Close           :: DYNAS83 ѿνλ
  ! PutLine         :: DYNAS83 ѿ˳ǼƤΰ
  ! initialized     :: DYNAS83 ѿꤵƤ뤫ݤ
  ! Inquire         :: DYNAS83 ѿ˳ǼƤμ
  ! EqualAxes       :: DYNAS83 ѿ˳ǼƤɸͤγǧ
  ! GetAxes         :: DYNAS83 ѿ˳ǼƤɸͤμ
  ! NonLinearOnGrid ::  (ȹ) γʻǤη׻
  ! TimeIntegration :: ʬμ¹
  ! WTplusGPiOnGrid :: $ \underline{W} \Dvect{T} + \Dvect{G} \pi $ η׻
  !                    (ץꥷåˡǻ)
  ! HDivOnGrid      :: $ \underline{h} \Dvect{D} $ η׻
  !                    (ץꥷåˡǻ)
  ! ------------    :: ------------
  ! Create          :: Constructor of "DYNAS83"
  ! Close           :: Deconstructor of "DYNAS83"
  ! PutLine         :: Print information of "DYNAS83"
  ! initialized     :: Check initialization of "DYNAS83"
  ! Inquire         :: Get information of "DYNAS83"
  ! EqualAxes       :: Confirm data of axes in "DYNAS83"
  ! GetAxes         :: Get data of axes in "DYNAS83"
  ! NonLinearOnGrid :: Calculate non-linear terms (non gravitational terms) 
  !                    on grid points
  ! TimeIntegration :: Execute time integration
  ! WTplusGPiOnGrid :: Calculate $ \underline{W} \Dvect{T} + \Dvect{G} \pi $
  !                    (For explicit scheme)
  ! HDivOnGrid      :: Calculate $ \underline{h} \Dvect{D} $
  !                    (For explicit scheme)
  !
  !== Usage
  !
  ! Ϥ, DYNAS83 ѿ, Create ǽꤷޤ.
  !  (ȹ) γʻǤη׻Ԥˤ
  ! NonLinearOnGrid ѤƤ.
  ! ʬԤݤˤ TimeIntegration ѤƤ.
  ! DYNAS83 ѿνλˤ Close Ѥޤ.
  ! ɸͤɬפȤʤˤ, GetAxes Ѥ
  ! ɸͤƤ.
  ! 
  ! First, initialize "DYNAS83" by "Create".
  ! Use NonLinearOnGrid for calculation of non-linear terms (non
  ! gravitational terms) on grid points.
  ! Use TimeIntegration for time integration.
  ! In order to terminate "DYNAS83", use "Close".
  ! If you need data of axes, get that by GetAxes.
  !
  !== References
  !
  ! * Arakawa, A., Suarez, M. J., 1983:
  !   Vertical differencing of the primitive equations
  !   in sigma coordinates.
  !   <i>Mon. Wea. Rev.</i>, <b>111</b>, 34--35.


  !----- ѥ桼ƥƥ -----
  use dc_types, only: DP, TOKEN
  use dyn_matrix, only: DYNMTX
  implicit none
  private
  public:: DYNAS83, Create, Close, PutLine, initialized
  public:: Inquire, EqualAxes
  public:: GetAxes, NonLinearOnGrid, TimeIntegration
  public:: WTplusGPiOnGrid, HDivOnGrid
!!$  public:: SetTime

  type DYNAS83
    !
    ! ޤ, Create  "DYNAS83" ѿꤷƲ.
    ! ꤵ줿 "DYNAS83" ѿѤݤˤ,
    ! Close ˤäƽλԤäƤ.
    !
    ! Initialize "DYNAS83" variable by "Create" before usage.
    ! If you reuse "DYNAS83" variable again for another application, 
    ! terminate by "Close".
    !
    logical:: initialized = .false.    ! ե饰. 
                                       ! Initialization flag
    integer:: nmax ! ȿ. 
                   ! Maximum truncated wavenumber
    integer:: imax ! ٳʻ. 
                   ! Number of grid points in longitude
    integer:: jmax ! ٳʻ. 
                   ! Number of grid points in latitude
    integer:: kmax ! ľؿ. 
                   ! Number of vertical level
    character(TOKEN):: time_integration_scheme
                              ! ʬˡ. 
                              ! ʲˡǽ. 
                              !
                              ! Time integration scheme. 
                              ! Available schemes are as follows.
                              ! 
                              ! * "Semi-implicit"
                              ! * "Explicit"
                              ! 
    real(DP), pointer:: z_Sigma (:) =>null()
                              ! $ \sigma $ ٥ (). 
                              ! Full $ \sigma $ level
    real(DP), pointer:: r_Sigma (:) =>null()
                              ! $ \sigma $ ٥ (Ⱦ). 
                              ! Half $ \sigma $ level
    real(DP), pointer:: z_DelSigma (:) =>null()
                              ! $ \Delta \sigma $ (). 
                              ! $ \Delta \sigma $ (Full)

    real(DP):: PI         ! $ \pi $ .    ߼Ψ.         Circular constant
    real(DP):: RPlanet    ! $ a $ .      Ⱦ.       Radius of planet
!!$    real(DP):: Omega      ! $ \Omega $ . ž®.     Angular velocity
!!$    real(DP):: Grav       ! $ g $ .      ϲ®.     Gravitational acceleration
    real(DP):: Cp         ! $ C_p $ .    絤갵Ǯ.   Specific heat of air at constant pressure
    real(DP):: RAir       ! $ R $ .      絤.   Gas constant of air
    real(DP):: EpsVT      ! $ 1/\epsilon_v - 1 $ .
    real(DP):: Kappa      ! $ \kappa = R/C_p $ .
                          ! 갵ǮФ. Ratio of gas constant to specific heat
    real(DP):: DelTime    ! $ \Delta t $ . ॹƥå. Time step
    real(DP), pointer:: xy_Cori (:,:) =>null()
                              ! $ f\equiv 2\Omega\sin\varphi $ . 
                              ! ꥪѥ᡼. Coriolis parameter
    integer, pointer:: nmo(:,:,:) =>null()
                              ! ڥȥź. 
                              ! Spectral subscript expression
    real(DP), pointer:: wz_rn (:,:) =>null()
                              ! $ -n \times (n+1) $ . ץ饷η. 
                              ! Laplacian coefficient
    real(DP), pointer:: wz_DiffVorDiv (:,:) =>null()
                              ! $ -K_{HD} [(-1)^{N_D/2}\nabla^{N_D}- (2/a^2)^{N_D/2}] $ . 
                              ! ư̿ʿȻ. 
                              ! Coefficient of horizontal momentum diffusion
    real(DP), pointer:: wz_DiffTherm (:,:) =>null()
                              ! $ -(-1)^{N_D/2}K_{HD}\nabla^{N_D} $ . 
                              ! Ǯ, ʿȻ. 
                              ! Coefficient of horizontal thermal and water diffusion
    real(DP), pointer:: w_Phis (:) =>null()
                              ! $ \Phi_s $ . ɽݥƥ󥷥. 
                              ! Surface geo-potential
    real(DP), pointer:: z_TempAvrXY (:) =>null()
                              ! ʿѲ (٥).
                              ! Average temperature on full sigma level
    real(DP), pointer:: r_TempAvrXY (:) =>null()
                              ! ʿѲ (Ⱦ٥). 
                              ! Average temperature on half sigma level
    real(DP), pointer:: z_HydroAlpha (:) =>null()
                              ! $ \alpha $ . ſ尵μη. 
                              ! Coefficient in hydrostatic equation.
    real(DP), pointer:: z_HydroBeta (:) =>null()
                              ! $ \beta $ . ſ尵μη. 
                              ! Coefficient in hydrostatic equation.
    real(DP), pointer:: z_TempInpolKappa (:) =>null()
                              ! $ \kappa $ . ٱľ֤η. 
                              ! Coefficient for vertical interpolation of temperature
    real(DP), pointer:: z_TempInpolA (:) =>null()
                              ! $ a $ . ٱľ֤η. 
                              ! Coefficient for vertical interpolation of temperature
    real(DP), pointer:: z_TempInpolB (:) =>null()
                              ! $ b $ . ٱľ֤η. 
                              ! Coefficient for vertical interpolation of temperature

    !-------------------------------------------------------
    !  ߥץꥷåˡΤι
    !  Matrices for semi-implicit method
    real(DP), pointer:: z_siMtxG (:)   =>null()
                              ! $ G = C_p \kappa T $
    real(DP), pointer:: zz_siMtxH (:,:)  =>null()
                              ! $ h = QS - R $
    real(DP), pointer:: zz_siMtxWH (:,:)  =>null()
                              ! $ W h $
    real(DP), pointer:: zz_siMtxGCt (:,:)  =>null()
                              ! $ G C^{T} $ ( $ C = \Delta \sigma $ )

    !-------------------------------------------------------
    !  ߥץꥷåȻʬѹ
    !  Matrixes for semi-implicit time integration
    type(DYNMTX):: dyn_mtx

    !-------------------------------------------------------
    !  ҥȥեؤΥǡ
    !  Settings for history data output
    real(DP):: history_interval_value
                              ! ҥȥǡνϴֳ֤ñ. 
                              ! Unit for interval of history data output
    character(TOKEN):: history_interval_unit
                              ! ҥȥǡνϴֳ֤ñ. 
                              ! Unit for interval of history data output
    character(TOKEN):: history_precision
                              ! ҥȥǡ. 
                              ! Precision of history data

  end type DYNAS83

  character(*), parameter:: version = &
    & '$Name: dcpam4-20070909 $' // &
    & '$Id: dyn_as83.f90,v 1.25 2007/09/09 15:23:50 morikawa Exp $'

  interface Create
    module procedure DynAS83Create
  end interface

  interface Close
    module procedure DynAS83Close
  end interface

  interface PutLine
    module procedure DynAS83PutLine
  end interface

  interface initialized
    module procedure DynAS83Initialized
  end interface

  interface NmlRead
    module procedure DynAS83NmlRead
  end interface

  interface Inquire
    module procedure DynAS83Inquire
  end interface

  interface EqualAxes
    module procedure DynAS83EqualAxes
  end interface

  interface GetAxes
    module procedure DynAS83GetAxes
  end interface

  interface NonLinearOnGrid
    module procedure DynAS83NonLinearOnGrid
  end interface

  interface TimeIntegration
    module procedure DynAS83TimeIntegration
  end interface

  interface WTplusGPiOnGrid
    module procedure DynAS83WTplusGPiOnGrid
  end interface

  interface HDivOnGrid
    module procedure DynAS83HDivOnGrid
  end interface

contains

  subroutine DynAS83Create( dyn_as, &
    & nmax, imax, jmax, kmax, &
    & PI, RPlanet, &
!!$    & Grav, Omega, &
    & Cp, RAir, EpsVT, &
    & DelTime, xy_Cori, nmo, wz_rn, wz_DiffVorDiv, wz_DiffTherm, &
    & r_SigmaSet, w_Phis, &
    & time_integration_scheme, &
    & history_interval_value, &
    & history_interval_unit, history_precision, &
    & nmlfile, err )
    ! 
    !  *dyn_as*  Arakawa and Suarez (1983) Ѥ黻
    ! Ԥޤ.
    ! ¾Υ֥롼ѤɬΥ֥롼ˤä DYNAS83 
    ! ѿꤷƤ.
    !
    ! NAMELIST Ѥˤϰ *nmlfile*  NAMELIST ե̾
    ! ͿƤ. NAMELIST ѿξܺ٤˴ؤƤ 
    ! NAMELIST#dyn_as83_nml 򻲾ȤƤ. 
    !
    ! Configure the settings for Arakawa and Suarez (1983) to *dyn_as*.
    ! Before other subroutines are used, initialize "DYNAS83"
    ! variable.
    !
    ! In order to use NAMELIST, specify a NAMELIST filename to 
    ! argument *nmlfile*. See "NAMELIST#dyn_as83_nml"
    ! for details about a NAMELIST group.
    !
    use dc_types, only: DP, STRING
    use dc_message, only: MessageNotify
    use dc_trace, only: BeginSub, EndSub, DbgMessage
    use dc_present, only: present_and_not_empty, present_and_true
    use dc_error, only: DC_ENOFILEREAD
    use dc_string, only: LChar, PutLine
    use dcpam_error, only: StoreError, DC_NOERR, &
      & DCPAM_EALREADYINIT, DCPAM_ENEGATIVE, &
      & DCPAM_ENOVARDEF, DCPAM_EARGSIZEMISMATCH, &
      & DCPAM_EFAILINIT, DCPAM_EBADSCHEME
    use dyn_matrix, only: Create
    use sigma_data, only: SIGDAT, Create, Get, Close
    implicit none
    type(DYNAS83), intent(inout):: dyn_as
    integer, intent(in):: nmax ! ȿ. 
                               ! Maximum truncated wavenumber
    integer, intent(in):: imax ! ٳʻ. 
                               ! Number of grid points in longitude
    integer, intent(in):: jmax ! ٳʻ. 
                               ! Number of grid points in latitude
    integer, intent(in):: kmax ! ľؿ. 
                               ! Number of vertical level
    real(DP), intent(in):: PI         ! $ \pi $ .    ߼Ψ.         Circular constant
    real(DP), intent(in):: RPlanet    ! $ a $ .      Ⱦ.       Radius of planet
!!$    real(DP), intent(in):: Omega      ! $ \Omega $ . ž®.     Angular velocity
!!$    real(DP), intent(in):: Grav       ! $ g $ .      ϲ®.     Gravitational acceleration
    real(DP), intent(in):: Cp         ! $ C_p $ .    絤갵Ǯ.   Specific heat of air at constant pressure
    real(DP), intent(in):: RAir       ! $ R $ .      絤.   Gas constant of air
    real(DP), intent(in):: EpsVT      ! $ 1/\epsilon_v - 1 $ .
    real(DP), intent(in):: DelTime ! $ \Delta t $ . ॹƥå. Time step
    real(DP), intent(in):: xy_Cori (0:imax-1, 0:jmax-1)
                              ! $ f\equiv 2\Omega\sin\varphi $ . 
                              ! ꥪѥ᡼. Coriolis parameter
    integer, intent(in):: nmo (1:2, 0:nmax, 0:nmax)
                              ! ڥȥź. 
                              ! Spectral subscript expression
    real(DP), intent(in):: wz_rn ((nmax+1)**2, 0:kmax-1)
                              ! $ -n \times (n+1) $ . ץ饷η. 
                              ! Laplacian coefficient
    real(DP), intent(in):: wz_DiffVorDiv ((nmax+1)**2, 0:kmax-1)
                              ! $ -K_{HD} [(-1)^{N_D/2}\nabla^{N_D}- (2/a^2)^{N_D/2}] $ . 
                              ! ư̿ʿȻ. 
                              ! Coefficient of horizontal momentum diffusion
    real(DP), intent(in):: wz_DiffTherm ((nmax+1)**2, 0:kmax-1)
                              ! $ -(-1)^{N_D/2}K_{HD}\nabla^{N_D} $ . 
                              ! Ǯ, ʿȻ. 
                              ! Coefficient of horizontal thermal and water diffusion
    real(DP), intent(in), optional:: r_SigmaSet (:)
                              ! ǥեȤǤ, kmax ͤ˱,
                              ! ưŪ $ \sigma $ ٥ (Ⱦ) 
                              ! ꤵ (kmax ͤΤ).
                              ! $ \sigma $ ٥ (Ⱦ). 
                              ! Ūꤹɬפ, 
                              ! ΰͿ.
                              !
                              ! By default, half $ \sigma $ level is 
                              ! specified automatically according to
                              ! the value of kmax (only the value is 
                              ! same as particular values).
                              ! If the half $ \sigma $ level is specified
                              ! manually, give this argument.
    real(DP), intent(in), optional:: w_Phis ((nmax+1)**2)
                              ! $ \Phi_s $ . ɽݥƥ󥷥. 
                              ! Surface geo-potential
    character(*), intent(in), optional:: time_integration_scheme
                              ! ʬˡ. 
                              ! ʲˡǽ. 
                              !
                              ! Time integration scheme. 
                              ! Available schemes are as follows.
                              ! 
                              ! * "Semi-implicit"
                              ! * "Explicit"
                              ! 
    real(DP), intent(in), optional:: history_interval_value
                              ! ҥȥǡνϴֳ֤ñ. 
                              ! Unit for interval of history data output
    character(*), intent(in), optional:: history_interval_unit
                              ! ҥȥǡνϴֳ֤ñ. 
                              ! Unit for interval of history data output
    character(*), intent(in), optional:: history_precision
                              ! ҥȥǡ. 
                              ! Precision of history data
    character(*), intent(in), optional :: nmlfile
                              ! NAMELIST ե̾. 
                              ! ΰ˶ʸʳͿ, 
                              ! ꤵ줿ե뤫 
                              ! NAMELIST ѿɤ߹ߤޤ. 
                              ! եɤ߹ʤˤϥ顼
                              ! ޤ.
                              !
                              ! NAMELIST ѿξܺ٤˴ؤƤ 
                              ! NAMELIST#dyn_as83_nml 
                              ! 򻲾ȤƤ. 
                              !
                              ! NAMELIST file name. 
                              ! If nonnull character is specified to
                              ! this argument, 
                              ! NAMELIST group name is loaded from the 
                              ! file. 
                              ! If the file can not be read, 
                              ! an error occurs.
                              ! 
                              ! See "NAMELIST#dyn_as83_nml" 
                              ! for details about a NAMELIST group.
                              ! 
    logical, intent(out), optional:: err
                              ! 㳰ѥե饰.
                              ! ǥեȤǤ, μ³ǥ顼
                              ! , ץ϶λޤ.
                              !  *err* Ϳ,
                              ! ץ϶λ, 
                              ! *err*  .true. ޤ.
                              !
                              ! Exception handling flag. 
                              ! By default, when error occur in 
                              ! this procedure, the program aborts. 
                              ! If this *err* argument is given, 
                              ! .true. is substituted to *err* and 
                              ! the program does not abort. 

    !-----------------------------------
    !  ѿ
    !  Work variables
    real(DP):: Kappa      ! $ \kappa = R/C_p $ .
                          ! 갵ǮФ. Ratio of gas constant to specific heat
    real(DP):: z_Sigma (0:kmax-1)
                              ! $ \sigma $ ٥ (). 
                              ! Full $ \sigma $ level
    real(DP):: r_Sigma (0:kmax)
                              ! $ \sigma $ ٥ (Ⱦ). 
                              ! Half $ \sigma $ level
    real(DP):: z_DelSigma (0:kmax-1)
                              ! $ \Delta \sigma $ (). 
                              ! $ \Delta \sigma $ (Full)
    real(DP):: z_TempAvrXY (0:kmax-1)
                              ! ʿѲ (٥).
                              ! Average temperature on full sigma level
    real(DP):: r_TempAvrXY (0:kmax)
                              ! ʿѲ (Ⱦ٥). 
                              ! Average temperature on half sigma level
    real(DP):: z_HydroAlpha (0:kmax-1)
                              ! $ \alpha $ . ſ尵μη. 
                              ! Coefficient in hydrostatic equation.
    real(DP):: z_HydroBeta (0:kmax-1)
                              ! $ \beta $ . ſ尵μη. 
                              ! Coefficient in hydrostatic equation.
    real(DP):: z_TempInpolKappa (0:kmax-1)
                              ! $ \kappa $ . ٱľ֤η. 
                              ! Coefficient for vertical interpolation of temperature
    real(DP):: z_TempInpolA (0:kmax-1)
                              ! $ a $ . ٱľ֤η. 
                              ! Coefficient for vertical interpolation of temperature
    real(DP):: z_TempInpolB (0:kmax-1)
                              ! $ b $ . ٱľ֤η. 
                              ! Coefficient for vertical interpolation of temperature
    real(DP):: z_siMtxG (0:kmax-1)
                              ! $ G = C_p \kappa T $
    real(DP):: zz_siMtxH (0:kmax-1, 0:kmax-1)
                              ! $ h = QS - R $
    real(DP):: zz_siMtxWH (0:kmax-1, 0:kmax-1)
                              ! $ W h $
    real(DP):: zz_siMtxGCt (0:kmax-1, 0:kmax-1)
                              ! $ G C^{T} $ ( $ C = \Delta \sigma $ )
    real(DP):: zz_siMtxW (0:kmax-1, 0:kmax-1)
                              ! $ W $ . 
                              ! ȯμǤȹθ̤ˤ벹η. 
                              ! Coefficient for correction of temperature 
                              ! by effort of linear gravitational terms

    real(DP):: zz_siMtxQ (0:kmax-1, 0:kmax-1)
                              ! $ Q = \DD{T}{\sigma} $
    real(DP):: zz_siMtxS (0:kmax-1, 0:kmax-1)
                              ! $ S = \DD{\sigma}{t} $
    real(DP):: zz_siMtxQS (0:kmax-1, 0:kmax-1)
                              ! $ QS $ . 
                              ! ѿ $ \sigma $ ήθ̤. 
                              ! This variable  corresponds to effort of $ \sigma $ advection
    real(DP):: zz_siMtxR (0:kmax-1, 0:kmax-1)
                              ! $ R $ . 
                              ! ȯȳݤ碌뤳ȤǵѲθ̤Ȥʤ.
                              ! Product R and divergence become effort of 
                              ! surface pressure tendency.
                              ! $ RD = \kappa T 
                              ! (\DD{\pi}{t} + \Dinv{\sigma}\DD{\sigma}{t}) $ .

    integer:: mmax            ! ȿ. 
                              ! Maximum truncated zonal wavenumber
    integer:: lmax            ! ȿ. 
                              ! Maximum truncated meridional wavenumber
    integer:: k, l            ! DO 롼Ѻѿ
                              ! Work variables for DO loop
    type(SIGDAT):: sig_dat
    integer:: stat
    character(STRING):: cause_c
    character(*), parameter:: subname = 'DynAS83Create'
  continue
    call BeginSub(subname, version)
    stat = DC_NOERR
    cause_c = ''

    !-----------------------------------------------------------------
    !  Υå
    !  Check initialization
    !-----------------------------------------------------------------
    if (dyn_as % initialized) then
      stat = DCPAM_EALREADYINIT
      cause_c = 'DYNAS83'
      goto 999
    end if

    !-----------------------------------------------------------------
    !  Υå
    !  Validation of arguments
    !-----------------------------------------------------------------
    if (nmax < 1) then
      stat = DCPAM_ENEGATIVE
      cause_c = 'nmax'
      goto 999
    end if
    if (imax < 1) then
      stat = DCPAM_ENEGATIVE
      cause_c = 'imax'
      goto 999
    end if
    if (jmax < 1) then
      stat = DCPAM_ENEGATIVE
      cause_c = 'jmax'
      goto 999
    end if
    if (kmax < 1) then
      stat = DCPAM_ENEGATIVE
      cause_c = 'kmax'
      goto 999
    end if

    !-----------------------------------------------------------------
    !  ȿʻ
    !  Configure wave number and grid point
    !-----------------------------------------------------------------
    dyn_as % nmax = nmax
    dyn_as % imax = imax
    dyn_as % jmax = jmax
    dyn_as % kmax = kmax

    !-----------------------------------------------------------------
    !  ʬˡ (ˤ)
    !  Configure time integration scheme (specified by an argument)
    !-----------------------------------------------------------------
    if ( present(time_integration_scheme ) ) then
      dyn_as % time_integration_scheme  = time_integration_scheme 
    else
      dyn_as % time_integration_scheme  = ''
    end if

    !-----------------------------------------------------------------
    !  ʪ
    !  Configure physical constants
    !-----------------------------------------------------------------
    dyn_as % PI = PI
    dyn_as % RPlanet = RPlanet
!!$    dyn_as % Omega = Omega
!!$    dyn_as % Grav = Grav
    dyn_as % Cp = Cp
    dyn_as % RAir = RAir
    dyn_as % EpsVT = EpsVT
    dyn_as % DelTime = DelTime

    Kappa = RAir / Cp
    dyn_as % Kappa = Kappa

    !-----------------------------------------------------------------
    !  ꥪѥ᡼
    !  Configure Coriolis parameter
    !-----------------------------------------------------------------
    allocate( dyn_as % xy_Cori (0:imax-1, 0:jmax-1) )
    dyn_as % xy_Cori = xy_Cori

    !-----------------------------------------------------------------
    !  ڥȥź֤ȥץ饷η
    !  Configure spectral subscript expression and Laplacian coefficient
    !-----------------------------------------------------------------
    mmax = dyn_as % nmax
    lmax = dyn_as % nmax
    allocate( dyn_as % nmo (1:2, 0:mmax, 0:lmax) )
    dyn_as % nmo = nmo

    allocate( dyn_as % wz_rn ((dyn_as%nmax+1)**2, 0:dyn_as%kmax-1) )
    dyn_as % wz_rn = wz_rn

    !-----------------------------------------------------------------
    !  ư̿ʿȻǮ, ʿȻ
    !  Configure coefficient of horizontal momentum diffusion, and
    !  coefficient of horizontal thermal and water diffusion
    !-----------------------------------------------------------------
    allocate( dyn_as % wz_DiffVorDiv ((dyn_as%nmax+1)**2, 0:dyn_as%kmax-1) )
    dyn_as % wz_DiffVorDiv = wz_DiffVorDiv

    allocate( dyn_as % wz_DiffTherm ((dyn_as%nmax+1)**2, 0:dyn_as%kmax-1) )
    dyn_as % wz_DiffTherm = wz_DiffTherm

    !-------------------------------------------------------------------
    !  Ϸǡ (ɽ $ \Phi $ ) Υڥȥͤ
    !  Configure spectral geography data (surface $ \Phi $ )
    !-------------------------------------------------------------------
    allocate( dyn_as % w_Phis ((dyn_as%nmax+1)**2) )
    if (present(w_Phis)) then
      dyn_as % w_Phis = w_Phis
    else
      dyn_as % w_Phis = 0.0_DP
    end if

    !-----------------------------------------------------------------
    !  ľʻ
    !  Configure values of vertical grid point
    !-----------------------------------------------------------------
    call Create( sig_dat = sig_dat, &             ! (inout)
      & kmax = dyn_as % kmax, &                   ! (in)
      & Cp = dyn_as % Cp, RAir = dyn_as % RAir, & ! (in)
      & r_SigmaSet = r_SigmaSet, &                ! (in)
      & nmlfile = nmlfile, &                      ! (in)
      & err = err )                               ! (out)

    if ( present_and_true(err) ) then
      stat = DCPAM_EFAILINIT
      cause_c = 'sig_dat'
      goto 999
    end if

    call Get( sig_dat = sig_dat, & ! (inout)
      & z_Sigma = z_Sigma, &       ! (out)
      & r_Sigma = r_Sigma, &       ! (out)
      & z_DelSigma = z_DelSigma )  ! (out)

    call Close( sig_dat = sig_dat ) ! (inout)

    allocate( dyn_as % z_Sigma(0:kmax-1) )
    allocate( dyn_as % r_Sigma(0:kmax) )
    allocate( dyn_as % z_DelSigma(0:kmax-1) )

    dyn_as % z_Sigma    = z_Sigma   
    dyn_as % r_Sigma    = r_Sigma   
    dyn_as % z_DelSigma = z_DelSigma

    !-----------------------------------------------------------------
    !  ҥȥեؤΥǡ
    !  Configure the settings for history data output
    !-----------------------------------------------------------------
    dyn_as % history_interval_value = 1.0
    dyn_as % history_interval_unit  = 'day'
    dyn_as % history_precision      = 'float'
    if ( present(history_interval_value) ) dyn_as % history_interval_value = history_interval_value 
    if ( present(history_interval_unit ) ) dyn_as % history_interval_unit  = history_interval_unit  
    if ( present(history_precision     ) ) dyn_as % history_precision      = history_precision      

    !-----------------------------------------------------------------
    !  NAMELIST 
    !  Values from NAMELIST
    !-----------------------------------------------------------------
    if ( present_and_not_empty(nmlfile) ) then
      call MessageNotify( 'M', subname, &
        & 'Loading NAMELIST file "%c" ...', &
        & c1=trim(nmlfile) )
      call NmlRead ( nmlfile = nmlfile, &        ! (in)
        & time_integration_scheme_ = &
        &   dyn_as % time_integration_scheme , & ! (inout)
!
        & history_interval_value = &
        &   dyn_as % history_interval_value, &   ! (inout)
!
        & history_interval_unit_ = &
        &   dyn_as % history_interval_unit, &    ! (inout)
!
        & history_precision_ = &
        &   dyn_as % history_precision, &        ! (inout)
!
        & err = err )                            ! (out)
      if ( present_and_true(err) ) then
        call MessageNotify( 'W', subname, &
          & '"%c" can not be read.', &
          & c1=trim(nmlfile) )
        stat = DC_ENOFILEREAD
        cause_c = nmlfile
        goto 999
      end if
    end if


    !-----------------------------------------------------------------
    !  ʬˡ (NAMELISTˤ)
    !  Configure time integration scheme (specified by an NAMELIST)
    !-----------------------------------------------------------------
    if ( trim(dyn_as % time_integration_scheme ) == '' ) then
      dyn_as % time_integration_scheme  = 'Semi-implicit'
    else
      select case ( LChar( trim(dyn_as % time_integration_scheme ) ) )
      case ('semi-implicit')
      case ('explicit')
      case default
        stat = DCPAM_EBADSCHEME
        cause_c = trim(dyn_as % time_integration_scheme )
        goto 999
      end select
    end if

    !-----------------------------------------------------------------
    !  ſ尵μη $ \alpha $ , $ \beta $ η׻
    !  Calculate coefficients $ \alpha $ and $ \beta $ in 
    !  hydrostatic equation.
    !-----------------------------------------------------------------
    do k = 0, kmax - 1
      z_HydroAlpha(k) = &
        & ( r_Sigma(k) / z_Sigma(k) ) ** Kappa - 1.0_DP

      z_HydroBeta(k) = &
        & 1.0_DP - ( r_Sigma(k+1) / z_Sigma(k) ) ** Kappa
    enddo

    allocate( dyn_as % z_HydroAlpha(0:kmax - 1) )
    allocate( dyn_as % z_HydroBeta(0:kmax - 1) )
    dyn_as % z_HydroAlpha = z_HydroAlpha
    dyn_as % z_HydroBeta = z_HydroBeta


    !-----------------------------------------------------------------
    !  ٱľ֤η $ \kappa $, $ a $ , $ b $ η׻
    !  Calculate coefficients $ \kappa $, $ a $ , $ b $ 
    !  for interpolation of temperature
    !-----------------------------------------------------------------
    do k = 0, kmax - 1
      z_TempInpolKappa(k) = &
        & ( r_Sigma(k) * z_HydroAlpha(k) &
        &   + r_Sigma(k+1) * z_HydroBeta(k) ) / z_DelSigma(k)
    enddo

    z_TempInpolA = 0.0_DP
    do k = 1, kmax - 1
      z_TempInpolA(k) = &
        & z_HydroAlpha(k) &
        &  / ( 1.0_DP - (z_Sigma(k) / z_Sigma(k-1)) ** Kappa )
    end do

    z_TempInpolB = 0.0_DP
    do k = 0, kmax - 2
      z_TempInpolB(k) = &
        & z_HydroBeta(k) / ( (z_Sigma(k) / z_Sigma(k+1) ) ** Kappa - 1.0_DP )
    end do

    allocate( dyn_as % z_TempInpolKappa(0:kmax - 1) )
    allocate( dyn_as % z_TempInpolA(0:kmax - 1) )
    allocate( dyn_as % z_TempInpolB(0:kmax - 1) )
    dyn_as % z_TempInpolKappa = z_TempInpolKappa
    dyn_as % z_TempInpolA = z_TempInpolA
    dyn_as % z_TempInpolB = z_TempInpolB


    !-----------------------------------------------------------------
    !  ʿѲ (٥롢Ⱦ٥) η׻
    !  Calculate average temperature about level and half level.
    !-----------------------------------------------------------------
    z_TempAvrXY = 300.0_DP
    r_TempAvrXY = 0.0_DP

    do k = 1, kmax - 1
      r_TempAvrXY(k) = &
        &   z_TempInpolA(k)   * z_TempAvrXY(k)   &
        & + z_TempInpolB(k-1) * z_TempAvrXY(k-1)
    enddo

    allocate( dyn_as % z_TempAvrXY(0:kmax - 1) )
    allocate( dyn_as % r_TempAvrXY(0:kmax) )
    dyn_as % z_TempAvrXY = z_TempAvrXY
    dyn_as % r_TempAvrXY = r_TempAvrXY


    !-----------------------------------------------------------------
    !  ߥץꥷåˡΤιη׻
    !  Calculate matrices for semi-implicit method
    !-----------------------------------------------------------------
    z_siMtxG = Cp * z_TempInpolKappa * z_TempAvrXY

    do k = 0, kmax - 1
      do l = 0, kmax - 1
        zz_siMtxGCt(k, l) = z_siMtxG(k) * z_DelSigma(l)
      end do
    end do

    zz_siMtxW = 0.0_DP
    do k = 0, kmax - 1
      do l = 0, k
        zz_siMtxW(k, l) = Cp * z_HydroAlpha(l)
      enddo

      do l = 0, k - 1
        zz_siMtxW(k, l) = zz_siMtxW(k, l) + Cp * z_HydroBeta(l)
      enddo
    enddo

    zz_siMtxS = 0.0_DP
    do k = 0, kmax - 1
      do l = 0, kmax - 1
        zz_siMtxS(k,l) = r_Sigma(k) * z_DelSigma(l)
      enddo
      do l = k, kmax - 1
        zz_siMtxS(k,l) = zz_siMtxS(k,l) - z_DelSigma(l)
      enddo
    enddo

    zz_siMtxQ = 0.0_DP
    do k = 0, kmax - 1
      zz_siMtxQ(k,k) = ( r_TempAvrXY(k) - z_TempAvrXY(k) ) / z_DelSigma(k)
    enddo
    do k = 0, kmax - 2
      zz_siMtxQ(k,k+1) = ( z_TempAvrXY(k) - r_TempAvrXY(k+1) ) / z_DelSigma(k)
    enddo

    zz_siMtxQS = 0.0_DP
    zz_siMtxQS = matmul(zz_siMtxQ, zz_siMtxS)

    zz_siMtxR = 0.0_DP
    do k = 0, kmax
      do l = k, kmax - 1
        zz_siMtxR(k,l) = &
          & - z_HydroAlpha(k) / z_DelSigma(k) * z_DelSigma(l) * z_TempAvrXY(k)
      enddo
      do l = k + 1, kmax - 1
        zz_siMtxR(k,l) = zz_siMtxR(k,l)  &
          & - z_HydroBeta(k) / z_DelSigma(k) * z_DelSigma(l) * z_TempAvrXY(k)
      enddo
    enddo

    zz_siMtxH = 0.0_DP
    zz_siMtxH = zz_siMtxQS - zz_siMtxR

    zz_siMtxWH = 0.0_DP
    zz_siMtxWH = matmul(zz_siMtxW, zz_siMtxH)

    allocate( dyn_as % z_siMtxG(0: kmax - 1) )
    allocate( dyn_as % zz_siMtxH(0: kmax - 1, 0: kmax - 1) )
    allocate( dyn_as % zz_siMtxWH(0: kmax - 1, 0: kmax - 1) )
    allocate( dyn_as % zz_siMtxGCt(0: kmax - 1, 0: kmax - 1) )
    dyn_as % z_siMtxG = z_siMtxG
    dyn_as % zz_siMtxH = zz_siMtxH
    dyn_as % zz_siMtxWH = zz_siMtxWH
    dyn_as % zz_siMtxGCt = zz_siMtxGCt

    !-----------------------------------------------------------------
    !  ߥץꥷåȻʬѹη׻
    !  Calculate matrixes for semi-implicit time integration
    !-----------------------------------------------------------------
    select case ( LChar( trim(dyn_as % time_integration_scheme ) ) )
    case ('semi-implicit')
      call Create( dyn_mtx = dyn_as % dyn_mtx, &                    ! (inout)
        & nmax = dyn_as % nmax, kmax = dyn_as % kmax, &             ! (in)
        & nmo = dyn_as % nmo, &                                     ! (in)
        & RPlanet = dyn_as % RPlanet, DelTime = dyn_as % DelTime, & ! (in)
        & zz_siMtxWH = dyn_as % zz_siMtxWH, &                       ! (in)
        & zz_siMtxGCt = dyn_as % zz_siMtxGCt, &                     ! (in)
        & w_DiffVorDiv = dyn_as % wz_DiffVorDiv(:,0), &             ! (in)
        & w_DiffTherm = dyn_as % wz_DiffTherm(:,0) )                ! (in)
    end select

    !-----------------------------------------------------------------
    !  λ, 㳰
    !  Termination and Exception handling
    !-----------------------------------------------------------------
    dyn_as % initialized = .true.
999 continue
    call StoreError(stat, subname, err, cause_c)
    call EndSub(subname)
  end subroutine DynAS83Create


  subroutine DynAS83Close( dyn_as, err )
    !
    ! DYNAS83 ѿνλԤޤ.
    ! ʤ, Ϳ줿 *dyn_as*  Create ˤäƽ
    ! Ƥʤ, ץϥ顼ȯޤ.
    !
    ! Deconstructor of "DYNAS83".
    ! Note that if *dyn_as* is not initialized by "Create" yet,
    ! error is occurred.
    !
    use dc_trace, only: BeginSub, EndSub
    use dc_types, only: STRING, STDOUT
    use dc_string, only: LChar, PutLine
    use dcpam_error, only: StoreError, DC_NOERR, DCPAM_ENOTINIT
    use dyn_matrix, only: Close
    implicit none
    type(DYNAS83), intent(inout):: dyn_as
    logical, intent(out), optional:: err
                              ! 㳰ѥե饰.
                              ! ǥեȤǤ, μ³ǥ顼
                              ! , ץ϶λޤ.
                              !  *err* Ϳ,
                              ! ץ϶λ, 
                              ! *err*  .true. ޤ.
                              !
                              ! Exception handling flag. 
                              ! By default, when error occur in 
                              ! this procedure, the program aborts. 
                              ! If this *err* argument is given, 
                              ! .true. is substituted to *err* and 
                              ! the program does not abort. 
    integer:: stat
    character(STRING):: cause_c
    character(*), parameter:: subname = 'DynAS83Close'
  continue
    call BeginSub(subname)
    stat = DC_NOERR
    cause_c = ''

    !-----------------------------------------------------------------
    !  Υå
    !  Check initialization
    !-----------------------------------------------------------------
    if (.not. dyn_as % initialized) then
      stat = DCPAM_ENOTINIT
      cause_c = 'DYNAS83'
      goto 999
    end if

    !-----------------------------------------------------------------
    !  "DYNAS83" ξõ
    !  Clear the settings for "DYNAS83"
    !-----------------------------------------------------------------
    deallocate( dyn_as % z_Sigma )
    deallocate( dyn_as % r_Sigma )
    deallocate( dyn_as % z_DelSigma )
    deallocate( dyn_as % nmo )
    deallocate( dyn_as % wz_rn )
    deallocate( dyn_as % wz_DiffVorDiv )
    deallocate( dyn_as % wz_DiffTherm )
    deallocate( dyn_as % z_TempAvrXY )
    deallocate( dyn_as % r_TempAvrXY )
    deallocate( dyn_as % z_HydroAlpha )
    deallocate( dyn_as % z_HydroBeta )
    deallocate( dyn_as % z_TempInpolKappa )
    deallocate( dyn_as % z_TempInpolA )
    deallocate( dyn_as % z_TempInpolB )
    deallocate( dyn_as % z_siMtxG )
    deallocate( dyn_as % zz_siMtxH )
    deallocate( dyn_as % zz_siMtxWH )
    deallocate( dyn_as % zz_siMtxGCt )

    !-----------------------------------------------------------------
    !  ߥץꥷåȻʬѹξõ
    !  Clear matrixes for semi-implicit time integration
    !-----------------------------------------------------------------
    select case ( LChar( trim(dyn_as % time_integration_scheme ) ) )
    case ('semi-implicit')
      call Close( dyn_mtx = dyn_as % dyn_mtx ) ! (inout)
    end select

    !-----------------------------------------------------------------
    !  λ, 㳰
    !  Termination and Exception handling
    !-----------------------------------------------------------------
    dyn_as % initialized = .false.
999 continue
    call StoreError(stat, subname, err, cause_c)
    call EndSub(subname)
  end subroutine DynAS83Close


  subroutine DynAS83PutLine( dyn_as, unit, indent, err )
    !
    !  *dyn_as* ξޤ.
    ! ǥեȤǤɸϤ˰ޤ. *unit* ֹ
    ! Ƥ뤳Ȥ, ѹ뤳ȤǽǤ.
    !
    ! Print information of *dyn_as*.
    ! By default messages are output to standard output.
    ! Unit number for output can be changed by *unit* argument.
    !
    use dc_types, only: STRING, STDOUT
    use dc_date, only: toChar
    use dc_trace, only: BeginSub, EndSub
    use dc_string, only: Printf, PutLine
    use dcpam_error, only: StoreError, DC_NOERR, DCPAM_ENOTINIT
    use dyn_matrix, only: PutLine
    implicit none
    type(DYNAS83), intent(in):: dyn_as
    integer, intent(in), optional:: unit
                              ! ֹ.
                              ! ǥեȤνɸ.
                              !
                              ! Unit number for output.
                              ! Default value is standard output.
    character(*), intent(in), optional:: indent
                              ! ɽåλ.
                              !
                              ! Indent of displayed messages.
    logical, intent(out), optional:: err
                              ! 㳰ѥե饰.
                              ! ǥեȤǤ, μ³ǥ顼
                              ! , ץ϶λޤ.
                              !  *err* Ϳ,
                              ! ץ϶λ, 
                              ! *err*  .true. ޤ.
                              !
                              ! Exception handling flag. 
                              ! By default, when error occur in 
                              ! this procedure, the program aborts. 
                              ! If this *err* argument is given, 
                              ! .true. is substituted to *err* and 
                              ! the program does not abort. 

    integer:: stat
    integer:: out_unit
    character(STRING):: cause_c
    integer:: indent_len
    character(STRING):: indent_str
    character(len = *), parameter:: subname = 'DynAS83PutLine'
  continue
    call BeginSub(subname)
    stat = DC_NOERR
    cause_c = ''

    !-----------------------------------------------------------------
    !  Υå
    !  Check initialization
    !-----------------------------------------------------------------
    if (present(unit)) then
      out_unit = unit
    else
      out_unit = STDOUT
    end if

    indent_len = 0
    indent_str = ''
    if (present(indent)) then
      if (len(indent) /= 0) then
        indent_len = len(indent)
        indent_str(1:indent_len) = indent
      end if
    end if

    !-----------------------------------------------------------------
    !  "DYNAS83" ΰ
    !  Print the settings for "DYNAS83"
    !-----------------------------------------------------------------
    if ( dyn_as % initialized ) then
      call Printf( out_unit, &
        & indent_str(1:indent_len) // &
        & '#<DYNAS83:: @initialized=%y @nmax=%d @imax=%d @jmax=%d @kmax=%d', &
        & i=(/dyn_as % nmax, dyn_as % imax, dyn_as % jmax, dyn_as % kmax/), &
        & l=(/dyn_as % initialized/) )

      call Printf( out_unit, &
        & indent_str(1:indent_len) // &
        & ' @time_integration_scheme=%c', &
        & c1=trim(dyn_as % time_integration_scheme) )

      call Printf( out_unit, &
        & indent_str(1:indent_len) // &
        & ' @history_interval_value=%f @history_interval_unit=%c @history_precision=%c', &
        & d=(/dyn_as % history_interval_value/), &
        & c1=trim(dyn_as % history_interval_unit), &
        & c2=trim(dyn_as % history_precision) )

      call Printf( out_unit, &
        & indent_str(1:indent_len) // &
        & ' @PI=%f @RPlanet=%f', &
        & d=(/dyn_as % PI, dyn_as % RPlanet/))

!!$      call Printf( out_unit, &
!!$        & indent_str(1:indent_len) // &
!!$        & ' @Omega=%f @Grav=%f', &
!!$        & d=(/dyn_as % Omega, dyn_as % Grav/))

      call Printf( out_unit, &
        & indent_str(1:indent_len) // &
        & ' @Cp=%f @RAir=%f @EpsVT=%f @Kappa=%f', &
        & d=(/dyn_as % Cp, dyn_as % RAir, dyn_as % EpsVT, &
        &     dyn_as % Kappa/))

      call PutLine( dyn_as % z_Sigma, unit = out_unit, &
        & lbounds = lbound(dyn_as % z_Sigma), &
        & ubounds = ubound(dyn_as % z_Sigma), &
        & indent = indent_str(1:indent_len) // &
        & ' @z_Sigma=' )

      call PutLine( dyn_as % r_Sigma, unit = out_unit, &
        & lbounds = lbound(dyn_as % r_Sigma), &
        & ubounds = ubound(dyn_as % r_Sigma), &
        & indent = indent_str(1:indent_len) // &
        & ' @r_Sigma=' )

      call PutLine( dyn_as % z_DelSigma, unit = out_unit, &
        & lbounds = lbound(dyn_as % z_DelSigma), &
        & ubounds = ubound(dyn_as % z_DelSigma), &
        & indent = indent_str(1:indent_len) // &
        & ' @z_DelSigma=' )

      call PutLine( dyn_as % xy_Cori, unit = out_unit, &
        & lbounds = lbound(dyn_as % xy_Cori), &
        & ubounds = ubound(dyn_as % xy_Cori), &
        & indent = indent_str(1:indent_len) // &
        & ' @xy_Cori=' )

      call PutLine( dyn_as % nmo, unit = out_unit, &
        & lbounds = lbound(dyn_as % nmo), &
        & ubounds = ubound(dyn_as % nmo), &
        & indent = indent_str(1:indent_len) // &
        & ' @nmo=' )

      call PutLine( dyn_as % wz_DiffVorDiv, unit = out_unit, &
        & lbounds = lbound(dyn_as % wz_DiffVorDiv), &
        & ubounds = ubound(dyn_as % wz_DiffVorDiv), &
        & indent = indent_str(1:indent_len) // &
        & ' @wz_DiffVorDiv=' )

      call PutLine( dyn_as % wz_DiffTherm, unit = out_unit, &
        & lbounds = lbound(dyn_as % wz_DiffTherm), &
        & ubounds = ubound(dyn_as % wz_DiffTherm), &
        & indent = indent_str(1:indent_len) // &
        & ' @wz_DiffTherm=' )

      call PutLine( dyn_as % w_Phis, unit = out_unit, &
        & lbounds = lbound(dyn_as % w_Phis), &
        & ubounds = ubound(dyn_as % w_Phis), &
        & indent = indent_str(1:indent_len) // &
        & ' @w_Phis=' )

      call PutLine( dyn_as % z_TempAvrXY, unit = out_unit, &
        & lbounds = lbound(dyn_as % z_TempAvrXY), &
        & ubounds = ubound(dyn_as % z_TempAvrXY), &
        & indent = indent_str(1:indent_len) // &
        & ' @z_TempAvrXY=' )

      call PutLine( dyn_as % r_TempAvrXY, unit = out_unit, &
        & lbounds = lbound(dyn_as % r_TempAvrXY), &
        & ubounds = ubound(dyn_as % r_TempAvrXY), &
        & indent = indent_str(1:indent_len) // &
        & ' @r_TempAvrXY=' )

      call PutLine( dyn_as % z_HydroAlpha, unit = out_unit, &
        & lbounds = lbound(dyn_as % z_HydroAlpha), &
        & ubounds = ubound(dyn_as % z_HydroAlpha), &
        & indent = indent_str(1:indent_len) // &
        & ' @z_HydroAlpha=' )

      call PutLine( dyn_as % z_HydroBeta, unit = out_unit, &
        & lbounds = lbound(dyn_as % z_HydroBeta), &
        & ubounds = ubound(dyn_as % z_HydroBeta), &
        & indent = indent_str(1:indent_len) // &
        & ' @z_HydroBeta=' )

      call PutLine( dyn_as % z_TempInpolKappa, unit = out_unit, &
        & lbounds = lbound(dyn_as % z_TempInpolKappa), &
        & ubounds = ubound(dyn_as % z_TempInpolKappa), &
        & indent = indent_str(1:indent_len) // &
        & ' @z_TempInpolKappa=' )

      call PutLine( dyn_as % z_TempInpolA, unit = out_unit, &
        & lbounds = lbound(dyn_as % z_TempInpolA), &
        & ubounds = ubound(dyn_as % z_TempInpolA), &
        & indent = indent_str(1:indent_len) // &
        & ' @z_TempInpolA=' )

      call PutLine( dyn_as % z_TempInpolB, unit = out_unit, &
        & lbounds = lbound(dyn_as % z_TempInpolB), &
        & ubounds = ubound(dyn_as % z_TempInpolB), &
        & indent = indent_str(1:indent_len) // &
        & ' @z_TempInpolB=' )

      call PutLine( dyn_as % z_siMtxG, unit = out_unit, &
        & lbounds = lbound(dyn_as % z_siMtxG), &
        & ubounds = ubound(dyn_as % z_siMtxG), &
        & indent = indent_str(1:indent_len) // &
        & ' @z_siMtxG=' )

      call PutLine( dyn_as % zz_siMtxH, unit = out_unit, &
        & lbounds = lbound(dyn_as % zz_siMtxH), &
        & ubounds = ubound(dyn_as % zz_siMtxH), &
        & indent = indent_str(1:indent_len) // &
        & ' @zz_siMtxH=' )

      call PutLine( dyn_as % zz_siMtxWH, unit = out_unit, &
        & lbounds = lbound(dyn_as % zz_siMtxWH), &
        & ubounds = ubound(dyn_as % zz_siMtxWH), &
        & indent = indent_str(1:indent_len) // &
        & ' @zz_siMtxWH=' )

      call PutLine( dyn_as % zz_siMtxGCt, unit = out_unit, &
        & lbounds = lbound(dyn_as % zz_siMtxGCt), &
        & ubounds = ubound(dyn_as % zz_siMtxGCt), &
        & indent = indent_str(1:indent_len) // &
        & ' @zz_siMtxGCt=' )

      call PutLine(dyn_as % dyn_mtx, out_unit, &
        & indent_str(1:indent_len) // &
        & ' ', err )

      call Printf( out_unit, indent_str(1:indent_len) // '>' )

    else
      call Printf( out_unit, &
        & indent_str(1:indent_len) // &
        & '#<DYNAS83:: @initialized=%y>', l=(/dyn_as % initialized/) )
    end if

    !-----------------------------------------------------------------
    !  λ, 㳰
    !  Termination and Exception handling
    !-----------------------------------------------------------------
999 continue
    call StoreError(stat, subname, err, cause_c)
    call EndSub(subname)
  end subroutine DynAS83PutLine

  logical function DynAS83Initialized( dyn_as ) result(result)
    !
    ! *dyn_as* ꤵƤˤ .true. ,
    ! ꤵƤʤˤ .false. ֤ޤ.
    !
    ! If *dyn_as* is initialized, .true. is returned.
    ! If *dyn_as* is not initialized, .false. is returned.
    !
    implicit none
    type(DYNAS83), intent(in):: dyn_as
  continue
    result = dyn_as % initialized
  end function DynAS83Initialized

  subroutine DynAS83Inquire( dyn_as, &
    & time_integration_scheme, &
    & err )
    !
    ! *dyn_as* ˳ǼƤ䤤碌ޤ.
    !
    ! ʤ, Ϳ줿 *dyn_as*  Create ˤäƽ
    ! Ƥʤ, ץϥ顼ȯޤ.
    !
    ! Inquire information stored in *dyn_as*
    !
    ! If *dyn_as* is not initialized by "Create" yet,
    ! error is occurred.
    !
    use dc_trace, only: BeginSub, EndSub
    use dc_types, only: STRING, STDOUT
    use dc_string, only: PutLine
    use dcpam_error, only: StoreError, DC_NOERR, DCPAM_ENOTINIT
    implicit none
    type(DYNAS83), intent(in):: dyn_as
    character(*), intent(out), optional:: time_integration_scheme
                              ! ʬˡ. 
                              ! ʲˡǽ. 
                              !
                              ! Time integration scheme. 
                              ! Available schemes are as follows.
                              ! 
                              ! * "Semi-implicit"
                              ! * "Explicit"
                              ! 
    logical, intent(out), optional:: err
                              ! 㳰ѥե饰.
                              ! ǥեȤǤ, μ³ǥ顼
                              ! , ץ϶λޤ.
                              !  *err* Ϳ,
                              ! ץ϶λ, 
                              ! *err*  .true. ޤ.
                              !
                              ! Exception handling flag. 
                              ! By default, when error occur in 
                              ! this procedure, the program aborts. 
                              ! If this *err* argument is given, 
                              ! .true. is substituted to *err* and 
                              ! the program does not abort. 
    integer:: stat
    character(STRING):: cause_c
    character(*), parameter:: subname = 'DynAS83Inquire'
  continue
    call BeginSub(subname)
    stat = DC_NOERR
    cause_c = ''

    !-----------------------------------------------------------------
    !  Υå
    !  Check initialization
    !-----------------------------------------------------------------
    if (.not. dyn_as % initialized) then
      stat = DCPAM_ENOTINIT
      cause_c = 'DYNAS83'
      goto 999
    end if

    !-----------------------------------------------------------------
    !  *dyn_as* ˳ǼμФ
    !  Fetch information stored in *dyn_as*
    !-----------------------------------------------------------------
    if ( present(time_integration_scheme) ) then
      time_integration_scheme = dyn_as % time_integration_scheme
    end if

    !-----------------------------------------------------------------
    !  λ, 㳰
    !  Termination and Exception handling
    !-----------------------------------------------------------------
999 continue
    call StoreError(stat, subname, err, cause_c)
    call EndSub(subname)
  end subroutine DynAS83Inquire

  subroutine DynAS83NmlRead( nmlfile, &
    & time_integration_scheme_, &
    & history_interval_value, &
    & history_interval_unit_, history_precision_, &
    & err )
    !
    ! NAMELIST ե *nmlfile* ͤϤ뤿
    ! ֥롼Ǥ. Create ǸƤӽФ뤳Ȥ
    ! ꤷƤޤ.
    !
    ! ͤ NAMELIST եǻꤵƤʤˤ,
    ! Ϥ줿ͤΤޤ֤ޤ.
    !
    ! ʤ, *nmlfile* ˶ʸͿ줿, ޤ
    ! Ϳ줿 *nmlfile* ɤ߹ळȤǤʤ, 
    ! ץϥ顼ȯޤ.
    !
    ! This is an internal subroutine to input values from 
    ! NAMELIST file *nmlfile*. This subroutine is expected to be
    ! called by "Create".
    !
    ! A value not specified in NAMELIST file is returned
    ! without change.
    !
    ! If *nmlfile* is empty, or *nmlfile* can not be read, 
    ! error is occurred.
    !
    use dc_trace, only: BeginSub, EndSub
    use dc_types, only: DP, STRING, TOKEN, STDOUT
    use dc_string, only: PutLine
    use dc_iounit, only: FileOpen
    use dc_message, only: MessageNotify
    use dc_present, only: present_and_true
    use dc_error, only: DC_NOERR, DC_ENOFILEREAD
    use dcpam_error, only: StoreError, DCPAM_ENMLARRAYINSUFF
    implicit none
    character(*), intent(in):: nmlfile
                              ! NAMELIST ե̾. 
                              ! NAMELIST file name
    character(*), intent(inout):: time_integration_scheme_
    character(TOKEN):: time_integration_scheme
                              ! ʬˡ. 
                              ! ʲˡǽ. 
                              !
                              ! Time integration scheme. 
                              ! Available schemes are as follows.
                              ! 
                              ! * "Semi-implicit"
                              ! * "Explicit"
                              ! 
    real(DP):: history_interval_value
                              ! ҥȥǡνϴֳ֤ñ. 
                              ! Unit for interval of history data output
    character(*), intent(inout):: history_interval_unit_
    character(TOKEN):: history_interval_unit
                              ! ҥȥǡνϴֳ֤ñ. 
                              ! Unit for interval of history data output
    character(*), intent(inout):: history_precision_
    character(TOKEN):: history_precision
                              ! ҥȥǡ. 
                              ! Precision of history data

    logical, intent(out), optional:: err
                              ! 㳰ѥե饰.
                              ! ǥեȤǤ, μ³ǥ顼
                              ! , ץ϶λޤ.
                              !  *err* Ϳ,
                              ! ץ϶λ, 
                              ! *err*  .true. ޤ.
                              !
                              ! Exception handling flag. 
                              ! By default, when error occur in 
                              ! this procedure, the program aborts. 
                              ! If this *err* argument is given, 
                              ! .true. is substituted to *err* and 
                              ! the program does not abort. 

    namelist /dyn_as83_nml/ &
      & time_integration_scheme, &
      & history_interval_value, history_interval_unit, &
      & history_precision
                              ! dyn_as83 ⥸塼
                              ! NAMELIST ѿ̾.
                              !
                              ! dyn_as83#Create Ѥݤ, 
                              ! ץʥ *nmlfile*  NAMELIST 
                              ! ե̾ꤹ뤳Ȥ, Υե뤫
                              !  NAMELIST ѿɤ߹ߤޤ.
                              !
                              ! NAMELIST group name for
                              ! "dyn_as83" module.
                              ! 
                              ! If a NAMELIST filename is specified to 
                              ! an optional argument *nmlfile* 
                              ! when "dyn_as83#Create" is used, 
                              ! this NAMELIST group is loaded from 
                              ! the file.

    !-----------------------------------
    !  ѿ
    !  Work variables
    integer:: stat
    character(STRING):: cause_c
    integer:: unit_nml        ! NAMELIST ե륪ץֹ. 
                              ! Unit number for NAMELIST file open
    integer:: iostat_nml      ! NAMELIST ɤ߹߻ IOSTAT. 
                              ! IOSTAT of NAMELIST read
    character(*), parameter:: subname = 'DynAS83NmlRead'
  continue
    call BeginSub( subname )
    stat = DC_NOERR
    cause_c = ''

    !-----------------------------------------------------------------
    !  ʸ NAMELIST ѿ
    !  Substitute character arguments to NAMELIST group
    !-----------------------------------------------------------------
    time_integration_scheme = time_integration_scheme_
    history_interval_unit = history_interval_unit_
    history_precision = history_precision_

    !----------------------------------------------------------------
    !  NAMELIST եΥץ
    !  Open NAMELIST file
    !----------------------------------------------------------------
    call FileOpen( unit = unit_nml, & ! (out)
      & file = nmlfile, mode = 'r', & ! (in)
      & err = err )                   ! (out)
    if ( present_and_true(err) ) then
      stat = DC_ENOFILEREAD
      cause_c = nmlfile
      goto 999
    end if


    !-----------------------------------------------------------------
    !  NAMELIST ѿμ
    !  Get NAMELIST group
    !-----------------------------------------------------------------
    read( unit = unit_nml, & ! (in)
      & nml = dyn_as83_nml, iostat = iostat_nml ) ! (out)
    if ( iostat_nml == 0 ) then
      call MessageNotify( 'M', subname, &
        & 'NAMELIST group "%c" is loaded from "%c".', &
        & c1='dyn_as83_nml', c2=trim(nmlfile) )
      write(STDOUT, nml = dyn_as83_nml)
    else
      call MessageNotify( 'W', subname, &
        & 'NAMELIST group "%c" is not found in "%c" (iostat=%d).', &
        & c1='dyn_as83_nml', c2=trim(nmlfile), &
        & i=(/iostat_nml/) )
    end if

    close( unit_nml )

    !-----------------------------------------------------------------
    !  NAMELIST ѿʸ
    !  Substitute NAMELIST group to character arguments
    !-----------------------------------------------------------------
    time_integration_scheme_ = time_integration_scheme
    history_interval_unit_ = history_interval_unit
    history_precision_ = history_precision

    !-----------------------------------------------------------------
    !  λ, 㳰
    !  Termination and Exception handling
    !-----------------------------------------------------------------
999 continue
    call StoreError( stat, subname, err, cause_c )
    call EndSub( subname )
  end subroutine DynAS83NmlRead

  subroutine DynAS83EqualAxes( dyn_as, &
    & z_Sigma, r_Sigma, &
    & err )
    !
    ! Ϳ줿 *z_Sigma*  *r_Sigma*  *dyn_as* 
    ! ݻٷ٥ǡȤǧޤ.
    ! ۤʤˤϥ顼ȯޤ.
    !
    ! ʤ, Ϳ줿 *dyn_as*  Create ˤäƽ
    ! Ƥʤ, ץϥ顼ȯޤ.
    !
    ! Confirm equality between given *z_Sigma*, *y_lat* and 
    ! longitude and latitude data stored in *dyn_as*.
    ! If the equality is not confirmed, error is occurred.
    !
    ! If *dyn_as* is not initialized by "Create" yet,
    ! error is occurred.
    !
    use dc_trace, only: BeginSub, EndSub
    use dc_types, only: STRING, STDOUT
    use dc_string, only: PutLine
    use dcpam_error, only: StoreError, DC_NOERR, DCPAM_ENOTINIT, DCPAM_EAXISMISMATCH
    use dc_message, only: MessageNotify
    use dc_string, only: JoinChar, toChar
    implicit none
    type(DYNAS83), intent(inout):: dyn_as
    real(DP), intent(in), optional:: z_Sigma (0:dyn_as%kmax-1)
                              ! $ \sigma $ ٥ (). 
                              ! Full $ \sigma $ level
    real(DP), intent(in), optional:: r_Sigma (0:dyn_as%kmax)
                              ! $ \sigma $ ٥ (Ⱦ). 
                              ! Half $ \sigma $ level
    logical, intent(out), optional:: err
                              ! 㳰ѥե饰.
                              ! ǥեȤǤ, μ³ǥ顼
                              ! , ץ϶λޤ.
                              !  *err* Ϳ,
                              ! ץ϶λ, 
                              ! *err*  .true. ޤ.
                              !
                              ! Exception handling flag. 
                              ! By default, when error occur in 
                              ! this procedure, the program aborts. 
                              ! If this *err* argument is given, 
                              ! .true. is substituted to *err* and 
                              ! the program does not abort. 

    !-----------------------------------
    !  ѿ
    !  Work variables
    real(DP):: z_SigmaStored (0:dyn_as%kmax-1)
                              ! $ \sigma $ ٥ (). 
                              ! Full $ \sigma $ level
    real(DP):: r_SigmaStored (0:dyn_as%kmax)
                              ! $ \sigma $ ٥ (Ⱦ). 
                              ! Half $ \sigma $ level

    logical:: z_Sigma_gt_check (0:dyn_as%kmax-1), z_Sigma_lt_check (0:dyn_as%kmax-1)
    logical:: r_Sigma_gt_check (0:dyn_as%kmax), r_Sigma_lt_check (0:dyn_as%kmax)
    real(DP), parameter:: check_accuracy = 1.0e-6_DP
    integer:: stat
    character(STRING):: cause_c
    character(*), parameter:: subname = 'DynAS83EqualAxes'
  continue
    call BeginSub(subname)
    stat = DC_NOERR
    cause_c = ''

    !-----------------------------------------------------------------
    !  Υå
    !  Check initialization
    !-----------------------------------------------------------------
    if (.not. dyn_as % initialized) then
      stat = DCPAM_ENOTINIT
      cause_c = 'DYNAS83'
      goto 999
    end if

    !-----------------------------------------------------------------
    !  ǡ
    !  Compare axes data
    !-----------------------------------------------------------------
    call GetAxes ( dyn_as = dyn_as, &                      ! (inout)
      & z_Sigma = z_SigmaStored, r_Sigma = r_SigmaStored ) ! (out)

    if ( present(z_Sigma) ) then
      z_Sigma_gt_check = z_Sigma > ( z_SigmaStored - check_accuracy )
      z_Sigma_lt_check = z_Sigma < ( z_SigmaStored + check_accuracy )
      if ( any(.not. z_Sigma_gt_check) .or. any(.not. z_Sigma_lt_check) ) then
!!$        call MessageNotify ( 'W', subname, &
!!$          & '%c (argument)=%*f', &
!!$          & c1 = 'z_Sigma', d = z_Sigma, n = (/dyn_as % imax/) )
!!$        call MessageNotify ( 'W', subname, &
!!$          & 'dyn_as %% %c =%*f', &
!!$          & c1 = 'z_Sigma', d = z_SigmaStored, n = (/dyn_as % imax/) )
        stat = DCPAM_EAXISMISMATCH
        cause_c = 'z_Sigma'
        goto 999
      end if
    end if

    if ( present(r_Sigma) ) then
      r_Sigma_gt_check = r_Sigma > ( r_SigmaStored - check_accuracy )
      r_Sigma_lt_check = r_Sigma < ( r_SigmaStored + check_accuracy )
      if ( any(.not. r_Sigma_gt_check) .or. any(.not. r_Sigma_lt_check) ) then
!!$        call MessageNotify ( 'W', subname, &
!!$          & '%c (argument)=%*f', &
!!$          & c1 = 'r_Sigma', d = r_Sigma, n = (/dyn_as % jmax/) )
!!$        call MessageNotify ( 'W', subname, &
!!$          & 'dyn_as %% %c =%*f', &
!!$          & c1 = 'r_Sigma', d = r_SigmaStored, n = (/dyn_as % jmax /) )
        stat = DCPAM_EAXISMISMATCH
        cause_c = 'r_Sigma'
        goto 999
      end if
    end if

    !-----------------------------------------------------------------
    !  λ, 㳰
    !  Termination and Exception handling
    !-----------------------------------------------------------------
999 continue
    call StoreError(stat, subname, err, cause_c)
    call EndSub(subname)
  end subroutine DynAS83EqualAxes

  subroutine DynAS83GetAxes( dyn_as, &
    & z_Sigma, r_Sigma, z_DelSigma, &
    & err )
    !
    ! *dyn_as* ˳ǼƤɸ֤ͤޤ.
    !
    ! ʤ, Ϳ줿 *dyn_as*  Create ˤäƽ
    ! Ƥʤ, ץϥ顼ȯޤ.
    !
    ! Return data of axes stored in *dyn_sp*.
    !
    ! If *dyn_as* is not initialized by "Create" yet,
    ! error is occurred.
    !
    use dc_trace, only: BeginSub, EndSub
    use dc_types, only: STRING, STDOUT
    use dc_string, only: PutLine
    use dcpam_error, only: StoreError, DC_NOERR, DCPAM_ENOTINIT
    implicit none
    type(DYNAS83), intent(inout):: dyn_as
    real(DP), intent(out), optional:: z_Sigma (0:dyn_as%kmax-1)
                              ! $ \sigma $ ٥ (). 
                              ! Full $ \sigma $ level
    real(DP), intent(out), optional:: r_Sigma (0:dyn_as%kmax)
                              ! $ \sigma $ ٥ (Ⱦ). 
                              ! Half $ \sigma $ level
    real(DP), intent(out), optional:: z_DelSigma (0:dyn_as%kmax-1)
                              ! $ \Delta \sigma $ (). 
                              ! $ \Delta \sigma $ (Full)
    logical, intent(out), optional:: err
                              ! 㳰ѥե饰.
                              ! ǥեȤǤ, μ³ǥ顼
                              ! , ץ϶λޤ.
                              !  *err* Ϳ,
                              ! ץ϶λ, 
                              ! *err*  .true. ޤ.
                              !
                              ! Exception handling flag. 
                              ! By default, when error occur in 
                              ! this procedure, the program aborts. 
                              ! If this *err* argument is given, 
                              ! .true. is substituted to *err* and 
                              ! the program does not abort. 
    integer:: stat
    character(STRING):: cause_c
    character(*), parameter:: subname = 'DynAS83GetAxes'
  continue
    call BeginSub(subname)
    stat = DC_NOERR
    cause_c = ''

    !-----------------------------------------------------------------
    !  Υå
    !  Check initialization
    !-----------------------------------------------------------------
    if (.not. dyn_as % initialized) then
      stat = DCPAM_ENOTINIT
      cause_c = 'DYNAS83'
      goto 999
    end if

    !-----------------------------------------------------------------
    !  $ \sigma $ (, Ⱦ) ٥
    !  Configure full and half $ \sigma $ level data
    !-----------------------------------------------------------------
    if ( present(z_Sigma) ) z_Sigma = dyn_as % z_Sigma
    if ( present(r_Sigma) ) r_Sigma = dyn_as % r_Sigma

    !-----------------------------------------------------------------
    ! $ \Delta \sigma $ () 
    ! Configure $ \Delta \sigma $ (Full)
    !-----------------------------------------------------------------
    if ( present(z_DelSigma) ) z_DelSigma = dyn_as % z_DelSigma

    !-----------------------------------------------------------------
    !  λ, 㳰
    !  Termination and Exception handling
    !-----------------------------------------------------------------
999 continue
    call StoreError(stat, subname, err, cause_c)
    call EndSub(subname)
  end subroutine DynAS83GetAxes


  subroutine DynAS83NonLinearOnGrid( dyn_as, &
    & xyz_U, xyz_V, xyz_Vor, xyz_Div, &
    & xyz_Temp, xyz_QVap, xy_GradLonPi, xy_GradLatPi, &
    & xyz_UAdv, xyz_VAdv, xyz_DTempDt, xyz_DQVapDt, &
    & xyz_KE, xyz_TempUAdv, xyz_TempVAdv, xy_DPiDt, &
    & xyz_QVapUAdv, xyz_QVapVAdv, &
    & err )
    !
    !  (ȹ) ʻǷ׻ޤ.
    !
    ! ʤ, Ϳ줿 *dyn_as*  Create ˤäƽ
    ! Ƥʤ, ץϥ顼ȯޤ.
    !
    ! Non-linear terms (non gravitational terms) are calculated on
    ! grid points
    !
    ! If *dyn_as* is not initialized by "Create" yet,
    ! error is occurred.
    !
    use dc_trace, only: BeginSub, EndSub
    use dc_types, only: STRING, STDOUT, DP
    use dc_string, only: LChar, PutLine
    use dcpam_error, only: StoreError, DC_NOERR, DCPAM_ENOTINIT
    implicit none
    type(DYNAS83), intent(inout):: dyn_as
    real(DP), intent(in):: xyz_U (0:dyn_as%imax-1, 0:dyn_as%jmax-1, 0:dyn_as%kmax-1)
                              ! $ U $ . ®. Zonal wind
    real(DP), intent(in):: xyz_V (0:dyn_as%imax-1, 0:dyn_as%jmax-1, 0:dyn_as%kmax-1)
                              ! $ V $ . ®. Meridional wind
    real(DP), intent(in):: xyz_Vor (0:dyn_as%imax-1, 0:dyn_as%jmax-1, 0:dyn_as%kmax-1)
                              ! $ \zeta $ . . Vorticity
    real(DP), intent(in):: xyz_Div (0:dyn_as%imax-1, 0:dyn_as%jmax-1, 0:dyn_as%kmax-1)
                              ! $ D $ . ȯ. Divergence
    real(DP), intent(in):: xyz_Temp (0:dyn_as%imax-1, 0:dyn_as%jmax-1, 0:dyn_as%kmax-1)
                              ! $ T $ . . Temperature
    real(DP), intent(in):: xyz_QVap (0:dyn_as%imax-1, 0:dyn_as%jmax-1, 0:dyn_as%kmax-1)
                              ! $ q $ . 漾. Specific humidity
    real(DP), intent(in):: xy_GradLonPi (0:dyn_as%imax-1, 0:dyn_as%jmax-1)
                              ! $ \DD{\pi}{x} $
    real(DP), intent(in):: xy_GradLatPi (0:dyn_as%imax-1, 0:dyn_as%jmax-1)
                              ! $ \DD{\pi}{y} $
    real(DP), intent(out):: xyz_UAdv (0:dyn_as%imax-1, 0:dyn_as%jmax-1, 0:dyn_as%kmax-1)
                              ! $ UA $ . ư̰ή.
                              ! Zonal advection of momentum
    real(DP), intent(out):: xyz_VAdv (0:dyn_as%imax-1, 0:dyn_as%jmax-1, 0:dyn_as%kmax-1)
                              ! $ VA $ . ̱ư̰ή. 
                              ! Meridional advection of momentum
    real(DP), intent(out):: xyz_DTempDt (0:dyn_as%imax-1, 0:dyn_as%jmax-1, 0:dyn_as%kmax-1)
                              ! $ H $ . ٻѲ. 
                              ! Temperature tendency
    real(DP), intent(out):: xyz_DQVapDt (0:dyn_as%imax-1, 0:dyn_as%jmax-1, 0:dyn_as%kmax-1)
                              ! $ R $ . 漾Ѳ. 
                              ! Specific humidity tendency
    real(DP), intent(out):: xyz_KE (0:dyn_as%imax-1, 0:dyn_as%jmax-1, 0:dyn_as%kmax-1)
                              ! $ E $ . ưͥ륮. 
                              ! Kinematic energy
    real(DP), intent(out):: xyz_TempUAdv (0:dyn_as%imax-1, 0:dyn_as%jmax-1, 0:dyn_as%kmax-1)
                              ! $ UT $ . ή. 
                              ! Zonal advection of temperature
    real(DP), intent(out):: xyz_TempVAdv (0:dyn_as%imax-1, 0:dyn_as%jmax-1, 0:dyn_as%kmax-1)
                              ! $ VT $ . ̰ή. 
                              ! Meridional advection of temperature
    real(DP), intent(out):: xy_DPiDt (0:dyn_as%imax-1, 0:dyn_as%jmax-1)
                              ! $ Z $ . ɽ̵Ѳ. 
                              ! Surface pressure tendency
    real(DP), intent(out):: xyz_QVapUAdv (0:dyn_as%imax-1, 0:dyn_as%jmax-1, 0:dyn_as%kmax-1)
                              ! $ Uq $ . 漾ή. 
                              ! Zonal advection of specific humidity
    real(DP), intent(out):: xyz_QVapVAdv (0:dyn_as%imax-1, 0:dyn_as%jmax-1, 0:dyn_as%kmax-1)
                              ! $ Vq $ . 漾̰ή. 
                              ! Meridional advection of specific humidity

    logical, intent(out), optional:: err
                              ! 㳰ѥե饰.
                              ! ǥեȤǤ, μ³ǥ顼
                              ! , ץ϶λޤ.
                              !  *err* Ϳ,
                              ! ץ϶λ, 
                              ! *err*  .true. ޤ.
                              !
                              ! Exception handling flag. 
                              ! By default, when error occur in 
                              ! this procedure, the program aborts. 
                              ! If this *err* argument is given, 
                              ! .true. is substituted to *err* and 
                              ! the program does not abort. 

    !-----------------------------------
    !  ѿ
    !  Work variables
    integer:: kmax            ! ľؿ. 
                              ! Number of vertical level
    real(DP):: r_Sigma (0:dyn_as%kmax)
                              ! $ \sigma $ ٥ (Ⱦ). 
                              ! Half $ \sigma $ level
    real(DP):: z_DelSigma (0:dyn_as%kmax-1)
                              ! $ \Delta \sigma $ (). 
                              ! $ \Delta \sigma $ (Full)
    real(DP):: RPlanet        ! $ a $ . Ⱦ. Radius of planet
    real(DP):: EpsVT          ! $ 1/\epsilon_v - 1 $ . 
    real(DP):: Cp             ! $ C_p $ . 絤갵Ǯ. Specific heat of air at constant pressure
    real(DP):: xy_Cori (0:dyn_as%imax-1, 0:dyn_as%jmax-1)
                              ! $ f\equiv 2\Omega\sin\varphi $ . 
                              ! ꥪѥ᡼. Coriolis parameter
    real(DP):: z_TempAvrXY (0:dyn_as%kmax-1)
                              ! ʿѲ (٥). 
                              ! Average temperature on full sigma level
    real(DP):: r_TempAvrXY (0:dyn_as%kmax)
                              ! ʿѲ (Ⱦ٥). 
                              ! Average temperature on half sigma level
    real(DP):: z_TempInpolA (0:dyn_as%kmax-1)
                              ! $ a $ . ٱľ֤η. 
                              ! Coefficient for vertical interpolation of temperature
    real(DP):: z_TempInpolB (0:dyn_as%kmax-1)
                              ! $ b $ . ٱľ֤η. 
                              ! Coefficient for vertical interpolation of temperature
    real(DP):: z_TempInpolKappa (0:dyn_as%kmax-1)
                              ! $ \kappa $ . ٱľ֤η. 
                              ! Coefficient for vertical interpolation of temperature
    real(DP):: z_HydroAlpha (0:dyn_as%kmax-1)
                              ! $ \alpha $ . ſ尵μη. 
                              ! Coefficient in hydrostatic equation.
    real(DP):: z_HydroBeta (0:dyn_as%kmax-1)
                              ! $ \beta $ . ſ尵μη. 
                              ! Coefficient in hydrostatic equation.

    real(DP):: xyz_PiAdv (0:dyn_as%imax-1, 0:dyn_as%jmax-1, 0:dyn_as%kmax-1)
                              ! $ \Dvect{v} \cdot \nabla \pi $ . 
                              ! $ \pi $ ΰή. Advection of $ \pi $
    real(DP):: xyz_PiAdvSum (0:dyn_as%imax-1, 0:dyn_as%jmax-1, 0:dyn_as%kmax-1)
                              ! $ \sum_k^K(\Dvect{v}\cdot\nabla\pi)\Delta\sigma $ . 
                              ! $ \pi $ ήѲ. Integral downward of advection of $ \pi $
    real(DP):: xyz_DivSum (0:dyn_as%imax-1, 0:dyn_as%jmax-1, 0:dyn_as%kmax-1)
                              ! $ \sum_k^K D\Delta\sigma $ .
                              ! ȯѲ. Integral downward of divergence
    real(DP):: xyr_SigmaDot (0:dyn_as%imax-1, 0:dyn_as%jmax-1, 0:dyn_as%kmax)
                              ! $ \dot{\sigma} $ .
                              ! ľή. Vertical flow
    real(DP):: xyr_SigmaDotNonGrav (0:dyn_as%imax-1, 0:dyn_as%jmax-1, 0:dyn_as%kmax)
                              ! $ \dot{\sigma} $ .
                              ! ľή (). Vertical flow (non gravitational)
    real(DP):: xyz_TempEdd (0:dyn_as%imax-1, 0:dyn_as%jmax-1, 0:dyn_as%kmax-1)
                              ! $ T' = T - \bar{T} $ . 
                              ! ٤ξ (٥). Temperature eddy (full level)
    real(DP):: xyr_TempEdd (0:dyn_as%imax-1, 0:dyn_as%jmax-1, 0:dyn_as%kmax)
                              ! $ \hat{T}' $ . 
                              ! ٤ξ (Ⱦ٥). Temperature eddy (half level)
    real(DP):: xyz_TempVir (0:dyn_as%imax-1, 0:dyn_as%jmax-1, 0:dyn_as%kmax-1)
                              ! $ T_v $ . 
                              ! . Virtual temperature
    real(DP):: xyz_TempVirEdd (0:dyn_as%imax-1, 0:dyn_as%jmax-1, 0:dyn_as%kmax-1)
                              ! $ {T_v}' = T_v - \bar{T} $ . 
                              ! ٤ξ. Virtual temperature eddy

    integer:: k               ! DO 롼Ѻѿ
                              ! Work variables for DO loop
    integer:: stat
    character(STRING):: cause_c
    character(*), parameter:: subname = 'DynAS83NonLinearOnGrid'
  continue
    call BeginSub(subname)
    stat = DC_NOERR
    cause_c = ''

    !-----------------------------------------------------------------
    !  Υå
    !  Check initialization
    !-----------------------------------------------------------------
    if (.not. dyn_as % initialized) then
      stat = DCPAM_ENOTINIT
      cause_c = 'DYNAS83'
      goto 999
    end if

    !-----------------------------------------------------------------
    !  *dyn_as* ˳Ǽɸ, ʪμФ
    !  Fetch coordinate information and physical constants from *dyn_as*
    !-----------------------------------------------------------------
    kmax = dyn_as % kmax
    r_Sigma = dyn_as % r_Sigma
    z_DelSigma = dyn_as % z_DelSigma
    RPlanet = dyn_as % RPlanet
    EpsVT = dyn_as % EpsVT
    Cp = dyn_as % Cp
    xy_Cori = dyn_as % xy_Cori
    z_TempAvrXY = dyn_as % z_TempAvrXY
    r_TempAvrXY = dyn_as % r_TempAvrXY
    z_TempInpolA = dyn_as % z_TempInpolA
    z_TempInpolB = dyn_as % z_TempInpolB
    z_TempInpolKappa = dyn_as % z_TempInpolKappa
    z_HydroAlpha = dyn_as % z_HydroAlpha
    z_HydroBeta = dyn_as % z_HydroBeta

    !-----------------------------------------------------------------
    !  $ \pi $ ΰή, $ \pi $ ήѲ, ȯѲη׻
    !  Calculate advection of $ \pi $, integral downward of advection 
    !  of $ \pi $, integral downward of divergence
    !-----------------------------------------------------------------
    do k = 0, kmax-1
      xyz_PiAdv(:,:,k) = &
        & ( xyz_U(:,:,k) * xy_GradLonPi + xyz_V(:,:,k) * xy_GradLatPi ) &
        & / RPlanet
    enddo

    xyz_PiAdvSum(:,:,kmax-1) = xyz_PiAdv(:,:,kmax-1) * z_DelSigma(kmax-1)
    do k = kmax-2, 0, -1
      xyz_PiAdvSum(:,:,k) = &
        & xyz_PiAdvSum(:,:,k+1) + xyz_PiAdv(:,:,k) * z_DelSigma(k)
    enddo

    xyz_DivSum(:,:,kmax-1) = xyz_Div(:,:,kmax-1) * z_DelSigma(kmax-1)
    do k = kmax-2, 0, -1
      xyz_DivSum(:,:,k) = &
        & xyz_DivSum(:,:,k+1) + xyz_Div(:,:,k) * z_DelSigma(k)
    enddo

    !-----------------------------------------------------------------
    !  ɽ̵Ѳ $ Z $ η׻
    !  Calculate surface pressure tendency $ Z $
    !-----------------------------------------------------------------
    xy_DPiDt = - xyz_PiAdvSum(:,:,0)

    select case ( LChar( trim( dyn_as % time_integration_scheme ) ) )
    case ('explicit')

      xy_DPiDt = xy_DPiDt - xyz_DivSum(:,:,0)

    end select

    !-------------------------------------------------------------------
    !  $ \dot{\sigma} $ η׻
    !  Calculate $ \dot{\sigma} $
    !-------------------------------------------------------------------
    do k = 1, kmax-1
      xyr_SigmaDot(:,:,k) = &
        & r_Sigma(k) * ( xyz_PiAdvSum(:,:,0) + xyz_DivSum(:,:,0) ) &
        & - ( xyz_PiAdvSum(:,:,k) + xyz_DivSum(:,:,k) )

      xyr_SigmaDotNonGrav(:,:,k) = &
        & r_Sigma(k) * xyz_PiAdvSum(:,:,0) - xyz_PiAdvSum(:,:,k)
    enddo

    ! 岼
    ! On upper and lower boundary
    xyr_SigmaDot(:,:,0)    = 0.0_DP
    xyr_SigmaDot(:,:,kmax) = 0.0_DP
    xyr_SigmaDotNonGrav(:,:,0)    = 0.0_DP
    xyr_SigmaDotNonGrav(:,:,kmax) = 0.0_DP

    !-------------------------------------------------------------------
    !  ٤ξ (٥), , ٤ξη׻
    !  Calculate temperature eddy (full level), virtual temperature, 
    !  virtual temperature eddy
    !-------------------------------------------------------------------
    do k = 0, kmax-1
      xyz_TempVir(:,:,k) = &
        & xyz_Temp(:,:,k) * ( 1.0_DP + ( EpsVT * xyz_QVap(:,:,k) ) )

      xyz_TempEdd(:,:,k) = xyz_Temp(:,:,k) - z_TempAvrXY(k)
      xyz_TempVirEdd(:,:,k) = xyz_TempVir(:,:,k) - z_TempAvrXY(k)
    enddo

    !-------------------------------------------------------------------
    !  ٤ξ (Ⱦ٥) η׻
    !  Calculate temperature eddy (half level)
    !-------------------------------------------------------------------
    xyr_TempEdd(:,:,0) = 0.0_DP
    xyr_TempEdd(:,:,kmax) = 0.0_DP
    do k = 1, kmax-1
      xyr_TempEdd(:,:,k) = &
        &   z_TempInpolA(k)   * xyz_Temp(:,:,k)   &
        & + z_TempInpolB(k-1) * xyz_Temp(:,:,k-1) &
        & - r_TempAvrXY(k)
    enddo

    !-----------------------------------------------------------------
    !  ư̰ή, ̱ư̰ήη׻
    !  Calculate advection of zonal momentum and meridional momentum
    !-----------------------------------------------------------------
    do k = 0, kmax-1
      xyz_UAdv(:,:,k) = &
        & ( xyz_Vor(:,:,k) + xy_Cori ) * xyz_V(:,:,k) &
        & - Cp * z_TempInpolKappa(k) &
        &         * xyz_TempVirEdd(:,:,k) * xy_GradLonPi / RPlanet

      xyz_VAdv(:,:,k) = &
        & - ( xyz_Vor(:,:,k) + xy_Cori ) * xyz_U(:,:,k) &
        & - Cp * z_TempInpolKappa(k) &
        &          * xyz_TempVirEdd(:,:,k) * xy_GradLatPi / RPlanet
    end do

    do k = 1, kmax-1
      xyz_UAdv(:,:,k) = xyz_UAdv(:,:,k) &
        & - 1.0_DP / ( 2.0_DP * z_DelSigma(k) ) &
        &    * xyr_SigmaDot(:,:,k) * ( xyz_U(:,:,k-1) - xyz_U(:,:,k) )

      xyz_VAdv(:,:,k) = xyz_VAdv(:,:,k) &
        & - 1.0_DP / ( 2.0_DP * z_DelSigma(k) ) &
        &    * xyr_SigmaDot(:,:,k) * ( xyz_V(:,:,k-1) - xyz_V(:,:,k) )
    end do

    do k = 0, kmax-2
      xyz_UAdv(:,:,k) = xyz_UAdv(:,:,k) &
        & - 1.0_DP / ( 2.0_DP * z_DelSigma(k) ) &
        &    * xyr_SigmaDot(:,:,k+1) * ( xyz_U(:,:,k) - xyz_U(:,:,k+1) )

      xyz_VAdv(:,:,k) = xyz_VAdv(:,:,k) &
        & - 1.0_DP / ( 2.0_DP * z_DelSigma(k) ) &
        &    * xyr_SigmaDot(:,:,k+1) * ( xyz_V(:,:,k) - xyz_V(:,:,k+1) )
    end do

    !-------------------------------------------------------------------
    !  ưͥ륮 (ޤ) η׻
    !  Calculate kinematic energy term 
    !  (including virtual temperature correction)
    !-------------------------------------------------------------------
    xyz_KE = 0.0_DP

    xyz_KE(:,:,0) = &
      & Cp * z_HydroAlpha(0) * ( xyz_TempVir(:,:,0) - xyz_Temp(:,:,0) )

    do k = 1, kmax-1
      xyz_KE(:,:,k) = xyz_KE(:,:,k-1) &
        & + Cp * z_HydroAlpha(k) * ( xyz_TempVir(:,:,k) - xyz_Temp(:,:,k) ) &
        & + Cp * z_HydroBeta(k-1) * ( xyz_TempVir(:,:,k-1) - xyz_Temp(:,:,k-1) )
    enddo

    xyz_KE = xyz_KE + ( xyz_U**2 + xyz_V**2 ) / 2.0_DP


    !-------------------------------------------------------------------
    !  ή, ̰ήη׻
    !  Calculate zonal and meridional advection of temperature
    !-------------------------------------------------------------------
    do k = 0, kmax-1
      xyz_TempUAdv(:,:,k) =  xyz_U(:,:,k) * xyz_TempEdd(:,:,k)
      xyz_TempVAdv(:,:,k) =  xyz_V(:,:,k) * xyz_TempEdd(:,:,k)
    end do

    !-------------------------------------------------------------------
    !  ٤λѲ $ H $ η׻
    !  Calculate temperature tendency term $ H $
    !-------------------------------------------------------------------
    do k = 0, kmax-1
      xyz_DTempDt(:,:,k) = &
        & xyz_TempEdd(:,:,k) * xyz_Div(:,:,k) &
!
        & + z_TempInpolKappa(k) * xyz_TempVir(:,:,k) * xyz_PiAdv(:,:,k) &
!
        & - z_HydroAlpha(k) / z_DelSigma(k) &
        &    * ( xyz_TempVir(:,:,k) * xyz_PiAdvSum(:,:,k) &
        &        + xyz_TempVirEdd(:,:,k) * xyz_DivSum(:,:,k) )
    enddo

    do k = 1, kmax-1
      xyz_DTempDt(:,:,k) = xyz_DTempDt(:,:,k) &
!
        & - xyr_SigmaDot(:,:,k) / z_DelSigma(k) &
        &     * ( xyr_TempEdd(:,:,k) - xyz_TempEdd(:,:,k) ) &
!
        & - xyr_SigmaDotNonGrav(:,:,k) / z_DelSigma(k) &
        &     * ( r_TempAvrXY(k) - z_TempAvrXY(k) )
    enddo

    do k = 0, kmax-2
      xyz_DTempDt(:,:,k) = xyz_DTempDt(:,:,k) &
!
        & - xyr_SigmaDot(:,:,k+1) / z_DelSigma(k) &
        &     * ( xyz_TempEdd(:,:,k) - xyr_TempEdd(:,:,k+1) ) &
!
        & - xyr_SigmaDotNonGrav(:,:,k+1) / z_DelSigma(k) &
        &     * ( z_TempAvrXY(k) - r_TempAvrXY(k+1) ) &
!
        & - z_HydroBeta(k) / z_DelSigma(k) &
        &    * (   xyz_TempVir(:,:,k) * xyz_PiAdvSum(:,:,k+1) &
        &        + xyz_TempVirEdd(:,:,k) * xyz_DivSum(:,:,k+1) )
    enddo

    !-------------------------------------------------------------------
    !  漾ή, 漾̰ήη׻
    !  Calculate zonal and meridional advection of specific humidity
    !-------------------------------------------------------------------
    do k = 0, kmax-1
      xyz_QVapUAdv(:,:,k) =  xyz_U(:,:,k) * xyz_QVap(:,:,k)
      xyz_QVapVAdv(:,:,k) =  xyz_V(:,:,k) * xyz_QVap(:,:,k)
    end do

    !-------------------------------------------------------------------
    !  漾Ѳ $ R $ η׻
    !  Calculate specific humidity tendency $ R $
    !-------------------------------------------------------------------
    xyz_DQVapDt = xyz_QVap * xyz_Div

    do k = 1, kmax-1
      xyz_DQVapDt(:,:,k) = xyz_DQVapDt(:,:,k) &
        & - xyr_SigmaDot(:,:,k) / ( 2.0_DP * z_DelSigma(k) )  &
        &      * ( xyz_QVap(:,:,k-1)  - xyz_QVap(:,:,k) )
    enddo

    do k = 0, kmax-2
      xyz_DQVapDt(:,:,k) = xyz_DQVapDt(:,:,k)  &
        & - xyr_SigmaDot(:,:,k+1) / ( 2.0_DP * z_DelSigma(k) ) &
        &      * ( xyz_QVap(:,:,k) - xyz_QVap(:,:,k+1) )
    enddo

    !-----------------------------------------------------------------
    !  λ, 㳰
    !  Termination and Exception handling
    !-----------------------------------------------------------------
999 continue
    call StoreError(stat, subname, err, cause_c)
    call EndSub(subname)
  end subroutine DynAS83NonLinearOnGrid

  subroutine DynAS83TimeIntegration( dyn_as, &
    & wz_DVorDtN, wz_DDivDtN, wz_DTempDtN, wz_DQVapDtN, w_DPiDtN, &
    & wz_VorB,    wz_DivB,    wz_TempB,    wz_QVapB,    w_PiB, &
    & wz_VorA,    wz_DivA,    wz_TempA,    wz_QVapA,    w_PiA, &
    & err )
    !
    ! ʬԤ, 
    !  $ t $ ʪ̤λѲ $ t-\Delta t$ ʪ̤
    !  $ t+\Delta t $ ʪ̤׻ޤ.
    !
    ! ʬˡˤϥ꡼ץեåѤƤޤ. 
    ! ȻˤʬѤƤޤ. 
    ! ǥեȤǤ, $ \Delta t $ 礭Ȥ뤿, ȹ 
    ! ߥץꥷåˡŬѤƤޤ.
    ! , Create  *time_integration_scheme* 
    ! "Explicit" ꤷˤϽȹ⥨ץꥷåˡˤä
    ! 򤭤ޤ.
    ! 
    ! ʤ, Ϳ줿 *dyn_as*  Create ˤäƽ
    ! Ƥʤ, ץϥ顼ȯޤ.
    !
    ! With time integration, calculate physical values at $ t+\Delta t $
    ! from tendency at $ t $ and physical values at $ t-\Delta t $ .
    !
    ! Leap-frog scheme is used as time integration scheme. 
    ! And forward difference is used to diffusion terms.
    ! By default, semi-implicit scheme is applied to gravitational terms 
    ! for extension of $ \Delta t $ . 
    ! If "Explicit" is specified to *time_integration_scheme* in "Create", 
    ! explicit scheme is applied to gravitational terms as well as 
    ! other terms.
    !
    ! If *dyn_as* is not initialized by "Create" yet,
    ! error is occurred.
    !
    use dc_trace, only: BeginSub, EndSub
    use dc_types, only: STRING, STDOUT, DP
    use dc_message, only: MessageNotify
    use dc_error, only: DC_EARGLACK
    use dcpam_error, only: StoreError, DC_NOERR, DCPAM_ENOTINIT
    use dc_string, only: LChar, PutLine
    use dyn_matrix, only: Solve
    implicit none
    type(DYNAS83), intent(inout):: dyn_as
    real(DP), intent(in):: wz_DVorDtN ((dyn_as%nmax+1)**2, 0:dyn_as%kmax-1)
                              ! $ \DD{\zeta}{t} (t) $ . Ѳ (ڥȥ). 
                              ! Vorticity tendency (spectral)
    real(DP), intent(in):: wz_DDivDtN ((dyn_as%nmax+1)**2, 0:dyn_as%kmax-1)
                              ! $ \DD{D}{t} (t) $ . ȯѲ (ڥȥ). 
                              ! Divergence tendency (spectral)
    real(DP), intent(in):: wz_DTempDtN ((dyn_as%nmax+1)**2, 0:dyn_as%kmax-1)
                              ! $ \DD{T}{t} (t) $ . Ѳ (ڥȥ). 
                              ! Temperature tendency (spectral)
    real(DP), intent(in):: wz_DQVapDtN ((dyn_as%nmax+1)**2, 0:dyn_as%kmax-1)
                              ! $ \DD{q}{t} (t) $ . 漾Ѳ (ڥȥ). 
                              ! Specific humidity tendency (spectral)
    real(DP), intent(in):: w_DPiDtN ((dyn_as%nmax+1)**2)
                              ! $ \DD{Ps}{t} (t) $ . ɽ̵Ѳ (ڥȥ). 
                              ! Surface pressure tendency (spectral)
    real(DP), intent(in):: wz_VorB ((dyn_as%nmax+1)**2, 0:dyn_as%kmax-1)
                              ! $ \zeta (t-\Delta t) $ .  (ڥȥ). 
                              ! Vorticity (spectral)
    real(DP), intent(in):: wz_DivB ((dyn_as%nmax+1)**2, 0:dyn_as%kmax-1)
                              ! $ D (t-\Delta t) $ . ȯ (ڥȥ). 
                              ! Divergence (spectral)
    real(DP), intent(in):: wz_TempB ((dyn_as%nmax+1)**2, 0:dyn_as%kmax-1)
                              ! $ T (t-\Delta t) $ .  (ڥȥ). 
                              ! Temperature (spectral)
    real(DP), intent(in):: w_PiB ((dyn_as%nmax+1)**2)
                              ! $ \pi = \ln P_s (t-\Delta t) $ . ɽ̵ (ڥȥ). 
                              ! Surface pressure (spectral)
    real(DP), intent(in):: wz_QVapB ((dyn_as%nmax+1)**2, 0:dyn_as%kmax-1)
                              ! $ q (t-\Delta t) $ . 漾 (ڥȥ). 
                              ! Specific humidity (spectral)
    real(DP), intent(out):: wz_VorA ((dyn_as%nmax+1)**2, 0:dyn_as%kmax-1)
                              ! $ \zeta (t+\Delta t) $ .  (ڥȥ). 
                              ! Vorticity (spectral)
    real(DP), intent(out):: wz_DivA ((dyn_as%nmax+1)**2, 0:dyn_as%kmax-1)
                              ! $ D (t+\Delta t) $ . ȯ (ڥȥ). 
                              ! Divergence (spectral)
    real(DP), intent(out):: wz_TempA ((dyn_as%nmax+1)**2, 0:dyn_as%kmax-1)
                              ! $ T (t+\Delta t) $ .  (ڥȥ). 
                              ! Temperature (spectral)
    real(DP), intent(out):: w_PiA ((dyn_as%nmax+1)**2)
                              ! $ \pi = \ln P_s (t+\Delta t) $ . ɽ̵ (ڥȥ). 
                              ! Surface pressure (spectral)
    real(DP), intent(out):: wz_QVapA ((dyn_as%nmax+1)**2, 0:dyn_as%kmax-1)
                              ! $ q (t+\Delta t) $ . 漾 (ڥȥ). 
                              ! Specific humidity (spectral)

    logical, intent(out), optional:: err
                              ! 㳰ѥե饰.
                              ! ǥեȤǤ, μ³ǥ顼
                              ! , ץ϶λޤ.
                              !  *err* Ϳ,
                              ! ץ϶λ, 
                              ! *err*  .true. ޤ.
                              !
                              ! Exception handling flag. 
                              ! By default, when error occur in 
                              ! this procedure, the program aborts. 
                              ! If this *err* argument is given, 
                              ! .true. is substituted to *err* and 
                              ! the program does not abort. 

    !-----------------------------------
    !  ѿ
    !  Work variables
    integer:: kmax            ! ľؿ. 
                              ! Number of vertical level
    real(DP):: RPlanet        ! $ a $ .   Ⱦ.       Radius of planet
    real(DP):: Cp             ! $ C_p $ . 絤갵Ǯ.   Specific heat of air at constant pressure
    real(DP):: DelTime        ! $ \Delta t $ . ॹƥå. Time step
    real(DP):: z_HydroAlpha (0:dyn_as%kmax-1)
                              ! $ \alpha $ . ſ尵μη. 
                              ! Coefficient in hydrostatic equation.
    real(DP):: z_HydroBeta (0:dyn_as%kmax-1)
                              ! $ \beta $ . ſ尵μη. 
                              ! Coefficient in hydrostatic equation.

    real(DP):: wz_DiffVorDiv ((dyn_as%nmax+1)**2, 0:dyn_as%kmax-1)
                              ! $ -K_{HD} [(-1)^{N_D/2}\nabla^{N_D}- (2/a^2)^{N_D/2}] $ . 
                              ! ư̿ʿȻ. 
                              ! Coefficient of horizontal momentum diffusion
    real(DP):: wz_DiffTherm ((dyn_as%nmax+1)**2, 0:dyn_as%kmax-1)
                              ! $ -(-1)^{N_D/2}K_{HD}\nabla^{N_D} $ . 
                              ! Ǯ, ʿȻ. 
                              ! Coefficient of horizontal thermal and water diffusion
    real(DP):: wz_rn ((dyn_as%nmax+1)**2, 0:dyn_as%kmax-1)
                              ! $ -n \times (n+1) $ . ץ饷η. 
                              ! Laplacian coefficient
                              !--
                              ! FLAPLA = wz_rn / RPlanet**2
                              !++
    real(DP):: w_Phis ((dyn_as%nmax+1)**2)
                              ! $ \Phi_s $ . ɽݥƥ󥷥. 
                              ! Surface geo-potential
    real(DP):: z_siMtxG (0:dyn_as%kmax-1)
                              ! $ G = C_p \kappa T $
    real(DP):: zz_siMtxH (0:dyn_as%kmax-1, 0:dyn_as%kmax-1)
                              ! $ h = QS - R $
    real(DP):: z_DelSigma (0:dyn_as%kmax-1)
                              ! $ \Delta \sigma $ (). 
                              ! $ \Delta \sigma $ (Full)

    real(DP):: wz_siTempW ((dyn_as%nmax+1)**2, 0:dyn_as%kmax-1)
                              !  (ߥץꥷåˡΤκѿ). 
                              ! Temperature (work variable for semi-implicit scheme)
    real(DP):: wz_siDTempDtW ((dyn_as%nmax+1)**2, 0:dyn_as%kmax-1)
                              ! $ \DD{T}{t} (t) $ . Ѳ (ڥȥ) κѿ. 
                              ! Temperature tendency (spectral) work variable

    real(DP):: w_siPiW ((dyn_as%nmax+1)**2)
                              ! $ \pi $ (ߥץꥷåˡΤκѿ). 
                              ! $ \pi $ (work variable for semi-implicit scheme)
    real(DP):: w_siDPiDtW ((dyn_as%nmax+1)**2)
                              ! $ \DD{Ps}{t} (t) $ . ɽ̵Ѳ (ڥȥ). 
                              ! Surface pressure tendency (spectral)
    real(DP):: wz_siPhiW ((dyn_as%nmax+1)**2, 0:dyn_as%kmax-1)
                              ! $ \Phi = \underline{W} \overline{ \Dvect{T} }^{t}$ .
                              ! (ߥץꥷåˡΤκѿ). 
                              ! (Work variable for semi-implicit scheme)
    real(DP):: wz_siDivAvrTime ((dyn_as%nmax+1)**2, 0:dyn_as%kmax-1)
                              ! ʿѤ $ \Dvect{D} $ (ߥץꥷåˡΤκѿ). 
                              ! Time average $ \Dvect{D} $ (work variable for semi-implicit scheme)

    integer:: k, kk           ! DO 롼Ѻѿ
                              ! Work variables for DO loop
    integer:: stat
    character(STRING):: cause_c
    character(*), parameter:: subname = 'DynAS83TimeIntegration'
  continue
    call BeginSub(subname)
    stat = DC_NOERR
    cause_c = ''

    !-----------------------------------------------------------------
    !  Υå
    !  Check initialization
    !-----------------------------------------------------------------
    if (.not. dyn_as % initialized) then
      stat = DCPAM_ENOTINIT
      cause_c = 'DYNAS83'
      goto 999
    end if

    !-----------------------------------------------------------------
    !  *dyn_as* ˳Ǽʪ, $ \Delta t $ μФ
    !  Fetch physical constant, $ \Delta t $ from *dyn_as*
    !-----------------------------------------------------------------
    kmax = dyn_as % kmax
    RPlanet = dyn_as % RPlanet
    Cp = dyn_as % Cp
    DelTime = dyn_as % DelTime
    z_DelSigma = dyn_as % z_DelSigma
    z_HydroAlpha = dyn_as % z_HydroAlpha
    z_HydroBeta = dyn_as % z_HydroBeta
    wz_rn = dyn_as % wz_rn
    wz_DiffVorDiv = dyn_as % wz_DiffVorDiv
    wz_DiffTherm = dyn_as % wz_DiffTherm
    w_Phis = dyn_as % w_Phis
    z_siMtxG = dyn_as % z_siMtxG
    zz_siMtxH = dyn_as % zz_siMtxH

    !-----------------------------------------------------------------
    !  ȹβʬ
    !  Integration non gravitational terms temporarily
    !-----------------------------------------------------------------
    select case ( LChar( trim(dyn_as % time_integration_scheme ) ) )
    case ('semi-implicit')

      wz_siTempW = &
        & wz_TempB * ( 1.0_DP - DelTime * wz_DiffTherm ) &
        & + wz_DTempDtN * DelTime

      w_siPiW = w_PiB + w_DPiDtN * DelTime

    end select

    !-----------------------------------------------------------------
    !  ݥƥ󥷥η׻
    !  Calculate geopotential
    !-----------------------------------------------------------------
    select case ( LChar( trim(dyn_as % time_integration_scheme ) ) )
    case ('semi-implicit')

      wz_siPhiW (:,0) = Cp * z_HydroAlpha(0) * wz_siTempW (:,0)

      do k = 1, kmax-1
        wz_siPhiW (:,k) = wz_siPhiW(:,k-1) &
          & + Cp * z_HydroAlpha(k) * wz_siTempW (:,k) &
          & + Cp * z_HydroBeta (k-1) * wz_siTempW (:,k-1)
      end do

    case ('explicit')

    end select

    !-----------------------------------------------------------------
    !  ȯαդη׻
    !  Calculate right side of divergence equation
    !-----------------------------------------------------------------
    select case ( LChar( trim(dyn_as % time_integration_scheme ) ) )
    case ('semi-implicit')

      do k = 0, kmax-1
        wz_siDivAvrTime(:,k) = &
          &   wz_DivB(:,k) * ( 1.0_DP + DelTime * wz_DiffVorDiv(:,k) ) &
          &                * ( 1.0_DP + DelTime * 2.0_DP * wz_DiffTherm(:,k) ) &
!
          & + DelTime &
          &     * (   wz_DDivDtN(:,k) &
          &         - wz_rn(:,k) / RPlanet**2  &
          &             * (   wz_siPhiW(:,k) &
          &                 + ( 1.0_DP + DelTime * 2.0_DP * wz_DiffTherm(:,k) ) &
          &                       * ( w_Phis + w_siPiW * z_siMtxG(k) ) &
          &               ) &
          &       )
      end do

    end select

    !-----------------------------------------------------------------
    !  ʿѤ $ \Dvect{D} $  LU ǲ
    !  Solve time average $ \Dvect{D} $ with LU matrix
    !-----------------------------------------------------------------
    select case ( LChar( trim(dyn_as % time_integration_scheme ) ) )
    case ('semi-implicit')

      call Solve( dyn_mtx = dyn_as % dyn_mtx, &    ! (inout)
        & wz_RSVector2Solution = wz_siDivAvrTime ) ! (inout)

    end select

    !-----------------------------------------------------------------
    !  , ɽλѲη׻
    !  Calculate tendency of temperature and surface pressure
    !-----------------------------------------------------------------
    select case ( LChar( trim(dyn_as % time_integration_scheme ) ) )
    case ('semi-implicit')

      wz_siDTempDtW = wz_DTempDtN
      do k = 0, kmax-1
        do kk = 0, kmax-1
          wz_siDTempDtW(:,k) = &
            &   wz_siDTempDtW(:,k) &
            & - zz_siMtxH(k,kk) * wz_siDivAvrTime(:,kk)
        end do
      end do

      w_siDPiDtW = w_DPiDtN
      do kk = 0, kmax-1
        w_siDPiDtW = &
          &   w_siDPiDtW &
          & - z_DelSigma(kk) * wz_siDivAvrTime(:,kk)
      end do

    end select

    !-----------------------------------------------------------------
    !  ʬ. Ȼʬ
    !  Time integration. Forward difference is applied to diffusion
    !-----------------------------------------------------------------

    !-------------------------------
    !  . Vorticity
    wz_VorA = &
      & ( wz_VorB + wz_DVorDtN * DelTime * 2.0_DP ) &
      &    / ( 1.0_DP + DelTime * 2.0_DP * wz_DiffVorDiv )

    !-------------------------------
    !  ȯ. Divergence
    select case ( LChar( trim(dyn_as % time_integration_scheme ) ) )
    case ('semi-implicit')

      wz_DivA  = &
        & wz_siDivAvrTime * 2.0_DP - wz_DivB

    case ('explicit')

      wz_DivA  = &
        & ( wz_DivB + wz_DDivDtN * DelTime * 2.0_DP ) &
        &    / ( 1.0_DP + DelTime * 2.0_DP * wz_DiffVorDiv )

    end select

    !-------------------------------
    !  . Temperature
    select case ( LChar( trim(dyn_as % time_integration_scheme ) ) )
    case ('semi-implicit')

      wz_TempA = &
        & ( wz_TempB + wz_siDTempDtW * DelTime * 2.0_DP ) &
        &    / ( 1.0_DP + DelTime * 2.0_DP * wz_DiffTherm )

    case ('explicit')

      wz_TempA = &
        & ( wz_TempB + wz_DTempDtN * DelTime * 2.0_DP ) &
        &    / ( 1.0_DP + DelTime * 2.0_DP * wz_DiffTherm )

    end select

    !-------------------------------
    !  漾. Specific humidity
    wz_QVapA = &
      & ( wz_QVapB + wz_DQVapDtN * DelTime * 2.0_DP ) &
      &    / ( 1.0_DP + DelTime * 2.0_DP * wz_DiffTherm )

    !-------------------------------
    !  ɽ̵. Surface pressure
    select case ( LChar( trim(dyn_as % time_integration_scheme ) ) )
    case ('semi-implicit')

      w_PiA  = w_PiB + w_siDPiDtW * DelTime * 2.0_DP

    case ('explicit')

      w_PiA  = w_PiB + w_DPiDtN * DelTime * 2.0_DP

    end select

    !-----------------------------------------------------------------
    !  λ, 㳰
    !  Termination and Exception handling
    !-----------------------------------------------------------------
999 continue
    call StoreError(stat, subname, err, cause_c)
    call EndSub(subname)
  end subroutine DynAS83TimeIntegration

  subroutine DynAS83WTplusGPiOnGrid( dyn_as, &
    & xyz_Temp, xy_Ps, &
    & xyz_exWTGPi, &
    & err )
    !
    ! ٤ɽ̵ $ \underline{W} \Dvect{T} + \Dvect{G} \pi $ 
    ! ޤ. ץꥷåˡѤݤ˸ƤФ뤳Ȥ
    ! ꤷƤޤ.
    !
    ! ʤ, Ϳ줿 *dyn_as*  Create ˤäƽ
    ! Ƥʤ, ץϥ顼ȯޤ.
    !
    ! Calculate $ \underline{W} \Dvect{T} + \Dvect{G} \pi $ from 
    ! temperature and surface pressure. This routine is expected to be 
    ! called when explicit scheme is used.
    !
    ! If *dyn_as* is not initialized by "Create" yet,
    ! error is occurred.
    !
    use dc_trace, only: BeginSub, EndSub
    use dc_types, only: STRING, STDOUT
    use dc_string, only: PutLine
    use dcpam_error, only: StoreError, DC_NOERR, DCPAM_ENOTINIT
    implicit none
    type(DYNAS83), intent(inout):: dyn_as
    real(DP), intent(in):: xyz_Temp (0:dyn_as%imax-1, 0:dyn_as%jmax-1, 0:dyn_as%kmax-1)
                              ! $ T $ .     . Temperature
    real(DP), intent(in):: xy_Ps (0:dyn_as%imax-1, 0:dyn_as%jmax-1)
                              ! $ P_s $ .   ɽ̵. Surface pressure
    real(DP), intent(out):: xyz_exWTGPi (0:dyn_as%imax-1, 0:dyn_as%jmax-1, 0:dyn_as%kmax-1)
                              ! $ \underline{W} \Dvect{T} + \Dvect{G} \pi $ .
    logical, intent(out), optional:: err
                              ! 㳰ѥե饰.
                              ! ǥեȤǤ, μ³ǥ顼
                              ! , ץ϶λޤ.
                              !  *err* Ϳ,
                              ! ץ϶λ, 
                              ! *err*  .true. ޤ.
                              !
                              ! Exception handling flag. 
                              ! By default, when error occur in 
                              ! this procedure, the program aborts. 
                              ! If this *err* argument is given, 
                              ! .true. is substituted to *err* and 
                              ! the program does not abort. 


    !-----------------------------------
    !  ѿ
    !  Work variables
    integer:: kmax            ! ľؿ. 
                              ! Number of vertical level
    real(DP):: xyz_exWT (0:dyn_as%imax-1, 0:dyn_as%jmax-1, 0:dyn_as%kmax-1)
                              ! $ \underline{W} \Dvect{T} $ .
    real(DP):: xyz_exGPi (0:dyn_as%imax-1, 0:dyn_as%jmax-1, 0:dyn_as%kmax-1)
                              ! $ \Dvect{G} \pi $ .

    real(DP):: Cp         ! $ C_p $ .    絤갵Ǯ.   Specific heat of air at constant pressure
    real(DP):: z_TempAvrXY (0:dyn_as%kmax-1)
                              ! ʿѲ (٥).
                              ! Average temperature on full sigma level
    real(DP):: z_TempInpolKappa (0:dyn_as%kmax-1)
                              ! $ \kappa $ . ٱľ֤η. 
                              ! Coefficient for vertical interpolation of temperature
    real(DP):: z_HydroAlpha (0:dyn_as%kmax-1)
                              ! $ \alpha $ . ſ尵μη. 
                              ! Coefficient in hydrostatic equation.
    real(DP):: z_HydroBeta (0:dyn_as%kmax-1)
                              ! $ \beta $ . ſ尵μη. 
                              ! Coefficient in hydrostatic equation.

    integer:: k               ! DO 롼Ѻѿ
                              ! Work variables for DO loop
    integer:: stat
    character(STRING):: cause_c
    character(*), parameter:: subname = 'DynAS83WTplusGPiOnGrid'
  continue
    call BeginSub(subname)
    stat = DC_NOERR
    cause_c = ''

    !-----------------------------------------------------------------
    !  Υå
    !  Check initialization
    !-----------------------------------------------------------------
    if (.not. dyn_as % initialized) then
      stat = DCPAM_ENOTINIT
      cause_c = 'DYNAS83'
      goto 999
    end if

    !-----------------------------------------------------------------
    !  *dyn_as* ˳Ǽɸ, ʪμФ
    !  Fetch coordinate information and physical constants from *dyn_as*
    !-----------------------------------------------------------------
    kmax = dyn_as % kmax
    Cp = dyn_as % Cp
    z_TempAvrXY = dyn_as % z_TempAvrXY
    z_TempInpolKappa = dyn_as % z_TempInpolKappa
    z_HydroAlpha = dyn_as % z_HydroAlpha
    z_HydroBeta = dyn_as % z_HydroBeta

    !-----------------------------------------------------------------
    !  $ \underline{W} \Dvect{T} $ η׻
    !  Calculate $ \underline{W} \Dvect{T} $
    !-----------------------------------------------------------------
    xyz_exWT = 0.0_DP

    xyz_exWT(:,:,0) = Cp * z_HydroAlpha(0) * xyz_Temp(:,:,0)

    do k = 1, kmax - 1
      xyz_exWT(:,:,k) = xyz_exWT(:,:,k-1) &
        & + Cp * z_HydroAlpha(k)  * xyz_Temp(:,:,k)   &
        & + Cp * z_HydroBeta(k-1) * xyz_Temp(:,:,k-1)
    enddo

    !-----------------------------------------------------------------
    !  $ \Dvect{G} \pi $ η׻
    !  Calculate $ \Dvect{G} \pi $
    !-----------------------------------------------------------------
    do k = 0, kmax - 1
      xyz_exGPi(:,:,k) = &
        & Cp * z_TempInpolKappa(k) * z_TempAvrXY(k) * log( xy_Ps )
    enddo

    !-----------------------------------------------------------------
    !  $ \underline{W} \Dvect{T} + \Dvect{G} \pi $ η׻
    !  Calculate $ \underline{W} \Dvect{T} + \Dvect{G} \pi $
    !-----------------------------------------------------------------
    xyz_exWTGPi = xyz_exWT + xyz_exGPi

    !-----------------------------------------------------------------
    !  λ, 㳰
    !  Termination and Exception handling
    !-----------------------------------------------------------------
999 continue
    call StoreError(stat, subname, err, cause_c)
    call EndSub(subname)
  end subroutine DynAS83WTplusGPiOnGrid

  subroutine DynAS83HDivOnGrid( dyn_as, &
    & xyz_Div, &
    & xyz_exHDiv, &
    & err )
    !
    ! ȯ $ \underline{h} \Dvect{D} $ 
    ! ޤ. ץꥷåˡѤݤ˸ƤФ뤳Ȥ
    ! ꤷƤޤ.
    !
    ! ʤ, Ϳ줿 *dyn_as*  Create ˤäƽ
    ! Ƥʤ, ץϥ顼ȯޤ.
    !
    ! Calculate $ \underline{h} \Dvect{D} $ from 
    ! divergence. This routine is expected to be 
    ! called when explicit scheme is used.
    !
    ! If *dyn_as* is not initialized by "Create" yet,
    ! error is occurred.
    !
    use dc_trace, only: BeginSub, EndSub
    use dc_types, only: STRING, STDOUT
    use dc_string, only: PutLine
    use dcpam_error, only: StoreError, DC_NOERR, DCPAM_ENOTINIT
    implicit none
    type(DYNAS83), intent(inout):: dyn_as
    real(DP), intent(in):: xyz_Div (0:dyn_as%imax-1, 0:dyn_as%jmax-1, 0:dyn_as%kmax-1)
                              ! $ D $ .     ȯ. Divergence
    real(DP), intent(out):: xyz_exHDiv (0:dyn_as%imax-1, 0:dyn_as%jmax-1, 0:dyn_as%kmax-1)
                              ! $ \underline{h} \Dvect{D} $ .

    logical, intent(out), optional:: err
                              ! 㳰ѥե饰.
                              ! ǥեȤǤ, μ³ǥ顼
                              ! , ץ϶λޤ.
                              !  *err* Ϳ,
                              ! ץ϶λ, 
                              ! *err*  .true. ޤ.
                              !
                              ! Exception handling flag. 
                              ! By default, when error occur in 
                              ! this procedure, the program aborts. 
                              ! If this *err* argument is given, 
                              ! .true. is substituted to *err* and 
                              ! the program does not abort. 

    !-----------------------------------
    !  ѿ
    !  Work variables
    integer:: kmax            ! ľؿ. 
                              ! Number of vertical level
    real(DP):: zz_siMtxH (0:dyn_as%kmax-1, 0:dyn_as%kmax-1)
                              ! $ h = QS - R $

    integer:: k, kk           ! DO 롼Ѻѿ
                              ! Work variables for DO loop

    integer:: stat
    character(STRING):: cause_c
    character(*), parameter:: subname = 'DynAS83HDivOnGrid'
  continue
    call BeginSub(subname)
    stat = DC_NOERR
    cause_c = ''

    !-----------------------------------------------------------------
    !  Υå
    !  Check initialization
    !-----------------------------------------------------------------
    if (.not. dyn_as % initialized) then
      stat = DCPAM_ENOTINIT
      cause_c = 'DYNAS83'
      goto 999
    end if

    !-----------------------------------------------------------------
    !  *dyn_as* ˳Ǽʪ, $ \Delta t $ μФ
    !  Fetch physical constant, $ \Delta t $ from *dyn_as*
    !-----------------------------------------------------------------
    kmax = dyn_as % kmax
    zz_siMtxH = dyn_as % zz_siMtxH

    !-----------------------------------------------------------------
    !  $ \underline{h} \Dvect{D} $ η׻
    !  Calculate $ \underline{h} \Dvect{D} $
    !-----------------------------------------------------------------
    xyz_exHDiv = 0.0_DP

    do k = 0, kmax - 1
      do kk = 0, kmax - 1
        xyz_exHDiv (:,:,k) = xyz_exHDiv (:,:,k) &
          & + zz_siMtxH (k,kk) * xyz_Div (:,:,kk)
      enddo
    enddo

    !-----------------------------------------------------------------
    !  λ, 㳰
    !  Termination and Exception handling
    !-----------------------------------------------------------------
999 continue
    call StoreError(stat, subname, err, cause_c)
    call EndSub(subname)
  end subroutine DynAS83HDivOnGrid


!!$  subroutine DynAS83Sample( dyn_as, err )
!!$    !--
!!$    ! DynAS83Sample 򵭽ҤƤ.
!!$    !++
!!$    ! ʤ, Ϳ줿 *dyn_as*  Create ˤäƽ
!!$    ! Ƥʤ, ץϥ顼ȯޤ.
!!$    !--
!!$    ! Describe brief of DynAS83Sample
!!$    !++
!!$    ! If *dyn_as* is not initialized by "Create" yet,
!!$    ! error is occurred.
!!$    !
!!$    use dc_trace, only: BeginSub, EndSub
!!$    use dc_types, only: STRING, STDOUT
!!$    use dc_string, only: PutLine
!!$    use dcpam_error, only: StoreError, DC_NOERR, DCPAM_ENOTINIT
!!$    implicit none
!!$    type(DYNAS83), intent(inout):: dyn_as
!!$    logical, intent(out), optional:: err
!!$                              ! 㳰ѥե饰.
!!$                              ! ǥեȤǤ, μ³ǥ顼
!!$                              ! , ץ϶λޤ.
!!$                              !  *err* Ϳ,
!!$                              ! ץ϶λ, 
!!$                              ! *err*  .true. ޤ.
!!$                              !
!!$                              ! Exception handling flag. 
!!$                              ! By default, when error occur in 
!!$                              ! this procedure, the program aborts. 
!!$                              ! If this *err* argument is given, 
!!$                              ! .true. is substituted to *err* and 
!!$                              ! the program does not abort. 
!!$    integer:: stat
!!$    character(STRING):: cause_c
!!$    character(*), parameter:: subname = 'DynAS83Sample'
!!$  continue
!!$    call BeginSub(subname)
!!$    stat = DC_NOERR
!!$    cause_c = ''
!!$
!!$    !-----------------------------------------------------------------
!!$    !  Υå
!!$    !  Check initialization
!!$    !-----------------------------------------------------------------
!!$    if (.not. dyn_as % initialized) then
!!$      stat = DCPAM_ENOTINIT
!!$      cause_c = 'DYNAS83'
!!$      goto 999
!!$    end if
!!$
!!$
!!$
!!$    !-----------------------------------------------------------------
!!$    !  λ, 㳰
!!$    !  Termination and Exception handling
!!$    !-----------------------------------------------------------------
!!$999 continue
!!$    call StoreError(stat, subname, err, cause_c)
!!$    call EndSub(subname)
!!$  end subroutine DynAS83Sample



end module dyn_as83
