!= ߲
!
!= Gravitational sedimentation process
!
! Authors::   Yoshiyuki O. TAKAHASHI
! Version::   $Id: grav_sed.f90,v 1.3 2014/05/07 09:39:24 murashin Exp $
! Tag Name::  $Name: dcpam5-20140630-2 $
! Copyright:: Copyright (C) GFD Dennou Club, 2008. All rights reserved.
! License::   See COPYRIGHT[link:../../../COPYRIGHT]
!
module grav_sed
  !
  != ߲
  !
  != Gravitational sedimentation process
  !
  ! <b>Note that Japanese and English are described in parallel.</b>
  !
  ! ߲׻⥸塼Ǥ.
  !
  ! This module calculates gravitational sedimentation. 
  !
  !== Procedures List
  !
  ! GravSed         :: ׻
  ! GravSedInit     :: 
  ! --------------- :: ------------
  ! GravSed         :: Calculation
  ! GravSedInit     :: Initialization
  !
  !== NAMELIST
  !
  ! NAMELIST#grav_sed_nml
  !
  !== References
  !
  ! * Conrath, B. J., 1975:
  !   Thermal structure of the Martian atmosphere during the dissipation of 
  !   the dust storm of 1971, 
  !   <i>Icarus</i>, <b>24</b>, 36--46.
  !
  ! * Lin, S.-J., and R. B. Rood, 1996:
  !   Multidimensional flux-form semi-Lagrangian transport scheme, 
  !   <i>Mon. Wea. Rev.</i>, <b>124</b>, 2046--2070.
  !

  ! ⥸塼 ; USE statements
  !

  ! ̷ѥ᥿
  ! Kind type parameter
  !
  use dc_types, only: DP, &      ! ټ¿. Double precision.
    &                 STRING     ! ʸ.       Strings.

  ! å
  ! Message output
  !
  use dc_message, only: MessageNotify

  ! ˴ؤ
  ! Settings of array for atmospheric composition
  !
  use composition, only: ncmax

  ! ʻ
  ! Grid points settings
  !
  use gridset, only: imax, & ! ٳʻ.
                             ! Number of grid points in longitude
    &                jmax, & ! ٳʻ.
                             ! Number of grid points in latitude
    &                kmax    ! ľؿ.
                             ! Number of vertical level


  ! ʸ ; Declaration statements
  !
  implicit none
  private

  ! ³
  ! Public procedure
  !
  public :: GravSed
  public :: GravSedInit


  ! ѿ
  ! Public variables
  !

  ! ѿ
  ! Private variables
  !
  logical, save :: grav_sed_inited = .false.
                              ! ե饰.
                              ! Initialization flag


  character(*), parameter:: module_name = 'grav_sed'
                              ! ⥸塼̾.
                              ! Module name
  character(*), parameter:: version = &
    & '$Name: dcpam5-20140630-2 $' // &
    & '$Id: grav_sed.f90,v 1.3 2014/05/07 09:39:24 murashin Exp $'
                              ! ⥸塼ΥС
                              ! Module version


  !------------------------------------------------------------------------------------

contains

  !------------------------------------------------------------------------------------
  ! Gravitational sedimentation is calculated.
  ! This routine works as adjustment routine, i.e., mass mixing ratio is 
  ! updated in this routine. 
  ! Current version calculates sedimentation of dust, only. 
  ! In this routine, gravitational sedimentation is calculated by the use of 
  ! a method following flux-form semi-Lagrangian transport scheme (Lin and 
  ! Rood, 1996). 
  !

  subroutine GravSed(                      &
    & SpcName,                             & ! (in )
    & xyr_Press, xyz_VirTemp, xyr_Height,  & ! (in )
    & xyz_QMix,                            & ! (out)
    & xy_SurfGravSedFlux                   & ! (out) optional
    & )

    ! ҥȥǡ
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoPut

    ! ˴ؤ
    ! Settings of array for atmospheric composition
    !
    use composition, only: a_QMixName, CompositionInqIndex

    ! 
    ! Time control
    !
    use timeset, only: &
      & TimeN, &              ! ƥå $ t $ λ. Time of step $ t $.
      & TimesetClockStart, TimesetClockStop, &
      & DelTime                 ! $ \Delta t $ [s]

    ! ʪ
    ! Physical constants settings
    !
    use constants, only: &
      & Grav, &
                              ! $ g $ [m s-2].
                              ! ϲ®.
                              ! Gravitational acceleration
      & GasRDry
                              ! $ R $ [J kg-1 K-1].
                              ! 絤ε.
                              ! Gas constant of air


    character(*), intent(in   ) :: SpcName
    real(DP)    , intent(in   ) :: xyr_Press  (0:imax-1, 1:jmax, 0:kmax)
    real(DP)    , intent(in   ) :: xyz_VirTemp(0:imax-1, 1:jmax, 1:kmax)
    real(DP)    , intent(in   ) :: xyr_Height (0:imax-1, 1:jmax, 0:kmax)
    real(DP)    , intent(inout) :: xyz_QMix   (0:imax-1, 1:jmax, 1:kmax)
    real(DP)    , intent(out  ), optional :: xy_SurfGravSedFlux(0:imax-1, 1:jmax)


    !
    ! local variables
    !
    ! rhod : dust density
    ! mfp  : mean free path
    ! mvc  : modecular viscosity coefficient
    ! rdia : particle diameter
    !
    real(DP) :: PartDen
    real(DP) :: PartDia
    real(DP) :: MeanFreePath
    real(DP) :: MolVisCoef

    real(DP) :: MeanFreePathRef
    real(DP) :: PressLambdaRef


    real(DP) :: xyz_DelAtmMass  (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_DelCompMass (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_DelZ        (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyr_SedVel      (0:imax-1, 1:jmax, 0:kmax)
    real(DP) :: xyr_FracSedDist (0:imax-1, 1:jmax, 0:kmax)
    real(DP) :: xyr_Dist        (0:imax-1, 1:jmax, 0:kmax)
    integer  :: xyr_KIndex      (0:imax-1, 1:jmax, 0:kmax)
    real(DP) :: xyr_QMixFlux    (0:imax-1, 1:jmax, 0:kmax)
    real(DP) :: xyr_IntQMixFlux (0:imax-1, 1:jmax, 0:kmax)
    real(DP) :: xyr_FracQMixFlux(0:imax-1, 1:jmax, 0:kmax)
    real(DP) :: xyz_DQMixDt     (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: xyz_QMixA       (0:imax-1, 1:jmax, 1:kmax)
    real(DP) :: LogPress
    real(DP) :: Press


    integer  :: i
    integer  :: j
    integer  :: k
    integer  :: kk


    ! ¹ʸ ; Executable statement
    !

    ! ǧ
    ! Initialization check
    !
    if ( .not. grav_sed_inited ) then
      call MessageNotify( 'E', module_name, 'This module has not been initialized.' )
    end if



    ! Calculation of mass in each layer and layer thickness in unit of meter
    !   Layer thickness is calculated by using mass of a layer.
!!$    xyz_Rho = xyz_Press / ( GasRDry * xyz_VirTemp )
    do k = 1, kmax
      xyz_DelAtmMass(:,:,k) = ( xyr_Press(:,:,k-1) - xyr_Press(:,:,k) ) / Grav
    end do
!!$    xyz_DelZ = xyz_DelAtmMass / xyz_Rho
    do k = 1, kmax
      xyz_DelZ(:,:,k) = xyr_Height(:,:,k) - xyr_Height(:,:,k-1)
    end do


    ! Calculation of mass of constituents in a layer
    xyz_DelCompMass = xyz_QMix * xyz_DelAtmMass


    !
    ! calculation of sedimentation terminal velocity
    !
    if ( SpcName == 'MarsDust' ) then

      !
      ! The values below are obtained from Conrath (1975). 
      ! Particle radius of 1e-6 m is assumed. 
      !
      MolVisCoef      = 1.5e-4_DP * 1.0e-3_DP * 1.0e2_DP
      MeanFreePathRef = 2.2e-4_DP * 1.0e-2_DP
      PressLambdaRef  = 25.0e2_DP
      PartDen         = 3.0e3_DP
      PartDia         = 2.0_DP * 1.0e-6_DP

      xyr_SedVel = &
        & aaa_SedVel( &
        &   MolVisCoef, MeanFreePathRef, PressLambdaRef, PartDen, PartDia, &
        &   max( xyr_Press, 1.0e-20_DP )                                      &
        & )
      k = kmax
      xyr_SedVel(:,:,k) = 0.0_DP

    else if ( SpcName == 'MarsH2OCloud' ) then

      !
      ! The values below are obtained from Conrath (1975). 
      !
      MolVisCoef      = 1.5e-4_DP * 1.0e-3_DP * 1.0e2_DP
      MeanFreePathRef = 2.2e-4_DP * 1.0e-2_DP
      PressLambdaRef  = 25.0e2_DP
      !
      ! Particle radius of 1e-6 m is assumed. 
      !
      PartDen         = 1.0e3_DP
      PartDia         = 2.0_DP * 1.0e-6_DP

      xyr_SedVel = &
        & aaa_SedVel(                                                      &
        &   MolVisCoef, MeanFreePathRef, PressLambdaRef, PartDen, PartDia, &
        &   max( xyr_Press, 1.0e-20_DP )                                      &
        & )
      k = kmax
      xyr_SedVel(:,:,k) = 0.0_DP

    else
      call MessageNotify( 'E', module_name, 'Specie %c is inappropriate', c1 = trim( SpcName ) )
    end if


    ! Calculation of sedimentation distance during a time step of 2 * DelTime
    xyr_Dist = abs( xyr_SedVel ) * 2.0_DP * DelTime
    do k = 0, kmax-1
      do j = 1, jmax
        do i = 0, imax-1

          ! A k index in which all mass of the layer does not fall is 
          ! searched. In addition, fractional sedimentation velocity is 
          ! calculated. 
          xyr_KIndex(i,j,k) = -1
          do kk = k+1, kmax-1
            ! If sedimentation velocity (distance) is positive, and all of 
            ! mass in kk layer does not fall, KIndex is kk.
            if ( ( xyr_Dist(i,j,k) >= 0.0_DP ) .and. &
              &  ( xyr_Dist(i,j,k) <= xyz_DelZ(i,j,kk) ) ) then
              xyr_KIndex     (i,j,k) = kk
              xyr_FracSedDist(i,j,k) = xyr_Dist(i,j,k)
            end if
            ! Sedimentation distance is decreased for preparation for next 
            ! layer.
            ! If xyz_Dist become negative, any mass of the upper layer does 
            ! not fall.
            xyr_Dist(i,j,k) = xyr_Dist(i,j,k) - xyz_DelZ(i,j,kk)
          end do
          ! Calculation for upper most layer.
          kk = kmax
          if ( xyr_Dist(i,j,k) >= 0.0_DP ) then
            xyr_KIndex     (i,j,k) = kk
            xyr_FracSedDist(i,j,k) = min( xyr_Dist(i,j,k), xyz_DelZ(i,j,kk) )
          end if

        end do
      end do
    end do
    ! K index and fractional sedimentation velocity at model top.
    ! No flux is assumed at the model top. 
    k = kmax
    xyr_KIndex     (:,:,k) = -1
    xyr_FracSedDist(:,:,k) = 0.0_DP


    ! Calculation of integer mass flux.
    xyr_IntQMixFlux = 0.0_DP
    do k = 0, kmax-1
      do j = 1, jmax
        do i = 0, imax-1

          do kk = k+1, xyr_KIndex(i,j,k)-1
            xyr_IntQMixFlux(i,j,k) = xyr_IntQMixFlux(i,j,k) &
              & + xyz_DelCompMass(i,j,kk)
          end do
          xyr_IntQMixFlux(i,j,k) = xyr_IntQMixFlux(i,j,k) / ( 2.0_DP * DelTime )

        end do
      end do
    end do

    ! Add sign of sedimentation velocity.
    ! This is equivalent to mulplying -1.
    xyr_IntQMixFlux = sign( 1.0_DP, xyr_SedVel ) * xyr_IntQMixFlux


    ! Calculation of fractional mass flux
    k = kmax
    xyr_FracQMixFlux(:,:,k) = 0.0_DP
    do k = kmax-1, 0, -1
      do j = 1, jmax
        do i = 0, imax-1
          kk = xyr_KIndex(i,j,k)
          !-----
          ! Simple method
!!$            xyrf_FracQMixFlux(i,j,k,n) =                       &
!!$              &   xyrf_FracSedDist(i,j,k,n) / xyz_DelZ(i,j,kk) &
!!$              & * xyzf_DelCompMass(i,j,kk,n)
          !-----
          ! Method considering exponential distribution of mass with height
          if ( xyr_Press(i,j,kk) == 0.0_DP ) then
            LogPress =                                                      &
              &   log( xyr_Press(i,j,kk-1) * 1.0e-1_DP / xyr_Press(i,j,kk-1) ) &
              & / xyz_DelZ(i,j,kk) * xyr_FracSedDist(i,j,k)                 &
              & + log( xyr_Press(i,j,kk-1) )
            Press = exp( LogPress )
            xyr_FracQMixFlux(i,j,k) =                                     &
              &   ( xyr_Press(i,j,kk-1) - Press                        )  &
              & / ( xyr_Press(i,j,kk-1) - xyr_Press(i,j,kk-1) * 1.0e-1_DP )  &
              & * xyz_DelCompMass(i,j,kk)
          else
            LogPress =                                           &
              &   log( xyr_Press(i,j,kk) / xyr_Press(i,j,kk-1) ) &
              & / xyz_DelZ(i,j,kk) * xyr_FracSedDist(i,j,k)      &
              & + log( xyr_Press(i,j,kk-1) )
            Press = exp( LogPress )
            xyr_FracQMixFlux(i,j,k) =                          &
              &   ( xyr_Press(i,j,kk-1) - Press             )  &
              & / ( xyr_Press(i,j,kk-1) - xyr_Press(i,j,kk) )  &
              & * xyz_DelCompMass(i,j,kk)
          end if
          !-----
          xyr_FracQMixFlux(i,j,k) = xyr_FracQMixFlux(i,j,k) &
            & / ( 2.0_DP * DelTime )
        end do
      end do
    end do

    ! Add sign of sedimentation velocity.
    ! This is equivalent to mulplying -1.
    xyr_FracQMixFlux = sign( 1.0_DP, xyr_SedVel ) * xyr_FracQMixFlux


    xyr_QMixFlux = xyr_IntQMixFlux + xyr_FracQMixFlux


    !
    ! estimate dust mixing ratio at next time step
    !
    do k = 1, kmax
      xyz_DQMixDt(:,:,k) =                                  &
        &   ( xyr_QMixFlux(:,:,k) - xyr_QMixFlux(:,:,k-1) ) &
        & / ( xyr_Press   (:,:,k) - xyr_Press   (:,:,k-1) ) &
        & * Grav
    end do


    xyz_QMixA = xyz_QMix + xyz_DQMixDt * 2.0_DP * DelTime

    xyz_QMix  = xyz_QMixA


    if ( present ( xy_SurfGravSedFlux ) ) then
      xy_SurfGravSedFlux = xyr_QMixFlux(:,:,0)
    end if


    ! ҥȥǡ
    ! History data output
    !
!!$    call HistoryAutoPut( TimeN, 'Surf'//trim(a_QMixName(n))//'GravSedFlux', &
!!$      & xyrf_QMixFlux(:,:,0,n)  )


  end subroutine GravSed

  !------------------------------------------------------------------------------------
  ! Sedimentation velocity is calculated by the use of a formula of Conrath (1975)
  !

  function aaa_SedVel(                                               &
    & MolVisCoef, MeanFreePathRef, PressLambdaRef, PartDen, PartDia, &
    & aaa_Press                                                      &
    & ) result( aaa_Result )

    ! ʪ
    ! Physical constants settings
    !
    use constants, only: &
      & Grav
                              ! $ g $ [m s-2].
                              ! ϲ®.
                              ! Gravitational acceleration


    real(DP), intent(in) :: MolVisCoef
    real(DP), intent(in) :: MeanFreePathRef
    real(DP), intent(in) :: PressLambdaRef
    real(DP), intent(in) :: PartDen
    real(DP), intent(in) :: PartDia
    real(DP), intent(in) :: aaa_Press(:,:,:)

    real(DP) :: aaa_Result(size(aaa_Press,1),size(aaa_Press,2),size(aaa_Press,3))

    !
    ! local variables
    !
    real(DP) :: aaa_MeanFreePath(size(aaa_Press,1),size(aaa_Press,2),size(aaa_Press,3))


    ! ¹ʸ ; Executable statement
    !

    aaa_MeanFreePath = MeanFreePathRef * ( PressLambdaRef / aaa_Press )
    aaa_Result =                                                &
      & - PartDen * Grav * PartDia**2 / ( 18.0_DP * MolVisCoef ) &
      & * ( 1.0_DP + 2.0_DP * aaa_MeanFreePath / PartDia )


  end function aaa_SedVel

  !--------------------------------------------------------------------------------------
!!$  !
!!$  ! This routine works as an adjustment-type one. 
!!$  !
!!$
!!$  subroutine dust_borrowingfrombelow( damassn, gdmmrn, gdmassg, ijs, ije )
!!$
!!$    use matype
!!$    use maparam, only : imax, jmax, kmax
!!$
!!$    implicit none
!!$
!!$    real(dp)    , intent(in   ) :: damassn( imax, jmax, kmax )
!!$    real(dp)    , intent(inout) :: gdmmrn ( imax, jmax, kmax )
!!$    real(dp)    , intent(inout) :: gdmassg( imax, jmax )
!!$    integer(i4b), intent(in   ) :: ijs, ije
!!$
!!$
!!$    !
!!$    ! local variables
!!$    !
!!$    ! ddm     : dust deficit mass
!!$    !
!!$    real(dp)                :: ddm
!!$
!!$    integer(i4b)            :: ij, k
!!$    integer(i4b), parameter :: j = 1
!!$
!!$
!!$    !
!!$    ! borrowing
!!$    !
!!$    do k = 1, kmax-1
!!$      do ij = ijs, ije
!!$        ddm = -min( gdmmrn( ij, j, k ), 0.0d0 ) * damassn( ij, j, k )
!!$        gdmmrn( ij, j, k   ) = max( gdmmrn( ij, j, k ), 0.0d0 )
!!$        gdmmrn( ij, j, k+1 ) = gdmmrn( ij, j, k+1 ) &
!!$          - ddm / damassn( ij, j, k+1 )
!!$      end do
!!$    end do
!!$    k = kmax
!!$    do ij = ijs, ije
!!$      ddm = -min( gdmmrn( ij, j, k ), 0.0d0 ) * damassn( ij, j, k )
!!$      gdmmrn ( ij, j, k ) = max( gdmmrn( ij, j, k ), 0.0d0 )
!!$      gdmassg( ij, j )    = gdmassg( ij, j ) + ddm
!!$    end do
!!$
!!$
!!$  end subroutine dust_borrowingfrombelow
!!$
  !--------------------------------------------------------------------------------------

  subroutine GravSedInit

    ! ⥸塼 ; USE statements
    !

    ! NAMELIST եϤ˴ؤ桼ƥƥ
    ! Utilities for NAMELIST file input
    !
    use namelist_util, only: namelist_filename, NmlutilMsg, NmlutilAryValid

    ! ե
    ! File I/O support
    !
    use dc_iounit, only: FileOpen

    ! ҥȥǡ
    ! History data output
    !
    use gtool_historyauto, only: HistoryAutoAddVariable

    ! ˴ؤ
    ! Settings of array for atmospheric composition
    !
    use composition, only: a_QMixName

    ! ʸ ; Declaration statements
    !
    implicit none

!!$    integer:: unit_nml        ! NAMELIST ե륪ץֹ.
!!$                              ! Unit number for NAMELIST file open
!!$    integer:: iostat_nml      ! NAMELIST ɤ߹߻ IOSTAT.
!!$                              ! IOSTAT of NAMELIST read

    integer  :: n

    ! NAMELIST ѿ
    ! NAMELIST group name
    !
!!$    namelist /grav_sed_nml/    &
!!$      & FlagConstDiffCoef,              &
!!$      & ConstDiffCoefM, ConstDiffCoefH, &
!!$!
!!$      & SquareVelMin, BulkRiNumMin,     &
!!$!
!!$      & MixLengthMax, TildeShMin, TildeSmMin, &
!!$      & VelDiffCoefMin, TempDiffCoefMin, QMixDiffCoefMin, &
!!$      & VelDiffCoefMax, TempDiffCoefMax, QMixDiffCoefMax, &
!!$!
!!$      & MYLv2ParamA1, MYLv2ParamB1, MYLv2ParamA2, MYLv2ParamB2, MYLv2ParamC1, &
!!$      & FlagCalcRiWithTv


    ! ¹ʸ ; Executable statement
    !

    if ( grav_sed_inited ) return


    ! ǥեͤ
    ! Default values settings
    !


    ! NAMELIST ɤ߹
    ! NAMELIST is input
    !
!!$    if ( trim(namelist_filename) /= '' ) then
!!$      call FileOpen( unit_nml, &          ! (out)
!!$        & namelist_filename, mode = 'r' ) ! (in)
!!$
!!$      rewind( unit_nml )
!!$      read( unit_nml, &                   ! (in)
!!$        & nml = grav_sed_nml, &           ! (out)
!!$        & iostat = iostat_nml )           ! (out)
!!$      close( unit_nml )
!!$
!!$      call NmlutilMsg( iostat_nml, module_name ) ! (in)
!!$      if ( iostat_nml == 0 ) write( STDOUT, nml = grav_sed_nml )
!!$    end if


    ! ҥȥǡϤΤΤؤѿϿ
    ! Register of variables for history data output
    !
!!$    do n = 1, ncmax
!!$      call HistoryAutoAddVariable( 'Surf'//trim(a_QMixName(n))//'GravSedFlux', &
!!$        & (/ 'lon ', 'lat ', 'time' /), &
!!$        & 'surface gravitational sedimentation flux of ' // trim(a_QMixName(n)), &
!!$        & 'kg m-2 s-1' )
!!$    end do


    !  ; Print
    !
    call MessageNotify( 'M', module_name, '----- Initialization Messages -----' )
    call MessageNotify( 'M', module_name, '-- version = %c', c1 = trim(version) )

    grav_sed_inited = .true.

  end subroutine GravSedInit

  !--------------------------------------------------------------------------------------

end module grav_sed
