
// -*- C++ -*-

// (c) COPYRIGHT OPeNDAP 2004
// Please read the full copyright statement in the file COPYRIGHT.
//
// Authors:
//      James Gallagher <jgallagher@opendap.org>

// Base class for methods common to all the netCDF DAP dat type classes.
// jhrg 08/31/04

#ifndef nc_access_h
#define nc_access_h 1

// #define DODS_DEBUG 1

#include "netcdf.h"

#include "BaseType.h"
#include "Constructor.h"
#include "Error.h"
#include "InternalErr.h"
#include "debug.h"

#include "ClientParams.h"
#include "nc_util.h"

using namespace std;

/** Interface for special methods used to perform translation. This interface
    is added to the BaseType interface; see NCArray, et cetera.
    
    @note Never add fields to this class without modifying NCArray, et c.,
    to ensure those fields are copied correctly.
    @see NCArray */
class NCAccess {
    string d_impl_version;

    NCAccess &operator=(const NCAccess &rhs) {
        throw InternalErr(__FILE__, __LINE__, "Assignment not implemented"); 
    }    

protected:
    BaseType *d_source;       /// Reference to source var if translated
    bool d_translated;
    
public:
    NCAccess() : d_impl_version(""), d_source(0), d_translated(false) {}
    NCAccess(const NCAccess &nca) { clone(nca); }
    virtual ~NCAccess() {
        DBG(cerr << "Delete d_source " << d_source << endl);
        delete d_source; d_source = 0;
    }
    
    /// clone
    void clone(const NCAccess &nca);
    
    /// Build a constraint that will retrieve this variable.
    virtual string build_constraint(int outtype, const size_t *start,
            const size_t *edges, const ptrdiff_t *stride) throw(Error);
            
    /// Scan a set of projection clauses and extract any info for this variable.
    virtual void store_projection(const string &proj);
    
    /// Can this type be converted to outtype?
    virtual bool is_convertable(int outtype);
    
    /// Get the netCDF type for this variable.
    virtual nc_type get_nc_type() throw(InternalErr);
    
    /// Read data from the variable.
    virtual void extract_values(void *values, int elements, int outtype)
        throw(Error);
    
    /// If created during translation, return the source variable.
    virtual BaseType *get_source() const;
    
    /// Set the source variable. Default version throws InternalErr.
    virtual void set_source(BaseType *s) throw(InternalErr);
    
    /// Get the implementation version string
    virtual string get_implementation_version() const { 
        return d_impl_version;
    }
    
    /// Set the implementation version string
    virtual void set_implementation_version(const string &v) {
        d_impl_version = v;
    }
    
    /// Is this variable the result of translation?
    virtual bool get_translated() const { return d_translated; }
    
    /// Set the \e translated property.
    virtual void set_translated(const bool t) { d_translated = t; }
    
    /// Flatten this variable.
    virtual VarList flatten(const ClientParams &cp, const string &parent_name);
    
    /// Does this variable contain a Sequence?
    virtual BaseType *find_child_sequence();
};

// $Log: NCAccess.h,v $
// Revision 1.8  2005/04/11 18:38:20  jimg
// Fixed a problem with NCSequence where nested sequences were not flagged
// but instead were translated. The extract_values software cannot process a
// nested sequence yet. Now the code inserts an attribute that notes that a
// nested sequence has been elided.
//
// Revision 1.7  2005/03/05 00:16:58  jimg
// checkpoint: working on memory leaks found using unit tests
//
// Revision 1.6  2005/02/26 00:43:20  jimg
// Check point: This version of the CL can now translate strings from the
// server into char arrays. This is controlled by two things: First a
// compile-time directive STRING_AS_ARRAY can be used to remove/include
// this feature. When included in the code, only Strings associated with
// variables created by the translation process will be turned into char
// arrays. Other String variables are assumed to be single character strings
// (although there may be a bug with the way these are handled, see
// NCAccess::extract_values()).
//
// Revision 1.5  2005/02/17 23:44:13  jimg
// Modifications for processing of command line projections combined
// with the limit stuff and projection info passed in from the API. I also
// consolodated some of the code by moving d_source from various
// classes to NCAccess. This may it so that DODvario() could be simplified
// as could build_constraint() and store_projection() in NCArray.
//
// Revision 1.4  2005/01/26 23:25:51  jimg
// Implemented a fix for Sequence access by row number when talking to a
// 3.4 or earlier server (which contains a bug in is_end_of_rows()).
//
// Revision 1.3  2004/11/30 22:11:35  jimg
// I replaced the flatten_*() functions with a flatten() method in
// NCAccess. The default version of this method is in NCAccess and works
// for the atomic types; constructors must provide a specialization.
// Then I removed the code that copied the variables from vectors to
// lists. The translation code in NCConnect was modified to use the
// new method.
//
// Revision 1.2  2004/10/22 21:51:34  jimg
// More massive changes: Translation of Sequences now works so long as the
// Sequence contains only atomic types.
//
// Revision 1.1  2004/09/08 22:10:54  jimg
// Added
// 

#endif // nc_access_h
