
// -*- mode: c++; c-basic-offset:4 -*-

// This file is part of libnc-dods, A C++ implementation of netCDF 3 which
// supports reading from local files and OPeNDAP servers.

// Copyright (c) 2004 OPeNDAP, Inc.
// Author: James Gallagher <jgallagher@opendap.org>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// You can contact OPeNDAP, Inc. at PO Box 112, Saunderstown, RI. 02874-0112.
 
#include <cppunit/TextTestRunner.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/extensions/HelperMacros.h>

// Tests for NCSequence 02/24/04 jhrg

#include <netcdf.h>

// #define DODS_DEBUG 

#include "NCConnect.h"
#include "NCSequence.h"
#include "NCStructure.h"
#include "NCArray.h"
#include "NCByte.h"
#include "nc_util.h"

#include "debug.h"

using namespace CppUnit;
using namespace std;

static void
delete_basetype(BaseType *btp)
{
    delete btp; btp = 0;
}

class NCSequenceTest : public TestFixture {
private:
    NCSequence *seq;
    NCSequence *s;
    ClientParams *cp;
    NCTypeFactory *factory;
public: 
    NCSequenceTest() {}
    ~NCSequenceTest() {}

    void setUp() {
        cp = new ClientParams("http://localhost/");
        seq = new NCSequence("test");
        s = new NCSequence("test");
        factory = new NCTypeFactory;
    }

    void tearDown() {
       delete factory;
       delete s;
       delete seq;
       delete cp;
    }
    
    CPPUNIT_TEST_SUITE( NCSequenceTest );

    CPPUNIT_TEST(store_projection_test);
    CPPUNIT_TEST(flatten_test1);
    CPPUNIT_TEST(flatten_test2);
    CPPUNIT_TEST(flatten_test3);
    CPPUNIT_TEST(flatten_test4);
    CPPUNIT_TEST(flatten_test5);
    CPPUNIT_TEST(flatten_test6);
    CPPUNIT_TEST(flatten_test7);

    CPPUNIT_TEST_SUITE_END();

    void store_projection_test() {
        seq->store_projection("");
        CPPUNIT_ASSERT(seq->d_start == -1 && seq->d_stop == -1 && seq->d_stride == -1);
        
        seq->store_projection("ready");
        CPPUNIT_ASSERT(seq->d_start == -1 && seq->d_stop == -1 && seq->d_stride == -1);

        seq->store_projection("ready[10:4:40]");
        CPPUNIT_ASSERT(seq->d_start == -1 && seq->d_stop == -1 && seq->d_stride == -1);

        seq->store_projection("test");
        CPPUNIT_ASSERT(seq->d_start == -1 && seq->d_stop == -1 && seq->d_stride == -1);
        
        seq->store_projection("test[17:3:53]");
        CPPUNIT_ASSERT(seq->d_start == 17 && seq->d_stop == 53 && seq->d_stride == 3);

        seq->store_projection("test[17:3:53],ready[9:2:18]");
        CPPUNIT_ASSERT(seq->d_start == 17 && seq->d_stop == 53 && seq->d_stride == 3);

        seq->store_projection("sample,test[17:3:53],ready[9:2:18]");
        CPPUNIT_ASSERT(seq->d_start == 17 && seq->d_stop == 53 && seq->d_stride == 3);

        seq->store_projection("sample[2:4;8],test[17:3:53],ready[9:2:18]");
        CPPUNIT_ASSERT(seq->d_start == 17 && seq->d_stop == 53 && seq->d_stride == 3);
        
        seq->store_projection("test[17:53]");
        CPPUNIT_ASSERT(seq->d_start == 17 && seq->d_stop == 53 && seq->d_stride == 1);

        seq->store_projection("test[17]");
        CPPUNIT_ASSERT(seq->d_start == 17 && seq->d_stop == 17 && seq->d_stride == 1);
    }
    // A Structure with two atomic types
    void flatten_test1() {
        Byte *b = factory->NewByte("b1");
        s->add_var(b);
        delete b; b = 0;
        Int32 *i32 = factory->NewInt32("i1");
        s->add_var(i32);
        delete i32; i32 = 0;
        Constructor::Vars_iter i = s->var_begin();
        CPPUNIT_ASSERT((*i)->name() == "b1" && (*i++)->type() == dods_byte_c);
        CPPUNIT_ASSERT((*i)->name() == "i1" && (*i++)->type() == dods_int32_c);
        CPPUNIT_ASSERT(i == s->var_end());
        
        VarList vars = s->flatten(*cp, "");
        VarListIter v = vars.begin();
        CPPUNIT_ASSERT((*v)->name() == "test.b1"
                       && (*v)->type() == dods_array_c
                       && (*v++)->var()->type() == dods_byte_c);
        CPPUNIT_ASSERT((*v)->name() == "test.i1"
                       && (*v)->type() == dods_array_c
                       && (*v++)->var()->type() == dods_int32_c);
        CPPUNIT_ASSERT(v == vars.end());
        
        for_each(vars.begin(), vars.end(), delete_basetype);
    }

    // with an array    
    void flatten_test2() {
        Byte *b = factory->NewByte("b1");
        s->add_var(b);
        delete b; b = 0;
        Int32 *i1 = factory->NewInt32("i1");
        Array *a = factory->NewArray("i1", i1);
        delete i1; i1 = 0;
        s->add_var(a);
        delete a; a = 0;
        Constructor::Vars_iter i = s->var_begin();
        CPPUNIT_ASSERT((*i)->name() == "b1" && (*i++)->type() == dods_byte_c);
        CPPUNIT_ASSERT((*i)->name() == "i1" && (*i)->type() == dods_array_c
                       && (*i++)->var()->type() == dods_int32_c);
        CPPUNIT_ASSERT(i == s->var_end());
        
        VarList vars = s->flatten(*cp, "");
        VarListIter v = vars.begin();
        CPPUNIT_ASSERT((*v)->name() == "test.b1" 
                       && (*v)->type() == dods_array_c
                       && (*v++)->var()->type() == dods_byte_c);
        CPPUNIT_ASSERT((*v)->name() == "test.i1"
                       && (*v)->type() == dods_array_c
                       && (*v++)->var()->type() == dods_int32_c);
        CPPUNIT_ASSERT(v == vars.end());

        for_each(vars.begin(), vars.end(), delete_basetype);
    }     
    
    // with a structure
    void flatten_test3() {
        Byte *b = factory->NewByte("b1");
        s->add_var(b);
        delete b; b = 0;
        Int32 *i32 = factory->NewInt32("i1");
        Structure *ss = factory->NewStructure("ss");
        ss->add_var(i32);
        delete i32; i32 = 0;
        s->add_var(ss);
        delete ss; ss = 0;
        
        DBG2(cerr << "Before flatten: "; s->print_decl(cerr, ""));
        Constructor::Vars_iter i = s->var_begin();
        CPPUNIT_ASSERT((*i)->name() == "b1" && (*i++)->type() == dods_byte_c);
        CPPUNIT_ASSERT((*i)->name() == "ss" && (*i++)->type() == dods_structure_c);
        CPPUNIT_ASSERT(i == s->var_end());
        
        VarList vars = s->flatten(*cp, "");
        VarListIter v = vars.begin();
        v++;
        DBG2(cerr << "flattened var: "; (*v)->print_decl(cerr, ""));
        CPPUNIT_ASSERT((*v)->name() == "test.ss.i1" 
                       && (*v)->type() == dods_array_c
                       && (*v++)->var()->type() == dods_int32_c);
        CPPUNIT_ASSERT(v == vars.end());

        for_each(vars.begin(), vars.end(), delete_basetype);
    }
    
    // with a grid
    void flatten_test4() {
        Byte *b = factory->NewByte("b1");
        s->add_var(b);
        delete  b; b = 0;
        Grid *g = factory->NewGrid("g");
        Int32 *i1 = factory->NewInt32("i1");
        Array *a = factory->NewArray("a", i1);
        delete i1; i1 = 0;
        a->append_dim(100);
        Float64 *f1 = factory->NewFloat64("f1");
        Array *m = factory->NewArray("m", f1);
        delete f1; f1 = 0;
        a->append_dim(100);
        g->add_var(a, array);
        g->add_var(m, maps);
        s->add_var(g);
        delete a; a = 0; delete m; m = 0; delete g; g = 0;
        DBG(cerr << "Before flatten: "; s->print_decl(cerr, ""));
        Constructor::Vars_iter i = s->var_begin();
        CPPUNIT_ASSERT((*i)->name() == "b1" && (*i++)->type() == dods_byte_c);
        CPPUNIT_ASSERT((*i)->name() == "g" && (*i++)->type() == dods_grid_c);
        CPPUNIT_ASSERT(i == s->var_end());
        
        VarList vars = s->flatten(*cp, "");
        VarListIter v = vars.begin();
        ++v;
        DBG(cerr << "flattened var: "; (*v)->print_decl(cerr, ""));
        CPPUNIT_ASSERT((*v)->name() == "test.g"
                       && (*v)->type() == dods_array_c
                       && (*v++)->var()->type() == dods_grid_c);
        CPPUNIT_ASSERT(v == vars.end());

        for_each(vars.begin(), vars.end(), delete_basetype);        
    }
    
    // with a sequence
    void flatten_test5() {
        Byte *b = factory->NewByte("b1");
        s->add_var(b);
        delete b; b = 0;
        Int32 *i32 = factory->NewInt32("i1");
        Sequence *ss = factory->NewSequence("ss");
        ss->add_var(i32);
        delete i32; i32 = 0;
        s->add_var(ss);
        delete ss; ss = 0;
        Constructor::Vars_iter i = s->var_begin();
        CPPUNIT_ASSERT((*i)->name() == "b1" && (*i++)->type() == dods_byte_c);
        CPPUNIT_ASSERT((*i)->name() == "ss" && (*i++)->type() == dods_sequence_c);
        CPPUNIT_ASSERT(i == s->var_end());
        
        VarList vars = s->flatten(*cp, "");
        VarListIter v = vars.begin();
        CPPUNIT_ASSERT((*v)->name() == "test.b1" && (*v)->type() == dods_array_c);
        CPPUNIT_ASSERT((*v)->get_attr_table().get_attr("translation") == "\"flatten\"");
        CPPUNIT_ASSERT(s->get_attr_table().get_attr("translation") 
                == "\"Elided inner Sequence named 'ss'\"");
        ++v;
        CPPUNIT_ASSERT(v == vars.end());
        for_each(vars.begin(), vars.end(), delete_basetype);
    }

    // with an array of structures
    void flatten_test6() {
        Byte *b = factory->NewByte("b1");
        s->add_var(b);
        delete b; b = 0;
        Int32 *i32 = factory->NewInt32("i1");
        Structure *ss = factory->NewStructure("a");
        ss->add_var(i32);
        delete i32; i32 = 0;
        Array *a = factory->NewArray("a");
        a->append_dim(200);
        a->add_var(ss);
        delete ss; ss = 0;
        s->add_var(a);
        delete a; a = 0;
        Constructor::Vars_iter i = s->var_begin();
        CPPUNIT_ASSERT((*i)->name() == "b1" && (*i++)->type() == dods_byte_c);
        CPPUNIT_ASSERT((*i)->name() == "a" && (*i)->type() == dods_array_c);
        BaseType *btp = (*i)->var();
        CPPUNIT_ASSERT(btp->name() == "a" && btp->type() == dods_structure_c);
        CPPUNIT_ASSERT(++i == s->var_end());
        
        VarList vars = s->flatten(*cp, "");
        VarListIter v = vars.begin();
        ++v;
        CPPUNIT_ASSERT((*v)->name() == "test.a.i1" && (*v)->type() == dods_array_c);
        CPPUNIT_ASSERT((*v++)->var()->type() == dods_int32_c);
        CPPUNIT_ASSERT(v == vars.end());

        for_each(vars.begin(), vars.end(), delete_basetype);
    }
    
    // with an array of sequences
    void flatten_test7() {
        Byte *b = factory->NewByte("b1");
        s->add_var(b);
        delete b; b = 0;
        Int32 *i32 = factory->NewInt32("i1");
        Sequence *ss = factory->NewSequence("as");
        ss->add_var(i32);
        delete i32; i32 = 0;
        Array *a = factory->NewArray("as", ss);
        delete ss; ss = 0;
        a->append_dim(200);
        s->add_var(a);
        delete a; a = 0;
        DBG(cerr << "Before flatten: "; s->print_decl(cerr, ""));
        Constructor::Vars_iter i = s->var_begin();
        CPPUNIT_ASSERT((*i)->name() == "b1" && (*i++)->type() == dods_byte_c);
        CPPUNIT_ASSERT((*i)->name() == "as" && (*i++)->type() == dods_array_c);
        CPPUNIT_ASSERT(i == s->var_end());
        
        VarList vars = s->flatten(*cp, "");
        VarListIter v = vars.begin();
        CPPUNIT_ASSERT((*v)->name() == "test.b1" && (*v)->type() == dods_array_c);

        CPPUNIT_ASSERT((*v)->get_attr_table().get_attr("translation") == "\"flatten\"");

        DBG(cerr << "Message: " << s->get_attr_table().get_attr("translation")
            << endl);
        CPPUNIT_ASSERT(s->get_attr_table().get_attr("translation") 
                == "\"Elided inner Sequence named 'as'\"");
        
        ++v;
#if 0
        DBG(cerr << "flattened var: "; (*v)->print_decl(cerr, ""));
        CPPUNIT_ASSERT((*v)->name() == "test.as.i1" && (*v)->type() == dods_array_c);
        CPPUNIT_ASSERT((*v++)->var()->type() == dods_int32_c);
#endif
        CPPUNIT_ASSERT(v == vars.end());

        for_each(vars.begin(), vars.end(), delete_basetype);
    }
    
};

CPPUNIT_TEST_SUITE_REGISTRATION(NCSequenceTest);

int 
main( int argc, char* argv[] )
{
    CppUnit::TextTestRunner runner;
    runner.addTest( CppUnit::TestFactoryRegistry::getRegistry().makeTest() );

    runner.run();

    return 0;
}

