#include "ruby.h"
#include "narray.h"
#include "nusdas.h"
#include "nwpl_capi.h"


static VALUE rb_cDate;


static VALUE rb_eNuSDaSError;
static VALUE rb_eNomemError;
static VALUE rb_eEmptytypeError;
static VALUE rb_eBadpathcharError;
static VALUE rb_eBadtypeError;
static VALUE rb_eNotypetabError;
static VALUE rb_eMaxfilesError;
static VALUE rb_eNodefError;
static VALUE rb_eDefsizeError;
static VALUE rb_eDefemapError;
static VALUE rb_eEmptytype1Error;
static VALUE rb_eEmptytype2Error;
static VALUE rb_eNotuniqueError;
static VALUE rb_eBadQueryError;
static VALUE rb_eNodataError;
static VALUE rb_eBufshortError;
static VALUE rb_eInternalError;
static VALUE rb_eUnknownelemError;


#define max(a,b) (a < b ? b : a)

#define GetCharUndef(name, len) \
  c##name[len] = '\0'; \
  str_fixlen(StringValueCStr(name), c##name, len);

#define GetChar(name, len) \
  char c##name[len+1]; \
  GetCharUndef(name, len);

#define GetType1 GetChar(type1, 8)
#define GetType2 GetChar(type2, 4)
#define GetType3 GetChar(type3, 4)
#define GetTypes \
  GetType1; \
  GetType2; \
  GetType3;

#define GetBasetime \
  N_SI4 cbasetime = get_ctime(basetime);
#define GetValidtime \
  N_SI4 cvalidtime = get_ctime(validtime);
#define GetMember GetChar(member, 4)
#define GetTimesAndMember \
  GetBasetime; \
  GetMember; \
  GetValidtime;

#define GetPlane GetChar(plane, 6)
#define GetElement GetChar(element, 6)
#define GetFmt GetChar(fmt, 2)

#define GetFullParams \
  GetTypes; \
  GetTimesAndMember; \
  GetPlane; \
  GetElement;


#define CreateNArray(fmt, rank, shape) \
  VALUE ary; \
  struct NARRAY *cary; \
  ary = na_make_object(nusfmt_to_natype(fmt), rank, shape, cNArray);	\
  GetNArray(ary, cary);


static void str_fixlen(char* from, char* to, int len)
{
  int i, n;
  n = strlen(from);
  for(i=0;i<max(n,len);i++){
    to[i] = from[i];
  }
  for(i=n;i<len;i++){
    to[i] = ' ';
  }
}

static int nusfmt_to_natype(char* fmt)
{
  if (strcmp(fmt,"I1")==0)
    return NA_BYTE;
  else if (strcmp(fmt,"I2")==0)
    return NA_SINT;
  else if (strcmp(fmt,"I4")==0)
    return NA_LINT;
  else if (strcmp(fmt,"R4")==0)
    return NA_SFLOAT;
  else if (strcmp(fmt,"R8")==0)
    return NA_DFLOAT;
  else
    rb_raise(rb_eArgError, "format is invalid");
  return -1;
}
static char* natype_to_nusfmt(int type)
{
  switch (type) {
  case NA_BYTE:
    return "I1";
  case NA_SINT:
    return "I2";
  case NA_LINT:
    return "I4";
  case NA_SFLOAT:
    return "R4";
  case NA_DFLOAT:
    return "R8";
  default:
    rb_raise(rb_eArgError, "type is invalid");
  }
  return NULL;
}

static N_SI4 get_ctime(VALUE obj)
{
  if (rb_obj_is_kind_of(obj, rb_cNumeric)){
    return (N_SI4)NUM2INT(obj);
  } else if (rb_class_of(obj)==rb_cTime || rb_class_of(obj)==rb_cDate) {
    int year = NUM2INT( rb_funcall(obj, rb_intern("year"), 0) );
    int month = NUM2INT( rb_funcall(obj, rb_intern("month"), 0) );
    int day = NUM2INT( rb_funcall(obj, rb_intern("day"), 0) );
    if (rb_class_of(obj)==rb_cTime) {
      int hour = NUM2INT( rb_funcall(obj, rb_intern("hour"), 0) );
      int min = NUM2INT( rb_funcall(obj, rb_intern("min"), 0) );
      return nwp_ymdhm2seq(year, month, day, hour, min);
    } else {
      return nwp_ymdhm2seq(year, month, day, 0, 0);
    }
  } else
    rb_raise(rb_eArgError, "date must be Numeric, Time or Date");

  return -1;
}

static int get_shape(char* type1, char* type2, char* type3,
		N_SI4 basetime, char* member, N_SI4 validtime,
		char* plane, char* element,
		N_SI4* shape)
{
  int rank = 2;
  int code;
  code = nusdas_inq_data(type1, type2, type3,
			 &basetime, member, &validtime,
			 plane, element,
			 N_GRID_SIZE, shape, &rank);
  if (code == 2)
      return 0;
  else if (code > 0)
    rb_raise(rb_eRuntimeError, "bug: nusdas_inq_data, code=%d", code);
  else if (code == -1)
    rb_raise(rb_eRuntimeError, "bug: nusdas_inq_data, code=%d", code);
  else if (code == -2)
    rb_raise(rb_eRuntimeError, "bug: nusdas_inq_data, code=%d", code);
  else if (code == -3)
    rb_raise(rb_eRuntimeError, "invalide request: nusdas_inq_data, code=%d", code);
  else
    rb_raise(rb_eRuntimeError, "failed: nusdas_inq_data, code=%d", code);
}



/*
 * read data
 *
 * read(type1, type2, type3, basetime, member, validtime, plane, element)
 *  arguments:
 *   type1, type2, type3, member, plane, element: String
 *   basetime, validtime: Integer (minuits from 00:00 1 Jan 1980)
 *  return:
 *   NArray
 *
 * e.g.
 * NuSDaS.read('_GSMGSLY', 'AASV', 'STD1', 108834840, ' ', 108835200, 'SURF', 'TSC')
 */
VALUE
rb_read(VALUE self,
	VALUE type1, VALUE type2, VALUE type3,
	VALUE basetime, VALUE member, VALUE validtime,
	VALUE plane, VALUE element,
	VALUE fmt)
{
  GetFullParams;
  GetFmt;

  int shape[2];
  N_SI4 len;
  int code;

  get_shape(ctype1, ctype2, ctype3,
	   cbasetime, cmember, cvalidtime,
	   cplane, celement,
	   shape);
  CreateNArray(cfmt, 2, shape);

  len = shape[0]*shape[1];
  code = nusdas_read(ctype1, ctype2, ctype3,
		     &cbasetime, cmember, &cvalidtime,
		     cplane, celement,
		     cary->ptr, cfmt, &len);

  if (code == len)
    return ary;
  else if (code > 0)
    rb_raise(rb_eRuntimeError, "only %d/%d data were read", code, len);
  else if (code == 0)
    rb_raise(rb_eRuntimeError, "data have not written");
  else if (code == -2)
    rb_raise(rb_eRuntimeError, "data are inhibited");
  else if (code == -4)
    rb_raise(rb_eRuntimeError, "bug");
  else if (code == -5)
    rb_raise(rb_eRuntimeError, "format is invalid");
  else
    rb_raise(rb_eRuntimeError, "failed: code=%d", code);

  return Qnil;
}

/*
 * write data
 *
 * write(type1, type2, type3, basetime, member, validtime, plane, element, data)
 *  arguments:
 *   type1, type2, type3, member, plane, element: String
 *   basetime, validtime: Integer (minuits from 00:00 1 Jan 1801)
 *   data: NArray
 *  return:
 *   nil
 */
VALUE
rb_write(VALUE self,
	 VALUE type1, VALUE type2, VALUE type3,
	 VALUE basetime, VALUE member, VALUE validtime,
	 VALUE plane, VALUE element,
	 VALUE data)
{
  GetFullParams;

  struct NARRAY *cary;
  N_SI4 len;
  int code;
  char *fmt;

  if (!IsNArray(data))
    rb_raise(rb_eArgError, "data must be NArray");
  if (NA_RANK(data) != 2)
    rb_raise(rb_eArgError, "rank of data must be 2");
  GetNArray(data, cary);
  len = NA_TOTAL(data);

  fmt = natype_to_nusfmt(NA_TYPE(data));
  code = nusdas_write(ctype1, ctype2, ctype3,
		      &cbasetime, cmember, &cvalidtime,
		      cplane, celement,
		      cary->ptr, fmt, &len);
  if (code == len)
    return Qtrue;
  else if (code > 0)
    rb_raise(rb_eRuntimeError, "only %d/%d data were written", code, len);
  else if (code == -1)
    rb_raise(rb_eRuntimeError, "either member, plane, or element is invalid");
  else if (code == -2)
    rb_raise(rb_eRuntimeError, "writing is inhibited");
  else if (code == -3)
    rb_raise(rb_eRuntimeError, "data is too short");
  else if (code == -4)
    rb_raise(rb_eRuntimeError, "narray type is not adapted");
  else if (code == -5)
    rb_raise(rb_eRuntimeError, "data record length exceeded the fixd record length");
  else if (code == -6)
    rb_raise(rb_eRuntimeError, "the missing method and RLEN comprssion cannot be used togeter");
  else if (code == -7)
    rb_raise(rb_eRuntimeError, "encode error");

  return Qnil;
}


VALUE
rb_iocntl(VALUE self,
	  VALUE param, VALUE value)
{
  rb_raise(rb_eRuntimeError, "sorry, this method has not defined yet");
  return Qnil;
}

/*
 * close files
 *
 * allfile_close(param)
 *  arguments:
 *   param: Integer
 *          NuSDaS::N_FOPEN_READ  : files opened for read
 *          NuSDaS::N_FOPEN_WRITE : files opened for write
 *          NuSDaS::N_FOPEN_ALL   : all files
 *  return: Integer (number of cloesd files) or nil (no files were closed)
 */
VALUE
rb_allfile_close(VALUE self,
		 VALUE param)
{
  N_SI4 code;
  code = nusdas_allfile_close( (N_SI4)NUM2INT(param) );
  if (code > 0)
    return INT2NUM((int)code);
  else if (code == 0)
    return Qnil;
  else
    rb_raise(rb_eRuntimeError, "%d files cannot be closeed", code);
}

/*
VALUE
rb_cut(VALUE self,
       VALUE type1, VALUE type2, VALUE type3,
       VALUE basetime, VALUE member, VALUE validtime,
       VALUE plane, VALUE element,
       VALUE fmt,
       VALUE ixstart, VALUE ixfinal, VALUE iystart, VALUE iyfinal)
{
  GetFullParams;
  GetFmt;

  N_SI4 cixstart, cixfinal, ciystart, ciyfinal;
  cixstart = (N_SI4)NUM2INT(ixstart);
  cixstart = (N_SI4)NUM2INT(ixfinal);
  ciystart = (N_SI4)NUM2INT(iystart);
  ciystart = (N_SI4)NUM2INT(iyfinal);

  N_SI4 shape[2];
  get_shape(ctype1, ctype2, ctype3,
	   cbasetime, cmember, cvalidtime,
	   cplane, celement,
	   shape);
  CreateNArray(cfmt, 2, shape);

  N_SI4 len = shape[0]*shape[1];
  N_SI4 code = nusdas_cut(ctype1, ctype2, ctype3,
			  &cbasetime, cmember, &cvalidtime,
			  cplane, celement,
			  cary, cfmt, &len,
			  &cixstar, &cixfinal, &ciystart, &ciyfinal);
  if (code == len)
    return ary;
  else if (code > 0)
    rb_raise(rb_eRuntimeError, "only %d/%d data were read", code, len);
  else if (code == 0)
    rb_raise(rb_eRuntimeError, "data have not written");
  else if (code == -2)
    rb_raise(rb_eRuntimeError, "data are inhibited");
  else if (code == -4)
    rb_raise(rb_eRuntimeError, "bug");
  else if (code == -5)
    rb_raise(rb_eRuntimeError, "format is invalid");
  else if (code == -8)
    rb_raise(rb_eRuntimeError, "region is invalid");
  else
    rb_raise(rb_eRuntimeError, "failed: code=%d", code);

  return Qnil;
}

VALUE
rb_cut_raw(VALUE self,
       VALUE type1, VALUE type2, VALUE type3,
       VALUE basetime, VALUE member, VALUE validtime,
       VALUE plane, VALUE element,
       VALUE ixstart, VALUE ixfinal, VALUE iystart, VALUE iyfinal)
{
  return Qnil;
}
*/

/*
 */
VALUE
rb_make_mask(VALUE self,
	     VALUE udata)
{
  rb_raise(rb_eRuntimeError, "sorry, this method has not defined yet");
  return Qnil;
}

VALUE
rb_onefile_close(VALUE self,
		 VALUE type1, VALUE type2, VALUE type3,
		 VALUE basetime, VALUE member, VALUE validtime)
{
  rb_raise(rb_eRuntimeError, "sorry, this method has not defined yet");
  return Qnil;
}

VALUE
rb_parameter_change(VALUE self,
		    VALUE param, VALUE value)
{
  rb_raise(rb_eRuntimeError, "sorry, this method has not defined yet");
  return Qnil;
}

/*
 * access to grid information
 *
 * grid(type1, type2, type3, basetime, member, validtime, "get")
 *  arguments:
 *   type1, type2, type3, member: String
 *   basetime, validtime: Integer (minuits from 00:00 1 Jan 1801)
 *  return:
 *   projection: String
 *   gridsize: Array[2] (nx,ny)
 *   gridinfo: Array[14] (basepoint_x, basepoint_y, basepoint_lat, basepoint_lon, distance_x, distance_y, standard_lat, standard_lon, standard2_lat, standard2_lon, other_lat, other_lon, other2_lat, other2_lon)
 *   value: String
 *
 * grid(type1, type2, type3, basetime, member, validtime, "put", projection, gridsize, gridinfo, value)
 *  arguments:
 *   type1, type2, type3, member: String
 *   basetime, validtime: Integer (minuits from 00:00 1 Jan 1801)
 *   projection: String
 *   gridsize: Array[2] (nx,ny)
 *   gridinfo: Array[14] (basepoint_x, basepoint_y, basepoint_lat, basepoint_lon, distance_x, distance_y, standard_lat, standard_lon, standard2_lat, standard2_lon, other_lat, other_lon, other2_lat, other2_lon)
 *   value: String
 *  return:
 *   nil
 */
VALUE
rb_grid(int argc, VALUE *argv, VALUE self)
{
  VALUE type1, type2, type3;
  VALUE basetime, member, validtime;
  VALUE getput;
  VALUE proj, gridsize, gridinfo, value;

  char cproj[3], cvalue[5];
  N_SI4 cgridsize[2];
  float cgridinfo[14];

  struct NARRAY* nary;
  VALUE* ary;
  int16_t *cary_sint;
  float *cary_float;
  int flag=0;
  int i;
  N_SI4 code;

  rb_scan_args(argc, argv, "74",
	       &type1, &type2, &type3,
	       &basetime, &member, &validtime,
	       &getput,
	       &proj, &gridsize, &gridinfo, &value);

  GetChar(getput,3);
  if (strcmp(cgetput,"get") || strcmp(cgetput,"GET")) {
    if (argc != 7)
      rb_raise(rb_eArgError, "wrong number of arguments for get (%d for 7)", argc);
    flag = 1;
  } else if (strcmp(cgetput,"put") || strcmp(cgetput,"PUT")) {
    if (argc != 11)
      rb_raise(rb_eArgError, "wrong number of arguments for put (%d for 11)", argc);
    if (TYPE(gridsize)==T_ARRAY) {
      if (RARRAY_LEN(gridsize)!=2)
	rb_raise(rb_eArgError, "length of gridsize must be 2");
      ary = RARRAY_PTR(gridsize);
      for (i=0;i<2;i++)
	cgridsize[i] = NUM2INT(ary[i]);
    } else if (IsNArray(gridsize)){
      if (NA_TOTAL(gridsize)!=2)
	rb_raise(rb_eArgError, "length of gridsize must be 2");
      gridsize = na_cast_object(gridsize, NA_SINT);
      GetNArray(gridsize,nary);
      cary_sint = NA_PTR_TYPE(nary,int16_t*);
      for (i=0;i<2;i++)
	cgridsize[i] = (N_SI4)cary_sint[i];
    } else
      rb_raise(rb_eArgError, "gridsize must be NArray of Array");

    if (TYPE(gridinfo)==T_ARRAY) {
      if (RARRAY_LEN(gridinfo)!=14)
	rb_raise(rb_eArgError, "length of gridsize must be 14");
      ary = RARRAY_PTR(gridinfo);
      for (i=0;i<14;i++)
	cgridinfo[i] = (float)NUM2DBL(ary[i]);
    } else if (IsNArray(gridinfo)){
      if (NA_TOTAL(gridinfo)!=14)
	rb_raise(rb_eArgError, "length of gridinfo must be 14");
      gridinfo = na_cast_object(gridinfo, NA_SFLOAT);
      GetNArray(gridinfo,nary);
      cary_float = NA_PTR_TYPE(nary,float*);
      for (i=0;i<14;i++)
	cgridinfo[i] = cary_float[i];
    } else
      rb_raise(rb_eArgError, "gridinfo must be NArray of Array");

    GetCharUndef(proj, 3);
    GetCharUndef(value, 4);
  }

  GetTypes;
  GetTimesAndMember;

  code = nusdas_grid(ctype1, ctype2, ctype3,
		     &cbasetime, cmember, &cvalidtime,
		     cproj, cgridsize, cgridinfo, cvalue,
		     cgetput);

  if (code == -5)
    rb_raise(rb_eRuntimeError, "getput is invalid");
  else if (code != 0)
    rb_raise(rb_eRuntimeError, "failed: code=%d", code);

  if ( flag==1 ) {
    proj = rb_str_new(cproj,4);
    gridsize = rb_ary_new();
    for(i=0;i<2;i++)
      rb_ary_push(gridsize, INT2NUM((int)cgridsize[i]));
    gridinfo = rb_ary_new();
    for(i=0;i<14;i++)
      rb_ary_push(gridinfo, rb_float_new((double)cgridinfo[i]));
    value = rb_str_new(cvalue,4);
    return rb_ary_new3(4, proj, gridsize, gridinfo, value);
  } else
    return Qnil;
}

VALUE
rb_info(int argc, VALUE *argv, VALUE self)
{
  return Qnil;
}

/*
 * inquire control information
 *
 * inq_cntl(type1, type2, type3, basetime, member, validtime, param)
 *  arguments:
 *   type1, type2, type3, member: String
 *   basetime, validtime: Integer (minuits from 00:00 1 Jan 1801)
 *   param: Integer
 *          NuSDaS::N_MEMBER_NUM      : number of members
 *          NuSDaS::N_MEMBER_LIST     : list of members
 *          NuSDaS::N_VALIDTIME_NUM   : number of validtimes
 *          NuSDaS::N_VALIDTIME_LIST  : list of validtime
 *          NuSDaS::N_VALIDTIME_LIST2 : list of validtime2
 *          NuSDaS::N_PLANE_NUM       : number of planes
 *          NuSDaS::N_PLANE_LIST      : list of planes
 *          NuSDaS::N_ELEMENT_NUM     : number of elements
 *          NuSDaS::N_ELEMENT_LIST    : list of elements
 *          NuSDaS::N_PROJECTION      : projection
 *  return:
 *   result
 */
VALUE
rb_inq_cntl(VALUE self,
	    VALUE type1, VALUE type2, VALUE type3,
	    VALUE basetime, VALUE member, VALUE validtime,
	    VALUE param)
{
  GetTypes;
  GetTimesAndMember;

  VALUE data;

  N_SI4 cparam;
  void* cdata;
  N_SI4 cdatasize;

  N_SI4 cparam2, len;
  int flag, i;
  N_SI4 code;

  cparam = (N_SI4)NUM2INT(param);
  cparam2 = 0;
  flag = -1;
  len = 0;
  switch( cparam ) {
  case N_MEMBER_NUM:
  case N_VALIDTIME_NUM:
  case N_PLANE_NUM:
  case N_ELEMENT_NUM:
    cdatasize = 1;
    cdata = xmalloc( sizeof(N_SI4) );
    flag = 1;
    break;
  case N_MEMBER_LIST:
  case N_VALIDTIME_LIST:
  case N_PLANE_LIST:
  case N_ELEMENT_LIST:
    switch( cparam ) {
    case N_MEMBER_LIST:
      cparam2 = N_MEMBER_NUM;
      len = 4;
      flag = 2;
      break;
    case N_VALIDTIME_LIST:
      cparam2 = N_VALIDTIME_NUM;
      len = 4;
      flag = 4;
      break;
    case N_PLANE_LIST:
      cparam2 = N_PLANE_NUM;
      len = 6;
      flag = 2;
      break;
    case N_ELEMENT_LIST:
      cparam2 = N_ELEMENT_NUM;
      len = 6;
      flag = 2;
      break;
    }
    cdatasize = 1;
    cdata = xmalloc(4);
    code = nusdas_inq_cntl(ctype1, ctype2, ctype3,
			   &cbasetime, cmember, &cbasetime,
			   cparam2, cdata, &cdatasize);
    if (code == 1)
      cdatasize = ((N_SI4*)cdata)[0];
    else if (code == -1)
      rb_raise(rb_eRuntimeError, "bug");
    else if (code == -2)
      rb_raise(rb_eRuntimeError, "bug");
    else if (code == -3)
      rb_raise(rb_eRuntimeError, "bug");
    else
      rb_raise(rb_eRuntimeError, "failed: nusdas_inq_cntl, code=%d",code);
    free(cdata);
    cdata = xmalloc( sizeof(char)*len*cdatasize );
    break;
  case N_PROJECTION:
    cdatasize = 1;
    cdata = xmalloc( sizeof(char)*4 );
    flag = 3;
    break;
  default:
    rb_raise(rb_eArgError, "param is invalid");
  }

  code = nusdas_inq_cntl(ctype1, ctype2, ctype3,
			 &cbasetime, cmember, &cvalidtime,
			 cparam, cdata, &cdatasize);
  if (code == -1)
    rb_raise(rb_eRuntimeError, "bug");
  else if (code == -2)
    rb_raise(rb_eRuntimeError, "bug");
  else if (code == -3)
    rb_raise(rb_eRuntimeError, "param is invalid");
  else if (code != cdatasize)
    rb_raise(rb_eRuntimeError, "failed: nusdas_inq_cntl, code=%d",code);
  switch (flag) {
  case 1:
    data = INT2NUM( ((int*)cdata)[0] );
    break;
  case 2:
    data = rb_ary_new();
    for (i=0;i<cdatasize;i++)
      rb_ary_push(data, rb_str_new(((char*)cdata)+i*len, len));
    break;
  case 3:
    data = rb_str_new((char*)cdata, 4);
    break;
  case 4:
    data = rb_ary_new();
    for (i=0;i<cdatasize;i++)
      rb_ary_push(data, INT2NUM((int)(((N_SI4*)cdata)[i])));
    break;
    data = INT2NUM( (int)(((N_SI4*)cdata)[0]) );
    break;
  default:
    data = Qnil;
  }
  free(cdata);
  return data;
}

/*
 * inquire data information
 *
 inq_data(type1, type2, type3, basetime, member, validtime, plane, element, param)
 *  arguments:
 *   type1, type2, type3, member, plane, element: String
 *   basetime, validtime: Integer (minuits from 00:00 1 Jan 1801)
 *   param: Integer
 *          NuSDaS::N_GRID_SIZE
 *          NuSDaS::N_PC_PACKING
 *          NuSDaS::N_MISSING_MODE
 *          NuSDaS::N_MISSING_VALUE
 *  return:
 *   result
 */
VALUE
rb_inq_data(VALUE self,
	    VALUE type1, VALUE type2, VALUE type3,
	    VALUE basetime, VALUE member, VALUE validtime,
	    VALUE plane, VALUE element,
	    VALUE param)
{
  GetFullParams;
  VALUE data;

  N_SI4 cparam, cnelems;
  void* cdata;

  int flag;
  int code;
  int len;

  cparam = (N_SI4)NUM2INT(param);
  switch(cparam) {
  case N_GRID_SIZE:
    cdata = xmalloc(4*2);
    cnelems = 2;
    flag = 1;
    break;
  case N_PC_PACKING:
  case N_MISSING_MODE:
    cdata = xmalloc(4);
    cnelems = 1;
    flag = 2;
    break;
  case N_MISSING_VALUE:
    cdata = xmalloc(4);
    cnelems = 1;
    code = nusdas_inq_data(ctype1, ctype2, ctype3,
			   &cbasetime, cmember, &cvalidtime,
			   cplane, celement,
			   N_PC_PACKING, cdata, &cnelems);
    if ( code == -1 )
      rb_raise(rb_eRuntimeError, "bug");
    else if ( code == -2 )
      rb_raise(rb_eRuntimeError, "bug");
    else if ( code == -3 )
      rb_raise(rb_eRuntimeError, "bug");
    else if ( code != 1 )
      rb_raise(rb_eRuntimeError, "bug");
    if ( strncmp((char*)cdata,"1PAC",4) == 0 ){
      len = 1;
      flag = 3;
    } else if ( strncmp((char*)cdata,"2PAC",4) == 0 ) {
      len = 2;
      flag = 4;
    } else if ( strncmp((char*)cdata,"2UPC",4) == 0 ) {
      len = 2;
      flag = 5;
    } else if ( strncmp((char*)cdata,"4PAC",4) == 0 ) {
      len = 4;
      flag = 6;
    } else if ( strncmp((char*)cdata,"N1I2",4) == 0 ) {
      len = 2;
      flag = 4;
    } else if ( strncmp((char*)cdata,"I1  ",4) == 0 ) {
      len = 1;
      flag = 3;
    } else if ( strncmp((char*)cdata,"I2  ",4) == 0 ) {
      len = 2;
      flag = 4;
    } else if ( strncmp((char*)cdata,"I4  ",4) == 0 ) {
      len = 4;
      flag = 7;
    } else if ( strncmp((char*)cdata,"R4  ",4) == 0 ) {
      len = 4;
      flag = 8;
    } else if ( strncmp((char*)cdata,"R8  ",4) == 0 ) {
      len = 8;
      flag = 9;
    } else {
      len = -1;
      flag = -1;
      rb_raise(rb_eRuntimeError, "bug");
    }
    free(cdata);
    cdata = xmalloc(len);
    cnelems = 1;
    break;
  default:
    flag = -1;
    cdata = NULL;
    rb_raise(rb_eArgError, "param is invalid");
  }

  code = nusdas_inq_data(ctype1, ctype2, ctype3,
			 &cbasetime, cmember, &cvalidtime,
			 cplane, celement,
			 cparam, cdata, &cnelems);
  if ( code == -1 )
    rb_raise(rb_eRuntimeError, "bug");
  else if ( code == -2 )
    rb_raise(rb_eRuntimeError, "bug");
  else if ( code == -3 )
    rb_raise(rb_eRuntimeError, "param is invalid");
  else if ( code != cnelems )
    rb_raise(rb_eRuntimeError, "bug");
  switch (flag) {
  case 1:
    data = rb_ary_new();
    rb_ary_push(data, INT2NUM((int)((N_SI4*)cdata)[0]));
    rb_ary_push(data, INT2NUM((int)((N_SI4*)cdata)[1]));
    break;
  case 2:
    data = rb_str_new((char*)cdata, 4);
    break;
  case 3:
    data = CHR2FIX( ((N_UI1*)cdata)[0] );
    break;
  case 4:
    data = INT2FIX( ((N_SI2*)cdata)[0] );
    break;
  case 5:
    data = UINT2NUM( (unsigned int)(((N_UI2*)cdata)[0]) );
    break;
  case 6:
    data = INT2FIX( ((N_SI4*)cdata)[0] );
    break;
  case 7:
    data = UINT2NUM( (unsigned int)(((N_UI4*)cdata)[0]) );
    break;
  case 8:
    data = rb_float_new( (double)(((float*)cdata)[0]) );
    break;
  case 9:
    data = rb_float_new( ((double*)cdata)[0] );
    break;
  default:
    data = Qnil;
  }

  free(cdata);
  return data;
}

VALUE
rb_inq_def(VALUE self,
	   VALUE type1, VALUE type2, VALUE type3,
	   VALUE param)
{
  rb_raise(rb_eRuntimeError, "sorry, this method has not defined yet");
  return Qnil;
}

/*
 * inquire list of basetimes
 *
 * inq_nrdbtime(type1, type2, type3, pflag=false)
 *  arguments:
 *   type1, type2, type3: String
 *   pflag: true or false, (if true, informations are printed out)
 *  return:
 *   btlist: Array
 */
VALUE
rb_inq_nrdbtime(int argc, VALUE *argv, VALUE self)
{
  VALUE type1, type2, type3;
  VALUE pflag;

  VALUE btlist;

  N_SI4 *cbtlist, cbtlistsize;
  N_SI4 cpflag;

  N_SI4 code;
  int i;

  rb_scan_args(argc, argv, "31", &type1, &type2, &type3, &pflag);
  GetTypes;
  cpflag = (pflag==Qtrue ? 1 : 0) ;

  cbtlistsize = 1;
  cbtlist = xmalloc(4);
  code = nusdas_inq_nrdbtime(ctype1, ctype2, ctype3,
			     cbtlist, &cbtlistsize, cpflag);
  if ( code == 1 )
    return rb_ary_new3(1, INT2NUM( (int)(cbtlist[0]) ) );
  else if ( code == -1 )
    rb_raise(rb_eRuntimeError, "file IO error");
  else if ( code == -2 )
    rb_raise(rb_eRuntimeError, "control part does not exit");
  else if ( code == -3 )
    rb_raise(rb_eRuntimeError, "record length is invalid");
  else if ( code == -4 )
    rb_raise(rb_eRuntimeError, "failed to open file or directory");
  else if ( code < 0 )
    rb_raise(rb_eRuntimeError, "failed");
  free(cbtlist);

  cbtlistsize = code;
  cbtlist = xmalloc(4*cbtlistsize);
  code = nusdas_inq_nrdbtime(ctype1, ctype2, ctype3,
			     cbtlist, &cbtlistsize, cpflag);
  if (code != cbtlistsize)
    rb_raise(rb_eRuntimeError, "bug");

  btlist = rb_ary_new();
  for (i=0;i<cbtlistsize;i++)
    rb_ary_push(btlist, INT2NUM((int)(cbtlist[i])));

  return btlist;
}

/*
 * inquire list of validtime
 *
 * inq_nrdvtime(type1, type2, type3, basetime=-1, pflag=false)
 *  arguments:
 *   type1, type2, type3: String
 *   basetime: Integer (minuits from 00:00 1 Jan 1801) if basetime=-1, returns validtimes for all basetime
 *   pflag: true or false, (if true, informations are printed out)
 *  return:
 *   vtlist: Array
 */
VALUE
rb_inq_nrdvtime(int argc, VALUE *argv, VALUE self)
{
  VALUE type1, type2, type3;
  VALUE basetime;
  VALUE pflag;

  VALUE vtlist;

  N_SI4 *cvtlist, cvtlistsize;
  N_SI4 cbasetime;
  N_SI4 cpflag;

  N_SI4 code;
  int i;


  rb_scan_args(argc, argv, "32", &type1, &type2, &type3, &basetime, &pflag);
  GetTypes;
  if (basetime==Qnil)
    cbasetime = -1;
  else
    cbasetime = get_ctime(basetime);
  cpflag = (pflag==Qtrue ? 1 : 0) ;

  cvtlistsize = 1;
  cvtlist = xmalloc(4);
  code = nusdas_inq_nrdvtime(ctype1, ctype2, ctype3,
			     cvtlist, &cvtlistsize,
			     &cbasetime, cpflag);
  if ( code == 1 )
    return INT2NUM( (int)(cvtlist[0]) );
  else if ( code < 0 )
    rb_raise(rb_eRuntimeError, "failed");
  free(cvtlist);

  cvtlistsize = code;
  cvtlist = xmalloc(4*cvtlistsize);
  code = nusdas_inq_nrdvtime(ctype1, ctype2, ctype3,
			     cvtlist, &cvtlistsize,
			     &cbasetime, cpflag);
  if (code != cvtlistsize)
    rb_raise(rb_eRuntimeError, "bug");

  vtlist = rb_ary_new();
  for (i=0;i<cvtlistsize;i++)
    rb_ary_push(vtlist, INT2NUM((int)(cvtlist[i])));

  return vtlist;
}

/*
 * inquire number of vertical levels for ETA, SIGM, and ZHYB coordinats
 *
 * subc_eta_inq_zn(type1, type2, type3, basetime, member, validtime, group)
 *  arguments:
 *   type1, type2, type3, member: String
 *   basetime, validtime: Integer (minuits from 00:00 1 Jan 1801)
 *   group: "ETA", "SIGM", or "ZHYB"
 *  return:
 *    j_n: Integer
 */
VALUE
rb_subc_eta_inq_nz(VALUE self,
		   VALUE type1, VALUE type2, VALUE type3,
		   VALUE basetime, VALUE member, VALUE validtime,
		   VALUE group)
{
  GetTypes;
  GetTimesAndMember;
  GetChar(group, 4);

  N_SI4 cn_levels;
  N_SI4 code;

  code = nusdas_subc_eta_inq_nz(ctype1, ctype2, ctype3,
				&cbasetime, cmember, &cvalidtime,
				cgroup, &cn_levels);
  if ( code < 0 )
    rb_raise(rb_eRuntimeError, "failed");

  return INT2NUM( (int)cn_levels );
}

/*
 * inquire number of south-north grid for RGAU grid
 *
 * subc_rgau_inq_jn(type1, type2, type3, basetime, member, validtime)
 *  arguments:
 *   type1, type2, type3, member: String
 *   basetime, validtime: Integer (minuits from 00:00 1 Jan 1801)
 *  return:
 *    j_n: Integer
 */
VALUE
rb_subc_rgau_inq_jn(VALUE self,
		    VALUE type1, VALUE type2, VALUE type3,
		    VALUE basetime, VALUE member, VALUE validtime)
{
  GetTypes;
  GetTimesAndMember;

  N_SI4 cj_n;
  N_SI4 code;

  code = nusdas_subc_rgau_inq_jn(ctype1, ctype2, ctype3,
				 &cbasetime, cmember, &cvalidtime,
				 &cj_n);
  if ( code == -2 )
    rb_raise(rb_eRuntimeError, "RGAU does not exits");
  else if ( code == -3 )
    rb_raise(rb_eRuntimeError, "record size is invalid");
  else if ( code < 0 )
    rb_raise(rb_eRuntimeError, "failed");

  return INT2NUM( (int)cj_n );
}

VALUE
rb_subc_delt(int argc, VALUE *argv, VALUE self)
{
  rb_raise(rb_eRuntimeError, "sorry, this method has not defined yet");
  return Qnil;
}

VALUE
rb_subc_delt_preset1(VALUE self,
		     VALUE type1, VALUE type2, VALUE type3,
		     VALUE delt)
{
  rb_raise(rb_eRuntimeError, "sorry, this method has not defined yet");
  return Qnil;
}

/*
 * access to parameters of ETA coordinate
 *
 * subc_eta(type1, type2, type3, basetime, member, validtime, "get")
 *  arguments:
 *   type1, type2, type3, member: String
 *   basetime, validtime: Integer (minuits from 00:00 1 Jan 1801)
 *  return:
 *   n_levels: Integer (number of levels)
 *   a, b: NArray[n_levels+1]
 *   c: Float
 *
 * subc_eta(type1, type2, type3, basetime, member, validtime, "put", a, b, c)
 *  arguments:
 *   type1, type2, type3, member: String
 *   basetime, validtime: Integer (minuits from 00:00 1 Jan 1801)
 *   a, b: NArray[n_levels+1] || Array[n_levels+1]
 *   c: Float
 *  return:
 *   nil
 */
VALUE
rb_subc_eta(int argc, VALUE *argv, VALUE self)
{
  VALUE type1, type2, type3;
  VALUE basetime, member, validtime;
  VALUE getput;
  VALUE a, b, c;
  VALUE obj;

  N_SI4 cn_levels;
  float *ca=NULL, *cb=NULL, cc;

  VALUE* ary;
  struct NARRAY* nary;
  float *cary;
  N_SI4 code;
  int flag=-1;

  int i;

  rb_scan_args(argc, argv, "73",
	       &type1, &type2, &type3,
	       &basetime, &member, &validtime,
	       &getput,
	       &a, &b, &c);

  GetTypes;
  GetTimesAndMember;

  GetChar(getput,3);

  if (strcmp(cgetput,"get") || strcmp(cgetput,"GET")) {
    if (argc != 7)
      rb_raise(rb_eArgError, "wrong number of arguments for get (%d for 7)", argc);
    code = nusdas_subc_eta_inq_nz(ctype1, ctype2, ctype3,
				  &cbasetime, cmember, &cvalidtime,
				  "ETA ", &cn_levels);
    if (code <= 0)
      rb_raise(rb_eRuntimeError, "failed: nusdas_subc_eta_inq_nz, code=%d", code);
    ca = xmalloc( sizeof(float)*(cn_levels+1) );
    cb = xmalloc( sizeof(float)*(cn_levels+1) );
    flag = 1;
  } else if (strcmp(cgetput,"put") || strcmp(cgetput,"PUT")) {
    if (argc != 10)
      rb_raise(rb_eArgError, "wrong number of arguments for put (%d for 11)", argc);

    if (TYPE(a)==T_ARRAY) {
      cn_levels = RARRAY_LEN(a)-1;
      ca = xmalloc( sizeof(float)*(cn_levels+1) );
      ary = RARRAY_PTR(a);
      for (i=0;i<cn_levels+1;i++)
	ca[i] = (float)NUM2DBL(ary[i]);
    } else if (IsNArray(a)){
      cn_levels = NA_TOTAL(a);
      ca = xmalloc( sizeof(float)*(cn_levels+1) );
      b = na_cast_object(b, NA_SFLOAT);
      GetNArray(b,nary);
      cary = NA_PTR_TYPE(nary,float*);
      for (i=0;i<cn_levels+1;i++)
	ca[i] = cary[i];
    } else
      rb_raise(rb_eArgError, "a must be NArray of Array");

    if (TYPE(b)==T_ARRAY) {
      if (RARRAY_LEN(b)!=cn_levels+1)
	rb_raise(rb_eArgError, "length of b must be the same as a");
      cb = xmalloc( sizeof(float)*(cn_levels+1) );
      ary = RARRAY_PTR(b);
      for (i=0;i<cn_levels+1;i++)
	cb[i] = (float)NUM2DBL(ary[i]);
    } else if (IsNArray(b)){
      if (NA_TOTAL(b)!=cn_levels+1)
	rb_raise(rb_eArgError, "length of b must be the same as a");
      cb = xmalloc( sizeof(float)*(cn_levels+1) );
      b = na_cast_object(b, NA_SFLOAT);
      GetNArray(b,nary);
      cary = NA_PTR_TYPE(nary,float*);
      for (i=0;i<cn_levels+1;i++)
	cb[i] = cary[i];
    } else
      rb_raise(rb_eArgError, "b must be NArray of Array");

    cc = NUM2DBL(c);
    flag = 2;
  }

  nusdas_subc_eta(ctype1, ctype2, ctype3,
		  &cbasetime, cmember, &cvalidtime,
		  &cn_levels, ca, cb, &cc,
		  cgetput);

  switch (flag) {
  case 1:
    a = rb_ary_new();
    b = rb_ary_new();
    for (i=0;i<cn_levels+1;i++) {
      rb_ary_push(a, rb_float_new((double)ca[i]) );
      rb_ary_push(b, rb_float_new((double)cb[i]) );
    }
    obj = rb_ary_new();
    rb_ary_push(obj, INT2NUM((int)cn_levels));
    rb_ary_push(obj, a);
    rb_ary_push(obj, b);
    rb_ary_push(obj, rb_float_new((double)cc) );
    break;
  case 2:
    obj = Qnil;
    break;
  default:
    rb_raise(rb_eRuntimeError, "failed");
    exit(-1);
  }

  free(ca);
  free(cb);
  return obj;
}

VALUE
rb_subc_preset1(int argc, VALUE *argv, VALUE self)
{
  rb_raise(rb_eRuntimeError, "sorry, this method has not defined yet");
  return Qnil;
}

VALUE
rb_subc_rgau(int argc, VALUE *argv, VALUE self)
{
  rb_raise(rb_eRuntimeError, "sorry, this method has not defined yet");
  return Qnil;
}

VALUE
rb_subc_rgau_preset1(VALUE self,
		     VALUE type1, VALUE type2, VALUE type3,
		     VALUE j, VALUE j_start, VALUE j_n,
		     VALUE i, VALUE i_start, VALUE i_n,
		     VALUE lat)
{
  rb_raise(rb_eRuntimeError, "sorry, this method has not defined yet");
  return Qnil;
}

/*
 * access to parameters for sigma coordinate
 *
 * subc_sigm(type1, type2, type3, basetime, member, validtime, "get")
 *  arguments:
 *   type1, type2, type3, member: String
 *   basetime, validtime: Integer (minuits from 00:00 1 Jan 1801)
 *  return:
 *   n_levels: Integer (number of levels)
 *   a, b: NArray[n_levels+1]
 *   c: Float
 *
 * subc_sigm(type1, type2, type3, basetime, member, validtime, "put", a, b, c)
 *  arguments:
 *   type1, type2, type3, member: String
 *   basetime, validtime: Integer (minuits from 00:00 1 Jan 1801)
 *   a, b: NArray[n_levels+1] || Array[n_levels+1]
 *   c: Float
 *  return:
 *   nil
 */
VALUE
rb_subc_sigm(int argc, VALUE *argv, VALUE self)
{
  VALUE type1, type2, type3;
  VALUE basetime, member, validtime;
  VALUE getput;
  VALUE a, b, c;
  VALUE obj;

  N_SI4 cn_levels;
  float *ca=NULL, *cb=NULL, cc;

  VALUE* ary;
  struct NARRAY* nary;
  float *cary;
  N_SI4 code;
  int flag=-1;

  int i;

  rb_scan_args(argc, argv, "73",
	       &type1, &type2, &type3,
	       &basetime, &member, &validtime,
	       &getput,
	       &a, &b, &c);

  GetTypes;
  GetTimesAndMember;

  GetChar(getput,3);

  if (strcmp(cgetput,"get") || strcmp(cgetput,"GET")) {
    if (argc != 7)
      rb_raise(rb_eArgError, "wrong number of arguments for get (%d for 7)", argc);
    code = nusdas_subc_eta_inq_nz(ctype1, ctype2, ctype3,
				  &cbasetime, cmember, &cvalidtime,
				  "SIGM", &cn_levels);
    if (code <= 0)
      rb_raise(rb_eRuntimeError, "failed: nusdas_subc_eta_inq_nz, code=%d", code);
    ca = xmalloc( sizeof(float)*(cn_levels+1) );
    cb = xmalloc( sizeof(float)*(cn_levels+1) );
    flag = 1;
  } else if (strcmp(cgetput,"put") || strcmp(cgetput,"PUT")) {
    if (argc != 10)
      rb_raise(rb_eArgError, "wrong number of arguments for put (%d for 11)", argc);

    if (TYPE(a)==T_ARRAY) {
      cn_levels = RARRAY_LEN(a)-1;
      ca = xmalloc( sizeof(float)*(cn_levels+1) );
      ary = RARRAY_PTR(a);
      for (i=0;i<cn_levels+1;i++)
	ca[i] = (float)NUM2DBL(ary[i]);
    } else if (IsNArray(a)){
      cn_levels = NA_TOTAL(a);
      ca = xmalloc( sizeof(float)*(cn_levels+1) );
      b = na_cast_object(b, NA_SFLOAT);
      GetNArray(b,nary);
      cary = NA_PTR_TYPE(nary,float*);
      for (i=0;i<cn_levels+1;i++)
	ca[i] = cary[i];
    } else
      rb_raise(rb_eArgError, "a must be NArray of Array");

    if (TYPE(b)==T_ARRAY) {
      if (RARRAY_LEN(b)!=cn_levels+1)
	rb_raise(rb_eArgError, "length of b must be the same as a");
      cb = xmalloc( sizeof(float)*(cn_levels+1) );
      ary = RARRAY_PTR(b);
      for (i=0;i<cn_levels+1;i++)
	cb[i] = (float)NUM2DBL(ary[i]);
    } else if (IsNArray(b)){
      if (NA_TOTAL(b)!=cn_levels+1)
	rb_raise(rb_eArgError, "length of b must be the same as a");
      cb = xmalloc( sizeof(float)*(cn_levels+1) );
      b = na_cast_object(b, NA_SFLOAT);
      GetNArray(b,nary);
      cary = NA_PTR_TYPE(nary,float*);
      for (i=0;i<cn_levels+1;i++)
	cb[i] = cary[i];
    } else
      rb_raise(rb_eArgError, "b must be NArray of Array");

    cc = NUM2DBL(c);
    flag = 2;
  }

  nusdas_subc_sigm(ctype1, ctype2, ctype3,
		   &cbasetime, cmember, &cvalidtime,
		   &cn_levels, ca, cb, &cc,
		   cgetput);

  switch (flag) {
  case 1:
    a = rb_ary_new();
    b = rb_ary_new();
    for (i=0;i<cn_levels+1;i++) {
      rb_ary_push(a, rb_float_new((double)ca[i]) );
      rb_ary_push(b, rb_float_new((double)cb[i]) );
    }
    obj = rb_ary_new();
    rb_ary_push(obj, INT2NUM((int)cn_levels));
    rb_ary_push(obj, a);
    rb_ary_push(obj, b);
    rb_ary_push(obj, rb_float_new((double)cc) );
    break;
  case 2:
    obj = Qnil;
    break;
  default:
    rb_raise(rb_eRuntimeError, "failed");
    exit(-1);
  }

  free(ca);
  free(cb);
  return obj;
}

VALUE
rb_subc_srf(int argc, VALUE *argv, VALUE self)
{
  rb_raise(rb_eRuntimeError, "sorry, this method has not defined yet");
  return Qnil;
}

VALUE
rb_subc_tdif(int argc, VALUE *argv, VALUE self)
{
  rb_raise(rb_eRuntimeError, "sorry, this method has not defined yet");
  return Qnil;
}

VALUE
rb_subc_zhyb(int argc, VALUE *argv, VALUE self)
{
  rb_raise(rb_eRuntimeError, "sorry, this method has not defined yet");
  return Qnil;
}

VALUE
rb_subc_zhyb_preset1(VALUE self,
		     VALUE type1, VALUE type2, VALUE type3,
		     VALUE nz, VALUE ptrf, VALUE presrf,
		     VALUE zrp, VALUE zrw,
		     VALUE vctrans_p, VALUE vctrans_w,
		     VALUE dvtrans_p, VALUE dvtrans_w)
{
  rb_raise(rb_eRuntimeError, "sorry, this method has not defined yet");
  return Qnil;
}

/*
 * Wraper of NuSDaS C library - handling NuSDaS data -
 *
 * methods are defined as class method
 */
void Init_nusdas_wrap()
{

  VALUE rb_mNumRu;
  VALUE rb_cNuSDaS;

  rb_require("narray");
  rb_require("date");

  rb_mNumRu = rb_define_module("NumRu");
  rb_cNuSDaS = rb_define_class_under(rb_mNumRu, "NuSDaS", rb_cObject);

  rb_cDate = rb_eval_string("Date");


  rb_define_const(rb_cNuSDaS, "N_NUSDAS_VERSION", INT2FIX(N_NUSDAS_VERSION));

  rb_define_const(rb_cNuSDaS, "N_ON", INT2FIX(N_ON));
  rb_define_const(rb_cNuSDaS, "N_OFF", INT2FIX(N_OFF));

  /* code for nusdas_allfile_close */
  rb_define_const(rb_cNuSDaS, "N_FOPEN_NOT", INT2FIX(N_FOPEN_NOT));
  rb_define_const(rb_cNuSDaS, "N_FOPEN_READ", INT2FIX(N_FOPEN_READ));
  rb_define_const(rb_cNuSDaS, "N_FOPEN_WRITE", INT2FIX(N_FOPEN_WRITE));
  rb_define_const(rb_cNuSDaS, "N_FOPEN_ALL", INT2FIX(N_FOPEN_ALL));

  /* code for nusdas_parameter_change */
  rb_define_const(rb_cNuSDaS, "N_PC_MISSING_UI1", INT2FIX(N_PC_MISSING_UI1));
  rb_define_const(rb_cNuSDaS, "N_PC_MISSING_SI2", INT2FIX(N_PC_MISSING_SI2));
  rb_define_const(rb_cNuSDaS, "N_PC_MISSING_SI4", INT2FIX(N_PC_MISSING_SI4));
  rb_define_const(rb_cNuSDaS, "N_PC_MISSING_R4", INT2FIX(N_PC_MISSING_R4));
  rb_define_const(rb_cNuSDaS, "N_PC_MISSING_R8", INT2FIX(N_PC_MISSING_R8));
  rb_define_const(rb_cNuSDaS, "N_PC_MASK_BIT", INT2FIX(N_PC_MASK_BIT));
  rb_define_const(rb_cNuSDaS, "N_PC_ID_SET", INT2FIX(N_PC_ID_SET));
  rb_define_const(rb_cNuSDaS, "N_PC_PACKING", INT2FIX(N_PC_PACKING));
  rb_define_const(rb_cNuSDaS, "N_PC_SIZEX", INT2FIX(N_PC_SIZEX));
  rb_define_const(rb_cNuSDaS, "N_PC_SIZEY", INT2FIX(N_PC_SIZEY));
  rb_define_const(rb_cNuSDaS, "N_PC_PANDORA", INT2FIX(N_PC_PANDORA));
  rb_define_const(rb_cNuSDaS, "N_PC_PANDORA_I1", INT2FIX(N_PC_PANDORA_I1));

  /* code for nusdas_iocntl */
  rb_define_const(rb_cNuSDaS, "N_IO_MARK_END", INT2FIX(N_IO_MARK_END));
  rb_define_const(rb_cNuSDaS, "N_IO_W_FCLOSE", INT2FIX(N_IO_W_FCLOSE));
  rb_define_const(rb_cNuSDaS, "N_IO_WARNING_OUT", INT2FIX(N_IO_WARNING_OUT));
  rb_define_const(rb_cNuSDaS, "N_IO_R_FCLOSE", INT2FIX(N_IO_R_FCLOSE));

  rb_define_const(rb_cNuSDaS, "N_IO_PUT", rb_str_new2(N_IO_PUT));
  rb_define_const(rb_cNuSDaS, "N_IO_GET", rb_str_new2(N_IO_GET));
  rb_define_const(rb_cNuSDaS, "N_MV_UI1", INT2FIX(N_MV_UI1));
  rb_define_const(rb_cNuSDaS, "N_MV_SI2", INT2FIX(N_MV_SI2));
  rb_define_const(rb_cNuSDaS, "N_MV_SI4", INT2FIX(N_MV_SI4));
  rb_define_const(rb_cNuSDaS, "N_MV_R4", rb_float_new((double)N_MV_R4));
  rb_define_const(rb_cNuSDaS, "N_MV_R8", rb_float_new((double)N_MV_R8));

  /* type of array */
  rb_define_const(rb_cNuSDaS, "N_I1", rb_str_new2(N_I1));
  rb_define_const(rb_cNuSDaS, "N_I2", rb_str_new2(N_I2));
  rb_define_const(rb_cNuSDaS, "N_I4", rb_str_new2(N_I4));
  rb_define_const(rb_cNuSDaS, "N_R4", rb_str_new2(N_R4));
  rb_define_const(rb_cNuSDaS, "N_R8", rb_str_new2(N_R8));
  rb_define_const(rb_cNuSDaS, "N_NC", rb_str_new2(N_NC));
  rb_define_const(rb_cNuSDaS, "N_P_1PAC", rb_str_new2(N_P_1PAC));
  rb_define_const(rb_cNuSDaS, "N_P_2PAC", rb_str_new2(N_P_2PAC));
  rb_define_const(rb_cNuSDaS, "N_P_4PAC", rb_str_new2(N_P_4PAC));
  rb_define_const(rb_cNuSDaS, "N_P_R4", rb_str_new2(N_P_R4));
  rb_define_const(rb_cNuSDaS, "N_P_R8", rb_str_new2(N_P_R8));
  rb_define_const(rb_cNuSDaS, "N_P_I1", rb_str_new2(N_P_I1));
  rb_define_const(rb_cNuSDaS, "N_P_I2", rb_str_new2(N_P_I2));
  rb_define_const(rb_cNuSDaS, "N_P_I4", rb_str_new2(N_P_I4));
  rb_define_const(rb_cNuSDaS, "N_P_N1I2", rb_str_new2(N_P_N1I2));
  rb_define_const(rb_cNuSDaS, "N_P_GRIB", rb_str_new2(N_P_GRIB));
  rb_define_const(rb_cNuSDaS, "N_P_RLEN", rb_str_new2(N_P_RLEN));
  rb_define_const(rb_cNuSDaS, "N_P_2UPC", rb_str_new2(N_P_2UPC));

  /* code for nusdas_inq*  */
  rb_define_const(rb_cNuSDaS, "N_MEMBER_NUM", INT2FIX(N_MEMBER_NUM));
  rb_define_const(rb_cNuSDaS, "N_MEMBER_LIST", INT2FIX(N_MEMBER_LIST));
  rb_define_const(rb_cNuSDaS, "N_VALIDTIME_NUM", INT2FIX(N_VALIDTIME_NUM));
  rb_define_const(rb_cNuSDaS, "N_VALIDTIME_LIST", INT2FIX(N_VALIDTIME_LIST));
  rb_define_const(rb_cNuSDaS, "N_PLANE_NUM", INT2FIX(N_PLANE_NUM));
  rb_define_const(rb_cNuSDaS, "N_PLANE_LIST", INT2FIX(N_PLANE_LIST));
  rb_define_const(rb_cNuSDaS, "N_ELEMENT_NUM", INT2FIX(N_ELEMENT_NUM));
  rb_define_const(rb_cNuSDaS, "N_ELEMENT_LIST", INT2FIX(N_ELEMENT_LIST));
  rb_define_const(rb_cNuSDaS, "N_ELEMENT_MAP", INT2FIX(N_ELEMENT_MAP));
  rb_define_const(rb_cNuSDaS, "N_GRID_SIZE", INT2FIX(N_GRID_SIZE));
  rb_define_const(rb_cNuSDaS, "N_GRID_DISTANCE", INT2FIX(N_GRID_DISTANCE));
  rb_define_const(rb_cNuSDaS, "N_GRID_BASEPOINT", INT2FIX(N_GRID_BASEPOINT));
  rb_define_const(rb_cNuSDaS, "N_VALIDTIME_UNIT", INT2FIX(N_VALIDTIME_UNIT));
  rb_define_const(rb_cNuSDaS, "N_VALIDTIME2_LIST", INT2FIX(N_VALIDTIME2_LIST));
  rb_define_const(rb_cNuSDaS, "N_PLANE2_LIST", INT2FIX(N_PLANE2_LIST));
  rb_define_const(rb_cNuSDaS, "N_BASETIME_NUM", INT2FIX(N_BASETIME_NUM));
  rb_define_const(rb_cNuSDaS, "N_BASETIME_LIST", INT2FIX(N_BASETIME_LIST));
  rb_define_const(rb_cNuSDaS, "N_MISSING_MODE", INT2FIX(N_MISSING_MODE));
  rb_define_const(rb_cNuSDaS, "N_MISSING_VALUE", INT2FIX(N_MISSING_VALUE));
  rb_define_const(rb_cNuSDaS, "N_PROJECTION", INT2FIX(N_PROJECTION));
  rb_define_const(rb_cNuSDaS, "N_STAND_LATLON", INT2FIX(N_STAND_LATLON));
  rb_define_const(rb_cNuSDaS, "N_SPARE_LATLON", INT2FIX(N_SPARE_LATLON));


  rb_eNuSDaSError = rb_define_class("NuSDasError", rb_eStandardError);

  rb_eNomemError = rb_define_class("Nomem", rb_eNuSDaSError);
  rb_eEmptytypeError = rb_define_class("Emptytype", rb_eNuSDaSError);
  rb_eBadpathcharError = rb_define_class("Badpathchar", rb_eNuSDaSError);
  rb_eBadtypeError = rb_define_class("Badtype", rb_eNuSDaSError);
  rb_eNotypetabError = rb_define_class("Notypetab", rb_eNuSDaSError);
  rb_eMaxfilesError = rb_define_class("Maxfiles", rb_eNuSDaSError);
  rb_eNodefError = rb_define_class("Nodef", rb_eNuSDaSError);
  rb_eDefsizeError = rb_define_class("Defsize", rb_eNuSDaSError);
  rb_eDefemapError = rb_define_class("Defemap", rb_eNuSDaSError);
  rb_eEmptytype1Error = rb_define_class("Emptytype1", rb_eNuSDaSError);
  rb_eEmptytype2Error = rb_define_class("Emptytype2", rb_eNuSDaSError);
  rb_eNotuniqueError = rb_define_class("Notunique", rb_eNuSDaSError);
  rb_eBadQueryError = rb_define_class("BadQuery", rb_eNuSDaSError);
  rb_eNodataError = rb_define_class("Nodata", rb_eNuSDaSError);
  rb_eBufshortError = rb_define_class("Bufshort", rb_eNuSDaSError);
  rb_eInternalError = rb_define_class("Internal", rb_eNuSDaSError);
  rb_eUnknownelemError = rb_define_class("Unknownelem", rb_eNuSDaSError);





  rb_define_singleton_method(rb_cNuSDaS, "read", rb_read, 9);
  rb_define_singleton_method(rb_cNuSDaS, "write", rb_write, 9);
  rb_define_singleton_method(rb_cNuSDaS, "iocntl", rb_iocntl, 2);
  rb_define_singleton_method(rb_cNuSDaS, "allfile_close", rb_allfile_close, 1);

  /* rb_define_singleton_method(rb_cNuSDaS, "cut", rb_cut, 13); */
  /* rb_define_singleton_method(rb_cNuSDaS, "cut_raw", rb_cut_raw, 12); */
  /*  rb_define_singleton_method(rb_cNuSDaS, "read_3d", rb_read_3d, 11); */
  /*  rb_define_singleton_method(rb_cNuSDaS, "write_3d", rb_write_3d, 11); */

  rb_define_singleton_method(rb_cNuSDaS, "make_mask", rb_make_mask, 1);
  /*  rb_define_singleton_method(rb_cNuSDaS, "set_mask", rb_set_mask, 4); */
  rb_define_singleton_method(rb_cNuSDaS, "onefile_close", rb_onefile_close, 6);
  rb_define_singleton_method(rb_cNuSDaS, "parameter_change", rb_parameter_change, 2);
  /*  rb_define_singleton_method(rb_cNuSDaS, "inq_parameter", rb_inq_parameter, 1); */
  /*  rb_define_singleton_method(rb_cNuSDaS, "parameter_reset", rb_parameter_reset, 1); */

  rb_define_singleton_method(rb_cNuSDaS, "grid", rb_grid, -1);
  rb_define_singleton_method(rb_cNuSDaS, "info", rb_info, -1);
  rb_define_singleton_method(rb_cNuSDaS, "inq_cntl", rb_inq_cntl, 7);
  rb_define_singleton_method(rb_cNuSDaS, "inq_data", rb_inq_data, 9);
  rb_define_singleton_method(rb_cNuSDaS, "inq_def", rb_inq_def, 4);
  rb_define_singleton_method(rb_cNuSDaS, "inq_nrdbtime", rb_inq_nrdbtime, -1);
  rb_define_singleton_method(rb_cNuSDaS, "inq_nrdvtime", rb_inq_nrdvtime, -1);
  rb_define_singleton_method(rb_cNuSDaS, "subc_eta_inq_nz", rb_subc_eta_inq_nz, 7);
  rb_define_singleton_method(rb_cNuSDaS, "subc_rgau_inq_jn", rb_subc_rgau_inq_jn, 6);
  /*  rb_define_singleton_method(rb_cNuSDaS, "subc_inq_subcinfo", rb_subc_inq_subcinfo, 8); */
  /*  rb_define_singleton_method(rb_cNuSDaS, "subc_scan_ds", rb_subc_scan_ds, 0); */

  rb_define_singleton_method(rb_cNuSDaS, "subc_delt", rb_subc_delt, -1);
  rb_define_singleton_method(rb_cNuSDaS, "subc_delt_preset1", rb_subc_delt_preset1, 4);
  rb_define_singleton_method(rb_cNuSDaS, "subc_eta", rb_subc_eta, -1);
  rb_define_singleton_method(rb_cNuSDaS, "subc_preset1", rb_subc_preset1, -1);
  rb_define_singleton_method(rb_cNuSDaS, "subc_rgau", rb_subc_rgau, -1);
  rb_define_singleton_method(rb_cNuSDaS, "subc_rgau_preset1", rb_subc_rgau_preset1, 10);
  rb_define_singleton_method(rb_cNuSDaS, "subc_sigm", rb_subc_sigm, -1);
  rb_define_singleton_method(rb_cNuSDaS, "subc_srf", rb_subc_srf, -1);
  rb_define_singleton_method(rb_cNuSDaS, "subc_tdif", rb_subc_tdif, -1);
  rb_define_singleton_method(rb_cNuSDaS, "subc_zhyb", rb_subc_zhyb, -1);
  rb_define_singleton_method(rb_cNuSDaS, "subc_zhyb_preset1", rb_subc_zhyb_preset1, 12);




  /*
  rb_define_singleton_method(rb_cNuSDaS, "close_esfflush");
  rb_define_singleton_method(rb_cNuSDaS, "dec_raw", rb_dec_raw, 3);
  rb_define_singleton_method(rb_cNuSDaS, "get_size");

  rb_define_singleton_method(rb_cNuSDaS, "list_type");
  rb_define_singleton_method(rb_cNuSDaS, "find_root");
  rb_define_singleton_method(rb_cNuSDaS, "list_dims");
  rb_define_singleton_method(rb_cNuSDaS, "inq_bydims");
  rb_define_singleton_method(rb_cNuSDaS, "getlist_bydims");
  rb_define_singleton_method(rb_cNuSDaS, "elemlist_ladess");
  rb_define_singleton_method(rb_cNuSDaS, "strerror");


  rb_define_singleton_method(rb_cNuSDaS, "nusdims_to_path");
  rb_define_singleton_method(rb_cNuSDaS, "split_type");
  rb_define_singleton_method(rb_cNuSDaS, "split_dims");
  rb_define_singleton_method(rb_cNuSDaS, "path_to_nusdims");
  */

}
