=begin
= grads_gridded_convert.rb: an extension of GrADS_Gridded class

by T Horinouchi

==Overview

This ruby program file provides additional methods to GrADS_Gridded
class.  The methods introduced here are those to convert the GrADS
gridded data into other formats (currently only NetCDF is available),
and vice versa (currently none is available).

==Usage

* to use this extention, RubyNetCDF must have been installed
* load grads_grdded.rb first by
     require "grads_grdded"
     require "grads_grdded_convert"

==Methods

---to_netcdf(filename_or_prefix, varnamelist=nil, separate=false)

   converts a grads dataset to NetCDF file(s).

   ARGUMENTS
   * filename_or_prefix (String): Output file name (path) or prefix.
     See the explanation of "separate" below. 
     This parameter includes the directory (i.e., it is a "path").
   * varnamelist (Array of String or nil): List of the names of the variables
     to output. If nil, all variables are selected.
   * separate (true or false): If true, output is made in one file
     named filename_or_prefix (in this case it is a filename). If false,
     output is made in multiple files --- one file for each variable.
     In this case, output file name is filename_or_prefix+varname+'.nc'
     (thus filename_or_prefix is a prefix).

   RETURN VALUE

   info to continue output, which is used by to_netcdf_append.
   Discard it if you do not append contents from another GrADS file

---to_netcdf_append(nctimeax,ncvars,varnamelist)

   appends the contents to NetCDF files created by to_netcdf.

   WARNING

   The GrADS file to be appended must have exactly the same variables
   with the same vertical levels and "options" as the one converted
   first by to_netcdf. This is ASSUMED, not checked, in the present 
   implimentation.

   ARGUMENTS

   * do not mind what they are unless you are intersted in the
     implimentation. Rather, stick to the example in the follwoing.

   EXAMPLE

      gr1 = GrADS_Gridded.open("grads1.ctl")
      append_info = gr1.to_netcdf("new.nc")
      gr2 = GrADS_Gridded.open("grads2.ctl")
      gr2.to_netcdf_append(*append_info)
      gr3 = GrADS_Gridded.open("grads3.ctl")
      gr3.to_netcdf_append(*append_info)

   * Note here that an asterisk is needed to use to_netcdf_append in this
     way (append_info is an Array of three elements, corresponding to those
     in the argument list).


=end

require "numru/netcdf"

class GrADS_Gridded
  include NumRu

  def set_regexp_units(regexp)
    # Set, if available, a regular expression to derive units from 
    # the description field. For instance, a description like
    # "zonal wind (m/s)" can be handled with this (see below).
    # regexp must have two pairs of parenthesis, so that $1 matches
    # the genuine description part (which will be "long_name") and 
    # $2 matches the units of the variables.
    #
    # EXAMPLE:
    # set_regexp_units( /(.*?)\s*\((.*)\)/ ) for "description (units)"

    @regexp_units = regexp
  end

  def to_netcdf_append(nctimeax,ncvars,varnamelist)

    nfile = nctimeax.length
    varnamelist, varindices =  interpret_vanamelist(varnamelist) 

    # < inquire the offset >
    off = nctimeax[0].shape_current[0]
    #p '####',off

    # < time >
    dim = self.dimensions[3]
    timevals = dim[:start]+dim[:increment]*NArray.float(dim[:len]).indgen!

    # < put variables >
    ntime = self.dimensions[3][:len]
    for t in 0..ntime-1
      if( (t+1)/(ntime/20)*(ntime/20) == (t+1) )
	print "appending #{t+1}-th / #{ntime}\n"    # print once a 20 times
      end
      for ifl in 0..nfile-1
	nctimeax[ifl].put(timevals[t],{"start"=>[t+off],"end"=>[t+off]})
      end
      varnamelist.each_index{|i1|
	i2 = varindices[i1]
	nlev = self.variables[i2][:nlev]
	if nlev == 1
	  xyslice = self.get(varnamelist[i1],0,t)
	  ncvars[i1].put(xyslice,{"start"=>[0,0,t+off],"end"=>[-1,-1,t+off]})
	else
	  for l in 0..nlev-1
	    xyslice = self.get(varnamelist[i1],l,t)
	    ncvars[i1].put(xyslice,{"start"=>[0,0,l,t+off],"end"=>[-1,-1,l,t+off]})
	  end
	end
      }
    end
  end

  def to_netcdf(filename_or_prefix, varnamelist=nil, separate=false)
    # if (!separate) 
    #    put all in the file named FILENAME_OR_PREFIX
    #    (in this case FILENAME_OR_PREFIX is a filename)
    # else
    #    create a file for each varable named FILENAME_OR_PREFIX+varname+'.nc'
    #    (in this case FILENAME_OR_PREFIX is the prefix of filenames)
    # end

    # < interpret varnamelist >
    varnamelist, varindices =  interpret_vanamelist(varnamelist) 
    nvars = varnamelist.length

    # < create output file(s) >
    flindices = Array.new
    ncfiles = Array.new
    if (!separate)
      nfile = 1
      ncfiles[0] = NetCDF.new(filename_or_prefix,'w+')
      nvars.times{|i| flindices.push(0)}
    else
      nfile = nvars
      nvars.times{|i| 
	ncfiles[i] = NetCDF.new(filename_or_prefix+varnamelist[i]+'.nc','w+')
	flindices.push(i)
      }
    end

    ncaxes = []
    nvars.times{ ncaxes.push([]) }
    nctimeax = []

    for ifl in 0..nfile-1

      ncfile = ncfiles[ifl]

      # < set global attributes >
      ncfile.put_att("title", self.title)
      ncfile.put_att("history", "Converted from #{self.ctlfilename} by #{$0}")

      # < define dimesnions & coordinate variables >
      ncdims = []
      axvals = []
      for i in 0..3
	dim = self.dimensions[i]
	name = dim[:name]
	len = dim[:len]
	nlev = len if (i == 2)
	if (i<=2)   # x,y,z
	  ncdims[i] = ncfile.def_dim(name,len) 
	  ncaxes[ifl][i] = ncax = ncfile.def_var(name,"sfloat",[ncdims[i]])
	else        # t
	  ncdims[i] = ncfile.def_dim(name,0)   # unlimited
	  ncaxes[ifl][i] = ncax = 
	    nctimeax[ifl] = ncfile.def_var(name,"float",[ncdims[i]]) 
	end
	ncax.put_att('long_name',dim[:description])
	if (i<=2)   # x,y,z
	  ncax.put_att('units',dim[:units])
	  case(dim[:flag])
	  when /LINEAR/i
	    axvals.push( dim[:start]+dim[:increment]*NArray.sfloat(len).indgen! )
	  when /LEVELS/i
	    axvals.push( dim[:levels] )
	  end
	else        # t
	  if(dim[:increment_units] =~ /day/)
	    ncax.put_att('units',dim[:start_units])  # assumed to be Julian day
	    timevals = dim[:start]+dim[:increment]*NArray.float(len).indgen!
	  else
	    raise "Sorry, increment units other than days are not supported"
	  end
	end
      end

    end

    # < define variables >
    ncvars = []
    varnamelist.each_index{|i1|
      i2 = varindices[i1]
      v=self.variables[i2]
      if (v[:nlev] == nlev)
	ncvars.push( nv = ncfiles[flindices[i1]].def_var( v[:name], "sfloat", 
					      ncdims[0..3] ) )
      elsif (v[:nlev] == 1)
	ncvars.push( nv = ncfiles[flindices[i1]].def_var( v[:name], "sfloat", 
					      ncdims[0..1]+ncdims[3..3] ) )
      else
	raise "Sorry. Currently, # of vertical levels must nlev or 1"
      end
      if(defined?(@regexp_units))
	@regexp_units =~ v[:description]
	if ($1 && $2)
	  long_name = $1
	  units = $2
	  nv.put_att("long_name",long_name)
	  nv.put_att("units",units)
	else
	  long_name = v[:description]
	  nv.put_att("long_name",long_name)
	end
      else
	long_name = v[:description]
	nv.put_att("long_name",long_name)
      end
      nv.put_att("missing_value",self.undef)
    }

    # < put coordinate variables >
    for ifl in 0..nfile-1
      ncfiles[ifl].enddef
      for i in 0..2
	ncaxes[ifl][i].put(axvals[i])
      end
    end

    # < put variables >
    ntime = self.dimensions[3][:len]
    for t in 0..ntime-1
      if( (t+1)/(ntime/20)*(ntime/20) == (t+1) )
	print "writing #{t+1}-th / #{ntime}\n"    # print once a 20 times
      end
      for ifl in 0..nfile-1
	nctimeax[ifl].put(timevals[t],{"start"=>[t],"end"=>[t]})
      end
      varnamelist.each_index{|i1|
	i2 = varindices[i1]
	nlev = self.variables[i2][:nlev]
	if nlev == 1
	  xyslice = self.get(varnamelist[i1],0,t)
	  ncvars[i1].put(xyslice,{"start"=>[0,0,t],"end"=>[-1,-1,t]})
	else
	  for l in 0..nlev-1
	    xyslice = self.get(varnamelist[i1],l,t)
	    ncvars[i1].put(xyslice,{"start"=>[0,0,l,t],"end"=>[-1,-1,l,t]})
	  end
	end
      }
    end
    [nctimeax,ncvars,varnamelist]
  end

  private
  def interpret_vanamelist(varnamelist)
    if (varnamelist)
      # if specified:
      varnamelist = [varnamelist] if (!varnamelist.is_a?(Array))  # if scalar
      nvars = varnamelist.length
      varindices = varnamelist.collect{|v|
	if ( self.varnames.include?(v) )
	  self.varnames.index(v)
	else
	  raise "variable #{v} does not exist in the file"
	end
      }
    else
      # if not specified:
      varnamelist = self.varnames
      nvars = varnamelist.length
      varindices = []
      for i in 0..nvars-1
	varindices.push(i)
      end
    end
    [varnamelist, varindices]
  end


end

if __FILE__ == $0 then
  require "grads_gridded"

  Dir.chdir('/murtmp/horinout/arps-data/much1-saigen-20020213_cont2/grads')
  ctlfilename = 'fi1-much-132x132x15.gradscntl'
  gr = GrADS_Gridded.open(ctlfilename)
  gr.set_regexp_units( /(.*?)\s*\((.*)\)/ )
  gr.dimensions[0][:units] = 'km'
  gr.dimensions[0][:description] = 'x'
  gr.dimensions[1][:units] = 'km'
  gr.dimensions[1][:description] = 'y'
  gr.dimensions[2][:units] = 'm'
  gr.dimensions[2][:description] = 'z'
  #gr.to_netcdf("tmp.nc",["prcrt1","uprt"])
  append_info = gr.to_netcdf("tmp.",["prcrt1"],true)

  # append
  ctlfilename = 'fi1-much-132x132x15.gradscntl.01'
  gr2 = GrADS_Gridded.open(ctlfilename)
  gr2.to_netcdf_append(*append_info)
end
