#include "arg.h"
#include "common.h"

#include <fstream>
#include <sstream>
#include <string>

// Export usage message (-h) to markdown format
// Automatically update the markdown docs

#define HELP_START_MARKER "<!-- HELP_START -->"
#define HELP_END_MARKER   "<!-- HELP_END -->"
#define NOTE_MESSAGE      "<!-- IMPORTANT: The list below is auto-generated by llama-gen-docs; do NOT modify it manually -->"

struct md_file {
    llama_example ex;
    std::string fname;
    std::string specific_section_header;
};

std::vector<md_file> md_files = {
    {LLAMA_EXAMPLE_CLI,        "tools/cli/README.md",        "CLI-specific params"},
    {LLAMA_EXAMPLE_COMPLETION, "tools/completion/README.md", "Completion-specific params"},
    {LLAMA_EXAMPLE_SERVER,     "tools/server/README.md",     "Server-specific params"},
};

static void write_table_header(std::ostringstream & ss) {
    ss << "| Argument | Explanation |\n";
    ss << "| -------- | ----------- |\n";
}

static void write_table_entry(std::ostringstream & ss, const common_arg & opt) {
    ss << "| `";
    // args
    auto all_args = opt.get_args();
    for (const auto & arg : all_args) {
    if (arg == all_args.front()) {
            ss << arg;
            if (all_args.size() > 1) ss << ", ";
        } else {
            ss << arg << (arg != all_args.back() ? ", " : "");
        }
    }
    // value hint
    if (opt.value_hint) {
        std::string md_value_hint(opt.value_hint);
        string_replace_all(md_value_hint, "|", "\\|");
        ss << " " << md_value_hint;
    }
    if (opt.value_hint_2) {
        std::string md_value_hint_2(opt.value_hint_2);
        string_replace_all(md_value_hint_2, "|", "\\|");
        ss << " " << md_value_hint_2;
    }
    // help text
    std::string md_help(opt.help);
    md_help = string_strip(md_help);
    string_replace_all(md_help, "\n", "<br/>");
    string_replace_all(md_help, "|", "\\|");
    ss << "` | " << md_help << " |\n";
}

static void write_table(std::ostringstream & ss, std::vector<common_arg *> & opts) {
    write_table_header(ss);
    for (const auto & opt : opts) {
        write_table_entry(ss, *opt);
    }
}

static void write_help(std::ostringstream & ss, const md_file & md) {
    common_params params;
    auto ctx_arg = common_params_parser_init(params, md.ex);

    std::vector<common_arg *> common_options;
    std::vector<common_arg *> sparam_options;
    std::vector<common_arg *> specific_options;
    for (auto & opt : ctx_arg.options) {
        // in case multiple LLAMA_EXAMPLE_* are set, we prioritize the LLAMA_EXAMPLE_* matching current example
        if (opt.is_sparam) {
            sparam_options.push_back(&opt);
        } else if (opt.in_example(ctx_arg.ex)) {
            specific_options.push_back(&opt);
        } else {
            common_options.push_back(&opt);
        }
    }

    ss << HELP_START_MARKER << "\n\n";

    ss << NOTE_MESSAGE << "\n\n";

    ss << "### Common params\n\n";
    write_table(ss, common_options);
    ss << "\n\n### Sampling params\n\n";
    write_table(ss, sparam_options);
    ss << "\n\n### " << md.specific_section_header << "\n\n";
    write_table(ss, specific_options);

    ss << "\n" << HELP_END_MARKER;
}

int main(int, char **) {
    for (const auto & md : md_files) {
        std::ifstream infile(md.fname);
        if (!infile.is_open()) {
            fprintf(stderr, "failed to open file '%s' for reading\n", md.fname.c_str());
            return 1;
        }

        std::ostringstream ss;
        ss << infile.rdbuf();
        infile.close();

        std::string content = ss.str();

        size_t help_start = content.find(HELP_START_MARKER);
        size_t help_end   = content.find(HELP_END_MARKER);

        if (help_start == std::string::npos || help_end == std::string::npos || help_end <= help_start) {
            fprintf(stderr, "failed to find help markers in file '%s'\n", md.fname.c_str());
            return 1;
        }

        std::ostringstream new_help_ss;
        write_help(new_help_ss, md);
        std::string new_help = new_help_ss.str();

        content = content.substr(0, help_start) + new_help + content.substr(help_end + strlen(HELP_END_MARKER));

        std::ofstream outfile(md.fname);
        if (!outfile.is_open()) {
            fprintf(stderr, "failed to open file '%s' for writing\n", md.fname.c_str());
            return 1;
        }
        outfile << content;
        outfile.close();

        printf("Updated help in '%s'\n", md.fname.c_str());
    }

    return 0;
}
