% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/reachability.R
\name{reachability}
\alias{reachability}
\alias{reachability_plot}
\alias{print.reachability}
\alias{plot.reachability}
\alias{as.reachability}
\alias{as.reachability.dendrogram}
\title{Reachability Distances}
\usage{
\method{print}{reachability}(x, ...)

\method{plot}{reachability}(
  x,
  order_labels = FALSE,
  xlab = "Order",
  ylab = "Reachability dist.",
  main = "Reachability Plot",
  ...
)

as.reachability(object, ...)

\method{as.reachability}{dendrogram}(object, ...)
}
\arguments{
\item{x}{object of class \code{reachability}.}

\item{...}{graphical parameters are passed on to \code{plot()},
or arguments for other methods.}

\item{order_labels}{whether to plot text labels for each points reachability
distance.}

\item{xlab}{x-axis label.}

\item{ylab}{y-axis label.}

\item{main}{Title of the plot.}

\item{object}{any object that can be coerced to class
\code{reachability}, such as an object of class \link{optics} or \link[stats:dendrogram]{stats::dendrogram}.}
}
\value{
An object of class \code{reachability} with components:
\item{order }{order to use for the data points in \code{x}. }
\item{reachdist }{reachability distance for each data point in \code{x}. }
}
\description{
Reachability distances can be plotted to show the hierarchical relationships between data points.
The idea was originally introduced by Ankerst et al (1999) for \link{OPTICS}. Later,
Sanders et al (2003) showed that the visualization is useful for other hierarchical
structures and introduced an algorithm to convert \link{dendrogram} representation to
reachability plots.
}
\details{
A reachability plot displays the points as vertical bars, were the height is the
reachability distance between two consecutive points.
The central idea behind reachability plots is that the ordering in which
points are plotted identifies underlying hierarchical density
representation as mountains and valleys of high and low reachability distance.
The original ordering algorithm OPTICS as described by Ankerst et al (1999)
introduced the notion of reachability plots.

OPTICS linearly orders the data points such that points
which are spatially closest become neighbors in the ordering. Valleys
represent clusters, which can be represented hierarchically. Although the
ordering is crucial to the structure of the reachability plot, its important
to note that OPTICS, like DBSCAN, is not entirely deterministic and, just
like the dendrogram, isomorphisms may exist

Reachability plots were shown to essentially convey the same information as
the more traditional dendrogram structure by Sanders et al (2003). An dendrograms
can be converted into reachability plots.

Different hierarchical representations, such as dendrograms or reachability
plots, may be preferable depending on the context. In smaller datasets,
cluster memberships may be more easily identifiable through a dendrogram
representation, particularly is the user is already familiar with tree-like
representations. For larger datasets however, a reachability plot may be
preferred for visualizing macro-level density relationships.

A variety of cluster extraction methods have been proposed using
reachability plots. Because both cluster extraction depend directly on the
ordering OPTICS produces, they are part of the \code{\link[=optics]{optics()}} interface.
Nonetheless, reachability plots can be created directly from other types of
linkage trees, and vice versa.

\emph{Note:} The reachability distance for the first point is by definition not defined
(it has no preceeding point).
Also, the reachability distances can be undefined when a point does not have enough
neighbors in the epsilon neighborhood. We represent these undefined cases as \code{Inf}
and represent them in the plot as a dashed line.
}
\examples{
set.seed(2)
n <- 20

x <- cbind(
  x = runif(4, 0, 1) + rnorm(n, sd = 0.1),
  y = runif(4, 0, 1) + rnorm(n, sd = 0.1)
)

plot(x, xlim = range(x), ylim = c(min(x) - sd(x), max(x) + sd(x)), pch = 20)
text(x = x, labels = 1:nrow(x), pos = 3)

### run OPTICS
res <- optics(x, eps = 10,  minPts = 2)
res

### plot produces a reachability plot.
plot(res)

### Manually extract reachability components from OPTICS
reach <- as.reachability(res)
reach

### plot still produces a reachability plot; points ids
### (rows in the original data) can be displayed with order_labels = TRUE
plot(reach, order_labels = TRUE)

### Reachability objects can be directly converted to dendrograms
dend <- as.dendrogram(reach)
dend
plot(dend)

### A dendrogram can be converted back into a reachability object
plot(as.reachability(dend))
}
\references{
Ankerst, M., M. M. Breunig, H.-P. Kriegel, J. Sander (1999).
OPTICS: Ordering Points To Identify the Clustering Structure. \emph{ACM
SIGMOD international conference on Management of data.} ACM Press. pp.
49--60.

Sander, J., X. Qin, Z. Lu, N. Niu, and A. Kovarsky (2003). Automatic
extraction of clusters from hierarchical clustering representations.
\emph{Pacific-Asia Conference on Knowledge Discovery and Data Mining.}
Springer Berlin Heidelberg.
}
\seealso{
\code{\link[=optics]{optics()}}, \code{\link[=as.dendrogram]{as.dendrogram()}}, and \code{\link[stats:hclust]{stats::hclust()}}.
}
\author{
Matthew Piekenbrock
}
\keyword{clustering}
\keyword{hierarchical}
\keyword{model}
