; AleoBFT Library
;
; Copyright (C) 2025 Provable Inc.
;
; License: See the LICENSE file distributed with this library.
;
; Authors: Alessandro Coglio (www.alessandrocoglio.info)
;          Eric McCarthy (bendyarm on GitHub)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package "ALEOBFT-PROPOSALS")

(include-book "system-states")
(include-book "committees")

(local (include-book "kestrel/utilities/nfix" :dir :system))

(local (include-book "kestrel/built-ins/disable" :dir :system))
(local (acl2::disable-most-builtin-logic-defuns))
(local (acl2::disable-builtin-rewrite-rules-for-defaults))
(set-induction-depth-limit 0)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defxdoc+ transitions-endorse
  :parents (transitions)
  :short "Transitions for proposal endorsement."
  :long
  (xdoc::topstring
   (xdoc::p
    "Here we define the system state changes caused by @('endorse') events.")
   (xdoc::p
    "If the network contains a message with a proposal addressed to a validator
     (this message is generated by a @('propose') event),
     the validator can consume that message and endorse the proposal,
     under suitable conditions.
     If the validator is faulty, there is in fact no condition:
     a faulty validator may sign anything.
     If the validator is correct,
     there are a number of checks that the validator performs.
     Either way, a successful endorsement results in
     a new endorsement message being added to the network:
     this is addressed to the author of the proposal."))
  :order-subtopics t
  :default-parent t)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define endorse-possiblep ((prop proposalp)
                           (endor addressp)
                           (systate system-statep))
  :returns (yes/no booleanp)
  :short "Check if an @('endorse') event is possible in a system state."
  :long
  (xdoc::topstring
   (xdoc::p
    "The @('prop') and @('endor') parameters of this function
     are the corresponding components of the @('endorse') event.")
   (xdoc::p
    "For the event to occur, the network must contain a proposal message
     that consists of the proposal and the endorser as destination:
     that is, the proposal must be addressed to the endorser.
     The message is isomorphic to the event, in fact.")
   (xdoc::p
    "If the endorser is faulty, there is no further condition.
     If the faulty validators decided not to endorse the proposal instead,
     that would be a non-event, i.e. it would not be represented by this event:
     an @('endorse') event captures the case in which
     the validator actually endorses the proposal.
     We could also add an event to model the situation in which
     a faulty validator consumes the proposal message,
     but does not send back an endorsement;
     however, the only difference is that, in our current model,
     the proposal message may sit in the network forever,
     which can be just ignored.")
   (xdoc::p
    "If the endorser is correct, the following conditions must hold:")
   (xdoc::ul
    (xdoc::li
     "The proposal author and the endorser must be both
      in the active committee at the proposal round,
      which the endorser must be thus able to calculate.")
    (xdoc::li
     "The endorser must have no certificate, in the DAG,
      with the same author and round as the proposal.")
    (xdoc::li
     "The endorser must not have already endorsed
      a proposal with the same author and round as the proposal to endorse.
      This is looked up in the component of the validator state
      that keeps track of the proposals endorsed so far,
      whose corresponding certificates are not in the validator's DAG yet.")
    (xdoc::li
     "If the proposal's round number is 1,
      the proposal must have no references to previous certificates.
      Otherwise, all the referenced certificates must be in the DAG,
      and their authors must form a non-empty quorum."))
   (xdoc::p
    "Some of these conditions are the same as in @(tsee propose-possiblep).
     This is unsurprising, because (correct) authors and endorsers
     must perform similar checks.")
   (xdoc::p
    "Note that, as in @(tsee propose-possiblep),
     the endorser does not check that the authors of the previous certificates
     are members of the active committee at the previous round.
     The reason is that this is an invariant, so it can be safely skipped.
     Indeed, we use @(tsee committee-validators-stake)
     to check the quorum the condition.")
   (xdoc::p
    "The same observation about (non-)empty committees applies here,
     which is made in the documentation of @(tsee propose-possiblep)."))
  (b* ((msg (make-message-proposal :proposal prop :destination endor))
       ((unless (set::in msg (get-network-state systate))) nil)
       ((unless (set::in (address-fix endor) (correct-addresses systate))) t)
       ((proposal prop) prop)
       ((validator-state vstate) (get-validator-state endor systate))
       (commtt (active-committee-at-round prop.round vstate.blockchain))
       ((unless commtt) nil)
       ((unless (set::in prop.author (committee-members commtt))) nil)
       ((unless (set::in (address-fix endor) (committee-members commtt))) nil)
       ((unless (set::emptyp
                 (certs-with-author+round prop.author prop.round vstate.dag)))
        nil)
       ((unless (set::emptyp
                 (props-with-author+round
                  prop.author prop.round vstate.endorsed)))
        nil)
       ((when (= prop.round 1)) (set::emptyp prop.previous))
       ((when (set::emptyp prop.previous)) nil)
       ((unless (set::subset prop.previous
                             (cert-set->author-set
                              (certs-with-round (1- prop.round) vstate.dag))))
        nil)
       (prev-commtt
        (active-committee-at-round (1- prop.round) vstate.blockchain))
       ((unless (>= (committee-validators-stake prop.previous prev-commtt)
                    (committee-quorum-stake prev-commtt)))
        nil))
    t)
  :guard-hints
  (("Goal"
    :in-theory (enable posp active-committee-at-previous-round-when-at-round)))
  :hooks (:fix))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define endorse-next ((prop proposalp)
                      (endor addressp)
                      (systate system-statep))
  :guard (endorse-possiblep prop endor systate)
  :returns (new-systate system-statep)
  :short "New system state resulting from an @('endorse') event."
  :long
  (xdoc::topstring
   (xdoc::p
    "The @('prop') and @('endor') parameters of this function
     are the corresponding components of the @('endorse') event.")
   (xdoc::p
    "The proposal message is removed from the network,
     and an endorsement message is added to the network.
     Additionally, if the endorser is correct,
     its state is updated to record that it has endorsed the proposal."))
  (b* (((proposal prop) prop)
       (network (get-network-state systate))
       (old-msg (make-message-proposal :proposal prop :destination endor))
       (new-msg (make-message-endorsement :proposal prop :endorser endor))
       (new-network (set::insert new-msg (set::delete old-msg network)))
       (systate (update-network-state new-network systate))
       ((unless (set::in (address-fix endor) (correct-addresses systate)))
        systate)
       ((validator-state vstate) (get-validator-state endor systate))
       (new-endorsed (set::insert (proposal-fix prop) vstate.endorsed))
       (new-vstate (change-validator-state vstate :endorsed new-endorsed))
       (systate (update-validator-state endor new-vstate systate)))
    systate)
  :hooks (:fix)

  ///

  (defret correct-addresses-of-endorse-next
    (equal (correct-addresses new-systate)
           (correct-addresses systate)))

  (local (in-theory (enable get-validator-state-of-update-validator-state)))

  (defret validator-state->round-of-endorse-next
    (equal (validator-state->round (get-validator-state val new-systate))
           (validator-state->round (get-validator-state val systate))))

  (defret validator-state->dag-of-endorse-next
    (equal (validator-state->dag (get-validator-state val new-systate))
           (validator-state->dag (get-validator-state val systate))))

  (defret validator-state->proposed-of-endorse-next
    (equal (validator-state->proposed (get-validator-state val new-systate))
           (validator-state->proposed (get-validator-state val systate))))

  (defret validator-state->endorsed-of-endorse-next
    (equal (validator-state->endorsed (get-validator-state val new-systate))
           (if (and (equal (address-fix val) (address-fix endor))
                    (set::in (address-fix val) (correct-addresses systate)))
               (set::insert (proposal-fix prop)
                            (validator-state->endorsed
                             (get-validator-state val systate)))
             (validator-state->endorsed (get-validator-state val systate)))))
  (in-theory (disable validator-state->proposed-of-endorse-next))

  (defret validator-state->last-of-endorse-next
    (equal (validator-state->last (get-validator-state val new-systate))
           (validator-state->last (get-validator-state val systate))))

  (defret validator-state->blockchain-of-endorse-next
    (equal (validator-state->blockchain (get-validator-state val new-systate))
           (validator-state->blockchain (get-validator-state val systate))))

  (defret validator-state->committed-of-endorse-next
    (equal (validator-state->committed (get-validator-state val new-systate))
           (validator-state->committed (get-validator-state val systate))))

  (defret get-network-state-of-endorse-next
    (equal (get-network-state new-systate)
           (set::insert (message-endorsement prop endor)
                        (set::delete (message-proposal prop endor)
                                     (get-network-state systate)))))
  (in-theory (disable get-network-state-of-endorse-next)))
