# -*- coding: utf-8 -*-
#!/usr/bin/python

# Script to pull one of the themes available at eclipsecolorthemes.org.
# The retrieved XML file is converted to the highlight Lua file format.

# In October 2022 eclipsecolorthemes.org went offline. See possible new source:
# https://github.com/eclipse-color-theme/eclipse-color-theme/issues/290

import requests
import sys
import xml.etree.ElementTree as ET
from datetime import date

if __name__ == '__main__':

    if (len(sys.argv) != 2):
        print("Please define the theme number to download from eclipsecolorthemes.org")
        sys.exit()

    themeID = sys.argv[1]

    print("Retrieving Theme #%s..." % themeID)
    r = requests.get(
        'http://www.eclipsecolorthemes.org/',
        params=[('view', 'empty'), ('action', 'download'), ('theme', themeID), ('type', 'xml')],
    )

    root = ET.fromstring(r.text)

    url = r.url
    today = date.today()

    scheme = root.attrib['name']
    author = root.attrib['author']
    id = root.attrib['id']

    if (id == ""):
        print("This ID is not known.")
        sys.exit()

    canvas = root.find('background').attrib['color']
    default = root.find('foreground').attrib['color']
    number = root.find('number').attrib['color']
    operator = root.find('operator').attrib['color']
    string = root.find('string').attrib['color']
    escape = root.find('commentTaskTag').attrib['color']
    preprocessor = root.find('annotation').attrib['color']
    stringPreProc = root.find('javadocLink').attrib['color']
    interpolation = root.find('commentTaskTag').attrib['color']
    lineComment = root.find('singleLineComment').attrib['color']
    blockComment = root.find('multiLineComment').attrib['color']
    lineNum = root.find('lineNumber').attrib['color']
    kwa = root.find('keyword').attrib['color']
    kwb = root.find('method').attrib['color']
    kwc = root.find('interface').attrib['color']
    kwd = root.find('field').attrib['color']
    kwe = root.find('staticField').attrib['color']
    kwf = root.find('localVariable').attrib['color']

    theme = """--[[%s | Theme generated by eclipse_color_themes.py
================================================================================
Author: %s
Source: %s
================================================================================]]

Description = "Eclipse #%s (%s)"
Categories = { "eclipse"}

--------------------------------------------------------------------------------
Canvas         = { Colour = "%s", Italic = false, Bold = false }
Default        = { Colour = "%s", Italic = false, Bold = false }
Number         = { Colour = "%s", Italic = false, Bold = false }
Operator       = { Colour = "%s", Italic = false, Bold = false }
String         = { Colour = "%s", Italic = false, Bold = false }
Escape         = { Colour = "%s", Italic = false, Bold = false }
StringPreProc  = { Colour = "%s", Italic = false, Bold = false }
Interpolation  = { Colour = "%s", Italic = false, Bold = false }
LineComment    = { Colour = "%s", Italic = false,  Bold = false }
BlockComment   = { Colour = "%s", Italic = false,  Bold = false }
LineNum        = { Colour = "%s", Italic = false, Bold = false }
PreProcessor   = { Colour = "%s", Italic = false, Bold = false }

Keywords = {
  { Colour = "%s", Italic = false, Bold = true  }, --> kwd 1
  { Colour = "%s", Italic = false, Bold = true  }, --> kwd 2
  { Colour = "%s", Italic = false, Bold = true  }, --> kwd 3
  { Colour = "%s", Italic = false, Bold = true  }, --> kwd 4
  { Colour = "%s", Italic = false, Bold = true  }, --> kwd 5
  { Colour = "%s", Italic = false, Bold = true  }, --> kwd 6
}

SemanticTokenTypes = {
  { Type = 'type', Style = Keywords[2] },
  { Type = 'class', Style =  Keywords[1] },
  { Type = 'struct', Style =  Keywords[4] },
  { Type = 'interface', Style = Keywords[1] },
  { Type = 'parameter', Style = Keywords[6] },
  { Type = 'variable', Style = Keywords[5] },
  { Type = 'enumMember', Style = Keywords[5] },
  { Type = 'function', Style = Keywords[4] },
  { Type = 'method', Style = Keywords[4] },
  { Type = 'keyword', Style =  Keywords[1]},
  { Type = 'number', Style = Number },
  { Type = 'regexp', Style = String },
  { Type = 'operator', Style = Operator },
}
""" % (today, author, url, id, scheme,
        canvas, default, number, operator, string, escape, stringPreProc,
        interpolation, lineComment, blockComment, lineNum, preprocessor,
        kwa, kwb, kwc, kwd, kwe, kwf)

    themeName = '%s.theme' % "_".join(scheme.lower().split())

    print("Saving Highlight Theme File %s..." % themeName)

    f = open(themeName, 'w')
    f.write(theme)
    f.close()
