//filter-slfi-strobe.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2012-2019
 *
 *  This file is part of roard a part of RoarAudio,
 *  a cross-platform sound system for both, home and professional use.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  RoarAudio is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 */

#include <roaraudio.h>
#include <libroarlight/libroarlight.h>

struct slfi_strobe {
 // config:
 ssize_t channel_brightness;
 ssize_t channel_frequency;
 ssize_t channel_trigger;
 uint8_t brightness;
 uint8_t frequency;
 uint8_t event;

 // runtime:
 uint8_t value_brightness;
 uint8_t value_frequency;
 uint8_t value_trigger;
};

static int __init(struct roar_slfi_inst * inst, const struct roar_keyval * para, ssize_t paralen) {
 struct slfi_strobe * self = roar_mm_malloc(sizeof(struct slfi_strobe));
 const struct roar_keyval * kv;
 ssize_t i;

 if ( self == NULL )
  return -1;

 memset(self, 0, sizeof(*self));
 self->channel_brightness =  -1;
 self->channel_frequency  =  -1;
 self->channel_trigger    =  -1;
 self->brightness         = 255;
 self->frequency          =  64;
 self->event              = ROAR_ROARDMX_EVENT_STROBE;
 inst->userdata = self;

 for (i = 0; i < paralen; i++) {
  kv = &(para[i]);
  if ( kv->key == NULL || kv->value == NULL )
   continue;
  if ( !strcmp(kv->key, "event") ) {
   self->event = roar_roardmx_str2event(kv->value) & ROAR_ROARDMX_MASK_EVENT;
  } else if ( !strcmp(kv->key, "brightness") ) {
   self->brightness = atoi(kv->value);
  } else if ( !strcmp(kv->key, "frequency") ) {
   self->frequency = atoi(kv->value);
  } else if ( !strcmp(kv->key, "brightnesschannel") ) {
   self->channel_brightness = atoi(kv->value);
  } else if ( !strcmp(kv->key, "frequencychannel") ) {
   self->channel_frequency = atoi(kv->value);
  } else if ( !strcmp(kv->key, "triggerchannel") ) {
   self->channel_trigger = atoi(kv->value);
  } else {
   ROAR_WARN("__init(*): Unknown parameter: %s", kv->key);
  }
 }

 return 0;
}

static inline void __set_channel(uint8_t * universe, ssize_t size_of_universe, ssize_t channel, uint8_t value) {
 if ( channel < 0 )
  return;
 if ( channel >= size_of_universe ) {
  ROAR_WARN("__set_channel(+): Universe too small for filter. Channel %llu not found.", (long long unsigned int)channel);
  return;
 }
 universe[channel] = value;
}

static int __update(struct roar_slfi_inst * inst, uint8_t * universe, ssize_t size_of_universe, int32_t usecspassed, const uint8_t * event, size_t eventlen) {
 struct slfi_strobe * self = inst->userdata;
 size_t i;

 (void)usecspassed;

 for (i = 0; i < eventlen; i++) {
  switch (event[i]) {
   case ROAR_ROARDMX_EVENT_STROBE|ROAR_ROARDMX_ETYPE_BEAT:
     self->value_trigger += 7; // prime value > 3 for maximum 'random' effect.
    break;
   case ROAR_ROARDMX_EVENT_STROBE|ROAR_ROARDMX_ETYPE_OFF:
     self->value_brightness = 0;
     self->value_frequency  = 0;
    break;
   case ROAR_ROARDMX_EVENT_STROBE|ROAR_ROARDMX_ETYPE_ON:
   case ROAR_ROARDMX_EVENT_STROBE|ROAR_ROARDMX_ETYPE_HOLD:
     self->value_brightness = self->brightness;
     self->value_frequency  = self->frequency;
    break;
  }
 }

 __set_channel(universe, size_of_universe, self->channel_brightness, self->value_brightness);
 __set_channel(universe, size_of_universe, self->channel_frequency,  self->value_frequency);
 __set_channel(universe, size_of_universe, self->channel_trigger,    self->value_trigger);

 return 0;
}

static const struct roar_slfi_filter filter[1] = {
 {
  .name = "strobe",
  .description = "Controls strobes via events",
  .flags = ROAR_SLFI_FLAG_NONE,
  .init = __init,
  .uninit = NULL,
  .update = __update,
  .ctl = NULL
 }
};

ROAR_DL_PLUGIN_REG_SLFI(filter);

// This is the plugin control block.
ROAR_DL_PLUGIN_START(filter_slfi_strobe) {
 // Here we set the name and vendor of our plugin.
 // If you have no Vendor ID you need to use ROAR_DL_PLUGIN_META_PRODUCT_NV().
 ROAR_DL_PLUGIN_META_PRODUCT_NIV("filter-slfi-strobe", ROAR_VID_ROARAUDIO, ROAR_VNAME_ROARAUDIO);

 // This sets the version of your plugin.
 ROAR_DL_PLUGIN_META_VERSION(ROAR_VERSION_STRING);

 // This sets the license of your plugin.
 // If there is no tag for the license you use you can just
 // use ROAR_DL_PLUGIN_META_LICENSE().
 ROAR_DL_PLUGIN_META_LICENSE_TAG(GPLv3_0);

 // This sets the author and contact infos.
 // There are several other macros to do this with other parameters.
 // See ROAR_DL_PLUGIN_META_CONTACT*() in the header or documentation.
 ROAR_DL_PLUGIN_META_CONTACT_FLNE("Philipp", "Schafft", "ph3-der-loewe", "lion@lion.leolix.org");

 // This sets the description for your plugin.
 ROAR_DL_PLUGIN_META_DESC("This is a plugin allows one to control common strobes via the strobe event.");

 // Load filters.
 ROAR_DL_PLUGIN_REG_FNFUNC(ROAR_DL_FN_FILTER);

// This is the end of the control block.
} ROAR_DL_PLUGIN_END

//ll
