<?php

/*
 * This file is part of the Symfony package.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Symfony\Component\Security\Core\Tests\Authentication\Token;

use PHPUnit\Framework\TestCase;
use Symfony\Component\Security\Core\Authentication\Token\SwitchUserToken;
use Symfony\Component\Security\Core\Authentication\Token\UsernamePasswordToken;
use Symfony\Component\Security\Core\Tests\Authentication\Token\Fixtures\CustomUser;
use Symfony\Component\Security\Core\User\InMemoryUser;

class SwitchUserTokenTest extends TestCase
{
    public function testSerialize()
    {
        $originalToken = new UsernamePasswordToken(new InMemoryUser('user', 'foo', ['ROLE_ADMIN', 'ROLE_ALLOWED_TO_SWITCH']), 'provider-key', ['ROLE_ADMIN', 'ROLE_ALLOWED_TO_SWITCH']);
        $token = new SwitchUserToken(new InMemoryUser('admin', 'bar', ['ROLE_USER']), 'provider-key', ['ROLE_USER'], $originalToken, 'https://symfony.com/blog');

        $unserializedToken = unserialize(serialize($token));

        $this->assertInstanceOf(SwitchUserToken::class, $unserializedToken);
        $this->assertSame('admin', $unserializedToken->getUserIdentifier());
        $this->assertSame('provider-key', $unserializedToken->getFirewallName());
        $this->assertEquals(['ROLE_USER'], $unserializedToken->getRoleNames());
        $this->assertSame('https://symfony.com/blog', $unserializedToken->getOriginatedFromUri());

        $unserializedOriginalToken = $unserializedToken->getOriginalToken();

        $this->assertInstanceOf(UsernamePasswordToken::class, $unserializedOriginalToken);
        $this->assertSame('user', $unserializedOriginalToken->getUserIdentifier());
        $this->assertSame('provider-key', $unserializedOriginalToken->getFirewallName());
        $this->assertEquals(['ROLE_ADMIN', 'ROLE_ALLOWED_TO_SWITCH'], $unserializedOriginalToken->getRoleNames());
    }

    public function testSerializeNullImpersonateUrl()
    {
        $originalToken = new UsernamePasswordToken(new InMemoryUser('user', 'foo', ['ROLE_ADMIN', 'ROLE_ALLOWED_TO_SWITCH']), 'provider-key', ['ROLE_ADMIN', 'ROLE_ALLOWED_TO_SWITCH']);
        $token = new SwitchUserToken(new InMemoryUser('admin', 'bar', ['ROLE_USER']), 'provider-key', ['ROLE_USER'], $originalToken);

        $unserializedToken = unserialize(serialize($token));

        $this->assertNull($unserializedToken->getOriginatedFromUri());
    }

    /**
     * Tests if an old version of SwitchUserToken can still be unserialized.
     *
     * The fixture was generated by running the following code with Symfony 4.4 and PHP 8.0.
     *
     * serialize(
     *     new SwitchUserToken(
     *         new CustomUser('john', ['ROLE_USER']),
     *         ['foo' => 'bar'],
     *         'main', ['ROLE_USER'],
     *         new UsernamePasswordToken(
     *             new CustomUser('jane', ['ROLE_USER']),
     *             ['foo' => 'bar'],
     *             'main',
     *             ['ROLE_USER']
     *         )
     *     )
     * )
     */
    public function testUnserializeOldToken()
    {
        /** @var SwitchUserToken $token */
        $token = unserialize(file_get_contents(__DIR__.'/Fixtures/switch-user-token-4.4.txt'));

        self::assertInstanceOf(SwitchUserToken::class, $token);
        self::assertInstanceOf(UsernamePasswordToken::class, $token->getOriginalToken());
        self::assertInstanceOf(CustomUser::class, $token->getUser());
        self::assertSame('john', $token->getUserIdentifier());
        self::assertSame('main', $token->getFirewallName());
        self::assertEquals(['ROLE_USER'], $token->getRoleNames());
        self::assertNull($token->getOriginatedFromUri());
    }
}
