/*
 * Copyright 2021-2025 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opentest4j.reporting.events.api;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.Writer;
import java.util.HashMap;
import java.util.Map;

import static java.util.Collections.unmodifiableMap;

class AttributeValueEscapingWriter extends Writer {

	private static final Map<Character, String> REPLACEMENTS_IN_ATTRIBUTE_VALUES;
	static {
		Map<Character, String> tmp = new HashMap<>(3);
		tmp.put('\n', "&#10;");
		tmp.put('\r', "&#13;");
		tmp.put('\t', "&#9;");
		REPLACEMENTS_IN_ATTRIBUTE_VALUES = unmodifiableMap(tmp);
	}

	private final BufferedWriter delegate;
	private boolean whitespaceReplacingEnabled;

	AttributeValueEscapingWriter(BufferedWriter delegate) {
		this.delegate = delegate;
	}

	void setWhitespaceReplacingEnabled(boolean whitespaceReplacingEnabled) {
		this.whitespaceReplacingEnabled = whitespaceReplacingEnabled;
	}

	@Override
	public void write(char[] cbuf, int off, int len) throws IOException {
		if (!whitespaceReplacingEnabled) {
			delegate.write(cbuf, off, len);
			return;
		}
		StringBuilder stringBuilder = new StringBuilder(len * 2);
		for (int i = off; i < off + len; i++) {
			char c = cbuf[i];
			String replacement = REPLACEMENTS_IN_ATTRIBUTE_VALUES.get(c);
			if (replacement != null) {
				stringBuilder.append(replacement);
			}
			else {
				stringBuilder.append(c);
			}
		}
		delegate.write(stringBuilder.toString());
	}

	@Override
	public void write(int c) throws IOException {
		if (whitespaceReplacingEnabled) {
			super.write(c);
		}
		else {
			delegate.write(c);
		}
	}

	@Override
	public void write(char[] cbuf) throws IOException {
		if (whitespaceReplacingEnabled) {
			super.write(cbuf);
		}
		else {
			delegate.write(cbuf);
		}
	}

	@Override
	public void write(String str) throws IOException {
		if (whitespaceReplacingEnabled) {
			super.write(str);
		}
		else {
			delegate.write(str);
		}
	}

	@Override
	public void write(String str, int off, int len) throws IOException {
		if (whitespaceReplacingEnabled) {
			super.write(str, off, len);
		}
		else {
			delegate.write(str, off, len);
		}
	}

	@Override
	public Writer append(CharSequence csq) throws IOException {
		if (whitespaceReplacingEnabled) {
			return super.append(csq);
		}
		else {
			return delegate.append(csq);
		}
	}

	@Override
	public Writer append(CharSequence csq, int start, int end) throws IOException {
		if (whitespaceReplacingEnabled) {
			return super.append(csq, start, end);
		}
		else {
			return delegate.append(csq, start, end);
		}
	}

	@Override
	public Writer append(char c) throws IOException {
		if (whitespaceReplacingEnabled) {
			return super.append(c);
		}
		else {
			return delegate.append(c);
		}
	}

	public void newLine() throws IOException {
		delegate.newLine();
	}

	@Override
	public void flush() throws IOException {
		delegate.flush();
	}

	@Override
	public void close() throws IOException {
		delegate.close();
	}
}
