module time_scheme
  use force_solv
  use Ellip_Slv

contains

subroutine time_schematic( it, zo, psio, zn, psin, zopt1 )
!-- calculating time integration
  use saveval_define
  implicit none
  integer, intent(in) :: it                               ! time step
  double precision, dimension(size(x),size(y)), intent(inout) :: zo   ! old zeta
  double precision, dimension(size(x),size(y)), intent(inout) :: psio ! psi of zo
  double precision, dimension(size(x),size(y)), intent(out) :: zn     ! new zeta
  double precision, dimension(size(x),size(y)), intent(out) :: psin   ! psi of zn
  double precision, dimension(size(x),size(y)), intent(inout), optional :: zopt1
                                           ! additional term
                                           ! L-F => zeta in t=n-1
                                           ! AB2 => force in t=n-1
  integer :: i, j, ct
  double precision, dimension(size(x),size(y)) :: zt, forcea_zeta, force_zeta

  force_zeta=0.0d0
  forcea_zeta=0.0d0
  zt=zo

  select case (time_flag(1:3))

  case ('EUL')  ! Euler scheme

     call calc_FORCE( zo, psio, force_zeta )

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

     do j=1,ny
        do i=1,nx
           zn(i,j)=zo(i,j)+dt*force_zeta(i,j)
        end do
     end do

!$omp end do
!$omp end parallel

     call set_bound( zn )

     psin=psio
     call Ellip_Jacobi_2d( x, y, zn, eps, boundary, psin,  &
  &                        init_flag=.false. )

  case ('RK3')  ! 3rd order Runge-Kutta scheme

     do ct=1,3

        call calc_FORCE( zo, psio, force_zeta )

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

        do j=1,ny
           do i=1,nx
              select case (ct)
              case (1)
                 forcea_zeta(i,j)=forcea_zeta(i,j)+force_zeta(i,j)
                 zo(i,j)=zt(i,j)+(0.5d0*dt)*force_zeta(i,j)
              case (2)                         
                 zo(i,j)=zt(i,j)+dt*(-forcea_zeta(i,j)+2.0d0*force_zeta(i,j))
                 forcea_zeta(i,j)=forcea_zeta(i,j)+4.0d0*force_zeta(i,j)
              case (3)                         
                 forcea_zeta(i,j)=forcea_zeta(i,j)+force_zeta(i,j)
                 zn(i,j)=zt(i,j)+(dt/6.0d0)*forcea_zeta(i,j)
              end select
           end do
        end do

!$omp end do
!$omp end parallel

        select case (ct)
        case (1,2)
           call set_bound( zo )

           call Ellip_Jacobi_2d( x, y, zo, eps, boundary, psio,  &
  &                              init_flag=.false. )
        case (3)
           call set_bound( zn )

           psin=psio
           call Ellip_Jacobi_2d( x, y, zn, eps, boundary, psin,  &
  &                              init_flag=.false. )
        end select

     end do

  case ('RK4')  ! 4th order Runge-Kutta scheme

     do ct=1,4

        call calc_FORCE( zo, psio, force_zeta )

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

        do j=1,ny
           do i=1,nx
              select case (ct)
              case (1)
                 forcea_zeta(i,j)=forcea_zeta(i,j)+force_zeta(i,j)
                 zo(i,j)=zt(i,j)+(0.5d0*dt)*force_zeta(i,j)
              case (2)                         
                 forcea_zeta(i,j)=forcea_zeta(i,j)+2.0d0*force_zeta(i,j)
                 zo(i,j)=zt(i,j)+(0.5d0*dt)*force_zeta(i,j)
              case (3)                         
                 forcea_zeta(i,j)=forcea_zeta(i,j)+2.0d0*force_zeta(i,j)
                 zo(i,j)=zt(i,j)+(dt)*force_zeta(i,j)
              case (4)                         
                 forcea_zeta(i,j)=forcea_zeta(i,j)+force_zeta(i,j)
                 zn(i,j)=zt(i,j)+(dt/6.0d0)*forcea_zeta(i,j)
              end select
           end do
        end do

!$omp end do
!$omp end parallel

        select case (ct)
        case (1,2,3)
           call set_bound( zo )

           call Ellip_Jacobi_2d( x, y, zo, eps, boundary, psio,  &
  &                              init_flag=.false. )
        case (4)
           call set_bound( zn )

           psin=psio
           call Ellip_Jacobi_2d( x, y, zn, eps, boundary, psin,  &
  &                              init_flag=.false. )
        end select

     end do

  case ('AB2')  ! 2nd order Adams-Bashforse scheme

     call calc_FORCE( zo, psio, force_zeta )

     if(it==1)then

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

        do j=1,ny
           do i=1,nx
              zn(i,j)=zo(i,j)+dt*force_zeta(i,j)
              zopt1(i,j)=force_zeta(i,j)
           end do
        end do

!$omp end do
!$omp end parallel

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

        do j=1,ny
           do i=1,nx
              zn(i,j)=zo(i,j)+dt*(1.5d0*force_zeta(i,j)-0.5d0*zopt1(i,j))
              zopt1(i,j)=force_zeta(i,j)
           end do
        end do

!$omp end do
!$omp end parallel

     end if

     call set_bound( zn )

     psin=psio
     call Ellip_Jacobi_2d( x, y, zn, eps, boundary, psin,  &
  &                        init_flag=.false. )

  case ('L-F')  ! Leap Frog scheme

     call calc_FORCE( zo, psio, force_zeta )

     if(it==1)then

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

        do j=1,ny
           do i=1,nx
              zn(i,j)=zo(i,j)+dt*force_zeta(i,j)
              zopt1(i,j)=zo(i,j)
           end do
        end do

!$omp end do
!$omp end parallel

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)

        do j=1,ny
           do i=1,nx
              zn(i,j)=zopt1(i,j)+2.0d0*dt*force_zeta(i,j)
              zopt1(i,j)=zo(i,j)
           end do
        end do

!$omp end do
!$omp end parallel

     end if

     call set_bound( zn )

     psin=psio
     call Ellip_Jacobi_2d( x, y, zn, eps, boundary, psin,  &
  &                        init_flag=.false. )

!  case ('C-N')  ! Crank-Nicolson scheme

!  case ('BEU')  ! Backward Euler scheme

  end select

end subroutine time_schematic

end module time_scheme
