!---------------------------------------------------------------
! Copyright (C) 2009-2015 GFD Dennou Club. All rights reserved.
!---------------------------------------------------------------

module Ellip_Slv   ! 代数演算を用いて楕円型偏微分方程式を解くモジュール

  use derivation

  public  :: Ellip_GauSei_2d,  &
  &          Ellip_Gausei_2df,  &
  &          Ellip_Gausei_2dd

  public  :: Ellip_GauSei_3d,  &
  &          Ellip_Gausei_3df,  &
  &          Ellip_Gausei_3dd

  public  :: Ellip_Jacobi_2d,  &
  &          Ellip_Jacobi_2df,  &
  &          Ellip_Jacobi_2dd

  public  :: Ellip_Jacobi_3d,  &
  &          Ellip_Jacobi_3df,  &
  &          Ellip_Jacobi_3dd

  private :: set_bound,  &
  &          setval_boundf,  &
  &          setval_boundd,  &
  &          set_coef,  &
  &          set_coed,  &
  &          calculate_boundf,  &
  &          calculate_boundd

  private :: set_bound_3d,  &
  &          setval_bound_3df,  &
  &          setval_bound_3dd,  &
  &          set_coe_3df,  &
  &          set_coe_3dd,  &
  &          calculate_bound_3df,  &
  &          calculate_bound_3dd

!  private :: check_bound

  interface Ellip_GauSei_2d
     module procedure Ellip_Gausei_2df,  &
  &                   Ellip_Gausei_2dd
  end interface Ellip_GauSei_2d

  interface Ellip_GauSei_3d
     module procedure Ellip_Gausei_3df,  &
  &                   Ellip_Gausei_3dd
  end interface Ellip_GauSei_3d

  interface Ellip_Jacobi_2d
     module procedure Ellip_Jacobi_2df,  &
  &                   Ellip_Jacobi_2dd
  end interface Ellip_Jacobi_2d

  interface Ellip_Jacobi_3d
     module procedure Ellip_Jacobi_3df,  &
  &                   Ellip_Jacobi_3dd
  end interface Ellip_Jacobi_3d

contains

subroutine Ellip_GauSei_2df( x, y, rho, eps, boundary, psi, bound_opt,  &
  &                          a, b, c, d, e, f, undef, inner_bound, init_flag,  &
  &                          accel, ln, check_flag )
! ガウス=ザイデル法による楕円型方程式の求積
! 各オプション配列は, ポアソン系の各微分項の値. デフォルトはゼロで設定される.
! $$a\dfrac{\partial ^2\psi}{\partial x^2} +b\dfrac{\partial ^2\psi}{\partial x\partial y} +c\dfrac{\partial ^2\psi}{\partial y^2} +d\dfrac{\partial \psi}{\partial x} +e\dfrac{\partial \psi}{\partial y} +f\psi =\rho $$
! の各係数に対応している.
  implicit none
  real, intent(in) :: x(:)  ! 領域の横座標
  real, intent(in) :: y(:)  ! 領域の縦座標
  real, intent(in) :: rho(size(x),size(y))  ! ポアソン方程式の強制項
                   ! rho =0 でラプラス方程式も求積可能
  real, intent(in) :: eps  ! 収束条件
  character(4), intent(in) :: boundary  ! 境界条件
                ! 4 文字で各辺の境界条件を与える.
                ! 1 文字目 : x 下端, 2 文字目 : y 左端, 3 文字目 : x 上端,
                ! 4 文字目 : y 右端
                ! boundary は 1 : 固定端境界, 2 : 自由端境界, 3 : 周期境界
  real, intent(in), optional :: bound_opt(size(x),size(y))  ! 境界での強制
                             ! ノイマン境界の場合 : フラックス値
  real, intent(in), optional :: a(size(x),size(y))  ! 各微分項の係数
  real, intent(in), optional :: b(size(x),size(y))  ! 各微分項の係数
  real, intent(in), optional :: c(size(x),size(y))  ! 各微分項の係数
  real, intent(in), optional :: d(size(x),size(y))  ! 各微分項の係数
  real, intent(in), optional :: e(size(x),size(y))  ! 各微分項の係数
  real, intent(in), optional :: f(size(x),size(y))  ! 各微分項の係数
  real, intent(inout) :: psi(size(x),size(y))  ! ポアソン方程式の解
  real, intent(in), optional :: undef  ! 未定義値
  integer, intent(in), optional :: inner_bound(size(x),size(y))
                             ! 内部領域の境界. 値に応じてその格子点で境界値計算
                             ! 1 = 固定端境界, 10 = 境界の内側.
                             ! 2 = y 方向自由端境界 (フラックスは上向き)
                             ! -2 = y 方向自由端境界 (フラックスは下向き)
                             ! 4 = x 方向自由端境界 (フラックスは右向き)
                             ! -4 = x 方向自由端境界 (フラックスは左向き)
                             ! 3 = 周期境界
                             ! 8 = |_, ~| で両方とも自由境界条件
                             ! -8 = |~, _| で両方とも自由境界条件
                             ! この引数が与えられなければ全領域を計算する.
                             ! 境界の内側格子点 (10) は反復計算を行わず,
                             ! undef で設定された値もしくはゼロが入る.
                             ! このときの境界値は bound_opt の値が用いられる.
  logical, intent(in), optional :: init_flag  ! psi の値をゼロで初期化するか.
                             ! .true. = 初期化する. .false. = 初期化しない.
                             ! デフォルトでは初期化する.
  real, intent(in), optional :: accel  ! SOR の加速係数 (0 < accel < 2)
                             ! デフォルト = 1
  integer, intent(in), optional :: ln  ! 反復回数
                             ! この値が与えられるとき, eps の値に無関係に
                             ! ln 回ループさせる.
  logical, intent(in), optional :: check_flag  ! true の場合, 最高階数の
                             ! 係数が負号でも stop する.
                             ! default = .false. (係数が 0 になる場合のみ stop)
  integer :: i, j, ix, jy, nl, counter
  integer :: nx  ! x 方向の配列要素
  integer :: ny  ! y 方向の配列要素
  integer :: signb  ! 各係数を計算するかどうか
  integer, dimension(size(x),size(y)) :: ib

  real :: defun
  real :: tmp, err, err_max
  real :: tmp_b, accc
  real :: bnd(size(x),size(y))
  real :: dx(size(x)), dy(size(y)), dx2(size(x)), dy2(size(y))
  real, dimension(size(x),size(y)) :: dxdy
  real, dimension(size(x),size(y)) :: at, bt, ct, dt, et, ft
  real, dimension(size(x),size(y)) :: adp, adm, cep, cem, ac, divi

  character(4) :: bound
  logical :: sor_flag
  logical, dimension(size(x),size(y)) :: inner_flag

  bound(1:4)=boundary(1:4)

  nx=size(x)
  ny=size(y)

!-- 応答関数の初期化

  if(present(init_flag))then
     if(init_flag.eqv..true.)then
        psi = 0.0
     end if
  else
     psi = 0.0
  end if

!-- 内部境界の判別フラグの設定

  if(present(inner_bound))then
     call set_bound( bound, ib, inner_flag, inner_bound )
  else
     call set_bound( bound, ib, inner_flag )
  end if

!-- 領域・内部境界における境界値の設定

  if(present(bound_opt))then
     call setval_boundf( ib, bnd, psi, bound_opt )
  else
     call setval_boundf( ib, bnd, psi )
  end if

!-- 未定義値の設定

  if(present(undef))then
     defun=undef
  else
     defun=0.0
  end if

!-- 係数の代入
!-- a, c については, 値が入れられていなければ, 全配列に 1 を代入する.
  if(present(a))then
     call set_coef( at, ext=a )
  else
     call set_coef( at, def=1.0 )
  end if

  if(present(c))then
     call set_coef( ct, ext=c )
  else
     call set_coef( ct, def=1.0 )
  end if

  if(present(b))then
     call set_coef( bt, ext=b )
     signb=1
  else
     call set_coef( bt, def=0.0 )
     signb=0
  end if

  if(present(d))then
     call set_coef( dt, ext=d )
  else
     call set_coef( dt, def=0.0 )
  end if

  if(present(e))then
     call set_coef( et, ext=e )
  else
     call set_coef( et, def=0.0 )
  end if

  if(present(f))then
     call set_coef( ft, ext=f )
  else
     call set_coef( ft, def=0.0 )
  end if

!-- 最高階数における係数チェック. (係数がゼロでないか調べる.)

  if(present(check_flag))then
     if(check_flag.eqv..true.)then
        call check_le_coef( at(2:nx-1,2:ny-1), 0.0, undeff=defun )
        call check_le_coef( ct(2:nx-1,2:ny-1), 0.0, undeff=defun )
     else
        call check_coef( at(2:nx-1,2:ny-1), 0.0, undeff=defun )
        call check_coef( ct(2:nx-1,2:ny-1), 0.0, undeff=defun )
     end if
  else
     call check_coef( at(2:nx-1,2:ny-1), 0.0, undeff=defun )
     call check_coef( ct(2:nx-1,2:ny-1), 0.0, undeff=defun )
  end if

!-- 加速係数の判定

  if(present(accel))then
     accc=accel
     sor_flag=.true.
  else
     sor_flag=.false.
  end if

!-- ln の処理
  if(present(ln))then
     nl=ln
  else
     nl=0
  end if

!-- 以下で先に格子間隔等の 1 回計算でよいものを求めておく.
!-- これらは 1 方向のみで変化すればよい.
!-- 格子点間隔の計算
  do i=2,nx-1
     dx(i)=(x(i+1)-x(i-1))*0.5
     dx2(i)=dx(i)**2
  end do
  do j=2,ny-1
     dy(j)=(y(j+1)-y(j-1))*0.5
     dy2(j)=dy(j)**2
  end do

  dx(1)=(x(2)-x(1))
  dx(nx)=(x(nx)-x(nx-1))
  dy(1)=(y(2)-y(1))
  dy(ny)=(y(ny)-y(ny-1))

  do j=1,ny
     do i=1,nx
        dxdy(i,j)=dx(i)*dy(j)
     end do
  end do

!-- ポアソン係数の計算
!-- 実際にポアソン係数が使用されるのは, 境界を除いた 1 回り内側なので,
!-- 計算量削減のため, ループをそのようにしておく.

  ac=0.0
  adp=0.0
  adm=0.0
  cep=0.0
  cem=0.0
  bt=0.0

!-- 最高次数係数 ac の計算
!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j)

  do j=2,ny-1
     do i=2,nx-1
        ac(i,j)=1.0/(2.0*(at(i,j)/dx2(i)+ct(i,j)/dy2(j))-ft(i,j))
        adp(i,j)=(at(i,j)/(dx2(i))+0.5*dt(i,j)/dx(i))*ac(i,j)
        adm(i,j)=(at(i,j)/(dx2(i))-0.5*dt(i,j)/dx(i))*ac(i,j)
        cep(i,j)=(ct(i,j)/(dy2(j))+0.5*et(i,j)/dy(j))*ac(i,j)
        cem(i,j)=(ct(i,j)/(dy2(j))-0.5*et(i,j)/dy(j))*ac(i,j)
        bt(i,j)=0.25*bt(i,j)/(dxdy(i,j))*ac(i,j)
     end do
  end do

!$omp end do
!$omp end parallel

  err_max=eps  ! while に入るための便宜的措置
  counter=0

!-- 実際のソルバ ---
  do while(err_max>=eps)
     err_max=0.0

     do j=1,ny
        do i=1,nx

!-- 以降, 反復計算に必要な点の周囲 8 点についてそれぞれ
!-- inner_flag のチェックと同時に適切な値をそれぞれ逐次計算する,
!-- 下で, 各方向の格子計算を行っているが, ib=8,-8 は 4 隅格子のときにしか
!-- case select しない. なぜなら, 真横や上下に隅格子がくることはありえない.
           if(inner_flag(i,j).eqv..false.)then  ! .false. なら領域計算開始
              tmp=-rho(i,j)*ac(i,j)
              tmp=tmp+adp(i,j)*psi(i+1,j)  &
 &                   +adm(i,j)*psi(i-1,j)  &
 &                   +cep(i,j)*psi(i,j+1)  &
 &                   +cem(i,j)*psi(i,j-1)

              if(signb==0)then  ! そもそも bt = 0 なら計算しない.
                 tmp_b=0.0
              else
                 tmp_b=bt(i,j)*(psi(i+1,j+1)+psi(i-1,j-1)  &
  &                            -psi(i+1,j-1)-psi(i-1,j+1))
              end if

              tmp=tmp+tmp_b

           else  ! .true. なら境界計算に移行.

              select case (ib(i,j))
              case (1)
                 tmp=bnd(i,j)

              case (2)  ! x 方向にフラックス一定, 上側が参照値
                 tmp=psi(i+1,j)-bnd(i,j)*dx(i)

              case (-2)  ! x 方向にフラックス一定, 下側が参照値
                 tmp=psi(i-1,j)+bnd(i,j)*dx(i)

              case (4)  ! y 方向にフラックス一定, 右側が参照値
                 tmp=psi(i,j+1)-bnd(i,j)*dy(j)

              case (-4)  ! y 方向にフラックス一定, 左側が参照値
                 tmp=psi(i,j-1)+bnd(i,j)*dy(j)

              case (3)  ! 周期境界
                 if(i==1)then
                    ix=nx-1
                 else if(i==nx)then
                    ix=2
                 else
                    ix=i
                 end if
                 if(j==1)then
                    jy=ny-1
                 else if(j==ny)then
                    jy=2
                 else
                    jy=j
                 end if

                 tmp=psi(ix,jy)

              case (7)  ! 両方フラックス一定で内部境界限定.
                 if((ib(i+1,j+1)==10).and.(ib(i+1,j)/=10).and.  &
  &                 (ib(i,j+1)/=10))then
                    tmp=0.5*(psi(i-1,j+1)+psi(i+1,j-1)  &
  &                         +bnd(i,j+1)*dx(i)+bnd(i+1,j)*dy(j))

                 else if((ib(i-1,j-1)==10).and.(ib(i-1,j)/=10).and.  &
  &                      (ib(i,j-1)/=10))then
                    tmp=0.5*(psi(i+1,j-1)+psi(i-1,j+1)  &
  &                         -bnd(i,j-1)*dx(i)-bnd(i-1,j)*dy(j))

                 end if

              case (-7)  ! 両方フラックス一定で内部境界限定.
                 if((ib(i-1,j+1)==10).and.(ib(i-1,j)/=10).and.  &
  &                 (ib(i,j+1)/=10))then
                    tmp=0.5*(psi(i+1,j+1)+psi(i-1,j-1)  &
  &                         -bnd(i,j+1)*dx(i)+bnd(i-1,j)*dy(j))

                 else if((ib(i+1,j-1)==10).and.(ib(i+1,j)/=10).and.  &
  &                 (ib(i,j-1)/=10))then
                    tmp=0.5*(psi(i-1,j-1)+psi(i+1,j+1)  &
  &                         +bnd(i,j-1)*dx(i)-bnd(i+1,j)*dy(j))

                 end if

              case (8)  ! 両方フラックス一定で左下角か右上角, もしくは内部境界.
                 if(i==1.and.j==1)then  ! -- 評価 1
                    tmp=psi(i+1,j+1)-0.5*(bnd(i+1,j)*dy(j)+bnd(i,j+1)*dx(i))

                 else if(i==nx.and.j==ny)then  ! -- 評価 2
                    tmp=psi(i-1,j-1)+0.5*(bnd(i-1,j)*dy(j)+bnd(i,j-1)*dx(i))

                 else if(ib(i-1,j)==10.and.ib(i,j-1)==10)then
                    ! -- 評価 1 と同じ
                    tmp=psi(i+1,j+1)-0.5*(bnd(i+1,j)*dy(j)+bnd(i,j+1)*dx(i))

                 else if(ib(i+1,j)==10.and.ib(i,j+1)==10)then
                    ! -- 評価 2 と同じ
                    tmp=psi(i-1,j-1)+0.5*(bnd(i-1,j)*dy(j)+bnd(i,j-1)*dx(i))

                 end if

              case (-8)  ! 両方フラックス一定で右下角か左上角
                 if(i==1.and.j==ny)then  ! -- 評価 1
                    tmp=psi(i+1,j-1)+0.5*(bnd(i+1,j)*dy(j)-bnd(i,j-1)*dx(i))

                 else if(i==nx.and.j==1)then  ! -- 評価 2
                    tmp=psi(i-1,j+1)+0.5*(-bnd(i-1,j)*dy(j)+bnd(i,j+1)*dx(i))

                 else if(ib(i-1,j)==10.and.ib(i,j+1)==10)then
                    ! -- 評価 1 と同じ
                    tmp=psi(i+1,j-1)+0.5*(bnd(i+1,j)*dy(j)-bnd(i,j-1)*dx(i))

                 else if(ib(i+1,j)==10.and.ib(i,j-1)==10)then
                    ! -- 評価 2 と同じ
                    tmp=psi(i-1,j+1)+0.5*(-bnd(i-1,j)*dy(j)+bnd(i,j+1)*dx(i))

                 end if
              end select

           end if

           if(sor_flag.eqv..true.)then
              tmp=(1.0-accc)*psi(i,j)+tmp*accc
           end if

           err=abs(tmp-psi(i,j))

!-- 最大誤差の更新
           if(err_max<=err)then
              err_max=err
           end if

           psi(i,j)=tmp

        end do
     end do

     if(nl/=0)then
        err_max=eps
        counter=counter+1
        if(counter==nl)then
           exit
        end if
     end if

  end do

!-- 境界の設定

  call calculate_boundf( ib, dx, dy, bnd, psi )

!-- 未定義領域には undef を代入する.

  do j=1,ny
     do i=1,nx
        if(ib(i,j)==10)then
           psi(i,j)=defun
        end if
     end do
  end do

end subroutine Ellip_GauSei_2df

!----------------------------------
!----------------------------------

subroutine Ellip_GauSei_2dd( x, y, rho, eps, boundary, psi, bound_opt,  &
  &                          a, b, c, d, e, f, undef, inner_bound, init_flag,  &
  &                          accel, ln, check_flag )
! ガウス=ザイデル法による楕円型方程式の求積
! 各オプション配列は, ポアソン系の各微分項の値. デフォルトはゼロで設定される.
! $$a\dfrac{\partial ^2\psi}{\partial x^2} +b\dfrac{\partial ^2\psi}{\partial x\partial y} +c\dfrac{\partial ^2\psi}{\partial y^2} +d\dfrac{\partial \psi}{\partial x} +e\dfrac{\partial \psi}{\partial y} +f\psi =\rho $$
! の各係数に対応している.
  implicit none
  double precision, intent(in) :: x(:)  ! 領域の横座標
  double precision, intent(in) :: y(:)  ! 領域の縦座標
  double precision, intent(in) :: rho(size(x),size(y))  ! ポアソン方程式の強制項
                   ! rho =0 でラプラス方程式も求積可能
  double precision, intent(in) :: eps  ! 収束条件
  character(4), intent(in) :: boundary  ! 境界条件
                ! 4 文字で各辺の境界条件を与える.
                ! 1 文字目 : x 下端, 2 文字目 : y 左端, 3 文字目 : x 上端,
                ! 4 文字目 : y 右端
                ! boundary は 1 : 固定端境界, 2 : 自由端境界, 3 : 周期境界
  double precision, intent(in), optional :: bound_opt(size(x),size(y))  ! 境界での強制
                             ! ノイマン境界の場合 : フラックス値
  double precision, intent(in), optional :: a(size(x),size(y))  ! 各微分項の係数
  double precision, intent(in), optional :: b(size(x),size(y))  ! 各微分項の係数
  double precision, intent(in), optional :: c(size(x),size(y))  ! 各微分項の係数
  double precision, intent(in), optional :: d(size(x),size(y))  ! 各微分項の係数
  double precision, intent(in), optional :: e(size(x),size(y))  ! 各微分項の係数
  double precision, intent(in), optional :: f(size(x),size(y))  ! 各微分項の係数
  double precision, intent(inout) :: psi(size(x),size(y))  ! ポアソン方程式の解
  double precision, intent(in), optional :: undef  ! 未定義値
  integer, intent(in), optional :: inner_bound(size(x),size(y))
                             ! 内部領域の境界. 値に応じてその格子点で境界値計算
                             ! 1 = 固定端境界, 10 = 境界の内側.
                             ! 2 = y 方向自由端境界 (フラックスは上向き)
                             ! -2 = y 方向自由端境界 (フラックスは下向き)
                             ! 4 = x 方向自由端境界 (フラックスは右向き)
                             ! -4 = x 方向自由端境界 (フラックスは左向き)
                             ! 3 = 周期境界
                             ! 8 = |_, ~| で両方とも自由境界条件
                             ! -8 = |~, _| で両方とも自由境界条件
                             ! この引数が与えられなければ全領域を計算する.
                             ! 境界の内側格子点 (10) は反復計算を行わず,
                             ! undef で設定された値もしくはゼロが入る.
                             ! このときの境界値は bound_opt の値が用いられる.
  logical, intent(in), optional :: init_flag  ! psi の値をゼロで初期化するか.
                             ! .true. = 初期化する. .false. = 初期化しない.
                             ! デフォルトでは初期化する.
  double precision, intent(in), optional :: accel  ! SOR の加速係数 (0 < accel < 2)
                             ! デフォルト = 1
  integer, intent(in), optional :: ln  ! 反復回数
                             ! この値が与えられるとき, eps の値に無関係に
                             ! ln 回ループさせる.
  logical, intent(in), optional :: check_flag  ! true の場合, 最高階数の
                             ! 係数が負号でも stop する.
                             ! default = .false. (係数が 0 になる場合のみ stop)
  integer :: i, j, ix, jy, nl, counter
  integer :: nx  ! x 方向の配列要素
  integer :: ny  ! y 方向の配列要素
  integer :: signb  ! 各係数を計算するかどうか
  integer, dimension(size(x),size(y)) :: ib

  double precision :: defun
  double precision :: tmp, err, err_max
  double precision :: tmp_b, accc
  double precision :: bnd(size(x),size(y))
  double precision :: dx(size(x)), dy(size(y)), dx2(size(x)), dy2(size(y))
  double precision, dimension(size(x),size(y)) :: dxdy
  double precision, dimension(size(x),size(y)) :: at, bt, ct, dt, et, ft
  double precision, dimension(size(x),size(y)) :: adp, adm, cep, cem, ac, divi

  character(4) :: bound
  logical :: sor_flag
  logical, dimension(size(x),size(y)) :: inner_flag

  bound(1:4)=boundary(1:4)

  nx=size(x)
  ny=size(y)

!-- 応答関数の初期化

  if(present(init_flag))then
     if(init_flag.eqv..true.)then
        psi = 0.0d0
     end if
  else
     psi = 0.0d0
  end if

!-- 内部境界の判別フラグの設定

  if(present(inner_bound))then
     call set_bound( bound, ib, inner_flag, inner_bound )
  else
     call set_bound( bound, ib, inner_flag )
  end if

!-- 領域・内部境界における境界値の設定

  if(present(bound_opt))then
     call setval_boundd( ib, bnd, psi, bound_opt )
  else
     call setval_boundd( ib, bnd, psi )
  end if

!-- 未定義値の設定

  if(present(undef))then
     defun=undef
  else
     defun=0.0d0
  end if

!-- 係数の代入
!-- a, c については, 値が入れられていなければ, 全配列に 1 を代入する.
  if(present(a))then
     call set_coed( at, ext=a )
  else
     call set_coed( at, def=1.0d0 )
  end if

  if(present(c))then
     call set_coed( ct, ext=c )
  else
     call set_coed( ct, def=1.0d0 )
  end if

  if(present(b))then
     call set_coed( bt, ext=b )
     signb=1
  else
     call set_coed( bt, def=0.0d0 )
     signb=0
  end if

  if(present(d))then
     call set_coed( dt, ext=d )
  else
     call set_coed( dt, def=0.0d0 )
  end if

  if(present(e))then
     call set_coed( et, ext=e )
  else
     call set_coed( et, def=0.0d0 )
  end if

  if(present(f))then
     call set_coed( ft, ext=f )
  else
     call set_coed( ft, def=0.0d0 )
  end if

!-- 最高階数における係数チェック. (係数がゼロでないか調べる.)

  if(present(check_flag))then
     if(check_flag.eqv..true.)then
        call check_le_coed( at(2:nx-1,2:ny-1), 0.0d0, undeff=defun )
        call check_le_coed( ct(2:nx-1,2:ny-1), 0.0d0, undeff=defun )
     else
        call check_coed( at(2:nx-1,2:ny-1), 0.0d0, undeff=defun )
        call check_coed( ct(2:nx-1,2:ny-1), 0.0d0, undeff=defun )
     end if
  else
     call check_coed( at(2:nx-1,2:ny-1), 0.0d0, undeff=defun )
     call check_coed( ct(2:nx-1,2:ny-1), 0.0d0, undeff=defun )
  end if

!-- 加速係数の判定

  if(present(accel))then
     accc=accel
     sor_flag=.true.
  else
     sor_flag=.false.
  end if

!-- ln の処理
  if(present(ln))then
     nl=ln
  else
     nl=0
  end if

!-- 以下で先に格子間隔等の 1 回計算でよいものを求めておく.
!-- これらは 1 方向のみで変化すればよい.
!-- 格子点間隔の計算
  do i=2,nx-1
     dx(i)=(x(i+1)-x(i-1))*0.5d0
     dx2(i)=dx(i)**2
  end do
  do j=2,ny-1
     dy(j)=(y(j+1)-y(j-1))*0.5d0
     dy2(j)=dy(j)**2
  end do

  dx(1)=(x(2)-x(1))
  dx(nx)=(x(nx)-x(nx-1))
  dy(1)=(y(2)-y(1))
  dy(ny)=(y(ny)-y(ny-1))

  do j=1,ny
     do i=1,nx
        dxdy(i,j)=dx(i)*dy(j)
     end do
  end do

!-- ポアソン係数の計算
!-- 実際にポアソン係数が使用されるのは, 境界を除いた 1 回り内側なので,
!-- 計算量削減のため, ループをそのようにしておく.

  ac=0.0d0
  adp=0.0d0
  adm=0.0d0
  cep=0.0d0
  cem=0.0d0
  bt=0.0d0

!-- 最高次数係数 ac の計算
!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j)

  do j=2,ny-1
     do i=2,nx-1
        ac(i,j)=1.0d0/(2.0d0*(at(i,j)/dx2(i)+ct(i,j)/dy2(j))-ft(i,j))
        adp(i,j)=(at(i,j)/(dx2(i))+0.5d0*dt(i,j)/dx(i))*ac(i,j)
        adm(i,j)=(at(i,j)/(dx2(i))-0.5d0*dt(i,j)/dx(i))*ac(i,j)
        cep(i,j)=(ct(i,j)/(dy2(j))+0.5d0*et(i,j)/dy(j))*ac(i,j)
        cem(i,j)=(ct(i,j)/(dy2(j))-0.5d0*et(i,j)/dy(j))*ac(i,j)
        bt(i,j)=0.25d0*bt(i,j)/(dxdy(i,j))*ac(i,j)
     end do
  end do

!$omp end do
!$omp end parallel

  err_max=eps  ! while に入るための便宜的措置
  counter=0

!-- 実際のソルバ ---
  do while(err_max>=eps)
     err_max=0.0d0

     do j=1,ny
        do i=1,nx

!-- 以降, 反復計算に必要な点の周囲 8 点についてそれぞれ
!-- inner_flag のチェックと同時に適切な値をそれぞれ逐次計算する,
!-- 下で, 各方向の格子計算を行っているが, ib=8,-8 は 4 隅格子のときにしか
!-- case select しない. なぜなら, 真横や上下に隅格子がくることはありえない.
           if(inner_flag(i,j).eqv..false.)then  ! .false. なら領域計算開始
              tmp=-rho(i,j)*ac(i,j)
              tmp=tmp+adp(i,j)*psi(i+1,j)  &
 &                   +adm(i,j)*psi(i-1,j)  &
 &                   +cep(i,j)*psi(i,j+1)  &
 &                   +cem(i,j)*psi(i,j-1)

              if(signb==0)then  ! そもそも bt = 0 なら計算しない.
                 tmp_b=0.0d0
              else
                 tmp_b=bt(i,j)*(psi(i+1,j+1)+psi(i-1,j-1)  &
  &                            -psi(i+1,j-1)-psi(i-1,j+1))
              end if

              tmp=tmp+tmp_b

           else  ! .true. なら境界計算に移行.

              select case (ib(i,j))
              case (1)
                 tmp=bnd(i,j)

              case (2)  ! x 方向にフラックス一定, 上側が参照値
                 tmp=psi(i+1,j)-bnd(i,j)*dx(i)

              case (-2)  ! x 方向にフラックス一定, 下側が参照値
                 tmp=psi(i-1,j)+bnd(i,j)*dx(i)

              case (4)  ! y 方向にフラックス一定, 右側が参照値
                 tmp=psi(i,j+1)-bnd(i,j)*dy(j)

              case (-4)  ! y 方向にフラックス一定, 左側が参照値
                 tmp=psi(i,j-1)+bnd(i,j)*dy(j)

              case (3)  ! 周期境界
                 if(i==1)then
                    ix=nx-1
                 else if(i==nx)then
                    ix=2
                 else
                    ix=i
                 end if
                 if(j==1)then
                    jy=ny-1
                 else if(j==ny)then
                    jy=2
                 else
                    jy=j
                 end if

                 tmp=psi(ix,jy)

              case (7)  ! 両方フラックス一定で内部境界限定.
                 if((ib(i+1,j+1)==10).and.(ib(i+1,j)/=10).and.  &
  &                 (ib(i,j+1)/=10))then
                    tmp=0.5d0*(psi(i-1,j+1)+psi(i+1,j-1)  &
  &                         +bnd(i,j+1)*dx(i)+bnd(i+1,j)*dy(j))

                 else if((ib(i-1,j-1)==10).and.(ib(i-1,j)/=10).and.  &
  &                      (ib(i,j-1)/=10))then
                    tmp=0.5d0*(psi(i+1,j-1)+psi(i-1,j+1)  &
  &                         -bnd(i,j-1)*dx(i)-bnd(i-1,j)*dy(j))

                 end if

              case (-7)  ! 両方フラックス一定で内部境界限定.
                 if((ib(i-1,j+1)==10).and.(ib(i-1,j)/=10).and.  &
  &                 (ib(i,j+1)/=10))then
                    tmp=0.5d0*(psi(i+1,j+1)+psi(i-1,j-1)  &
  &                         -bnd(i,j+1)*dx(i)+bnd(i-1,j)*dy(j))

                 else if((ib(i+1,j-1)==10).and.(ib(i+1,j)/=10).and.  &
  &                 (ib(i,j-1)/=10))then
                    tmp=0.5d0*(psi(i-1,j-1)+psi(i+1,j+1)  &
  &                         +bnd(i,j-1)*dx(i)-bnd(i+1,j)*dy(j))

                 end if

              case (8)  ! 両方フラックス一定で左下角か右上角, もしくは内部境界.
                 if(i==1.and.j==1)then  ! -- 評価 1
                    tmp=psi(i+1,j+1)-0.5d0*(bnd(i+1,j)*dy(j)+bnd(i,j+1)*dx(i))

                 else if(i==nx.and.j==ny)then  ! -- 評価 2
                    tmp=psi(i-1,j-1)+0.5d0*(bnd(i-1,j)*dy(j)+bnd(i,j-1)*dx(i))

                 else if(ib(i-1,j)==10.and.ib(i,j-1)==10)then
                    ! -- 評価 1 と同じ
                    tmp=psi(i+1,j+1)-0.5d0*(bnd(i+1,j)*dy(j)+bnd(i,j+1)*dx(i))

                 else if(ib(i+1,j)==10.and.ib(i,j+1)==10)then
                    ! -- 評価 2 と同じ
                    tmp=psi(i-1,j-1)+0.5d0*(bnd(i-1,j)*dy(j)+bnd(i,j-1)*dx(i))

                 end if

              case (-8)  ! 両方フラックス一定で右下角か左上角
                 if(i==1.and.j==ny)then  ! -- 評価 1
                    tmp=psi(i+1,j-1)+0.5d0*(bnd(i+1,j)*dy(j)-bnd(i,j-1)*dx(i))

                 else if(i==nx.and.j==1)then  ! -- 評価 2
                    tmp=psi(i-1,j+1)+0.5d0*(-bnd(i-1,j)*dy(j)+bnd(i,j+1)*dx(i))

                 else if(ib(i-1,j)==10.and.ib(i,j+1)==10)then
                    ! -- 評価 1 と同じ
                    tmp=psi(i+1,j-1)+0.5d0*(bnd(i+1,j)*dy(j)-bnd(i,j-1)*dx(i))

                 else if(ib(i+1,j)==10.and.ib(i,j-1)==10)then
                    ! -- 評価 2 と同じ
                    tmp=psi(i-1,j+1)+0.5d0*(-bnd(i-1,j)*dy(j)+bnd(i,j+1)*dx(i))

                 end if
              end select

           end if

           if(sor_flag.eqv..true.)then
              tmp=(1.0d0-accc)*psi(i,j)+tmp*accc
           end if

           err=abs(tmp-psi(i,j))

!-- 最大誤差の更新
           if(err_max<=err)then
              err_max=err
           end if

           psi(i,j)=tmp

        end do
     end do

     if(nl/=0)then
        err_max=eps
        counter=counter+1
        if(counter==nl)then
           exit
        end if
     end if

  end do

!-- 境界の設定

  call calculate_boundd( ib, dx, dy, bnd, psi )

!-- 未定義領域には undef を代入する.

  do j=1,ny
     do i=1,nx
        if(ib(i,j)==10)then
           psi(i,j)=defun
        end if
     end do
  end do

end subroutine Ellip_GauSei_2dd

!----------------------------------
!----------------------------------

subroutine Ellip_Jacobi_2df( x, y, rho, eps, boundary, psi, bound_opt,  &
  &                          a, b, c, d, e, f, undef, inner_bound, init_flag,  &
  &                          accel, ln, check_flag )
  ! openmp によるスレッド並列が可能.
  ! ガウス・ザイデル法ではアルゴリズムの点から並列化が困難と思われたので,
  ! 並列計算によるポアソン方程式の求積が必要となるなら,
  ! ヤコビ法のものを使用されたい.
  implicit none
  real, intent(in) :: x(:)  ! 領域の横座標
  real, intent(in) :: y(:)  ! 領域の縦座標
  real, intent(in) :: rho(size(x),size(y))  ! ポアソン方程式の強制項
                   ! rho =0 でラプラス方程式も求積可能
  real, intent(in) :: eps  ! 収束条件
  character(4), intent(in) :: boundary  ! 境界条件
                ! 4 文字で各辺の境界条件を与える.
                ! 1 文字目 : x 下端, 2 文字目 : y 左端, 3 文字目 : x 上端,
                ! 4 文字目 : y 右端
                ! boundary は 1 : 固定端境界, 2 : 自由端境界, 3 : 周期境界
  real, intent(in), optional :: bound_opt(size(x),size(y))  ! 境界での強制
                             ! ノイマン境界の場合 : フラックス値
  real, intent(in), optional :: a(size(x),size(y))  ! 各微分項の係数
  real, intent(in), optional :: b(size(x),size(y))  ! 各微分項の係数
  real, intent(in), optional :: c(size(x),size(y))  ! 各微分項の係数
  real, intent(in), optional :: d(size(x),size(y))  ! 各微分項の係数
  real, intent(in), optional :: e(size(x),size(y))  ! 各微分項の係数
  real, intent(in), optional :: f(size(x),size(y))  ! 各微分項の係数
  real, intent(in), optional :: undef  ! 未定義値
  integer, intent(in), optional :: inner_bound(size(x),size(y))
                             ! 内部領域の境界. 値に応じてその格子点で境界値計算
                             ! 1 = 固定端境界, 10 = 境界の内側.
                             ! 2 = y 方向自由端境界 (フラックスは上向き)
                             ! -2 = y 方向自由端境界 (フラックスは下向き)
                             ! 4 = x 方向自由端境界 (フラックスは右向き)
                             ! -4 = x 方向自由端境界 (フラックスは左向き)
                             ! 3 = 周期境界, -3 = 隅領域で両方とも周期境界
                             ! 8 = |_, ~| で両方とも自由境界条件
                             ! -8 = |~, _| で両方とも自由境界条件
                             ! この引数が与えられなければ全領域を計算する.
                             ! 境界の内側格子点 (10) は反復計算を行わず,
                             ! undef で設定された値もしくはゼロが入る.
                             ! このときの境界値は bound_opt の値が用いられる.
  real, intent(inout) :: psi(size(x),size(y))  ! ポアソン方程式の解
  logical, intent(in), optional :: init_flag  ! psi の値をゼロで初期化するか.
                             ! .true. = 初期化する. .false. = 初期化しない.
                             ! デフォルトでは初期化する.
  real, intent(in), optional :: accel  ! SOR の加速係数 (0 < accel < 2)
                             ! デフォルト = 1
  integer, intent(in), optional :: ln  ! 反復回数
                             ! この値が与えられるとき, eps の値に無関係に
                             ! ln 回ループさせる.
  logical, intent(in), optional :: check_flag  ! true の場合, 最高階数の
                             ! 係数が負号でも stop する.
                             ! default = .false. (係数が 0 になる場合のみ stop)

  integer :: i, j, ix, jy, nl, counter
  integer :: nx  ! x 方向の配列要素
  integer :: ny  ! y 方向の配列要素
  integer :: signb  ! クロスターム b が存在するか
  integer, dimension(size(x),size(y)) :: ib

  real :: defun
  real :: err, err_max, accc, emax
  real :: bnd(size(x),size(y))
  real :: dx(size(x)), dy(size(y)), dx2(size(x)), dy2(size(y))
  real, dimension(size(x),size(y)) :: dxdy
  real, dimension(size(x),size(y)) :: at, bt, ct, dt, et, ft
  real, dimension(size(x),size(y)) :: adp, adm, cep, cem, ac
  real, dimension(size(x),size(y)) :: tmp, tmp_b, divi

  character(4) :: bound
  logical :: sor_flag
  logical, dimension(size(x),size(y)) :: inner_flag

  bound(1:4)=boundary(1:4)

  nx=size(x)
  ny=size(y)

!-- 応答関数の初期化

  if(present(init_flag))then
     if(init_flag.eqv..true.)then
        psi = 0.0
     end if
  else
     psi = 0.0
  end if

!-- 内部境界の判別フラグの設定

  if(present(inner_bound))then
     call set_bound( bound, ib, inner_flag, inner_bound )
  else
     call set_bound( bound, ib, inner_flag )
  end if

!-- 領域・内部境界における境界値の設定

  if(present(bound_opt))then
     call setval_boundf( ib, bnd, psi, bound_opt )
  else
     call setval_boundf( ib, bnd, psi )
  end if

!-- 未定義値の設定

  if(present(undef))then
     defun=undef
  else
     defun=0.0
  end if

!-- 係数の代入
!-- a, c については, 値が入れられていなければ, 全配列に 1 を代入する.

  if(present(a))then
     call set_coef( at, ext=a )
  else
     call set_coef( at, def=1.0 )
  end if

  if(present(c))then
     call set_coef( ct, ext=c )
  else
     call set_coef( ct, def=1.0 )
  end if

  if(present(b))then
     call set_coef( bt, ext=b )
     signb=1
  else
     call set_coef( bt, def=0.0 )
     signb=0
  end if

  if(present(d))then
     call set_coef( dt, ext=d )
  else
     call set_coef( dt, def=0.0 )
  end if

  if(present(e))then
     call set_coef( et, ext=e )
  else
     call set_coef( et, def=0.0 )
  end if

  if(present(f))then
     call set_coef( ft, ext=f )
  else
     call set_coef( ft, def=0.0 )
  end if

!-- 最高階数における係数チェック. (係数がゼロでないか調べる.)

  if(present(check_flag))then
     if(check_flag.eqv..true.)then
        call check_le_coef( at(2:nx-1,2:ny-1), 0.0, undeff=defun )
        call check_le_coef( ct(2:nx-1,2:ny-1), 0.0, undeff=defun )
     else
        call check_coef( at(2:nx-1,2:ny-1), 0.0, undeff=defun )
        call check_coef( ct(2:nx-1,2:ny-1), 0.0, undeff=defun )
     end if
  else
     call check_coef( at(2:nx-1,2:ny-1), 0.0, undeff=defun )
     call check_coef( ct(2:nx-1,2:ny-1), 0.0, undeff=defun )
  end if

!-- 加速係数の判定

  if(present(accel))then
     accc=accel
     sor_flag=.true.
  else
     sor_flag=.false.
  end if

!-- ln の処理
  if(present(ln))then
     nl=ln
  else
     nl=0
  end if

!-- 以下で先に格子間隔等の 1 回計算でよいものを求めておく.
!-- これらは 1 方向のみで変化すればよい.
!-- 格子点間隔の計算
  do i=2,nx-1
     dx(i)=(x(i+1)-x(i-1))*0.5
     dx2(i)=dx(i)**2
  end do
  do j=2,ny-1
     dy(j)=(y(j+1)-y(j-1))*0.5
     dy2(j)=dy(j)**2
  end do

  dx(1)=(x(2)-x(1))
  dx(nx)=(x(nx)-x(nx-1))
  dy(1)=(y(2)-y(1))
  dy(ny)=(y(ny)-y(ny-1))

  do j=1,ny
     do i=1,nx
        dxdy(i,j)=dx(i)*dy(j)
     end do
  end do

!-- ポアソン係数の計算
!-- 実際にポアソン係数が使用されるのは, 境界を除いた 1 回り内側なので,
!-- 計算量削減のため, ループをそのようにしておく.

  ac=0.0
  adp=0.0
  adm=0.0
  cep=0.0
  cem=0.0
  bt=0.0

!-- 最高次数係数 ac の計算
!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j)

  do j=2,ny-1
     do i=2,nx-1
        ac(i,j)=1.0/(2.0*(at(i,j)/dx2(i)+ct(i,j)/dy2(j))-ft(i,j))
        adp(i,j)=(at(i,j)/(dx2(i))+0.5*dt(i,j)/dx(i))*ac(i,j)
        adm(i,j)=(at(i,j)/(dx2(i))-0.5*dt(i,j)/dx(i))*ac(i,j)
        cep(i,j)=(ct(i,j)/(dy2(j))+0.5*et(i,j)/dy(j))*ac(i,j)
        cem(i,j)=(ct(i,j)/(dy2(j))-0.5*et(i,j)/dy(j))*ac(i,j)
        bt(i,j)=0.25*bt(i,j)/(dxdy(i,j))*ac(i,j)
     end do
  end do

!$omp end do
!$omp end parallel

  err_max=eps  ! while に入るための便宜的措置
  counter=0

!-- 実際のソルバ ---
  do while(err_max>=eps)
     err_max=0.0
!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,ix,jy)
     do j=1,ny
        do i=1,nx

!-- 以降, 反復計算に必要な点の周囲 8 点についてそれぞれ
!-- inner_flag のチェックと同時に適切な値をそれぞれ逐次計算する,
!-- 下で, 各方向の格子計算を行っているが, ib=8,-8 は 4 隅格子のときにしか
!-- case select しない. なぜなら, 真横や上下に隅格子がくることはありえない.
           if(inner_flag(i,j).eqv..false.)then  ! .false. なら領域計算開始
              tmp(i,j)=-rho(i,j)*ac(i,j)
              tmp(i,j)=tmp(i,j)+adp(i,j)*psi(i+1,j)  &
 &                             +adm(i,j)*psi(i-1,j)  &
 &                             +cep(i,j)*psi(i,j+1)  &
 &                             +cem(i,j)*psi(i,j-1)
              if(signb==0)then  ! そもそも bt = 0 なら計算しない.
                  tmp_b(i,j)=0.0
              else
                  tmp_b(i,j)=bt(i,j)*(psi(i+1,j+1)+psi(i-1,j-1)  &
  &                             -psi(i+1,j-1)-psi(i-1,j+1))
              end if

              tmp(i,j)=tmp(i,j)+tmp_b(i,j)

           else  ! .true. なら境界計算に移行.

              select case (ib(i,j))
              case (1)
                 tmp(i,j)=bnd(i,j)

              case (2)  ! x 方向にフラックス一定, 上側が参照値
                 tmp(i,j)=psi(i+1,j)-bnd(i,j)*dx(i)

              case (-2)  ! x 方向にフラックス一定, 下側が参照値
                 tmp(i,j)=psi(i-1,j)+bnd(i,j)*dx(i)

              case (4)  ! y 方向にフラックス一定, 右側が参照値
                 tmp(i,j)=psi(i,j+1)-bnd(i,j)*dy(j)

              case (-4)  ! y 方向にフラックス一定, 左側が参照値
                 tmp(i,j)=psi(i,j-1)+bnd(i,j)*dy(j)

              case (3)  ! 周期境界
                 if(i==1)then
                    ix=nx-1
                 else if(i==nx)then
                    ix=2
                 else
                    ix=i
                 end if
                 if(j==1)then
                    jy=ny-1
                 else if(j==ny)then
                    jy=2
                 else
                    jy=j
                 end if

                 tmp(i,j)=psi(ix,jy)

              case (7)  ! 両方フラックス一定で内部境界限定.
                 if((ib(i+1,j+1)==10).and.(ib(i+1,j)/=10).and.  &
  &                 (ib(i,j+1)/=10))then
                    tmp(i,j)=0.5*(psi(i-1,j)+psi(i,j-1))  &
  &                          +0.5*bnd(i,j)*(dy(j)+dx(i))

                 else if((ib(i-1,j-1)==10).and.(ib(i-1,j)/=10).and.  &
  &                      (ib(i,j-1)/=10))then
                    tmp(i,j)=0.5*(psi(i+1,j)+psi(i,j+1))  &
  &                          -0.5*bnd(i,j)*(dy(j)+dx(i))

                 end if

              case (-7)  ! 両方フラックス一定で内部境界限定.
                 if((ib(i-1,j+1)==10).and.(ib(i-1,j)/=10).and.  &
  &                 (ib(i,j+1)/=10))then
                    tmp(i,j)=0.5*(psi(i+1,j)+psi(i,j-1))  &
  &                          +0.5*bnd(i,j)*(dy(j)-dx(i))

                 else if((ib(i+1,j-1)==10).and.(ib(i+1,j)/=10).and.  &
  &                 (ib(i,j-1)/=10))then
                    tmp(i,j)=0.5*(psi(i-1,j)+psi(i,j+1))  &
  &                          +0.5*bnd(i,j)*(-dy(j)+dx(i))

                 end if

              case (8)  ! 両方フラックス一定で左下角か右上角, もしくは内部境界.
                 if(i==1.and.j==1)then  ! -- 評価 1
                    tmp(i,j)=psi(i+1,j+1)-0.5*(bnd(i+1,j)*dy(j)+bnd(i,j+1)*dx(i))

                 else if(i==nx.and.j==ny)then  ! -- 評価 2
                    tmp(i,j)=psi(i-1,j-1)+0.5*(bnd(i-1,j)*dy(j)+bnd(i,j-1)*dx(i))

                 else if(ib(i-1,j)==10.and.ib(i,j-1)==10)then
                    ! -- 評価 1 と同じ
                    tmp(i,j)=psi(i+1,j+1)-0.5*(bnd(i+1,j)*dy(j)+bnd(i,j+1)*dx(i))

                 else if(ib(i+1,j)==10.and.ib(i,j+1)==10)then
                    ! -- 評価 2 と同じ
                    tmp(i,j)=psi(i-1,j-1)+0.5*(bnd(i-1,j)*dy(j)+bnd(i,j-1)*dx(i))

                 end if

              case (-8)  ! 両方フラックス一定で右下角か左上角
                 if(i==1.and.j==ny)then  ! -- 評価 1
                    tmp(i,j)=psi(i+1,j-1)+0.5*(bnd(i+1,j)*dy(j)-bnd(i,j-1)*dx(i))

                 else if(i==nx.and.j==1)then  ! -- 評価 2
                    tmp(i,j)=psi(i-1,j+1)+0.5*(-bnd(i-1,j)*dy(j)+bnd(i,j+1)*dx(i))

                 else if(ib(i-1,j)==10.and.ib(i,j+1)==10)then
                    ! -- 評価 1 と同じ
                    tmp(i,j)=psi(i+1,j-1)+0.5*(bnd(i+1,j)*dy(j)-bnd(i,j-1)*dx(i))

                 else if(ib(i+1,j)==10.and.ib(i,j-1)==10)then
                    ! -- 評価 2 と同じ
                    tmp(i,j)=psi(i-1,j+1)+0.5*(-bnd(i-1,j)*dy(j)+bnd(i,j+1)*dx(i))

                 end if
              end select

           end if

           if(sor_flag.eqv..true.)then
              tmp(i,j)=(1.0-accc)*psi(i,j)+tmp(i,j)*accc
           end if

        end do
     end do
!$omp end do
!$omp end parallel

!!-- ここまでは if 文のインデントを調節しない.
!        end if
!
!        tmp=tmp/ac(i,j)
!-- 誤差の計算 ---
     do j=2,ny-1
        do i=2,nx-1
           err=abs(tmp(i,j)-psi(i,j))

!-- 最大誤差の更新
           if(err_max<=err)then
              err_max=err
           end if
        end do
     end do

!     write(*,*) "*** MESSAGE (Ellip_Slv) *** : error max is", err, err_max

!-- 一斉更新

     do j=1,ny
        do i=1,nx
           psi(i,j)=tmp(i,j)
        end do
     end do

     if(nl/=0)then
        counter=counter+1
        if(counter==nl)then
           exit
        else
           err_max=eps
        end if
     end if

!!-- 特異摂動回避処理
!     if(present(epsm))then
!        emax=check_diff_error_2d( x, y, psi, rho, at, bt, ct, dt, et, ft, ib )
!        if(emax<=epsm)then
!           exit
!        end if
!     end if

  end do
  
!-- 境界の設定

  call calculate_boundf( ib, dx, dy, bnd, psi )

!-- 未定義領域には undef を代入する.

  do j=1,ny
     do i=1,nx
        if(ib(i,j)==10)then
           psi(i,j)=defun
        end if
     end do
  end do

end subroutine Ellip_Jacobi_2df

!----------------------------------
!----------------------------------

subroutine Ellip_Jacobi_2dd( x, y, rho, eps, boundary, psi, bound_opt,  &
  &                          a, b, c, d, e, f, undef, inner_bound, init_flag,  &
  &                          accel, ln, check_flag )
  ! openmp によるスレッド並列が可能.
  ! ガウス・ザイデル法ではアルゴリズムの点から並列化が困難と思われたので,
  ! 並列計算によるポアソン方程式の求積が必要となるなら,
  ! ヤコビ法のものを使用されたい.
  implicit none
  double precision, intent(in) :: x(:)  ! 領域の横座標
  double precision, intent(in) :: y(:)  ! 領域の縦座標
  double precision, intent(in) :: rho(size(x),size(y))  ! ポアソン方程式の強制項
                   ! rho =0 でラプラス方程式も求積可能
  double precision, intent(in) :: eps  ! 収束条件
  character(4), intent(in) :: boundary  ! 境界条件
                ! 4 文字で各辺の境界条件を与える.
                ! 1 文字目 : x 下端, 2 文字目 : y 左端, 3 文字目 : x 上端,
                ! 4 文字目 : y 右端
                ! boundary は 1 : 固定端境界, 2 : 自由端境界, 3 : 周期境界
  double precision, intent(in), optional :: bound_opt(size(x),size(y))  ! 境界での強制
                             ! ノイマン境界の場合 : フラックス値
  double precision, intent(in), optional :: a(size(x),size(y))  ! 各微分項の係数
  double precision, intent(in), optional :: b(size(x),size(y))  ! 各微分項の係数
  double precision, intent(in), optional :: c(size(x),size(y))  ! 各微分項の係数
  double precision, intent(in), optional :: d(size(x),size(y))  ! 各微分項の係数
  double precision, intent(in), optional :: e(size(x),size(y))  ! 各微分項の係数
  double precision, intent(in), optional :: f(size(x),size(y))  ! 各微分項の係数
  double precision, intent(in), optional :: undef  ! 未定義値
  integer, intent(in), optional :: inner_bound(size(x),size(y))
                             ! 内部領域の境界. 値に応じてその格子点で境界値計算
                             ! 1 = 固定端境界, 10 = 境界の内側.
                             ! 2 = y 方向自由端境界 (フラックスは上向き)
                             ! -2 = y 方向自由端境界 (フラックスは下向き)
                             ! 4 = x 方向自由端境界 (フラックスは右向き)
                             ! -4 = x 方向自由端境界 (フラックスは左向き)
                             ! 3 = 周期境界, -3 = 隅領域で両方とも周期境界
                             ! 8 = |_, ~| で両方とも自由境界条件
                             ! -8 = |~, _| で両方とも自由境界条件
                             ! この引数が与えられなければ全領域を計算する.
                             ! 境界の内側格子点 (10) は反復計算を行わず,
                             ! undef で設定された値もしくはゼロが入る.
                             ! このときの境界値は bound_opt の値が用いられる.
  double precision, intent(inout) :: psi(size(x),size(y))  ! ポアソン方程式の解
  logical, intent(in), optional :: init_flag  ! psi の値をゼロで初期化するか.
                             ! .true. = 初期化する. .false. = 初期化しない.
                             ! デフォルトでは初期化する.
  double precision, intent(in), optional :: accel  ! SOR の加速係数 (0 < accel < 2)
                             ! デフォルト = 1
  integer, intent(in), optional :: ln  ! 反復回数
                             ! この値が与えられるとき, eps の値に無関係に
                             ! ln 回ループさせる.
  logical, intent(in), optional :: check_flag  ! true の場合, 最高階数の
                             ! 係数が負号でも stop する.
                             ! default = .false. (係数が 0 になる場合のみ stop)

  integer :: i, j, ix, jy, nl, counter
  integer :: nx  ! x 方向の配列要素
  integer :: ny  ! y 方向の配列要素
  integer :: signb  ! クロスターム b が存在するか
  integer, dimension(size(x),size(y)) :: ib

  double precision :: defun
  double precision :: err, err_max, accc, emax
  double precision :: bnd(size(x),size(y))
  double precision :: dx(size(x)), dy(size(y)), dx2(size(x)), dy2(size(y))
  double precision, dimension(size(x),size(y)) :: dxdy
  double precision, dimension(size(x),size(y)) :: at, bt, ct, dt, et, ft
  double precision, dimension(size(x),size(y)) :: adp, adm, cep, cem, ac
  double precision, dimension(size(x),size(y)) :: tmp, tmp_b, divi

  character(4) :: bound
  logical :: sor_flag
  logical, dimension(size(x),size(y)) :: inner_flag

  bound(1:4)=boundary(1:4)

  nx=size(x)
  ny=size(y)

!-- 応答関数の初期化

  if(present(init_flag))then
     if(init_flag.eqv..true.)then
        psi = 0.0d0
     end if
  else
     psi = 0.0d0
  end if

!-- 内部境界の判別フラグの設定

  if(present(inner_bound))then
     call set_bound( bound, ib, inner_flag, inner_bound )
  else
     call set_bound( bound, ib, inner_flag )
  end if

!-- 領域・内部境界における境界値の設定

  if(present(bound_opt))then
     call setval_boundd( ib, bnd, psi, bound_opt )
  else
     call setval_boundd( ib, bnd, psi )
  end if

!-- 未定義値の設定

  if(present(undef))then
     defun=undef
  else
     defun=0.0d0
  end if

!-- 係数の代入
!-- a, c については, 値が入れられていなければ, 全配列に 1 を代入する.

  if(present(a))then
     call set_coed( at, ext=a )
  else
     call set_coed( at, def=1.0d0 )
  end if

  if(present(c))then
     call set_coed( ct, ext=c )
  else
     call set_coed( ct, def=1.0d0 )
  end if

  if(present(b))then
     call set_coed( bt, ext=b )
     signb=1
  else
     call set_coed( bt, def=0.0d0 )
     signb=0
  end if

  if(present(d))then
     call set_coed( dt, ext=d )
  else
     call set_coed( dt, def=0.0d0 )
  end if

  if(present(e))then
     call set_coed( et, ext=e )
  else
     call set_coed( et, def=0.0d0 )
  end if

  if(present(f))then
     call set_coed( ft, ext=f )
  else
     call set_coed( ft, def=0.0d0 )
  end if

!-- 最高階数における係数チェック. (係数がゼロでないか調べる.)

  if(present(check_flag))then
     if(check_flag.eqv..true.)then
        call check_le_coed( at(2:nx-1,2:ny-1), 0.0d0, undeff=defun )
        call check_le_coed( ct(2:nx-1,2:ny-1), 0.0d0, undeff=defun )
     else
        call check_coed( at(2:nx-1,2:ny-1), 0.0d0, undeff=defun )
        call check_coed( ct(2:nx-1,2:ny-1), 0.0d0, undeff=defun )
     end if
  else
     call check_coed( at(2:nx-1,2:ny-1), 0.0d0, undeff=defun )
     call check_coed( ct(2:nx-1,2:ny-1), 0.0d0, undeff=defun )
  end if

!-- 加速係数の判定

  if(present(accel))then
     accc=accel
     sor_flag=.true.
  else
     sor_flag=.false.
  end if

!-- ln の処理
  if(present(ln))then
     nl=ln
  else
     nl=0
  end if

!-- 以下で先に格子間隔等の 1 回計算でよいものを求めておく.
!-- これらは 1 方向のみで変化すればよい.
!-- 格子点間隔の計算
  do i=2,nx-1
     dx(i)=(x(i+1)-x(i-1))*0.5d0
     dx2(i)=dx(i)**2
  end do
  do j=2,ny-1
     dy(j)=(y(j+1)-y(j-1))*0.5d0
     dy2(j)=dy(j)**2
  end do

  dx(1)=(x(2)-x(1))
  dx(nx)=(x(nx)-x(nx-1))
  dy(1)=(y(2)-y(1))
  dy(ny)=(y(ny)-y(ny-1))

  do j=1,ny
     do i=1,nx
        dxdy(i,j)=dx(i)*dy(j)
     end do
  end do

!-- ポアソン係数の計算
!-- 実際にポアソン係数が使用されるのは, 境界を除いた 1 回り内側なので,
!-- 計算量削減のため, ループをそのようにしておく.

  ac=0.0d0
  adp=0.0d0
  adm=0.0d0
  cep=0.0d0
  cem=0.0d0
  bt=0.0d0

!-- 最高次数係数 ac の計算
!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j)

  do j=2,ny-1
     do i=2,nx-1
        ac(i,j)=1.0d0/(2.0d0*(at(i,j)/dx2(i)+ct(i,j)/dy2(j))-ft(i,j))
        adp(i,j)=(at(i,j)/(dx2(i))+0.5d0*dt(i,j)/dx(i))*ac(i,j)
        adm(i,j)=(at(i,j)/(dx2(i))-0.5d0*dt(i,j)/dx(i))*ac(i,j)
        cep(i,j)=(ct(i,j)/(dy2(j))+0.5d0*et(i,j)/dy(j))*ac(i,j)
        cem(i,j)=(ct(i,j)/(dy2(j))-0.5d0*et(i,j)/dy(j))*ac(i,j)
        bt(i,j)=0.25d0*bt(i,j)/(dxdy(i,j))*ac(i,j)
     end do
  end do

!$omp end do
!$omp end parallel

  err_max=eps  ! while に入るための便宜的措置
  counter=0

!-- 実際のソルバ ---
  do while(err_max>=eps)
     err_max=0.0d0
!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,ix,jy)
     do j=1,ny
        do i=1,nx

!-- 以降, 反復計算に必要な点の周囲 8 点についてそれぞれ
!-- inner_flag のチェックと同時に適切な値をそれぞれ逐次計算する,
!-- 下で, 各方向の格子計算を行っているが, ib=8,-8 は 4 隅格子のときにしか
!-- case select しない. なぜなら, 真横や上下に隅格子がくることはありえない.
           if(inner_flag(i,j).eqv..false.)then  ! .false. なら領域計算開始
              tmp(i,j)=-rho(i,j)*ac(i,j)
              tmp(i,j)=tmp(i,j)+adp(i,j)*psi(i+1,j)  &
 &                             +adm(i,j)*psi(i-1,j)  &
 &                             +cep(i,j)*psi(i,j+1)  &
 &                             +cem(i,j)*psi(i,j-1)
              if(signb==0)then  ! そもそも bt = 0 なら計算しない.
                  tmp_b(i,j)=0.0d0
              else
                  tmp_b(i,j)=bt(i,j)*(psi(i+1,j+1)+psi(i-1,j-1)  &
  &                             -psi(i+1,j-1)-psi(i-1,j+1))
              end if

              tmp(i,j)=tmp(i,j)+tmp_b(i,j)

           else  ! .true. なら境界計算に移行.

              select case (ib(i,j))
              case (1)
                 tmp(i,j)=bnd(i,j)

              case (2)  ! x 方向にフラックス一定, 上側が参照値
                 tmp(i,j)=psi(i+1,j)-bnd(i,j)*dx(i)

              case (-2)  ! x 方向にフラックス一定, 下側が参照値
                 tmp(i,j)=psi(i-1,j)+bnd(i,j)*dx(i)

              case (4)  ! y 方向にフラックス一定, 右側が参照値
                 tmp(i,j)=psi(i,j+1)-bnd(i,j)*dy(j)

              case (-4)  ! y 方向にフラックス一定, 左側が参照値
                 tmp(i,j)=psi(i,j-1)+bnd(i,j)*dy(j)

              case (3)  ! 周期境界
                 if(i==1)then
                    ix=nx-1
                 else if(i==nx)then
                    ix=2
                 else
                    ix=i
                 end if
                 if(j==1)then
                    jy=ny-1
                 else if(j==ny)then
                    jy=2
                 else
                    jy=j
                 end if

                 tmp(i,j)=psi(ix,jy)

              case (7)  ! 両方フラックス一定で内部境界限定.
                 if((ib(i+1,j+1)==10).and.(ib(i+1,j)/=10).and.  &
  &                 (ib(i,j+1)/=10))then
                    tmp(i,j)=0.5d0*(psi(i-1,j)+psi(i,j-1))  &
  &                          +0.5d0*bnd(i,j)*(dy(j)+dx(i))

                 else if((ib(i-1,j-1)==10).and.(ib(i-1,j)/=10).and.  &
  &                      (ib(i,j-1)/=10))then
                    tmp(i,j)=0.5d0*(psi(i+1,j)+psi(i,j+1))  &
  &                          -0.5d0*bnd(i,j)*(dy(j)+dx(i))

                 end if

              case (-7)  ! 両方フラックス一定で内部境界限定.
                 if((ib(i-1,j+1)==10).and.(ib(i-1,j)/=10).and.  &
  &                 (ib(i,j+1)/=10))then
                    tmp(i,j)=0.5d0*(psi(i+1,j)+psi(i,j-1))  &
  &                          +0.5d0*bnd(i,j)*(dy(j)-dx(i))

                 else if((ib(i+1,j-1)==10).and.(ib(i+1,j)/=10).and.  &
  &                 (ib(i,j-1)/=10))then
                    tmp(i,j)=0.5d0*(psi(i-1,j)+psi(i,j+1))  &
  &                          +0.5d0*bnd(i,j)*(-dy(j)+dx(i))

                 end if

              case (8)  ! 両方フラックス一定で左下角か右上角, もしくは内部境界.
                 if(i==1.and.j==1)then  ! -- 評価 1
                    tmp(i,j)=psi(i+1,j+1)-0.5d0*(bnd(i+1,j)*dy(j)+bnd(i,j+1)*dx(i))

                 else if(i==nx.and.j==ny)then  ! -- 評価 2
                    tmp(i,j)=psi(i-1,j-1)+0.5d0*(bnd(i-1,j)*dy(j)+bnd(i,j-1)*dx(i))

                 else if(ib(i-1,j)==10.and.ib(i,j-1)==10)then
                    ! -- 評価 1 と同じ
                    tmp(i,j)=psi(i+1,j+1)-0.5d0*(bnd(i+1,j)*dy(j)+bnd(i,j+1)*dx(i))

                 else if(ib(i+1,j)==10.and.ib(i,j+1)==10)then
                    ! -- 評価 2 と同じ
                    tmp(i,j)=psi(i-1,j-1)+0.5d0*(bnd(i-1,j)*dy(j)+bnd(i,j-1)*dx(i))

                 end if

              case (-8)  ! 両方フラックス一定で右下角か左上角
                 if(i==1.and.j==ny)then  ! -- 評価 1
                    tmp(i,j)=psi(i+1,j-1)+0.5d0*(bnd(i+1,j)*dy(j)-bnd(i,j-1)*dx(i))

                 else if(i==nx.and.j==1)then  ! -- 評価 2
                    tmp(i,j)=psi(i-1,j+1)+0.5d0*(-bnd(i-1,j)*dy(j)+bnd(i,j+1)*dx(i))

                 else if(ib(i-1,j)==10.and.ib(i,j+1)==10)then
                    ! -- 評価 1 と同じ
                    tmp(i,j)=psi(i+1,j-1)+0.5d0*(bnd(i+1,j)*dy(j)-bnd(i,j-1)*dx(i))

                 else if(ib(i+1,j)==10.and.ib(i,j-1)==10)then
                    ! -- 評価 2 と同じ
                    tmp(i,j)=psi(i-1,j+1)+0.5d0*(-bnd(i-1,j)*dy(j)+bnd(i,j+1)*dx(i))

                 end if
              end select

           end if

           if(sor_flag.eqv..true.)then
              tmp(i,j)=(1.0d0-accc)*psi(i,j)+tmp(i,j)*accc
           end if

        end do
     end do
!$omp end do
!$omp end parallel

!!-- ここまでは if 文のインデントを調節しない.
!        end if
!
!        tmp=tmp/ac(i,j)
!-- 誤差の計算 ---
     do j=2,ny-1
        do i=2,nx-1
           err=abs(tmp(i,j)-psi(i,j))

!-- 最大誤差の更新
           if(err_max<=err)then
              err_max=err
           end if
        end do
     end do

!     write(*,*) "*** MESSAGE (Ellip_Slv) *** : error max is", err, err_max

!-- 一斉更新

     do j=1,ny
        do i=1,nx
           psi(i,j)=tmp(i,j)
        end do
     end do

     if(nl/=0)then
        counter=counter+1
        if(counter==nl)then
           exit
        else
           err_max=eps
        end if
     end if

!!-- 特異摂動回避処理
!     if(present(epsm))then
!        emax=check_diff_error_2d( x, y, psi, rho, at, bt, ct, dt, et, ft, ib )
!        if(emax<=epsm)then
!           exit
!        end if
!     end if

  end do
  
!-- 境界の設定

  call calculate_boundd( ib, dx, dy, bnd, psi )

!-- 未定義領域には undef を代入する.

  do j=1,ny
     do i=1,nx
        if(ib(i,j)==10)then
           psi(i,j)=defun
        end if
     end do
  end do

end subroutine Ellip_Jacobi_2dd

!----------------------------------
!----------------------------------

subroutine Ellip_GauSei_3df(x, y, z, rho, eps, boundary, psi, bound_opt,  &
  &                         xa, ya, za, a, b, c, d, e, f, g,  &
  &                         undef, inner_bound, init_flag, accel, ln, check_flag )
! ガウス=ザイデル法による楕円型方程式の求積
! 各オプション配列は, ポアソン系の各微分項の値. デフォルトはゼロで設定される.
! $$xa\dfrac{\partial ^2\psi}{\partial x^2} +ya\dfrac{\partial ^2\psi}{\partial y^2} +za\dfrac{\partial ^2\psi}{\partial z^2} +a\dfrac{\partial ^2\psi}{\partial x\partial y} +b\dfrac{\partial ^2\psi}{\partial y\partial z} +c\dfrac{\partial ^2\psi}{\partial z\partial x} +d\dfrac{\partial \psi}{\partial x} +e\dfrac{\partial \psi}{\partial y} +f\dfrac{\partial \psi}{\partial z} +g\psi =\rho $$
! の各係数に対応している.
  implicit none
  real, intent(in) :: x(:)  ! 領域の x 座標
  real, intent(in) :: y(:)  ! 領域の y 座標
  real, intent(in) :: z(:)  ! 領域の z 座標
  real, intent(in) :: rho(size(x),size(y),size(z))  ! ポアソン方程式の強制項
                   ! rho =0 でラプラス方程式も求積可能
  real, intent(in) :: eps  ! 収束条件
  character(6), intent(in) :: boundary  ! 境界条件
                ! 4 文字で各辺の境界条件を与える.
                ! 1 文字目 : x 下端, 2 文字目 : y 左端, 3 文字目 : x 上端,
                ! 4 文字目 : y 右端
                ! boundary は 1 : 固定端境界, 2 : 自由端境界, 3 : 周期境界
  real, intent(in), optional :: bound_opt(size(x),size(y),size(z))
                             ! 境界での強制
                             ! ノイマン境界の場合 : フラックス値
  real, intent(in), optional :: xa(size(x),size(y),size(z))  ! 各微分項の係数
  real, intent(in), optional :: ya(size(x),size(y),size(z))  ! 各微分項の係数
  real, intent(in), optional :: za(size(x),size(y),size(z))  ! 各微分項の係数
  real, intent(in), optional :: a(size(x),size(y),size(z))  ! 各微分項の係数
  real, intent(in), optional :: b(size(x),size(y),size(z))  ! 各微分項の係数
  real, intent(in), optional :: c(size(x),size(y),size(z))  ! 各微分項の係数
  real, intent(in), optional :: d(size(x),size(y),size(z))  ! 各微分項の係数
  real, intent(in), optional :: e(size(x),size(y),size(z))  ! 各微分項の係数
  real, intent(in), optional :: f(size(x),size(y),size(z))  ! 各微分項の係数
  real, intent(in), optional :: g(size(x),size(y),size(z))  ! 各微分項の係数
  real, intent(inout) :: psi(size(x),size(y),size(z))  ! ポアソン方程式の解
  real, intent(in), optional :: undef  ! 未定義値
  integer, intent(in), optional :: inner_bound(size(x),size(y),size(z))
                             ! 内部領域の境界. 値に応じてその格子点で境界値計算
                             ! 1 = 固定端境界, 10 = 境界の内側.
                             ! 2 = y 方向自由端境界 (フラックスは上向き)
                             ! -2 = y 方向自由端境界 (フラックスは下向き)
                             ! 4 = x 方向自由端境界 (フラックスは右向き)
                             ! -4 = x 方向自由端境界 (フラックスは左向き)
                             ! 3 = 周期境界
                             ! 8 = |_, ~| で両方とも自由境界条件
                             ! -8 = |~, _| で両方とも自由境界条件
                             ! この引数が与えられなければ全領域を計算する.
                             ! 境界の内側格子点 (10) は反復計算を行わず,
                             ! undef で設定された値もしくはゼロが入る.
                             ! このときの境界値は bound_opt の値が用いられる.
  logical, intent(in), optional :: init_flag  ! psi の値をゼロで初期化するか.
                             ! .true. = 初期化する. .false. = 初期化しない.
                             ! デフォルトでは初期化する.
  real, intent(in), optional :: accel  ! SOR の加速係数 (0 < accel < 2)
                             ! デフォルト = 1
  integer, intent(in), optional :: ln  ! 反復回数
                             ! この値が与えられるとき, eps の値に無関係に
                             ! ln 回ループさせる.
  logical, intent(in), optional :: check_flag  ! true の場合, 最高階数の
                             ! 係数が負号でも stop する.
                             ! default = .false. (係数が 0 になる場合のみ stop)

  integer :: i, j, k, ix, jy, kz, nl, counter
  integer :: nx  ! x 方向の配列要素
  integer :: ny  ! y 方向の配列要素
  integer :: nz  ! z 方向の配列要素
  integer :: signa, signb, signc  ! 各係数を計算するかどうか
  integer, dimension(size(x),size(y),size(z)) :: ib

  real :: defun
  real :: tmp, err, err_max
  real :: tmp_b, accc
  real :: bnd(size(x),size(y),size(z))
  real :: dx(size(x)), dy(size(y)), dx2(size(x)), dy2(size(y))
  real :: dz(size(z)), dz2(size(z))
  real, dimension(size(x),size(y)) :: dxdy
  real, dimension(size(y),size(z)) :: dydz
  real, dimension(size(x),size(z)) :: dxdz
  real, dimension(size(x),size(y),size(z)) :: xt, yt, zt, at, bt, ct, dt, et, ft, gt
  real, dimension(size(x),size(y),size(z)) :: xdp, xdm, yep, yem, zfp, zfm, xyz, divi

  character(6) :: bound
  logical :: sor_flag
  logical, dimension(size(x),size(y),size(z)) :: inner_flag

  bound(1:6)=boundary(1:6)

  nx=size(x)
  ny=size(y)
  nz=size(z)

!-- 応答関数の初期化

  if(present(init_flag))then
     if(init_flag.eqv..true.)then
        psi = 0.0
     end if
  else
     psi = 0.0
  end if

!-- 内部境界の判別フラグの設定

  if(present(inner_bound))then
     call set_bound_3d( bound, ib, inner_flag, inner_bound )
  else
     call set_bound_3d( bound, ib, inner_flag )
  end if

!-- 領域・内部境界における境界値の設定

  if(present(bound_opt))then
     call setval_bound_3df( ib, bnd, psi, bound_opt )
  else
     call setval_bound_3df( ib, bnd, psi )
  end if

!-- 未定義値の設定

  if(present(undef))then
     defun=undef
  else
     defun=0.0
  end if

!-- 係数の代入
!-- xa, ya, za については, 値が入れられていなければ, 全配列に 1 を代入する.
  if(present(xa))then
     call set_coe_3df( xt, ext=xa )
  else
     call set_coe_3df( xt, def=1.0 )
  end if

  if(present(ya))then
     call set_coe_3df( yt, ext=ya )
  else
     call set_coe_3df( yt, def=1.0 )
  end if

  if(present(za))then
     call set_coe_3df( zt, ext=za )
  else
     call set_coe_3df( zt, def=1.0 )
  end if

  if(present(a))then
     call set_coe_3df( at, ext=a )
     signa=1
  else
     call set_coe_3df( at, def=0.0 )
     signa=0
  end if

  if(present(b))then
     call set_coe_3df( bt, ext=b )
     signb=1
  else
     call set_coe_3df( bt, def=0.0 )
     signb=0
  end if

  if(present(c))then
     signc=1
     call set_coe_3df( ct, ext=c )
  else
     call set_coe_3df( ct, def=0.0 )
     signc=0
  end if

  if(present(d))then
     call set_coe_3df( dt, ext=d )
  else
     call set_coe_3df( dt, def=0.0 )
  end if

  if(present(e))then
     call set_coe_3df( et, ext=e )
  else
     call set_coe_3df( et, def=0.0 )
  end if

  if(present(f))then
     call set_coe_3df( ft, ext=f )
  else
     call set_coe_3df( ft, def=0.0 )
  end if

  if(present(g))then
     call set_coe_3df( gt, ext=g )
  else
     call set_coe_3df( gt, def=0.0 )
  end if

!-- 最高階数における係数チェック. (係数がゼロでないか調べる.)

  if(present(check_flag))then
     if(check_flag.eqv..true.)then
        call check_le_coe_3df( xt(2:nx-1,2:ny-1,2:nz-1), 0.0, undeff=defun )
        call check_le_coe_3df( yt(2:nx-1,2:ny-1,2:nz-1), 0.0, undeff=defun )
        call check_le_coe_3df( zt(2:nx-1,2:ny-1,2:nz-1), 0.0, undeff=defun )
     else
        call check_coe_3df( xt(2:nx-1,2:ny-1,2:nz-1), 0.0, undeff=defun )
        call check_coe_3df( yt(2:nx-1,2:ny-1,2:nz-1), 0.0, undeff=defun )
        call check_coe_3df( zt(2:nx-1,2:ny-1,2:nz-1), 0.0, undeff=defun )
     end if
  else
     call check_coe_3df( xt(2:nx-1,2:ny-1,2:nz-1), 0.0, undeff=defun )
     call check_coe_3df( yt(2:nx-1,2:ny-1,2:nz-1), 0.0, undeff=defun )
     call check_coe_3df( zt(2:nx-1,2:ny-1,2:nz-1), 0.0, undeff=defun )
  end if

!-- 加速係数の判定

  if(present(accel))then
     accc=accel
     sor_flag=.true.
  else
     sor_flag=.false.
  end if

!-- ln の処理
  if(present(ln))then
     nl=ln
  else
     nl=0
  end if

!-- 以下で先に格子間隔等の 1 回計算でよいものを求めておく.
!-- これらは 1 方向のみで変化すればよい.
!-- 格子点間隔の計算
  do i=2,nx-1
     dx(i)=(x(i+1)-x(i-1))*0.5
     dx2(i)=dx(i)**2
  end do
  do j=2,ny-1
     dy(j)=(y(j+1)-y(j-1))*0.5
     dy2(j)=dy(j)**2
  end do
  do k=2,nz-1
     dz(k)=(z(k+1)-z(k-1))*0.5
     dz2(k)=dz(k)**2
  end do

  dx(1)=(x(2)-x(1))
  dx(nx)=(x(nx)-x(nx-1))
  dy(1)=(y(2)-y(1))
  dy(ny)=(y(ny)-y(ny-1))
  dz(1)=(z(2)-z(1))
  dz(nz)=(z(nz)-z(nz-1))

  do j=1,ny
     do i=1,nx
        dxdy(i,j)=dx(i)*dy(j)
     end do
  end do
  do k=1,nz
     do j=1,ny
        dydz(j,k)=dy(j)*dz(k)
     end do
  end do
  do k=1,nz
     do i=1,nx
        dxdz(i,k)=dx(i)*dz(k)
     end do
  end do

!-- ポアソン係数の計算
!-- 実際にポアソン係数が使用されるのは, 境界を除いた 1 回り内側なので,
!-- 計算量削減のため, ループをそのようにしておく.

  xyz=0.0
  xdp=0.0
  xdm=0.0
  yep=0.0
  yem=0.0
  zfp=0.0
  zfm=0.0

!-- 最高次数係数 ac の計算
!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)

  do k=2,nz-1
     do j=2,ny-1
        do i=2,nx-1
           xyz(i,j,k)=1.0/(2.0*(xt(i,j,k)/dx2(i)+yt(i,j,k)/dy2(j)+zt(i,j,k)/dz2(k))-gt(i,j,k))
           xdp(i,j,k)=(xt(i,j,k)/(dx2(i))+0.5*dt(i,j,k)/dx(i))*xyz(i,j,k)
           xdm(i,j,k)=(xt(i,j,k)/(dx2(i))-0.5*dt(i,j,k)/dx(i))*xyz(i,j,k)
           yep(i,j,k)=(yt(i,j,k)/(dy2(j))+0.5*et(i,j,k)/dy(j))*xyz(i,j,k)
           yem(i,j,k)=(yt(i,j,k)/(dy2(j))-0.5*et(i,j,k)/dy(j))*xyz(i,j,k)
           zfp(i,j,k)=(zt(i,j,k)/(dz2(k))+0.5*ft(i,j,k)/dz(k))*xyz(i,j,k)
           zfm(i,j,k)=(zt(i,j,k)/(dz2(k))-0.5*ft(i,j,k)/dz(k))*xyz(i,j,k)
           if(signa==1)then
              at(i,j,k)=0.25*at(i,j,k)/(dxdy(i,j))*xyz(i,j,k)
           else
              at(i,j,k)=0.0
           end if
           if(signb==1)then
              bt(i,j,k)=0.25*bt(i,j,k)/(dydz(j,k))*xyz(i,j,k)
           else
              bt(i,j,k)=0.0
           end if
           if(signc==1)then
              ct(i,j,k)=0.25*ct(i,j,k)/(dxdz(i,k))*xyz(i,j,k)
           else
              ct(i,j,k)=0.0
           end if
        end do
     end do
  end do

!$omp end do
!$omp end parallel

  err_max=eps  ! while に入るための便宜的措置
  counter=0

!-- 実際のソルバ ---
  do while(err_max>=eps)
     err_max=0.0

     do k=1,nz
        do j=1,ny
           do i=1,nx

!-- 以降, 反復計算に必要な点の周囲 8 点についてそれぞれ
!-- inner_flag のチェックと同時に適切な値をそれぞれ逐次計算する,
!-- 下で, 各方向の格子計算を行っているが, ib=8,-8 は 4 隅格子のときにしか
!-- case select しない. なぜなら, 真横や上下に隅格子がくることはありえない.
              if(inner_flag(i,j,k).eqv..false.)then  ! .false. なら領域計算開始

                 tmp=-rho(i,j,k)*xyz(i,j,k)
                 tmp=tmp+xdp(i,j,k)*psi(i+1,j,k)  &
  &                     +xdm(i,j,k)*psi(i-1,j,k)  &
  &                     +yep(i,j,k)*psi(i,j+1,k)  &
  &                     +yem(i,j,k)*psi(i,j-1,k)  &
  &                     +zfp(i,j,k)*psi(i,j,k+1)  &
  &                     +zfm(i,j,k)*psi(i,j,k-1)

                 if(signa==1)then
                    tmp_b=at(i,j,k)*(psi(i+1,j+1,k)+psi(i-1,j-1,k)  &
  &                                 -psi(i+1,j-1,k)-psi(i-1,j+1,k))
                 else
                    tmp_b=0.0
                 end if
                 if(signb==1)then  ! そもそも bt = 0 なら計算しない.
                    tmp_b=tmp_b+bt(i,j,k)*(psi(i,j+1,k+1)+psi(i,j-1,k-1)  &
  &                                       -psi(i,j-1,k+1)-psi(i,j+1,k-1))
                 end if
                 if(signc==1)then  ! そもそも bt = 0 なら計算しない.
                    tmp_b=tmp_b+ct(i,j,k)*(psi(i+1,j,k+1)+psi(i-1,j,k-1)  &
  &                                       -psi(i-1,j,k+1)-psi(i+1,j,k-1))
                 end if
              
                 tmp=tmp+tmp_b

              else   ! .true. なら境界計算開始.

                 select case (ib(i,j,k))
                 case (1)
                    tmp=bnd(i,j,k)

                 case (2)  ! x 方向にフラックス一定, 上側が参照値
                    tmp=psi(i+1,j,k)-bnd(i,j,k)*dx(i)

                 case (-2)  ! x 方向にフラックス一定, 下側が参照値
                    tmp=psi(i-1,j,k)+bnd(i,j,k)*dx(i)

                 case (4)  ! y 方向にフラックス一定, 右側が参照値
                    tmp=psi(i,j+1,k)-bnd(i,j,k)*dy(j)

                 case (-4)  ! y 方向にフラックス一定, 左側が参照値
                    tmp=psi(i,j-1,k)+bnd(i,j,k)*dy(j)

                 case (6)  ! y 方向にフラックス一定, 右側が参照値
                    tmp=psi(i,j,k+1)-bnd(i,j,k)*dz(k)

                 case (-6)  ! y 方向にフラックス一定, 左側が参照値
                    tmp=psi(i,j,k-1)+bnd(i,j,k)*dz(k)

                 case (3)  ! 12 辺, もしくは 8 点で周期境界を判断
                    if(i==1)then
                       ix=nx-1
                    else if(i==nx)then
                       ix=2
                    else
                       ix=i
                    end if
                    if(j==1)then
                       jy=ny-1
                    else if(j==ny)then
                       jy=2
                    else
                       jy=j
                    end if
                    if(k==1)then
                       kz=nz-1
                    else if(k==nz)then
                       kz=2
                    else
                       kz=k
                    end if
                    tmp=psi(ix,jy,kz)

                 case (8)  ! 両方フラックス一定で z 面の x, y 右上か左下角.
                    if(i==1.and.j==1)then  ! -- 評価 1
                       tmp=psi(i+1,j+1,k)  &
  &                        -0.5*(bnd(i+1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))

                    else if(i==nx.and.j==ny)then  ! -- 評価 2
                       tmp=psi(i-1,j-1,k)  &
  &                        +0.5*(bnd(i-1,j,k)*dy(j)+bnd(i,j-1,k)*dx(i))

                    else if(ib(i-1,j,k)==10.and.ib(i,j-1,k)==10)then
                       ! -- 評価 1 と同じ
                       tmp=psi(i+1,j+1,k)  &
  &                        -0.5*(bnd(i+1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))

                    else if(ib(i+1,j,k)==10.and.ib(i,j+1,k)==10)then
                       ! -- 評価 2 と同じ
                       tmp=psi(i-1,j-1,k)  &
  &                        +0.5*(bnd(i-1,j,k)*dy(j)+bnd(i,j-1,k)*dx(i))

                    end if

                 case (-8)  ! 両方フラックス一定で z 面の x, y 右下か左上角.
                    if(i==1.and.j==ny)then  ! -- 評価 1
                       tmp=psi(i+1,j-1,k)  &
  &                        +0.5*(bnd(i+1,j,k)*dy(j)-bnd(i,j-1,k)*dx(i))

                    else if(i==nx.and.j==1)then  ! -- 評価 2
                       tmp=psi(i-1,j+1,k)  &
  &                        +0.5*(-bnd(i-1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))

                    else if(ib(i-1,j,k)==10.and.ib(i,j+1,k)==10)then
                       ! -- 評価 1 と同じ
                       tmp=psi(i+1,j-1,k)  &
  &                        +0.5*(bnd(i+1,j,k)*dy(j)-bnd(i,j-1,k)*dx(i))

                    else if(ib(i+1,j,k)==10.and.ib(i,j-1,k)==10)then
                       ! -- 評価 2 と同じ
                       tmp=psi(i-1,j+1,k)  &
  &                        +0.5*(-bnd(i-1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))
                    end if

                 case (12)  ! 両方フラックス一定で y 面の x, z 右上か左下角.
                    if(i==1.and.k==1)then  ! -- 評価 1
                       tmp=psi(i+1,j,k+1)  &
  &                        -0.5*(bnd(i+1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                    else if(i==nx.and.k==nz)then  ! -- 評価 2
                       tmp=psi(i-1,j,k-1)  &
  &                        +0.5*(bnd(i-1,j,k)*dz(k)+bnd(i,j,k-1)*dx(i))

                    else if(ib(i-1,j,k)==10.and.ib(i,j,k-1)==10)then
                       ! -- 評価 1 と同じ
                       tmp=psi(i+1,j,k+1)  &
  &                        -0.5*(bnd(i+1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                    else if(ib(i+1,j,k)==10.and.ib(i,j,k+1)==10)then
                       ! -- 評価 2 と同じ
                       tmp=psi(i-1,j,k-1)  &
  &                        +0.5*(bnd(i-1,j,k)*dz(k)+bnd(i,j,k-1)*dx(i))
                    end if

                 case (-12)  ! 両方フラックス一定で y 面の x, z 右下か左上角.
                    if(i==1.and.k==nz)then  ! -- 評価 1
                       tmp=psi(i+1,j,k-1)  &
  &                        +0.5*(bnd(i+1,j,k)*dz(k)-bnd(i,j,k-1)*dx(i))

                    else if(i==nx.and.k==1)then  ! -- 評価 2
                       tmp=psi(i-1,j,k+1)  &
  &                        +0.5*(-bnd(i-1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                    else if(ib(i-1,j,k)==10.and.ib(i,j,k+1)==10)then
                       ! -- 評価 1 と同じ
                       tmp=psi(i+1,j,k-1)  &
  &                        +0.5*(bnd(i+1,j,k)*dz(k)-bnd(i,j,k-1)*dx(i))

                    else if(ib(i+1,j,k)==10.and.ib(i,j,k-1)==10)then
                       ! -- 評価 2 と同じ
                       tmp=psi(i-1,j,k+1)  &
  &                        +0.5*(-bnd(i-1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))
                    end if

                 case (24)  ! 両方フラックス一定で x 面の y, z 右上か左下角.
                    if(j==1.and.k==1)then  ! -- 評価 1
                       tmp=psi(i,j+1,k+1)  &
  &                        -0.5*(bnd(i,j+1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                    else if(j==ny.and.k==nz)then  ! -- 評価 2
                       tmp=psi(i,j-1,k-1)  &
  &                        +0.5*(bnd(i,j-1,k)*dz(k)+bnd(i,j,k-1)*dy(j))

                    else if(ib(i,j-1,k)==10.and.ib(i,j,k-1)==10)then
                       ! -- 評価 1 と同じ
                       tmp=psi(i,j+1,k+1)  &
  &                        -0.5*(bnd(i,j+1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                    else if(ib(i,j+1,k)==10.and.ib(i,j,k+1)==10)then
                       ! -- 評価 2 と同じ
                       tmp=psi(i,j-1,k-1)  &
  &                        +0.5*(bnd(i,j-1,k)*dz(k)+bnd(i,j,k-1)*dy(j))
                    end if

                 case (-24)  ! 両方フラックス一定で x 面の y, z 右下か左上角.
                    if(j==1.and.k==nz)then  ! -- 評価 1
                       tmp=psi(i,j+1,k-1)  &
  &                        +0.5*(bnd(i,j+1,k)*dz(k)-bnd(i,j,k-1)*dy(j))

                    else if(j==ny.and.k==1)then  ! -- 評価 2
                       tmp=psi(i,j-1,k+1)  &
  &                        +0.5*(-bnd(i,j-1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                    else if(ib(i,j-1,k)==10.and.ib(i,j,k+1)==10)then
                       ! -- 評価 1 と同じ
                       tmp=psi(i,j+1,k-1)  &
  &                        +0.5*(bnd(i,j+1,k)*dz(k)-bnd(i,j,k-1)*dy(j))

                    else if(ib(i,j+1,k)==10.and.ib(i,j,k-1)==10)then
                       ! -- 評価 2 と同じ
                       tmp=psi(i,j-1,k+1)  &
  &                        +0.5*(-bnd(i,j-1,k)*dz(k)+bnd(i,j,k+1)*dy(j))
                    end if

                 !-- 以降, 隅領域なので, 個別に設定.
                 case (11)  ! 両方フラックス一定で (1,1,1) 点, もしくは内部領域.
                    tmp=psi(i+1,j+1,k+1)  &
  &                           -(bnd(i,j+1,k+1)*dx(i)  &
  &                            +bnd(i+1,j,k+1)*dy(j)  &
  &                            +bnd(i+1,j+1,k)*dz(k))/3.0

                 case (13)  ! 両方フラックス一定で (nx,1,1) 点, もしくは内部領域.
                    tmp=psi(i-1,j+1,k+1)  &
  &                           -(-bnd(i,j+1,k+1)*dx(i)  &
  &                             +bnd(i-1,j,k+1)*dy(j)  &
  &                             +bnd(i-1,j+1,k)*dz(k))/3.0

                 case (17)  ! 両方フラックス一定で (1,ny,1) 点, もしくは内部領域.
                    tmp=psi(i+1,j-1,k+1)  &
  &                           -(bnd(i,j-1,k+1)*dx(i)  &
  &                            -bnd(i+1,j,k+1)*dy(j)  &
  &                            +bnd(i+1,j-1,k)*dz(k))/3.0

                 case (19)  ! 両方フラックス一定で (nx,ny,1) 点, もしくは内部領域.
                    tmp=psi(i-1,j-1,k+1)  &
  &                           -(-bnd(i,j-1,k+1)*dx(i)  &
  &                             -bnd(i-1,j,k+1)*dy(j)  &
  &                             +bnd(i-1,j-1,k)*dz(k))/3.0

                 case (23)  ! 両方フラックス一定で (1,1,nz) 点, もしくは内部領域.
                    tmp=psi(i+1,j+1,k-1)  &
  &                           -(bnd(i,j+1,k-1)*dx(i)  &
  &                            +bnd(i+1,j,k-1)*dy(j)  &
  &                            -bnd(i+1,j+1,k)*dz(k))/3.0

                 case (29)  ! 両方フラックス一定で (nx,1,nz) 点, もしくは内部領域.
                    tmp=psi(i-1,j+1,k-1)  &
  &                           -(-bnd(i,j+1,k-1)*dx(i)  &
  &                             +bnd(i-1,j,k-1)*dy(j)  &
  &                             -bnd(i-1,j+1,k)*dz(k))/3.0

                 case (31)  ! 両方フラックス一定で (1,ny,nz) 点, もしくは内部領域.
                    tmp=psi(i+1,j-1,k-1)  &
  &                           -(bnd(i,j-1,k-1)*dx(i)  &
  &                            -bnd(i+1,j,k-1)*dy(j)  &
  &                            -bnd(i+1,j-1,k)*dz(k))/3.0

                 case (37)  ! 両方フラックス一定で (nx,ny,nz) 点, もしくは内部領域.
                    tmp=psi(i-1,j-1,k-1)  &
  &                           -(-bnd(i,j-1,k-1)*dx(i)  &
  &                             -bnd(i-1,j,k-1)*dy(j)  &
  &                             -bnd(i-1,j-1,k)*dz(k))/3.0

                 case (-11)  ! 両方フラックス一定で (i+1,j+1,k+1) に 10 が設定.
                    tmp=(psi(i-1,j+1,k+1)+psi(i+1,j-1,k+1)+psi(i+1,j+1,k-1)  &
  &                            +bnd(i,j+1,k+1)*dx(i)+bnd(i+1,j,k+1)*dy(j)  &
  &                            +bnd(i+1,j+1,k)*dz(k))/3.0

                 case (-13)  ! 両方フラックス一定で (i-1,j+1,k+1) に 10 が設定.
                    tmp=(psi(i+1,j+1,k+1)+psi(i-1,j-1,k+1)+psi(i-1,j+1,k-1)  &
  &                            -bnd(i,j+1,k+1)*dx(i)+bnd(i-1,j,k+1)*dy(j)  &
  &                            +bnd(i-1,j+1,k)*dz(k))/3.0

                 case (-17)  ! 両方フラックス一定で (i+1,j-1,k+1) に 10 が設定.
                    tmp=(psi(i-1,j-1,k+1)+psi(i+1,j+1,k+1)+psi(i+1,j-1,k-1)  &
  &                            +bnd(i,j-1,k+1)*dx(i)-bnd(i+1,j,k+1)*dy(j)  &
  &                            +bnd(i+1,j-1,k)*dz(k))/3.0

                 case (-19)  ! 両方フラックス一定で (i-1,j-1,k+1) に 10 が設定.
                    tmp=(psi(i+1,j-1,k+1)+psi(i-1,j+1,k+1)+psi(i-1,j-1,k-1)  &
  &                            -bnd(i,j-1,k+1)*dx(i)-bnd(i-1,j,k+1)*dy(j)  &
  &                            +bnd(i-1,j-1,k)*dz(k))/3.0

                 case (-23)  ! 両方フラックス一定で (i+1,j+1,k-1) に 10 が設定.
                    tmp=(psi(i-1,j+1,k-1)+psi(i+1,j-1,k-1)+psi(i+1,j+1,k+1)  &
  &                            +bnd(i,j+1,k-1)*dx(i)+bnd(i+1,j,k-1)*dy(j)  &
  &                            -bnd(i+1,j+1,k)*dz(k))/3.0

                 case (-29)  ! 両方フラックス一定で (i-1,j+1,k-1) に 10 が設定.
                    tmp=(psi(i+1,j+1,k-1)+psi(i-1,j-1,k-1)+psi(i-1,j+1,k+1)  &
  &                            -bnd(i,j+1,k-1)*dx(i)+bnd(i-1,j,k-1)*dy(j)  &
  &                            -bnd(i-1,j+1,k)*dz(k))/3.0

                 case (-31)  ! 両方フラックス一定で (i+1,j-1,k-1) に 10 が設定.
                    tmp=(psi(i-1,j-1,k-1)+psi(i+1,j+1,k-1)+psi(i+1,j-1,k+1)  &
  &                            +bnd(i,j-1,k-1)*dx(i)-bnd(i+1,j,k-1)*dy(j)  &
  &                            -bnd(i+1,j-1,k)*dz(k))/3.0

                 case (-37)  ! 両方フラックス一定で (i-1,j-1,k-1) に 10 が設定.
                    tmp=(psi(i+1,j-1,k-1)+psi(i-1,j+1,k-1)+psi(i-1,j-1,k+1)  &
  &                            -bnd(i,j-1,k-1)*dx(i)-bnd(i-1,j,k-1)*dy(j)  &
  &                            -bnd(i-1,j-1,k)*dz(k))/3.0

                 end select

              end if

              if(sor_flag.eqv..true.)then
                 tmp=(1.0-accc)*psi(i,j,k)+tmp*accc
              end if

              err=abs(tmp-psi(i,j,k))

!-- 最大誤差の更新
              if(err_max<=err)then
                 err_max=err
              end if

              psi(i,j,k)=tmp

           end do
        end do
     end do

     if(nl/=0)then
        counter=counter+1
        if(counter==nl)then
           exit
        else
           err_max=eps
        end if
     end if

  end do

!-- 境界の設定

  call calculate_bound_3df( ib, dx, dy, dz, bnd, psi )

!-- 未定義領域には undef を代入する.

  do k=1,nz
     do j=1,ny
        do i=1,nx
           if(ib(i,j,k)==10)then
              psi(i,j,k)=defun
           end if
        end do
     end do
  end do

end subroutine Ellip_GauSei_3df

!----------------------------------
!----------------------------------

subroutine Ellip_GauSei_3dd(x, y, z, rho, eps, boundary, psi, bound_opt,  &
  &                         xa, ya, za, a, b, c, d, e, f, g,  &
  &                         undef, inner_bound, init_flag, accel, ln, check_flag )
! ガウス=ザイデル法による楕円型方程式の求積
! 各オプション配列は, ポアソン系の各微分項の値. デフォルトはゼロで設定される.
! $$xa\dfrac{\partial ^2\psi}{\partial x^2} +ya\dfrac{\partial ^2\psi}{\partial y^2} +za\dfrac{\partial ^2\psi}{\partial z^2} +a\dfrac{\partial ^2\psi}{\partial x\partial y} +b\dfrac{\partial ^2\psi}{\partial y\partial z} +c\dfrac{\partial ^2\psi}{\partial z\partial x} +d\dfrac{\partial \psi}{\partial x} +e\dfrac{\partial \psi}{\partial y} +f\dfrac{\partial \psi}{\partial z} +g\psi =\rho $$
! の各係数に対応している.
  implicit none
  double precision, intent(in) :: x(:)  ! 領域の x 座標
  double precision, intent(in) :: y(:)  ! 領域の y 座標
  double precision, intent(in) :: z(:)  ! 領域の z 座標
  double precision, intent(in) :: rho(size(x),size(y),size(z))  ! ポアソン方程式の強制項
                   ! rho =0 でラプラス方程式も求積可能
  double precision, intent(in) :: eps  ! 収束条件
  character(6), intent(in) :: boundary  ! 境界条件
                ! 4 文字で各辺の境界条件を与える.
                ! 1 文字目 : x 下端, 2 文字目 : y 左端, 3 文字目 : x 上端,
                ! 4 文字目 : y 右端
                ! boundary は 1 : 固定端境界, 2 : 自由端境界, 3 : 周期境界
  double precision, intent(in), optional :: bound_opt(size(x),size(y),size(z))
                             ! 境界での強制
                             ! ノイマン境界の場合 : フラックス値
  double precision, intent(in), optional :: xa(size(x),size(y),size(z))  ! 各微分項の係数
  double precision, intent(in), optional :: ya(size(x),size(y),size(z))  ! 各微分項の係数
  double precision, intent(in), optional :: za(size(x),size(y),size(z))  ! 各微分項の係数
  double precision, intent(in), optional :: a(size(x),size(y),size(z))  ! 各微分項の係数
  double precision, intent(in), optional :: b(size(x),size(y),size(z))  ! 各微分項の係数
  double precision, intent(in), optional :: c(size(x),size(y),size(z))  ! 各微分項の係数
  double precision, intent(in), optional :: d(size(x),size(y),size(z))  ! 各微分項の係数
  double precision, intent(in), optional :: e(size(x),size(y),size(z))  ! 各微分項の係数
  double precision, intent(in), optional :: f(size(x),size(y),size(z))  ! 各微分項の係数
  double precision, intent(in), optional :: g(size(x),size(y),size(z))  ! 各微分項の係数
  double precision, intent(inout) :: psi(size(x),size(y),size(z))  ! ポアソン方程式の解
  double precision, intent(in), optional :: undef  ! 未定義値
  integer, intent(in), optional :: inner_bound(size(x),size(y),size(z))
                             ! 内部領域の境界. 値に応じてその格子点で境界値計算
                             ! 1 = 固定端境界, 10 = 境界の内側.
                             ! 2 = y 方向自由端境界 (フラックスは上向き)
                             ! -2 = y 方向自由端境界 (フラックスは下向き)
                             ! 4 = x 方向自由端境界 (フラックスは右向き)
                             ! -4 = x 方向自由端境界 (フラックスは左向き)
                             ! 3 = 周期境界
                             ! 8 = |_, ~| で両方とも自由境界条件
                             ! -8 = |~, _| で両方とも自由境界条件
                             ! この引数が与えられなければ全領域を計算する.
                             ! 境界の内側格子点 (10) は反復計算を行わず,
                             ! undef で設定された値もしくはゼロが入る.
                             ! このときの境界値は bound_opt の値が用いられる.
  logical, intent(in), optional :: init_flag  ! psi の値をゼロで初期化するか.
                             ! .true. = 初期化する. .false. = 初期化しない.
                             ! デフォルトでは初期化する.
  double precision, intent(in), optional :: accel  ! SOR の加速係数 (0 < accel < 2)
                             ! デフォルト = 1
  integer, intent(in), optional :: ln  ! 反復回数
                             ! この値が与えられるとき, eps の値に無関係に
                             ! ln 回ループさせる.
  logical, intent(in), optional :: check_flag  ! true の場合, 最高階数の
                             ! 係数が負号でも stop する.
                             ! default = .false. (係数が 0 になる場合のみ stop)

  integer :: i, j, k, ix, jy, kz, nl, counter
  integer :: nx  ! x 方向の配列要素
  integer :: ny  ! y 方向の配列要素
  integer :: nz  ! z 方向の配列要素
  integer :: signa, signb, signc  ! 各係数を計算するかどうか
  integer, dimension(size(x),size(y),size(z)) :: ib

  double precision :: defun
  double precision :: tmp, err, err_max
  double precision :: tmp_b, accc
  double precision :: bnd(size(x),size(y),size(z))
  double precision :: dx(size(x)), dy(size(y)), dx2(size(x)), dy2(size(y))
  double precision :: dz(size(z)), dz2(size(z))
  double precision, dimension(size(x),size(y)) :: dxdy
  double precision, dimension(size(y),size(z)) :: dydz
  double precision, dimension(size(x),size(z)) :: dxdz
  double precision, dimension(size(x),size(y),size(z)) :: xt, yt, zt, at, bt, ct, dt, et, ft, gt
  double precision, dimension(size(x),size(y),size(z)) :: xdp, xdm, yep, yem, zfp, zfm, xyz, divi

  character(6) :: bound
  logical :: sor_flag
  logical, dimension(size(x),size(y),size(z)) :: inner_flag

  bound(1:6)=boundary(1:6)

  nx=size(x)
  ny=size(y)
  nz=size(z)

!-- 応答関数の初期化

  if(present(init_flag))then
     if(init_flag.eqv..true.)then
        psi = 0.0d0
     end if
  else
     psi = 0.0d0
  end if

!-- 内部境界の判別フラグの設定

  if(present(inner_bound))then
     call set_bound_3d( bound, ib, inner_flag, inner_bound )
  else
     call set_bound_3d( bound, ib, inner_flag )
  end if

!-- 領域・内部境界における境界値の設定

  if(present(bound_opt))then
     call setval_bound_3dd( ib, bnd, psi, bound_opt )
  else
     call setval_bound_3dd( ib, bnd, psi )
  end if

!-- 未定義値の設定

  if(present(undef))then
     defun=undef
  else
     defun=0.0d0
  end if

!-- 係数の代入
!-- xa, ya, za については, 値が入れられていなければ, 全配列に 1 を代入する.
  if(present(xa))then
     call set_coe_3dd( xt, ext=xa )
  else
     call set_coe_3dd( xt, def=1.0d0 )
  end if

  if(present(ya))then
     call set_coe_3dd( yt, ext=ya )
  else
     call set_coe_3dd( yt, def=1.0d0 )
  end if

  if(present(za))then
     call set_coe_3dd( zt, ext=za )
  else
     call set_coe_3dd( zt, def=1.0d0 )
  end if

  if(present(a))then
     call set_coe_3dd( at, ext=a )
     signa=1
  else
     call set_coe_3dd( at, def=0.0d0 )
     signa=0
  end if

  if(present(b))then
     call set_coe_3dd( bt, ext=b )
     signb=1
  else
     call set_coe_3dd( bt, def=0.0d0 )
     signb=0
  end if

  if(present(c))then
     signc=1
     call set_coe_3dd( ct, ext=c )
  else
     call set_coe_3dd( ct, def=0.0d0 )
     signc=0
  end if

  if(present(d))then
     call set_coe_3dd( dt, ext=d )
  else
     call set_coe_3dd( dt, def=0.0d0 )
  end if

  if(present(e))then
     call set_coe_3dd( et, ext=e )
  else
     call set_coe_3dd( et, def=0.0d0 )
  end if

  if(present(f))then
     call set_coe_3dd( ft, ext=f )
  else
     call set_coe_3dd( ft, def=0.0d0 )
  end if

  if(present(g))then
     call set_coe_3dd( gt, ext=g )
  else
     call set_coe_3dd( gt, def=0.0d0 )
  end if

!-- 最高階数における係数チェック. (係数がゼロでないか調べる.)

  if(present(check_flag))then
     if(check_flag.eqv..true.)then
        call check_le_coe_3dd( xt(2:nx-1,2:ny-1,2:nz-1), 0.0d0, undeff=defun )
        call check_le_coe_3dd( yt(2:nx-1,2:ny-1,2:nz-1), 0.0d0, undeff=defun )
        call check_le_coe_3dd( zt(2:nx-1,2:ny-1,2:nz-1), 0.0d0, undeff=defun )
     else
        call check_coe_3dd( xt(2:nx-1,2:ny-1,2:nz-1), 0.0d0, undeff=defun )
        call check_coe_3dd( yt(2:nx-1,2:ny-1,2:nz-1), 0.0d0, undeff=defun )
        call check_coe_3dd( zt(2:nx-1,2:ny-1,2:nz-1), 0.0d0, undeff=defun )
     end if
  else
     call check_coe_3dd( xt(2:nx-1,2:ny-1,2:nz-1), 0.0d0, undeff=defun )
     call check_coe_3dd( yt(2:nx-1,2:ny-1,2:nz-1), 0.0d0, undeff=defun )
     call check_coe_3dd( zt(2:nx-1,2:ny-1,2:nz-1), 0.0d0, undeff=defun )
  end if

!-- 加速係数の判定

  if(present(accel))then
     accc=accel
     sor_flag=.true.
  else
     sor_flag=.false.
  end if

!-- ln の処理
  if(present(ln))then
     nl=ln
  else
     nl=0
  end if

!-- 以下で先に格子間隔等の 1 回計算でよいものを求めておく.
!-- これらは 1 方向のみで変化すればよい.
!-- 格子点間隔の計算
  do i=2,nx-1
     dx(i)=(x(i+1)-x(i-1))*0.5d0
     dx2(i)=dx(i)**2
  end do
  do j=2,ny-1
     dy(j)=(y(j+1)-y(j-1))*0.5d0
     dy2(j)=dy(j)**2
  end do
  do k=2,nz-1
     dz(k)=(z(k+1)-z(k-1))*0.5d0
     dz2(k)=dz(k)**2
  end do

  dx(1)=(x(2)-x(1))
  dx(nx)=(x(nx)-x(nx-1))
  dy(1)=(y(2)-y(1))
  dy(ny)=(y(ny)-y(ny-1))
  dz(1)=(z(2)-z(1))
  dz(nz)=(z(nz)-z(nz-1))

  do j=1,ny
     do i=1,nx
        dxdy(i,j)=dx(i)*dy(j)
     end do
  end do
  do k=1,nz
     do j=1,ny
        dydz(j,k)=dy(j)*dz(k)
     end do
  end do
  do k=1,nz
     do i=1,nx
        dxdz(i,k)=dx(i)*dz(k)
     end do
  end do

!-- ポアソン係数の計算
!-- 実際にポアソン係数が使用されるのは, 境界を除いた 1 回り内側なので,
!-- 計算量削減のため, ループをそのようにしておく.

  xyz=0.0d0
  xdp=0.0d0
  xdm=0.0d0
  yep=0.0d0
  yem=0.0d0
  zfp=0.0d0
  zfm=0.0d0

!-- 最高次数係数 ac の計算
!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)

  do k=2,nz-1
     do j=2,ny-1
        do i=2,nx-1
           xyz(i,j,k)=1.0d0/(2.0d0*(xt(i,j,k)/dx2(i)+yt(i,j,k)/dy2(j)+zt(i,j,k)/dz2(k))-gt(i,j,k))
           xdp(i,j,k)=(xt(i,j,k)/(dx2(i))+0.5d0*dt(i,j,k)/dx(i))*xyz(i,j,k)
           xdm(i,j,k)=(xt(i,j,k)/(dx2(i))-0.5d0*dt(i,j,k)/dx(i))*xyz(i,j,k)
           yep(i,j,k)=(yt(i,j,k)/(dy2(j))+0.5d0*et(i,j,k)/dy(j))*xyz(i,j,k)
           yem(i,j,k)=(yt(i,j,k)/(dy2(j))-0.5d0*et(i,j,k)/dy(j))*xyz(i,j,k)
           zfp(i,j,k)=(zt(i,j,k)/(dz2(k))+0.5d0*ft(i,j,k)/dz(k))*xyz(i,j,k)
           zfm(i,j,k)=(zt(i,j,k)/(dz2(k))-0.5d0*ft(i,j,k)/dz(k))*xyz(i,j,k)
           if(signa==1)then
              at(i,j,k)=0.25d0*at(i,j,k)/(dxdy(i,j))*xyz(i,j,k)
           else
              at(i,j,k)=0.0d0
           end if
           if(signb==1)then
              bt(i,j,k)=0.25d0*bt(i,j,k)/(dydz(j,k))*xyz(i,j,k)
           else
              bt(i,j,k)=0.0d0
           end if
           if(signc==1)then
              ct(i,j,k)=0.25d0*ct(i,j,k)/(dxdz(i,k))*xyz(i,j,k)
           else
              ct(i,j,k)=0.0d0
           end if
        end do
     end do
  end do

!$omp end do
!$omp end parallel

  err_max=eps  ! while に入るための便宜的措置
  counter=0

!-- 実際のソルバ ---
  do while(err_max>=eps)
     err_max=0.0d0

     do k=1,nz
        do j=1,ny
           do i=1,nx

!-- 以降, 反復計算に必要な点の周囲 8 点についてそれぞれ
!-- inner_flag のチェックと同時に適切な値をそれぞれ逐次計算する,
!-- 下で, 各方向の格子計算を行っているが, ib=8,-8 は 4 隅格子のときにしか
!-- case select しない. なぜなら, 真横や上下に隅格子がくることはありえない.
              if(inner_flag(i,j,k).eqv..false.)then  ! .false. なら領域計算開始

                 tmp=-rho(i,j,k)*xyz(i,j,k)
                 tmp=tmp+xdp(i,j,k)*psi(i+1,j,k)  &
  &                     +xdm(i,j,k)*psi(i-1,j,k)  &
  &                     +yep(i,j,k)*psi(i,j+1,k)  &
  &                     +yem(i,j,k)*psi(i,j-1,k)  &
  &                     +zfp(i,j,k)*psi(i,j,k+1)  &
  &                     +zfm(i,j,k)*psi(i,j,k-1)

                 if(signa==1)then
                    tmp_b=at(i,j,k)*(psi(i+1,j+1,k)+psi(i-1,j-1,k)  &
  &                                 -psi(i+1,j-1,k)-psi(i-1,j+1,k))
                 else
                    tmp_b=0.0d0
                 end if
                 if(signb==1)then  ! そもそも bt = 0 なら計算しない.
                    tmp_b=tmp_b+bt(i,j,k)*(psi(i,j+1,k+1)+psi(i,j-1,k-1)  &
  &                                       -psi(i,j-1,k+1)-psi(i,j+1,k-1))
                 end if
                 if(signc==1)then  ! そもそも bt = 0 なら計算しない.
                    tmp_b=tmp_b+ct(i,j,k)*(psi(i+1,j,k+1)+psi(i-1,j,k-1)  &
  &                                       -psi(i-1,j,k+1)-psi(i+1,j,k-1))
                 end if
              
                 tmp=tmp+tmp_b

              else   ! .true. なら境界計算開始.

                 select case (ib(i,j,k))
                 case (1)
                    tmp=bnd(i,j,k)

                 case (2)  ! x 方向にフラックス一定, 上側が参照値
                    tmp=psi(i+1,j,k)-bnd(i,j,k)*dx(i)

                 case (-2)  ! x 方向にフラックス一定, 下側が参照値
                    tmp=psi(i-1,j,k)+bnd(i,j,k)*dx(i)

                 case (4)  ! y 方向にフラックス一定, 右側が参照値
                    tmp=psi(i,j+1,k)-bnd(i,j,k)*dy(j)

                 case (-4)  ! y 方向にフラックス一定, 左側が参照値
                    tmp=psi(i,j-1,k)+bnd(i,j,k)*dy(j)

                 case (6)  ! y 方向にフラックス一定, 右側が参照値
                    tmp=psi(i,j,k+1)-bnd(i,j,k)*dz(k)

                 case (-6)  ! y 方向にフラックス一定, 左側が参照値
                    tmp=psi(i,j,k-1)+bnd(i,j,k)*dz(k)

                 case (3)  ! 12 辺, もしくは 8 点で周期境界を判断
                    if(i==1)then
                       ix=nx-1
                    else if(i==nx)then
                       ix=2
                    else
                       ix=i
                    end if
                    if(j==1)then
                       jy=ny-1
                    else if(j==ny)then
                       jy=2
                    else
                       jy=j
                    end if
                    if(k==1)then
                       kz=nz-1
                    else if(k==nz)then
                       kz=2
                    else
                       kz=k
                    end if
                    tmp=psi(ix,jy,kz)

                 case (8)  ! 両方フラックス一定で z 面の x, y 右上か左下角.
                    if(i==1.and.j==1)then  ! -- 評価 1
                       tmp=psi(i+1,j+1,k)  &
  &                        -0.5d0*(bnd(i+1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))

                    else if(i==nx.and.j==ny)then  ! -- 評価 2
                       tmp=psi(i-1,j-1,k)  &
  &                        +0.5d0*(bnd(i-1,j,k)*dy(j)+bnd(i,j-1,k)*dx(i))

                    else if(ib(i-1,j,k)==10.and.ib(i,j-1,k)==10)then
                       ! -- 評価 1 と同じ
                       tmp=psi(i+1,j+1,k)  &
  &                        -0.5d0*(bnd(i+1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))

                    else if(ib(i+1,j,k)==10.and.ib(i,j+1,k)==10)then
                       ! -- 評価 2 と同じ
                       tmp=psi(i-1,j-1,k)  &
  &                        +0.5d0*(bnd(i-1,j,k)*dy(j)+bnd(i,j-1,k)*dx(i))

                    end if

                 case (-8)  ! 両方フラックス一定で z 面の x, y 右下か左上角.
                    if(i==1.and.j==ny)then  ! -- 評価 1
                       tmp=psi(i+1,j-1,k)  &
  &                        +0.5d0*(bnd(i+1,j,k)*dy(j)-bnd(i,j-1,k)*dx(i))

                    else if(i==nx.and.j==1)then  ! -- 評価 2
                       tmp=psi(i-1,j+1,k)  &
  &                        +0.5d0*(-bnd(i-1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))

                    else if(ib(i-1,j,k)==10.and.ib(i,j+1,k)==10)then
                       ! -- 評価 1 と同じ
                       tmp=psi(i+1,j-1,k)  &
  &                        +0.5d0*(bnd(i+1,j,k)*dy(j)-bnd(i,j-1,k)*dx(i))

                    else if(ib(i+1,j,k)==10.and.ib(i,j-1,k)==10)then
                       ! -- 評価 2 と同じ
                       tmp=psi(i-1,j+1,k)  &
  &                        +0.5d0*(-bnd(i-1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))
                    end if

                 case (12)  ! 両方フラックス一定で y 面の x, z 右上か左下角.
                    if(i==1.and.k==1)then  ! -- 評価 1
                       tmp=psi(i+1,j,k+1)  &
  &                        -0.5d0*(bnd(i+1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                    else if(i==nx.and.k==nz)then  ! -- 評価 2
                       tmp=psi(i-1,j,k-1)  &
  &                        +0.5d0*(bnd(i-1,j,k)*dz(k)+bnd(i,j,k-1)*dx(i))

                    else if(ib(i-1,j,k)==10.and.ib(i,j,k-1)==10)then
                       ! -- 評価 1 と同じ
                       tmp=psi(i+1,j,k+1)  &
  &                        -0.5d0*(bnd(i+1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                    else if(ib(i+1,j,k)==10.and.ib(i,j,k+1)==10)then
                       ! -- 評価 2 と同じ
                       tmp=psi(i-1,j,k-1)  &
  &                        +0.5d0*(bnd(i-1,j,k)*dz(k)+bnd(i,j,k-1)*dx(i))
                    end if

                 case (-12)  ! 両方フラックス一定で y 面の x, z 右下か左上角.
                    if(i==1.and.k==nz)then  ! -- 評価 1
                       tmp=psi(i+1,j,k-1)  &
  &                        +0.5d0*(bnd(i+1,j,k)*dz(k)-bnd(i,j,k-1)*dx(i))

                    else if(i==nx.and.k==1)then  ! -- 評価 2
                       tmp=psi(i-1,j,k+1)  &
  &                        +0.5d0*(-bnd(i-1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                    else if(ib(i-1,j,k)==10.and.ib(i,j,k+1)==10)then
                       ! -- 評価 1 と同じ
                       tmp=psi(i+1,j,k-1)  &
  &                        +0.5d0*(bnd(i+1,j,k)*dz(k)-bnd(i,j,k-1)*dx(i))

                    else if(ib(i+1,j,k)==10.and.ib(i,j,k-1)==10)then
                       ! -- 評価 2 と同じ
                       tmp=psi(i-1,j,k+1)  &
  &                        +0.5d0*(-bnd(i-1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))
                    end if

                 case (24)  ! 両方フラックス一定で x 面の y, z 右上か左下角.
                    if(j==1.and.k==1)then  ! -- 評価 1
                       tmp=psi(i,j+1,k+1)  &
  &                        -0.5d0*(bnd(i,j+1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                    else if(j==ny.and.k==nz)then  ! -- 評価 2
                       tmp=psi(i,j-1,k-1)  &
  &                        +0.5d0*(bnd(i,j-1,k)*dz(k)+bnd(i,j,k-1)*dy(j))

                    else if(ib(i,j-1,k)==10.and.ib(i,j,k-1)==10)then
                       ! -- 評価 1 と同じ
                       tmp=psi(i,j+1,k+1)  &
  &                        -0.5d0*(bnd(i,j+1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                    else if(ib(i,j+1,k)==10.and.ib(i,j,k+1)==10)then
                       ! -- 評価 2 と同じ
                       tmp=psi(i,j-1,k-1)  &
  &                        +0.5d0*(bnd(i,j-1,k)*dz(k)+bnd(i,j,k-1)*dy(j))
                    end if

                 case (-24)  ! 両方フラックス一定で x 面の y, z 右下か左上角.
                    if(j==1.and.k==nz)then  ! -- 評価 1
                       tmp=psi(i,j+1,k-1)  &
  &                        +0.5d0*(bnd(i,j+1,k)*dz(k)-bnd(i,j,k-1)*dy(j))

                    else if(j==ny.and.k==1)then  ! -- 評価 2
                       tmp=psi(i,j-1,k+1)  &
  &                        +0.5d0*(-bnd(i,j-1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                    else if(ib(i,j-1,k)==10.and.ib(i,j,k+1)==10)then
                       ! -- 評価 1 と同じ
                       tmp=psi(i,j+1,k-1)  &
  &                        +0.5d0*(bnd(i,j+1,k)*dz(k)-bnd(i,j,k-1)*dy(j))

                    else if(ib(i,j+1,k)==10.and.ib(i,j,k-1)==10)then
                       ! -- 評価 2 と同じ
                       tmp=psi(i,j-1,k+1)  &
  &                        +0.5d0*(-bnd(i,j-1,k)*dz(k)+bnd(i,j,k+1)*dy(j))
                    end if

                 !-- 以降, 隅領域なので, 個別に設定.
                 case (11)  ! 両方フラックス一定で (1,1,1) 点, もしくは内部領域.
                    tmp=psi(i+1,j+1,k+1)  &
  &                           -(bnd(i,j+1,k+1)*dx(i)  &
  &                            +bnd(i+1,j,k+1)*dy(j)  &
  &                            +bnd(i+1,j+1,k)*dz(k))/3.0d0

                 case (13)  ! 両方フラックス一定で (nx,1,1) 点, もしくは内部領域.
                    tmp=psi(i-1,j+1,k+1)  &
  &                           -(-bnd(i,j+1,k+1)*dx(i)  &
  &                             +bnd(i-1,j,k+1)*dy(j)  &
  &                             +bnd(i-1,j+1,k)*dz(k))/3.0d0

                 case (17)  ! 両方フラックス一定で (1,ny,1) 点, もしくは内部領域.
                    tmp=psi(i+1,j-1,k+1)  &
  &                           -(bnd(i,j-1,k+1)*dx(i)  &
  &                            -bnd(i+1,j,k+1)*dy(j)  &
  &                            +bnd(i+1,j-1,k)*dz(k))/3.0d0

                 case (19)  ! 両方フラックス一定で (nx,ny,1) 点, もしくは内部領域.
                    tmp=psi(i-1,j-1,k+1)  &
  &                           -(-bnd(i,j-1,k+1)*dx(i)  &
  &                             -bnd(i-1,j,k+1)*dy(j)  &
  &                             +bnd(i-1,j-1,k)*dz(k))/3.0d0

                 case (23)  ! 両方フラックス一定で (1,1,nz) 点, もしくは内部領域.
                    tmp=psi(i+1,j+1,k-1)  &
  &                           -(bnd(i,j+1,k-1)*dx(i)  &
  &                            +bnd(i+1,j,k-1)*dy(j)  &
  &                            -bnd(i+1,j+1,k)*dz(k))/3.0d0

                 case (29)  ! 両方フラックス一定で (nx,1,nz) 点, もしくは内部領域.
                    tmp=psi(i-1,j+1,k-1)  &
  &                           -(-bnd(i,j+1,k-1)*dx(i)  &
  &                             +bnd(i-1,j,k-1)*dy(j)  &
  &                             -bnd(i-1,j+1,k)*dz(k))/3.0d0

                 case (31)  ! 両方フラックス一定で (1,ny,nz) 点, もしくは内部領域.
                    tmp=psi(i+1,j-1,k-1)  &
  &                           -(bnd(i,j-1,k-1)*dx(i)  &
  &                            -bnd(i+1,j,k-1)*dy(j)  &
  &                            -bnd(i+1,j-1,k)*dz(k))/3.0d0

                 case (37)  ! 両方フラックス一定で (nx,ny,nz) 点, もしくは内部領域.
                    tmp=psi(i-1,j-1,k-1)  &
  &                           -(-bnd(i,j-1,k-1)*dx(i)  &
  &                             -bnd(i-1,j,k-1)*dy(j)  &
  &                             -bnd(i-1,j-1,k)*dz(k))/3.0d0

                 case (-11)  ! 両方フラックス一定で (i+1,j+1,k+1) に 10 が設定.
                    tmp=(psi(i-1,j+1,k+1)+psi(i+1,j-1,k+1)+psi(i+1,j+1,k-1)  &
  &                            +bnd(i,j+1,k+1)*dx(i)+bnd(i+1,j,k+1)*dy(j)  &
  &                            +bnd(i+1,j+1,k)*dz(k))/3.0d0

                 case (-13)  ! 両方フラックス一定で (i-1,j+1,k+1) に 10 が設定.
                    tmp=(psi(i+1,j+1,k+1)+psi(i-1,j-1,k+1)+psi(i-1,j+1,k-1)  &
  &                            -bnd(i,j+1,k+1)*dx(i)+bnd(i-1,j,k+1)*dy(j)  &
  &                            +bnd(i-1,j+1,k)*dz(k))/3.0d0

                 case (-17)  ! 両方フラックス一定で (i+1,j-1,k+1) に 10 が設定.
                    tmp=(psi(i-1,j-1,k+1)+psi(i+1,j+1,k+1)+psi(i+1,j-1,k-1)  &
  &                            +bnd(i,j-1,k+1)*dx(i)-bnd(i+1,j,k+1)*dy(j)  &
  &                            +bnd(i+1,j-1,k)*dz(k))/3.0d0

                 case (-19)  ! 両方フラックス一定で (i-1,j-1,k+1) に 10 が設定.
                    tmp=(psi(i+1,j-1,k+1)+psi(i-1,j+1,k+1)+psi(i-1,j-1,k-1)  &
  &                            -bnd(i,j-1,k+1)*dx(i)-bnd(i-1,j,k+1)*dy(j)  &
  &                            +bnd(i-1,j-1,k)*dz(k))/3.0d0

                 case (-23)  ! 両方フラックス一定で (i+1,j+1,k-1) に 10 が設定.
                    tmp=(psi(i-1,j+1,k-1)+psi(i+1,j-1,k-1)+psi(i+1,j+1,k+1)  &
  &                            +bnd(i,j+1,k-1)*dx(i)+bnd(i+1,j,k-1)*dy(j)  &
  &                            -bnd(i+1,j+1,k)*dz(k))/3.0d0

                 case (-29)  ! 両方フラックス一定で (i-1,j+1,k-1) に 10 が設定.
                    tmp=(psi(i+1,j+1,k-1)+psi(i-1,j-1,k-1)+psi(i-1,j+1,k+1)  &
  &                            -bnd(i,j+1,k-1)*dx(i)+bnd(i-1,j,k-1)*dy(j)  &
  &                            -bnd(i-1,j+1,k)*dz(k))/3.0d0

                 case (-31)  ! 両方フラックス一定で (i+1,j-1,k-1) に 10 が設定.
                    tmp=(psi(i-1,j-1,k-1)+psi(i+1,j+1,k-1)+psi(i+1,j-1,k+1)  &
  &                            +bnd(i,j-1,k-1)*dx(i)-bnd(i+1,j,k-1)*dy(j)  &
  &                            -bnd(i+1,j-1,k)*dz(k))/3.0d0

                 case (-37)  ! 両方フラックス一定で (i-1,j-1,k-1) に 10 が設定.
                    tmp=(psi(i+1,j-1,k-1)+psi(i-1,j+1,k-1)+psi(i-1,j-1,k+1)  &
  &                            -bnd(i,j-1,k-1)*dx(i)-bnd(i-1,j,k-1)*dy(j)  &
  &                            -bnd(i-1,j-1,k)*dz(k))/3.0d0

                 end select

              end if

              if(sor_flag.eqv..true.)then
                 tmp=(1.0d0-accc)*psi(i,j,k)+tmp*accc
              end if

              err=abs(tmp-psi(i,j,k))

!-- 最大誤差の更新
              if(err_max<=err)then
                 err_max=err
              end if

              psi(i,j,k)=tmp

           end do
        end do
     end do

     if(nl/=0)then
        counter=counter+1
        if(counter==nl)then
           exit
        else
           err_max=eps
        end if
     end if

  end do

!-- 境界の設定

  call calculate_bound_3dd( ib, dx, dy, dz, bnd, psi )

!-- 未定義領域には undef を代入する.

  do k=1,nz
     do j=1,ny
        do i=1,nx
           if(ib(i,j,k)==10)then
              psi(i,j,k)=defun
           end if
        end do
     end do
  end do

end subroutine Ellip_GauSei_3dd

!----------------------------------
!----------------------------------

subroutine Ellip_Jacobi_3df(x, y, z, rho, eps, boundary, psi, bound_opt,  &
  &                         xa, ya, za, a, b, c, d, e, f, g,  &
  &                         undef, inner_bound, init_flag, accel, ln, check_flag )
use omp_lib
! openmp によるスレッド並列が可能.
! ガウス・ザイデル法ではアルゴリズムの点から並列化が困難と思われたので,
! 並列計算によるポアソン方程式の求積が必要となるなら,
! ヤコビ法のものを使用されたい.
! 各オプション配列は, ポアソン系の各微分項の値. デフォルトはゼロで設定される.
! $$xa\dfrac{\partial ^2\psi}{\partial x^2} +ya\dfrac{\partial ^2\psi}{\partial y^2} +za\dfrac{\partial ^2\psi}{\partial z^2} +a\dfrac{\partial ^2\psi}{\partial x\partial y} +b\dfrac{\partial ^2\psi}{\partial y\partial z} +c\dfrac{\partial ^2\psi}{\partial z\partial x} +d\dfrac{\partial \psi}{\partial x} +e\dfrac{\partial \psi}{\partial y} +f\dfrac{\partial \psi}{\partial z} +g\psi =\rho $$
! の各係数に対応している.
  implicit none
  real, intent(in) :: x(:)  ! 領域の x 座標
  real, intent(in) :: y(:)  ! 領域の y 座標
  real, intent(in) :: z(:)  ! 領域の z 座標
  real, intent(in) :: rho(size(x),size(y),size(z))  ! ポアソン方程式の強制項
                   ! rho =0 でラプラス方程式も求積可能
  real, intent(in) :: eps  ! 収束条件
  character(6), intent(in) :: boundary  ! 境界条件
                ! 4 文字で各辺の境界条件を与える.
                ! 1 文字目 : x 下端, 2 文字目 : y 左端, 3 文字目 : x 上端,
                ! 4 文字目 : y 右端
                ! boundary は 1 : 固定端境界, 2 : 自由端境界, 3 : 周期境界
  real, intent(in), optional :: bound_opt(size(x),size(y),size(z))
                             ! 境界での強制
                             ! ノイマン境界の場合 : フラックス値
  real, intent(in), optional :: xa(size(x),size(y),size(z))  ! 各微分項の係数
  real, intent(in), optional :: ya(size(x),size(y),size(z))  ! 各微分項の係数
  real, intent(in), optional :: za(size(x),size(y),size(z))  ! 各微分項の係数
  real, intent(in), optional :: a(size(x),size(y),size(z))  ! 各微分項の係数
  real, intent(in), optional :: b(size(x),size(y),size(z))  ! 各微分項の係数
  real, intent(in), optional :: c(size(x),size(y),size(z))  ! 各微分項の係数
  real, intent(in), optional :: d(size(x),size(y),size(z))  ! 各微分項の係数
  real, intent(in), optional :: e(size(x),size(y),size(z))  ! 各微分項の係数
  real, intent(in), optional :: f(size(x),size(y),size(z))  ! 各微分項の係数
  real, intent(in), optional :: g(size(x),size(y),size(z))  ! 各微分項の係数
  real, intent(inout) :: psi(size(x),size(y),size(z))  ! ポアソン方程式の解
  real, intent(in), optional :: undef  ! 未定義値
  integer, intent(in), optional :: inner_bound(size(x),size(y),size(z))
                             ! 内部領域の境界. 値に応じてその格子点で境界値計算
                             ! 1 = 固定端境界, 10 = 境界の内側.
                             ! 2 = y 方向自由端境界 (フラックスは上向き)
                             ! -2 = y 方向自由端境界 (フラックスは下向き)
                             ! 4 = x 方向自由端境界 (フラックスは右向き)
                             ! -4 = x 方向自由端境界 (フラックスは左向き)
                             ! 3 = 周期境界
                             ! 8 = |_, ~| で両方とも自由境界条件
                             ! -8 = |~, _| で両方とも自由境界条件
                             ! この引数が与えられなければ全領域を計算する.
                             ! 境界の内側格子点 (10) は反復計算を行わず,
                             ! undef で設定された値もしくはゼロが入る.
                             ! このときの境界値は bound_opt の値が用いられる.
  logical, intent(in), optional :: init_flag  ! psi の値をゼロで初期化するか.
                             ! .true. = 初期化する. .false. = 初期化しない.
                             ! デフォルトでは初期化する.
  real, intent(in), optional :: accel  ! SOR の加速係数 (0 < accel < 2)
                             ! デフォルト = 1
  integer, intent(in), optional :: ln  ! 反復回数
                             ! この値が与えられるとき, eps の値に無関係に
                             ! ln 回ループさせる.
  logical, intent(in), optional :: check_flag  ! true の場合, 最高階数の
                             ! 係数が負号でも stop する.
                             ! default = .false. (係数が 0 になる場合のみ stop)

  integer :: i, j, k, ix, jy, kz, nl, counter
  integer :: nx  ! x 方向の配列要素
  integer :: ny  ! y 方向の配列要素
  integer :: nz  ! z 方向の配列要素
  integer :: signa, signb, signc  ! 各係数を計算するかどうか
  integer, dimension(size(x),size(y),size(z)) :: ib

  real :: defun
  real :: err, err_max, accc
  real :: bnd(size(x),size(y),size(z))
  real :: dx(size(x)), dy(size(y)), dx2(size(x)), dy2(size(y))
  real :: dz(size(z)), dz2(size(z))
  real, dimension(size(x),size(y)) :: dxdy
  real, dimension(size(y),size(z)) :: dydz
  real, dimension(size(x),size(z)) :: dxdz
  real, dimension(size(x),size(y),size(z)) :: xt, yt, zt, at, bt, ct, dt, et, ft, gt
  real, dimension(size(x),size(y),size(z)) :: xdp, xdm, yep, yem, zfp, zfm, xyz, divi
  real, dimension(size(x),size(y),size(z)) :: tmp, tmp_b

  character(6) :: bound
  logical :: sor_flag
  logical, dimension(size(x),size(y),size(z)) :: inner_flag

  bound(1:6)=boundary(1:6)

  nx=size(x)
  ny=size(y)
  nz=size(z)

!-- 応答関数の初期化

  if(present(init_flag))then
     if(init_flag.eqv..true.)then
        psi = 0.0
     end if
  else
     psi = 0.0
  end if

!-- 内部境界の判別フラグの設定

  if(present(inner_bound))then
     call set_bound_3d( bound, ib, inner_flag, inner_bound )
  else
     call set_bound_3d( bound, ib, inner_flag )
  end if

!-- 領域・内部境界における境界値の設定

  if(present(bound_opt))then
     call setval_bound_3df( ib, bnd, psi, bound_opt )
  else
     call setval_bound_3df( ib, bnd, psi )
  end if

!-- 未定義値の設定

  if(present(undef))then
     defun=undef
  else
     defun=0.0
  end if

!-- 係数の代入
!-- xa, ya, za については, 値が入れられていなければ, 全配列に 1 を代入する.
  if(present(xa))then
     call set_coe_3df( xt, ext=xa )
  else
     call set_coe_3df( xt, def=1.0 )
  end if

  if(present(ya))then
     call set_coe_3df( yt, ext=ya )
  else
     call set_coe_3df( yt, def=1.0 )
  end if

  if(present(za))then
     call set_coe_3df( zt, ext=za )
  else
     call set_coe_3df( zt, def=1.0 )
  end if

  if(present(a))then
     call set_coe_3df( at, ext=a )
     signa=1
  else
     call set_coe_3df( at, def=0.0 )
     signa=0
  end if

  if(present(b))then
     call set_coe_3df( bt, ext=b )
     signb=1
  else
     call set_coe_3df( bt, def=0.0 )
     signb=0
  end if

  if(present(c))then
     signc=1
     call set_coe_3df( ct, ext=c )
  else
     call set_coe_3df( ct, def=0.0 )
     signc=0
  end if

  if(present(d))then
     call set_coe_3df( dt, ext=d )
  else
     call set_coe_3df( dt, def=0.0 )
  end if

  if(present(e))then
     call set_coe_3df( et, ext=e )
  else
     call set_coe_3df( et, def=0.0 )
  end if

  if(present(f))then
     call set_coe_3df( ft, ext=f )
  else
     call set_coe_3df( ft, def=0.0 )
  end if

  if(present(g))then
     call set_coe_3df( gt, ext=g )
  else
     call set_coe_3df( gt, def=0.0 )
  end if

!-- 最高階数における係数チェック. (係数がゼロでないか調べる.)

  if(present(check_flag))then
     if(check_flag.eqv..true.)then
        call check_le_coe_3df( xt(2:nx-1,2:ny-1,2:nz-1), 0.0, undeff=defun )
        call check_le_coe_3df( yt(2:nx-1,2:ny-1,2:nz-1), 0.0, undeff=defun )
        call check_le_coe_3df( zt(2:nx-1,2:ny-1,2:nz-1), 0.0, undeff=defun )
     else
        call check_coe_3df( xt(2:nx-1,2:ny-1,2:nz-1), 0.0, undeff=defun )
        call check_coe_3df( yt(2:nx-1,2:ny-1,2:nz-1), 0.0, undeff=defun )
        call check_coe_3df( zt(2:nx-1,2:ny-1,2:nz-1), 0.0, undeff=defun )
     end if
  else
     call check_coe_3df( xt(2:nx-1,2:ny-1,2:nz-1), 0.0, undeff=defun )
     call check_coe_3df( yt(2:nx-1,2:ny-1,2:nz-1), 0.0, undeff=defun )
     call check_coe_3df( zt(2:nx-1,2:ny-1,2:nz-1), 0.0, undeff=defun )
  end if

!-- 加速係数の判定

  if(present(accel))then
     accc=accel
     sor_flag=.true.
  else
     sor_flag=.false.
  end if

!-- ln の処理
  if(present(ln))then
     nl=ln
  else
     nl=0
  end if

!-- 以下で先に格子間隔等の 1 回計算でよいものを求めておく.
!-- これらは 1 方向のみで変化すればよい.
!-- 格子点間隔の計算
  do i=2,nx-1
     dx(i)=(x(i+1)-x(i-1))*0.5
     dx2(i)=dx(i)**2
  end do
  do j=2,ny-1
     dy(j)=(y(j+1)-y(j-1))*0.5
     dy2(j)=dy(j)**2
  end do
  do k=2,nz-1
     dz(k)=(z(k+1)-z(k-1))*0.5
     dz2(k)=dz(k)**2
  end do

  dx(1)=(x(2)-x(1))
  dx(nx)=(x(nx)-x(nx-1))
  dy(1)=(y(2)-y(1))
  dy(ny)=(y(ny)-y(ny-1))
  dz(1)=(z(2)-z(1))
  dz(nz)=(z(nz)-z(nz-1))

  do j=1,ny
     do i=1,nx
        dxdy(i,j)=dx(i)*dy(j)
     end do
  end do
  do k=1,nz
     do j=1,ny
        dydz(j,k)=dy(j)*dz(k)
     end do
  end do
  do k=1,nz
     do i=1,nx
        dxdz(i,k)=dx(i)*dz(k)
     end do
  end do

!-- ポアソン係数の計算
!-- 実際にポアソン係数が使用されるのは, 境界を除いた 1 回り内側なので,
!-- 計算量削減のため, ループをそのようにしておく.

  xyz=0.0
  xdp=0.0
  xdm=0.0
  yep=0.0
  yem=0.0
  zfp=0.0
  zfm=0.0

!-- 最高次数係数 ac の計算
!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)

  do k=2,nz-1
     do j=2,ny-1
        do i=2,nx-1
           xyz(i,j,k)=1.0/(2.0*(xt(i,j,k)/dx2(i)+yt(i,j,k)/dy2(j)+zt(i,j,k)/dz2(k))-gt(i,j,k))
           xdp(i,j,k)=(xt(i,j,k)/(dx2(i))+0.5*dt(i,j,k)/dx(i))*xyz(i,j,k)
           xdm(i,j,k)=(xt(i,j,k)/(dx2(i))-0.5*dt(i,j,k)/dx(i))*xyz(i,j,k)
           yep(i,j,k)=(yt(i,j,k)/(dy2(j))+0.5*et(i,j,k)/dy(j))*xyz(i,j,k)
           yem(i,j,k)=(yt(i,j,k)/(dy2(j))-0.5*et(i,j,k)/dy(j))*xyz(i,j,k)
           zfp(i,j,k)=(zt(i,j,k)/(dz2(k))+0.5*ft(i,j,k)/dz(k))*xyz(i,j,k)
           zfm(i,j,k)=(zt(i,j,k)/(dz2(k))-0.5*ft(i,j,k)/dz(k))*xyz(i,j,k)
           if(signa==1)then
              at(i,j,k)=0.25*at(i,j,k)/(dxdy(i,j))*xyz(i,j,k)
           else
              at(i,j,k)=0.0
           end if
           if(signb==1)then
              bt(i,j,k)=0.25*bt(i,j,k)/(dydz(j,k))*xyz(i,j,k)
           else
              bt(i,j,k)=0.0
           end if
           if(signc==1)then
              ct(i,j,k)=0.25*ct(i,j,k)/(dxdz(i,k))*xyz(i,j,k)
           else
              ct(i,j,k)=0.0
           end if
        end do
     end do
  end do

!$omp end do
!$omp end parallel

  err_max=eps  ! while に入るための便宜的措置
  counter=0

!-- 実際のソルバ ---
  do while(err_max>=eps)
     err_max=0.0

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k,ix,jy,kz)
     do k=1,nz
        do j=1,ny
           do i=1,nx

!-- 以降, 反復計算に必要な点の周囲 8 点についてそれぞれ
!-- inner_flag のチェックと同時に適切な値をそれぞれ逐次計算する,
!-- 下で, 各方向の格子計算を行っているが, ib=8,-8 は 4 隅格子のときにしか
!-- case select しない. なぜなら, 真横や上下に隅格子がくることはありえない.
              if(inner_flag(i,j,k).eqv..false.)then  ! .false. なら領域計算開始

                 tmp(i,j,k)=-rho(i,j,k)*xyz(i,j,k)
                 tmp(i,j,k)=tmp(i,j,k)+xdp(i,j,k)*psi(i+1,j,k)  &
  &                                   +xdm(i,j,k)*psi(i-1,j,k)  &
  &                                   +yep(i,j,k)*psi(i,j+1,k)  &
  &                                   +yem(i,j,k)*psi(i,j-1,k)  &
  &                                   +zfp(i,j,k)*psi(i,j,k+1)  &
  &                                   +zfm(i,j,k)*psi(i,j,k-1)


                 if(signa==1)then
                    tmp_b(i,j,k)=at(i,j,k)*(psi(i+1,j+1,k)+psi(i-1,j-1,k)  &
  &                                        -psi(i+1,j-1,k)-psi(i-1,j+1,k))
                 else
                    tmp_b(i,j,k)=0.0
                 end if
                 if(signb==1)then  ! そもそも bt = 0 なら計算しない.
                    tmp_b(i,j,k)=tmp_b(i,j,k)  &
  &                             +bt(i,j,k)*(psi(i,j+1,k+1)+psi(i,j-1,k-1)  &
  &                                        -psi(i,j-1,k+1)-psi(i,j+1,k-1))
                 end if
                 if(signc==1)then  ! そもそも bt = 0 なら計算しない.
                    tmp_b(i,j,k)=tmp_b(i,j,k)  &
  &                             +ct(i,j,k)*(psi(i+1,j,k+1)+psi(i-1,j,k-1)  &
  &                                        -psi(i-1,j,k+1)-psi(i+1,j,k-1))
                 end if
              
                 tmp(i,j,k)=tmp(i,j,k)+tmp_b(i,j,k)

              else   ! .true. なら境界計算開始.

                 select case (ib(i,j,k))
                 case (1)
                    tmp(i,j,k)=bnd(i,j,k)

                 case (2)  ! x 方向にフラックス一定, 上側が参照値
                    tmp(i,j,k)=psi(i+1,j,k)-bnd(i,j,k)*dx(i)

                 case (-2)  ! x 方向にフラックス一定, 下側が参照値
                    tmp(i,j,k)=psi(i-1,j,k)+bnd(i,j,k)*dx(i)

                 case (4)  ! y 方向にフラックス一定, 右側が参照値
                    tmp(i,j,k)=psi(i,j+1,k)-bnd(i,j,k)*dy(j)

                 case (-4)  ! y 方向にフラックス一定, 左側が参照値
                    tmp(i,j,k)=psi(i,j-1,k)+bnd(i,j,k)*dy(j)

                 case (6)  ! y 方向にフラックス一定, 右側が参照値
                    tmp(i,j,k)=psi(i,j,k+1)-bnd(i,j,k)*dz(k)

                 case (-6)  ! y 方向にフラックス一定, 左側が参照値
                    tmp(i,j,k)=psi(i,j,k-1)+bnd(i,j,k)*dz(k)

                 case (3)  ! 12 辺, もしくは 8 点で周期境界を判断
                    if(i==1)then
                       ix=nx-1
                    else if(i==nx)then
                       ix=2
                    else
                       ix=i
                    end if
                    if(j==1)then
                       jy=ny-1
                    else if(j==ny)then
                       jy=2
                    else
                       jy=j
                    end if
                    if(k==1)then
                       kz=nz-1
                    else if(k==nz)then
                       kz=2
                    else
                       kz=k
                    end if
                    tmp(i,j,k)=psi(ix,jy,kz)

                 case (8)  ! 両方フラックス一定で z 面の x, y 右上か左下角.
                    if(i==1.and.j==1)then  ! -- 評価 1
                       tmp(i,j,k)=psi(i+1,j+1,k)  &
  &                        -0.5*(bnd(i+1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))

                    else if(i==nx.and.j==ny)then  ! -- 評価 2
                       tmp(i,j,k)=psi(i-1,j-1,k)  &
  &                        +0.5*(bnd(i-1,j,k)*dy(j)+bnd(i,j-1,k)*dx(i))

                    else if(ib(i-1,j,k)==10.and.ib(i,j-1,k)==10)then
                       ! -- 評価 1 と同じ
                       tmp(i,j,k)=psi(i+1,j+1,k)  &
  &                        -0.5*(bnd(i+1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))

                    else if(ib(i+1,j,k)==10.and.ib(i,j+1,k)==10)then
                       ! -- 評価 2 と同じ
                       tmp(i,j,k)=psi(i-1,j-1,k)  &
  &                        +0.5*(bnd(i-1,j,k)*dy(j)+bnd(i,j-1,k)*dx(i))

                    end if

                 case (-8)  ! 両方フラックス一定で z 面の x, y 右下か左上角.
                    if(i==1.and.j==ny)then  ! -- 評価 1
                       tmp(i,j,k)=psi(i+1,j-1,k)  &
  &                        +0.5*(bnd(i+1,j,k)*dy(j)-bnd(i,j-1,k)*dx(i))

                    else if(i==nx.and.j==1)then  ! -- 評価 2
                       tmp(i,j,k)=psi(i-1,j+1,k)  &
  &                        +0.5*(-bnd(i-1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))

                    else if(ib(i-1,j,k)==10.and.ib(i,j+1,k)==10)then
                       ! -- 評価 1 と同じ
                       tmp(i,j,k)=psi(i+1,j-1,k)  &
  &                        +0.5*(bnd(i+1,j,k)*dy(j)-bnd(i,j-1,k)*dx(i))

                    else if(ib(i+1,j,k)==10.and.ib(i,j-1,k)==10)then
                       ! -- 評価 2 と同じ
                       tmp(i,j,k)=psi(i-1,j+1,k)  &
  &                        +0.5*(-bnd(i-1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))
                    end if

                 case (12)  ! 両方フラックス一定で y 面の x, z 右上か左下角.
                    if(i==1.and.k==1)then  ! -- 評価 1
                       tmp(i,j,k)=psi(i+1,j,k+1)  &
  &                        -0.5*(bnd(i+1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                    else if(i==nx.and.k==nz)then  ! -- 評価 2
                       tmp(i,j,k)=psi(i-1,j,k-1)  &
  &                        +0.5*(bnd(i-1,j,k)*dz(k)+bnd(i,j,k-1)*dx(i))

                    else if(ib(i-1,j,k)==10.and.ib(i,j,k-1)==10)then
                       ! -- 評価 1 と同じ
                       tmp(i,j,k)=psi(i+1,j,k+1)  &
  &                        -0.5*(bnd(i+1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                    else if(ib(i+1,j,k)==10.and.ib(i,j,k+1)==10)then
                       ! -- 評価 2 と同じ
                       tmp(i,j,k)=psi(i-1,j,k-1)  &
  &                        +0.5*(bnd(i-1,j,k)*dz(k)+bnd(i,j,k-1)*dx(i))

                    end if

                 case (-12)  ! 両方フラックス一定で y 面の x, z 右下か左上角.
                    if(i==1.and.k==nz)then  ! -- 評価 1
                       tmp(i,j,k)=psi(i+1,j,k-1)  &
  &                        +0.5*(bnd(i+1,j,k)*dz(k)-bnd(i,j,k-1)*dx(i))

                    else if(i==nx.and.k==1)then  ! -- 評価 2
                       tmp(i,j,k)=psi(i-1,j,k+1)  &
  &                        +0.5*(-bnd(i-1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                    else if(ib(i-1,j,k)==10.and.ib(i,j,k+1)==10)then
                       ! -- 評価 1 と同じ
                       tmp(i,j,k)=psi(i+1,j,k-1)  &
  &                        +0.5*(bnd(i+1,j,k)*dz(k)-bnd(i,j,k-1)*dx(i))

                    else if(ib(i+1,j,k)==10.and.ib(i,j,k-1)==10)then
                       ! -- 評価 2 と同じ
                       tmp(i,j,k)=psi(i-1,j,k+1)  &
  &                        +0.5*(-bnd(i-1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                    end if

                 case (24)  ! 両方フラックス一定で x 面の y, z 右上か左下角.
                    if(j==1.and.k==1)then  ! -- 評価 1
                       tmp(i,j,k)=psi(i,j+1,k+1)  &
  &                        -0.5*(bnd(i,j+1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                    else if(j==ny.and.k==nz)then  ! -- 評価 2
                       tmp(i,j,k)=psi(i,j-1,k-1)  &
  &                        +0.5*(bnd(i,j-1,k)*dz(k)+bnd(i,j,k-1)*dy(j))

                    else if(ib(i,j-1,k)==10.and.ib(i,j,k-1)==10)then
                       ! -- 評価 1 と同じ
                       tmp(i,j,k)=psi(i,j+1,k+1)  &
  &                        -0.5*(bnd(i,j+1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                    else if(ib(i,j+1,k)==10.and.ib(i,j,k+1)==10)then
                       ! -- 評価 2 と同じ
                       tmp(i,j,k)=psi(i,j-1,k-1)  &
  &                        +0.5*(bnd(i,j-1,k)*dz(k)+bnd(i,j,k-1)*dy(j))

                    end if

                 case (-24)  ! 両方フラックス一定で x 面の y, z 右下か左上角.
                    if(j==1.and.k==nz)then  ! -- 評価 1
                       tmp(i,j,k)=psi(i,j+1,k-1)  &
  &                        +0.5*(bnd(i,j+1,k)*dz(k)-bnd(i,j,k-1)*dy(j))

                    else if(j==ny.and.k==1)then  ! -- 評価 2
                       tmp(i,j,k)=psi(i,j-1,k+1)  &
  &                        +0.5*(-bnd(i,j-1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                    else if(ib(i,j-1,k)==10.and.ib(i,j,k+1)==10)then
                       ! -- 評価 1 と同じ
                       tmp(i,j,k)=psi(i,j+1,k-1)  &
  &                        +0.5*(bnd(i,j+1,k)*dz(k)-bnd(i,j,k-1)*dy(j))

                    else if(ib(i,j+1,k)==10.and.ib(i,j,k-1)==10)then
                       ! -- 評価 2 と同じ
                       tmp(i,j,k)=psi(i,j-1,k+1)  &
  &                        +0.5*(-bnd(i,j-1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                    end if

                 !-- 以降, 隅領域なので, 個別に設定.
                 case (11)  ! 両方フラックス一定で (1,1,1) 点, もしくは内部領域.
                    tmp(i,j,k)=psi(i+1,j+1,k+1)  &
  &                           -(bnd(i,j+1,k+1)*dx(i)  &
  &                            +bnd(i+1,j,k+1)*dy(j)  &
  &                            +bnd(i+1,j+1,k)*dz(k))/3.0

                 case (13)  ! 両方フラックス一定で (nx,1,1) 点, もしくは内部領域.
                    tmp(i,j,k)=psi(i-1,j+1,k+1)  &
  &                           -(-bnd(i,j+1,k+1)*dx(i)  &
  &                             +bnd(i-1,j,k+1)*dy(j)  &
  &                             +bnd(i-1,j+1,k)*dz(k))/3.0

                 case (17)  ! 両方フラックス一定で (1,ny,1) 点, もしくは内部領域.
                    tmp(i,j,k)=psi(i+1,j-1,k+1)  &
  &                           -(bnd(i,j-1,k+1)*dx(i)  &
  &                            -bnd(i+1,j,k+1)*dy(j)  &
  &                            +bnd(i+1,j-1,k)*dz(k))/3.0

                 case (19)  ! 両方フラックス一定で (nx,ny,1) 点, もしくは内部領域.
                    tmp(i,j,k)=psi(i-1,j-1,k+1)  &
  &                           -(-bnd(i,j-1,k+1)*dx(i)  &
  &                             -bnd(i-1,j,k+1)*dy(j)  &
  &                             +bnd(i-1,j-1,k)*dz(k))/3.0

                 case (23)  ! 両方フラックス一定で (1,1,nz) 点, もしくは内部領域.
                    tmp(i,j,k)=psi(i+1,j+1,k-1)  &
  &                           -(bnd(i,j+1,k-1)*dx(i)  &
  &                            +bnd(i+1,j,k-1)*dy(j)  &
  &                            -bnd(i+1,j+1,k)*dz(k))/3.0

                 case (29)  ! 両方フラックス一定で (nx,1,nz) 点, もしくは内部領域.
                    tmp(i,j,k)=psi(i-1,j+1,k-1)  &
  &                           -(-bnd(i,j+1,k-1)*dx(i)  &
  &                             +bnd(i-1,j,k-1)*dy(j)  &
  &                             -bnd(i-1,j+1,k)*dz(k))/3.0

                 case (31)  ! 両方フラックス一定で (1,ny,nz) 点, もしくは内部領域.
                    tmp(i,j,k)=psi(i+1,j-1,k-1)  &
  &                           -(bnd(i,j-1,k-1)*dx(i)  &
  &                            -bnd(i+1,j,k-1)*dy(j)  &
  &                            -bnd(i+1,j-1,k)*dz(k))/3.0

                 case (37)  ! 両方フラックス一定で (nx,ny,nz) 点, もしくは内部領域.
                    tmp(i,j,k)=psi(i-1,j-1,k-1)  &
  &                           -(-bnd(i,j-1,k-1)*dx(i)  &
  &                             -bnd(i-1,j,k-1)*dy(j)  &
  &                             -bnd(i-1,j-1,k)*dz(k))/3.0

                 case (-11)  ! 両方フラックス一定で (i+1,j+1,k+1) に 10 が設定.
                    tmp(i,j,k)=(psi(i-1,j+1,k+1)+psi(i+1,j-1,k+1)+psi(i+1,j+1,k-1)  &
  &                            +bnd(i,j+1,k+1)*dx(i)+bnd(i+1,j,k+1)*dy(j)  &
  &                            +bnd(i+1,j+1,k)*dz(k))/3.0

                 case (-13)  ! 両方フラックス一定で (i-1,j+1,k+1) に 10 が設定.
                    tmp(i,j,k)=(psi(i+1,j+1,k+1)+psi(i-1,j-1,k+1)+psi(i-1,j+1,k-1)  &
  &                            -bnd(i,j+1,k+1)*dx(i)+bnd(i-1,j,k+1)*dy(j)  &
  &                            +bnd(i-1,j+1,k)*dz(k))/3.0

                 case (-17)  ! 両方フラックス一定で (i+1,j-1,k+1) に 10 が設定.
                    tmp(i,j,k)=(psi(i-1,j-1,k+1)+psi(i+1,j+1,k+1)+psi(i+1,j-1,k-1)  &
  &                            +bnd(i,j-1,k+1)*dx(i)-bnd(i+1,j,k+1)*dy(j)  &
  &                            +bnd(i+1,j-1,k)*dz(k))/3.0

                 case (-19)  ! 両方フラックス一定で (i-1,j-1,k+1) に 10 が設定.
                    tmp(i,j,k)=(psi(i+1,j-1,k+1)+psi(i-1,j+1,k+1)+psi(i-1,j-1,k-1)  &
  &                            -bnd(i,j-1,k+1)*dx(i)-bnd(i-1,j,k+1)*dy(j)  &
  &                            +bnd(i-1,j-1,k)*dz(k))/3.0

                 case (-23)  ! 両方フラックス一定で (i+1,j+1,k-1) に 10 が設定.
                    tmp(i,j,k)=(psi(i-1,j+1,k-1)+psi(i+1,j-1,k-1)+psi(i+1,j+1,k+1)  &
  &                            +bnd(i,j+1,k-1)*dx(i)+bnd(i+1,j,k-1)*dy(j)  &
  &                            -bnd(i+1,j+1,k)*dz(k))/3.0

                 case (-29)  ! 両方フラックス一定で (i-1,j+1,k-1) に 10 が設定.
                    tmp(i,j,k)=(psi(i+1,j+1,k-1)+psi(i-1,j-1,k-1)+psi(i-1,j+1,k+1)  &
  &                            -bnd(i,j+1,k-1)*dx(i)+bnd(i-1,j,k-1)*dy(j)  &
  &                            -bnd(i-1,j+1,k)*dz(k))/3.0

                 case (-31)  ! 両方フラックス一定で (i+1,j-1,k-1) に 10 が設定.
                    tmp(i,j,k)=(psi(i-1,j-1,k-1)+psi(i+1,j+1,k-1)+psi(i+1,j-1,k+1)  &
  &                            +bnd(i,j-1,k-1)*dx(i)-bnd(i+1,j,k-1)*dy(j)  &
  &                            -bnd(i+1,j-1,k)*dz(k))/3.0

                 case (-37)  ! 両方フラックス一定で (i-1,j-1,k-1) に 10 が設定.
                    tmp(i,j,k)=(psi(i+1,j-1,k-1)+psi(i-1,j+1,k-1)+psi(i-1,j-1,k+1)  &
  &                            -bnd(i,j-1,k-1)*dx(i)-bnd(i-1,j,k-1)*dy(j)  &
  &                            -bnd(i-1,j-1,k)*dz(k))/3.0

                 end select

              end if

              if(sor_flag.eqv..true.)then
                 tmp(i,j,k)=(1.0-accc)*psi(i,j,k)+tmp(i,j,k)*accc
              end if

           end do
        end do
     end do
!$omp end do
!$omp end parallel

!-- 最大誤差の更新
     do k=2,nz-1
        do j=2,ny-1
           do i=2,nx-1
              err=abs(tmp(i,j,k)-psi(i,j,k))
              if(err_max<=err)then
                 err_max=err
              end if
           end do
        end do
     end do

!-- 一斉更新

     do k=1,nz
        do j=1,ny
           do i=1,nx
              psi(i,j,k)=tmp(i,j,k)
           end do
        end do
     end do

     if(nl/=0)then
        counter=counter+1
        if(counter==nl)then
           exit
        else
           err_max=eps
        end if
     end if

  end do

!-- 境界の設定

  call calculate_bound_3df( ib, dx, dy, dz, bnd, psi )

!-- 未定義領域には undef を代入する.

  do k=1,nz
     do j=1,ny
        do i=1,nx
           if(ib(i,j,k)==10)then
              psi(i,j,k)=defun
           end if
        end do
     end do
  end do

end subroutine Ellip_Jacobi_3df

!----------------------------------
!----------------------------------

subroutine Ellip_Jacobi_3dd(x, y, z, rho, eps, boundary, psi, bound_opt,  &
  &                         xa, ya, za, a, b, c, d, e, f, g,  &
  &                         undef, inner_bound, init_flag, accel, ln, check_flag )
use omp_lib
! openmp によるスレッド並列が可能.
! ガウス・ザイデル法ではアルゴリズムの点から並列化が困難と思われたので,
! 並列計算によるポアソン方程式の求積が必要となるなら,
! ヤコビ法のものを使用されたい.
! 各オプション配列は, ポアソン系の各微分項の値. デフォルトはゼロで設定される.
! $$xa\dfrac{\partial ^2\psi}{\partial x^2} +ya\dfrac{\partial ^2\psi}{\partial y^2} +za\dfrac{\partial ^2\psi}{\partial z^2} +a\dfrac{\partial ^2\psi}{\partial x\partial y} +b\dfrac{\partial ^2\psi}{\partial y\partial z} +c\dfrac{\partial ^2\psi}{\partial z\partial x} +d\dfrac{\partial \psi}{\partial x} +e\dfrac{\partial \psi}{\partial y} +f\dfrac{\partial \psi}{\partial z} +g\psi =\rho $$
! の各係数に対応している.
  implicit none
  double precision, intent(in) :: x(:)  ! 領域の x 座標
  double precision, intent(in) :: y(:)  ! 領域の y 座標
  double precision, intent(in) :: z(:)  ! 領域の z 座標
  double precision, intent(in) :: rho(size(x),size(y),size(z))  ! ポアソン方程式の強制項
                   ! rho =0 でラプラス方程式も求積可能
  double precision, intent(in) :: eps  ! 収束条件
  character(6), intent(in) :: boundary  ! 境界条件
                ! 4 文字で各辺の境界条件を与える.
                ! 1 文字目 : x 下端, 2 文字目 : y 左端, 3 文字目 : x 上端,
                ! 4 文字目 : y 右端
                ! boundary は 1 : 固定端境界, 2 : 自由端境界, 3 : 周期境界
  double precision, intent(in), optional :: bound_opt(size(x),size(y),size(z))
                             ! 境界での強制
                             ! ノイマン境界の場合 : フラックス値
  double precision, intent(in), optional :: xa(size(x),size(y),size(z))  ! 各微分項の係数
  double precision, intent(in), optional :: ya(size(x),size(y),size(z))  ! 各微分項の係数
  double precision, intent(in), optional :: za(size(x),size(y),size(z))  ! 各微分項の係数
  double precision, intent(in), optional :: a(size(x),size(y),size(z))  ! 各微分項の係数
  double precision, intent(in), optional :: b(size(x),size(y),size(z))  ! 各微分項の係数
  double precision, intent(in), optional :: c(size(x),size(y),size(z))  ! 各微分項の係数
  double precision, intent(in), optional :: d(size(x),size(y),size(z))  ! 各微分項の係数
  double precision, intent(in), optional :: e(size(x),size(y),size(z))  ! 各微分項の係数
  double precision, intent(in), optional :: f(size(x),size(y),size(z))  ! 各微分項の係数
  double precision, intent(in), optional :: g(size(x),size(y),size(z))  ! 各微分項の係数
  double precision, intent(inout) :: psi(size(x),size(y),size(z))  ! ポアソン方程式の解
  double precision, intent(in), optional :: undef  ! 未定義値
  integer, intent(in), optional :: inner_bound(size(x),size(y),size(z))
                             ! 内部領域の境界. 値に応じてその格子点で境界値計算
                             ! 1 = 固定端境界, 10 = 境界の内側.
                             ! 2 = y 方向自由端境界 (フラックスは上向き)
                             ! -2 = y 方向自由端境界 (フラックスは下向き)
                             ! 4 = x 方向自由端境界 (フラックスは右向き)
                             ! -4 = x 方向自由端境界 (フラックスは左向き)
                             ! 3 = 周期境界
                             ! 8 = |_, ~| で両方とも自由境界条件
                             ! -8 = |~, _| で両方とも自由境界条件
                             ! この引数が与えられなければ全領域を計算する.
                             ! 境界の内側格子点 (10) は反復計算を行わず,
                             ! undef で設定された値もしくはゼロが入る.
                             ! このときの境界値は bound_opt の値が用いられる.
  logical, intent(in), optional :: init_flag  ! psi の値をゼロで初期化するか.
                             ! .true. = 初期化する. .false. = 初期化しない.
                             ! デフォルトでは初期化する.
  double precision, intent(in), optional :: accel  ! SOR の加速係数 (0 < accel < 2)
                             ! デフォルト = 1
  integer, intent(in), optional :: ln  ! 反復回数
                             ! この値が与えられるとき, eps の値に無関係に
                             ! ln 回ループさせる.
  logical, intent(in), optional :: check_flag  ! true の場合, 最高階数の
                             ! 係数が負号でも stop する.
                             ! default = .false. (係数が 0 になる場合のみ stop)

  integer :: i, j, k, ix, jy, kz, nl, counter
  integer :: nx  ! x 方向の配列要素
  integer :: ny  ! y 方向の配列要素
  integer :: nz  ! z 方向の配列要素
  integer :: signa, signb, signc  ! 各係数を計算するかどうか
  integer, dimension(size(x),size(y),size(z)) :: ib

  double precision :: defun
  double precision :: err, err_max, accc
  double precision :: bnd(size(x),size(y),size(z))
  double precision :: dx(size(x)), dy(size(y)), dx2(size(x)), dy2(size(y))
  double precision :: dz(size(z)), dz2(size(z))
  double precision, dimension(size(x),size(y)) :: dxdy
  double precision, dimension(size(y),size(z)) :: dydz
  double precision, dimension(size(x),size(z)) :: dxdz
  double precision, dimension(size(x),size(y),size(z)) :: xt, yt, zt, at, bt, ct, dt, et, ft, gt
  double precision, dimension(size(x),size(y),size(z)) :: xdp, xdm, yep, yem, zfp, zfm, xyz, divi
  double precision, dimension(size(x),size(y),size(z)) :: tmp, tmp_b

  character(6) :: bound
  logical :: sor_flag
  logical, dimension(size(x),size(y),size(z)) :: inner_flag

  bound(1:6)=boundary(1:6)

  nx=size(x)
  ny=size(y)
  nz=size(z)

!-- 応答関数の初期化

  if(present(init_flag))then
     if(init_flag.eqv..true.)then
        psi = 0.0d0
     end if
  else
     psi = 0.0d0
  end if

!-- 内部境界の判別フラグの設定

  if(present(inner_bound))then
     call set_bound_3d( bound, ib, inner_flag, inner_bound )
  else
     call set_bound_3d( bound, ib, inner_flag )
  end if

!-- 領域・内部境界における境界値の設定

  if(present(bound_opt))then
     call setval_bound_3dd( ib, bnd, psi, bound_opt )
  else
     call setval_bound_3dd( ib, bnd, psi )
  end if

!-- 未定義値の設定

  if(present(undef))then
     defun=undef
  else
     defun=0.0d0
  end if

!-- 係数の代入
!-- xa, ya, za については, 値が入れられていなければ, 全配列に 1 を代入する.
  if(present(xa))then
     call set_coe_3dd( xt, ext=xa )
  else
     call set_coe_3dd( xt, def=1.0d0 )
  end if

  if(present(ya))then
     call set_coe_3dd( yt, ext=ya )
  else
     call set_coe_3dd( yt, def=1.0d0 )
  end if

  if(present(za))then
     call set_coe_3dd( zt, ext=za )
  else
     call set_coe_3dd( zt, def=1.0d0 )
  end if

  if(present(a))then
     call set_coe_3dd( at, ext=a )
     signa=1
  else
     call set_coe_3dd( at, def=0.0d0 )
     signa=0
  end if

  if(present(b))then
     call set_coe_3dd( bt, ext=b )
     signb=1
  else
     call set_coe_3dd( bt, def=0.0d0 )
     signb=0
  end if

  if(present(c))then
     signc=1
     call set_coe_3dd( ct, ext=c )
  else
     call set_coe_3dd( ct, def=0.0d0 )
     signc=0
  end if

  if(present(d))then
     call set_coe_3dd( dt, ext=d )
  else
     call set_coe_3dd( dt, def=0.0d0 )
  end if

  if(present(e))then
     call set_coe_3dd( et, ext=e )
  else
     call set_coe_3dd( et, def=0.0d0 )
  end if

  if(present(f))then
     call set_coe_3dd( ft, ext=f )
  else
     call set_coe_3dd( ft, def=0.0d0 )
  end if

  if(present(g))then
     call set_coe_3dd( gt, ext=g )
  else
     call set_coe_3dd( gt, def=0.0d0 )
  end if

!-- 最高階数における係数チェック. (係数がゼロでないか調べる.)

  if(present(check_flag))then
     if(check_flag.eqv..true.)then
        call check_le_coe_3dd( xt(2:nx-1,2:ny-1,2:nz-1), 0.0d0, undeff=defun )
        call check_le_coe_3dd( yt(2:nx-1,2:ny-1,2:nz-1), 0.0d0, undeff=defun )
        call check_le_coe_3dd( zt(2:nx-1,2:ny-1,2:nz-1), 0.0d0, undeff=defun )
     else
        call check_coe_3dd( xt(2:nx-1,2:ny-1,2:nz-1), 0.0d0, undeff=defun )
        call check_coe_3dd( yt(2:nx-1,2:ny-1,2:nz-1), 0.0d0, undeff=defun )
        call check_coe_3dd( zt(2:nx-1,2:ny-1,2:nz-1), 0.0d0, undeff=defun )
     end if
  else
     call check_coe_3dd( xt(2:nx-1,2:ny-1,2:nz-1), 0.0d0, undeff=defun )
     call check_coe_3dd( yt(2:nx-1,2:ny-1,2:nz-1), 0.0d0, undeff=defun )
     call check_coe_3dd( zt(2:nx-1,2:ny-1,2:nz-1), 0.0d0, undeff=defun )
  end if

!-- 加速係数の判定

  if(present(accel))then
     accc=accel
     sor_flag=.true.
  else
     sor_flag=.false.
  end if

!-- ln の処理
  if(present(ln))then
     nl=ln
  else
     nl=0
  end if

!-- 以下で先に格子間隔等の 1 回計算でよいものを求めておく.
!-- これらは 1 方向のみで変化すればよい.
!-- 格子点間隔の計算
  do i=2,nx-1
     dx(i)=(x(i+1)-x(i-1))*0.5d0
     dx2(i)=dx(i)**2
  end do
  do j=2,ny-1
     dy(j)=(y(j+1)-y(j-1))*0.5d0
     dy2(j)=dy(j)**2
  end do
  do k=2,nz-1
     dz(k)=(z(k+1)-z(k-1))*0.5d0
     dz2(k)=dz(k)**2
  end do

  dx(1)=(x(2)-x(1))
  dx(nx)=(x(nx)-x(nx-1))
  dy(1)=(y(2)-y(1))
  dy(ny)=(y(ny)-y(ny-1))
  dz(1)=(z(2)-z(1))
  dz(nz)=(z(nz)-z(nz-1))

  do j=1,ny
     do i=1,nx
        dxdy(i,j)=dx(i)*dy(j)
     end do
  end do
  do k=1,nz
     do j=1,ny
        dydz(j,k)=dy(j)*dz(k)
     end do
  end do
  do k=1,nz
     do i=1,nx
        dxdz(i,k)=dx(i)*dz(k)
     end do
  end do

!-- ポアソン係数の計算
!-- 実際にポアソン係数が使用されるのは, 境界を除いた 1 回り内側なので,
!-- 計算量削減のため, ループをそのようにしておく.

  xyz=0.0d0
  xdp=0.0d0
  xdm=0.0d0
  yep=0.0d0
  yem=0.0d0
  zfp=0.0d0
  zfm=0.0d0

!-- 最高次数係数 ac の計算
!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)

  do k=2,nz-1
     do j=2,ny-1
        do i=2,nx-1
           xyz(i,j,k)=1.0d0/(2.0d0*(xt(i,j,k)/dx2(i)+yt(i,j,k)/dy2(j)+zt(i,j,k)/dz2(k))-gt(i,j,k))
           xdp(i,j,k)=(xt(i,j,k)/(dx2(i))+0.5d0*dt(i,j,k)/dx(i))*xyz(i,j,k)
           xdm(i,j,k)=(xt(i,j,k)/(dx2(i))-0.5d0*dt(i,j,k)/dx(i))*xyz(i,j,k)
           yep(i,j,k)=(yt(i,j,k)/(dy2(j))+0.5d0*et(i,j,k)/dy(j))*xyz(i,j,k)
           yem(i,j,k)=(yt(i,j,k)/(dy2(j))-0.5d0*et(i,j,k)/dy(j))*xyz(i,j,k)
           zfp(i,j,k)=(zt(i,j,k)/(dz2(k))+0.5d0*ft(i,j,k)/dz(k))*xyz(i,j,k)
           zfm(i,j,k)=(zt(i,j,k)/(dz2(k))-0.5d0*ft(i,j,k)/dz(k))*xyz(i,j,k)
           if(signa==1)then
              at(i,j,k)=0.25d0*at(i,j,k)/(dxdy(i,j))*xyz(i,j,k)
           else
              at(i,j,k)=0.0d0
           end if
           if(signb==1)then
              bt(i,j,k)=0.25d0*bt(i,j,k)/(dydz(j,k))*xyz(i,j,k)
           else
              bt(i,j,k)=0.0d0
           end if
           if(signc==1)then
              ct(i,j,k)=0.25d0*ct(i,j,k)/(dxdz(i,k))*xyz(i,j,k)
           else
              ct(i,j,k)=0.0d0
           end if
        end do
     end do
  end do

!$omp end do
!$omp end parallel

  err_max=eps  ! while に入るための便宜的措置
  counter=0

!-- 実際のソルバ ---
  do while(err_max>=eps)
     err_max=0.0d0

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k,ix,jy,kz)
     do k=1,nz
        do j=1,ny
           do i=1,nx

!-- 以降, 反復計算に必要な点の周囲 8 点についてそれぞれ
!-- inner_flag のチェックと同時に適切な値をそれぞれ逐次計算する,
!-- 下で, 各方向の格子計算を行っているが, ib=8,-8 は 4 隅格子のときにしか
!-- case select しない. なぜなら, 真横や上下に隅格子がくることはありえない.
              if(inner_flag(i,j,k).eqv..false.)then  ! .false. なら領域計算開始

                 tmp(i,j,k)=-rho(i,j,k)*xyz(i,j,k)
                 tmp(i,j,k)=tmp(i,j,k)+xdp(i,j,k)*psi(i+1,j,k)  &
  &                                   +xdm(i,j,k)*psi(i-1,j,k)  &
  &                                   +yep(i,j,k)*psi(i,j+1,k)  &
  &                                   +yem(i,j,k)*psi(i,j-1,k)  &
  &                                   +zfp(i,j,k)*psi(i,j,k+1)  &
  &                                   +zfm(i,j,k)*psi(i,j,k-1)


                 if(signa==1)then
                    tmp_b(i,j,k)=at(i,j,k)*(psi(i+1,j+1,k)+psi(i-1,j-1,k)  &
  &                                        -psi(i+1,j-1,k)-psi(i-1,j+1,k))
                 else
                    tmp_b(i,j,k)=0.0d0
                 end if
                 if(signb==1)then  ! そもそも bt = 0 なら計算しない.
                    tmp_b(i,j,k)=tmp_b(i,j,k)  &
  &                             +bt(i,j,k)*(psi(i,j+1,k+1)+psi(i,j-1,k-1)  &
  &                                        -psi(i,j-1,k+1)-psi(i,j+1,k-1))
                 end if
                 if(signc==1)then  ! そもそも bt = 0 なら計算しない.
                    tmp_b(i,j,k)=tmp_b(i,j,k)  &
  &                             +ct(i,j,k)*(psi(i+1,j,k+1)+psi(i-1,j,k-1)  &
  &                                        -psi(i-1,j,k+1)-psi(i+1,j,k-1))
                 end if
              
                 tmp(i,j,k)=tmp(i,j,k)+tmp_b(i,j,k)

              else   ! .true. なら境界計算開始.

                 select case (ib(i,j,k))
                 case (1)
                    tmp(i,j,k)=bnd(i,j,k)

                 case (2)  ! x 方向にフラックス一定, 上側が参照値
                    tmp(i,j,k)=psi(i+1,j,k)-bnd(i,j,k)*dx(i)

                 case (-2)  ! x 方向にフラックス一定, 下側が参照値
                    tmp(i,j,k)=psi(i-1,j,k)+bnd(i,j,k)*dx(i)

                 case (4)  ! y 方向にフラックス一定, 右側が参照値
                    tmp(i,j,k)=psi(i,j+1,k)-bnd(i,j,k)*dy(j)

                 case (-4)  ! y 方向にフラックス一定, 左側が参照値
                    tmp(i,j,k)=psi(i,j-1,k)+bnd(i,j,k)*dy(j)

                 case (6)  ! y 方向にフラックス一定, 右側が参照値
                    tmp(i,j,k)=psi(i,j,k+1)-bnd(i,j,k)*dz(k)

                 case (-6)  ! y 方向にフラックス一定, 左側が参照値
                    tmp(i,j,k)=psi(i,j,k-1)+bnd(i,j,k)*dz(k)

                 case (3)  ! 12 辺, もしくは 8 点で周期境界を判断
                    if(i==1)then
                       ix=nx-1
                    else if(i==nx)then
                       ix=2
                    else
                       ix=i
                    end if
                    if(j==1)then
                       jy=ny-1
                    else if(j==ny)then
                       jy=2
                    else
                       jy=j
                    end if
                    if(k==1)then
                       kz=nz-1
                    else if(k==nz)then
                       kz=2
                    else
                       kz=k
                    end if
                    tmp(i,j,k)=psi(ix,jy,kz)

                 case (8)  ! 両方フラックス一定で z 面の x, y 右上か左下角.
                    if(i==1.and.j==1)then  ! -- 評価 1
                       tmp(i,j,k)=psi(i+1,j+1,k)  &
  &                        -0.5d0*(bnd(i+1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))

                    else if(i==nx.and.j==ny)then  ! -- 評価 2
                       tmp(i,j,k)=psi(i-1,j-1,k)  &
  &                        +0.5d0*(bnd(i-1,j,k)*dy(j)+bnd(i,j-1,k)*dx(i))

                    else if(ib(i-1,j,k)==10.and.ib(i,j-1,k)==10)then
                       ! -- 評価 1 と同じ
                       tmp(i,j,k)=psi(i+1,j+1,k)  &
  &                        -0.5d0*(bnd(i+1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))

                    else if(ib(i+1,j,k)==10.and.ib(i,j+1,k)==10)then
                       ! -- 評価 2 と同じ
                       tmp(i,j,k)=psi(i-1,j-1,k)  &
  &                        +0.5d0*(bnd(i-1,j,k)*dy(j)+bnd(i,j-1,k)*dx(i))

                    end if

                 case (-8)  ! 両方フラックス一定で z 面の x, y 右下か左上角.
                    if(i==1.and.j==ny)then  ! -- 評価 1
                       tmp(i,j,k)=psi(i+1,j-1,k)  &
  &                        +0.5d0*(bnd(i+1,j,k)*dy(j)-bnd(i,j-1,k)*dx(i))

                    else if(i==nx.and.j==1)then  ! -- 評価 2
                       tmp(i,j,k)=psi(i-1,j+1,k)  &
  &                        +0.5d0*(-bnd(i-1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))

                    else if(ib(i-1,j,k)==10.and.ib(i,j+1,k)==10)then
                       ! -- 評価 1 と同じ
                       tmp(i,j,k)=psi(i+1,j-1,k)  &
  &                        +0.5d0*(bnd(i+1,j,k)*dy(j)-bnd(i,j-1,k)*dx(i))

                    else if(ib(i+1,j,k)==10.and.ib(i,j-1,k)==10)then
                       ! -- 評価 2 と同じ
                       tmp(i,j,k)=psi(i-1,j+1,k)  &
  &                        +0.5d0*(-bnd(i-1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))
                    end if

                 case (12)  ! 両方フラックス一定で y 面の x, z 右上か左下角.
                    if(i==1.and.k==1)then  ! -- 評価 1
                       tmp(i,j,k)=psi(i+1,j,k+1)  &
  &                        -0.5d0*(bnd(i+1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                    else if(i==nx.and.k==nz)then  ! -- 評価 2
                       tmp(i,j,k)=psi(i-1,j,k-1)  &
  &                        +0.5d0*(bnd(i-1,j,k)*dz(k)+bnd(i,j,k-1)*dx(i))

                    else if(ib(i-1,j,k)==10.and.ib(i,j,k-1)==10)then
                       ! -- 評価 1 と同じ
                       tmp(i,j,k)=psi(i+1,j,k+1)  &
  &                        -0.5d0*(bnd(i+1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                    else if(ib(i+1,j,k)==10.and.ib(i,j,k+1)==10)then
                       ! -- 評価 2 と同じ
                       tmp(i,j,k)=psi(i-1,j,k-1)  &
  &                        +0.5d0*(bnd(i-1,j,k)*dz(k)+bnd(i,j,k-1)*dx(i))

                    end if

                 case (-12)  ! 両方フラックス一定で y 面の x, z 右下か左上角.
                    if(i==1.and.k==nz)then  ! -- 評価 1
                       tmp(i,j,k)=psi(i+1,j,k-1)  &
  &                        +0.5d0*(bnd(i+1,j,k)*dz(k)-bnd(i,j,k-1)*dx(i))

                    else if(i==nx.and.k==1)then  ! -- 評価 2
                       tmp(i,j,k)=psi(i-1,j,k+1)  &
  &                        +0.5d0*(-bnd(i-1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                    else if(ib(i-1,j,k)==10.and.ib(i,j,k+1)==10)then
                       ! -- 評価 1 と同じ
                       tmp(i,j,k)=psi(i+1,j,k-1)  &
  &                        +0.5d0*(bnd(i+1,j,k)*dz(k)-bnd(i,j,k-1)*dx(i))

                    else if(ib(i+1,j,k)==10.and.ib(i,j,k-1)==10)then
                       ! -- 評価 2 と同じ
                       tmp(i,j,k)=psi(i-1,j,k+1)  &
  &                        +0.5d0*(-bnd(i-1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                    end if

                 case (24)  ! 両方フラックス一定で x 面の y, z 右上か左下角.
                    if(j==1.and.k==1)then  ! -- 評価 1
                       tmp(i,j,k)=psi(i,j+1,k+1)  &
  &                        -0.5d0*(bnd(i,j+1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                    else if(j==ny.and.k==nz)then  ! -- 評価 2
                       tmp(i,j,k)=psi(i,j-1,k-1)  &
  &                        +0.5d0*(bnd(i,j-1,k)*dz(k)+bnd(i,j,k-1)*dy(j))

                    else if(ib(i,j-1,k)==10.and.ib(i,j,k-1)==10)then
                       ! -- 評価 1 と同じ
                       tmp(i,j,k)=psi(i,j+1,k+1)  &
  &                        -0.5d0*(bnd(i,j+1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                    else if(ib(i,j+1,k)==10.and.ib(i,j,k+1)==10)then
                       ! -- 評価 2 と同じ
                       tmp(i,j,k)=psi(i,j-1,k-1)  &
  &                        +0.5d0*(bnd(i,j-1,k)*dz(k)+bnd(i,j,k-1)*dy(j))

                    end if

                 case (-24)  ! 両方フラックス一定で x 面の y, z 右下か左上角.
                    if(j==1.and.k==nz)then  ! -- 評価 1
                       tmp(i,j,k)=psi(i,j+1,k-1)  &
  &                        +0.5d0*(bnd(i,j+1,k)*dz(k)-bnd(i,j,k-1)*dy(j))

                    else if(j==ny.and.k==1)then  ! -- 評価 2
                       tmp(i,j,k)=psi(i,j-1,k+1)  &
  &                        +0.5d0*(-bnd(i,j-1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                    else if(ib(i,j-1,k)==10.and.ib(i,j,k+1)==10)then
                       ! -- 評価 1 と同じ
                       tmp(i,j,k)=psi(i,j+1,k-1)  &
  &                        +0.5d0*(bnd(i,j+1,k)*dz(k)-bnd(i,j,k-1)*dy(j))

                    else if(ib(i,j+1,k)==10.and.ib(i,j,k-1)==10)then
                       ! -- 評価 2 と同じ
                       tmp(i,j,k)=psi(i,j-1,k+1)  &
  &                        +0.5d0*(-bnd(i,j-1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                    end if

                 !-- 以降, 隅領域なので, 個別に設定.
                 case (11)  ! 両方フラックス一定で (1,1,1) 点, もしくは内部領域.
                    tmp(i,j,k)=psi(i+1,j+1,k+1)  &
  &                           -(bnd(i,j+1,k+1)*dx(i)  &
  &                            +bnd(i+1,j,k+1)*dy(j)  &
  &                            +bnd(i+1,j+1,k)*dz(k))/3.0d0

                 case (13)  ! 両方フラックス一定で (nx,1,1) 点, もしくは内部領域.
                    tmp(i,j,k)=psi(i-1,j+1,k+1)  &
  &                           -(-bnd(i,j+1,k+1)*dx(i)  &
  &                             +bnd(i-1,j,k+1)*dy(j)  &
  &                             +bnd(i-1,j+1,k)*dz(k))/3.0d0

                 case (17)  ! 両方フラックス一定で (1,ny,1) 点, もしくは内部領域.
                    tmp(i,j,k)=psi(i+1,j-1,k+1)  &
  &                           -(bnd(i,j-1,k+1)*dx(i)  &
  &                            -bnd(i+1,j,k+1)*dy(j)  &
  &                            +bnd(i+1,j-1,k)*dz(k))/3.0d0

                 case (19)  ! 両方フラックス一定で (nx,ny,1) 点, もしくは内部領域.
                    tmp(i,j,k)=psi(i-1,j-1,k+1)  &
  &                           -(-bnd(i,j-1,k+1)*dx(i)  &
  &                             -bnd(i-1,j,k+1)*dy(j)  &
  &                             +bnd(i-1,j-1,k)*dz(k))/3.0d0

                 case (23)  ! 両方フラックス一定で (1,1,nz) 点, もしくは内部領域.
                    tmp(i,j,k)=psi(i+1,j+1,k-1)  &
  &                           -(bnd(i,j+1,k-1)*dx(i)  &
  &                            +bnd(i+1,j,k-1)*dy(j)  &
  &                            -bnd(i+1,j+1,k)*dz(k))/3.0d0

                 case (29)  ! 両方フラックス一定で (nx,1,nz) 点, もしくは内部領域.
                    tmp(i,j,k)=psi(i-1,j+1,k-1)  &
  &                           -(-bnd(i,j+1,k-1)*dx(i)  &
  &                             +bnd(i-1,j,k-1)*dy(j)  &
  &                             -bnd(i-1,j+1,k)*dz(k))/3.0d0

                 case (31)  ! 両方フラックス一定で (1,ny,nz) 点, もしくは内部領域.
                    tmp(i,j,k)=psi(i+1,j-1,k-1)  &
  &                           -(bnd(i,j-1,k-1)*dx(i)  &
  &                            -bnd(i+1,j,k-1)*dy(j)  &
  &                            -bnd(i+1,j-1,k)*dz(k))/3.0d0

                 case (37)  ! 両方フラックス一定で (nx,ny,nz) 点, もしくは内部領域.
                    tmp(i,j,k)=psi(i-1,j-1,k-1)  &
  &                           -(-bnd(i,j-1,k-1)*dx(i)  &
  &                             -bnd(i-1,j,k-1)*dy(j)  &
  &                             -bnd(i-1,j-1,k)*dz(k))/3.0d0

                 case (-11)  ! 両方フラックス一定で (i+1,j+1,k+1) に 10 が設定.
                    tmp(i,j,k)=(psi(i-1,j+1,k+1)+psi(i+1,j-1,k+1)+psi(i+1,j+1,k-1)  &
  &                            +bnd(i,j+1,k+1)*dx(i)+bnd(i+1,j,k+1)*dy(j)  &
  &                            +bnd(i+1,j+1,k)*dz(k))/3.0d0

                 case (-13)  ! 両方フラックス一定で (i-1,j+1,k+1) に 10 が設定.
                    tmp(i,j,k)=(psi(i+1,j+1,k+1)+psi(i-1,j-1,k+1)+psi(i-1,j+1,k-1)  &
  &                            -bnd(i,j+1,k+1)*dx(i)+bnd(i-1,j,k+1)*dy(j)  &
  &                            +bnd(i-1,j+1,k)*dz(k))/3.0d0

                 case (-17)  ! 両方フラックス一定で (i+1,j-1,k+1) に 10 が設定.
                    tmp(i,j,k)=(psi(i-1,j-1,k+1)+psi(i+1,j+1,k+1)+psi(i+1,j-1,k-1)  &
  &                            +bnd(i,j-1,k+1)*dx(i)-bnd(i+1,j,k+1)*dy(j)  &
  &                            +bnd(i+1,j-1,k)*dz(k))/3.0d0

                 case (-19)  ! 両方フラックス一定で (i-1,j-1,k+1) に 10 が設定.
                    tmp(i,j,k)=(psi(i+1,j-1,k+1)+psi(i-1,j+1,k+1)+psi(i-1,j-1,k-1)  &
  &                            -bnd(i,j-1,k+1)*dx(i)-bnd(i-1,j,k+1)*dy(j)  &
  &                            +bnd(i-1,j-1,k)*dz(k))/3.0d0

                 case (-23)  ! 両方フラックス一定で (i+1,j+1,k-1) に 10 が設定.
                    tmp(i,j,k)=(psi(i-1,j+1,k-1)+psi(i+1,j-1,k-1)+psi(i+1,j+1,k+1)  &
  &                            +bnd(i,j+1,k-1)*dx(i)+bnd(i+1,j,k-1)*dy(j)  &
  &                            -bnd(i+1,j+1,k)*dz(k))/3.0d0

                 case (-29)  ! 両方フラックス一定で (i-1,j+1,k-1) に 10 が設定.
                    tmp(i,j,k)=(psi(i+1,j+1,k-1)+psi(i-1,j-1,k-1)+psi(i-1,j+1,k+1)  &
  &                            -bnd(i,j+1,k-1)*dx(i)+bnd(i-1,j,k-1)*dy(j)  &
  &                            -bnd(i-1,j+1,k)*dz(k))/3.0d0

                 case (-31)  ! 両方フラックス一定で (i+1,j-1,k-1) に 10 が設定.
                    tmp(i,j,k)=(psi(i-1,j-1,k-1)+psi(i+1,j+1,k-1)+psi(i+1,j-1,k+1)  &
  &                            +bnd(i,j-1,k-1)*dx(i)-bnd(i+1,j,k-1)*dy(j)  &
  &                            -bnd(i+1,j-1,k)*dz(k))/3.0d0

                 case (-37)  ! 両方フラックス一定で (i-1,j-1,k-1) に 10 が設定.
                    tmp(i,j,k)=(psi(i+1,j-1,k-1)+psi(i-1,j+1,k-1)+psi(i-1,j-1,k+1)  &
  &                            -bnd(i,j-1,k-1)*dx(i)-bnd(i-1,j,k-1)*dy(j)  &
  &                            -bnd(i-1,j-1,k)*dz(k))/3.0d0

                 end select

              end if

              if(sor_flag.eqv..true.)then
                 tmp(i,j,k)=(1.0d0-accc)*psi(i,j,k)+tmp(i,j,k)*accc
              end if

           end do
        end do
     end do
!$omp end do
!$omp end parallel

!-- 最大誤差の更新
     do k=2,nz-1
        do j=2,ny-1
           do i=2,nx-1
              err=abs(tmp(i,j,k)-psi(i,j,k))
              if(err_max<=err)then
                 err_max=err
              end if
           end do
        end do
     end do

!-- 一斉更新

     do k=1,nz
        do j=1,ny
           do i=1,nx
              psi(i,j,k)=tmp(i,j,k)
           end do
        end do
     end do

     if(nl/=0)then
        counter=counter+1
        if(counter==nl)then
           exit
        else
           err_max=eps
        end if
     end if

  end do

!-- 境界の設定

  call calculate_bound_3dd( ib, dx, dy, dz, bnd, psi )

!-- 未定義領域には undef を代入する.

  do k=1,nz
     do j=1,ny
        do i=1,nx
           if(ib(i,j,k)==10)then
              psi(i,j,k)=defun
           end if
        end do
     end do
  end do

end subroutine Ellip_Jacobi_3dd

!----------------------------------
!----------------------------------

!-----------------------------------
! 以下, private subroutine
!-----------------------------------

subroutine set_bound( bound, ib, inner_flag, inner_bound )
! 各反復法ルーチンにおいて設定される境界条件のフラグをチェック, 設定する.
  implicit none
  character(4), intent(in) :: bound   ! 領域境界のフラグ
  integer, intent(inout) :: ib(:,:)   ! 全境界の判別整数
  logical, intent(inout) :: inner_flag(size(ib,1),size(ib,2))
                            ! 全領域境界フラグ
  integer, intent(in), optional :: inner_bound(size(ib,1),size(ib,2))
                            ! 内部領域境界の判別整数
  integer :: i, j, nx, ny

  nx=size(ib,1)
  ny=size(ib,2)

!-- 出力変数の初期化

  ib=0
  inner_flag=.true.

!-- 周期境界の設定確認.
!-- 周期境界なので, 両端とも 3 が設定されていないといけない.
  if(bound(1:1)=='3')then
     if(bound(3:3)/='3')then
        write(*,*) "### ERROR ###"
        write(*,*) "if bound = 3, bound(1:1)==bound(3:3). STOP."
        stop
     end if
  end if

  if(bound(3:3)=='3')then
     if(bound(1:1)/='3')then
        write(*,*) "### ERROR ###"
        write(*,*) "if bound = 3, bound(1:1)==bound(3:3). STOP."
        stop
     end if
  end if

  if(bound(2:2)=='3')then
     if(bound(4:4)/='3')then
        write(*,*) "### ERROR ###"
        write(*,*) "if bound = 3, bound(2:2)==bound(4:4). STOP."
        stop
     end if
  end if

  if(bound(4:4)=='3')then
     if(bound(2:2)/='3')then
        write(*,*) "### ERROR ###"
        write(*,*) "if bound = 3, bound(2:2)==bound(4:4). STOP."
        stop
     end if
  end if

  select case (bound(1:1))
  case ('1')
     do i=2,nx-1
        ib(i,1)=1
     end do

  case ('2')
     do i=2,nx-1
        ib(i,1)=4  ! y 方向のフラックスで参照値は上側
     end do

  case ('3')
     do i=2,nx-1
        ib(i,1)=3  ! y 方向 (x 軸) 周期境界
     end do
  end select

  select case (bound(2:2))
  case ('1')
     do j=2,ny-1
        ib(1,j)=1
     end do

  case ('2')
     do j=2,ny-1
        ib(1,j)=2  ! x 方向のフラックスで参照値は右側
     end do

  case ('3')
     do j=2,ny-1
        ib(1,j)=3  ! x 方向 (y 軸) 周期境界
     end do
  end select

  select case (bound(3:3))
  case ('1')
     do i=2,nx-1
        ib(i,ny)=1
     end do

  case ('2')
     do i=2,nx-1
        ib(i,ny)=-4  ! y 方向のフラックスで参照値は下側
     end do

  case ('3')
     do i=2,nx-1
        ib(i,ny)=3  ! y 方向 (x 軸) 周期境界
     end do
  end select

  select case (bound(4:4))
  case ('1')
     do j=2,ny-1
        ib(nx,j)=1
     end do

  case ('2')
     do j=2,ny-1
        ib(nx,j)=-2  ! x 方向のフラックスで参照値は左側
     end do

  case ('3')
     do j=2,ny-1
        ib(nx,j)=3  ! x 方向 (y 軸) 周期境界
     end do
  end select

!-- 領域隅境界は 2 辺が重なるので, 境界での強制方法は以下の順に優先する.
!-- (1) どちらかが 1 のとき, 隅領域は 1,
!-- (2) (1) 以外でどちらかが 3 のとき, 隅領域は 3.
!-- (3) (1), (2) 以外で 2. つまり, 両軸とも 2 なら 2 となる.

  if(bound(1:1)=='1')then
     ib(1,1)=1
     ib(nx,1)=1
  end if
  if(bound(2:2)=='1')then
     ib(1,1)=1
     ib(1,ny)=1
  end if
  if(bound(3:3)=='1')then
     ib(1,ny)=1
     ib(nx,ny)=1
  end if
  if(bound(4:4)=='1')then
     ib(nx,1)=1
     ib(nx,ny)=1
  end if

!-- 4 隅とも周期境界の場合
  if(bound(1:2)=='33')then
     ib(1,1)=3
     ib(1,ny)=3
     ib(nx,1)=3
     ib(nx,ny)=3
  end if

!-- どちらかが周期境界の場合, ib の符号が設定されていなければ -3 を指定.
  if(bound(1:1)=='3'.or.bound(2:2)=='3')then
     if(ib(1,1)==0)then
        ib(1,1)=3
     end if
     if(ib(nx,1)==0)then
        ib(nx,1)=3
     end if
     if(ib(1,ny)==0)then
        ib(1,ny)=3
     end if
     if(ib(nx,ny)==0)then
        ib(nx,ny)=3
     end if
  end if

!-- 領域の隅であることを 8, -8 で示す.
!-- 8 の場合, 左下か右上であることを, -8 の場合, 左上か右下であることを示す.

  if(ib(1,1)==0)then
     ib(1,1)=8
  end if
  if(ib(nx,1)==0)then
     ib(nx,1)=-8
  end if
  if(ib(1,ny)==0)then
     ib(1,ny)=-8
  end if
  if(ib(nx,ny)==0)then
     ib(nx,ny)=8
  end if

!-- 内部境界の設定

  if(present(inner_bound))then
     do j=2,ny-1
        do i=2,nx-1
           ib(i,j)=inner_bound(i,j)
           if(ib(i,j)==0)then
              inner_flag(i,j)=.false.
           end if
        end do
     end do
  else   ! 内部領域が設定されていない場合は, 内部全て計算する.
     do j=2,ny-1
        do i=2,nx-1
           inner_flag(i,j)=.false.
        end do
     end do
  end if

end subroutine set_bound



subroutine setval_boundf( ib, bnd, psi, bound_opt )
! 境界値を境界条件判別整数をもとに設定する.
  implicit none
  integer, intent(in) :: ib(:,:)  ! 境界条件判別整数
  real, intent(inout) :: bnd(size(ib,1),size(ib,2))  ! 境界での値
  real, intent(inout) :: psi(size(ib,1),size(ib,2))  ! 応答
  real, intent(in), optional :: bound_opt(size(ib,1),size(ib,2))  ! 境界での値
  integer :: i, j, nx, ny

  nx=size(ib,1)
  ny=size(ib,2)

!-- 境界値の設定
  if(present(bound_opt))then
     do j=1,ny
        do i=1,nx
           bnd(i,j)=bound_opt(i,j)
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           bnd(i,j)=0.0
        end do
     end do
  end if

!-- 境界条件の代入 "ib(i,j)==1 の場合のみ"
!-- 内部領域についてもここで代入してしまう.

  do j=1,ny
     do i=1,nx
        if(ib(i,j)==1)then
           psi(i,j)=bnd(i,j)
        end if
     end do
  end do

end subroutine setval_boundf


subroutine setval_boundd( ib, bnd, psi, bound_opt )
! 境界値を境界条件判別整数をもとに設定する.
  implicit none
  integer, intent(in) :: ib(:,:)  ! 境界条件判別整数
  double precision, intent(inout) :: bnd(size(ib,1),size(ib,2))  ! 境界での値
  double precision, intent(inout) :: psi(size(ib,1),size(ib,2))  ! 応答
  double precision, intent(in), optional :: bound_opt(size(ib,1),size(ib,2))  ! 境界での値
  integer :: i, j, nx, ny

  nx=size(ib,1)
  ny=size(ib,2)

!-- 境界値の設定
  if(present(bound_opt))then
     do j=1,ny
        do i=1,nx
           bnd(i,j)=bound_opt(i,j)
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           bnd(i,j)=0.0d0
        end do
     end do
  end if

!-- 境界条件の代入 "ib(i,j)==1 の場合のみ"
!-- 内部領域についてもここで代入してしまう.

  do j=1,ny
     do i=1,nx
        if(ib(i,j)==1)then
           psi(i,j)=bnd(i,j)
        end if
     end do
  end do

end subroutine setval_boundd


subroutine set_coef( coe, ext, def )
! 2 次元配列に ext で指定された値もしくは def で指定された一定値を代入する.
! ext, def どちらも optional であるが, 必ずどちらかは指定されていないといけない.
  implicit none
  real, intent(inout) :: coe(:,:)  ! 代入される配列
  real, intent(in), optional :: ext(size(coe,1),size(coe,2))  ! 代入する配列
  real, intent(in), optional :: def  ! 代入する一定値
  integer :: i, j, nx, ny

  nx=size(coe,1)
  ny=size(coe,2)

  if(present(ext))then
     do j=1,ny
        do i=1,nx
           coe(i,j)=ext(i,j)
        end do
     end do
  else if(present(def))then
     do j=1,ny
        do i=1,nx
           coe(i,j)=def
        end do
     end do
  else
     write(*,*) "### ERROR ###"
     write(*,*) "subroutine set_coe must be set optional argument 'ext' or 'def'"
     write(*,*) "STOP."
     stop
  end if

end subroutine set_coef

subroutine set_coed( coe, ext, def )
! 2 次元配列に ext で指定された値もしくは def で指定された一定値を代入する.
! ext, def どちらも optional であるが, 必ずどちらかは指定されていないといけない.
  implicit none
  double precision, intent(inout) :: coe(:,:)  ! 代入される配列
  double precision, intent(in), optional :: ext(size(coe,1),size(coe,2))  ! 代入する配列
  double precision, intent(in), optional :: def  ! 代入する一定値
  integer :: i, j, nx, ny

  nx=size(coe,1)
  ny=size(coe,2)

  if(present(ext))then
     do j=1,ny
        do i=1,nx
           coe(i,j)=ext(i,j)
        end do
     end do
  else if(present(def))then
     do j=1,ny
        do i=1,nx
           coe(i,j)=def
        end do
     end do
  else
     write(*,*) "### ERROR ###"
     write(*,*) "subroutine set_coe must be set optional argument 'ext' or 'def'"
     write(*,*) "STOP."
     stop
  end if

end subroutine set_coed


subroutine check_coef( coe, aval, undeff )
! 2 次元配列に aval が代入されていないかをチェックする.
  implicit none
  real, intent(inout) :: coe(:,:)  ! 代入される配列
  real, intent(in) :: aval         ! チェックされる値
  real, intent(in), optional :: undeff
  integer :: i, j, nx, ny

  nx=size(coe,1)
  ny=size(coe,2)

  if(present(undeff))then
     do j=1,ny
        do i=1,nx
           if(coe(i,j)/=undeff)then
              if(coe(i,j)==aval)then
                 write(*,*) "### ERROR (Ellip_Slv module) ###"
                 write(*,*) "Detect a certain value", aval, i, j
                 write(*,*) "STOP."
                 stop
              end if
           end if
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           if(coe(i,j)==aval)then
              write(*,*) "### ERROR (Ellip_Slv module) ###"
              write(*,*) "Detect a certain value", aval, i, j
              write(*,*) "STOP."
              stop
           end if
        end do
     end do
  end if

end subroutine check_coef


subroutine check_coed( coe, aval, undeff )
! 2 次元配列に aval が代入されていないかをチェックする.
  implicit none
  double precision, intent(inout) :: coe(:,:)  ! 代入される配列
  double precision, intent(in) :: aval         ! チェックされる値
  double precision, intent(in), optional :: undeff
  integer :: i, j, nx, ny

  nx=size(coe,1)
  ny=size(coe,2)

  if(present(undeff))then
     do j=1,ny
        do i=1,nx
           if(coe(i,j)/=undeff)then
              if(coe(i,j)==aval)then
                 write(*,*) "### ERROR (Ellip_Slv module) ###"
                 write(*,*) "Detect a certain value", aval, i, j
                 write(*,*) "STOP."
                 stop
              end if
           end if
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           if(coe(i,j)==aval)then
              write(*,*) "### ERROR (Ellip_Slv module) ###"
              write(*,*) "Detect a certain value", aval, i, j
              write(*,*) "STOP."
              stop
           end if
        end do
     end do
  end if

end subroutine check_coed


subroutine check_le_coef( coe, aval, undeff )
! 2 次元配列に less equal aval をチェックする.
  implicit none
  real, intent(inout) :: coe(:,:)  ! 代入される配列
  real, intent(in) :: aval         ! チェックされる値
  real, intent(in), optional :: undeff
  integer :: i, j, nx, ny

  nx=size(coe,1)
  ny=size(coe,2)

  if(present(undeff))then
     do j=1,ny
        do i=1,nx
           if(coe(i,j)/=undeff)then
              if(coe(i,j)<=aval)then
                 write(*,*) "### ERROR (Ellip_Slv module) ###"
                 write(*,*) "Detect a certain value", coe(i,j), i, j
                 write(*,*) "STOP."
                 stop
              end if
           end if
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           if(coe(i,j)<=aval)then
              write(*,*) "### ERROR (Ellip_Slv module) ###"
              write(*,*) "Detect a certain value", coe(i,j), i, j
              write(*,*) "STOP."
              stop
           end if
        end do
     end do
  end if

end subroutine check_le_coef


subroutine check_le_coed( coe, aval, undeff )
! 2 次元配列に less equal aval をチェックする.
  implicit none
  double precision, intent(inout) :: coe(:,:)  ! 代入される配列
  double precision, intent(in) :: aval         ! チェックされる値
  double precision, intent(in), optional :: undeff
  integer :: i, j, nx, ny

  nx=size(coe,1)
  ny=size(coe,2)

  if(present(undeff))then
     do j=1,ny
        do i=1,nx
           if(coe(i,j)/=undeff)then
              if(coe(i,j)<=aval)then
                 write(*,*) "### ERROR (Ellip_Slv module) ###"
                 write(*,*) "Detect a certain value", coe(i,j), i, j
                 write(*,*) "STOP."
                 stop
              end if
           end if
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           if(coe(i,j)<=aval)then
              write(*,*) "### ERROR (Ellip_Slv module) ###"
              write(*,*) "Detect a certain value", coe(i,j), i, j
              write(*,*) "STOP."
              stop
           end if
        end do
     end do
  end if

end subroutine check_le_coed


subroutine calculate_boundf( ib, dx, dy, bnd, psi )
! ib 判別を元に, ノイマン型, 周期境界型について境界値を計算する.
  integer, intent(in) :: ib(:,:)  ! 境界整数判別
  real, intent(in) :: dx(size(ib,1))  ! x 方向の格子解像度
  real, intent(in) :: dy(size(ib,2))  ! y 方向の格子解像度
  real, intent(in) :: bnd(size(ib,1),size(ib,2))  ! 境界値(ノイマン型のみ使用)
  real, intent(inout) :: psi(size(ib,1),size(ib,2))  ! 応答関数
  integer :: i, j, nx, ny, ix, jy

  nx=size(ib,1)
  ny=size(ib,2)

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,ix,jy)
  do j=1,ny
     do i=1,nx
        if(ib(i,j)/=0)then   ! 計算領域点が select 文に入るのを防ぐ.
           select case (ib(i,j))
           case (1)
              psi(i,j)=bnd(i,j)

           case (2)  ! x 方向にフラックス一定, 上側が参照値
              psi(i,j)=psi(i+1,j)-bnd(i,j)*dx(i)

           case (-2)  ! x 方向にフラックス一定, 下側が参照値
              psi(i,j)=psi(i-1,j)+bnd(i,j)*dx(i)

           case (4)  ! y 方向にフラックス一定, 右側が参照値
              psi(i,j)=psi(i,j+1)-bnd(i,j)*dy(j)

           case (-4)  ! y 方向にフラックス一定, 左側が参照値
              psi(i,j)=psi(i,j-1)+bnd(i,j)*dy(j)

           case (3)  ! 周期境界
              if(i==1)then
                 ix=nx-1
              else if(i==nx)then
                 ix=2
              else
                 ix=i
              end if
              if(j==1)then
                 jy=ny-1
              else if(j==ny)then
                 jy=2
              else
                 jy=j
              end if

              psi(i,j)=psi(ix,jy)

           case (7)  ! 両方フラックス一定で内部境界限定.
              if((ib(i+1,j+1)==10).and.(ib(i+1,j)/=10).and.  &
  &              (ib(i,j+1)/=10))then
                 psi(i,j)=0.5*(psi(i-1,j)+psi(i,j-1))  &
  &                       +0.5*bnd(i,j)*(dy(j)+dx(i))

              else if((ib(i-1,j-1)==10).and.(ib(i-1,j)/=10).and.  &
  &                   (ib(i,j-1)/=10))then
                 psi(i,j)=0.5*(psi(i+1,j)+psi(i,j+1))  &
  &                       -0.5*bnd(i,j)*(dy(j)+dx(i))

              end if

           case (-7)  ! 両方フラックス一定で内部境界限定.
              if((ib(i-1,j+1)==10).and.(ib(i-1,j)/=10).and.  &
  &              (ib(i,j+1)/=10))then
                 psi(i,j)=0.5*(psi(i+1,j)+psi(i,j-1))  &
  &                       +0.5*bnd(i,j)*(dy(j)-dx(i))

              else if((ib(i+1,j-1)==10).and.(ib(i+1,j)/=10).and.  &
  &              (ib(i,j-1)/=10))then
                 psi(i,j)=0.5*(psi(i-1,j)+psi(i,j+1))  &
  &                       +0.5*bnd(i,j)*(-dy(j)+dx(i))

              end if

           case (8)  ! 両方フラックス一定で左下角か右上角, もしくは内部境界.
              if(i==1.and.j==1)then  ! -- 評価 1
                 psi(i,j)=psi(i+1,j+1)-0.5*(bnd(i+1,j)*dy(j)+bnd(i,j+1)*dx(i))

              else if(i==nx.and.j==ny)then  ! -- 評価 2
                 psi(i,j)=psi(i-1,j-1)+0.5*(bnd(i-1,j)*dy(j)+bnd(i,j-1)*dx(i))

              else if(ib(i-1,j)==10.and.ib(i,j-1)==10)then
                 ! -- 評価 1 と同じ
                 psi(i,j)=psi(i+1,j+1)-0.5*(bnd(i+1,j)*dy(j)+bnd(i,j+1)*dx(i))

              else if(ib(i+1,j)==10.and.ib(i,j+1)==10)then
                 ! -- 評価 2 と同じ
                 psi(i,j)=psi(i-1,j-1)+0.5*(bnd(i-1,j)*dy(j)+bnd(i,j-1)*dx(i))

              end if

           case (-8)  ! 両方フラックス一定で右下角か左上角
              if(i==1.and.j==ny)then  ! -- 評価 1
                 psi(i,j)=psi(i+1,j-1)+0.5*(bnd(i+1,j)*dy(j)-bnd(i,j-1)*dx(i))

              else if(i==nx.and.j==1)then  ! -- 評価 2
                 psi(i,j)=psi(i-1,j+1)+0.5*(-bnd(i-1,j)*dy(j)+bnd(i,j+1)*dx(i))

              else if(ib(i-1,j)==10.and.ib(i,j+1)==10)then
                 ! -- 評価 1 と同じ
                 psi(i,j)=psi(i+1,j-1)+0.5*(bnd(i+1,j)*dy(j)-bnd(i,j-1)*dx(i))

              else if(ib(i+1,j)==10.and.ib(i,j-1)==10)then
                 ! -- 評価 2 と同じ
                 psi(i,j)=psi(i-1,j+1)+0.5*(-bnd(i-1,j)*dy(j)+bnd(i,j+1)*dx(i))

              end if
           end select

        end if
     end do
  end do
!$omp end do
!$omp end parallel

end subroutine calculate_boundf


subroutine calculate_boundd( ib, dx, dy, bnd, psi )
! ib 判別を元に, ノイマン型, 周期境界型について境界値を計算する.
  integer, intent(in) :: ib(:,:)  ! 境界整数判別
  double precision, intent(in) :: dx(size(ib,1))  ! x 方向の格子解像度
  double precision, intent(in) :: dy(size(ib,2))  ! y 方向の格子解像度
  double precision, intent(in) :: bnd(size(ib,1),size(ib,2))  ! 境界値(ノイマン型のみ使用)
  double precision, intent(inout) :: psi(size(ib,1),size(ib,2))  ! 応答関数
  integer :: i, j, nx, ny, ix, jy

  nx=size(ib,1)
  ny=size(ib,2)

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,ix,jy)
  do j=1,ny
     do i=1,nx
        if(ib(i,j)/=0)then   ! 計算領域点が select 文に入るのを防ぐ.
           select case (ib(i,j))
           case (1)
              psi(i,j)=bnd(i,j)

           case (2)  ! x 方向にフラックス一定, 上側が参照値
              psi(i,j)=psi(i+1,j)-bnd(i,j)*dx(i)

           case (-2)  ! x 方向にフラックス一定, 下側が参照値
              psi(i,j)=psi(i-1,j)+bnd(i,j)*dx(i)

           case (4)  ! y 方向にフラックス一定, 右側が参照値
              psi(i,j)=psi(i,j+1)-bnd(i,j)*dy(j)

           case (-4)  ! y 方向にフラックス一定, 左側が参照値
              psi(i,j)=psi(i,j-1)+bnd(i,j)*dy(j)

           case (3)  ! 周期境界
              if(i==1)then
                 ix=nx-1
              else if(i==nx)then
                 ix=2
              else
                 ix=i
              end if
              if(j==1)then
                 jy=ny-1
              else if(j==ny)then
                 jy=2
              else
                 jy=j
              end if

              psi(i,j)=psi(ix,jy)

           case (7)  ! 両方フラックス一定で内部境界限定.
              if((ib(i+1,j+1)==10).and.(ib(i+1,j)/=10).and.  &
  &              (ib(i,j+1)/=10))then
                 psi(i,j)=0.5d0*(psi(i-1,j)+psi(i,j-1))  &
  &                       +0.5d0*bnd(i,j)*(dy(j)+dx(i))

              else if((ib(i-1,j-1)==10).and.(ib(i-1,j)/=10).and.  &
  &                   (ib(i,j-1)/=10))then
                 psi(i,j)=0.5d0*(psi(i+1,j)+psi(i,j+1))  &
  &                       -0.5d0*bnd(i,j)*(dy(j)+dx(i))

              end if

           case (-7)  ! 両方フラックス一定で内部境界限定.
              if((ib(i-1,j+1)==10).and.(ib(i-1,j)/=10).and.  &
  &              (ib(i,j+1)/=10))then
                 psi(i,j)=0.5d0*(psi(i+1,j)+psi(i,j-1))  &
  &                       +0.5d0*bnd(i,j)*(dy(j)-dx(i))

              else if((ib(i+1,j-1)==10).and.(ib(i+1,j)/=10).and.  &
  &              (ib(i,j-1)/=10))then
                 psi(i,j)=0.5d0*(psi(i-1,j)+psi(i,j+1))  &
  &                       +0.5d0*bnd(i,j)*(-dy(j)+dx(i))

              end if

           case (8)  ! 両方フラックス一定で左下角か右上角, もしくは内部境界.
              if(i==1.and.j==1)then  ! -- 評価 1
                 psi(i,j)=psi(i+1,j+1)-0.5d0*(bnd(i+1,j)*dy(j)+bnd(i,j+1)*dx(i))

              else if(i==nx.and.j==ny)then  ! -- 評価 2
                 psi(i,j)=psi(i-1,j-1)+0.5d0*(bnd(i-1,j)*dy(j)+bnd(i,j-1)*dx(i))

              else if(ib(i-1,j)==10.and.ib(i,j-1)==10)then
                 ! -- 評価 1 と同じ
                 psi(i,j)=psi(i+1,j+1)-0.5d0*(bnd(i+1,j)*dy(j)+bnd(i,j+1)*dx(i))

              else if(ib(i+1,j)==10.and.ib(i,j+1)==10)then
                 ! -- 評価 2 と同じ
                 psi(i,j)=psi(i-1,j-1)+0.5d0*(bnd(i-1,j)*dy(j)+bnd(i,j-1)*dx(i))

              end if

           case (-8)  ! 両方フラックス一定で右下角か左上角
              if(i==1.and.j==ny)then  ! -- 評価 1
                 psi(i,j)=psi(i+1,j-1)+0.5d0*(bnd(i+1,j)*dy(j)-bnd(i,j-1)*dx(i))

              else if(i==nx.and.j==1)then  ! -- 評価 2
                 psi(i,j)=psi(i-1,j+1)+0.5d0*(-bnd(i-1,j)*dy(j)+bnd(i,j+1)*dx(i))

              else if(ib(i-1,j)==10.and.ib(i,j+1)==10)then
                 ! -- 評価 1 と同じ
                 psi(i,j)=psi(i+1,j-1)+0.5d0*(bnd(i+1,j)*dy(j)-bnd(i,j-1)*dx(i))

              else if(ib(i+1,j)==10.and.ib(i,j-1)==10)then
                 ! -- 評価 2 と同じ
                 psi(i,j)=psi(i-1,j+1)+0.5d0*(-bnd(i-1,j)*dy(j)+bnd(i,j+1)*dx(i))

              end if
           end select

        end if
     end do
  end do
!$omp end do
!$omp end parallel

end subroutine calculate_boundd


subroutine set_bound_3d( bound, ib, inner_flag, inner_bound )
! 各反復法ルーチンにおいて設定される境界条件のフラグをチェック, 設定する.
  implicit none
  character(6), intent(in) :: bound     ! 領域境界のフラグ
  integer, intent(inout) :: ib(:,:,:)   ! 全境界の判別整数
  logical, intent(inout) :: inner_flag(size(ib,1),size(ib,2),size(ib,3))
                            ! 全領域境界フラグ
  integer, intent(in), optional :: inner_bound(size(ib,1),size(ib,2),size(ib,3))
                            ! 内部領域境界の判別整数
  integer :: i, j, k, nx, ny, nz

  nx=size(ib,1)
  ny=size(ib,2)
  nz=size(ib,3)

!-- 出力変数の初期化

  ib=0
  inner_flag=.true.

!-- 周期境界の設定確認.
!-- 周期境界なので, 両端とも 3 が設定されていないといけない.
  if(bound(1:1)=='3')then
     if(bound(3:3)/='3')then
        write(*,*) "### ERROR ###"
        write(*,*) "if bound = 3, bound(1:1)==bound(3:3). STOP."
        stop
     end if
  end if

  if(bound(3:3)=='3')then
     if(bound(1:1)/='3')then
        write(*,*) "### ERROR ###"
        write(*,*) "if bound = 3, bound(1:1)==bound(3:3). STOP."
        stop
     end if
  end if

  if(bound(2:2)=='3')then
     if(bound(4:4)/='3')then
        write(*,*) "### ERROR ###"
        write(*,*) "if bound = 3, bound(2:2)==bound(4:4). STOP."
        stop
     end if
  end if

  if(bound(4:4)=='3')then
     if(bound(2:2)/='3')then
        write(*,*) "### ERROR ###"
        write(*,*) "if bound = 3, bound(2:2)==bound(4:4). STOP."
        stop
     end if
  end if

  if(bound(5:5)=='3')then
     if(bound(6:6)/='3')then
        write(*,*) "### ERROR ###"
        write(*,*) "if bound = 3, bound(5:5)==bound(6:6). STOP."
        stop
     end if
  end if

  if(bound(6:6)=='3')then
     if(bound(5:5)/='3')then
        write(*,*) "### ERROR ###"
        write(*,*) "if bound = 3, bound(5:5)==bound(6:6). STOP."
        stop
     end if
  end if

  select case (bound(1:1))
  case ('1')
     do k=2,nz-1
        do i=2,nx-1
           ib(i,1,k)=1
        end do
     end do

  case ('2')
     do k=2,nz-1
        do i=2,nx-1
           ib(i,1,k)=4  ! y 方向のフラックスで参照値は上側
        end do
     end do

  case ('3')
     do k=2,nz-1
        do i=2,nx-1
           ib(i,1,k)=3  ! y 方向 (x 軸) 周期境界
        end do
     end do
  end select

  select case (bound(2:2))
  case ('1')
     do k=2,nz-1
        do j=2,ny-1
           ib(1,j,k)=1
        end do
     end do

  case ('2')
     do k=2,nz-1
        do j=2,ny-1
           ib(1,j,k)=2  ! x 方向のフラックスで参照値は右側
        end do
     end do

  case ('3')
     do k=2,nz-1
        do j=2,ny-1
           ib(1,j,k)=3  ! x 方向 (y 軸) 周期境界
        end do
     end do
  end select

  select case (bound(3:3))
  case ('1')
     do k=2,nz-1
        do i=2,nx-1
           ib(i,ny,k)=1
        end do
     end do

  case ('2')
     do k=2,nz-1
        do i=2,nx-1
           ib(i,ny,k)=-4  ! y 方向のフラックスで参照値は下側
        end do
     end do

  case ('3')
     do k=2,nz-1
        do i=2,nx-1
           ib(i,ny,k)=3  ! y 方向 (x 軸) 周期境界
        end do
     end do
  end select

  select case (bound(4:4))
  case ('1')
     do k=2,nz-1
        do j=2,ny-1
           ib(nx,j,k)=1
        end do
     end do

  case ('2')
     do k=2,nz-1
        do j=2,ny-1
           ib(nx,j,k)=-2  ! x 方向のフラックスで参照値は左側
        end do
     end do

  case ('3')
     do k=2,nz-1
        do j=2,ny-1
           ib(nx,j,k)=3  ! x 方向 (y 軸) 周期境界
        end do
     end do
  end select

  select case (bound(5:5))
  case ('1')
     do j=2,ny-1
        do i=2,nx-1
           ib(i,j,1)=1
        end do
     end do

  case ('2')
     do j=2,ny-1
        do i=2,nx-1
           ib(i,j,1)=6  ! z 方向のフラックスで参照値は下面
        end do
     end do

  case ('3')
     do j=2,ny-1
        do i=2,nx-1
           ib(i,j,1)=3
        end do
     end do
  end select

  select case (bound(6:6))
  case ('1')
     do j=2,ny-1
        do i=2,nx-1
           ib(i,j,nz)=1
        end do
     end do

  case ('2')
     do j=2,ny-1
        do i=2,nx-1
           ib(i,j,nz)=-6  ! z 方向のフラックスで参照値は上面
        end do
     end do

  case ('3')
     do j=2,ny-1
        do i=2,nx-1
           ib(i,j,nz)=3
        end do
     end do
  end select

!-- 領域隅境界は 2 辺が重なるので, 境界での強制方法は以下の順に優先する.
!-- (1) どちらかが 1 のとき, 隅領域は 1,
!-- (2) (1) 以外でどちらかが 3 のとき, 隅領域は 3.
!-- (3) (1), (2) 以外で 2. つまり, 両軸とも 2 なら 2 となる.
!-- 境界辺は 12 本あるので, 12 箇所設定.
!-- 各境界辺は以下のように x, y, z で指定.
!-- (1,1,0) -> z 軸に伸びる nx=1, ny=1 面を通る辺.

!-- [注意] : 1, 3 の場合は境界辺だけでなく, 境界点も同時に設定できるので
!--          ここで一緒に設定してしまう.

  if(bound(1:1)=='1')then  ! (1,1,0), (nx,1,0), (0,1,1), (0,1,nz) 辺
     ib(1,1,1:nz)=1
     ib(nx,1,1:nz)=1
     ib(1:nx,1,1)=1
     ib(1:nx,1,nz)=1
  end if
  if(bound(2:2)=='1')then  ! (1,1,0), (1,ny,0), (1,0,1), (1,0,nz) 辺
     ib(1,1,1:nz)=1
     ib(1,ny,1:nz)=1
     ib(1,1:ny,1)=1
     ib(1,1:ny,nz)=1
  end if
  if(bound(3:3)=='1')then  ! (1,ny,0), (nx,ny,0), (0,ny,1), (0,ny,nz) 辺
     ib(1,ny,1:nz)=1
     ib(nx,ny,1:nz)=1
     ib(1:nx,ny,1)=1
     ib(1:nx,ny,nz)=1
  end if
  if(bound(4:4)=='1')then  ! (nx,1,0), (nx,ny,0), (nx,0,1), (nx,0,nz) 辺
     ib(nx,1,1:nz)=1
     ib(nx,ny,1:nz)=1
     ib(nx,1:ny,1)=1
     ib(nx,1:ny,nz)=1
  end if
  if(bound(5:5)=='1')then  ! (0,1,1), (0,ny,1), (1,0,1), (nx,0,1) 辺
     ib(1:nx,1,1)=1
     ib(1:nx,ny,1)=1
     ib(1,1:ny,1)=1
     ib(nx,1:ny,1)=1
  end if
  if(bound(6:6)=='1')then  ! (0,1,nz), (0,ny,nz), (1,0,nz), (nx,0,nz) 辺
     ib(1:nx,1,nz)=1
     ib(1:nx,ny,nz)=1
     ib(1,1:ny,nz)=1
     ib(nx,1:ny,nz)=1
  end if

!-- 12 本とも周期境界の場合
  if(bound(1:1)=='3'.and.bound(2:2)=='3'.and.bound(5:5)=='3')then
     ib(1,1,1:nz)=3
     ib(1,ny,1:nz)=3
     ib(nx,1,1:nz)=3
     ib(nx,ny,1:nz)=3
     ib(1:nx,1,1)=3
     ib(1:nx,1,nz)=3
     ib(1:nx,ny,1)=3
     ib(1:nx,ny,nz)=3
     ib(1,1:ny,1)=3
     ib(nx,1:ny,1)=3
     ib(1,1:ny,nz)=3
     ib(nx,1:ny,nz)=3
  end if

!-- 12 辺のうち, いずれかで周期境界の場合, ib の値が設定されていなければ
!-- -3 を設定する. -3 は固定境界 1 より優先度が小さいので, ここで設定.
  if(bound(1:1)=='3'.or.bound(2:2)=='3'.or.bound(5:5)=='3')then
     do k=1,nz
        do i=1,nx
           if(ib(i,1,k)==0)then
              ib(i,1,k)=3
           end if
           if(ib(i,ny,k)==0)then
              ib(i,ny,k)=3
           end if
        end do
     end do
     do k=1,nz
        do j=1,ny
           if(ib(1,j,k)==0)then
              ib(1,j,k)=3
           end if
           if(ib(nx,j,k)==0)then
              ib(nx,j,k)=3
           end if
        end do
     end do
     do j=1,ny
        do i=1,nx
           if(ib(i,j,1)==0)then
              ib(i,j,1)=3
           end if
           if(ib(i,j,nz)==0)then
              ib(i,j,nz)=3
           end if
        end do
     end do
  end if

!-- 自由境界条件の設定
!-- [注意] : ここでは, 境界辺と境界点で設定値が異なるので,
!--          別々に設定する.
!-- 境界辺の設定には, ib = 0 であることを確認した後,
!-- 隣接 2 面に設定されている値の積となるように設定する.
!-- 符号も含めて設定する.
!-- 各面は符号を省いて 2, 4, 6 で設定されているので
!-- それぞれの面が隣接する 12 辺は
!-- 8, 12, 24 という値が符号つきで設定される.

  do k=2,nz-1
     if(ib(1,1,k)==0)then
        ib(1,1,k)=ib(2,1,k)*ib(1,2,k)
     end if
     if(ib(1,ny,k)==0)then
        ib(1,ny,k)=ib(2,ny,k)*ib(1,ny-1,k)
     end if
     if(ib(nx,1,k)==0)then
        ib(nx,1,k)=ib(nx-1,1,k)*ib(nx,2,k)
     end if
     if(ib(nx,ny,k)==0)then
        ib(nx,ny,k)=ib(nx-1,ny,k)*ib(nx,ny-1,k)
     end if
  end do

  do j=2,ny-1
     if(ib(1,j,1)==0)then
        ib(1,j,1)=ib(2,j,1)*ib(1,j,2)
     end if
     if(ib(nx,j,1)==0)then
        ib(nx,j,1)=ib(nx-1,j,1)*ib(nx,j,2)
     end if
     if(ib(1,j,nz)==0)then
        ib(1,j,nz)=ib(2,j,nz)*ib(1,j,nz-1)
     end if
     if(ib(nx,j,nz)==0)then
        ib(nx,j,nz)=ib(nx-1,j,nz)*ib(nx,j,nz-1)
     end if
  end do

  do i=2,nx-1
     if(ib(i,1,1)==0)then
        ib(i,1,1)=ib(i,1,2)*ib(i,2,1)
     end if
     if(ib(i,ny,1)==0)then
        ib(i,ny,1)=ib(i,ny,2)*ib(i,ny-1,1)
     end if
     if(ib(i,1,nz)==0)then
        ib(i,1,nz)=ib(i,1,nz-1)*ib(i,2,nz)
     end if
     if(ib(i,ny,nz)==0)then
        ib(i,ny,nz)=ib(i,ny,nz-1)*ib(i,ny-1,nz)
     end if
  end do

!-- 8 境界点の設定.
!-- 8 点しかないので, 以下の素数を各点に対応させる.
!-- (1,1,1) = 11
!-- (nx,1,1) = 13
!-- (1,ny,1) = 17
!-- (nx,ny,1) = 19
!-- (1,1,nz) = 23
!-- (nx,1,nz) = 29
!-- (1,ny,nz) = 31
!-- (nx,ny,nz) = 33

  if(ib(1,1,1)==0)then
     ib(1,1,1)=11
  end if
  if(ib(nx,1,1)==0)then
     ib(nx,1,1)=13
  end if
  if(ib(1,ny,1)==0)then
     ib(1,ny,1)=17
  end if
  if(ib(nx,ny,1)==0)then
     ib(nx,ny,1)=19
  end if
  if(ib(1,1,nz)==0)then
     ib(1,1,nz)=23
  end if
  if(ib(nx,1,nz)==0)then
     ib(nx,1,nz)=29
  end if
  if(ib(1,ny,nz)==0)then
     ib(1,ny,nz)=31
  end if
  if(ib(nx,ny,nz)==0)then
     ib(nx,ny,nz)=37
  end if

!-- 内部境界の設定

  if(present(inner_bound))then
     do k=2,nz-1
        do j=2,ny-1
           do i=2,nx-1
              ib(i,j,k)=inner_bound(i,j,k)
              if(ib(i,j,k)==0)then
                 inner_flag(i,j,k)=.false.
              end if
           end do
        end do
     end do
  else   ! 内部領域が設定されていない場合は内部全て計算する.
     do k=2,nz-1
        do j=2,ny-1
           do i=2,nx-1
              inner_flag(i,j,k)=.false.
           end do
        end do
     end do
  end if

end subroutine set_bound_3d


subroutine setval_bound_3df( ib, bnd, psi, bound_opt )
! 境界値を境界条件判別整数をもとに設定する.
  implicit none
  integer, intent(in) :: ib(:,:,:)  ! 境界条件判別整数
  real, intent(inout) :: bnd(size(ib,1),size(ib,2),size(ib,3))  ! 境界での値
  real, intent(inout) :: psi(size(ib,1),size(ib,2),size(ib,3))  ! 応答
  real, intent(in), optional :: bound_opt(size(ib,1),size(ib,2),size(ib,3))  ! 境界での値
  integer :: i, j, k, nx, ny, nz

  nx=size(ib,1)
  ny=size(ib,2)
  nz=size(ib,3)

!-- 境界値の設定
  if(present(bound_opt))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              bnd(i,j,k)=bound_opt(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              bnd(i,j,k)=0.0
           end do
        end do
     end do
  end if

!-- 境界条件の代入 "ib(i,j)==1 の場合のみ"
!-- 内部領域についてもここで代入してしまう.

  do k=1,nz
     do j=1,ny
        do i=1,nx
           if(ib(i,j,k)==1)then
              psi(i,j,k)=bnd(i,j,k)
           end if
        end do
     end do
  end do

end subroutine setval_bound_3df


subroutine setval_bound_3dd( ib, bnd, psi, bound_opt )
! 境界値を境界条件判別整数をもとに設定する.
  implicit none
  integer, intent(in) :: ib(:,:,:)  ! 境界条件判別整数
  double precision, intent(inout) :: bnd(size(ib,1),size(ib,2),size(ib,3))  ! 境界での値
  double precision, intent(inout) :: psi(size(ib,1),size(ib,2),size(ib,3))  ! 応答
  double precision, intent(in), optional :: bound_opt(size(ib,1),size(ib,2),size(ib,3))  ! 境界での値
  integer :: i, j, k, nx, ny, nz

  nx=size(ib,1)
  ny=size(ib,2)
  nz=size(ib,3)

!-- 境界値の設定
  if(present(bound_opt))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              bnd(i,j,k)=bound_opt(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              bnd(i,j,k)=0.0d0
           end do
        end do
     end do
  end if

!-- 境界条件の代入 "ib(i,j)==1 の場合のみ"
!-- 内部領域についてもここで代入してしまう.

  do k=1,nz
     do j=1,ny
        do i=1,nx
           if(ib(i,j,k)==1)then
              psi(i,j,k)=bnd(i,j,k)
           end if
        end do
     end do
  end do

end subroutine setval_bound_3dd


subroutine set_coe_3df( coe, ext, def )
! 2 次元配列に ext で指定された値もしくは def で指定された一定値を代入する.
! ext, def どちらも optional であるが, 必ずどちらかは指定されていないといけない.
  implicit none
  real, intent(inout) :: coe(:,:,:)  ! 代入される配列
  real, intent(in), optional :: ext(size(coe,1),size(coe,2),size(coe,3))  ! 代入する配列
  real, intent(in), optional :: def  ! 代入する一定値
  integer :: i, j, k, nx, ny, nz

  nx=size(coe,1)
  ny=size(coe,2)
  nz=size(coe,3)

  if(present(ext))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              coe(i,j,k)=ext(i,j,k)
           end do
        end do
     end do
  else if(present(def))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              coe(i,j,k)=def
           end do
        end do
     end do
  else
     write(*,*) "### ERROR ###"
     write(*,*) "subroutine set_coe must be set optional argument 'ext' or 'def'"
     write(*,*) "STOP."
     stop
  end if

end subroutine set_coe_3df


subroutine set_coe_3dd( coe, ext, def )
! 2 次元配列に ext で指定された値もしくは def で指定された一定値を代入する.
! ext, def どちらも optional であるが, 必ずどちらかは指定されていないといけない.
  implicit none
  double precision, intent(inout) :: coe(:,:,:)  ! 代入される配列
  double precision, intent(in), optional :: ext(size(coe,1),size(coe,2),size(coe,3))  ! 代入する配列
  double precision, intent(in), optional :: def  ! 代入する一定値
  integer :: i, j, k, nx, ny, nz

  nx=size(coe,1)
  ny=size(coe,2)
  nz=size(coe,3)

  if(present(ext))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              coe(i,j,k)=ext(i,j,k)
           end do
        end do
     end do
  else if(present(def))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              coe(i,j,k)=def
           end do
        end do
     end do
  else
     write(*,*) "### ERROR ###"
     write(*,*) "subroutine set_coe must be set optional argument 'ext' or 'def'"
     write(*,*) "STOP."
     stop
  end if

end subroutine set_coe_3dd


subroutine check_coe_3df( coe, aval, undeff )
! 3 次元配列に aval で指定された値が入っていないかを検出する.
  implicit none
  real, intent(inout) :: coe(:,:,:)  ! 代入される配列
  real, intent(in) :: aval           ! 検出される値
  real, intent(in), optional :: undeff
  integer :: i, j, k, nx, ny, nz

  nx=size(coe,1)
  ny=size(coe,2)
  nz=size(coe,3)

  if(present(undeff))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(coe(i,j,k)/=undeff)then
                 if(coe(i,j,k)==aval)then
                    write(*,*) "### ERROR (Ellip_Slv module) ###"
                    write(*,*) "Detect a certain value", aval, i, j, k
                    write(*,*) "STOP."
                    stop
                 end if
              end if
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(coe(i,j,k)==aval)then
                 write(*,*) "### ERROR (Ellip_Slv module) ###"
                 write(*,*) "Detect a certain value", aval, i, j, k
                 write(*,*) "STOP."
                 stop
              end if
           end do
        end do
     end do
  end if

end subroutine check_coe_3df


subroutine check_coe_3dd( coe, aval, undeff )
! 3 次元配列に aval で指定された値が入っていないかを検出する.
  implicit none
  double precision, intent(inout) :: coe(:,:,:)  ! 代入される配列
  double precision, intent(in) :: aval           ! 検出される値
  double precision, intent(in), optional :: undeff
  integer :: i, j, k, nx, ny, nz

  nx=size(coe,1)
  ny=size(coe,2)
  nz=size(coe,3)

  if(present(undeff))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(coe(i,j,k)/=undeff)then
                 if(coe(i,j,k)==aval)then
                    write(*,*) "### ERROR (Ellip_Slv module) ###"
                    write(*,*) "Detect a certain value", aval, i, j, k
                    write(*,*) "STOP."
                    stop
                 end if
              end if
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(coe(i,j,k)==aval)then
                 write(*,*) "### ERROR (Ellip_Slv module) ###"
                 write(*,*) "Detect a certain value", aval, i, j, k
                 write(*,*) "STOP."
                 stop
              end if
           end do
        end do
     end do
  end if

end subroutine check_coe_3dd


subroutine check_le_coe_3df( coe, aval, undeff )
! 3 次元配列に less equal aval を検出する.
  implicit none
  real, intent(inout) :: coe(:,:,:)  ! 代入される配列
  real, intent(in) :: aval           ! 検出される値
  real, intent(in), optional :: undeff
  integer :: i, j, k, nx, ny, nz

  nx=size(coe,1)
  ny=size(coe,2)
  nz=size(coe,3)

  if(present(undeff))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(coe(i,j,k)/=undeff)then
                 if(coe(i,j,k)<=aval)then
                    write(*,*) "### ERROR (Ellip_Slv module) ###"
                    write(*,*) "Detect a certain value", coe(i,j,k), i, j, k
                    write(*,*) "STOP."
                    stop
                 end if
              end if
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(coe(i,j,k)<=aval)then
                 write(*,*) "### ERROR (Ellip_Slv module) ###"
                 write(*,*) "Detect a certain value", coe(i,j,k), i, j, k
                 write(*,*) "STOP."
                 stop
              end if
           end do
        end do
     end do
  end if

end subroutine check_le_coe_3df


subroutine check_le_coe_3dd( coe, aval, undeff )
! 3 次元配列に less equal aval を検出する.
  implicit none
  double precision, intent(inout) :: coe(:,:,:)  ! 代入される配列
  double precision, intent(in) :: aval           ! 検出される値
  double precision, intent(in), optional :: undeff
  integer :: i, j, k, nx, ny, nz

  nx=size(coe,1)
  ny=size(coe,2)
  nz=size(coe,3)

  if(present(undeff))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(coe(i,j,k)/=undeff)then
                 if(coe(i,j,k)<=aval)then
                    write(*,*) "### ERROR (Ellip_Slv module) ###"
                    write(*,*) "Detect a certain value", coe(i,j,k), i, j, k
                    write(*,*) "STOP."
                    stop
                 end if
              end if
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(coe(i,j,k)<=aval)then
                 write(*,*) "### ERROR (Ellip_Slv module) ###"
                 write(*,*) "Detect a certain value", coe(i,j,k), i, j, k
                 write(*,*) "STOP."
                 stop
              end if
           end do
        end do
     end do
  end if

end subroutine check_le_coe_3dd


subroutine calculate_bound_3df( ib, dx, dy, dz, bnd, psi )
! ib 判別を元に, ノイマン型, 周期境界型について境界値を計算する.
  integer, intent(in) :: ib(:,:,:)  ! 境界整数判別
  real, intent(in) :: dx(size(ib,1))  ! x 方向の格子解像度
  real, intent(in) :: dy(size(ib,2))  ! y 方向の格子解像度
  real, intent(in) :: dz(size(ib,3))  ! z 方向の格子解像度
  real, intent(in) :: bnd(size(ib,1),size(ib,2),size(ib,3))
                                      ! 境界値(ノイマン型のみ使用)
  real, intent(inout) :: psi(size(ib,1),size(ib,2),size(ib,3))  ! 応答関数
  integer :: i, j, k, nx, ny, nz, ix, jy, kz

  nx=size(ib,1)
  ny=size(ib,2)
  nz=size(ib,3)

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k,ix,jy,kz)
  do k=1,nz
     do j=1,ny
        do i=1,nx
           if(ib(i,j,k)/=0)then
              select case (ib(i,j,k))
              case (1)
                 psi(i,j,k)=bnd(i,j,k)

              case (2)  ! x 方向にフラックス一定, 上側が参照値
                 psi(i,j,k)=psi(i+1,j,k)-bnd(i,j,k)*dx(i)

              case (-2)  ! x 方向にフラックス一定, 下側が参照値
                 psi(i,j,k)=psi(i-1,j,k)+bnd(i,j,k)*dx(i)

              case (4)  ! y 方向にフラックス一定, 右側が参照値
                 psi(i,j,k)=psi(i,j+1,k)-bnd(i,j,k)*dy(j)

              case (-4)  ! y 方向にフラックス一定, 左側が参照値
                 psi(i,j,k)=psi(i,j-1,k)+bnd(i,j,k)*dy(j)

              case (6)  ! y 方向にフラックス一定, 右側が参照値
                 psi(i,j,k)=psi(i,j,k+1)-bnd(i,j,k)*dz(k)

              case (-6)  ! y 方向にフラックス一定, 左側が参照値
                 psi(i,j,k)=psi(i,j,k-1)+bnd(i,j,k)*dz(k)

              case (3)  ! 12 辺, もしくは 8 点で周期境界を判断
                 if(i==1)then
                    ix=nx-1
                 else if(i==nx)then
                    ix=2
                 else
                    ix=i
                 end if
                 if(j==1)then
                    jy=ny-1
                 else if(j==ny)then
                    jy=2
                 else
                    jy=j
                 end if
                 if(k==1)then
                    kz=nz-1
                 else if(k==nz)then
                    kz=2
                 else
                    kz=k
                 end if
                 psi(i,j,k)=psi(ix,jy,kz)

              case (8)  ! 両方フラックス一定で z 面の x, y 右上か左下角.
                 if(i==1.and.j==1)then  ! -- 評価 1
                    psi(i,j,k)=psi(i+1,j+1,k)-0.5*(bnd(i+1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))

                 else if(i==nx.and.j==ny)then  ! -- 評価 2
                    psi(i,j,k)=psi(i-1,j-1,k)+0.5*(bnd(i-1,j,k)*dy(j)+bnd(i,j-1,k)*dx(i))

                 else if(ib(i-1,j,k)==10.and.ib(i,j-1,k)==10)then
                    ! -- 評価 1 と同じ
                    psi(i,j,k)=psi(i+1,j+1,k)-0.5*(bnd(i+1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))

                 else if(ib(i+1,j,k)==10.and.ib(i,j+1,k)==10)then
                    ! -- 評価 2 と同じ
                    psi(i,j,k)=psi(i-1,j-1,k)+0.5*(bnd(i-1,j,k)*dy(j)+bnd(i,j-1,k)*dx(i))

                 end if

              case (-8)  ! 両方フラックス一定で z 面の x, y 右下か左上角.
                 if(i==1.and.j==ny)then  ! -- 評価 1
                    psi(i,j,k)=psi(i+1,j-1,k)+0.5*(bnd(i+1,j,k)*dy(j)-bnd(i,j-1,k)*dx(i))

                 else if(i==nx.and.j==1)then  ! -- 評価 2
                    psi(i,j,k)=psi(i-1,j+1,k)+0.5*(-bnd(i-1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))

                 else if(ib(i-1,j,k)==10.and.ib(i,j+1,k)==10)then
                    ! -- 評価 1 と同じ
                    psi(i,j,k)=psi(i+1,j-1,k)+0.5*(bnd(i+1,j,k)*dy(j)-bnd(i,j-1,k)*dx(i))

                 else if(ib(i+1,j,k)==10.and.ib(i,j-1,k)==10)then
                    ! -- 評価 2 と同じ
                    psi(i,j,k)=psi(i-1,j+1,k)+0.5*(-bnd(i-1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))
                 end if

              case (12)  ! 両方フラックス一定で y 面の x, z 右上か左下角.
                 if(i==1.and.k==1)then  ! -- 評価 1
                    psi(i,j,k)=psi(i+1,j,k+1)-0.5*(bnd(i+1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                 else if(i==nx.and.k==nz)then  ! -- 評価 2
                    psi(i,j,k)=psi(i-1,j,k-1)+0.5*(bnd(i-1,j,k)*dz(k)+bnd(i,j,k-1)*dx(i))

                 else if(ib(i-1,j,k)==10.and.ib(i,j,k-1)==10)then
                    ! -- 評価 1 と同じ
                    psi(i,j,k)=psi(i+1,j,k+1)-0.5*(bnd(i+1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                 else if(ib(i+1,j,k)==10.and.ib(i,j,k+1)==10)then
                    ! -- 評価 2 と同じ
                    psi(i,j,k)=psi(i-1,j,k-1)+0.5*(bnd(i-1,j,k)*dz(k)+bnd(i,j,k-1)*dx(i))

                 end if

              case (-12)  ! 両方フラックス一定で y 面の x, z 右下か左上角.
                 if(i==1.and.k==nz)then  ! -- 評価 1
                    psi(i,j,k)=psi(i+1,j,k-1)+0.5*(bnd(i+1,j,k)*dz(k)-bnd(i,j,k-1)*dx(i))

                 else if(i==nx.and.k==1)then  ! -- 評価 2
                    psi(i,j,k)=psi(i-1,j,k+1)+0.5*(-bnd(i-1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                 else if(ib(i-1,j,k)==10.and.ib(i,j,k+1)==10)then
                    ! -- 評価 1 と同じ
                    psi(i,j,k)=psi(i+1,j,k-1)+0.5*(bnd(i+1,j,k)*dz(k)-bnd(i,j,k-1)*dx(i))

                 else if(ib(i+1,j,k)==10.and.ib(i,j,k-1)==10)then
                    ! -- 評価 2 と同じ
                    psi(i,j,k)=psi(i-1,j,k+1)+0.5*(-bnd(i-1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                 end if

              case (24)  ! 両方フラックス一定で x 面の y, z 右上か左下角.
                 if(j==1.and.k==1)then  ! -- 評価 1
                    psi(i,j,k)=psi(i,j+1,k+1)-0.5*(bnd(i,j+1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                 else if(j==ny.and.k==nz)then  ! -- 評価 2
                    psi(i,j,k)=psi(i,j-1,k-1)+0.5*(bnd(i,j-1,k)*dz(k)+bnd(i,j,k-1)*dy(j))

                 else if(ib(i,j-1,k)==10.and.ib(i,j,k-1)==10)then
                    ! -- 評価 1 と同じ
                    psi(i,j,k)=psi(i,j+1,k+1)-0.5*(bnd(i,j+1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                 else if(ib(i,j+1,k)==10.and.ib(i,j,k+1)==10)then
                    ! -- 評価 2 と同じ
                    psi(i,j,k)=psi(i,j-1,k-1)+0.5*(bnd(i,j-1,k)*dz(k)+bnd(i,j,k-1)*dy(j))

                 end if

              case (-24)  ! 両方フラックス一定で x 面の y, z 右下か左上角.
                 if(j==1.and.k==nz)then  ! -- 評価 1
                    psi(i,j,k)=psi(i,j+1,k-1)+0.5*(bnd(i,j+1,k)*dz(k)-bnd(i,j,k-1)*dy(j))

                 else if(j==ny.and.k==1)then  ! -- 評価 2
                    psi(i,j,k)=psi(i,j-1,k+1)+0.5*(-bnd(i,j-1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                 else if(ib(i,j-1,k)==10.and.ib(i,j,k+1)==10)then
                    ! -- 評価 1 と同じ
                    psi(i,j,k)=psi(i,j+1,k-1)+0.5*(bnd(i,j+1,k)*dz(k)-bnd(i,j,k-1)*dy(j))

                 else if(ib(i,j+1,k)==10.and.ib(i,j,k-1)==10)then
                    ! -- 評価 2 と同じ
                    psi(i,j,k)=psi(i,j-1,k+1)+0.5*(-bnd(i,j-1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                 end if

              !-- 以降, 隅領域なので, 個別に設定.
              case (11)  ! 両方フラックス一定で (1,1,1) 点, もしくは内部領域.
                 psi(i,j,k)=psi(i+1,j+1,k+1)  &
  &                        -(bnd(i,j+1,k+1)*dx(i)  &
  &                         +bnd(i+1,j,k+1)*dy(j)  &
  &                         +bnd(i+1,j+1,k)*dz(k))/3.0

              case (13)  ! 両方フラックス一定で (nx,1,1) 点, もしくは内部領域.
                 psi(i,j,k)=psi(i-1,j+1,k+1)  &
  &                        -(-bnd(i,j+1,k+1)*dx(i)  &
  &                          +bnd(i-1,j,k+1)*dy(j)  &
  &                          +bnd(i-1,j+1,k)*dz(k))/3.0

              case (17)  ! 両方フラックス一定で (1,ny,1) 点, もしくは内部領域.
                 psi(i,j,k)=psi(i+1,j-1,k+1)  &
  &                        -(bnd(i,j-1,k+1)*dx(i)  &
  &                         -bnd(i+1,j,k+1)*dy(j)  &
  &                         +bnd(i+1,j-1,k)*dz(k))/3.0

              case (19)  ! 両方フラックス一定で (nx,ny,1) 点, もしくは内部領域.
                 psi(i,j,k)=psi(i-1,j-1,k+1)  &
  &                        -(-bnd(i,j-1,k+1)*dx(i)  &
  &                          -bnd(i-1,j,k+1)*dy(j)  &
  &                          +bnd(i-1,j-1,k)*dz(k))/3.0

              case (23)  ! 両方フラックス一定で (1,1,nz) 点, もしくは内部領域.
                 psi(i,j,k)=psi(i+1,j+1,k-1)  &
  &                        -(bnd(i,j+1,k-1)*dx(i)  &
  &                         +bnd(i+1,j,k-1)*dy(j)  &
  &                         -bnd(i+1,j+1,k)*dz(k))/3.0

              case (29)  ! 両方フラックス一定で (nx,1,nz) 点, もしくは内部領域.
                 psi(i,j,k)=psi(i-1,j+1,k-1)  &
  &                        -(-bnd(i,j+1,k-1)*dx(i)  &
  &                          +bnd(i-1,j,k-1)*dy(j)  &
  &                          -bnd(i-1,j+1,k)*dz(k))/3.0

              case (31)  ! 両方フラックス一定で (1,ny,nz) 点, もしくは内部領域.
                 psi(i,j,k)=psi(i+1,j-1,k-1)  &
  &                        -(bnd(i,j-1,k-1)*dx(i)  &
  &                         -bnd(i+1,j,k-1)*dy(j)  &
  &                         -bnd(i+1,j-1,k)*dz(k))/3.0

              case (37)  ! 両方フラックス一定で (nx,ny,nz) 点, もしくは内部領域.
                 psi(i,j,k)=psi(i-1,j-1,k-1)  &
  &                        -(-bnd(i,j-1,k-1)*dx(i)  &
  &                          -bnd(i-1,j,k-1)*dy(j)  &
  &                          -bnd(i-1,j-1,k)*dz(k))/3.0

              case (-11)  ! 両方フラックス一定で (i+1,j+1,k+1) に 10 が設定.
                 psi(i,j,k)=(psi(i-1,j+1,k+1)+psi(i+1,j-1,k+1)+psi(i+1,j+1,k-1)  &
  &                         +bnd(i,j+1,k+1)*dx(i)+bnd(i+1,j,k+1)*dy(j)  &
  &                         +bnd(i+1,j+1,k)*dz(k))/3.0

              case (-13)  ! 両方フラックス一定で (i-1,j+1,k+1) に 10 が設定.
                 psi(i,j,k)=(psi(i+1,j+1,k+1)+psi(i-1,j-1,k+1)+psi(i-1,j+1,k-1)  &
  &                         -bnd(i,j+1,k+1)*dx(i)+bnd(i-1,j,k+1)*dy(j)  &
  &                         +bnd(i-1,j+1,k)*dz(k))/3.0

              case (-17)  ! 両方フラックス一定で (i+1,j-1,k+1) に 10 が設定.
                 psi(i,j,k)=(psi(i-1,j-1,k+1)+psi(i+1,j+1,k+1)+psi(i+1,j-1,k-1)  &
  &                         +bnd(i,j-1,k+1)*dx(i)-bnd(i+1,j,k+1)*dy(j)  &
  &                         +bnd(i+1,j-1,k)*dz(k))/3.0

              case (-19)  ! 両方フラックス一定で (i-1,j-1,k+1) に 10 が設定.
                 psi(i,j,k)=(psi(i+1,j-1,k+1)+psi(i-1,j+1,k+1)+psi(i-1,j-1,k-1)  &
  &                         -bnd(i,j-1,k+1)*dx(i)-bnd(i-1,j,k+1)*dy(j)  &
  &                         +bnd(i-1,j-1,k)*dz(k))/3.0

              case (-23)  ! 両方フラックス一定で (i+1,j+1,k-1) に 10 が設定.
                 psi(i,j,k)=(psi(i-1,j+1,k-1)+psi(i+1,j-1,k-1)+psi(i+1,j+1,k+1)  &
  &                         +bnd(i,j+1,k-1)*dx(i)+bnd(i+1,j,k-1)*dy(j)  &
  &                         -bnd(i+1,j+1,k)*dz(k))/3.0

              case (-29)  ! 両方フラックス一定で (i-1,j+1,k-1) に 10 が設定.
                 psi(i,j,k)=(psi(i+1,j+1,k-1)+psi(i-1,j-1,k-1)+psi(i-1,j+1,k+1)  &
  &                         -bnd(i,j+1,k-1)*dx(i)+bnd(i-1,j,k-1)*dy(j)  &
  &                         -bnd(i-1,j+1,k)*dz(k))/3.0

              case (-31)  ! 両方フラックス一定で (i+1,j-1,k-1) に 10 が設定.
                 psi(i,j,k)=(psi(i-1,j-1,k-1)+psi(i+1,j+1,k-1)+psi(i+1,j-1,k+1)  &
  &                         +bnd(i,j-1,k-1)*dx(i)-bnd(i+1,j,k-1)*dy(j)  &
  &                         -bnd(i+1,j-1,k)*dz(k))/3.0

              case (-37)  ! 両方フラックス一定で (i-1,j-1,k-1) に 10 が設定.
                 psi(i,j,k)=(psi(i+1,j-1,k-1)+psi(i-1,j+1,k-1)+psi(i-1,j-1,k+1)  &
  &                         -bnd(i,j-1,k-1)*dx(i)-bnd(i-1,j,k-1)*dy(j)  &
  &                         -bnd(i-1,j-1,k)*dz(k))/3.0

              end select

           end if
        end do
     end do
  end do
!$omp end do
!$omp end parallel

end subroutine calculate_bound_3df


subroutine calculate_bound_3dd( ib, dx, dy, dz, bnd, psi )
! ib 判別を元に, ノイマン型, 周期境界型について境界値を計算する.
  integer, intent(in) :: ib(:,:,:)  ! 境界整数判別
  double precision, intent(in) :: dx(size(ib,1))  ! x 方向の格子解像度
  double precision, intent(in) :: dy(size(ib,2))  ! y 方向の格子解像度
  double precision, intent(in) :: dz(size(ib,3))  ! z 方向の格子解像度
  double precision, intent(in) :: bnd(size(ib,1),size(ib,2),size(ib,3))
                                      ! 境界値(ノイマン型のみ使用)
  double precision, intent(inout) :: psi(size(ib,1),size(ib,2),size(ib,3))  ! 応答関数
  integer :: i, j, k, nx, ny, nz, ix, jy, kz

  nx=size(ib,1)
  ny=size(ib,2)
  nz=size(ib,3)

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k,ix,jy,kz)
  do k=1,nz
     do j=1,ny
        do i=1,nx
           if(ib(i,j,k)/=0)then
              select case (ib(i,j,k))
              case (1)
                 psi(i,j,k)=bnd(i,j,k)

              case (2)  ! x 方向にフラックス一定, 上側が参照値
                 psi(i,j,k)=psi(i+1,j,k)-bnd(i,j,k)*dx(i)

              case (-2)  ! x 方向にフラックス一定, 下側が参照値
                 psi(i,j,k)=psi(i-1,j,k)+bnd(i,j,k)*dx(i)

              case (4)  ! y 方向にフラックス一定, 右側が参照値
                 psi(i,j,k)=psi(i,j+1,k)-bnd(i,j,k)*dy(j)

              case (-4)  ! y 方向にフラックス一定, 左側が参照値
                 psi(i,j,k)=psi(i,j-1,k)+bnd(i,j,k)*dy(j)

              case (6)  ! y 方向にフラックス一定, 右側が参照値
                 psi(i,j,k)=psi(i,j,k+1)-bnd(i,j,k)*dz(k)

              case (-6)  ! y 方向にフラックス一定, 左側が参照値
                 psi(i,j,k)=psi(i,j,k-1)+bnd(i,j,k)*dz(k)

              case (3)  ! 12 辺, もしくは 8 点で周期境界を判断
                 if(i==1)then
                    ix=nx-1
                 else if(i==nx)then
                    ix=2
                 else
                    ix=i
                 end if
                 if(j==1)then
                    jy=ny-1
                 else if(j==ny)then
                    jy=2
                 else
                    jy=j
                 end if
                 if(k==1)then
                    kz=nz-1
                 else if(k==nz)then
                    kz=2
                 else
                    kz=k
                 end if
                 psi(i,j,k)=psi(ix,jy,kz)

              case (8)  ! 両方フラックス一定で z 面の x, y 右上か左下角.
                 if(i==1.and.j==1)then  ! -- 評価 1
                    psi(i,j,k)=psi(i+1,j+1,k)-0.5d0*(bnd(i+1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))

                 else if(i==nx.and.j==ny)then  ! -- 評価 2
                    psi(i,j,k)=psi(i-1,j-1,k)+0.5d0*(bnd(i-1,j,k)*dy(j)+bnd(i,j-1,k)*dx(i))

                 else if(ib(i-1,j,k)==10.and.ib(i,j-1,k)==10)then
                    ! -- 評価 1 と同じ
                    psi(i,j,k)=psi(i+1,j+1,k)-0.5d0*(bnd(i+1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))

                 else if(ib(i+1,j,k)==10.and.ib(i,j+1,k)==10)then
                    ! -- 評価 2 と同じ
                    psi(i,j,k)=psi(i-1,j-1,k)+0.5d0*(bnd(i-1,j,k)*dy(j)+bnd(i,j-1,k)*dx(i))

                 end if

              case (-8)  ! 両方フラックス一定で z 面の x, y 右下か左上角.
                 if(i==1.and.j==ny)then  ! -- 評価 1
                    psi(i,j,k)=psi(i+1,j-1,k)+0.5d0*(bnd(i+1,j,k)*dy(j)-bnd(i,j-1,k)*dx(i))

                 else if(i==nx.and.j==1)then  ! -- 評価 2
                    psi(i,j,k)=psi(i-1,j+1,k)+0.5d0*(-bnd(i-1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))

                 else if(ib(i-1,j,k)==10.and.ib(i,j+1,k)==10)then
                    ! -- 評価 1 と同じ
                    psi(i,j,k)=psi(i+1,j-1,k)+0.5d0*(bnd(i+1,j,k)*dy(j)-bnd(i,j-1,k)*dx(i))

                 else if(ib(i+1,j,k)==10.and.ib(i,j-1,k)==10)then
                    ! -- 評価 2 と同じ
                    psi(i,j,k)=psi(i-1,j+1,k)+0.5d0*(-bnd(i-1,j,k)*dy(j)+bnd(i,j+1,k)*dx(i))
                 end if

              case (12)  ! 両方フラックス一定で y 面の x, z 右上か左下角.
                 if(i==1.and.k==1)then  ! -- 評価 1
                    psi(i,j,k)=psi(i+1,j,k+1)-0.5d0*(bnd(i+1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                 else if(i==nx.and.k==nz)then  ! -- 評価 2
                    psi(i,j,k)=psi(i-1,j,k-1)+0.5d0*(bnd(i-1,j,k)*dz(k)+bnd(i,j,k-1)*dx(i))

                 else if(ib(i-1,j,k)==10.and.ib(i,j,k-1)==10)then
                    ! -- 評価 1 と同じ
                    psi(i,j,k)=psi(i+1,j,k+1)-0.5d0*(bnd(i+1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                 else if(ib(i+1,j,k)==10.and.ib(i,j,k+1)==10)then
                    ! -- 評価 2 と同じ
                    psi(i,j,k)=psi(i-1,j,k-1)+0.5d0*(bnd(i-1,j,k)*dz(k)+bnd(i,j,k-1)*dx(i))

                 end if

              case (-12)  ! 両方フラックス一定で y 面の x, z 右下か左上角.
                 if(i==1.and.k==nz)then  ! -- 評価 1
                    psi(i,j,k)=psi(i+1,j,k-1)+0.5d0*(bnd(i+1,j,k)*dz(k)-bnd(i,j,k-1)*dx(i))

                 else if(i==nx.and.k==1)then  ! -- 評価 2
                    psi(i,j,k)=psi(i-1,j,k+1)+0.5d0*(-bnd(i-1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                 else if(ib(i-1,j,k)==10.and.ib(i,j,k+1)==10)then
                    ! -- 評価 1 と同じ
                    psi(i,j,k)=psi(i+1,j,k-1)+0.5d0*(bnd(i+1,j,k)*dz(k)-bnd(i,j,k-1)*dx(i))

                 else if(ib(i+1,j,k)==10.and.ib(i,j,k-1)==10)then
                    ! -- 評価 2 と同じ
                    psi(i,j,k)=psi(i-1,j,k+1)+0.5d0*(-bnd(i-1,j,k)*dz(k)+bnd(i,j,k+1)*dx(i))

                 end if

              case (24)  ! 両方フラックス一定で x 面の y, z 右上か左下角.
                 if(j==1.and.k==1)then  ! -- 評価 1
                    psi(i,j,k)=psi(i,j+1,k+1)-0.5d0*(bnd(i,j+1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                 else if(j==ny.and.k==nz)then  ! -- 評価 2
                    psi(i,j,k)=psi(i,j-1,k-1)+0.5d0*(bnd(i,j-1,k)*dz(k)+bnd(i,j,k-1)*dy(j))

                 else if(ib(i,j-1,k)==10.and.ib(i,j,k-1)==10)then
                    ! -- 評価 1 と同じ
                    psi(i,j,k)=psi(i,j+1,k+1)-0.5d0*(bnd(i,j+1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                 else if(ib(i,j+1,k)==10.and.ib(i,j,k+1)==10)then
                    ! -- 評価 2 と同じ
                    psi(i,j,k)=psi(i,j-1,k-1)+0.5d0*(bnd(i,j-1,k)*dz(k)+bnd(i,j,k-1)*dy(j))

                 end if

              case (-24)  ! 両方フラックス一定で x 面の y, z 右下か左上角.
                 if(j==1.and.k==nz)then  ! -- 評価 1
                    psi(i,j,k)=psi(i,j+1,k-1)+0.5d0*(bnd(i,j+1,k)*dz(k)-bnd(i,j,k-1)*dy(j))

                 else if(j==ny.and.k==1)then  ! -- 評価 2
                    psi(i,j,k)=psi(i,j-1,k+1)+0.5d0*(-bnd(i,j-1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                 else if(ib(i,j-1,k)==10.and.ib(i,j,k+1)==10)then
                    ! -- 評価 1 と同じ
                    psi(i,j,k)=psi(i,j+1,k-1)+0.5d0*(bnd(i,j+1,k)*dz(k)-bnd(i,j,k-1)*dy(j))

                 else if(ib(i,j+1,k)==10.and.ib(i,j,k-1)==10)then
                    ! -- 評価 2 と同じ
                    psi(i,j,k)=psi(i,j-1,k+1)+0.5d0*(-bnd(i,j-1,k)*dz(k)+bnd(i,j,k+1)*dy(j))

                 end if

              !-- 以降, 隅領域なので, 個別に設定.
              case (11)  ! 両方フラックス一定で (1,1,1) 点, もしくは内部領域.
                 psi(i,j,k)=psi(i+1,j+1,k+1)  &
  &                        -(bnd(i,j+1,k+1)*dx(i)  &
  &                         +bnd(i+1,j,k+1)*dy(j)  &
  &                         +bnd(i+1,j+1,k)*dz(k))/3.0d0

              case (13)  ! 両方フラックス一定で (nx,1,1) 点, もしくは内部領域.
                 psi(i,j,k)=psi(i-1,j+1,k+1)  &
  &                        -(-bnd(i,j+1,k+1)*dx(i)  &
  &                          +bnd(i-1,j,k+1)*dy(j)  &
  &                          +bnd(i-1,j+1,k)*dz(k))/3.0d0

              case (17)  ! 両方フラックス一定で (1,ny,1) 点, もしくは内部領域.
                 psi(i,j,k)=psi(i+1,j-1,k+1)  &
  &                        -(bnd(i,j-1,k+1)*dx(i)  &
  &                         -bnd(i+1,j,k+1)*dy(j)  &
  &                         +bnd(i+1,j-1,k)*dz(k))/3.0d0

              case (19)  ! 両方フラックス一定で (nx,ny,1) 点, もしくは内部領域.
                 psi(i,j,k)=psi(i-1,j-1,k+1)  &
  &                        -(-bnd(i,j-1,k+1)*dx(i)  &
  &                          -bnd(i-1,j,k+1)*dy(j)  &
  &                          +bnd(i-1,j-1,k)*dz(k))/3.0d0

              case (23)  ! 両方フラックス一定で (1,1,nz) 点, もしくは内部領域.
                 psi(i,j,k)=psi(i+1,j+1,k-1)  &
  &                        -(bnd(i,j+1,k-1)*dx(i)  &
  &                         +bnd(i+1,j,k-1)*dy(j)  &
  &                         -bnd(i+1,j+1,k)*dz(k))/3.0d0

              case (29)  ! 両方フラックス一定で (nx,1,nz) 点, もしくは内部領域.
                 psi(i,j,k)=psi(i-1,j+1,k-1)  &
  &                        -(-bnd(i,j+1,k-1)*dx(i)  &
  &                          +bnd(i-1,j,k-1)*dy(j)  &
  &                          -bnd(i-1,j+1,k)*dz(k))/3.0d0

              case (31)  ! 両方フラックス一定で (1,ny,nz) 点, もしくは内部領域.
                 psi(i,j,k)=psi(i+1,j-1,k-1)  &
  &                        -(bnd(i,j-1,k-1)*dx(i)  &
  &                         -bnd(i+1,j,k-1)*dy(j)  &
  &                         -bnd(i+1,j-1,k)*dz(k))/3.0d0

              case (37)  ! 両方フラックス一定で (nx,ny,nz) 点, もしくは内部領域.
                 psi(i,j,k)=psi(i-1,j-1,k-1)  &
  &                        -(-bnd(i,j-1,k-1)*dx(i)  &
  &                          -bnd(i-1,j,k-1)*dy(j)  &
  &                          -bnd(i-1,j-1,k)*dz(k))/3.0d0

              case (-11)  ! 両方フラックス一定で (i+1,j+1,k+1) に 10 が設定.
                 psi(i,j,k)=(psi(i-1,j+1,k+1)+psi(i+1,j-1,k+1)+psi(i+1,j+1,k-1)  &
  &                         +bnd(i,j+1,k+1)*dx(i)+bnd(i+1,j,k+1)*dy(j)  &
  &                         +bnd(i+1,j+1,k)*dz(k))/3.0d0

              case (-13)  ! 両方フラックス一定で (i-1,j+1,k+1) に 10 が設定.
                 psi(i,j,k)=(psi(i+1,j+1,k+1)+psi(i-1,j-1,k+1)+psi(i-1,j+1,k-1)  &
  &                         -bnd(i,j+1,k+1)*dx(i)+bnd(i-1,j,k+1)*dy(j)  &
  &                         +bnd(i-1,j+1,k)*dz(k))/3.0d0

              case (-17)  ! 両方フラックス一定で (i+1,j-1,k+1) に 10 が設定.
                 psi(i,j,k)=(psi(i-1,j-1,k+1)+psi(i+1,j+1,k+1)+psi(i+1,j-1,k-1)  &
  &                         +bnd(i,j-1,k+1)*dx(i)-bnd(i+1,j,k+1)*dy(j)  &
  &                         +bnd(i+1,j-1,k)*dz(k))/3.0d0

              case (-19)  ! 両方フラックス一定で (i-1,j-1,k+1) に 10 が設定.
                 psi(i,j,k)=(psi(i+1,j-1,k+1)+psi(i-1,j+1,k+1)+psi(i-1,j-1,k-1)  &
  &                         -bnd(i,j-1,k+1)*dx(i)-bnd(i-1,j,k+1)*dy(j)  &
  &                         +bnd(i-1,j-1,k)*dz(k))/3.0d0

              case (-23)  ! 両方フラックス一定で (i+1,j+1,k-1) に 10 が設定.
                 psi(i,j,k)=(psi(i-1,j+1,k-1)+psi(i+1,j-1,k-1)+psi(i+1,j+1,k+1)  &
  &                         +bnd(i,j+1,k-1)*dx(i)+bnd(i+1,j,k-1)*dy(j)  &
  &                         -bnd(i+1,j+1,k)*dz(k))/3.0d0

              case (-29)  ! 両方フラックス一定で (i-1,j+1,k-1) に 10 が設定.
                 psi(i,j,k)=(psi(i+1,j+1,k-1)+psi(i-1,j-1,k-1)+psi(i-1,j+1,k+1)  &
  &                         -bnd(i,j+1,k-1)*dx(i)+bnd(i-1,j,k-1)*dy(j)  &
  &                         -bnd(i-1,j+1,k)*dz(k))/3.0d0

              case (-31)  ! 両方フラックス一定で (i+1,j-1,k-1) に 10 が設定.
                 psi(i,j,k)=(psi(i-1,j-1,k-1)+psi(i+1,j+1,k-1)+psi(i+1,j-1,k+1)  &
  &                         +bnd(i,j-1,k-1)*dx(i)-bnd(i+1,j,k-1)*dy(j)  &
  &                         -bnd(i+1,j-1,k)*dz(k))/3.0d0

              case (-37)  ! 両方フラックス一定で (i-1,j-1,k-1) に 10 が設定.
                 psi(i,j,k)=(psi(i+1,j-1,k-1)+psi(i-1,j+1,k-1)+psi(i-1,j-1,k+1)  &
  &                         -bnd(i,j-1,k-1)*dx(i)-bnd(i-1,j,k-1)*dy(j)  &
  &                         -bnd(i-1,j-1,k)*dz(k))/3.0d0

              end select

           end if
        end do
     end do
  end do
!$omp end do
!$omp end parallel

end subroutine calculate_bound_3dd


real function check_diff_error_2df( x, y, psi, rho, at, bt, ct, dt, et, ft, ib )
  implicit none
  real, intent(in) :: x(:)
  real, intent(in) :: y(:)
  real, intent(in) :: psi(size(x),size(y))
  real, intent(in) :: rho(size(x),size(y))
  real, intent(in) :: at(size(x),size(y))
  real, intent(in) :: bt(size(x),size(y))
  real, intent(in) :: ct(size(x),size(y))
  real, intent(in) :: dt(size(x),size(y))
  real, intent(in) :: et(size(x),size(y))
  real, intent(in) :: ft(size(x),size(y))
  integer, intent(in) :: ib(size(x),size(y))
  integer :: nx, ny, i, j
  real, parameter :: undef=-999.0
  real :: emax
  real, dimension(size(x),size(y)) :: dp2dx2, dp2dy2, dpdx, dpdy, dp2dxy
  real, dimension(size(x),size(y)) :: xi, errmax

  nx=size(x)
  ny=size(y)
  emax=0.0

  xi=psi

  do j=1,ny
     do i=1,nx
        if(ib(i,j)==10)then
           xi(i,j)=undef
        end if
     end do
  end do

  call grad_2d( x, y, xi, dpdx, dpdy, undeff=undef )

  do j=1,ny
     call laplacian_1d( x, xi(:,j), dp2dx2(:,j), undef=undef )
     call grad_1d( x, dpdy(:,j), dp2dxy(:,j), undef=undef )
  end do
  do i=1,nx
     call laplacian_1d( y, xi(i,:), dp2dy2(i,:), undef=undef )
  end do

  do j=1,ny
     do i=1,nx
        if(xi(i,j)/=undef)then
           errmax(i,j)=at(i,j)*dp2dx2(i,j)  &
  &                   +ct(i,j)*dp2dy2(i,j)  &
  &                   +bt(i,j)*dp2dxy(i,j)  &
  &                   +dt(i,j)*dpdx(i,j)  &
  &                   +et(i,j)*dpdy(i,j)  &
  &                   +ft(i,j)*xi(i,j)  &
  &                   -rho(i,j)
           errmax(i,j)=abs(errmax(i,j))
        else
           errmax(i,j)=0.0
        end if
        if(emax<errmax(i,j))then
           emax=errmax(i,j)
        end if
     end do
  end do

  check_diff_error_2df=emax

  return

end function check_diff_error_2df


double precision function check_diff_error_2dd( x, y, psi, rho, at, bt, ct, dt, et, ft, ib )
  implicit none
  double precision, intent(in) :: x(:)
  double precision, intent(in) :: y(:)
  double precision, intent(in) :: psi(size(x),size(y))
  double precision, intent(in) :: rho(size(x),size(y))
  double precision, intent(in) :: at(size(x),size(y))
  double precision, intent(in) :: bt(size(x),size(y))
  double precision, intent(in) :: ct(size(x),size(y))
  double precision, intent(in) :: dt(size(x),size(y))
  double precision, intent(in) :: et(size(x),size(y))
  double precision, intent(in) :: ft(size(x),size(y))
  integer, intent(in) :: ib(size(x),size(y))
  integer :: nx, ny, i, j
  double precision, parameter :: undef=-999.0
  double precision :: emax
  double precision, dimension(size(x),size(y)) :: dp2dx2, dp2dy2, dpdx, dpdy, dp2dxy
  double precision, dimension(size(x),size(y)) :: xi, errmax

  nx=size(x)
  ny=size(y)
  emax=0.0

  xi=psi

  do j=1,ny
     do i=1,nx
        if(ib(i,j)==10)then
           xi(i,j)=undef
        end if
     end do
  end do

  call grad_2d( x, y, xi, dpdx, dpdy, undeff=undef )

  do j=1,ny
     call laplacian_1d( x, xi(:,j), dp2dx2(:,j), undef=undef )
     call grad_1d( x, dpdy(:,j), dp2dxy(:,j), undef=undef )
  end do
  do i=1,nx
     call laplacian_1d( y, xi(i,:), dp2dy2(i,:), undef=undef )
  end do

  do j=1,ny
     do i=1,nx
        if(xi(i,j)/=undef)then
           errmax(i,j)=at(i,j)*dp2dx2(i,j)  &
  &                   +ct(i,j)*dp2dy2(i,j)  &
  &                   +bt(i,j)*dp2dxy(i,j)  &
  &                   +dt(i,j)*dpdx(i,j)  &
  &                   +et(i,j)*dpdy(i,j)  &
  &                   +ft(i,j)*xi(i,j)  &
  &                   -rho(i,j)
           errmax(i,j)=abs(errmax(i,j))
        else
           errmax(i,j)=0.0
        end if
        if(emax<errmax(i,j))then
           emax=errmax(i,j)
        end if
     end do
  end do

  check_diff_error_2dd=emax

  return

end function check_diff_error_2dd


end module Ellip_Slv
