!---------------------------------------------------------------
! Copyright (C) 2009-2015 GFD Dennou Club. All rights reserved.
!---------------------------------------------------------------

module Derivation  ! ʬ黻׻⥸塼

  use Basis

  public :: grad_1d,  &
  &         grad_1df,  &
  &         grad_1dd

  public :: grad_2d,  &
  &         grad_2df,  &
  &         grad_2dd

  public :: grad_3d,  &
  &         grad_3df,  &
  &         grad_3dd

  public :: div,  &
  &         divf,  &
  &         divd

  public :: curl,  &
  &         curlf,  &
  &         curld

  public :: div_3d,  &
  &         div_3df,  &
  &         div_3dd

  public :: curl_3d,  &
  &         curl_3df,  &
  &         curl_3dd

  public :: grad4_1d,  &
  &         grad4_1df,  &
  &         grad4_1dd

  public :: grad4_2d,  &
  &         grad4_2df,  &
  &         grad4_2dd

  interface grad_1d
     module procedure grad_1df,  &
  &                   grad_1dd
  end interface grad_1d

  interface grad_2d
     module procedure grad_2df,  &
  &                   grad_2dd
  end interface grad_2d

  interface grad_3d
     module procedure grad_3df,  &
  &                   grad_3dd
  end interface grad_3d

  interface div
     module procedure divf,  &
  &                   divd
  end interface div

  interface curl
     module procedure curlf,  &
  &                   curld
  end interface curl

  interface div_3d
     module procedure div_3df,  &
  &                   div_3dd
  end interface div_3d

  interface curl_3d
     module procedure curl_3df,  &
  &                   curl_3dd
  end interface curl_3d

  interface grad4_1d
     module procedure grad4_1df,  &
  &                   grad4_1dd
  end interface grad4_1d

  interface grad4_2d
     module procedure grad4_2df,  &
  &                   grad4_2dd
  end interface grad4_2d

  interface laplacian_1d
     module procedure laplacian_1df,  &
  &                   laplacian_1dd
  end interface laplacian_1d

  interface laplacian_2d
     module procedure laplacian_2df,  &
  &                   laplacian_2dd
  end interface laplacian_2d

  interface laplacian_3d
     module procedure laplacian_3df,  &
  &                   laplacian_3dd
  end interface laplacian_3d

contains

subroutine divf( x, y, u, v, val, undeff, hx, hy )
! 2ȯ׻롼
! 
! $\frac{\partial u}{\partial x} +\frac{\partial v}{\partial y} $ 
! 2 ʬǽ񤭴,  $(i,j)$ Ǥȯ
! $\frac{u_{i+1,j}-u_{i-1,j}}{2dx} + \frac{v_{i,j+1}-v_{i,j-1}}{2dy} $
! ȤǤ. Ѥ2ȯ׻.
! ǡ­ʤΰǤ, 1 κʬǷ׻Τ, ٤
! .
! ¼Ūˤ, grad_1d 롼Ȥ߹碌뤳ȤǷ׻Ԥ.
! du/dx = grad_1d( x, u, dudx ), dv/dy = grad_1d( y, v, dvdy ) ȤǷ׻Ԥ,
! ⼫ưŪ˷׻ǽ.
  implicit none
  real, intent(in) :: x(:)  ! x ζֺɸ [m]
  real, intent(in) :: y(:)  ! y ζֺɸ [m]
  real, intent(in) :: u(size(x),size(y))  ! x б 2 ٥ȥʬ
  real, intent(in) :: v(size(x),size(y))  ! y б 2 ٥ȥʬ
  real, intent(inout) :: val(size(x),size(y))  ! 2ȯ
  real, intent(in), optional :: undeff
  real, intent(in), optional :: hx(size(x),size(y))  ! x Υ
  real, intent(in), optional :: hy(size(x),size(y))  ! y Υ
  integer :: i   ! 졼ź
  integer :: j   ! 졼ź
  integer :: nx  ! ǿ 1 
  integer :: ny  ! ǿ 2 
  real :: scalex(size(x),size(y)), scaley(size(x),size(y))
  real :: dudx(size(x),size(y)), dvdy(size(x),size(y))
  real :: tmpu(size(x),size(y)), tmpv(size(x),size(y))

  nx=size(x)
  ny=size(y)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "div" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, v ),  &
  &                                     "div" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, val ),  &
  &                                     "div" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, hx ),  &
  &                                        "div" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, hy ),  &
  &                                        "div" )
     end if
  end if

!-- ѿ.

  if(present(hx))then
     do j=1,ny
        do i=1,nx
           scalex(i,j)=hx(i,j)
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           scalex(i,j)=1.0
        end do
     end do
  end if

  if(present(hy))then
     do j=1,ny
        do i=1,nx
           scaley(i,j)=hy(i,j)
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           scaley(i,j)=1.0
        end do
     end do
  end if

!-- ˤ, x  dvdy ׻, y  dudx ׻.

  if(present(undeff))then
     do j=1,ny
        do i=1,nx
           if(u(i,j)/=undeff.and.v(i,j)/=undeff.and.  &
  &           scalex(i,j)/=undeff.and.scaley(i,j)/=undeff)then
              tmpu(i,j)=scaley(i,j)*u(i,j)
              tmpv(i,j)=scalex(i,j)*v(i,j)
           else
              tmpu(i,j)=undeff
              tmpv(i,j)=undeff
           end if
        end do
     end do

     do i=1,nx
        call grad_1df( y, tmpv(i,:), dvdy(i,:), undeff )
     end do
     do j=1,ny
        call grad_1df( x, tmpu(:,j), dudx(:,j), undeff )
     end do

     do j=1,ny
        do i=1,nx
           if(dudx(i,j)/=undeff.and.dvdy(i,j)/=undeff.and.  &
  &           scalex(i,j)/=undeff.and.scaley(i,j)/=undeff)then
              if(scalex(i,j)/=0.0.and.scaley(i,j)/=0.0)then
                 val(i,j)=(dudx(i,j)+dvdy(i,j))/(scalex(i,j)*scaley(i,j))
              else
                 val(i,j)=0.0
              end if
           else
              val(i,j)=undeff
           end if
        end do
     end do

  else

     do j=1,ny
        do i=1,nx
           tmpu(i,j)=scaley(i,j)*u(i,j)
           tmpv(i,j)=scalex(i,j)*v(i,j)
        end do
     end do

     do i=1,nx
        call grad_1df( y, tmpv(i,:), dvdy(i,:) )
     end do
     do j=1,ny
        call grad_1df( x, tmpu(:,j), dudx(:,j) )
     end do

     do j=1,ny
        do i=1,nx
           if(scalex(i,j)/=0.0.and.scaley(i,j)/=0.0)then
              val(i,j)=(dudx(i,j)+dvdy(i,j))/(scalex(i,j)*scaley(i,j))
           else
              val(i,j)=0.0
           end if
        end do
     end do
  end if

end subroutine divf

!-----------------------------------------
!-----------------------------------------

subroutine divd( x, y, u, v, val, undeff, hx, hy )
! 2ȯ׻롼
! 
! $\frac{\partial u}{\partial x} +\frac{\partial v}{\partial y} $ 
! 2 ʬǽ񤭴,  $(i,j)$ Ǥȯ
! $\frac{u_{i+1,j}-u_{i-1,j}}{2dx} + \frac{v_{i,j+1}-v_{i,j-1}}{2dy} $
! ȤǤ. Ѥ2ȯ׻.
! ǡ­ʤΰǤ, 1 κʬǷ׻Τ, ٤
! .
! ¼Ūˤ, grad_1d 롼Ȥ߹碌뤳ȤǷ׻Ԥ.
! du/dx = grad_1d( x, u, dudx ), dv/dy = grad_1d( y, v, dvdy ) ȤǷ׻Ԥ,
! ⼫ưŪ˷׻ǽ.
  implicit none
  double precision, intent(in) :: x(:)  ! x ζֺɸ [m]
  double precision, intent(in) :: y(:)  ! y ζֺɸ [m]
  double precision, intent(in) :: u(size(x),size(y))  ! x б 2 ٥ȥʬ
  double precision, intent(in) :: v(size(x),size(y))  ! y б 2 ٥ȥʬ
  double precision, intent(inout) :: val(size(x),size(y))  ! 2ȯ
  double precision, intent(in), optional :: undeff
  double precision, intent(in), optional :: hx(size(x),size(y))  ! x Υ
  double precision, intent(in), optional :: hy(size(x),size(y))  ! y Υ
  integer :: i   ! 졼ź
  integer :: j   ! 졼ź
  integer :: nx  ! ǿ 1 
  integer :: ny  ! ǿ 2 
  double precision :: scalex(size(x),size(y)), scaley(size(x),size(y))
  double precision :: dudx(size(x),size(y)), dvdy(size(x),size(y))
  double precision :: tmpu(size(x),size(y)), tmpv(size(x),size(y))

  nx=size(x)
  ny=size(y)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "div" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, v ),  &
  &                                     "div" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, val ),  &
  &                                     "div" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, hx ),  &
  &                                        "div" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, hy ),  &
  &                                        "div" )
     end if
  end if

!-- ѿ.

  if(present(hx))then
     do j=1,ny
        do i=1,nx
           scalex(i,j)=hx(i,j)
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           scalex(i,j)=1.0d0
        end do
     end do
  end if

  if(present(hy))then
     do j=1,ny
        do i=1,nx
           scaley(i,j)=hy(i,j)
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           scaley(i,j)=1.0d0
        end do
     end do
  end if

!-- ˤ, x  dvdy ׻, y  dudx ׻.

  if(present(undeff))then
     do j=1,ny
        do i=1,nx
           if(u(i,j)/=undeff.and.v(i,j)/=undeff.and.  &
  &           scalex(i,j)/=undeff.and.scaley(i,j)/=undeff)then
              tmpu(i,j)=scaley(i,j)*u(i,j)
              tmpv(i,j)=scalex(i,j)*v(i,j)
           else
              tmpu(i,j)=undeff
              tmpv(i,j)=undeff
           end if
        end do
     end do

     do i=1,nx
        call grad_1dd( y, tmpv(i,:), dvdy(i,:), undeff )
     end do
     do j=1,ny
        call grad_1dd( x, tmpu(:,j), dudx(:,j), undeff )
     end do

     do j=1,ny
        do i=1,nx
           if(dudx(i,j)/=undeff.and.dvdy(i,j)/=undeff.and.  &
  &           scalex(i,j)/=undeff.and.scaley(i,j)/=undeff)then
              if(scalex(i,j)/=0.0d0.and.scaley(i,j)/=0.0d0)then
                 val(i,j)=(dudx(i,j)+dvdy(i,j))/(scalex(i,j)*scaley(i,j))
              else
                 val(i,j)=0.0d0
              end if
           else
              val(i,j)=undeff
           end if
        end do
     end do

  else

     do j=1,ny
        do i=1,nx
           tmpu(i,j)=scaley(i,j)*u(i,j)
           tmpv(i,j)=scalex(i,j)*v(i,j)
        end do
     end do

     do i=1,nx
        call grad_1dd( y, tmpv(i,:), dvdy(i,:) )
     end do
     do j=1,ny
        call grad_1dd( x, tmpu(:,j), dudx(:,j) )
     end do

     do j=1,ny
        do i=1,nx
           if(scalex(i,j)/=0.0d0.and.scaley(i,j)/=0.0d0)then
              val(i,j)=(dudx(i,j)+dvdy(i,j))/(scalex(i,j)*scaley(i,j))
           else
              val(i,j)=0.0d0
           end if
        end do
     end do
  end if

end subroutine divd

!-----------------------------------------
!-----------------------------------------

subroutine curlf( x, y, u, v, val, undeff, hx, hy, ord )
! 2 ٷ׻롼
!
! x, y μμ㤤ɬ¤٤뤳.
!
! u, v ϶ִθ֤, Ǥդ2٤׻ǽ
! , du/dz-dw/dx ׻ȤΤ, (x,z,u,w) ν֤, ord ץ false.
!
! $\frac{\partial v}{\partial x} -\frac{\partial u}{\partial y} $ 
! 2 ʬǽ񤭴,  $(i,j)$ Ǥȯ
! $\frac{v_{i,j+1}-v_{i,j-1}}{2dx} -\frac{u_{i+1,j}-u_{i-1,j}}{2dy} $
! ȤǤ. Ѥ2ȯ׻.
! ǡ­ʤΰǤ, 1 κʬǷ׻Τ, ٤
! .
! ¼Ūˤ grad_1d ʬ׻ôΤ, η׻⼫ưǹԤ.
  implicit none
  real, intent(in) :: x(:)  ! x ζֺɸ [m]
  real, intent(in) :: y(:)  ! y ζֺɸ [m]
  real, intent(in) :: u(size(x),size(y))  ! x б 2 ٥ȥʬ
  real, intent(in) :: v(size(x),size(y))  ! y б 2 ٥ȥʬ
  real, intent(inout) :: val(size(x),size(y))  ! 2
  real, intent(in), optional :: undeff
  real, intent(in), optional :: hx(size(x),size(y))  ! x Υ
  real, intent(in), optional :: hy(size(x),size(y))  ! y Υ
  logical, intent(in),  optional :: ord  ! ʬν֤ؤ륪ץ.
                 ! true ʤ, ؤʤ, false ʤ, ؤ.
                 ! ǥեȤ true, du/dz-dw/dx ׻ȤΤѤ.
  integer :: i   ! 졼ź
  integer :: j   ! 졼ź
  integer :: nx  ! ǿ 1 
  integer :: ny  ! ǿ 2 
  logical :: order
  real :: scalex(size(x),size(y)), scaley(size(x),size(y))
  real :: dvdx(size(x),size(y)), dudy(size(x),size(y))
  real :: tmpu(size(x),size(y)), tmpv(size(x),size(y))

  nx=size(x)
  ny=size(y)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "curl" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, v ),  &
  &                                     "curl" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, val ),  &
  &                                     "curl" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, hx ),  &
  &                                        "curl" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, hy ),  &
  &                                        "curl" )
     end if
  end if

!-- ѿ.

  if(present(hx))then
     do j=1,ny
        do i=1,nx
           scalex(i,j)=hx(i,j)
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           scalex(i,j)=1.0
        end do
     end do
  end if

  if(present(hy))then
     do j=1,ny
        do i=1,nx
           scaley(i,j)=hy(i,j)
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           scaley(i,j)=1.0
        end do
     end do
  end if

!-- ˤ, x  dvdy ׻, y  dudx ׻.

  if(present(undeff))then
     do j=1,ny
        do i=1,nx
           if(u(i,j)/=undeff.and.v(i,j)/=undeff.and.  &
  &           scalex(i,j)/=undeff.and.scaley(i,j)/=undeff)then
              tmpu(i,j)=scalex(i,j)*u(i,j)
              tmpv(i,j)=scaley(i,j)*v(i,j)
           else
              tmpu(i,j)=undeff
              tmpv(i,j)=undeff
           end if
        end do
     end do

     do i=1,nx
        call grad_1df( y, tmpu(i,:), dudy(i,:), undeff )
     end do
     do j=1,ny
        call grad_1df( x, tmpv(:,j), dvdx(:,j), undeff )
     end do

     do j=1,ny
        do i=1,nx
           if(dudy(i,j)/=undeff.and.dvdx(i,j)/=undeff.and.  &
  &           scalex(i,j)/=undeff.and.scaley(i,j)/=undeff)then
              if(scalex(i,j)/=0.0.and.scaley(i,j)/=0.0)then
                 val(i,j)=(dvdx(i,j)-dudy(i,j))/(scalex(i,j)*scaley(i,j))
              else
                 val(i,j)=0.0
              end if
           else
              val(i,j)=undeff
           end if
        end do
     end do

  else

     do j=1,ny
        do i=1,nx
           tmpu(i,j)=scalex(i,j)*u(i,j)
           tmpv(i,j)=scaley(i,j)*v(i,j)
        end do
     end do

     do i=1,nx
        call grad_1df( y, tmpu(i,:), dudy(i,:) )
     end do
     do j=1,ny
        call grad_1df( x, tmpv(:,j), dvdx(:,j) )
     end do

     do j=1,ny
        do i=1,nx
           if(scalex(i,j)/=0.0.and.scaley(i,j)/=0.0)then
              val(i,j)=(dvdx(i,j)-dudy(i,j))/(scalex(i,j)*scaley(i,j))
           else
              val(i,j)=0.0
           end if
        end do
     end do
  end if

!-- žν֤դˤ륪ץ
!-- false ʤ, ֤ؤƽϤ.

  if(present(ord))then
     order=ord
  else
     order=.true.
  end if

  if(order.eqv..false.)then
     do j=1,ny
        do i=1,nx
           val(i,j)=-val(i,j)
        end do
     end do
  end if

end subroutine curlf

!-----------------------------------------
!-----------------------------------------

subroutine curld( x, y, u, v, val, undeff, hx, hy, ord )
! 2 ٷ׻롼
!
! x, y μμ㤤ɬ¤٤뤳.
!
! u, v ϶ִθ֤, Ǥդ2٤׻ǽ
! , du/dz-dw/dx ׻ȤΤ, (x,z,u,w) ν֤, ord ץ false.
!
! $\frac{\partial v}{\partial x} -\frac{\partial u}{\partial y} $ 
! 2 ʬǽ񤭴,  $(i,j)$ Ǥȯ
! $\frac{v_{i,j+1}-v_{i,j-1}}{2dx} -\frac{u_{i+1,j}-u_{i-1,j}}{2dy} $
! ȤǤ. Ѥ2ȯ׻.
! ǡ­ʤΰǤ, 1 κʬǷ׻Τ, ٤
! .
! ¼Ūˤ grad_1d ʬ׻ôΤ, η׻⼫ưǹԤ.
  implicit none
  double precision, intent(in) :: x(:)  ! x ζֺɸ [m]
  double precision, intent(in) :: y(:)  ! y ζֺɸ [m]
  double precision, intent(in) :: u(size(x),size(y))  ! x б 2 ٥ȥʬ
  double precision, intent(in) :: v(size(x),size(y))  ! y б 2 ٥ȥʬ
  double precision, intent(inout) :: val(size(x),size(y))  ! 2
  double precision, intent(in), optional :: undeff
  double precision, intent(in), optional :: hx(size(x),size(y))  ! x Υ
  double precision, intent(in), optional :: hy(size(x),size(y))  ! y Υ
  logical, intent(in),  optional :: ord  ! ʬν֤ؤ륪ץ.
                 ! true ʤ, ؤʤ, false ʤ, ؤ.
                 ! ǥեȤ true, du/dz-dw/dx ׻ȤΤѤ.
  integer :: i   ! 졼ź
  integer :: j   ! 졼ź
  integer :: nx  ! ǿ 1 
  integer :: ny  ! ǿ 2 
  logical :: order
  double precision :: scalex(size(x),size(y)), scaley(size(x),size(y))
  double precision :: dvdx(size(x),size(y)), dudy(size(x),size(y))
  double precision :: tmpu(size(x),size(y)), tmpv(size(x),size(y))

  nx=size(x)
  ny=size(y)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "curl" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, v ),  &
  &                                     "curl" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, val ),  &
  &                                     "curl" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, hx ),  &
  &                                        "curl" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, hy ),  &
  &                                        "curl" )
     end if
  end if

!-- ѿ.

  if(present(hx))then
     do j=1,ny
        do i=1,nx
           scalex(i,j)=hx(i,j)
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           scalex(i,j)=1.0d0
        end do
     end do
  end if

  if(present(hy))then
     do j=1,ny
        do i=1,nx
           scaley(i,j)=hy(i,j)
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           scaley(i,j)=1.0d0
        end do
     end do
  end if

!-- ˤ, x  dvdy ׻, y  dudx ׻.

  if(present(undeff))then
     do j=1,ny
        do i=1,nx
           if(u(i,j)/=undeff.and.v(i,j)/=undeff.and.  &
  &           scalex(i,j)/=undeff.and.scaley(i,j)/=undeff)then
              tmpu(i,j)=scalex(i,j)*u(i,j)
              tmpv(i,j)=scaley(i,j)*v(i,j)
           else
              tmpu(i,j)=undeff
              tmpv(i,j)=undeff
           end if
        end do
     end do

     do i=1,nx
        call grad_1dd( y, tmpu(i,:), dudy(i,:), undeff )
     end do
     do j=1,ny
        call grad_1dd( x, tmpv(:,j), dvdx(:,j), undeff )
     end do

     do j=1,ny
        do i=1,nx
           if(dudy(i,j)/=undeff.and.dvdx(i,j)/=undeff.and.  &
  &           scalex(i,j)/=undeff.and.scaley(i,j)/=undeff)then
              if(scalex(i,j)/=0.0d0.and.scaley(i,j)/=0.0d0)then
                 val(i,j)=(dvdx(i,j)-dudy(i,j))/(scalex(i,j)*scaley(i,j))
              else
                 val(i,j)=0.0d0
              end if
           else
              val(i,j)=undeff
           end if
        end do
     end do

  else

     do j=1,ny
        do i=1,nx
           tmpu(i,j)=scalex(i,j)*u(i,j)
           tmpv(i,j)=scaley(i,j)*v(i,j)
        end do
     end do

     do i=1,nx
        call grad_1dd( y, tmpu(i,:), dudy(i,:) )
     end do
     do j=1,ny
        call grad_1dd( x, tmpv(:,j), dvdx(:,j) )
     end do

     do j=1,ny
        do i=1,nx
           if(scalex(i,j)/=0.0d0.and.scaley(i,j)/=0.0d0)then
              val(i,j)=(dvdx(i,j)-dudy(i,j))/(scalex(i,j)*scaley(i,j))
           else
              val(i,j)=0.0d0
           end if
        end do
     end do
  end if

!-- žν֤դˤ륪ץ
!-- false ʤ, ֤ؤƽϤ.

  if(present(ord))then
     order=ord
  else
     order=.true.
  end if

  if(order.eqv..false.)then
     do j=1,ny
        do i=1,nx
           val(i,j)=-val(i,j)
        end do
     end do
  end if

end subroutine curld

!-----------------------------------------
!-----------------------------------------

subroutine div_3df( x, y, z, u, v, w, val, undeff, hx, hy, hz )
! 3ȯ׻롼
!
! $\frac{\partial u}{\partial x} +\frac{\partial v}{\partial y} +\frac{\partial w}{\partial z} $ 
! 2 ʬǽ񤭴,  $(i,j,k)$ Ǥȯ
! $\frac{u_{i+1,j,k}-u_{i-1,j,k}}{2dx} + \frac{v_{i,j+1,k}-v_{i,j-1,k}}{2dy} + \frac{w_{i,j,k+1}-w_{i,j,k-1}}{2dz} $
! ȤǤ. Ѥ 3 ȯ׻.
! ǡ­ʤΰǤ, 1 κʬǷ׻Τ, ٤
! .
! ¼Ūˤ, grad_1d 롼󤬷׻ԤΤ, ν⼫ưǹԤ.
  implicit none
  real, intent(in) :: x(:)  ! x ζֺɸ [m]
  real, intent(in) :: y(:)  ! y ζֺɸ [m]
  real, intent(in) :: z(:)  ! z ζֺɸ [m]
  real, intent(in) :: u(size(x),size(y),size(z))  ! x б 2 ٥ȥʬ
  real, intent(in) :: v(size(x),size(y),size(z))  ! y б 2 ٥ȥʬ
  real, intent(in) :: w(size(x),size(y),size(z))  ! y б 2 ٥ȥʬ
  real, intent(inout) :: val(size(x),size(y),size(z))  ! 3 ȯ
  real, intent(in), optional :: undeff
  real, intent(in), optional :: hx(size(x),size(y),size(z))  ! x Υ
  real, intent(in), optional :: hy(size(x),size(y),size(z))  ! y Υ
  real, intent(in), optional :: hz(size(x),size(y),size(z))  ! z Υ
  integer :: i   ! 졼ź
  integer :: j   ! 졼ź
  integer :: k   ! 졼ź
  integer :: nx  ! ǿ 1 
  integer :: ny  ! ǿ 2 
  integer :: nz  ! ǿ 3 
  real, dimension(size(x),size(y),size(z)) :: scalex, scaley, scalez
  real, dimension(size(x),size(y),size(z)) :: dudx, dvdy, dwdz
  real, dimension(size(x),size(y),size(z)) :: tmpu, tmpv, tmpw
  real, allocatable, dimension(:,:) :: tmpy1, tmpy2, tmpz1, tmpz2  ! å

!-- OpenMP ؿ
!$ integer :: OMP_GET_THREAD_NUM, OMP_GET_MAX_THREADS
  integer :: ompnum, omppe

  nx=size(x)
  ny=size(y)
  nz=size(z)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, u ),  &
  &                                     "div_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, v ),  &
  &                                     "div_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, w ),  &
  &                                     "div_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, val ),  &
  &                                     "div_3d" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hx ),  &
  &                                        "div_3d" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hy ),  &
  &                                        "div_3d" )
     end if
     if(present(hz))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hz ),  &
  &                                        "div_3d" )
     end if
  end if

!-- å
  ompnum=1
  omppe=1
!$   ompnum=OMP_GET_MAX_THREADS()  ! OpenMP ͭξϤͭ.

  allocate(tmpy1(ny,ompnum))
  allocate(tmpy2(ny,ompnum))
  allocate(tmpz1(nz,ompnum))
  allocate(tmpz2(nz,ompnum))

!-- ѿ.

  if(present(hx))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=hx(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=1.0
           end do
        end do
     end do
  end if

  if(present(hy))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=hy(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=1.0
           end do
        end do
     end do
  end if

  if(present(hz))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=hz(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=1.0
           end do
        end do
     end do
  end if

!-- ˤ, x  dvdy ׻, y  dudx ׻.

  if(present(undeff))then
!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(u(i,j,k)/=undeff.and.v(i,j,k)/=undeff.and.w(i,j,k)/=undeff  &
  &              .and.scalex(i,j,k)/=undeff.and.scaley(i,j,k)/=undeff.and.  &
  &              scalez(i,j,k)/=undeff)then
                 tmpu(i,j,k)=scaley(i,j,k)*scalez(i,j,k)*u(i,j,k)
                 tmpv(i,j,k)=scalez(i,j,k)*scalex(i,j,k)*v(i,j,k)
                 tmpw(i,j,k)=scalex(i,j,k)*scaley(i,j,k)*w(i,j,k)
              else
                 tmpu(i,j,k)=undeff
                 tmpv(i,j,k)=undeff
                 tmpw(i,j,k)=undeff
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,k,omppe)
     do k=1,nz
        do i=1,nx
           !-- å
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
           tmpy1(1:ny,omppe)=tmpv(i,1:ny,k)

           call grad_1df( y, tmpy1(1:ny,omppe), tmpy2(1:ny,omppe), undeff )

           !-- å夫
           dvdy(i,1:ny,k)=tmpy2(1:ny,omppe)
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
     do k=1,nz
        do j=1,ny
           call grad_1df( x, tmpu(:,j,k), dudx(:,j,k), undeff )
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
     do j=1,ny
        do i=1,nx
           !-- å
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
           tmpz1(1:nz,omppe)=tmpw(i,j,1:nz)

           call grad_1df( z, tmpz1(1:nz,omppe), tmpz2(1:nz,omppe), undeff )

           !-- å夫
           dwdz(i,j,1:nz)=tmpz2(1:nz,omppe)
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(dudx(i,j,k)/=undeff.and.dvdy(i,j,k)/=undeff.and.  &
  &              dwdz(i,j,k)/=undeff.and.scalex(i,j,k)/=undeff.and.  &
  &              scaley(i,j,k)/=undeff.and.scalez(i,j,k)/=undeff)then
                 if(scalex(i,j,k)/=0.0.and.scaley(i,j,k)/=0.0.and.  &
  &                 scalez(i,j,k)/=0.0)then
                    val(i,j,k)=(dudx(i,j,k)+dvdy(i,j,k)+dwdz(i,j,k))/  &
  &                            (scalex(i,j,k)*scaley(i,j,k)*scalez(i,j,k))
                 else
                    val(i,j,k)=0.0
                 end if
              else
                 val(i,j,k)=undeff
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

  else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              tmpu(i,j,k)=scaley(i,j,k)*scalez(i,j,k)*u(i,j,k)
              tmpv(i,j,k)=scalez(i,j,k)*scalex(i,j,k)*v(i,j,k)
              tmpw(i,j,k)=scalex(i,j,k)*scaley(i,j,k)*w(i,j,k)
           end do
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,k,omppe)
     do k=1,nz
        do i=1,nx
           !-- å
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
           tmpy1(1:ny,omppe)=tmpv(i,1:ny,k)

           call grad_1df( y, tmpy1(1:ny,omppe), tmpy2(1:ny,omppe) )

           !-- å夫
           dvdy(i,1:ny,k)=tmpy2(1:ny,omppe)
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
     do k=1,nz
        do j=1,ny
           call grad_1df( x, tmpu(:,j,k), dudx(:,j,k) )
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
     do j=1,ny
        do i=1,nx
           !-- å
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
           tmpz1(1:nz,omppe)=tmpw(i,j,1:nz)

           call grad_1df( z, tmpz1(1:nz,omppe), tmpz2(1:nz,omppe) )

           !-- å夫
           dwdz(i,j,1:nz)=tmpz2(1:nz,omppe)
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(scalex(i,j,k)/=0.0.and.scaley(i,j,k)/=0.0.and.scalez(i,j,k)/=0.0)then
                 val(i,j,k)=(dudx(i,j,k)+dvdy(i,j,k)+dwdz(i,j,k))/  &
  &                         (scalex(i,j,k)*scaley(i,j,k)*scalez(i,j,k))
              else
                 val(i,j,k)=0.0
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

  end if

end subroutine div_3df

!-----------------------------------------
!-----------------------------------------

subroutine div_3dd( x, y, z, u, v, w, val, undeff, hx, hy, hz )
! 3ȯ׻롼
!
! $\frac{\partial u}{\partial x} +\frac{\partial v}{\partial y} +\frac{\partial w}{\partial z} $ 
! 2 ʬǽ񤭴,  $(i,j,k)$ Ǥȯ
! $\frac{u_{i+1,j,k}-u_{i-1,j,k}}{2dx} + \frac{v_{i,j+1,k}-v_{i,j-1,k}}{2dy} + \frac{w_{i,j,k+1}-w_{i,j,k-1}}{2dz} $
! ȤǤ. Ѥ 3 ȯ׻.
! ǡ­ʤΰǤ, 1 κʬǷ׻Τ, ٤
! .
! ¼Ūˤ, grad_1d 롼󤬷׻ԤΤ, ν⼫ưǹԤ.
  implicit none
  double precision, intent(in) :: x(:)  ! x ζֺɸ [m]
  double precision, intent(in) :: y(:)  ! y ζֺɸ [m]
  double precision, intent(in) :: z(:)  ! z ζֺɸ [m]
  double precision, intent(in) :: u(size(x),size(y),size(z))  ! x б 2 ٥ȥʬ
  double precision, intent(in) :: v(size(x),size(y),size(z))  ! y б 2 ٥ȥʬ
  double precision, intent(in) :: w(size(x),size(y),size(z))  ! y б 2 ٥ȥʬ
  double precision, intent(inout) :: val(size(x),size(y),size(z))  ! 3 ȯ
  double precision, intent(in), optional :: undeff
  double precision, intent(in), optional :: hx(size(x),size(y),size(z))  ! x Υ
  double precision, intent(in), optional :: hy(size(x),size(y),size(z))  ! y Υ
  double precision, intent(in), optional :: hz(size(x),size(y),size(z))  ! z Υ
  integer :: i   ! 졼ź
  integer :: j   ! 졼ź
  integer :: k   ! 졼ź
  integer :: nx  ! ǿ 1 
  integer :: ny  ! ǿ 2 
  integer :: nz  ! ǿ 3 
  double precision, dimension(size(x),size(y),size(z)) :: scalex, scaley, scalez
  double precision, dimension(size(x),size(y),size(z)) :: dudx, dvdy, dwdz
  double precision, dimension(size(x),size(y),size(z)) :: tmpu, tmpv, tmpw
  double precision, allocatable, dimension(:,:) :: tmpy1, tmpy2, tmpz1, tmpz2  ! å

!-- OpenMP ؿ
!$ integer :: OMP_GET_THREAD_NUM, OMP_GET_MAX_THREADS
  integer :: ompnum, omppe

  nx=size(x)
  ny=size(y)
  nz=size(z)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, u ),  &
  &                                     "div_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, v ),  &
  &                                     "div_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, w ),  &
  &                                     "div_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, val ),  &
  &                                     "div_3d" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hx ),  &
  &                                        "div_3d" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hy ),  &
  &                                        "div_3d" )
     end if
     if(present(hz))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hz ),  &
  &                                        "div_3d" )
     end if
  end if

!-- å
  ompnum=1
  omppe=1
!$   ompnum=OMP_GET_MAX_THREADS()  ! OpenMP ͭξϤͭ.

  allocate(tmpy1(ny,ompnum))
  allocate(tmpy2(ny,ompnum))
  allocate(tmpz1(nz,ompnum))
  allocate(tmpz2(nz,ompnum))

!-- ѿ.

  if(present(hx))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=hx(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=1.0d0
           end do
        end do
     end do
  end if

  if(present(hy))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=hy(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=1.0d0
           end do
        end do
     end do
  end if

  if(present(hz))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=hz(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=1.0d0
           end do
        end do
     end do
  end if

!-- ˤ, x  dvdy ׻, y  dudx ׻.

  if(present(undeff))then
!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(u(i,j,k)/=undeff.and.v(i,j,k)/=undeff.and.w(i,j,k)/=undeff  &
  &              .and.scalex(i,j,k)/=undeff.and.scaley(i,j,k)/=undeff.and.  &
  &              scalez(i,j,k)/=undeff)then
                 tmpu(i,j,k)=scaley(i,j,k)*scalez(i,j,k)*u(i,j,k)
                 tmpv(i,j,k)=scalez(i,j,k)*scalex(i,j,k)*v(i,j,k)
                 tmpw(i,j,k)=scalex(i,j,k)*scaley(i,j,k)*w(i,j,k)
              else
                 tmpu(i,j,k)=undeff
                 tmpv(i,j,k)=undeff
                 tmpw(i,j,k)=undeff
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,k,omppe)
     do k=1,nz
        do i=1,nx
           !-- å
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
           tmpy1(1:ny,omppe)=tmpv(i,1:ny,k)

           call grad_1dd( y, tmpy1(1:ny,omppe), tmpy2(1:ny,omppe), undeff )

           !-- å夫
           dvdy(i,1:ny,k)=tmpy2(1:ny,omppe)
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
     do k=1,nz
        do j=1,ny
           call grad_1dd( x, tmpu(:,j,k), dudx(:,j,k), undeff )
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
     do j=1,ny
        do i=1,nx
           !-- å
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
           tmpz1(1:nz,omppe)=tmpw(i,j,1:nz)

           call grad_1dd( z, tmpz1(1:nz,omppe), tmpz2(1:nz,omppe), undeff )

           !-- å夫
           dwdz(i,j,1:nz)=tmpz2(1:nz,omppe)
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(dudx(i,j,k)/=undeff.and.dvdy(i,j,k)/=undeff.and.  &
  &              dwdz(i,j,k)/=undeff.and.scalex(i,j,k)/=undeff.and.  &
  &              scaley(i,j,k)/=undeff.and.scalez(i,j,k)/=undeff)then
                 if(scalex(i,j,k)/=0.0d0.and.scaley(i,j,k)/=0.0d0.and.  &
  &                 scalez(i,j,k)/=0.0d0)then
                    val(i,j,k)=(dudx(i,j,k)+dvdy(i,j,k)+dwdz(i,j,k))/  &
  &                            (scalex(i,j,k)*scaley(i,j,k)*scalez(i,j,k))
                 else
                    val(i,j,k)=0.0d0
                 end if
              else
                 val(i,j,k)=undeff
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

  else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              tmpu(i,j,k)=scaley(i,j,k)*scalez(i,j,k)*u(i,j,k)
              tmpv(i,j,k)=scalez(i,j,k)*scalex(i,j,k)*v(i,j,k)
              tmpw(i,j,k)=scalex(i,j,k)*scaley(i,j,k)*w(i,j,k)
           end do
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,k,omppe)
     do k=1,nz
        do i=1,nx
           !-- å
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
           tmpy1(1:ny,omppe)=tmpv(i,1:ny,k)

           call grad_1dd( y, tmpy1(1:ny,omppe), tmpy2(1:ny,omppe) )

           !-- å夫
           dvdy(i,1:ny,k)=tmpy2(1:ny,omppe)
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
     do k=1,nz
        do j=1,ny
           call grad_1dd( x, tmpu(:,j,k), dudx(:,j,k) )
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
     do j=1,ny
        do i=1,nx
           !-- å
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
           tmpz1(1:nz,omppe)=tmpw(i,j,1:nz)

           call grad_1dd( z, tmpz1(1:nz,omppe), tmpz2(1:nz,omppe) )

           !-- å夫
           dwdz(i,j,1:nz)=tmpz2(1:nz,omppe)
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(scalex(i,j,k)/=0.0d0.and.scaley(i,j,k)/=0.0d0.and.scalez(i,j,k)/=0.0d0)then
                 val(i,j,k)=(dudx(i,j,k)+dvdy(i,j,k)+dwdz(i,j,k))/  &
  &                         (scalex(i,j,k)*scaley(i,j,k)*scalez(i,j,k))
              else
                 val(i,j,k)=0.0d0
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

  end if

end subroutine div_3dd

!-----------------------------------------
!-----------------------------------------

subroutine curl_3df( x, y, z, u, v, w, zeta, eta, xi, undeff, hx, hy, hz )
! 3 ٤׻.
! ν֤ϱϤ x, y, z ΥǥȺɸ,
! б٥ȥʬ u, v, w ,
! б٥٥ȥʬ zeta, eta, xi ׻.
! ¼ grad_1d ׻ôΤ, ν⼫ưǹԤ.
  implicit none
  real, intent(in) :: x(:)  ! x ζֺɸ [m]
  real, intent(in) :: y(:)  ! y ζֺɸ [m]
  real, intent(in) :: z(:)  ! z ζֺɸ [m]
  real, intent(in) :: u(size(x),size(y),size(z))  ! x б 2 ٥ȥʬ
  real, intent(in) :: v(size(x),size(y),size(z))  ! y б 2 ٥ȥʬ
  real, intent(in) :: w(size(x),size(y),size(z))  ! y б 2 ٥ȥʬ
  real, intent(inout) :: zeta(size(x),size(y),size(z))  ! ٥٥ȥ x ʬ
  real, intent(inout) :: eta(size(x),size(y),size(z))  ! ٥٥ȥ y ʬ
  real, intent(inout) :: xi(size(x),size(y),size(z))  ! ٥٥ȥ z ʬ
  real, intent(in), optional :: undeff
  real, intent(in), optional :: hx(size(x),size(y),size(z))  ! x Υ
  real, intent(in), optional :: hy(size(x),size(y),size(z))  ! y Υ
  real, intent(in), optional :: hz(size(x),size(y),size(z))  ! z Υ
  integer :: i   ! 졼ź
  integer :: j   ! 졼ź
  integer :: k   ! 졼ź
  integer :: nx  ! ǿ 1 
  integer :: ny  ! ǿ 2 
  integer :: nz  ! ǿ 3 
  real, dimension(size(x),size(y),size(z)) :: scalex, scaley, scalez
  real, allocatable, dimension(:,:,:) :: tmpyz1, tmpyz2, tmpyz3  ! å
  real, allocatable, dimension(:,:,:) :: tmpsyz1, tmpsyz2  ! å
  real, allocatable, dimension(:,:,:) :: tmpxz1, tmpxz2, tmpxz3  ! å
  real, allocatable, dimension(:,:,:) :: tmpsxz1, tmpsxz2  ! å

!-- OpenMP ؿ
!$ integer :: OMP_GET_THREAD_NUM, OMP_GET_MAX_THREADS
  integer :: ompnum, omppe

  nx=size(x)
  ny=size(y)
  nz=size(z)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, u ),  &
  &                                     "curl_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, v ),  &
  &                                     "curl_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, w ),  &
  &                                     "curl_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, xi ),  &
  &                                     "curl_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, eta ),  &
  &                                     "curl_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, zeta ),  &
  &                                     "curl_3d" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hx ),  &
  &                                        "curl_3d" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hy ),  &
  &                                        "curl_3d" )
     end if
     if(present(hz))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hz ),  &
  &                                        "curl_3d" )
     end if
  end if

!-- å
  ompnum=1
  omppe=1
!$   ompnum=OMP_GET_MAX_THREADS()  ! OpenMP ͭξϤͭ.

  allocate(tmpyz1(ny,nz,ompnum))
  allocate(tmpyz2(ny,nz,ompnum))
  allocate(tmpyz3(ny,nz,ompnum))
  allocate(tmpsyz1(ny,nz,ompnum))
  allocate(tmpsyz2(ny,nz,ompnum))
  allocate(tmpxz1(nx,nz,ompnum))
  allocate(tmpxz2(nx,nz,ompnum))
  allocate(tmpxz3(nx,nz,ompnum))
  allocate(tmpsxz1(nx,nz,ompnum))
  allocate(tmpsxz2(nx,nz,ompnum))

!-- ѿ.

  if(present(hx))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=hx(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=1.0
           end do
        end do
     end do
  end if

  if(present(hy))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=hy(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=1.0
           end do
        end do
     end do
  end if

  if(present(hz))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=hz(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=1.0
           end do
        end do
     end do
  end if

  if(present(undeff))then

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,omppe)
! x ˿ľ̾Ǥ x α٥٥ȥ 3 Ƿ׻.
     do i=1,nx
        !-- å
!$      omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
        tmpyz1(1:ny,1:nz,omppe)=v(i,1:ny,1:nz)
        tmpyz2(1:ny,1:nz,omppe)=w(i,1:ny,1:nz)
        tmpsyz1(1:ny,1:nz,omppe)=scaley(i,1:ny,1:nz)
        tmpsyz2(1:ny,1:nz,omppe)=scalez(i,1:ny,1:nz)

        !-- curl(y,z,v,w,zeta,hy,hz)
        call curlf( y(1:ny), z(1:nz), tmpyz1(1:ny,1:nz,omppe),  &
  &                tmpyz2(1:ny,1:nz,omppe), tmpyz3(1:ny,1:nz,omppe), undeff,  &
  &                hx=tmpsyz1(1:ny,1:nz,omppe), hy=tmpsyz2(1:ny,1:nz,omppe) )

        !-- å夫
        zeta(i,1:ny,1:nz)=tmpyz3(1:ny,1:nz,omppe)
     end do
!$omp end do
!$omp end parallel

  else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,omppe)
! x ˿ľ̾Ǥ x α٥٥ȥ 3 Ƿ׻.
     do i=1,nx
        !-- å
!$      omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
        tmpyz1(1:ny,1:nz,omppe)=v(i,1:ny,1:nz)
        tmpyz2(1:ny,1:nz,omppe)=w(i,1:ny,1:nz)
        tmpsyz1(1:ny,1:nz,omppe)=scaley(i,1:ny,1:nz)
        tmpsyz2(1:ny,1:nz,omppe)=scalez(i,1:ny,1:nz)

        !-- curl(y,z,v,w,zeta,hy,hz)
        call curlf( y(1:ny), z(1:nz), tmpyz1(1:ny,1:nz,omppe),  &
  &                tmpyz2(1:ny,1:nz,omppe), tmpyz3(1:ny,1:nz,omppe),  &
  &                hx=tmpsyz1(1:ny,1:nz,omppe), hy=tmpsyz2(1:ny,1:nz,omppe) )

        !-- å夫
        zeta(i,1:ny,1:nz)=tmpyz3(1:ny,1:nz,omppe)
     end do
!$omp end do
!$omp end parallel

  end if

! y ˿ľ̾Ǥ y α٥٥ȥ 3 Ƿ׻.

  if(present(undeff))then

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,omppe)
     do j=1,ny
        !-- å
!$      omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
        tmpxz1(1:nx,1:nz,omppe)=u(1:nx,j,1:nz)
        tmpxz2(1:nx,1:nz,omppe)=w(1:nx,j,1:nz)
        tmpsxz1(1:nx,1:nz,omppe)=scalex(1:nx,j,1:nz)
        tmpsxz2(1:nx,1:nz,omppe)=scalez(1:nx,j,1:nz)

        !-- curl(x,z,u,w,eta,hx,hz)
        call curlf( x(1:nx), z(1:nz), tmpxz1(1:nx,1:nz,omppe),  &
  &                tmpxz2(1:nx,1:nz,omppe), tmpxz3(1:nx,1:nz,omppe),  &
  &                undeff, ord=.false., &
  &                hx=tmpsxz1(1:nx,1:nz,omppe), hy=tmpsxz2(1:nx,1:nz,omppe) )

        !-- å夫
        eta(1:nx,j,1:nz)=tmpxz3(1:nx,1:nz,omppe)
     end do
!$omp end do
!$omp end parallel

  else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,omppe)
     do j=1,ny
        !-- å
!$      omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
        tmpxz1(1:nx,1:nz,omppe)=u(1:nx,j,1:nz)
        tmpxz2(1:nx,1:nz,omppe)=w(1:nx,j,1:nz)
        tmpsxz1(1:nx,1:nz,omppe)=scalex(1:nx,j,1:nz)
        tmpsxz2(1:nx,1:nz,omppe)=scalez(1:nx,j,1:nz)

        !-- curl(x,z,u,w,eta,hx,hz)
        call curlf( x(1:nx), z(1:nz), tmpxz1(1:nx,1:nz,omppe),  &
  &                tmpxz2(1:nx,1:nz,omppe), tmpxz3(1:nx,1:nz,omppe), ord=.false., &
  &                hx=tmpsxz1(1:nx,1:nz,omppe), hy=tmpsxz2(1:nx,1:nz,omppe) )

        !-- å夫
        eta(1:nx,j,1:nz)=tmpxz3(1:nx,1:nz,omppe)
     end do
!$omp end do
!$omp end parallel

  end if

! z ˿ľ̾Ǥ z α٥٥ȥ 3 Ƿ׻.

  if(present(undeff))then

!$omp parallel default(shared)
!$omp do schedule(runtime) private(k)
     do k=1,nz
        call curlf( x, y, u(:,:,k), v(:,:,k), xi(:,:,k), undeff,  &
  &                hx=scalex(:,:,k), hy=scaley(:,:,k) )
     end do
!$omp end do
!$omp end parallel

  else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(k)
     do k=1,nz
        call curlf( x, y, u(:,:,k), v(:,:,k), xi(:,:,k),  &
  &                hx=scalex(:,:,k), hy=scaley(:,:,k) )
     end do
!$omp end do
!$omp end parallel

  end if

end subroutine curl_3df

!-----------------------------------------
!-----------------------------------------

subroutine curl_3dd( x, y, z, u, v, w, zeta, eta, xi, undeff, hx, hy, hz )
! 3 ٤׻.
! ν֤ϱϤ x, y, z ΥǥȺɸ,
! б٥ȥʬ u, v, w ,
! б٥٥ȥʬ zeta, eta, xi ׻.
! ¼ grad_1d ׻ôΤ, ν⼫ưǹԤ.
  implicit none
  double precision, intent(in) :: x(:)  ! x ζֺɸ [m]
  double precision, intent(in) :: y(:)  ! y ζֺɸ [m]
  double precision, intent(in) :: z(:)  ! z ζֺɸ [m]
  double precision, intent(in) :: u(size(x),size(y),size(z))  ! x б 2 ٥ȥʬ
  double precision, intent(in) :: v(size(x),size(y),size(z))  ! y б 2 ٥ȥʬ
  double precision, intent(in) :: w(size(x),size(y),size(z))  ! y б 2 ٥ȥʬ
  double precision, intent(inout) :: zeta(size(x),size(y),size(z))  ! ٥٥ȥ x ʬ
  double precision, intent(inout) :: eta(size(x),size(y),size(z))  ! ٥٥ȥ y ʬ
  double precision, intent(inout) :: xi(size(x),size(y),size(z))  ! ٥٥ȥ z ʬ
  double precision, intent(in), optional :: undeff
  double precision, intent(in), optional :: hx(size(x),size(y),size(z))  ! x Υ
  double precision, intent(in), optional :: hy(size(x),size(y),size(z))  ! y Υ
  double precision, intent(in), optional :: hz(size(x),size(y),size(z))  ! z Υ
  integer :: i   ! 졼ź
  integer :: j   ! 졼ź
  integer :: k   ! 졼ź
  integer :: nx  ! ǿ 1 
  integer :: ny  ! ǿ 2 
  integer :: nz  ! ǿ 3 
  double precision, dimension(size(x),size(y),size(z)) :: scalex, scaley, scalez
  double precision, allocatable, dimension(:,:,:) :: tmpyz1, tmpyz2, tmpyz3  ! å
  double precision, allocatable, dimension(:,:,:) :: tmpsyz1, tmpsyz2  ! å
  double precision, allocatable, dimension(:,:,:) :: tmpxz1, tmpxz2, tmpxz3  ! å
  double precision, allocatable, dimension(:,:,:) :: tmpsxz1, tmpsxz2  ! å

!-- OpenMP ؿ
!$ integer :: OMP_GET_THREAD_NUM, OMP_GET_MAX_THREADS
  integer :: ompnum, omppe

  nx=size(x)
  ny=size(y)
  nz=size(z)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, u ),  &
  &                                     "curl_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, v ),  &
  &                                     "curl_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, w ),  &
  &                                     "curl_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, xi ),  &
  &                                     "curl_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, eta ),  &
  &                                     "curl_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, zeta ),  &
  &                                     "curl_3d" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hx ),  &
  &                                        "curl_3d" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hy ),  &
  &                                        "curl_3d" )
     end if
     if(present(hz))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hz ),  &
  &                                        "curl_3d" )
     end if
  end if

!-- å
  ompnum=1
  omppe=1
!$   ompnum=OMP_GET_MAX_THREADS()  ! OpenMP ͭξϤͭ.

  allocate(tmpyz1(ny,nz,ompnum))
  allocate(tmpyz2(ny,nz,ompnum))
  allocate(tmpyz3(ny,nz,ompnum))
  allocate(tmpsyz1(ny,nz,ompnum))
  allocate(tmpsyz2(ny,nz,ompnum))
  allocate(tmpxz1(nx,nz,ompnum))
  allocate(tmpxz2(nx,nz,ompnum))
  allocate(tmpxz3(nx,nz,ompnum))
  allocate(tmpsxz1(nx,nz,ompnum))
  allocate(tmpsxz2(nx,nz,ompnum))

!-- ѿ.

  if(present(hx))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=hx(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=1.0d0
           end do
        end do
     end do
  end if

  if(present(hy))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=hy(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=1.0d0
           end do
        end do
     end do
  end if

  if(present(hz))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=hz(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=1.0d0
           end do
        end do
     end do
  end if

  if(present(undeff))then

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,omppe)
! x ˿ľ̾Ǥ x α٥٥ȥ 3 Ƿ׻.
     do i=1,nx
        !-- å
!$      omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
        tmpyz1(1:ny,1:nz,omppe)=v(i,1:ny,1:nz)
        tmpyz2(1:ny,1:nz,omppe)=w(i,1:ny,1:nz)
        tmpsyz1(1:ny,1:nz,omppe)=scaley(i,1:ny,1:nz)
        tmpsyz2(1:ny,1:nz,omppe)=scalez(i,1:ny,1:nz)

        !-- curl(y,z,v,w,zeta,hy,hz)
        call curld( y(1:ny), z(1:nz), tmpyz1(1:ny,1:nz,omppe),  &
  &                 tmpyz2(1:ny,1:nz,omppe), tmpyz3(1:ny,1:nz,omppe), undeff,  &
  &                 hx=tmpsyz1(1:ny,1:nz,omppe), hy=tmpsyz2(1:ny,1:nz,omppe) )

        !-- å夫
        zeta(i,1:ny,1:nz)=tmpyz3(1:ny,1:nz,omppe)
     end do
!$omp end do
!$omp end parallel

  else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,omppe)
! x ˿ľ̾Ǥ x α٥٥ȥ 3 Ƿ׻.
     do i=1,nx
        !-- å
!$      omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
        tmpyz1(1:ny,1:nz,omppe)=v(i,1:ny,1:nz)
        tmpyz2(1:ny,1:nz,omppe)=w(i,1:ny,1:nz)
        tmpsyz1(1:ny,1:nz,omppe)=scaley(i,1:ny,1:nz)
        tmpsyz2(1:ny,1:nz,omppe)=scalez(i,1:ny,1:nz)

        !-- curl(y,z,v,w,zeta,hy,hz)
        call curld( y(1:ny), z(1:nz), tmpyz1(1:ny,1:nz,omppe),  &
  &                 tmpyz2(1:ny,1:nz,omppe), tmpyz3(1:ny,1:nz,omppe),  &
  &                 hx=tmpsyz1(1:ny,1:nz,omppe), hy=tmpsyz2(1:ny,1:nz,omppe) )

        !-- å夫
        zeta(i,1:ny,1:nz)=tmpyz3(1:ny,1:nz,omppe)
     end do
!$omp end do
!$omp end parallel

  end if

! y ˿ľ̾Ǥ y α٥٥ȥ 3 Ƿ׻.

  if(present(undeff))then

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,omppe)
     do j=1,ny
        !-- å
!$      omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
        tmpxz1(1:nx,1:nz,omppe)=u(1:nx,j,1:nz)
        tmpxz2(1:nx,1:nz,omppe)=w(1:nx,j,1:nz)
        tmpsxz1(1:nx,1:nz,omppe)=scalex(1:nx,j,1:nz)
        tmpsxz2(1:nx,1:nz,omppe)=scalez(1:nx,j,1:nz)

        !-- curl(x,z,u,w,eta,hx,hz)
        call curld( x(1:nx), z(1:nz), tmpxz1(1:nx,1:nz,omppe),  &
  &                 tmpxz2(1:nx,1:nz,omppe), tmpxz3(1:nx,1:nz,omppe),  &
  &                 undeff, ord=.false., &
  &                 hx=tmpsxz1(1:nx,1:nz,omppe), hy=tmpsxz2(1:nx,1:nz,omppe) )

        !-- å夫
        eta(1:nx,j,1:nz)=tmpxz3(1:nx,1:nz,omppe)
     end do
!$omp end do
!$omp end parallel

  else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,omppe)
     do j=1,ny
        !-- å
!$      omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
        tmpxz1(1:nx,1:nz,omppe)=u(1:nx,j,1:nz)
        tmpxz2(1:nx,1:nz,omppe)=w(1:nx,j,1:nz)
        tmpsxz1(1:nx,1:nz,omppe)=scalex(1:nx,j,1:nz)
        tmpsxz2(1:nx,1:nz,omppe)=scalez(1:nx,j,1:nz)

        !-- curl(x,z,u,w,eta,hx,hz)
        call curld( x(1:nx), z(1:nz), tmpxz1(1:nx,1:nz,omppe),  &
  &                 tmpxz2(1:nx,1:nz,omppe), tmpxz3(1:nx,1:nz,omppe), ord=.false., &
  &                 hx=tmpsxz1(1:nx,1:nz,omppe), hy=tmpsxz2(1:nx,1:nz,omppe) )

        !-- å夫
        eta(1:nx,j,1:nz)=tmpxz3(1:nx,1:nz,omppe)
     end do
!$omp end do
!$omp end parallel

  end if

! z ˿ľ̾Ǥ z α٥٥ȥ 3 Ƿ׻.

  if(present(undeff))then

!$omp parallel default(shared)
!$omp do schedule(runtime) private(k)
     do k=1,nz
        call curld( x, y, u(:,:,k), v(:,:,k), xi(:,:,k), undeff,  &
  &                 hx=scalex(:,:,k), hy=scaley(:,:,k) )
     end do
!$omp end do
!$omp end parallel

  else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(k)
     do k=1,nz
        call curld( x, y, u(:,:,k), v(:,:,k), xi(:,:,k),  &
  &                 hx=scalex(:,:,k), hy=scaley(:,:,k) )
     end do
!$omp end do
!$omp end parallel

  end if

end subroutine curl_3dd

!-----------------------------------------
!-----------------------------------------

subroutine grad_1df( x, u, val, undef, hx )
! 1 Υ顼ѿθۤ׻
! $\frac{\partial p}{\partial x} $ 
! 2 ʬǽ񤭴,  $(i)$ Ǥθۤ
! $\frac{p_{i+1}-p_{i-1}}{2dx} $
! ȤǤ. Ѥ 1 ۤ׻.
! ǡ­ʤΰǤ, 1 κʬǷ׻Τ, ٤
! .
  implicit none
  real, intent(in) :: x(:)  ! ֺɸ
  real, intent(in) :: u(size(x))  ! ζб 1 顼
  real, intent(inout) :: val(size(x))  ! 顼ͤ x θ
  real, intent(in), optional :: undef
  real, intent(in), optional :: hx(size(x))  ! x Υ
  integer :: i  ! 졼ź
  integer :: nx  ! ǿ
  real :: scalex(size(x))

  nx=size(x)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( nx, u ),  &
  &                                     "grad_1d" )
     call check_array_size_dmp_message( check_array_size_1d( nx, val ),  &
  &                                     "grad_1d" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_1d( nx, hx ),  &
  &                                        "grad_1d" )
     end if
  end if

  if(present(hx))then
     do i=1,nx
        scalex(i)=hx(i)
     end do
  else
     do i=1,nx
        scalex(i)=1.0
     end do
  end if

  if(present(undef))then
     do i=2,nx-1
        if(u(i+1)==undef.or.u(i-1)==undef.or.scalex(i)==undef)then
           val(i)=undef
        else
           if(scalex(i)/=0.0)then
              val(i)=(u(i+1)-u(i-1))/(scalex(i)*(x(i+1)-x(i-1)))
           else
              val(i)=0.0
           end if
        end if
     end do
!-- ǡΤʤξüν ---
     if(u(1)==undef.or.u(2)==undef.or.scalex(1)==undef)then
        val(1)=undef
     else
        if(scalex(1)/=0.0)then
           val(1)=(u(2)-u(1))/(scalex(1)*(x(2)-x(1)))
        else
           val(1)=0.0
        end if
     end if
     if(u(nx)==undef.or.u(nx-1)==undef.or.scalex(nx)==undef)then
        val(nx)=undef
     else
        if(scalex(nx)/=0.0)then
           val(nx)=(u(nx)-u(nx-1))/(scalex(nx)*(x(nx)-x(nx-1)))
        else
           val(nx)=0.0
        end if
     end if
  else
     do i=2,nx-1
        if(scalex(i)/=0.0)then
           val(i)=(u(i+1)-u(i-1))/(scalex(i)*(x(i+1)-x(i-1)))
        else
           val(i)=0.0
        end if
     end do
!-- ǡΤʤξüν ---
     if(scalex(1)/=0.0)then
        val(1)=(u(2)-u(1))/(scalex(1)*(x(2)-x(1)))
     else
        val(1)=0.0
     end if
     if(scalex(nx)/=0.0)then
        val(nx)=(u(nx)-u(nx-1))/(scalex(nx)*(x(nx)-x(nx-1)))
     else
        val(nx)=0.0
     end if
  end if

end subroutine grad_1df

!-----------------------------------------
!-----------------------------------------

subroutine grad_1dd( x, u, val, undef, hx )
! 1 Υ顼ѿθۤ׻
! $\frac{\partial p}{\partial x} $ 
! 2 ʬǽ񤭴,  $(i)$ Ǥθۤ
! $\frac{p_{i+1}-p_{i-1}}{2dx} $
! ȤǤ. Ѥ 1 ۤ׻.
! ǡ­ʤΰǤ, 1 κʬǷ׻Τ, ٤
! .
  implicit none
  double precision, intent(in) :: x(:)  ! ֺɸ
  double precision, intent(in) :: u(size(x))  ! ζб 1 顼
  double precision, intent(inout) :: val(size(x))  ! 顼ͤ x θ
  double precision, intent(in), optional :: undef
  double precision, intent(in), optional :: hx(size(x))  ! x Υ
  integer :: i  ! 졼ź
  integer :: nx  ! ǿ
  double precision :: scalex(size(x))

  nx=size(x)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( nx, u ),  &
  &                                     "grad_1d" )
     call check_array_size_dmp_message( check_array_size_1d( nx, val ),  &
  &                                     "grad_1d" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_1d( nx, hx ),  &
  &                                        "grad_1d" )
     end if
  end if

  if(present(hx))then
     do i=1,nx
        scalex(i)=hx(i)
     end do
  else
     do i=1,nx
        scalex(i)=1.0d0
     end do
  end if

  if(present(undef))then
     do i=2,nx-1
        if(u(i+1)==undef.or.u(i-1)==undef.or.scalex(i)==undef)then
           val(i)=undef
        else
           if(scalex(i)/=0.0d0)then
              val(i)=(u(i+1)-u(i-1))/(scalex(i)*(x(i+1)-x(i-1)))
           else
              val(i)=0.0d0
           end if
        end if
     end do
!-- ǡΤʤξüν ---
     if(u(1)==undef.or.u(2)==undef.or.scalex(1)==undef)then
        val(1)=undef
     else
        if(scalex(1)/=0.0d0)then
           val(1)=(u(2)-u(1))/(scalex(1)*(x(2)-x(1)))
        else
           val(1)=0.0d0
        end if
     end if
     if(u(nx)==undef.or.u(nx-1)==undef.or.scalex(nx)==undef)then
        val(nx)=undef
     else
        if(scalex(nx)/=0.0d0)then
           val(nx)=(u(nx)-u(nx-1))/(scalex(nx)*(x(nx)-x(nx-1)))
        else
           val(nx)=0.0d0
        end if
     end if
  else
     do i=2,nx-1
        if(scalex(i)/=0.0d0)then
           val(i)=(u(i+1)-u(i-1))/(scalex(i)*(x(i+1)-x(i-1)))
        else
           val(i)=0.0d0
        end if
     end do
!-- ǡΤʤξüν ---
     if(scalex(1)/=0.0d0)then
        val(1)=(u(2)-u(1))/(scalex(1)*(x(2)-x(1)))
     else
        val(1)=0.0d0
     end if
     if(scalex(nx)/=0.0d0)then
        val(nx)=(u(nx)-u(nx-1))/(scalex(nx)*(x(nx)-x(nx-1)))
     else
        val(nx)=0.0d0
     end if
  end if

end subroutine grad_1dd

!-----------------------------------------
!-----------------------------------------

subroutine grad_2df( x, y, u, valx, valy, undeff, hx, hy )
  ! 1 顼ۤΥ롼Ѥ 2 ۤΥ٥ȥ׻
  ! $\nabla _hp =\left(\frac{\partial p}{\partial x} ,\; \frac{\partial p}{\partial y} \right) $ 
  ! 2 ʬǽ񤭴,  $(i,j)$ Ǥθۤ
  ! $\left(\frac{p_{i+1,j}-p_{i-1,j}}{2dx} ,\; \frac{p_{i,j+1}-p_{i,j-1}}{2dy} \right) $
  ! ȤǤ. Ѥ2ۤ׻.
  ! ǡ­ʤΰǤ, 1 κʬǷ׻Τ, ٤
  ! .
  ! Ѥ 1 ۷׻롼 2 ƤӽФȤˤƤ.
  implicit none
  real, intent(in) :: x(:)  ! x κɸѿ [m]
  real, intent(in) :: y(:)  ! y κɸѿ [m]
  real, intent(in) :: u(size(x),size(y))  ! ۤȤ 2 顼ʬ
  real, intent(inout) :: valx(size(x),size(y))  ! ׻줿 y  2 ۥ٥ȥ
  real, intent(inout) :: valy(size(x),size(y))  ! ׻줿 y  2 ۥ٥ȥ
  real, intent(in), optional :: undeff
  real, intent(in), optional :: hx(size(x),size(y))  ! x Υ
  real, intent(in), optional :: hy(size(x),size(y))  ! y Υ
  integer :: i   ! 졼ź
  integer :: j   ! 졼ź
  integer :: nx  ! x ǿ(size ؿǼưŪ˷׻)
  integer :: ny  ! y ǿ(size ؿǼưŪ˷׻)
  real :: scalex(size(x),size(y)), scaley(size(x),size(y))

  nx=size(x)
  ny=size(y)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "grad_2d" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, valx ),  &
  &                                     "grad_2d" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, valy ),  &
  &                                     "grad_2d" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, hx ),  &
  &                                        "grad_2d" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, hy ),  &
  &                                        "grad_2d" )
     end if
  end if

  if(present(hx))then
     do j=1,ny
        do i=1,nx
           scalex(i,j)=hx(i,j)
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           scalex(i,j)=1.0
        end do
     end do
  end if

  if(present(hy))then
     do j=1,ny
        do i=1,nx
           scaley(i,j)=hy(i,j)
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           scaley(i,j)=1.0
        end do
     end do
  end if

  if(present(undeff))then
     do i=1,ny
        call grad_1df(x, u(:,i), valx(:,i), undeff)
     end do

     do i=1,nx
        call grad_1df(y, u(i,:), valy(i,:), undeff)
     end do

     do j=1,ny
        do i=1,nx
           if(u(i,j)/=undeff.and.scalex(i,j)/=undeff.and.scaley(i,j)/=undeff)then
              if(scalex(i,j)/=0.0)then
                 valx(i,j)=valx(i,j)/scalex(i,j)
              else
                 valx(i,j)=0.0
              end if
              if(scaley(i,j)/=0.0)then
                 valy(i,j)=valy(i,j)/scaley(i,j)
              else
                 valy(i,j)=0.0
              end if
!-- , else ʤΤ, grad_1d 롼ǤǤ undeff äƤ뤫ƱȤ
!-- ʤΤ, 䰦.
           end if
        end do
     end do

  else

     do i=1,ny
        call grad_1df(x, u(:,i), valx(:,i) )
     end do

     do i=1,nx
        call grad_1df(y, u(i,:), valy(i,:) )
     end do

     do j=1,ny
        do i=1,nx
           if(scalex(i,j)/=0.0)then
              valx(i,j)=valx(i,j)/scalex(i,j)
           else
              valx(i,j)=0.0
           end if
           if(scaley(i,j)/=0.0)then
              valy(i,j)=valy(i,j)/scaley(i,j)
           else
              valy(i,j)=0.0
           end if
        end do
     end do

  end if

end subroutine grad_2df

!-----------------------------------------
!-----------------------------------------

subroutine grad_2dd( x, y, u, valx, valy, undeff, hx, hy )
  ! 1 顼ۤΥ롼Ѥ 2 ۤΥ٥ȥ׻
  ! $\nabla _hp =\left(\frac{\partial p}{\partial x} ,\; \frac{\partial p}{\partial y} \right) $ 
  ! 2 ʬǽ񤭴,  $(i,j)$ Ǥθۤ
  ! $\left(\frac{p_{i+1,j}-p_{i-1,j}}{2dx} ,\; \frac{p_{i,j+1}-p_{i,j-1}}{2dy} \right) $
  ! ȤǤ. Ѥ2ۤ׻.
  ! ǡ­ʤΰǤ, 1 κʬǷ׻Τ, ٤
  ! .
  ! Ѥ 1 ۷׻롼 2 ƤӽФȤˤƤ.
  implicit none
  double precision, intent(in) :: x(:)  ! x κɸѿ [m]
  double precision, intent(in) :: y(:)  ! y κɸѿ [m]
  double precision, intent(in) :: u(size(x),size(y))  ! ۤȤ 2 顼ʬ
  double precision, intent(inout) :: valx(size(x),size(y))  ! ׻줿 y  2 ۥ٥ȥ
  double precision, intent(inout) :: valy(size(x),size(y))  ! ׻줿 y  2 ۥ٥ȥ
  double precision, intent(in), optional :: undeff
  double precision, intent(in), optional :: hx(size(x),size(y))  ! x Υ
  double precision, intent(in), optional :: hy(size(x),size(y))  ! y Υ
  integer :: i   ! 졼ź
  integer :: j   ! 졼ź
  integer :: nx  ! x ǿ(size ؿǼưŪ˷׻)
  integer :: ny  ! y ǿ(size ؿǼưŪ˷׻)
  double precision :: scalex(size(x),size(y)), scaley(size(x),size(y))

  nx=size(x)
  ny=size(y)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "grad_2d" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, valx ),  &
  &                                     "grad_2d" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, valy ),  &
  &                                     "grad_2d" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, hx ),  &
  &                                        "grad_2d" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, hy ),  &
  &                                        "grad_2d" )
     end if
  end if

  if(present(hx))then
     do j=1,ny
        do i=1,nx
           scalex(i,j)=hx(i,j)
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           scalex(i,j)=1.0d0
        end do
     end do
  end if

  if(present(hy))then
     do j=1,ny
        do i=1,nx
           scaley(i,j)=hy(i,j)
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           scaley(i,j)=1.0d0
        end do
     end do
  end if

  if(present(undeff))then
     do i=1,ny
        call grad_1dd(x, u(:,i), valx(:,i), undeff)
     end do

     do i=1,nx
        call grad_1dd(y, u(i,:), valy(i,:), undeff)
     end do

     do j=1,ny
        do i=1,nx
           if(u(i,j)/=undeff.and.scalex(i,j)/=undeff.and.scaley(i,j)/=undeff)then
              if(scalex(i,j)/=0.0d0)then
                 valx(i,j)=valx(i,j)/scalex(i,j)
              else
                 valx(i,j)=0.0d0
              end if
              if(scaley(i,j)/=0.0)then
                 valy(i,j)=valy(i,j)/scaley(i,j)
              else
                 valy(i,j)=0.0d0
              end if
!-- , else ʤΤ, grad_1d 롼ǤǤ undeff äƤ뤫ƱȤ
!-- ʤΤ, 䰦.
           end if
        end do
     end do

  else

     do i=1,ny
        call grad_1dd(x, u(:,i), valx(:,i) )
     end do

     do i=1,nx
        call grad_1dd(y, u(i,:), valy(i,:) )
     end do

     do j=1,ny
        do i=1,nx
           if(scalex(i,j)/=0.0d0)then
              valx(i,j)=valx(i,j)/scalex(i,j)
           else
              valx(i,j)=0.0d0
           end if
           if(scaley(i,j)/=0.0d0)then
              valy(i,j)=valy(i,j)/scaley(i,j)
           else
              valy(i,j)=0.0d0
           end if
        end do
     end do

  end if

end subroutine grad_2dd

!-----------------------------------------
!-----------------------------------------

subroutine grad_3df( x, y, z, u, valx, valy, valz, undeff, hx, hy, hz )
  ! 1 顼ۤΥ롼Ѥ 3 ۤΥ٥ȥ׻.
  ! $\nabla p =\left(\frac{\partial p}{\partial x} ,\; \frac{\partial p}{\partial y} ,\; \frac{\partial p}{\partial z} \right) $ 
  ! 2 ʬǽ񤭴,  $(i,j,k)$ Ǥθۤ
  ! $\left(\frac{p_{i+1,j,k}-p_{i-1,j,k}}{2dx} ,\; \frac{p_{i,j+1,k}-p_{i,j-1,k}}{2dy} ,\; \frac{p_{i,j,k+1}-p_{i,j,k-1}}{2dz} \right) $
  ! ȤǤ. Ѥ 3 ۤ׻.
  ! ǡ­ʤΰǤ, 1 κʬǷ׻Τ, ٤
  ! .
  ! Ѥ 1 ۷׻롼 3 ƤӽФȤˤƤ.
  implicit none
  real, intent(in) :: x(:)  ! x κɸѿ [m]
  real, intent(in) :: y(:)  ! y κɸѿ [m]
  real, intent(in) :: z(:)  ! z κɸѿ [m]
  real, intent(in) :: u(size(x),size(y),size(z))  ! ۤȤ 2 顼ʬ
  real, intent(inout) :: valx(size(x),size(y),size(z))  ! ׻줿 y  2 ۥ٥ȥ
  real, intent(inout) :: valy(size(x),size(y),size(z))  ! ׻줿 y  2 ۥ٥ȥ
  real, intent(inout) :: valz(size(x),size(y),size(z))  ! ׻줿 z  2 ۥ٥ȥ
  real, intent(in), optional :: undeff
  real, intent(in), optional :: hx(size(x),size(y),size(z))  ! x Υ
  real, intent(in), optional :: hy(size(x),size(y),size(z))  ! y Υ
  real, intent(in), optional :: hz(size(x),size(y),size(z))  ! z Υ
  integer :: i   ! 졼ź
  integer :: j   ! 졼ź
  integer :: k   ! 졼ź
  integer :: nx  ! ǿ 1 
  integer :: ny  ! ǿ 2 
  integer :: nz  ! ǿ 3 
  real, allocatable, dimension(:,:) :: tmpy1, tmpy2, tmpz1, tmpz2  ! å
  real, dimension(size(x),size(y),size(z)) :: scalex, scaley, scalez

!-- OpenMP ؿ
!$ integer :: OMP_GET_THREAD_NUM, OMP_GET_MAX_THREADS
  integer :: ompnum, omppe

  nx=size(x)
  ny=size(y)
  nz=size(z)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, u ),  &
  &                                     "grad_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, valx ),  &
  &                                     "grad_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, valy ),  &
  &                                     "grad_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, valz ),  &
  &                                     "grad_3d" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hx ),  &
  &                                        "grad_3d" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hy ),  &
  &                                        "grad_3d" )
     end if
     if(present(hz))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hz ),  &
  &                                        "grad_3d" )
     end if
  end if

!-- å
  ompnum=1
  omppe=1
!$   ompnum=OMP_GET_MAX_THREADS()  ! OpenMP ͭξϤͭ.

  allocate(tmpy1(ny,ompnum))
  allocate(tmpy2(ny,ompnum))
  allocate(tmpz1(nz,ompnum))
  allocate(tmpz2(nz,ompnum))

  if(present(hx))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=hx(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=1.0
           end do
        end do
     end do
  end if

  if(present(hy))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=hy(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=1.0
           end do
        end do
     end do
  end if

  if(present(hz))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=hz(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=1.0
           end do
        end do
     end do
  end if

  if(present(undeff))then
!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)
     do i=1,nz
        do j=1,ny
           call grad_1df( x, u(:,j,i), valx(:,j,i), undeff )
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
     do i=1,nz
        do j=1,nx
           !-- å
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
           tmpy1(1:ny,omppe)=u(j,1:ny,i)

           !-- du/dy => valy
           call grad_1df( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe), undeff )

           !-- å夫
           valy(j,1:ny,i)=tmpy2(1:ny,omppe)
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
     do i=1,ny
        do j=1,nx
           !-- å
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
           tmpz1(1:nz,omppe)=u(j,i,1:nz)

           !-- du/dz => valz
           call grad_1df( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe), undeff )

           !-- å夫
           valz(j,i,1:nz)=tmpz2(1:nz,omppe)
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(u(i,j,k)/=undeff.and.scalex(i,j,k)/=undeff.and.  &
  &              scaley(i,j,k)/=undeff.and.scalez(i,j,k)/=undeff)then
                 if(scalex(i,j,k)/=0.0)then
                    valx(i,j,k)=valx(i,j,k)/scalex(i,j,k)
                 else
                    valx(i,j,k)=0.0
                 end if
                 if(scaley(i,j,k)/=0.0)then
                    valy(i,j,k)=valy(i,j,k)/scaley(i,j,k)
                 else
                    valy(i,j,k)=0.0
                 end if
                 if(scalez(i,j,k)/=0.0)then
                    valz(i,j,k)=valz(i,j,k)/scalez(i,j,k)
                 else
                    valz(i,j,k)=0.0
                 end if
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

  else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)
     do i=1,nz
        do j=1,ny
           call grad_1df(x, u(:,j,i), valx(:,j,i))
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
     do i=1,nz
        do j=1,nx
           !-- å
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
           tmpy1(1:ny,omppe)=u(j,1:ny,i)

           !-- du/dy => valy
           call grad_1df( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe) )

           !-- å夫
           valy(j,1:ny,i)=tmpy2(1:ny,omppe)
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
     do i=1,ny
        do j=1,nx
           !-- å
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
           tmpz1(1:nz,omppe)=u(j,i,1:nz)

           !-- du/dz => valz
           call grad_1df( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe) )

           !-- å夫
           valz(j,i,1:nz)=tmpz2(1:nz,omppe)
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(scalex(i,j,k)/=0.0)then
                 valx(i,j,k)=valx(i,j,k)/scalex(i,j,k)
              else
                 valx(i,j,k)=0.0
              end if
              if(scaley(i,j,k)/=0.0)then
                 valy(i,j,k)=valy(i,j,k)/scaley(i,j,k)
              else
                 valy(i,j,k)=0.0
              end if
              if(scalez(i,j,k)/=0.0)then
                 valz(i,j,k)=valz(i,j,k)/scalez(i,j,k)
              else
                 valz(i,j,k)=0.0
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

  end if

end subroutine grad_3df

!-----------------------------------------
!-----------------------------------------

subroutine grad_3dd( x, y, z, u, valx, valy, valz, undeff, hx, hy, hz )
  ! 1 顼ۤΥ롼Ѥ 3 ۤΥ٥ȥ׻.
  ! $\nabla p =\left(\frac{\partial p}{\partial x} ,\; \frac{\partial p}{\partial y} ,\; \frac{\partial p}{\partial z} \right) $ 
  ! 2 ʬǽ񤭴,  $(i,j,k)$ Ǥθۤ
  ! $\left(\frac{p_{i+1,j,k}-p_{i-1,j,k}}{2dx} ,\; \frac{p_{i,j+1,k}-p_{i,j-1,k}}{2dy} ,\; \frac{p_{i,j,k+1}-p_{i,j,k-1}}{2dz} \right) $
  ! ȤǤ. Ѥ 3 ۤ׻.
  ! ǡ­ʤΰǤ, 1 κʬǷ׻Τ, ٤
  ! .
  ! Ѥ 1 ۷׻롼 3 ƤӽФȤˤƤ.
  implicit none
  double precision, intent(in) :: x(:)  ! x κɸѿ [m]
  double precision, intent(in) :: y(:)  ! y κɸѿ [m]
  double precision, intent(in) :: z(:)  ! z κɸѿ [m]
  double precision, intent(in) :: u(size(x),size(y),size(z))  ! ۤȤ 2 顼ʬ
  double precision, intent(inout) :: valx(size(x),size(y),size(z))  ! ׻줿 y  2 ۥ٥ȥ
  double precision, intent(inout) :: valy(size(x),size(y),size(z))  ! ׻줿 y  2 ۥ٥ȥ
  double precision, intent(inout) :: valz(size(x),size(y),size(z))  ! ׻줿 z  2 ۥ٥ȥ
  double precision, intent(in), optional :: undeff
  double precision, intent(in), optional :: hx(size(x),size(y),size(z))  ! x Υ
  double precision, intent(in), optional :: hy(size(x),size(y),size(z))  ! y Υ
  double precision, intent(in), optional :: hz(size(x),size(y),size(z))  ! z Υ
  integer :: i   ! 졼ź
  integer :: j   ! 졼ź
  integer :: k   ! 졼ź
  integer :: nx  ! ǿ 1 
  integer :: ny  ! ǿ 2 
  integer :: nz  ! ǿ 3 
  double precision, allocatable, dimension(:,:) :: tmpy1, tmpy2, tmpz1, tmpz2  ! å
  double precision, dimension(size(x),size(y),size(z)) :: scalex, scaley, scalez

!-- OpenMP ؿ
!$ integer :: OMP_GET_THREAD_NUM, OMP_GET_MAX_THREADS
  integer :: ompnum, omppe

  nx=size(x)
  ny=size(y)
  nz=size(z)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, u ),  &
  &                                     "grad_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, valx ),  &
  &                                     "grad_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, valy ),  &
  &                                     "grad_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, valz ),  &
  &                                     "grad_3d" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hx ),  &
  &                                        "grad_3d" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hy ),  &
  &                                        "grad_3d" )
     end if
     if(present(hz))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hz ),  &
  &                                        "grad_3d" )
     end if
  end if

!-- å
  ompnum=1
  omppe=1
!$   ompnum=OMP_GET_MAX_THREADS()  ! OpenMP ͭξϤͭ.

  allocate(tmpy1(ny,ompnum))
  allocate(tmpy2(ny,ompnum))
  allocate(tmpz1(nz,ompnum))
  allocate(tmpz2(nz,ompnum))

  if(present(hx))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=hx(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=1.0d0
           end do
        end do
     end do
  end if

  if(present(hy))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=hy(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=1.0d0
           end do
        end do
     end do
  end if

  if(present(hz))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=hz(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=1.0d0
           end do
        end do
     end do
  end if

  if(present(undeff))then
!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)
     do i=1,nz
        do j=1,ny
           call grad_1dd( x, u(:,j,i), valx(:,j,i), undeff )
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
     do i=1,nz
        do j=1,nx
           !-- å
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
           tmpy1(1:ny,omppe)=u(j,1:ny,i)

           !-- du/dy => valy
           call grad_1dd( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe), undeff )

           !-- å夫
           valy(j,1:ny,i)=tmpy2(1:ny,omppe)
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
     do i=1,ny
        do j=1,nx
           !-- å
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
           tmpz1(1:nz,omppe)=u(j,i,1:nz)

           !-- du/dz => valz
           call grad_1dd( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe), undeff )

           !-- å夫
           valz(j,i,1:nz)=tmpz2(1:nz,omppe)
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(u(i,j,k)/=undeff.and.scalex(i,j,k)/=undeff.and.  &
  &              scaley(i,j,k)/=undeff.and.scalez(i,j,k)/=undeff)then
                 if(scalex(i,j,k)/=0.0d0)then
                    valx(i,j,k)=valx(i,j,k)/scalex(i,j,k)
                 else
                    valx(i,j,k)=0.0d0
                 end if
                 if(scaley(i,j,k)/=0.0d0)then
                    valy(i,j,k)=valy(i,j,k)/scaley(i,j,k)
                 else
                    valy(i,j,k)=0.0d0
                 end if
                 if(scalez(i,j,k)/=0.0d0)then
                    valz(i,j,k)=valz(i,j,k)/scalez(i,j,k)
                 else
                    valz(i,j,k)=0.0d0
                 end if
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

  else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j)
     do i=1,nz
        do j=1,ny
           call grad_1dd(x, u(:,j,i), valx(:,j,i))
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
     do i=1,nz
        do j=1,nx
           !-- å
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
           tmpy1(1:ny,omppe)=u(j,1:ny,i)

           !-- du/dy => valy
           call grad_1dd( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe) )

           !-- å夫
           valy(j,1:ny,i)=tmpy2(1:ny,omppe)
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
     do i=1,ny
        do j=1,nx
           !-- å
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
           tmpz1(1:nz,omppe)=u(j,i,1:nz)

           !-- du/dz => valz
           call grad_1dd( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe) )

           !-- å夫
           valz(j,i,1:nz)=tmpz2(1:nz,omppe)
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(scalex(i,j,k)/=0.0d0)then
                 valx(i,j,k)=valx(i,j,k)/scalex(i,j,k)
              else
                 valx(i,j,k)=0.0d0
              end if
              if(scaley(i,j,k)/=0.0d0)then
                 valy(i,j,k)=valy(i,j,k)/scaley(i,j,k)
              else
                 valy(i,j,k)=0.0d0
              end if
              if(scalez(i,j,k)/=0.0d0)then
                 valz(i,j,k)=valz(i,j,k)/scalez(i,j,k)
              else
                 valz(i,j,k)=0.0d0
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

  end if

end subroutine grad_3dd

!-----------------------------------------
!-----------------------------------------

subroutine grad4_1df( x, u, val, undef, hx )
! 1 Υ顼ѿθۤ׻
! $\frac{\partial p}{\partial x} $ 
! 4 ʬǽ񤭴,  $(i)$ Ǥθۤ
! $(2/3)*\frac{p_{i+1}-p_{i-1}}{dx} -(p_{i+2}-p_{i-2})/(12dx)$
! ȤǤ. Ѥ 1 ۤ׻.
! ǡ­ʤΰǤ, 1 κʬǷ׻Τ, ٤
! .
  implicit none
  real, intent(in) :: x(:)  ! ֺɸ
  real, intent(in) :: u(size(x))  ! ζб 1 顼
  real, intent(inout) :: val(size(x))  ! 顼ͤ x θ
  real, intent(in), optional :: undef
  real, intent(in), optional :: hx(size(x))  ! x Υ
  integer :: i  ! 졼ź
  integer :: nx  ! ǿ
  real :: scalex(size(x)), dx(size(x)), coe23, coe112

  nx=size(x)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( nx, u ),  &
  &                                     "grad4_1d" )
     call check_array_size_dmp_message( check_array_size_1d( nx, val ),  &
  &                                     "grad4_1d" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_1d( nx, hx ),  &
  &                                        "grad4_1d" )
     end if
  end if

  do i=2,nx-1
     dx(i)=0.5*(x(i+1)-x(i-1))
  end do
  dx(1)=x(2)-x(1)
  dx(nx)=x(nx)-x(nx-1)
  coe23=2.0/3.0
  coe112=1.0/12.0

  if(present(hx))then
     do i=1,nx
        scalex(i)=hx(i)
     end do
  else
     do i=1,nx
        scalex(i)=1.0
     end do
  end if

  if(present(undef))then
     do i=3,nx-2
        if(u(i+1)==undef.or.u(i-1)==undef.or.  &
  &        u(i+2)==undef.or.u(i-2)==undef.or.scalex(i)==undef)then
           val(i)=undef
        else
           if(scalex(i)/=0.0)then
              val(i)=(coe23*(u(i+1)-u(i-1))-coe112*(u(i+2)-u(i-2)))  &
  &                  /(scalex(i)*dx(i))
           else
              val(i)=0.0
           end if
        end if
     end do

!-- ǡΤʤξüν (ξü 1 ¦) ---
     if(u(1)==undef.or.u(3)==undef.or.scalex(2)==undef)then
        val(2)=undef
     else
        if(scalex(2)/=0.0)then
           val(2)=0.5*(u(3)-u(1))/(scalex(2)*dx(2))
        else
           val(2)=0.0
        end if
     end if

     if(u(nx)==undef.or.u(nx-2)==undef.or.scalex(nx-1)==undef)then
        val(nx-1)=undef
     else
        if(scalex(nx-1)/=0.0)then
           val(nx-1)=0.5*(u(nx)-u(nx-2))/(scalex(nx-1)*dx(nx-1))
        else
           val(nx-1)=0.0
        end if
     end if

!-- ǡΤʤξüν ---
     if(u(1)==undef.or.u(2)==undef.or.scalex(1)==undef)then
        val(1)=undef
     else
        if(scalex(1)/=0.0)then
           val(1)=(u(2)-u(1))/(scalex(1)*(x(2)-x(1)))
        else
           val(1)=0.0
        end if
     end if
     if(u(nx)==undef.or.u(nx-1)==undef.or.scalex(nx)==undef)then
        val(nx)=undef
     else
        if(scalex(nx)/=0.0)then
           val(nx)=(u(nx)-u(nx-1))/(scalex(nx)*(x(nx)-x(nx-1)))
        else
           val(nx)=0.0
        end if
     end if
  else
     do i=3,nx-2
        if(scalex(i)/=0.0)then
           val(i)=(coe23*(u(i+1)-u(i-1))-coe112*(u(i+2)-u(i-2)))  &
  &                  /(scalex(i)*dx(i))
        else
           val(i)=0.0
        end if
     end do

!-- ǡΤʤξüν (ξü 1 ¦) ---
     if(scalex(2)/=0.0)then
        val(2)=0.5*(u(3)-u(1))/(scalex(2)*dx(2))
     else
        val(2)=0.0
     end if

     if(scalex(nx-1)/=0.0)then
        val(nx-1)=0.5*(u(nx)-u(nx-2))/(scalex(nx-1)*dx(nx-1))
     else
        val(nx-1)=0.0
     end if

!-- ǡΤʤξüν ---
     if(scalex(1)/=0.0)then
        val(1)=(u(2)-u(1))/(scalex(1)*(x(2)-x(1)))
     else
        val(1)=0.0
     end if
     if(scalex(nx)/=0.0)then
        val(nx)=(u(nx)-u(nx-1))/(scalex(nx)*(x(nx)-x(nx-1)))
     else
        val(nx)=0.0
     end if
  end if

end subroutine grad4_1df

!-----------------------------------------
!-----------------------------------------

subroutine grad4_1dd( x, u, val, undef, hx )
! 1 Υ顼ѿθۤ׻
! $\frac{\partial p}{\partial x} $ 
! 4 ʬǽ񤭴,  $(i)$ Ǥθۤ
! $(2/3)*\frac{p_{i+1}-p_{i-1}}{dx} -(p_{i+2}-p_{i-2})/(12dx)$
! ȤǤ. Ѥ 1 ۤ׻.
! ǡ­ʤΰǤ, 1 κʬǷ׻Τ, ٤
! .
  implicit none
  double precision, intent(in) :: x(:)  ! ֺɸ
  double precision, intent(in) :: u(size(x))  ! ζб 1 顼
  double precision, intent(inout) :: val(size(x))  ! 顼ͤ x θ
  double precision, intent(in), optional :: undef
  double precision, intent(in), optional :: hx(size(x))  ! x Υ
  integer :: i  ! 졼ź
  integer :: nx  ! ǿ
  double precision :: scalex(size(x)), dx(size(x)), coe23, coe112

  nx=size(x)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( nx, u ),  &
  &                                     "grad4_1d" )
     call check_array_size_dmp_message( check_array_size_1d( nx, val ),  &
  &                                     "grad4_1d" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_1d( nx, hx ),  &
  &                                        "grad4_1d" )
     end if
  end if

  do i=2,nx-1
     dx(i)=0.5d0*(x(i+1)-x(i-1))
  end do
  dx(1)=x(2)-x(1)
  dx(nx)=x(nx)-x(nx-1)
  coe23=2.0d0/3.0d0
  coe112=1.0d0/12.0d0

  if(present(hx))then
     do i=1,nx
        scalex(i)=hx(i)
     end do
  else
     do i=1,nx
        scalex(i)=1.0d0
     end do
  end if

  if(present(undef))then
     do i=3,nx-2
        if(u(i+1)==undef.or.u(i-1)==undef.or.  &
  &        u(i+2)==undef.or.u(i-2)==undef.or.scalex(i)==undef)then
           val(i)=undef
        else
           if(scalex(i)/=0.0d0)then
              val(i)=(coe23*(u(i+1)-u(i-1))-coe112*(u(i+2)-u(i-2)))  &
  &                  /(scalex(i)*dx(i))
           else
              val(i)=0.0d0
           end if
        end if
     end do

!-- ǡΤʤξüν (ξü 1 ¦) ---
     if(u(1)==undef.or.u(3)==undef.or.scalex(2)==undef)then
        val(2)=undef
     else
        if(scalex(2)/=0.0d0)then
           val(2)=0.5d0*(u(3)-u(1))/(scalex(2)*dx(2))
        else
           val(2)=0.0d0
        end if
     end if

     if(u(nx)==undef.or.u(nx-2)==undef.or.scalex(nx-1)==undef)then
        val(nx-1)=undef
     else
        if(scalex(nx-1)/=0.0d0)then
           val(nx-1)=0.5d0*(u(nx)-u(nx-2))/(scalex(nx-1)*dx(nx-1))
        else
           val(nx-1)=0.0d0
        end if
     end if

!-- ǡΤʤξüν ---
     if(u(1)==undef.or.u(2)==undef.or.scalex(1)==undef)then
        val(1)=undef
     else
        if(scalex(1)/=0.0d0)then
           val(1)=(u(2)-u(1))/(scalex(1)*(x(2)-x(1)))
        else
           val(1)=0.0d0
        end if
     end if
     if(u(nx)==undef.or.u(nx-1)==undef.or.scalex(nx)==undef)then
        val(nx)=undef
     else
        if(scalex(nx)/=0.0d0)then
           val(nx)=(u(nx)-u(nx-1))/(scalex(nx)*(x(nx)-x(nx-1)))
        else
           val(nx)=0.0d0
        end if
     end if
  else
     do i=3,nx-2
        if(scalex(i)/=0.0d0)then
           val(i)=(coe23*(u(i+1)-u(i-1))-coe112*(u(i+2)-u(i-2)))  &
  &                  /(scalex(i)*dx(i))
        else
           val(i)=0.0d0
        end if
     end do

!-- ǡΤʤξüν (ξü 1 ¦) ---
     if(scalex(2)/=0.0d0)then
        val(2)=0.5d0*(u(3)-u(1))/(scalex(2)*dx(2))
     else
        val(2)=0.0d0
     end if

     if(scalex(nx-1)/=0.0d0)then
        val(nx-1)=0.5d0*(u(nx)-u(nx-2))/(scalex(nx-1)*dx(nx-1))
     else
        val(nx-1)=0.0d0
     end if

!-- ǡΤʤξüν ---
     if(scalex(1)/=0.0d0)then
        val(1)=(u(2)-u(1))/(scalex(1)*(x(2)-x(1)))
     else
        val(1)=0.0d0
     end if
     if(scalex(nx)/=0.0d0)then
        val(nx)=(u(nx)-u(nx-1))/(scalex(nx)*(x(nx)-x(nx-1)))
     else
        val(nx)=0.0d0
     end if
  end if

end subroutine grad4_1dd

!-----------------------------------------
!-----------------------------------------

subroutine grad4_2df( x, y, u, valx, valy, undeff, hx, hy )
  ! 1 顼ۤΥ롼Ѥ 2 ۤΥ٥ȥ׻
  ! $\nabla _hp =\left(\frac{\partial p}{\partial x} ,\; \frac{\partial p}{\partial y} \right) $ 
  ! 4 ʬǽ񤭴,  $(i,j)$ Ǥθۤ
  ! $\left(\frac{p_{i+1,j}-p_{i-1,j}}{2dx} ,\; \frac{p_{i,j+1}-p_{i,j-1}}{2dy} \right) $
  ! $(2/3)*\frac{p_{i+1,j}-p_{i-1,j}}{dx} -(p_{i+2,j}-p_{i-2,j})/(12dx),\; 
  ! (2/3)*\frac{p_{i,j+1}-p_{i,j-1}}{dy} -(p_{i,j+2}-p_{i,j-2})/(12dy)$
  ! ȤǤ. Ѥ2ۤ׻.
  ! ǡ­ʤΰǤ, 1 κʬǷ׻Τ, ٤
  ! .
  ! Ѥ 1 ۷׻롼 2 ƤӽФȤˤƤ.
  implicit none
  real, intent(in) :: x(:)  ! x κɸѿ [m]
  real, intent(in) :: y(:)  ! y κɸѿ [m]
  real, intent(in) :: u(size(x),size(y))  ! ۤȤ 2 顼ʬ
  real, intent(inout) :: valx(size(x),size(y))  ! ׻줿 y  2 ۥ٥ȥ
  real, intent(inout) :: valy(size(x),size(y))  ! ׻줿 y  2 ۥ٥ȥ
  real, intent(in), optional :: undeff
  real, intent(in), optional :: hx(size(x),size(y))  ! x Υ
  real, intent(in), optional :: hy(size(x),size(y))  ! y Υ
  integer :: i   ! 졼ź
  integer :: j   ! 졼ź
  integer :: nx  ! x ǿ(size ؿǼưŪ˷׻)
  integer :: ny  ! y ǿ(size ؿǼưŪ˷׻)
  real :: scalex(size(x),size(y)), scaley(size(x),size(y))

  nx=size(x)
  ny=size(y)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "grad4_2d" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, valx ),  &
  &                                     "grad4_2d" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, valy ),  &
  &                                     "grad4_2d" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, hx ),  &
  &                                        "grad4_2d" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, hy ),  &
  &                                        "grad4_2d" )
     end if
  end if

  if(present(hx))then
     do j=1,ny
        do i=1,nx
           scalex(i,j)=hx(i,j)
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           scalex(i,j)=1.0
        end do
     end do
  end if

  if(present(hy))then
     do j=1,ny
        do i=1,nx
           scaley(i,j)=hy(i,j)
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           scaley(i,j)=1.0
        end do
     end do
  end if

  if(present(undeff))then
     do i=1,ny
        call grad4_1df(x, u(:,i), valx(:,i), undeff)
     end do

     do i=1,nx
        call grad4_1df(y, u(i,:), valy(i,:), undeff)
     end do

     do j=1,ny
        do i=1,nx
           if(u(i,j)/=undeff.and.scalex(i,j)/=undeff.and.  &
  &           scaley(i,j)/=undeff)then
              if(scalex(i,j)/=0.0)then
                 valx(i,j)=valx(i,j)/scalex(i,j)
              else
                 valx(i,j)=0.0
              end if
              if(scaley(i,j)/=0.0)then
                 valy(i,j)=valy(i,j)/scaley(i,j)
              else
                 valy(i,j)=0.0
              end if
!-- , else ʤΤ, grad_1d 롼ǤǤ undeff äƤ뤫ƱȤ
!-- ʤΤ, 䰦.
           end if
        end do
     end do

  else

     do i=1,ny
        call grad4_1df(x, u(:,i), valx(:,i) )
     end do

     do i=1,nx
        call grad4_1df(y, u(i,:), valy(i,:) )
     end do

     do j=1,ny
        do i=1,nx
           if(scalex(i,j)/=0.0)then
              valx(i,j)=valx(i,j)/scalex(i,j)
           else
              valx(i,j)=0.0
           end if
           if(scaley(i,j)/=0.0)then
              valy(i,j)=valy(i,j)/scaley(i,j)
           else
              valy(i,j)=0.0
           end if
        end do
     end do

  end if

end subroutine grad4_2df

!-----------------------------------------
!-----------------------------------------

subroutine grad4_2dd( x, y, u, valx, valy, undeff, hx, hy )
  ! 1 顼ۤΥ롼Ѥ 2 ۤΥ٥ȥ׻
  ! $\nabla _hp =\left(\frac{\partial p}{\partial x} ,\; \frac{\partial p}{\partial y} \right) $ 
  ! 4 ʬǽ񤭴,  $(i,j)$ Ǥθۤ
  ! $\left(\frac{p_{i+1,j}-p_{i-1,j}}{2dx} ,\; \frac{p_{i,j+1}-p_{i,j-1}}{2dy} \right) $
  ! $(2/3)*\frac{p_{i+1,j}-p_{i-1,j}}{dx} -(p_{i+2,j}-p_{i-2,j})/(12dx),\; 
  ! (2/3)*\frac{p_{i,j+1}-p_{i,j-1}}{dy} -(p_{i,j+2}-p_{i,j-2})/(12dy)$
  ! ȤǤ. Ѥ2ۤ׻.
  ! ǡ­ʤΰǤ, 1 κʬǷ׻Τ, ٤
  ! .
  ! Ѥ 1 ۷׻롼 2 ƤӽФȤˤƤ.
  implicit none
  double precision, intent(in) :: x(:)  ! x κɸѿ [m]
  double precision, intent(in) :: y(:)  ! y κɸѿ [m]
  double precision, intent(in) :: u(size(x),size(y))  ! ۤȤ 2 顼ʬ
  double precision, intent(inout) :: valx(size(x),size(y))  ! ׻줿 y  2 ۥ٥ȥ
  double precision, intent(inout) :: valy(size(x),size(y))  ! ׻줿 y  2 ۥ٥ȥ
  double precision, intent(in), optional :: undeff
  double precision, intent(in), optional :: hx(size(x),size(y))  ! x Υ
  double precision, intent(in), optional :: hy(size(x),size(y))  ! y Υ
  integer :: i   ! 졼ź
  integer :: j   ! 졼ź
  integer :: nx  ! x ǿ(size ؿǼưŪ˷׻)
  integer :: ny  ! y ǿ(size ؿǼưŪ˷׻)
  double precision :: scalex(size(x),size(y)), scaley(size(x),size(y))

  nx=size(x)
  ny=size(y)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "grad4_2d" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, valx ),  &
  &                                     "grad4_2d" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, valy ),  &
  &                                     "grad4_2d" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, hx ),  &
  &                                        "grad4_2d" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, hy ),  &
  &                                        "grad4_2d" )
     end if
  end if

  if(present(hx))then
     do j=1,ny
        do i=1,nx
           scalex(i,j)=hx(i,j)
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           scalex(i,j)=1.0d0
        end do
     end do
  end if

  if(present(hy))then
     do j=1,ny
        do i=1,nx
           scaley(i,j)=hy(i,j)
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           scaley(i,j)=1.0d0
        end do
     end do
  end if

  if(present(undeff))then
     do i=1,ny
        call grad4_1dd(x, u(:,i), valx(:,i), undeff)
     end do

     do i=1,nx
        call grad4_1dd(y, u(i,:), valy(i,:), undeff)
     end do

     do j=1,ny
        do i=1,nx
           if(u(i,j)/=undeff.and.scalex(i,j)/=undeff.and.  &
  &           scaley(i,j)/=undeff)then
              if(scalex(i,j)/=0.0d0)then
                 valx(i,j)=valx(i,j)/scalex(i,j)
              else
                 valx(i,j)=0.0d0
              end if
              if(scaley(i,j)/=0.0d0)then
                 valy(i,j)=valy(i,j)/scaley(i,j)
              else
                 valy(i,j)=0.0d0
              end if
!-- , else ʤΤ, grad_1d 롼ǤǤ undeff äƤ뤫ƱȤ
!-- ʤΤ, 䰦.
           end if
        end do
     end do

  else

     do i=1,ny
        call grad4_1dd(x, u(:,i), valx(:,i) )
     end do

     do i=1,nx
        call grad4_1dd(y, u(i,:), valy(i,:) )
     end do

     do j=1,ny
        do i=1,nx
           if(scalex(i,j)/=0.0d0)then
              valx(i,j)=valx(i,j)/scalex(i,j)
           else
              valx(i,j)=0.0d0
           end if
           if(scaley(i,j)/=0.0d0)then
              valy(i,j)=valy(i,j)/scaley(i,j)
           else
              valy(i,j)=0.0d0
           end if
        end do
     end do

  end if

end subroutine grad4_2dd

!-----------------------------------------
!-----------------------------------------

subroutine zast_2_w_2d( x, y, zeta, zf, zt, u, v, w, wh, undef )
! terrain following ɸϤɾƤ®ʬǥȺɸϤǤ
! ®ʬѴ.
! , terrain following Ϥοʿ®ϥǥȺɸϤǤ
! 礭Ѳʤ, Υ롼Ǥϱľ®ΤѴԤ.
! ޤ, 3 ǥȷϤγʻѴΤǤϤʤ, terrain following Ϥ
! ®ʬǥȷѴ.
! , ʿˤϥǥȷϤˤΤбƤ.
! ⤷, ɸѴ, vert_coord_conv ⥸塼ѤΤ.
  implicit none
  real, dimension(:), intent(in) :: x  ! κɸ [m]
  real, dimension(:), intent(in) :: y  ! κɸ [m]
  real, dimension(size(x),size(y)), intent(in) :: zeta  ! terrain ϱľɸ [m]
  real, dimension(size(x),size(y)), intent(in) :: zf  ! ɽ̹ٺɸ [m]
  real, dimension(size(x),size(y)), intent(in) :: zt  ! ɸǾü [m]
  real, dimension(size(x),size(y)), intent(in) :: u  ! zeta ˤ [m/s]
  real, dimension(size(x),size(y)), intent(in) :: v  ! zeta ˤ [m/s]
  real, dimension(size(x),size(y)), intent(in) :: w  ! zeta ˤľ [m/s]
  real, dimension(size(x),size(y)), intent(inout) :: wh  ! ǥȷϤˤľ [m/s]
  real, intent(in), optional :: undef  ! »
  integer :: i, j, nx, ny
  real, dimension(size(x),size(y)) :: dx, dy
  real :: j31, j32, jd, coe

  nx=size(x)
  ny=size(y)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, zeta ),  &
  &                                     "zast_2_w_2d" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, zf ),  &
  &                                     "zast_2_w_2d" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, zt ),  &
  &                                     "zast_2_w_2d" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "zast_2_w_2d" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, v ),  &
  &                                     "zast_2_w_2d" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, w ),  &
  &                                     "zast_2_w_2d" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, wh ),  &
  &                                     "zast_2_w_2d" )
  end if

  call grad_2d( x, y, zf, dx, dy )

  if(present(undef))then
     do j=1,ny
        do i=1,nx
           if(u(i,j)/=undef.and.v(i,j)/=undef.and.w(i,j)/=undef)then
              jd=1.0-zf(i,j)/zt(i,j)
              coe=zeta(i,j)/zt(i,j)-1.0
              j31=coe*dx(i,j)
              j32=coe*dy(i,j)
              wh(i,j)=(u(i,j)*j31+v(i,j)*j32+w(i,j))/jd
           else
              wh(i,j)=undef
           end if
        end do
     end do
  else
     do j=1,ny
        do i=1,nx
           jd=1.0-zf(i,j)/zt(i,j)
           coe=zeta(i,j)/zt(i,j)-1.0
           j31=coe*dx(i,j)
           j32=coe*dy(i,j)
           wh(i,j)=(u(i,j)*j31+v(i,j)*j32+w(i,j))/jd
        end do
     end do
  end if

end subroutine zast_2_w_2d

!-----------------------------------------
!-----------------------------------------

subroutine zast_2_w_3d( x, y, zeta, zf, zt, u, v, w, wh, undef )
! terrain following ɸϤɾƤ®ʬǥȺɸϤǤ
! ®ʬѴ.
! , terrain following Ϥοʿ®ϥǥȺɸϤǤ
! 礭Ѳʤ, Υ롼Ǥϱľ®ΤѴԤ.
! ޤ, 3 ǥȷϤγʻѴΤǤϤʤ, terrain following Ϥ
! ®ʬǥȷѴ.
! , ʿˤϥǥȷϤˤΤбƤ.
! ⤷, ɸѴ, vert_coord_conv ⥸塼ѤΤ.
  implicit none
  real, dimension(:,:,:), intent(in) :: zeta  ! terrain Ϥαľɸ [m]
  real, dimension(size(zeta,1)), intent(in) :: x  ! κɸ [m]
  real, dimension(size(zeta,2)), intent(in) :: y  ! κɸ [m]
  real, dimension(size(zeta,1),size(zeta,2)), intent(in) :: zf  ! ɽ̹ٺɸ [m]
  real, dimension(size(zeta,1),size(zeta,2)), intent(in) :: zt  ! ɸǾü [m]
  real, dimension(size(zeta,1),size(zeta,2),size(zeta,3)), intent(in) :: u  ! zeta ˤ [m/s]
  real, dimension(size(zeta,1),size(zeta,2),size(zeta,3)), intent(in) :: v  ! zeta ˤ [m/s]
  real, dimension(size(zeta,1),size(zeta,2),size(zeta,3)), intent(in) :: w  ! zeta ˤľ [m/s]
  real, dimension(size(zeta,1),size(zeta,2),size(zeta,3)), intent(inout) :: wh  ! ǥȷϤˤľ [m/s]
  real, intent(in), optional :: undef  ! »
  integer :: k, nx, ny, nz

  nx=size(zeta,1)
  ny=size(zeta,2)
  nz=size(zeta,3)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( nx, x ),  &
  &                                     "zast_2_w_3d" )
     call check_array_size_dmp_message( check_array_size_1d( ny, y ),  &
  &                                     "zast_2_w_3d" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, zf ),  &
  &                                     "zast_2_w_3d" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, zt ),  &
  &                                     "zast_2_w_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, u ),  &
  &                                     "zast_2_w_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, v ),  &
  &                                     "zast_2_w_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, w ),  &
  &                                     "zast_2_w_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, wh ),  &
  &                                     "zast_2_w_3d" )
  end if

  if(present(undef))then

!$omp parallel default(shared)
!$omp do schedule(runtime) private(k)
     do k=1,nz
        call zast_2_w_2d( x, y, zeta(:,:,k), zf, zt,  &
  &                       u(:,:,k), v(:,:,k), w(:,:,k),  &
  &                       wh(:,:,k), undef )
     end do
!$omp end do
!$omp end parallel

  else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(k)
     do k=1,nz
        call zast_2_w_2d( x, y, zeta(:,:,k), zf, zt,  &
  &                       u(:,:,k), v(:,:,k), w(:,:,k),  &
  &                       wh(:,:,k) )
     end do
!$omp end do
!$omp end parallel

  end if

end subroutine zast_2_w_3d

!-----------------------------------------
!-----------------------------------------

subroutine laplacian_1df( x, u, val, undef, hx )
! 1 Υ顼ѿΥץ饷׻
! $\frac{\partial ^2p}{\partial x^2} $ 
! 2 ʬǽ񤭴,  $(i)$ Ǥθۤ
! $\frac{p_{i+1}+p_{i-1}-2p_i}{dx^2} $
! ȤǤ. Ѥ 1 ץ饷׻.
! ǡ­ʤΰǤ, undef .
! option  undef Ƥʤ, 0.0 .
! At each boundary, res = (val_n-val_{n-1})/(x_n-x_{n-1})^2
!                        -(val_n-val_{n-2})/{(x_n-x_{n-1})(x_n-x_{n-2})}
  implicit none
  real, intent(in) :: x(:)  ! x κɸѿ [m]
  real, intent(in) :: u(size(x))  ! ζб 1 顼
  real, intent(inout) :: val(size(x))  ! 顼ͤ x θ
  real, intent(in), optional :: undef
  real, intent(in), optional :: hx(size(x))  ! x Υ
  integer :: i  ! 졼ź
  integer :: nx  ! ǿ
  real :: scalex(size(x)), tmpu(size(x)), tmpv(size(x))

  nx=size(x)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( nx, u ),  &
  &                                     "laplacian_1d" )
     call check_array_size_dmp_message( check_array_size_1d( nx, val ),  &
  &                                     "laplacian_1d" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_1d( nx, hx ),  &
  &                                        "laplacian_1d" )
     end if
  end if

!-- ѿ.

  if(present(hx))then
     do i=1,nx
        if(hx(i)==0.0)then
           if(present(undef))then
              scalex(i)=undef
           else
              scalex(i)=0.0
           end if
        else
           scalex(i)=1.0/(hx(i)**2)
        end if
     end do
     if(present(undef))then
        call grad_1df( x, scalex, tmpv, undef=undef )
        call grad_1df( x, u, tmpu, undef=undef )
     else
        call grad_1df( x, scalex, tmpv )
        call grad_1df( x, u, tmpu )
     end if
  else
     do i=1,nx
        scalex(i)=1.0
     end do
     tmpv=0.0
     tmpu=0.0
  end if

  if(present(undef))then
     do i=2,nx-1
        if(u(i+1)==undef.or.u(i-1)==undef.or.u(i)==undef.or.  &
  &        scalex(i)==undef.or.tmpu(i)==undef.or.tmpv(i)==undef)then
           val(i)=undef
        else
           val(i)=4.0*((u(i+1)+u(i-1)-2.0*u(i))/((x(i+1)-x(i-1))**2))  &
  &              *scalex(i)+0.5*tmpu(i)*tmpv(i)
        end if
     end do
!-- ǡΤʤξüν ---
     if(u(1)==undef.or.u(2)==undef.or.u(3)==undef.or.  &
  &     scalex(1)==undef.or.tmpu(1)==undef.or.tmpv(1)==undef)then
        val(1)=undef
     else
        val(1)=((u(3)-u(1))/((x(3)-x(1))*(x(2)-x(1)))  &
  &            -(u(2)-u(1))/((x(2)-x(1))**2))*scalex(1)  &
  &            +0.5*tmpu(1)*tmpv(1)
     end if
     if(u(nx)==undef.or.u(nx-1)==undef.or.u(nx-2)==undef.or.  &
  &     scalex(nx)==undef.or.tmpu(nx)==undef.or.tmpv(nx)==undef)then
        val(nx)=undef
     else
        val(nx)=((u(nx)-u(nx-1))/((x(nx)-x(nx-1))**2)  &
  &            -(u(nx)-u(nx-2))/((x(nx)-x(nx-1))*(x(nx)-x(nx-2))))*scalex(nx)  &
  &            +0.5*tmpu(nx)*tmpv(nx)
     end if
  else
     do i=2,nx-1
        val(i)=4.0*((u(i+1)+u(i-1)-2.0*u(i))/((x(i+1)-x(i-1))**2))  &
  &           *scalex(i)+0.5*tmpu(i)*tmpv(i)
     end do
!-- ǡΤʤξüν ---
     val(1)=((u(3)-u(1))/((x(3)-x(1))*(x(2)-x(1)))  &
  &         -(u(2)-u(1))/((x(2)-x(1))**2))*scalex(1)  &
  &         +0.5*tmpu(1)*tmpv(1)
     val(nx)=((u(nx)-u(nx-1))/((x(nx)-x(nx-1))**2)  &
  &         -(u(nx)-u(nx-2))/((x(nx)-x(nx-1))*(x(nx)-x(nx-2))))*scalex(nx)  &
  &         +0.5*tmpu(nx)*tmpv(nx)
  end if

end subroutine laplacian_1df

!-----------------------------------------
!-----------------------------------------

subroutine laplacian_1dd( x, u, val, undef, hx )
! 1 Υ顼ѿΥץ饷׻
! $\frac{\partial ^2p}{\partial x^2} $ 
! 2 ʬǽ񤭴,  $(i)$ Ǥθۤ
! $\frac{p_{i+1}+p_{i-1}-2p_i}{dx^2} $
! ȤǤ. Ѥ 1 ץ饷׻.
! ǡ­ʤΰǤ, undef .
! option  undef Ƥʤ, 0.0d0 .
! At each boundary, res = (val_n-val_{n-1})/(x_n-x_{n-1})^2
!                        -(val_n-val_{n-2})/{(x_n-x_{n-1})(x_n-x_{n-2})}
  implicit none
  double precision, intent(in) :: x(:)  ! x κɸѿ [m]
  double precision, intent(in) :: u(size(x))  ! ζб 1 顼
  double precision, intent(inout) :: val(size(x))  ! 顼ͤ x θ
  double precision, intent(in), optional :: undef
  double precision, intent(in), optional :: hx(size(x))  ! x Υ
  integer :: i  ! 졼ź
  integer :: nx  ! ǿ
  double precision :: scalex(size(x)), tmpu(size(x)), tmpv(size(x))

  nx=size(x)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_1d( nx, u ),  &
  &                                     "laplacian_1d" )
     call check_array_size_dmp_message( check_array_size_1d( nx, val ),  &
  &                                     "laplacian_1d" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_1d( nx, hx ),  &
  &                                        "laplacian_1d" )
     end if
  end if

!-- ѿ.

  if(present(hx))then
     do i=1,nx
        if(hx(i)==0.0d0)then
           if(present(undef))then
              scalex(i)=undef
           else
              scalex(i)=0.0d0
           end if
        else
           scalex(i)=1.0d0/(hx(i)**2)
        end if
     end do
     if(present(undef))then
        call grad_1dd( x, scalex, tmpv, undef=undef )
        call grad_1dd( x, u, tmpu, undef=undef )
     else
        call grad_1dd( x, scalex, tmpv )
        call grad_1dd( x, u, tmpu )
     end if
  else
     do i=1,nx
        scalex(i)=1.0d0
     end do
     tmpv=0.0d0
     tmpu=0.0d0
  end if

  if(present(undef))then
     do i=2,nx-1
        if(u(i+1)==undef.or.u(i-1)==undef.or.u(i)==undef.or.  &
  &        scalex(i)==undef.or.tmpu(i)==undef.or.tmpv(i)==undef)then
           val(i)=undef
        else
           val(i)=4.0d0*((u(i+1)+u(i-1)-2.0d0*u(i))/((x(i+1)-x(i-1))**2))  &
  &              *scalex(i)+0.5d0*tmpu(i)*tmpv(i)
        end if
     end do
!-- ǡΤʤξüν ---
     if(u(1)==undef.or.u(2)==undef.or.u(3)==undef.or.  &
  &     scalex(1)==undef.or.tmpu(1)==undef.or.tmpv(1)==undef)then
        val(1)=undef
     else
        val(1)=((u(3)-u(1))/((x(3)-x(1))*(x(2)-x(1)))  &
  &            -(u(2)-u(1))/((x(2)-x(1))**2))*scalex(1)  &
  &            +0.5d0*tmpu(1)*tmpv(1)
     end if
     if(u(nx)==undef.or.u(nx-1)==undef.or.u(nx-2)==undef.or.  &
  &     scalex(nx)==undef.or.tmpu(nx)==undef.or.tmpv(nx)==undef)then
        val(nx)=undef
     else
        val(nx)=((u(nx)-u(nx-1))/((x(nx)-x(nx-1))**2)  &
  &            -(u(nx)-u(nx-2))/((x(nx)-x(nx-1))*(x(nx)-x(nx-2))))*scalex(nx)  &
  &            +0.5d0*tmpu(nx)*tmpv(nx)
     end if
  else
     do i=2,nx-1
        val(i)=4.0d0*((u(i+1)+u(i-1)-2.0d0*u(i))/((x(i+1)-x(i-1))**2))  &
  &           *scalex(i)+0.5d0*tmpu(i)*tmpv(i)
     end do
!-- ǡΤʤξüν ---
     val(1)=((u(3)-u(1))/((x(3)-x(1))*(x(2)-x(1)))  &
  &         -(u(2)-u(1))/((x(2)-x(1))**2))*scalex(1)  &
  &         +0.5d0*tmpu(1)*tmpv(1)
     val(nx)=((u(nx)-u(nx-1))/((x(nx)-x(nx-1))**2)  &
  &         -(u(nx)-u(nx-2))/((x(nx)-x(nx-1))*(x(nx)-x(nx-2))))*scalex(nx)  &
  &         +0.5d0*tmpu(nx)*tmpv(nx)
  end if

end subroutine laplacian_1dd

!-----------------------------------------
!-----------------------------------------

subroutine laplacian_2df( x, y, u, val, undef, hx, hy )
! 2 Υ顼ѿΥץ饷׻
! $\frac{\partial ^2p}{\partial x^2} $ 
! 2 ʬǽ񤭴,  $(i)$ Ǥθۤ
! $\frac{p_{i+1}+p_{i-1}-2p_i}{dx^2} $
! ȤǤ. Ѥ 1 ץ饷׻.
! ǡ­ʤΰǤ, undef .
! option  undef Ƥʤ, 0.0 .
! ľɸϤؤбϤޤ.
  implicit none
  real, intent(in) :: x(:)  ! x κɸѿ [m]
  real, intent(in) :: y(:)  ! y κɸѿ [m]
  real, intent(in) :: u(size(x),size(y))  ! ζб 2 顼
  real, intent(inout) :: val(size(x),size(y))  ! 顼ͤ 2 Υץ饷
  real, intent(in), optional :: undef
  real, intent(in), optional :: hx(size(x),size(y))  ! x Υ
  real, intent(in), optional :: hy(size(x),size(y))  ! y Υ
  integer :: i, j  ! 졼ź
  integer :: nx, ny  ! ǿ
  real :: scalex(size(x),size(y)), scaley(size(x),size(y))
  real :: sx(size(x),size(y)), sy(size(x),size(y))
  real :: tmpu(size(x),size(y)), tmpv(size(x),size(y))
  real :: tmpsx(size(x),size(y)), tmpsy(size(x),size(y))
  real :: tmpsu(size(x),size(y)), tmpsv(size(x),size(y))
  real :: ddu(size(x),size(y)), ddv(size(x),size(y))

  nx=size(x)
  ny=size(y)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "laplacian_2d" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, val ),  &
  &                                     "laplacian_2d" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, hx ),  &
  &                                        "laplacian_2d" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, hy ),  &
  &                                        "laplacian_2d" )
     end if
  end if

!-- ѿ.

  if(present(hx).or.present(hy))then
     if(present(hx))then
        do j=1,ny
           do i=1,nx
              sx(i,j)=hx(i,j)
           end do
        end do
     else
        sx=1.0
     end if
     if(present(hy))then
        do j=1,ny
           do i=1,nx
              sy(i,j)=hy(i,j)
           end do
        end do
     else
        sy=1.0
     end if

     if(present(undef))then
        do j=1,ny
           do i=1,nx
              if(sx(i,j)/=undef.and.sx(i,j)/=0.0.and.sy(i,j)/=undef.and.  &
  &              sy(i,j)/=0.0)then
                 scalex(i,j)=1.0/(sx(i,j)**2)
                 scaley(i,j)=1.0/(sy(i,j)**2)
                 tmpsx(i,j)=(sx(i,j)/sy(i,j))**2
                 tmpsy(i,j)=(sy(i,j)/sx(i,j))**2
              else
                 scalex(i,j)=undef
                 scaley(i,j)=undef
                 tmpsx(i,j)=undef
                 tmpsy(i,j)=undef
              end if
           end do
        end do
        do j=1,ny
           call grad_1df( x, tmpsy(:,j), tmpsu(:,j), undef=undef )
        end do
        do i=1,nx
           call grad_1df( y, tmpsx(i,:), tmpsv(i,:), undef=undef )
        end do
        call grad_2df( x, y, u, ddu, ddv, undeff=undef )
     else
        do j=1,ny
           do i=1,nx
              scalex(i,j)=1.0/(sx(i,j)**2)
              scaley(i,j)=1.0/(sy(i,j)**2)
              tmpsx(i,j)=(sx(i,j)/sy(i,j))**2
              tmpsy(i,j)=(sy(i,j)/sx(i,j))**2
           end do
        end do
        do j=1,ny
           call grad_1df( x, tmpsy(:,j), tmpsu(:,j) )
        end do
        do i=1,nx
           call grad_1df( y, tmpsx(i,:), tmpsv(i,:) )
        end do
        call grad_2df( x, y, u, ddu, ddv )
     end if
  else
     scalex=1.0
     scaley=1.0
     tmpsu=0.0
     tmpsv=0.0
     ddu=0.0
     ddv=0.0
  end if

  if(present(undef))then

     val=undef

     do j=1,ny
        call laplacian_1df( x, u(1:nx,j), tmpu(1:nx,j), undef )
     end do
     do i=1,nx
        call laplacian_1df( y, u(i,1:ny), tmpv(i,1:ny), undef )
     end do

     do j=1,ny
        do i=1,nx
           if(tmpu(i,j)/=undef.and.tmpv(i,j)/=undef.and.  &
  &           scalex(i,j)/=undef.and.scaley(i,j)/=undef.and.  &
  &           tmpsu(i,j)/=undef.and.tmpsv(i,j)/=undef.and.  &
  &           ddu(i,j)/=undef.and.ddv(i,j)/=undef)then
              val(i,j)=tmpu(i,j)*scalex(i,j)  &
  &                   +tmpv(i,j)*scaley(i,j)  &
  &                   +0.5*scalex(i,j)*tmpsv(i,j)*ddv(i,j)  &
  &                   +0.5*scaley(i,j)*tmpsu(i,j)*ddu(i,j)
           end if
        end do
     end do

  else

     val=0.0

     do j=1,ny
        call laplacian_1df( x, u(1:nx,j), tmpu(1:nx,j) )
     end do
     do i=1,nx
        call laplacian_1df( y, u(i,1:ny), tmpv(i,1:ny) )
     end do

     do j=1,ny
        do i=1,nx
           val(i,j)=tmpu(i,j)*scalex(i,j)  &
  &                +tmpv(i,j)*scaley(i,j)  &
  &                +0.5*scalex(i,j)*tmpsv(i,j)*ddv(i,j)  &
  &                +0.5*scaley(i,j)*tmpsu(i,j)*ddu(i,j)
        end do
     end do

  end if

end subroutine laplacian_2df

!-----------------------------------------
!-----------------------------------------

subroutine laplacian_2dd( x, y, u, val, undef, hx, hy )
! 2 Υ顼ѿΥץ饷׻
! $\frac{\partial ^2p}{\partial x^2} $ 
! 2 ʬǽ񤭴,  $(i)$ Ǥθۤ
! $\frac{p_{i+1}+p_{i-1}-2p_i}{dx^2} $
! ȤǤ. Ѥ 1 ץ饷׻.
! ǡ­ʤΰǤ, undef .
! option  undef Ƥʤ, 0.0d0 .
! ľɸϤؤбϤޤ.
  implicit none
  double precision, intent(in) :: x(:)  ! x κɸѿ [m]
  double precision, intent(in) :: y(:)  ! y κɸѿ [m]
  double precision, intent(in) :: u(size(x),size(y))  ! ζб 2 顼
  double precision, intent(inout) :: val(size(x),size(y))  ! 顼ͤ 2 Υץ饷
  double precision, intent(in), optional :: undef
  double precision, intent(in), optional :: hx(size(x),size(y))  ! x Υ
  double precision, intent(in), optional :: hy(size(x),size(y))  ! y Υ
  integer :: i, j  ! 졼ź
  integer :: nx, ny  ! ǿ
  double precision :: scalex(size(x),size(y)), scaley(size(x),size(y))
  double precision :: sx(size(x),size(y)), sy(size(x),size(y))
  double precision :: tmpu(size(x),size(y)), tmpv(size(x),size(y))
  double precision :: tmpsx(size(x),size(y)), tmpsy(size(x),size(y))
  double precision :: tmpsu(size(x),size(y)), tmpsv(size(x),size(y))
  double precision :: ddu(size(x),size(y)), ddv(size(x),size(y))

  nx=size(x)
  ny=size(y)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, u ),  &
  &                                     "laplacian_2d" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, val ),  &
  &                                     "laplacian_2d" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, hx ),  &
  &                                        "laplacian_2d" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, hy ),  &
  &                                        "laplacian_2d" )
     end if
  end if

!-- ѿ.

  if(present(hx).or.present(hy))then
     if(present(hx))then
        do j=1,ny
           do i=1,nx
              sx(i,j)=hx(i,j)
           end do
        end do
     else
        sx=1.0d0
     end if
     if(present(hy))then
        do j=1,ny
           do i=1,nx
              sy(i,j)=hy(i,j)
           end do
        end do
     else
        sy=1.0d0
     end if

     if(present(undef))then
        do j=1,ny
           do i=1,nx
              if(sx(i,j)/=undef.and.sx(i,j)/=0.0d0.and.sy(i,j)/=undef.and.  &
  &              sy(i,j)/=0.0d0)then
                 scalex(i,j)=1.0d0/(sx(i,j)**2)
                 scaley(i,j)=1.0d0/(sy(i,j)**2)
                 tmpsx(i,j)=(sx(i,j)/sy(i,j))**2
                 tmpsy(i,j)=(sy(i,j)/sx(i,j))**2
              else
                 scalex(i,j)=undef
                 scaley(i,j)=undef
                 tmpsx(i,j)=undef
                 tmpsy(i,j)=undef
              end if
           end do
        end do
        do j=1,ny
           call grad_1dd( x, tmpsy(:,j), tmpsu(:,j), undef=undef )
        end do
        do i=1,nx
           call grad_1dd( y, tmpsx(i,:), tmpsv(i,:), undef=undef )
        end do
        call grad_2dd( x, y, u, ddu, ddv, undeff=undef )
     else
        do j=1,ny
           do i=1,nx
              scalex(i,j)=1.0d0/(sx(i,j)**2)
              scaley(i,j)=1.0d0/(sy(i,j)**2)
              tmpsx(i,j)=(sx(i,j)/sy(i,j))**2
              tmpsy(i,j)=(sy(i,j)/sx(i,j))**2
           end do
        end do
        do j=1,ny
           call grad_1dd( x, tmpsy(:,j), tmpsu(:,j) )
        end do
        do i=1,nx
           call grad_1dd( y, tmpsx(i,:), tmpsv(i,:) )
        end do
        call grad_2dd( x, y, u, ddu, ddv )
     end if
  else
     scalex=1.0d0
     scaley=1.0d0
     tmpsu=0.0d0
     tmpsv=0.0d0
     ddu=0.0d0
     ddv=0.0d0
  end if

  if(present(undef))then

     val=undef

     do j=1,ny
        call laplacian_1dd( x, u(1:nx,j), tmpu(1:nx,j), undef )
     end do
     do i=1,nx
        call laplacian_1dd( y, u(i,1:ny), tmpv(i,1:ny), undef )
     end do

     do j=1,ny
        do i=1,nx
           if(tmpu(i,j)/=undef.and.tmpv(i,j)/=undef.and.  &
  &           scalex(i,j)/=undef.and.scaley(i,j)/=undef.and.  &
  &           tmpsu(i,j)/=undef.and.tmpsv(i,j)/=undef.and.  &
  &           ddu(i,j)/=undef.and.ddv(i,j)/=undef)then
              val(i,j)=tmpu(i,j)*scalex(i,j)  &
  &                   +tmpv(i,j)*scaley(i,j)  &
  &                   +0.5d0*scalex(i,j)*tmpsv(i,j)*ddv(i,j)  &
  &                   +0.5d0*scaley(i,j)*tmpsu(i,j)*ddu(i,j)
           end if
        end do
     end do

  else

     val=0.0d0

     do j=1,ny
        call laplacian_1dd( x, u(1:nx,j), tmpu(1:nx,j) )
     end do
     do i=1,nx
        call laplacian_1dd( y, u(i,1:ny), tmpv(i,1:ny) )
     end do

     do j=1,ny
        do i=1,nx
           val(i,j)=tmpu(i,j)*scalex(i,j)  &
  &                +tmpv(i,j)*scaley(i,j)  &
  &                +0.5d0*scalex(i,j)*tmpsv(i,j)*ddv(i,j)  &
  &                +0.5d0*scaley(i,j)*tmpsu(i,j)*ddu(i,j)
        end do
     end do

  end if

end subroutine laplacian_2dd

!-----------------------------------------
!-----------------------------------------

subroutine laplacian_3df( x, y, z, u, val, undef, hx, hy, hz )
! 3 Υ顼ѿΥץ饷׻
! $\frac{\partial ^2p}{\partial x^2} $ 
! 2 ʬǽ񤭴,  $(i)$ Ǥθۤ
! $\frac{p_{i+1}+p_{i-1}-2p_i}{dx^2} $
! ȤǤ. Ѥ 1 ץ饷׻.
! ǡ­ʤΰǤ, undef .
! option  undef Ƥʤ, 0.0 .
! ľɸϤؤбϤޤ.
  implicit none
  real, intent(in) :: x(:)  ! x κɸѿ [m]
  real, intent(in) :: y(:)  ! y κɸѿ [m]
  real, intent(in) :: z(:)  ! z κɸѿ [m]
  real, intent(in) :: u(size(x),size(y),size(z))  ! ζб 3 顼
  real, intent(inout) :: val(size(x),size(y),size(z))  ! 顼ͤ 3 Υץ饷
  real, intent(in), optional :: undef
  real, intent(in), optional :: hx(size(x),size(y),size(z))  ! x Υ
  real, intent(in), optional :: hy(size(x),size(y),size(z))  ! y Υ
  real, intent(in), optional :: hz(size(x),size(y),size(z))  ! z Υ
  integer :: i, j, k  ! 졼ź
  integer :: nx, ny, nz  ! ǿ
  real :: scalex(size(x),size(y),size(z))
  real :: scaley(size(x),size(y),size(z))
  real :: scalez(size(x),size(y),size(z))
  real :: sx(size(x),size(y),size(z))
  real :: sy(size(x),size(y),size(z))
  real :: sz(size(x),size(y),size(z))
  real :: tmpu(size(x),size(y),size(z)), tmpv(size(x),size(y),size(z))
  real :: tmpw(size(x),size(y),size(z))
  real :: tmpsu(size(x),size(y),size(z)), tmpsv(size(x),size(y),size(z))
  real :: tmpsw(size(x),size(y),size(z))
  real :: h1(size(x),size(y),size(z)), h2(size(x),size(y),size(z))
  real :: h3(size(x),size(y),size(z))
  real :: ddu(size(x),size(y),size(z)), ddv(size(x),size(y),size(z))
  real :: ddw(size(x),size(y),size(z))
  real, allocatable, dimension(:,:) :: tmpy1, tmpy2, tmpz1, tmpz2  ! å

!-- OpenMP ؿ
!$ integer :: OMP_GET_THREAD_NUM, OMP_GET_MAX_THREADS
  integer :: ompnum, omppe  ! ǿ

  nx=size(x)
  ny=size(y)
  nz=size(z)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, u ),  &
  &                                     "laplacian_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, val ),  &
  &                                     "laplacian_3d" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hx ),  &
  &                                        "laplacian_3d" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hy ),  &
  &                                        "laplacian_3d" )
     end if
     if(present(hz))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hz ),  &
  &                                        "laplacian_3d" )
     end if
  end if

!-- å
  ompnum=1
  omppe=1
!$   ompnum=OMP_GET_MAX_THREADS()  ! OpenMP ͭξϤͭ.

  allocate(tmpy1(ny,ompnum))
  allocate(tmpy2(ny,ompnum))
  allocate(tmpz1(nz,ompnum))
  allocate(tmpz2(nz,ompnum))

!-- ѿ.

  if(present(hx).or.present(hy).or.present(hz))then

     if(present(hx))then
        sx=hx
     else
        sx=1.0
     end if
     if(present(hy))then
        sy=hy
     else
        sy=1.0
     end if
     if(present(hz))then
        sz=hz
     else
        sz=1.0
     end if

     if(present(undef))then

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,k)
        do k=1,nz
           do j=1,ny
              do i=1,nx
                 if(sx(i,j,k)/=undef.and.sy(i,j,k)/=undef.and.sz(i,j,k)/=undef)then
                    scalex(i,j,k)=(sy(i,j,k)*sz(i,j,k)/sx(i,j,k))**2
                    scaley(i,j,k)=(sz(i,j,k)*sx(i,j,k)/sy(i,j,k))**2
                    scalez(i,j,k)=(sx(i,j,k)*sy(i,j,k)/sz(i,j,k))**2
                    h1(i,j,k)=1.0/(sx(i,j,k)**2)
                    h2(i,j,k)=1.0/(sy(i,j,k)**2)
                    h3(i,j,k)=1.0/(sz(i,j,k)**2)
                 else
                    scalex(i,j,k)=undef
                    scaley(i,j,k)=undef
                    scalez(i,j,k)=undef
                    h1(i,j,k)=undef
                    h2(i,j,k)=undef
                    h3(i,j,k)=undef
                 end if
              end do
           end do
        end do
!$omp end do
!$omp end parallel

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,k)
        do k=1,nz
           do j=1,ny
              do i=1,nx
                 scalex(i,j,k)=(sy(i,j,k)*sz(i,j,k)/sx(i,j,k))**2
                 scaley(i,j,k)=(sz(i,j,k)*sx(i,j,k)/sy(i,j,k))**2
                 scalez(i,j,k)=(sx(i,j,k)*sy(i,j,k)/sz(i,j,k))**2
                 h1(i,j,k)=1.0/(sx(i,j,k)**2)
                 h2(i,j,k)=1.0/(sy(i,j,k)**2)
                 h3(i,j,k)=1.0/(sz(i,j,k)**2)
              end do
           end do
        end do
!$omp end do
!$omp end parallel

     end if

     if(present(undef))then
!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
        do k=1,nz
           do j=1,ny
              call grad_1df( x, scalex(:,j,k), tmpsu(:,j,k), undef=undef )
           end do
        end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,k,omppe)
        do k=1,nz
           do i=1,nx
              !-- å
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
              tmpy1(1:ny,omppe)=scaley(i,1:ny,k)

              !-- dhy/dy => tmpsv
              call grad_1df( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe), undef=undef )

              !-- å夫
              tmpsv(i,1:ny,k)=tmpy2(1:ny,omppe)
           end do
        end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,omppe)
        do j=1,ny
           do i=1,nx
              !-- å
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
              tmpz1(1:nz,omppe)=scalex(i,j,1:nz)

              !-- dhx/dz => tmpsw
              call grad_1df( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe), undef=undef )

              !-- å夫
              tmpsw(i,j,1:nz)=tmpz2(1:nz,omppe)
           end do
        end do
!$omp end do
!$omp end parallel

        call grad_3df( x, y, z, u, ddu, ddv, ddw, undeff=undef )

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
        do k=1,nz
           do j=1,ny
              call grad_1df( x, scalex(:,j,k), tmpsu(:,j,k) )
           end do
        end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,k,omppe)
        do k=1,nz
           do i=1,nx
              !-- å
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
              tmpy1(1:ny,omppe)=scaley(i,1:ny,k)

              !-- dhy/dy => tmpsv
              call grad_1df( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe) )

              !-- å夫
              tmpsv(i,1:ny,k)=tmpy2(1:ny,omppe)
           end do
        end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,omppe)
        do j=1,ny
           do i=1,nx
              !-- å
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
              tmpz1(1:nz,omppe)=scalex(i,j,1:nz)

              !-- dhx/dz => tmpsw
              call grad_1df( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe) )

              !-- å夫
              tmpsw(i,j,1:nz)=tmpz2(1:nz,omppe)
           end do
        end do
!$omp end do
!$omp end parallel

        call grad_3df( x, y, z, u, ddu, ddv, ddw )
     end if

  else

     scalex=1.0
     scaley=1.0
     scalez=1.0
     tmpsu=0.0
     tmpsv=0.0
     tmpsw=0.0
     ddu=0.0
     ddv=0.0
     ddw=0.0

  end if

  if(present(undef))then

     val=undef

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
     do k=1,nz
        do j=1,ny
           call laplacian_1df( x, u(:,j,k), tmpu(:,j,k), undef )
        end do
     end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,k,omppe)
     do k=1,nz
        do i=1,nx
           !-- å
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
           tmpy1(1:ny,omppe)=u(i,1:ny,k)

           !-- d^2u/dy^2 => tmpv
           call laplacian_1df( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe), undef )

           !-- å夫
           tmpv(i,1:ny,k)=tmpy2(1:ny,omppe)
        end do
     end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,omppe)
     do j=1,ny
        do i=1,nx
           !-- å
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
           tmpz1(1:nz,omppe)=u(i,j,1:nz)

           !-- d^2u/dz^2 => tmpw
           call laplacian_1df( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe), undef )

           !-- å夫
           tmpw(i,j,1:nz)=tmpz2(1:nz,omppe)
        end do
     end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,k)

     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(tmpu(i,j,k)/=undef.and.tmpv(i,j,k)/=undef.and.  &
  &              h1(i,j,k)/=undef.and.h2(i,j,k)/=undef.and.  &
  &              h3(i,j,k)/=undef.and.tmpsu(i,j,k)/=undef.and.  &
  &              tmpsv(i,j,k)/=undef.and.tmpsw(i,j,k)/=undef.and.  &
  &              ddu(i,j,k)/=undef.and.ddv(i,j,k)/=undef.and.ddw(i,j,k)/=undef)then
                 val(i,j,k)=h1(i,j,k)*tmpu(i,j,k)  &
  &                        +h2(i,j,k)*tmpv(i,j,k)  &
  &                        +h3(i,j,k)*tmpw(i,j,k)  &
  &                        +0.5*h2(i,j,k)*h3(i,j,k)*ddu(i,j,k)*tmpsu(i,j,k)  &
  &                        +0.5*h3(i,j,k)*h1(i,j,k)*ddv(i,j,k)*tmpsv(i,j,k)  &
  &                        +0.5*h1(i,j,k)*h2(i,j,k)*ddw(i,j,k)*tmpsw(i,j,k)
              end if
           end do
        end do
     end do

!$omp end do
!$omp end parallel

  else

     val=0.0

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
     do k=1,nz
        do j=1,ny
           call laplacian_1df( x, u(:,j,k), tmpu(:,j,k) )
        end do
     end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,k,omppe)
     do k=1,nz
        do i=1,nx
           !-- å
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
           tmpy1(1:ny,omppe)=u(i,1:ny,k)

           !-- d^2u/dy^2 => tmpv
           call laplacian_1df( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe) )

           !-- å夫
           tmpv(i,1:ny,k)=tmpy2(1:ny,omppe)
        end do
     end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,omppe)
     do j=1,ny
        do i=1,nx
           !-- å
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
           tmpz1(1:nz,omppe)=u(i,j,1:nz)

           !-- d^2u/dz^2 => tmpw
           call laplacian_1df( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe) )

           !-- å夫
           tmpw(i,j,1:nz)=tmpz2(1:nz,omppe)
        end do
     end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,k)

     do k=1,nz
        do j=1,ny
           do i=1,nx
              val(i,j,k)=h1(i,j,k)*tmpu(i,j,k)  &
  &                     +h2(i,j,k)*tmpv(i,j,k)  &
  &                     +h3(i,j,k)*tmpw(i,j,k)  &
  &                     +0.5*h2(i,j,k)*h3(i,j,k)*ddu(i,j,k)*tmpsu(i,j,k)  &
  &                     +0.5*h3(i,j,k)*h1(i,j,k)*ddv(i,j,k)*tmpsv(i,j,k)  &
  &                     +0.5*h1(i,j,k)*h2(i,j,k)*ddw(i,j,k)*tmpsw(i,j,k)
           end do
        end do
     end do

!$omp end do
!$omp end parallel

  end if

end subroutine laplacian_3df

!-----------------------------------------
!-----------------------------------------

subroutine laplacian_3dd( x, y, z, u, val, undef, hx, hy, hz )
! 3 Υ顼ѿΥץ饷׻
! $\frac{\partial ^2p}{\partial x^2} $ 
! 2 ʬǽ񤭴,  $(i)$ Ǥθۤ
! $\frac{p_{i+1}+p_{i-1}-2p_i}{dx^2} $
! ȤǤ. Ѥ 1 ץ饷׻.
! ǡ­ʤΰǤ, undef .
! option  undef Ƥʤ, 0.0 .
! ľɸϤؤбϤޤ.
  implicit none
  double precision, intent(in) :: x(:)  ! x κɸѿ [m]
  double precision, intent(in) :: y(:)  ! y κɸѿ [m]
  double precision, intent(in) :: z(:)  ! z κɸѿ [m]
  double precision, intent(in) :: u(size(x),size(y),size(z))  ! ζб 3 顼
  double precision, intent(inout) :: val(size(x),size(y),size(z))  ! 顼ͤ 3 Υץ饷
  double precision, intent(in), optional :: undef
  double precision, intent(in), optional :: hx(size(x),size(y),size(z))  ! x Υ
  double precision, intent(in), optional :: hy(size(x),size(y),size(z))  ! y Υ
  double precision, intent(in), optional :: hz(size(x),size(y),size(z))  ! z Υ
  integer :: i, j, k  ! 졼ź
  integer :: nx, ny, nz  ! ǿ
  double precision :: scalex(size(x),size(y),size(z))
  double precision :: scaley(size(x),size(y),size(z))
  double precision :: scalez(size(x),size(y),size(z))
  double precision :: sx(size(x),size(y),size(z))
  double precision :: sy(size(x),size(y),size(z))
  double precision :: sz(size(x),size(y),size(z))
  double precision :: tmpu(size(x),size(y),size(z)), tmpv(size(x),size(y),size(z))
  double precision :: tmpw(size(x),size(y),size(z))
  double precision :: tmpsu(size(x),size(y),size(z)), tmpsv(size(x),size(y),size(z))
  double precision :: tmpsw(size(x),size(y),size(z))
  double precision :: h1(size(x),size(y),size(z)), h2(size(x),size(y),size(z))
  double precision :: h3(size(x),size(y),size(z))
  double precision :: ddu(size(x),size(y),size(z)), ddv(size(x),size(y),size(z))
  double precision :: ddw(size(x),size(y),size(z))
  double precision, allocatable, dimension(:,:) :: tmpy1, tmpy2, tmpz1, tmpz2  ! å

!-- OpenMP ؿ
!$ integer :: OMP_GET_THREAD_NUM, OMP_GET_MAX_THREADS
  integer :: ompnum, omppe  ! ǿ

  nx=size(x)
  ny=size(y)
  nz=size(z)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, u ),  &
  &                                     "laplacian_3d" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, val ),  &
  &                                     "laplacian_3d" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hx ),  &
  &                                        "laplacian_3d" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hy ),  &
  &                                        "laplacian_3d" )
     end if
     if(present(hz))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hz ),  &
  &                                        "laplacian_3d" )
     end if
  end if

!-- å
  ompnum=1
  omppe=1
!$   ompnum=OMP_GET_MAX_THREADS()  ! OpenMP ͭξϤͭ.

  allocate(tmpy1(ny,ompnum))
  allocate(tmpy2(ny,ompnum))
  allocate(tmpz1(nz,ompnum))
  allocate(tmpz2(nz,ompnum))

!-- ѿ.

  if(present(hx).or.present(hy).or.present(hz))then

     if(present(hx))then
        sx=hx
     else
        sx=1.0
     end if
     if(present(hy))then
        sy=hy
     else
        sy=1.0
     end if
     if(present(hz))then
        sz=hz
     else
        sz=1.0
     end if

     if(present(undef))then

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,k)
        do k=1,nz
           do j=1,ny
              do i=1,nx
                 if(sx(i,j,k)/=undef.and.sy(i,j,k)/=undef.and.sz(i,j,k)/=undef)then
                    scalex(i,j,k)=(sy(i,j,k)*sz(i,j,k)/sx(i,j,k))**2
                    scaley(i,j,k)=(sz(i,j,k)*sx(i,j,k)/sy(i,j,k))**2
                    scalez(i,j,k)=(sx(i,j,k)*sy(i,j,k)/sz(i,j,k))**2
                    h1(i,j,k)=1.0/(sx(i,j,k)**2)
                    h2(i,j,k)=1.0/(sy(i,j,k)**2)
                    h3(i,j,k)=1.0/(sz(i,j,k)**2)
                 else
                    scalex(i,j,k)=undef
                    scaley(i,j,k)=undef
                    scalez(i,j,k)=undef
                    h1(i,j,k)=undef
                    h2(i,j,k)=undef
                    h3(i,j,k)=undef
                 end if
              end do
           end do
        end do
!$omp end do
!$omp end parallel

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,k)
        do k=1,nz
           do j=1,ny
              do i=1,nx
                 scalex(i,j,k)=(sy(i,j,k)*sz(i,j,k)/sx(i,j,k))**2
                 scaley(i,j,k)=(sz(i,j,k)*sx(i,j,k)/sy(i,j,k))**2
                 scalez(i,j,k)=(sx(i,j,k)*sy(i,j,k)/sz(i,j,k))**2
                 h1(i,j,k)=1.0/(sx(i,j,k)**2)
                 h2(i,j,k)=1.0/(sy(i,j,k)**2)
                 h3(i,j,k)=1.0/(sz(i,j,k)**2)
              end do
           end do
        end do
!$omp end do
!$omp end parallel

     end if

     if(present(undef))then
!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
        do k=1,nz
           do j=1,ny
              call grad_1dd( x, scalex(:,j,k), tmpsu(:,j,k), undef=undef )
           end do
        end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,k,omppe)
        do k=1,nz
           do i=1,nx
              !-- å
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
              tmpy1(1:ny,omppe)=scaley(i,1:ny,k)

              !-- dhy/dy => tmpsv
              call grad_1dd( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe), undef=undef )

              !-- å夫
              tmpsv(i,1:ny,k)=tmpy2(1:ny,omppe)
           end do
        end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,omppe)
        do j=1,ny
           do i=1,nx
              !-- å
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
              tmpz1(1:nz,omppe)=scalex(i,j,1:nz)

              !-- dhx/dz => tmpsw
              call grad_1dd( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe), undef=undef )

              !-- å夫
              tmpsw(i,j,1:nz)=tmpz2(1:nz,omppe)
           end do
        end do
!$omp end do
!$omp end parallel

        call grad_3dd( x, y, z, u, ddu, ddv, ddw, undeff=undef )

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
        do k=1,nz
           do j=1,ny
              call grad_1dd( x, scalex(:,j,k), tmpsu(:,j,k) )
           end do
        end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,k,omppe)
        do k=1,nz
           do i=1,nx
              !-- å
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
              tmpy1(1:ny,omppe)=scaley(i,1:ny,k)

              !-- dhy/dy => tmpsv
              call grad_1dd( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe) )

              !-- å夫
              tmpsv(i,1:ny,k)=tmpy2(1:ny,omppe)
           end do
        end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,omppe)
        do j=1,ny
           do i=1,nx
              !-- å
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
              tmpz1(1:nz,omppe)=scalex(i,j,1:nz)

              !-- dhx/dz => tmpsw
              call grad_1dd( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe) )

              !-- å夫
              tmpsw(i,j,1:nz)=tmpz2(1:nz,omppe)
           end do
        end do
!$omp end do
!$omp end parallel

        call grad_3dd( x, y, z, u, ddu, ddv, ddw )
     end if

  else

     scalex=1.0
     scaley=1.0
     scalez=1.0
     tmpsu=0.0
     tmpsv=0.0
     tmpsw=0.0
     ddu=0.0
     ddv=0.0
     ddw=0.0

  end if

  if(present(undef))then

     val=undef

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
     do k=1,nz
        do j=1,ny
           call laplacian_1dd( x, u(:,j,k), tmpu(:,j,k), undef )
        end do
     end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,k,omppe)
     do k=1,nz
        do i=1,nx
           !-- å
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
           tmpy1(1:ny,omppe)=u(i,1:ny,k)

           !-- d^2u/dy^2 => tmpv
           call laplacian_1dd( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe), undef )

           !-- å夫
           tmpv(i,1:ny,k)=tmpy2(1:ny,omppe)
        end do
     end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,omppe)
     do j=1,ny
        do i=1,nx
           !-- å
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
           tmpz1(1:nz,omppe)=u(i,j,1:nz)

           !-- d^2u/dz^2 => tmpw
           call laplacian_1dd( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe), undef )

           !-- å夫
           tmpw(i,j,1:nz)=tmpz2(1:nz,omppe)
        end do
     end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,k)

     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(tmpu(i,j,k)/=undef.and.tmpv(i,j,k)/=undef.and.  &
  &              h1(i,j,k)/=undef.and.h2(i,j,k)/=undef.and.  &
  &              h3(i,j,k)/=undef.and.tmpsu(i,j,k)/=undef.and.  &
  &              tmpsv(i,j,k)/=undef.and.tmpsw(i,j,k)/=undef.and.  &
  &              ddu(i,j,k)/=undef.and.ddv(i,j,k)/=undef.and.ddw(i,j,k)/=undef)then
                 val(i,j,k)=h1(i,j,k)*tmpu(i,j,k)  &
  &                        +h2(i,j,k)*tmpv(i,j,k)  &
  &                        +h3(i,j,k)*tmpw(i,j,k)  &
  &                        +0.5*h2(i,j,k)*h3(i,j,k)*ddu(i,j,k)*tmpsu(i,j,k)  &
  &                        +0.5*h3(i,j,k)*h1(i,j,k)*ddv(i,j,k)*tmpsv(i,j,k)  &
  &                        +0.5*h1(i,j,k)*h2(i,j,k)*ddw(i,j,k)*tmpsw(i,j,k)
              end if
           end do
        end do
     end do

!$omp end do
!$omp end parallel

  else

     val=0.0

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
     do k=1,nz
        do j=1,ny
           call laplacian_1dd( x, u(:,j,k), tmpu(:,j,k) )
        end do
     end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,k,omppe)
     do k=1,nz
        do i=1,nx
           !-- å
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
           tmpy1(1:ny,omppe)=u(i,1:ny,k)

           !-- d^2u/dy^2 => tmpv
           call laplacian_1dd( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe) )

           !-- å夫
           tmpv(i,1:ny,k)=tmpy2(1:ny,omppe)
        end do
     end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,omppe)
     do j=1,ny
        do i=1,nx
           !-- å
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
           tmpz1(1:nz,omppe)=u(i,j,1:nz)

           !-- d^2u/dz^2 => tmpw
           call laplacian_1dd( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe) )

           !-- å夫
           tmpw(i,j,1:nz)=tmpz2(1:nz,omppe)
        end do
     end do
!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i,j,k)

     do k=1,nz
        do j=1,ny
           do i=1,nx
              val(i,j,k)=h1(i,j,k)*tmpu(i,j,k)  &
  &                     +h2(i,j,k)*tmpv(i,j,k)  &
  &                     +h3(i,j,k)*tmpw(i,j,k)  &
  &                     +0.5*h2(i,j,k)*h3(i,j,k)*ddu(i,j,k)*tmpsu(i,j,k)  &
  &                     +0.5*h3(i,j,k)*h1(i,j,k)*ddv(i,j,k)*tmpsv(i,j,k)  &
  &                     +0.5*h1(i,j,k)*h2(i,j,k)*ddw(i,j,k)*tmpsw(i,j,k)
           end do
        end do
     end do

!$omp end do
!$omp end parallel

  end if

end subroutine laplacian_3dd

!-----------------------------------------
!-----------------------------------------

subroutine z_2_zeta( z, zf, zt, zeta )
! ɸ z  terrain following ɸѴ.
  implicit none
  real, intent(in) :: z(:,:,:)    ! ǥȴɸϱľɸ [m]
  real, intent(in) :: zf(size(z,1),size(z,2))   ! ɽ̹ [m]
  real, intent(in) :: zt(size(z,1),size(z,2))   ! ǹ [m]
  real, intent(inout) :: zeta(size(z,1),size(z,2),size(z,3))  ! terrain following ɸ [m]

  integer :: i, j, k
  integer :: nx, ny, nz

  nx=size(z,1)
  ny=size(z,2)
  nz=size(z,3)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, zf ),  &
  &                                     "z_2_zeta" )
     call check_array_size_dmp_message( check_array_size_2d( nx, ny, zt ),  &
  &                                     "z_2_zeta" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, zeta ),  &
  &                                     "z_2_zeta" )
  end if

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,k)

  do k=1,nz
     do j=1,ny
        do i=1,nx
           zeta(i,j,k)=zt(i,j)*(z(i,j,k)-zf(i,j))/(zt(i,j)-zf(i,j))
        end do
     end do
  end do

!$omp end do
!$omp end parallel

end subroutine z_2_zeta

!-----------------------------------------
!-----------------------------------------


subroutine turb_visc( signal, x, y, z, u, v, w, rho, nuh, nuv, val, undef, hx, hy, hz, sfctau )
! ǥȺɸϤˤήǴ׻.
  implicit none
  character(1) :: signal  ! ǥȺɸϤβܤήʬȽꤹ.
                  ! [1] = ǥȺɸϤˤ x ɸʬ ( u ʬ)
                  ! [2] = ǥȺɸϤˤ y ɸʬ ( v ʬ)
                  ! [3] = ǥȺɸϤˤ z ɸʬ ( w ʬ)
  real, intent(in) :: x(:)  ! x ζֺɸ [m]
  real, intent(in) :: y(:)  ! y ζֺɸ [m]
  real, intent(in) :: z(:)  ! z ζֺɸ [m]
  real, intent(in) :: u(size(x),size(y),size(z))  ! x б 2 ٥ȥʬ
  real, intent(in) :: v(size(x),size(y),size(z))  ! y б 2 ٥ȥʬ
  real, intent(in) :: w(size(x),size(y),size(z))  ! y б 2 ٥ȥʬ
  real, intent(in) :: rho(size(x),size(y),size(z))  ! ܾ̩ [kg/m^3]
  real, intent(in) :: nuh(size(x),size(y),size(z))  ! ʿǴ
  real, intent(in) :: nuv(size(x),size(y),size(z))  ! ľǴ
  real, intent(inout) :: val(size(x),size(y),size(z))  ! ήեå
  real, intent(in), optional :: undef
  real, intent(in), optional :: hx(size(x),size(y),size(z))  ! x Υ
  real, intent(in), optional :: hy(size(x),size(y),size(z))  ! y Υ
  real, intent(in), optional :: hz(size(x),size(y),size(z))  ! z Υ
  real, intent(in), optional :: sfctau(size(x),size(y))  ! ɽ̤Υեå
  integer :: i   ! 졼ź
  integer :: j   ! 졼ź
  integer :: k   ! 졼ź
  integer :: nx  ! ǿ 1 
  integer :: ny  ! ǿ 2 
  integer :: nz  ! ǿ 3 
  real, dimension(size(x),size(y),size(z),3) :: tau  ! signal 
              ! Ѥ 1,2,3 ̤˿ľʱ
  character(1) :: signaltau(3)
  integer :: id
  real, dimension(size(x),size(y)) :: stau
  real, dimension(size(x),size(y),size(z)) :: scalex, scaley, scalez

  signaltau=(/ '1', '2', '3' /)
  nx=size(x)
  ny=size(y)
  nz=size(z)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, u ),  &
  &                                     "turb_visc" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, v ),  &
  &                                     "turb_visc" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, w ),  &
  &                                     "turb_visc" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, rho ),  &
  &                                     "turb_visc" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, nuh ),  &
  &                                     "turb_visc" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, nuv ),  &
  &                                     "turb_visc" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, val ),  &
  &                                     "turb_visc" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hx ),  &
  &                                        "turb_visc" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hy ),  &
  &                                        "turb_visc" )
     end if
     if(present(hz))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hz ),  &
  &                                        "turb_visc" )
     end if
     if(present(sfctau))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, sfctau ),  &
  &                                        "turb_visc" )
     end if
  end if

  if(present(hx))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=hx(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=1.0
           end do
        end do
     end do
  end if

  if(present(hy))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=hy(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=1.0
           end do
        end do
     end do
  end if

  if(present(hz))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=hz(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=1.0
           end do
        end do
     end do
  end if

  val=0.0

  do id=1,3
     if(present(sfctau))then
        stau(:,:)=sfctau(:,:)
        if(present(undef))then
           call Reynolds_stress( signal//signaltau(id),  &
  &             x, y, z, u, v, w, rho, nuh, nuv, tau(:,:,:,id),  &
  &             hx=scalex, hy=scaley, hz=scalez, sfctau=stau, undef=undef )
        else
           call Reynolds_stress( signal//signaltau(id),  &
  &             x, y, z, u, v, w, rho, nuh, nuv, tau(:,:,:,id),  &
  &             hx=scalex, hy=scaley, hz=scalez, sfctau=stau )
        end if
     else
        if(present(undef))then
           call Reynolds_stress( signal//signaltau(id),  &
  &             x, y, z, u, v, w, rho, nuh, nuv, tau(:,:,:,id),  &
  &             hx=scalex, hy=scaley, hz=scalez, undef=undef )
        else
           call Reynolds_stress( signal//signaltau(id),  &
  &             x, y, z, u, v, w, rho, nuh, nuv, tau(:,:,:,id),  &
  &             hx=scalex, hy=scaley, hz=scalez )
        end if
     end if
  end do

!-- ήη׻, 3 ȯԤȤǤ뤿,
!-- ʲǤ, div_3d ѤƷ׻Ԥ.

  if(present(undef))then
     call div_3df( x, y, z, tau(:,:,:,1), tau(:,:,:,2), tau(:,:,:,3), val,  &
  &                hx=scalex, hy=scaley, hz=scalez, undeff=undef )
  else
     call div_3df( x, y, z, tau(:,:,:,1), tau(:,:,:,2), tau(:,:,:,3), val,  &
  &                hx=scalex, hy=scaley, hz=scalez )
  end if

end subroutine

!-----------------------------------------
!-----------------------------------------

subroutine Reynolds_stress( signal, x, y, z, u, v, w, rho, nuh, nuv, val,  &
  &                         undef, hx, hy, hz, sfctau )
! ǥȺɸϤˤ쥤Υ륺ϥƥ󥽥׻.
  implicit none
  character(2) :: signal  ! ׻ƥ󥽥ʬ.
                  ! ['11', '22', '33'] = 줾гѥƥ󥽥ʬ
                  ! ['12', '13', '21', '23', '31', '32'] = 줾г
                  ! ƥ󥽥ʬ. , оΥƥ󥽥Ǥ뤿, '12'='21' 
                  ! ׻Ƥ뤳Ȥ.
  real, intent(in) :: x(:)  ! x ζֺɸ [m]
  real, intent(in) :: y(:)  ! y ζֺɸ [m]
  real, intent(in) :: z(:)  ! z ζֺɸ [m]
  real, intent(in) :: u(size(x),size(y),size(z))  ! x б 2 ٥ȥʬ
  real, intent(in) :: v(size(x),size(y),size(z))  ! y б 2 ٥ȥʬ
  real, intent(in) :: w(size(x),size(y),size(z))  ! y б 2 ٥ȥʬ
  real, intent(in) :: rho(size(x),size(y),size(z))  ! ܾ̩ [kg/m^3]
  real, intent(in) :: nuh(size(x),size(y),size(z))  ! ʿǴ
  real, intent(in) :: nuv(size(x),size(y),size(z))  ! ľǴ
  real, intent(inout) :: val(size(x),size(y),size(z))  ! ׻줿ƥ󥽥ʬ
! , ʲΥץϻѤƤʤ.
  real, intent(in), optional :: undef
  real, intent(in), optional :: hx(size(x),size(y),size(z))  ! x Υ
  real, intent(in), optional :: hy(size(x),size(y),size(z))  ! y Υ
  real, intent(in), optional :: hz(size(x),size(y),size(z))  ! z Υ
  real, intent(in), optional :: sfctau(size(x),size(y))  ! ǥǲؤǤαϤͿƤ, ͤ.
        ! ͤϲ⤻, ñ val κǲؤ.
  integer :: i   ! 졼ź
  integer :: j   ! 졼ź
  integer :: k   ! 졼ź
  integer :: nx  ! ǿ 1 
  integer :: ny  ! ǿ 2 
  integer :: nz  ! ǿ 3 
  real :: stau(size(x),size(y))
  real :: sxx(size(x),size(y),size(z)), nu(size(x),size(y),size(z))
  real, dimension(size(x),size(y),size(z)) :: scalex, scaley, scalez
  logical, dimension(size(x),size(y),size(z)) :: undeflag

  nx=size(x)
  ny=size(y)
  nz=size(z)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, u ),  &
  &                                     "Reynolds_stress" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, v ),  &
  &                                     "Reynolds_stress" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, w ),  &
  &                                     "Reynolds_stress" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, rho ),  &
  &                                     "Reynolds_stress" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, nuh ),  &
  &                                     "Reynolds_stress" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, nuv ),  &
  &                                     "Reynolds_stress" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, val ),  &
  &                                     "Reynolds_stress" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hx ),  &
  &                                        "Reynolds_stress" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hy ),  &
  &                                        "Reynolds_stress" )
     end if
     if(present(hz))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hz ),  &
  &                                        "Reynolds_stress" )
     end if
     if(present(sfctau))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, sfctau ),  &
  &                                        "Reynolds_stress" )
     end if
  end if

  undeflag=.false.

  if(present(hx))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=hx(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=1.0
           end do
        end do
     end do
  end if

  if(present(hy))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=hy(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=1.0
           end do
        end do
     end do
  end if

  if(present(hz))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=hz(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=1.0
           end do
        end do
     end do
  end if

  val=0.0
  stau=0.0

!-- ɽ̥եåͤ.

  if(present(sfctau))then
     if(signal(2:2)=='3'.and.signal(1:1)/='3')then
        stau(:,:)=sfctau(:,:)
     end if
  end if

!-- [NOTE]
!-- ʲ, ʸ case  or Ǥʤ, 
!-- if ʸҤǤϤʤ, if ʸɽ case Ʊ褦˸.
!-- Ϥ, 夫 if 򤿤ɤ뤬, ɤξ 2 ʾ if 
!-- פʤȤΤǤ뤿˲ǽȤʤǤ,
!-- ɽ if  2 ѥʾ˹פƤޤ褦ʾʸǤ,
!-- case ѤˤѤ뤳ȤǤʤȤ.
!-- ܥ饤֥ǤΤ褦ʶ路ɽ򤷤Ƥɬ NOTE .

!-- ϥƥ󥽥η׻
  if(present(undef))then
     call deform_tensor( signal, x, y, z, u, v, w, sxx,  &
  &                      hx=scalex, hy=scaley, hz=scalez, undef=undef )
  else
     call deform_tensor( signal, x, y, z, u, v, w, sxx,  &
  &                      hx=scalex, hy=scaley, hz=scalez )
  end if

  if(signal(1:2)=='12'.or.signal(1:2)=='21')then

     do k=1,nz
        do j=1,ny
           do i=1,nx
              nu(i,j,k)=nuh(i,j,k)
           end do
        end do
     end do

  else if(signal(1:2)=='23'.or.signal(1:2)=='32')then

     if(signal(2:2)=='3')then
        do k=1,nz
           do j=1,ny
              do i=1,nx
                 nu(i,j,k)=nuv(i,j,k)
              end do
           end do
        end do
     else
        do k=1,nz
           do j=1,ny
              do i=1,nx
                 nu(i,j,k)=nuh(i,j,k)
              end do
           end do
        end do
     end if

  else if(signal(1:2)=='13'.or.signal(1:2)=='31')then

     if(signal(2:2)=='3')then
        do k=1,nz
           do j=1,ny
              do i=1,nx
                 nu(i,j,k)=nuv(i,j,k)
              end do
           end do
        end do
     else
        do k=1,nz
           do j=1,ny
              do i=1,nx
                 nu(i,j,k)=nuh(i,j,k)
              end do
           end do
        end do
     end if

  else if(signal(1:2)=='11')then

     if(present(undef))then
        call div_3df( x, y, z, u, v, w, val, hx=scalex, hy=scaley, hz=scalez, undeff=undef )
     else
        call div_3df( x, y, z, u, v, w, val, hx=scalex, hy=scaley, hz=scalez )
     end if

     do k=1,nz
        do j=1,ny
           do i=1,nx
              nu(i,j,k)=nuh(i,j,k)
           end do
        end do
     end do

  else if(signal(1:2)=='22')then

     if(present(undef))then
        call div_3df( x, y, z, u, v, w, val, hx=scalex, hy=scaley, hz=scalez, undeff=undef )
     else
        call div_3df( x, y, z, u, v, w, val, hx=scalex, hy=scaley, hz=scalez )
     end if

     do k=1,nz
        do j=1,ny
           do i=1,nx
              nu(i,j,k)=nuh(i,j,k)
           end do
        end do
     end do

  else if(signal(1:2)=='33')then

     if(present(undef))then
        call div_3df( x, y, z, u, v, w, val, hx=scalex, hy=scaley, hz=scalez, undeff=undef )
     else
        call div_3df( x, y, z, u, v, w, val, hx=scalex, hy=scaley, hz=scalez )
     end if

     do k=1,nz
        do j=1,ny
           do i=1,nx
              nu(i,j,k)=nuv(i,j,k)
           end do
        end do
     end do
  end if

  if(present(undef))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(val(i,j,k)==undef.or.nu(i,j,k)==undef.or.sxx(i,j,k)==undef)then
                 undeflag(i,j,k)=.true.
              end if
           end do
        end do
     end do
  end if

!-- ʲ, ǲؤɽ̥եå뤫ɤΥץΤ, ̥롼

  if(present(sfctau))then
     do j=1,ny
        do i=1,nx
           if(undeflag(i,j,1).eqv..false.)then
              val(i,j,1)=stau(i,j)
           else
              val(i,j,1)=undef
           end if
        end do
     end do
  else
!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j)
     do j=1,ny
        do i=1,nx
           if(undeflag(i,j,1).eqv..false.)then
              val(i,j,1)=rho(i,j,1)*nu(i,j,1)*(sxx(i,j,1)-(2.0/3.0)*val(i,j,1))
           else
              val(i,j,1)=undef
           end if
        end do
     end do
!$omp end do
!$omp end parallel
  end if

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
  do k=2,nz
     do j=1,ny
        do i=1,nx
           if(undeflag(i,j,k).eqv..false.)then
              val(i,j,k)=rho(i,j,k)*nu(i,j,k)*(sxx(i,j,k)-(2.0/3.0)*val(i,j,k))
           else
              val(i,j,k)=undef
           end if
        end do
     end do
  end do
!$omp end do
!$omp end parallel

end subroutine

!-----------------------------------------
!-----------------------------------------

subroutine deform_tensor( signal, x, y, z, u, v, w, val, undef, hx, hy, hz )
! ǥȺɸϤˤѷ®٥ƥ󥽥׻.
  implicit none
  character(2) :: signal  ! ׻ƥ󥽥ʬ.
                  ! ['11', '22', '33'] = 줾гѥƥ󥽥ʬ
                  ! ['12', '13', '21', '23', '31', '32'] = 줾г
                  ! ƥ󥽥ʬ. , оΥƥ󥽥Ǥ뤿, '12'='21' 
                  ! ׻Ƥ뤳Ȥ.
  real, intent(in) :: x(:)  ! x ζֺɸ [m]
  real, intent(in) :: y(:)  ! y ζֺɸ [m]
  real, intent(in) :: z(:)  ! z ζֺɸ [m]
  real, intent(in) :: u(size(x),size(y),size(z))  ! x б 2 ٥ȥʬ
  real, intent(in) :: v(size(x),size(y),size(z))  ! y б 2 ٥ȥʬ
  real, intent(in) :: w(size(x),size(y),size(z))  ! y б 2 ٥ȥʬ
  real, intent(inout) :: val(size(x),size(y),size(z))  ! ׻줿ƥ󥽥ʬ
! , ʲΥץϻѤƤʤ.
  real, intent(in), optional :: undef
  real, intent(in), optional :: hx(size(x),size(y),size(z))  ! x Υ
  real, intent(in), optional :: hy(size(x),size(y),size(z))  ! y Υ
  real, intent(in), optional :: hz(size(x),size(y),size(z))  ! z Υ
  integer :: i   ! 졼ź
  integer :: j   ! 졼ź
  integer :: k   ! 졼ź
  integer :: nx  ! ǿ 1 
  integer :: ny  ! ǿ 2 
  integer :: nz  ! ǿ 3 
  real, allocatable, dimension(:,:) :: tmpy1, tmpy2, tmpz1, tmpz2  ! å
  real, dimension(size(x),size(y),size(z)) :: tmpu, tmpv, tmpw
  real, dimension(size(x),size(y),size(z)) :: scalex, scaley, scalez
  real, dimension(size(x),size(y),size(z)) :: ddx, ddy, ddz
  logical, dimension(size(x),size(y),size(z)) :: undeflag

!-- OpenMP ؿ
!$ integer :: OMP_GET_THREAD_NUM, OMP_GET_MAX_THREADS
  integer :: ompnum, omppe  ! ǿ

  nx=size(x)
  ny=size(y)
  nz=size(z)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, u ),  &
  &                                     "deform_tensor" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, v ),  &
  &                                     "deform_tensor" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, w ),  &
  &                                     "deform_tensor" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, val ),  &
  &                                     "deform_tensor" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hx ),  &
  &                                        "deform_tensor" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hy ),  &
  &                                        "deform_tensor" )
     end if
     if(present(hz))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hz ),  &
  &                                        "deform_tensor" )
     end if
  end if

!-- å
  ompnum=1
  omppe=1
!$   ompnum=OMP_GET_MAX_THREADS()  ! OpenMP ͭξϤͭ.

  allocate(tmpy1(ny,ompnum))
  allocate(tmpy2(ny,ompnum))
  allocate(tmpz1(nz,ompnum))
  allocate(tmpz2(nz,ompnum))

  undeflag=.false.

  if(present(hx))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=hx(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=1.0
           end do
        end do
     end do
  end if

  if(present(hy))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=hy(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=1.0
           end do
        end do
     end do
  end if

  if(present(hz))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=hz(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=1.0
           end do
        end do
     end do
  end if

!-- [NOTE]
!-- ʲ, ʸ case  or Ǥʤ, 
!-- if ʸҤǤϤʤ, if ʸɽ case Ʊ褦˸.
!-- Ϥ, 夫 if 򤿤ɤ뤬, ɤξ 2 ʾ if 
!-- פʤȤΤǤ뤿˲ǽȤʤǤ,
!-- ɽ if  2 ѥʾ˹פƤޤ褦ʾʸǤ,
!-- case ѤˤѤ뤳ȤǤʤȤ.
!-- ܥ饤֥ǤΤ褦ʶ路ɽ򤷤Ƥɬ NOTE .

  if(present(undef))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(u(i,j,k)==undef.or.v(i,j,k)==undef.or.w(i,j,k)==undef)then
                 undeflag(i,j,k)=.true.
              end if
           end do
        end do
     end do
  end if

  if(signal(1:2)=='12'.or.signal(1:2)=='21')then

     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(undeflag(i,j,k).eqv..false.)then
                 tmpu(i,j,k)=u(i,j,k)/scalex(i,j,k)
                 tmpv(i,j,k)=v(i,j,k)/scaley(i,j,k)
              else
                 tmpu(i,j,k)=undef
                 tmpv(i,j,k)=undef
              end if
           end do
        end do
     end do

     if(present(undef))then
!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
        do k=1,nz
           do j=1,ny
              call grad_1d( x, tmpv(:,j,k), ddx(:,j,k), undef=undef )
           end do
        end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,k,omppe)
        do k=1,nz
           do i=1,nx
              !-- å
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
              tmpy1(1:ny,omppe)=tmpu(i,1:ny,k)

              !-- dtmpu/dy => ddy
              call grad_1d( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe), undef=undef )

              !-- å夫
              ddy(i,1:ny,k)=tmpy2(1:ny,omppe)
           end do
        end do
!$omp end do
!$omp end parallel

        do k=1,nz
           do j=1,ny
              do i=1,nx
                 if((ddx(i,j,k)==undef).or.(ddy(i,j,k)==undef))then
                    undeflag(i,j,k)=.true.
                 end if
              end do
           end do
        end do

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
        do k=1,nz
           do j=1,ny
              call grad_1d( x, tmpv(:,j,k), ddx(:,j,k) )
           end do
        end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k,omppe)
        do k=1,nz
           do i=1,nx
              !-- å
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
              tmpy1(1:ny,omppe)=tmpu(i,1:ny,k)

              !-- dtmpu/dy => ddy
              call grad_1d( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe) )

              !-- å夫
              ddy(i,1:ny,k)=tmpy2(1:ny,omppe)
           end do
        end do
!$omp end do
!$omp end parallel

     end if

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(undeflag(i,j,k).eqv..false.)then
                 val(i,j,k)=ddx(i,j,k)*scaley(i,j,k)/scalex(i,j,k)+  &
  &                         ddy(i,j,k)*scalex(i,j,k)/scaley(i,j,k)
              else
                 val(i,j,k)=undef
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

  else if(signal(1:2)=='23'.or.signal(1:2)=='32')then

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(undeflag(i,j,k).eqv..false.)then
                 tmpv(i,j,k)=v(i,j,k)/scaley(i,j,k)
                 tmpw(i,j,k)=w(i,j,k)/scalez(i,j,k)
              else
                 tmpv(i,j,k)=undef
                 tmpw(i,j,k)=undef
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

     if(present(undef))then
!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,k,omppe)
        do k=1,nz
           do i=1,nx
              !-- å
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
              tmpy1(1:ny,omppe)=tmpw(i,1:ny,k)

              !-- dtmpw/dy => ddy
              call grad_1d( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe), undef=undef )

              !-- å夫
              ddy(i,1:ny,k)=tmpy2(1:ny,omppe)
           end do
        end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
        do j=1,ny
           do i=1,nx
              !-- å
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
              tmpz1(1:nz,omppe)=tmpv(i,j,1:nz)

              !-- dtmpv/dz => ddz
              call grad_1d( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe), undef=undef )

              !-- å夫
              ddz(i,j,1:nz)=tmpz2(1:nz,omppe)
           end do
        end do
!$omp end do
!$omp end parallel

        do k=1,nz
           do j=1,ny
              do i=1,nx
                 if((ddy(i,j,k)==undef).or.(ddz(i,j,k)==undef))then
                    undeflag(i,j,k)=.true.
                 end if
              end do
           end do
        end do

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,k,omppe)
        do k=1,nz
           do i=1,nx
              !-- å
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
              tmpy1(1:ny,omppe)=tmpw(i,1:ny,k)

              !-- dtmpw/dy => ddy
              call grad_1d( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe) )

              !-- å夫
              ddy(i,1:ny,k)=tmpy2(1:ny,omppe)
           end do
        end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
        do j=1,ny
           do i=1,nx
              !-- å
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
              tmpz1(1:nz,omppe)=tmpv(i,j,1:nz)

              !-- dtmpv/dz => ddz
              call grad_1d( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe) )

              !-- å夫
              ddz(i,j,1:nz)=tmpz2(1:nz,omppe)
           end do
        end do
!$omp end do
!$omp end parallel

     end if

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(undeflag(i,j,k).eqv..false.)then
                 val(i,j,k)=ddy(i,j,k)*scalez(i,j,k)/scaley(i,j,k)+  &
  &                         ddz(i,j,k)*scaley(i,j,k)/scalez(i,j,k)
              else
                 val(i,j,k)=undef
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

  else if(signal(1:2)=='13'.or.signal(1:2)=='31')then

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(undeflag(i,j,k).eqv..false.)then
                 tmpu(i,j,k)=u(i,j,k)/scalex(i,j,k)
                 tmpw(i,j,k)=w(i,j,k)/scalez(i,j,k)
              else
                 tmpu(i,j,k)=undef
                 tmpw(i,j,k)=undef
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

     if(present(undef))then
!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
        do j=1,ny
           do i=1,nx
              !-- å
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
              tmpz1(1:nz,omppe)=tmpu(i,j,1:nz)

              !-- dtmpu/dz => ddz
              call grad_1d( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe), undef=undef )

              !-- å夫
              ddz(i,j,1:nz)=tmpz2(1:nz,omppe)
           end do
        end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
        do k=1,nz
           do j=1,ny
              call grad_1d( x, tmpw(:,j,k), ddx(:,j,k), undef=undef )
           end do
        end do
!$omp end do
!$omp end parallel

        do k=1,nz
           do j=1,ny
              do i=1,nx
                 if((ddz(i,j,k)==undef).or.(ddx(i,j,k)==undef))then
                    undeflag(i,j,k)=.true.
                 end if
              end do
           end do
        end do

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
        do j=1,ny
           do i=1,nx
              !-- å
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
              tmpz1(1:nz,omppe)=tmpu(i,j,1:nz)

              !-- dtmpu/dz => ddz
              call grad_1d( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe) )

              !-- å夫
              ddz(i,j,1:nz)=tmpz2(1:nz,omppe)
           end do
        end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
        do k=1,nz
           do j=1,ny
              call grad_1d( x, tmpw(:,j,k), ddx(:,j,k) )
           end do
        end do
!$omp end do
!$omp end parallel
     end if

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(undeflag(i,j,k).eqv..false.)then
                 val(i,j,k)=ddz(i,j,k)*scalex(i,j,k)/scalez(i,j,k)+  &
  &                         ddx(i,j,k)*scalez(i,j,k)/scalex(i,j,k)
              else
                 val(i,j,k)=undef
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

  else if(signal(1:2)=='11')then

     !-- scale  undef ꤷƤʤΤ, undef ׻Ϥʤ.
!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,k,omppe)
     do k=1,nz
        do i=1,nx
           !-- å
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
           tmpy1(1:ny,omppe)=scalex(i,1:ny,k)

           !-- dhx/dy => ddy
           call grad_1d( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe) )

           !-- å夫
           ddy(i,1:ny,k)=tmpy2(1:ny,omppe)
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
     do j=1,ny
        do i=1,nx
           !-- å
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
           tmpz1(1:nz,omppe)=scalex(i,j,1:nz)

           !-- dhx/dz => ddz
           call grad_1d( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe) )

           !-- å夫
           ddz(i,j,1:nz)=tmpz2(1:nz,omppe)
        end do
     end do
!$omp end do
!$omp end parallel

     if(present(undef))then
!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
        do k=1,nz
           do j=1,ny
              call grad_1d( x, u(:,j,k), ddx(:,j,k), undef=undef )
           end do
        end do
!$omp end do
!$omp end parallel

        do k=1,nz
           do j=1,ny
              do i=1,nx
                 if(ddx(i,j,k)==undef)then
                    undeflag(i,j,k)=.true.
                 end if
              end do
           end do
        end do

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
        do k=1,nz
           do j=1,ny
              call grad_1d( x, u(:,j,k), ddx(:,j,k) )
           end do
        end do
!$omp end do
!$omp end parallel

     end if

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(undeflag(i,j,k).eqv..false.)then
                 val(i,j,k)=2.0*(ddx(i,j,k)+ddy(i,j,k)*v(i,j,k)/scaley(i,j,k)+  &
  &                              ddz(i,j,k)*w(i,j,k)/scalez(i,j,k))/scalex(i,j,k)
              else
                 val(i,j,k)=undef
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

  else if(signal(1:2)=='22')then

     if(present(undef))then
!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,k,omppe)
        do k=1,nz
           do i=1,nx
              !-- å
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
              tmpy1(1:ny,omppe)=v(i,1:ny,k)

              !-- dv/dy => ddy
              call grad_1d( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe), undef=undef )

              !-- å夫
              ddy(i,1:ny,k)=tmpy2(1:ny,omppe)
           end do
        end do
!$omp end do
!$omp end parallel

        do k=1,nz
           do j=1,ny
              do i=1,nx
                 if(ddy(i,j,k)==undef)then
                    undeflag(i,j,k)=.true.
                 end if
              end do
           end do
        end do

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,k,omppe)
        do k=1,nz
           do i=1,nx
              !-- å
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
              tmpy1(1:ny,omppe)=v(i,1:ny,k)

              !-- dv/dy => ddy
              call grad_1d( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe) )

              !-- å夫
              ddy(i,1:ny,k)=tmpy2(1:ny,omppe)
           end do
        end do
!$omp end do
!$omp end parallel

     end if

     !-- scale  undef ꤷƤʤΤ, undef ׻Ϥʤ.
!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
     do j=1,ny
        do i=1,nx
           !-- å
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
           tmpz1(1:nz,omppe)=scaley(i,j,1:nz)

           !-- dhy/dz => ddz
           call grad_1d( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe) )

           !-- å夫
           ddz(i,j,1:nz)=tmpz2(1:nz,omppe)
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
     do k=1,nz
        do j=1,ny
           call grad_1d( x, scaley(:,j,k), ddx(:,j,k) )
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(undeflag(i,j,k).eqv..false.)then
                 val(i,j,k)=2.0*(ddy(i,j,k)+ddz(i,j,k)*w(i,j,k)/scalez(i,j,k)+  &
  &                              ddx(i,j,k)*u(i,j,k)/scalex(i,j,k))/scaley(i,j,k)
              else
                 val(i,j,k)=undef
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

  else if(signal(1:2)=='33')then

     !-- scale  undef ꤷƤʤΤ, undef ׻Ϥʤ.
!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,k,omppe)
     do k=1,nz
        do i=1,nx
           !-- å
!$         omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
           tmpy1(1:ny,omppe)=scalez(i,1:ny,k)

           !-- dhz/dy => ddy
           call grad_1d( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe) )

           !-- å夫
           ddy(i,1:ny,k)=tmpy2(1:ny,omppe)
        end do
     end do
!$omp end do
!$omp end parallel

     if(present(undef))then
!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
        do j=1,ny
           do i=1,nx
              !-- å
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
              tmpz1(1:nz,omppe)=w(i,j,1:nz)

              !-- dw/dz => ddz
              call grad_1d( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe), undef=undef )

              !-- å夫
              ddz(i,j,1:nz)=tmpz2(1:nz,omppe)
           end do
        end do
!$omp end do
!$omp end parallel

        do k=1,nz
           do j=1,ny
              do i=1,nx
                 if(ddz(i,j,k)==undef)then
                    undeflag(i,j,k)=.true.
                 end if
              end do
           end do
        end do

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
        do j=1,ny
           do i=1,nx
              !-- å
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
              tmpz1(1:nz,omppe)=w(i,j,1:nz)

              !-- dw/dz => ddz
              call grad_1d( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe) )

              !-- å夫
              ddz(i,j,1:nz)=tmpz2(1:nz,omppe)
           end do
        end do
!$omp end do
!$omp end parallel

     end if

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
     do k=1,nz
        do j=1,ny
           call grad_1d( x, scalez(:,j,k), ddx(:,j,k) )
        end do
     end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(undeflag(i,j,k).eqv..false.)then
                 val(i,j,k)=2.0*(ddz(i,j,k)+ddx(i,j,k)*u(i,j,k)/scalex(i,j,k)+  &
  &                              ddy(i,j,k)*v(i,j,k)/scaley(i,j,k))/scalez(i,j,k)
              else
                 val(i,j,k)=undef
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

  end if

end subroutine

!-----------------------------------------
!-----------------------------------------

subroutine turb_diff( x, y, z, phi, rho, nuh, nuv, val, undef, hx, hy, hz, sfcphi )
! ήȻ׻.
  implicit none
  real, intent(in) :: x(:)  ! x ζֺɸ [m]
  real, intent(in) :: y(:)  ! y ζֺɸ [m]
  real, intent(in) :: z(:)  ! z ζֺɸ [m]
  real, intent(in) :: phi(size(x),size(y),size(z))  ! ήȻ׻륹顼
  real, intent(in) :: rho(size(x),size(y),size(z))  ! ܾ̩ [kg/m^3]
  real, intent(in) :: nuh(size(x),size(y),size(z))  ! ʿǴ
  real, intent(in) :: nuv(size(x),size(y),size(z))  ! ľǴ
  real, intent(inout) :: val(size(x),size(y),size(z))  ! 3 ȯ
  real, intent(in), optional :: undef
  real, intent(in), optional :: hx(size(x),size(y),size(z))  ! x Υ
  real, intent(in), optional :: hy(size(x),size(y),size(z))  ! y Υ
  real, intent(in), optional :: hz(size(x),size(y),size(z))  ! z Υ
  real, intent(in), optional :: sfcphi(size(x),size(y))      ! ɽ̤Υեå
  integer :: i   ! 졼ź
  integer :: j   ! 졼ź
  integer :: k   ! 졼ź
  integer :: nx  ! ǿ 1 
  integer :: ny  ! ǿ 2 
  integer :: nz  ! ǿ 3 
  real, dimension(size(x),size(y),size(z),3) :: tau  ! signal 
              ! Ѥ 1,2,3 ̤˿ľʱ
  character(1) :: signaltau(3)
  integer :: id
  real, dimension(size(x),size(y)) :: stau
  real, dimension(size(x),size(y),size(z)) :: scalex, scaley, scalez

  signaltau=(/ '1', '2', '3' /)

  nx=size(x)
  ny=size(y)
  nz=size(z)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, phi ),  &
  &                                     "turb_diff" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, rho ),  &
  &                                     "turb_diff" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, nuh ),  &
  &                                     "turb_diff" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, nuv ),  &
  &                                     "turb_diff" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, val ),  &
  &                                     "turb_diff" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hx ),  &
  &                                        "turb_diff" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hy ),  &
  &                                        "turb_diff" )
     end if
     if(present(hz))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hz ),  &
  &                                        "turb_diff" )
     end if
     if(present(sfcphi))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, sfcphi ),  &
  &                                        "turb_diff" )
     end if
  end if

  if(present(hx))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=hx(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=1.0
           end do
        end do
     end do
  end if

  if(present(hy))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=hy(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=1.0
           end do
        end do
     end do
  end if

  if(present(hz))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=hz(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=1.0
           end do
        end do
     end do
  end if

  val=0.0

  do id=1,3
     if(present(sfcphi))then
        stau(:,:)=sfcphi(:,:)
        if(present(undef))then
           call Reynolds_scal( signaltau(id),  &
  &             x, y, z, phi, rho, nuh, nuv, tau(:,:,:,id),  &
  &             hx=scalex, hy=scaley, hz=scalez, sfcphi=stau, undef=undef )
        else
           call Reynolds_scal( signaltau(id),  &
  &             x, y, z, phi, rho, nuh, nuv, tau(:,:,:,id),  &
  &             hx=scalex, hy=scaley, hz=scalez, sfcphi=stau )
        end if
     else
        if(present(undef))then
           call Reynolds_scal( signaltau(id),  &
  &             x, y, z, phi, rho, nuh, nuv, tau(:,:,:,id),  &
  &             hx=scalex, hy=scaley, hz=scalez, undef=undef )
        else
           call Reynolds_scal( signaltau(id),  &
  &             x, y, z, phi, rho, nuh, nuv, tau(:,:,:,id),  &
  &             hx=scalex, hy=scaley, hz=scalez )
        end if
     end if
  end do

!-- ή 3 ȯƱȤʤΤ, div_3d ǤޤȤ.
  if(present(undef))then
     call div_3df( x, y, z, tau(:,:,:,1), tau(:,:,:,2), tau(:,:,:,3), val,  &
  &               hx=scalex, hy=scaley, hz=scalez, undeff=undef )
  else
     call div_3df( x, y, z, tau(:,:,:,1), tau(:,:,:,2), tau(:,:,:,3), val,  &
  &               hx=scalex, hy=scaley, hz=scalez )
  end if

end subroutine

!-----------------------------------------
!-----------------------------------------

subroutine Reynolds_scal( signal, x, y, z, phi, rho, nuh, nuv, val, undef, hx, hy, hz, sfcphi )
! 顼̤ήեå׻.
  implicit none
  character(1) :: signal  ! ǥȺɸϤβܤήեåʬȽꤹ.
                          ! [1] = ǥȺɸϤˤ x ɸʬ
                          ! [2] = ǥȺɸϤˤ y ɸʬ
                          ! [3] = ǥȺɸϤˤ z ɸʬ
  real, intent(in) :: x(:)  ! x ζֺɸ [m]
  real, intent(in) :: y(:)  ! y ζֺɸ [m]
  real, intent(in) :: z(:)  ! z ζֺɸ [m]
  real, intent(in) :: phi(size(x),size(y),size(z))  ! x б 2 ٥ȥʬ
  real, intent(in) :: rho(size(x),size(y),size(z))  ! ܾ̩ [kg/m^3]
  real, intent(in) :: nuh(size(x),size(y),size(z))  ! ʿȻ
  real, intent(in) :: nuv(size(x),size(y),size(z))  ! ľȻ
  real, intent(inout) :: val(size(x),size(y),size(z))  ! ׻줿ƥ󥽥ʬ
! , ʲΥץϻѤƤʤ.
  real, intent(in), optional :: undef
  real, intent(in), optional :: hx(size(x),size(y),size(z))  ! x Υ
  real, intent(in), optional :: hy(size(x),size(y),size(z))  ! y Υ
  real, intent(in), optional :: hz(size(x),size(y),size(z))  ! z Υ
  real, intent(in), optional :: sfcphi(size(x),size(y))  ! ǥǲؤǤΥեåͿƤ, ͤ.
        ! ͤϲ⤻, ñ val κǲؤ.
  integer :: i   ! 졼ź
  integer :: j   ! 졼ź
  integer :: k   ! 졼ź
  integer :: nx  ! ǿ 1 
  integer :: ny  ! ǿ 2 
  integer :: nz  ! ǿ 3 
  real, allocatable, dimension(:,:) :: tmpy1, tmpy2, tmpz1, tmpz2  ! å
  real, dimension(size(x),size(y)) :: stau
  real, dimension(size(x),size(y),size(z)) :: scalex, scaley, scalez
  logical, dimension(size(x),size(y),size(z)) :: undeflag

!-- OpenMP ؿ
!$ integer :: OMP_GET_THREAD_NUM, OMP_GET_MAX_THREADS
  integer :: ompnum, omppe  ! ǿ

  nx=size(x)
  ny=size(y)
  nz=size(z)

  if(check_array_size_iflag>0)then
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, phi ),  &
  &                                     "Reynolds_scal" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, rho ),  &
  &                                     "Reynolds_scal" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, nuh ),  &
  &                                     "Reynolds_scal" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, nuv ),  &
  &                                     "Reynolds_scal" )
     call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, val ),  &
  &                                     "Reynolds_scal" )
     if(present(hx))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hx ),  &
  &                                        "Reynolds_scal" )
     end if
     if(present(hy))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hy ),  &
  &                                        "Reynolds_scal" )
     end if
     if(present(hz))then
        call check_array_size_dmp_message( check_array_size_3d( nx, ny, nz, hz ),  &
  &                                        "Reynolds_scal" )
     end if
     if(present(sfcphi))then
        call check_array_size_dmp_message( check_array_size_2d( nx, ny, sfcphi ),  &
  &                                        "Reynolds_scal" )
     end if
  end if

!-- å
  ompnum=1
  omppe=1
!$   ompnum=OMP_GET_MAX_THREADS()  ! OpenMP ͭξϤͭ.

  allocate(tmpy1(ny,ompnum))
  allocate(tmpy2(ny,ompnum))
  allocate(tmpz1(nz,ompnum))
  allocate(tmpz2(nz,ompnum))

  undeflag=.false.

  if(present(hx))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=hx(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalex(i,j,k)=1.0
           end do
        end do
     end do
  end if

  if(present(hy))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=hy(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scaley(i,j,k)=1.0
           end do
        end do
     end do
  end if

  if(present(hz))then
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=hz(i,j,k)
           end do
        end do
     end do
  else
     do k=1,nz
        do j=1,ny
           do i=1,nx
              scalez(i,j,k)=1.0
           end do
        end do
     end do
  end if

  val=0.0

  if(present(sfcphi))then
     if(signal(1:1)=='3')then
        stau(:,:)=sfcphi(:,:)
     end if
  end if

  select case (signal(1:1))
  case ('1')
     if(present(undef))then
!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
        do k=1,nz
           do j=1,ny
              call grad_1d( x, phi(:,j,k), val(:,j,k), undef=undef )
           end do
        end do
!$omp end do
!$omp end parallel

        do k=1,nz
           do j=1,ny
              do i=1,nx
                 if(val(i,j,k)==undef.or.nuh(i,j,k)==undef.or.nuv(i,j,k)==undef)then
                    undeflag(i,j,k)=.true.
                 end if
              end do
           end do
        end do

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
        do k=1,nz
           do j=1,ny
              call grad_1d( x, phi(:,j,k), val(:,j,k) )
           end do
        end do
!$omp end do
!$omp end parallel

     end if

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(undeflag(i,j,k).eqv..false.)then
                 val(i,j,k)=rho(i,j,k)*nuh(i,j,k)*val(i,j,k)/scalex(i,j,k)
              else
                 val(i,j,k)=undef
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

  case ('2')
     if(present(undef))then
!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,k,omppe)
        do k=1,nz
           do i=1,nx
              !-- å
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
              tmpy1(1:ny,omppe)=phi(i,1:ny,k)

              !-- dphi/dy=>val
              call grad_1d( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe), undef=undef )

              !-- å夫
              val(i,1:ny,k)=tmpy2(1:ny,omppe)
           end do
        end do
!$omp end do
!$omp end parallel

        do k=1,nz
           do j=1,ny
              do i=1,nx
                 if(val(i,j,k)==undef.or.nuh(i,j,k)==undef.or.nuv(i,j,k)==undef)then
                    undeflag(i,j,k)=.true.
                 end if
              end do
           end do
        end do

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,k,omppe)
        do k=1,nz
           do i=1,nx
              !-- å
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
              tmpy1(1:ny,omppe)=phi(i,1:ny,k)

              !-- dphi/dy=>val
              call grad_1d( y(1:ny), tmpy1(1:ny,omppe), tmpy2(1:ny,omppe) )

              !-- å夫
              val(i,1:ny,k)=tmpy2(1:ny,omppe)
           end do
        end do
!$omp end do
!$omp end parallel

     end if

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(undeflag(i,j,k).eqv..false.)then
                 val(i,j,k)=rho(i,j,k)*nuh(i,j,k)*val(i,j,k)/scaley(i,j,k)
              else
                 val(i,j,k)=undef
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

  case ('3')
     if(present(undef))then
!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
        do j=1,ny
           do i=1,nx
              !-- å
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
              tmpz1(1:nz,omppe)=phi(i,j,1:nz)

              !-- dphi/dz=>val
              call grad_1d( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe), undef=undef )

              !-- å夫
              val(i,j,1:nz)=tmpz2(1:nz,omppe)
           end do
        end do
!$omp end do
!$omp end parallel

        do j=1,ny
           do i=1,nx
              do k=1,nz
                 if(val(i,j,k)==undef.or.nuh(i,j,k)==undef.or.nuv(i,j,k)==undef)then
                    undeflag(i,j,k)=.true.
                 end if
              end do
           end do
        end do

     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i,j,omppe)
        do j=1,ny
           do i=1,nx
              !-- å
!$            omppe=OMP_GET_THREAD_NUM()+1  ! OpenMP ͭξ
              tmpz1(1:nz,omppe)=phi(i,j,1:nz)

              !-- dphi/dz=>val
              call grad_1d( z(1:nz), tmpz1(1:nz,omppe), tmpz2(1:nz,omppe) )

              !-- å夫
              val(i,j,1:nz)=tmpz2(1:nz,omppe)
           end do
        end do
!$omp end do
!$omp end parallel

     end if

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(i,j,k)
     do k=1,nz
        do j=1,ny
           do i=1,nx
              if(undeflag(i,j,k).eqv..false.)then
                 val(i,j,k)=rho(i,j,k)*nuv(i,j,k)*val(i,j,k)/scalez(i,j,k)
              else
                 val(i,j,k)=undef
              end if
           end do
        end do
     end do
!$omp end do
!$omp end parallel

     do j=1,ny
        do i=1,nx
           if(present(sfcphi))then
              if(undeflag(i,j,1).eqv..false.)then
                 val(i,j,1)=stau(i,j)
              else
                 val(i,j,1)=undef
              end if
           end if
        end do
     end do

  end select

end subroutine

!-----------------------------------------
!-----------------------------------------

! ʲ, ꥸʥ zast_2_vel_?d Ǥ뤬, CReSS ǤϿʿ®˴ؤ
! terrain following ʬѲϤʤΤ, ΤȤʲ롼
! ȤӤʤ.
!subroutine zast_2_vel_2d( x, y, zast, u, v, w, uh, vh, wh, undef )
!! terrain following ɸϤɾƤ®ʿȱľѴ.
!! , 3 ǥȷϤѴΤǤϤʤ, terrain following ɸϤ
!! ®ʬǥȷѴ.
!! , ʿˤϥǥȷϤˤΤбƤ.
!! ⤷, ɸѴ, vert_coord_conv ⥸塼ѤΤ.
!  implicit none
!  real, dimension(:), intent(in) :: x  ! κɸ [m]
!  real, dimension(:), intent(in) :: y  ! κɸ [m]
!  real, dimension(size(x),size(y)), intent(in) :: zast  ! terrain Ϥ,  1  [m]
!  real, dimension(size(x),size(y)), intent(in) :: u  ! zast ˤ [m/s]
!  real, dimension(size(x),size(y)), intent(in) :: v  ! zast ˤ [m/s]
!  real, dimension(size(x),size(y)), intent(in) :: w  ! zast ˤľ [m/s]
!  real, dimension(size(x),size(y)), intent(inout) :: uh  ! ǥȷϤˤ [m/s]
!  real, dimension(size(x),size(y)), intent(inout) :: vh  ! ǥȷϤˤ [m/s]
!  real, dimension(size(x),size(y)), intent(inout) :: wh  ! ǥȷϤˤľ [m/s]
!  real, intent(in), optional :: undef  ! »
!  integer :: i, j, nx, ny
!  real, dimension(size(x),size(y)) :: dx, dy
!  real :: cosx, sinx, cosy, siny
!
!  nx=size(x)
!  ny=size(y)
!
!  call grad_2d( x, y, zast, dx, dy )
!
!  if(present(undef))then
!     do j=1,ny
!        do i=1,nx
!           if(u(i,j)/=undef.and.v(i,j)/=undef.and.w(i,j)/=undef)then
!              cosx=1.0/sqrt(1.0+dx(i,j)*dx(i,j))
!              cosy=1.0/sqrt(1.0+dy(i,j)*dy(i,j))
!              sinx=dx(i,j)*cosx
!              siny=dy(i,j)*cosy
!              uh(i,j)=u(i,j)*cosx
!              vh(i,j)=v(i,j)*cosy
!              wh(i,j)=w(i,j)+u(i,j)*sinx+v(i,j)*siny
!           else
!              uh(i,j)=undef
!              vh(i,j)=undef
!              wh(i,j)=undef
!           end if
!        end do
!     end do
!  else
!     do j=1,ny
!        do i=1,nx
!           cosx=1.0/sqrt(1.0+dx(i,j)*dx(i,j))
!           cosy=1.0/sqrt(1.0+dy(i,j)*dy(i,j))
!           sinx=dx(i,j)*cosx
!           siny=dy(i,j)*cosy
!           uh(i,j)=u(i,j)*cosx
!           vh(i,j)=v(i,j)*cosy
!           wh(i,j)=w(i,j)+u(i,j)*sinx+v(i,j)*siny
!        end do
!     end do
!  end if
!
!end subroutine zast_2_vel_2d
!
!!-----------------------------------------
!!-----------------------------------------
!
!subroutine zast_2_vel_3d( x, y, zast, u, v, w, uh, vh, wh, undef )
!! terrain following ɸϤɾƤ®ʿȱľѴ.
!! , 3 ǥȷϤѴΤǤϤʤ, terrain following ɸϤ
!! ®ʬǥȷѴ.
!! , ʿˤϥǥȷϤˤΤбƤ.
!! ⤷, ɸѴ, vert_coord_conv ⥸塼ѤΤ.
!  implicit none
!  real, dimension(:,:,:), intent(in) :: zast  ! terrain Ϥιٺɸ [m]
!  real, dimension(size(zast,1)), intent(in) :: x  ! κɸ [m]
!  real, dimension(size(zast,2)), intent(in) :: y  ! κɸ [m]
!  real, dimension(size(zast,1),size(zast,2),size(zast,3)), intent(in) :: u  ! zast ˤ [m/s]
!  real, dimension(size(zast,1),size(zast,2),size(zast,3)), intent(in) :: v  ! zast ˤ [m/s]
!  real, dimension(size(zast,1),size(zast,2),size(zast,3)), intent(in) :: w  ! zast ˤľ [m/s]
!  real, dimension(size(zast,1),size(zast,2),size(zast,3)), intent(inout) :: uh  ! ǥȷϤˤ [m/s]
!  real, dimension(size(zast,1),size(zast,2),size(zast,3)), intent(inout) :: vh  ! ǥȷϤˤ [m/s]
!  real, dimension(size(zast,1),size(zast,2),size(zast,3)), intent(inout) :: wh  ! ǥȷϤˤľ [m/s]
!  real, intent(in), optional :: undef  ! »
!  integer :: k, nx, ny, nz
!
!  nx=size(zast,1)
!  ny=size(zast,2)
!  nz=size(zast,3)
!
!  if(present(undef))then
!
!!$omp parallel default(shared)
!!$omp do schedule(runtime) private(k)
!     do k=1,nz
!        call zast_2_vel_2d( x, y, zast(:,:,k), u(:,:,k), v(:,:,k), w(:,:,k),  &
!  &                         uh(:,:,k), vh(:,:,k), wh(:,:,k), undef )
!     end do
!!$omp end do
!!$omp end parallel
!
!  else
!
!!$omp parallel default(shared)
!!$omp do schedule(runtime) private(k)
!     do k=1,nz
!        call zast_2_vel_2d( x, y, zast(:,:,k), u(:,:,k), v(:,:,k), w(:,:,k),  &
!  &                         uh(:,:,k), vh(:,:,k), wh(:,:,k) )
!     end do
!!$omp end do
!!$omp end parallel
!
!  end if
!
!end subroutine zast_2_vel_3d


end module Derivation
