program calc_TCenter_nc
! 順圧渦モデルで計算された渦中心を計算して接線平均量を
! 別の NetCDF データに出力するプログラム.
! 本プログラムでの渦中心は第一推定値近傍の風速の最小値格子点として定義される.

  use gtool_history
  use max_min
  use typhoon_analy
  use Math_Const
  use Algebra

  implicit none

  integer, parameter :: ncmax=10

  integer :: i, j, k, l, m, nx, ny, nz, nr, nt, vm
  integer :: tcx, tcy
  integer, dimension(ncmax) :: vz
  real :: tmpmin
  real, allocatable, dimension(:) :: x, y, z, t, r, theta
  real, allocatable, dimension(:,:) :: ubar, vbar
  real, allocatable, dimension(:,:) :: u, v, ws, vt, ur
  real, allocatable, dimension(:,:) :: posi, posix, posiy
  real, allocatable, dimension(:,:,:) :: val, utmp, vtmp
  real, pointer, dimension(:) :: a
  character(1000) :: vlname
  character(100) :: vuname

!-- namelist variables
  integer, dimension(2) :: fxrange, fyrange  ! first guess ranges for TCenter
  integer :: nc        ! variable number for azimuthal average
  integer :: vcflag(ncmax)   ! variable number for azimuthal average
  integer :: ntheta    ! azimuthal angle interval
  real :: rmax         ! azimuthally averaged maximum radius
  character(1000) :: finame, foname  ! input and output Netcdf file names
  character(100) :: xname, yname, zname, tname  ! each coordinate name
  character(100) :: vxname, vyname  ! name of each component of wind
  character(100), dimension(ncmax) :: vname  ! averaged variable's name
  logical :: zflag(ncmax)   ! Flag for vertical structure

  namelist /basic /finame, foname, xname, yname, zname, tname, vxname, vyname,  &
  &                nc, vname, vcflag, zflag, fxrange, fyrange, rmax, ntheta
  read(5,nml=basic)

  nullify(a)
  call HistoryGetPointer( trim(adjustl(finame)), trim(adjustl(xname)), a )
  nx=size(a)
  deallocate(a)

  nullify(a)
  call HistoryGetPointer( trim(adjustl(finame)), trim(adjustl(yname)), a )
  ny=size(a)
  deallocate(a)

  nullify(a)
  call HistoryGetPointer( trim(adjustl(finame)), trim(adjustl(zname)), a )
  nz=size(a)
  deallocate(a)

  nullify(a)
  call HistoryGetPointer( trim(adjustl(finame)), trim(adjustl(tname)), a )
  nt=size(a)
  deallocate(a)

  allocate(x(nx))
  allocate(y(ny))
  allocate(z(nz))
  allocate(t(nt))
  allocate(u(nx,ny))
  allocate(v(nx,ny))
  allocate(ws(nx,ny))
  allocate(val(nx,ny,nz))
  allocate(ur(nx,ny))
  allocate(vt(nx,ny))
  allocate(utmp(nx,ny,nz))
  allocate(vtmp(nx,ny,nz))
  allocate(posi(nx,ny))
  allocate(posix(nx,ny))
  allocate(posiy(nx,ny))

  call HistoryGet( trim(adjustl(finame)), trim(adjustl(xname)), x )
  call HistoryGet( trim(adjustl(finame)), trim(adjustl(yname)), y )
  call HistoryGet( trim(adjustl(finame)), trim(adjustl(zname)), z )
  call HistoryGet( trim(adjustl(finame)), trim(adjustl(tname)), t )

  nr=int(rmax/(x(2)-x(1)))

  allocate(r(nr))
  allocate(theta(ntheta))
  allocate(ubar(nr,nz))
  allocate(vbar(nr,nz))

  do i=1,nr
     r(i)=x(i)-x(1)
  end do

  theta=(/((2.0*pi*real(i)/real(ntheta-1)),i=1,ntheta)/)

  call HistoryCreate( &                        ! ヒストリー作成
  &    file=trim(foname),  &
  &    title='Azimuthally averaged data', &
  &    source='test',   &
  &    institution='',       &
  &    dims=(/'r', 'z','t'/),  &
  &    dimsizes=(/nr,nz,0/),  &
  &    longnames=(/'R-coordinate', 'Z-coordinate', '        Time'/),  &
  &    units=(/'m', 'm', 's'/),  &
  &    origin=real(t(1)), interval=real(t(2)-t(1)) )

  call HistoryAddVariable( &                   ! 変数定義
  &    varname='vt',  &
  &    dims=(/'r','t'/),  &
  &    longname='Azimuthally averaged tangential wind',  &
  &    units='m s-1', xtype='float' )

  do m=1,nc
     if(len_trim(adjustl(vname(m)))/=0)then
        call HistoryGetAttr( trim(adjustl(finame)), trim(adjustl(vname(m))),  &
  &                          'long_name', vlname )
        call HistoryGetAttr( trim(adjustl(finame)), trim(adjustl(vname(m))),  &
  &                          'units', vuname )

        if(zflag(m).eqv..true.)then
           call HistoryAddVariable( &
  &             varname=trim(adjustl(vname(m))),  &
  &             dims=(/'r', 'z', 't'/),  &
  &             longname='Azimuthally Averaged '//trim(adjustl(vlname)),  &
  &             units=trim(adjustl(vuname)), xtype='float' )
        else
           call HistoryAddVariable( &
  &             varname=trim(adjustl(vname(m))),  &
  &             dims=(/'r', 't'/),  &
  &             longname='Azimuthally Averaged '//trim(adjustl(vlname)),  &
  &             units=trim(adjustl(vuname)), xtype='float' )
        end if
     end if
  end do

  call HistoryPut( 'r', r )
  call HistoryPut( 'z', z )

  do k=1,nt
     call HistoryGet( trim(adjustl(finame)), trim(adjustl(vxname)),  &
  &                   u, range=trim(adjustl(tname))//'=^'  &
  &                            //trim(adjustl( i2c_convert(k) )) )
     call HistoryGet( trim(adjustl(finame)), trim(adjustl(vyname)),  &
  &                   v, range=trim(adjustl(tname))//'=^'  &
  &                            //trim(adjustl( i2c_convert(k) )) )
     call abst_2d( u, v, ws )
     call Min_val_2d( ws(fxrange(1):fxrange(2),fyrange(1):fyrange(2)),  &
  &                   tcx, tcy, tmpmin )

     tcx=tcx+fxrange(1)-1
     tcy=tcy+fyrange(1)-1

     write(*,*) t(k), tcx, tcy, x(tcx), y(tcy)

     do j=1,ny
        do i=1,nx
           posi(i,j)=sqrt((x(i)-x(tcx))**2+(y(j)-y(tcy))**2)
           if(posi(i,j)/=0.0)then
              posix(i,j)=(x(i)-x(tcx))/posi(i,j)
              posiy(i,j)=(y(j)-y(tcy))/posi(i,j)
           end if
        end do
     end do

     call vec_prod_2d( posix, posiy, u, v, vt )
     call tangent_mean_scal( x, y, x(tcx), y(tcy), vt, r, theta, vbar(1:nr,1),  &
  &                          stdopt=.true. )
     call HistoryPut( 'vt', vbar(1:nr,1) )

     vm=0
     do m=1,nc
        if(len_trim(adjustl(vname(m)))/=0)then
           if(zflag(m).eqv..true.)then
              call HistoryGet( trim(adjustl(finame)), trim(adjustl(vname(m))),  &
  &                            val(1:nx,1:ny,1:nz), range=trim(adjustl(tname))//'=^'  &
  &                                                     //trim(adjustl(i2c_convert(k))) )
              vz(m)=nz
           else
              call HistoryGet( trim(adjustl(finame)), trim(adjustl(vname(m))),  &
  &                            val(1:nx,1:ny,1), range=trim(adjustl(tname))//'=^'  &
  &                                                  //trim(adjustl(i2c_convert(k))) )
              vz(m)=1
           end if
           if(vcflag(m)==0)then  ! スカラー量の平均
              do l=1,vz(m)
                 call tangent_mean_scal( x, y, x(tcx), y(tcy), val(1:nx,1:ny,l),  &
  &                                      r, theta, vbar(1:nr,l), stdopt=.true. )
              end do
              if(zflag(m).eqv..true.)then
                 call HistoryPut( trim(adjustl(vname(m))), vbar )
              else
                 call HistoryPut( trim(adjustl(vname(m))), vbar(1:nr,1) )
              end if
           else  ! ベクトル量の平均
              if(vm==0)then  ! 2 成分読み込んでいない.
                 if(vcflag(m)==1)then
                    utmp=val
                 else
                    vtmp=val
                 end if
                 vm=m
              else  ! 2 成分読み込んだ. -> 接線平均開始.
                 if(vcflag(m)==1)then
                    utmp=val
                 else
                    vtmp=val
                 end if
                 do l=1,vz(m)
                    call dot_prod_2d( posix, posiy, utmp(:,:,l), vtmp(:,:,l), ur )
                    call vec_prod_2d( posix, posiy, utmp(:,:,l), vtmp(:,:,l), vt )
                    call tangent_mean_scal( x, y, x(tcx), y(tcy), ur,  &
  &                                         r, theta, ubar(1:nr,l), stdopt=.true. )
                    call tangent_mean_scal( x, y, x(tcx), y(tcy), vt,  &
  &                                         r, theta, vbar(1:nr,l), stdopt=.true. )
                 end do
                 if(zflag(m).eqv..true.)then
                    if(vcflag(m)==1)then  ! m == x-coord 成分
                       call HistoryPut( trim(adjustl(vname(m))), ubar )
                       call HistoryPut( trim(adjustl(vname(vm))), vbar )
                    else  ! m == y-coord 成分
                       call HistoryPut( trim(adjustl(vname(vm))), ubar )
                       call HistoryPut( trim(adjustl(vname(m))), vbar )
                    end if
                 else
                    if(vcflag(m)==1)then  ! m == x-coord 成分
                       call HistoryPut( trim(adjustl(vname(m))), ubar(1:nr,1) )
                       call HistoryPut( trim(adjustl(vname(vm))), vbar(1:nr,1) )
                    else  ! m == y-coord 成分
                       call HistoryPut( trim(adjustl(vname(vm))), ubar(1:nr,1) )
                       call HistoryPut( trim(adjustl(vname(m))), vbar(1:nr,1) )
                    end if
                 end if
                 vm=0
              end if
           end if
        end if
     end do
  end do

  call HistoryClose

end program
