program adjust_2d
! 地衡風調節を浅水系線形モデルで検証するプログラム
! 今や, 地形起伏や定常一様な一般風下での波動伝播もシミュレート可能.
! 2 次元モデル

  use file_operate
  use Derivation
  use Phys_Const
  use basis
  use gtool_history

  implicit none

! namelist values
  integer :: nt, dmpstep
  real :: corioli, dt, mean_height
  character(100) :: fname, oname
  real :: nu, kappa, tau
  real, dimension(3) :: regist_coe

  integer :: i, j, it, counter
  integer :: nx, ny, dn, bound
  real :: cp, lambda, bound_phase, beta
  real, allocatable, dimension(:) :: ini_x, ini_y, time, dx, dy
  real, allocatable, dimension(:,:) :: height, ubar, vbar
  real, allocatable, dimension(:,:) :: ini_h, ini_u, ini_v
  real, allocatable, dimension(:,:) :: old_h, old_u, old_v
  real, allocatable, dimension(:,:) :: mid_h, mid_u, mid_v
  real, allocatable, dimension(:,:) :: force_h, force_u, force_v
  real, allocatable, dimension(:,:) :: h, u, v
  real, allocatable, dimension(:,:) :: dhdx, dudx, dvdx, dvdy, dudy, dhdy
  real, allocatable, dimension(:,:) :: dh2dx, du2dx, dv2dx
  real, allocatable, dimension(:,:) :: dh2dy, du2dy, dv2dy
  real, allocatable, dimension(:,:) :: divuv
  real, allocatable, dimension(:,:) :: dheightx, dheighty
  real, allocatable, dimension(:,:,:) :: dval_h, dval_u, dval_v
  real, allocatable, dimension(:,:,:) :: dval_pe, dval_ke, dval_ae
  real, allocatable, dimension(:,:,:) :: dmp_tmp
  character(10) :: x_axis, y_axis, val_h, val_u, val_v, val_height
  character(10) :: val_ubar, val_vbar
  character(3) :: steady_flag, regist_flag

! reading namelist
  namelist /input /corioli, beta, fname, oname, nt, nx, ny, dt, dmpstep,  &
  &                x_axis, y_axis, val_height, val_h, val_u, val_v,   &
  &                val_ubar, val_vbar, mean_height, bound, steady_flag,  &
  &                regist_flag, regist_coe
  read(5,nml=input)

  dn=nt/dmpstep+1
  nu=regist_coe(1)
  kappa=regist_coe(2)
  tau=regist_coe(3)

! allocate array
  allocate(ini_x(nx))
  allocate(ini_y(ny))
  allocate(height(nx,ny))
  allocate(ubar(nx,ny))
  allocate(vbar(nx,ny))
  allocate(ini_h(nx,ny))
  allocate(ini_u(nx,ny))
  allocate(ini_v(nx,ny))
  allocate(old_h(nx,ny))
  allocate(old_u(nx,ny))
  allocate(old_v(nx,ny))
  allocate(mid_h(nx,ny))
  allocate(mid_u(nx,ny))
  allocate(mid_v(nx,ny))
  allocate(force_h(nx,ny))
  allocate(force_u(nx,ny))
  allocate(force_v(nx,ny))
  allocate(dhdx(nx,ny))
  allocate(dudx(nx,ny))
  allocate(dvdx(nx,ny))
  allocate(dhdy(nx,ny))
  allocate(dudy(nx,ny))
  allocate(dvdy(nx,ny))
  allocate(dh2dx(nx,ny))
  allocate(du2dx(nx,ny))
  allocate(dv2dx(nx,ny))
  allocate(dh2dy(nx,ny))
  allocate(du2dy(nx,ny))
  allocate(dv2dy(nx,ny))
  allocate(divuv(nx,ny))
  allocate(dheightx(nx,ny))
  allocate(dheighty(nx,ny))
  allocate(h(nx,ny))
  allocate(u(nx,ny))
  allocate(v(nx,ny))
  allocate(dx(nx))
  allocate(dy(ny))
  allocate(dval_h(nx,ny,dn))
  allocate(dval_u(nx,ny,dn))
  allocate(dval_v(nx,ny,dn))
  allocate(dval_pe(nx,ny,dn))
  allocate(dval_ke(nx,ny,dn))
  allocate(dval_ae(nx,ny,dn))
  allocate(dmp_tmp(nx,ny,dn))
  allocate(time(dn))

  time=(/((dt*dmpstep*real(i-1)),i=1,dn)/)

!-- reading initial data

  call HistoryGet( trim(fname), trim(x_axis), ini_x )
  call HistoryGet( trim(fname), trim(y_axis), ini_y )
  call HistoryGet( trim(fname), trim(val_height), height )
  call HistoryGet( trim(fname), trim(val_ubar), ubar )
  call HistoryGet( trim(fname), trim(val_vbar), vbar )
  call HistoryGet( trim(fname), trim(val_h), ini_h )
  call HistoryGet( trim(fname), trim(val_u), ini_u )
  call HistoryGet( trim(fname), trim(val_v), ini_v )

! calculate each parameter

  cp=sqrt(g*height(1,1))
  bound_phase=cp
  lambda=cp/corioli

  write(*,*) "phase speed is ", cp, "m/s."
  write(*,*) "deformation radius is ", lambda, "m."

!-- calculating fluid depth gradient.
!-- (because of constant for time except non-linear.)
!-- base state wind is assumed steady.

  call grad_2d( ini_x, ini_y, height, dheightx, dheighty )

!-- opening history file

  call HistoryCreate( file=trim(oname), title='shallow result data',  &
  &                   source='test', institution='test',  &
  &                   dims=(/'x', 'y', 't'/),  &
  &                   dimsizes=(/nx,ny,dn/),  &
  &                   longnames=(/'x-coordinate', 'y-coordinate', 'time        '/),  &
  &                   units=(/'m','m','s'/), origin=0.0, interval=0.0 )

  call HistoryPut( 'x', ini_x )
  call HistoryPut( 'y', ini_y )
  call HistoryPut( 't', time )

  call HistoryAddVariable( varname='h', dims=(/'x','y','t'/),  &
  &                        longname='height', units='m', xtype='float' )
  call HistoryAddVariable( varname='u', dims=(/'x','y','t'/),  &
  &                        longname='zonal wind', units='m s-1', xtype='float' )
  call HistoryAddVariable( varname='v', dims=(/'x','y','t'/),  &
  &                        longname='meridional wind', units='m s-1',  &
  &                        xtype='float' )
  call HistoryAddVariable( varname='pe', dims=(/'x','y','t'/),  &
  &                        longname='potential energy', units='J kg-1',  &
  &                        xtype='float' )
  call HistoryAddVariable( varname='ke', dims=(/'x','y','t'/),  &
  &                        longname='kinetic energy', units='J kg-1',  &
  &                        xtype='float' )
  call HistoryAddVariable( varname='ae', dims=(/'x','y','t'/),  &
  &                        longname='all energy', units='J kg-1',  &
  &                        xtype='float' )

  if(steady_flag(1:1)=='o')then
     call HistoryAddVariable( varname='ht', dims=(/'x','y','t'/),  &
  &                           longname='terrain height', units='m',  &
  &                           xtype='float' )
  end if
  if(steady_flag(2:2)=='o')then
     call HistoryAddVariable( varname='ub', dims=(/'x','y','t'/),  &
  &                           longname='zonal basic flow', units='m s-1',  &
  &                           xtype='float' )
  end if
  if(steady_flag(3:3)=='o')then
     call HistoryAddVariable( varname='vb', dims=(/'x','y','t'/),  &
  &                           longname='meridional basic flow', units='m s-1',  &
  &                           xtype='float' )
  end if

!-- define grid interval

  do i=2,nx-1
     dx(i)=(ini_x(i+1)-ini_x(i-1))*0.5
  end do
  dx(1)=ini_x(2)-ini_x(1)
  dx(nx)=ini_x(nx)-ini_x(nx-1)

  do j=2,ny-1
     dy(j)=(ini_y(j+1)-ini_y(j-1))*0.5
  end do
  dy(1)=ini_y(2)-ini_y(1)
  dy(ny)=ini_y(ny)-ini_y(ny-1)

  do j=1,ny
     do i=1,nx
        old_h(i,j)=ini_h(i,j)
        old_u(i,j)=ini_u(i,j)
        old_v(i,j)=ini_v(i,j)
        dval_h(i,j,1)=ini_h(i,j)
        dval_u(i,j,1)=ini_u(i,j)
        dval_v(i,j,1)=ini_v(i,j)
        dval_pe(i,j,1)=0.5*g*ini_h(i,j)*ini_h(i,j)
        dval_ke(i,j,1)=0.5*(ini_v(i,j)*ini_v(i,j)+ini_u(i,j)*ini_u(i,j))
        dval_ae(i,j,1)=dval_pe(i,j,1)+dval_ke(i,j,1)
     end do
  end do

  counter=1

  write(*,*) "*******************************************"
  write(*,*) "File damp (time =", 0.0, "[s])."
  write(*,*) "*******************************************"

  write(*,*) "starting time integration"

! time step start

  do it=1,nt

! initialization of force array

     do j=1,ny
        do i=1,nx
           force_h(i,j)=0.0
           force_u(i,j)=0.0
           force_v(i,j)=0.0
        end do
     end do

     write(*,*) "This step is ", it, "(time =", real(it)*dt, "[s])."

  ! compute gradient

     call grad_2d( ini_x, ini_y, old_h, dhdx, dhdy )
     call grad_2d( ini_x, ini_y, old_u, dudx, dudy )
     call grad_2d( ini_x, ini_y, old_v, dvdx, dvdy )

     if(regist_flag(1:1)=='o')then
        call laplacian_1d( ini_x, old_h, dh2dx )
        call laplacian_1d( ini_x, old_u, du2dx )
        call laplacian_1d( ini_x, old_v, dv2dx )
        call laplacian_1d( ini_y, old_h, dh2dy )
        call laplacian_1d( ini_y, old_u, du2dy )
        call laplacian_1d( ini_y, old_v, dv2dy )
     end if

     do j=1,ny
        do i=1,nx
           divuv(i,j)=dudx(i,j)+dvdy(i,j)
        end do
     end do

  ! compute forcing term


!$omp parallel default(shared)
!$omp do private(i,j)

     do j=2,ny-1
        do i=2,nx-1
           force_h(i,j)=-(mean_height-height(i,j))*(divuv(i,j))  &
  &                     +(ubar(i,j)+old_u(i,j))*dheightx(i,j)  &
  &                     +(vbar(i,j)+old_v(i,j))*dheighty(i,j)  &
  &                     +ubar(i,j)*dhdx(i,j)+vbar(i,j)*dhdy(i,j)
           force_u(i,j)=-g*dhdx(i,j)  &
  &                     -ubar(i,j)*dudx(i,j)  &
  &                     -vbar(i,j)*dudy(i,j)  &
  &                     +(corioli+beta*ini_y(j))*old_v(i,j)
           force_v(i,j)=-g*dhdy(i,j)  &
  &                     -ubar(i,j)*dvdx(i,j)  &
  &                     -vbar(i,j)*dvdy(i,j)  &
  &                     -(corioli+beta*ini_y(j))*old_u(i,j)

           if(regist_flag(1:1)=='o')then
              force_h(i,j)=force_h(i,j)+kappa*(dh2dx(i,j)+dh2dy(i,j))
              force_u(i,j)=force_u(i,j)+nu*(du2dx(i,j)+du2dy(i,j))
              force_v(i,j)=force_v(i,j)+nu*(dv2dx(i,j)+dv2dy(i,j))
           end if

           if(regist_flag(2:2)=='o')then
              force_h(i,j)=force_h(i,j)-tau*old_h(i,j)
              force_u(i,j)=force_u(i,j)-tau*old_u(i,j)
              force_v(i,j)=force_v(i,j)-tau*old_v(i,j)
           end if
        end do
     end do

!$omp end do
!$omp end parallel

  ! time integration

     if(it==1)then
        do j=2,ny-1
           do i=2,nx-1
              h(i,j)=old_h(i,j)+dt*force_h(i,j)
              u(i,j)=old_u(i,j)+dt*force_u(i,j)
              v(i,j)=old_v(i,j)+dt*force_v(i,j)
           end do
        end do
     else
        do j=2,ny-1
           do i=2,nx-1
              h(i,j)=mid_h(i,j)+2.0*dt*force_h(i,j)
              u(i,j)=mid_u(i,j)+2.0*dt*force_u(i,j)
              v(i,j)=mid_v(i,j)+2.0*dt*force_v(i,j)
           end do
        end do
     end if

  ! boundary coditions

     select case (bound)
     case (1)  ! Neumann conditiona
        do i=2,nx-1
           h(i,1)=h(i,2)
           u(i,1)=u(i,2)
           v(i,1)=v(i,2)
           h(i,ny)=h(i,ny-1)
           u(i,ny)=u(i,ny-1)
           v(i,ny)=v(i,ny-1)
        end do
        do j=2,ny-1
           h(1,j)=h(2,j)
           u(1,j)=u(2,j)
           v(1,j)=v(2,j)
           h(nx,j)=h(nx-1,j)
           u(nx,j)=u(nx-1,j)
           v(nx,j)=v(nx-1,j)
        end do
        h(1,1)=(h(1,2)+h(2,1))*0.5
        u(1,1)=(u(1,2)+u(2,1))*0.5
        v(1,1)=(v(1,2)+v(2,1))*0.5
        h(nx,1)=(h(nx-1,1)+h(nx,2))*0.5
        u(nx,1)=(u(nx-1,1)+u(nx,2))*0.5
        v(nx,1)=(v(nx-1,1)+v(nx,2))*0.5
        h(1,ny)=(h(1,ny-1)+h(2,ny))*0.5
        u(1,ny)=(u(1,ny-1)+u(2,ny))*0.5
        v(1,ny)=(v(1,ny-1)+v(2,ny))*0.5
        h(nx,ny)=(h(nx-1,ny)+h(nx,ny-1))*0.5
        u(nx,ny)=(u(nx-1,ny)+u(nx,ny-1))*0.5
        v(nx,ny)=(v(nx-1,ny)+v(nx,ny-1))*0.5
     case (2)  ! open boundary condition
        if(it==1)then
           do i=2,nx-1
              h(i,1)=old_h(i,1)+bound_phase*dt*dhdy(i,1)
              v(i,1)=old_v(i,1)+bound_phase*dt*dvdy(i,1)
              u(i,1)=old_u(i,1)+bound_phase*dt*dudy(i,1)
              h(i,ny)=old_h(i,ny)-bound_phase*dt*dhdy(i,ny)
              v(i,ny)=old_v(i,ny)-bound_phase*dt*dvdy(i,ny)
              u(i,ny)=old_u(i,ny)-bound_phase*dt*dudy(i,ny)
           end do
           do j=2,ny-1
              h(1,j)=old_h(1,j)+bound_phase*dt*dhdx(1,j)
              v(1,j)=old_v(1,j)+bound_phase*dt*dvdx(1,j)
              u(1,j)=old_u(1,j)+bound_phase*dt*dudx(1,j)
              h(nx,j)=old_h(nx,j)-bound_phase*dt*dhdx(nx,j)
              v(nx,j)=old_v(nx,j)-bound_phase*dt*dvdx(nx,j)
              u(nx,j)=old_u(nx,j)-bound_phase*dt*dudx(nx,j)
           end do
           h(1,1)=old_h(1,1)+bound_phase*dt*(dhdx(1,1)+dhdy(1,1))
           v(1,1)=old_v(1,1)+bound_phase*dt*(dvdx(1,1)+dvdy(1,1))
           u(1,1)=old_u(1,1)+bound_phase*dt*(dudx(1,1)+dudy(1,1))
           h(nx,1)=old_h(nx,1)+bound_phase*dt*(dhdx(nx,1)+dhdy(nx,1))
           v(nx,1)=old_v(nx,1)+bound_phase*dt*(dvdx(nx,1)+dvdy(nx,1))
           u(nx,1)=old_u(nx,1)+bound_phase*dt*(dudx(nx,1)+dudy(nx,1))
           h(1,ny)=old_h(1,ny)+bound_phase*dt*(dhdx(1,ny)+dhdy(1,ny))
           v(1,ny)=old_v(1,ny)+bound_phase*dt*(dvdx(1,ny)+dvdy(1,ny))
           u(1,ny)=old_u(1,ny)+bound_phase*dt*(dudx(1,ny)+dudy(1,ny))
           h(nx,ny)=old_h(nx,ny)+bound_phase*dt*(dhdx(nx,ny)+dhdy(nx,ny))
           v(nx,ny)=old_v(nx,ny)+bound_phase*dt*(dvdx(nx,ny)+dvdy(nx,ny))
           u(nx,ny)=old_u(nx,ny)+bound_phase*dt*(dudx(nx,ny)+dudy(nx,ny))
        else
           do i=2,nx-1
              h(i,1)=mid_h(i,1)+bound_phase*dt*dhdy(i,1)
              v(i,1)=mid_v(i,1)+bound_phase*dt*dvdy(i,1)
              u(i,1)=mid_u(i,1)+bound_phase*dt*dudy(i,1)
              h(i,ny)=mid_h(i,ny)-bound_phase*dt*dhdy(i,ny)
              v(i,ny)=mid_v(i,ny)-bound_phase*dt*dvdy(i,ny)
              u(i,ny)=mid_u(i,ny)-bound_phase*dt*dudy(i,ny)
           end do
           do j=2,ny-1
              h(1,j)=mid_h(1,j)+bound_phase*dt*dhdx(1,j)
              v(1,j)=mid_v(1,j)+bound_phase*dt*dvdx(1,j)
              u(1,j)=mid_u(1,j)+bound_phase*dt*dudx(1,j)
              h(nx,j)=mid_h(nx,j)-bound_phase*dt*dhdx(nx,j)
              v(nx,j)=mid_v(nx,j)-bound_phase*dt*dvdx(nx,j)
              u(nx,j)=mid_u(nx,j)-bound_phase*dt*dudx(nx,j)
           end do
           h(1,1)=mid_h(1,1)+bound_phase*dt*(dhdx(1,1)+dhdy(1,1))
           v(1,1)=mid_v(1,1)+bound_phase*dt*(dvdx(1,1)+dvdy(1,1))
           u(1,1)=mid_u(1,1)+bound_phase*dt*(dudx(1,1)+dudy(1,1))
           h(nx,1)=mid_h(nx,1)+bound_phase*dt*(dhdx(nx,1)+dhdy(nx,1))
           v(nx,1)=mid_v(nx,1)+bound_phase*dt*(dvdx(nx,1)+dvdy(nx,1))
           u(nx,1)=mid_u(nx,1)+bound_phase*dt*(dudx(nx,1)+dudy(nx,1))
           h(1,ny)=mid_h(1,ny)+bound_phase*dt*(dhdx(1,ny)+dhdy(1,ny))
           v(1,ny)=mid_v(1,ny)+bound_phase*dt*(dvdx(1,ny)+dvdy(1,ny))
           u(1,ny)=mid_u(1,ny)+bound_phase*dt*(dudx(1,ny)+dudy(1,ny))
           h(nx,ny)=mid_h(nx,ny)+bound_phase*dt*(dhdx(nx,ny)+dhdy(nx,ny))
           v(nx,ny)=mid_v(nx,ny)+bound_phase*dt*(dvdx(nx,ny)+dvdy(nx,ny))
           u(nx,ny)=mid_u(nx,ny)+bound_phase*dt*(dudx(nx,ny)+dudy(nx,ny))
        end if
     case(3)  ! periodic condition
        do i=2,nx-1
           h(i,1)=h(i,ny-1)
           u(i,1)=u(i,ny-1)
           v(i,1)=v(i,ny-1)
           h(i,ny)=h(i,2)
           u(i,ny)=u(i,2)
           v(i,ny)=v(i,2)
        end do
        do j=2,ny-1
           h(1,j)=h(nx-1,j)
           u(1,j)=u(nx-1,j)
           v(1,j)=v(nx-1,j)
           h(nx,j)=h(2,j)
           u(nx,j)=u(2,j)
           v(nx,j)=v(2,j)
        end do
        h(1,1)=(h(nx,ny-1)+h(nx-1,ny))*0.5
        u(1,1)=(u(nx,ny-1)+u(nx-1,ny))*0.5
        v(1,1)=(v(nx,ny-1)+v(nx-1,ny))*0.5
        h(nx,1)=(h(2,ny)+h(1,ny-1))*0.5
        u(nx,1)=(u(2,ny)+u(1,ny-1))*0.5
        v(nx,1)=(v(2,ny)+v(1,ny-1))*0.5
        h(1,ny)=(h(nx-1,1)+h(nx,2))*0.5
        u(1,ny)=(u(nx-1,1)+u(nx,2))*0.5
        v(1,ny)=(v(nx-1,1)+v(nx,2))*0.5
        h(nx,ny)=(h(1,2)+h(2,1))*0.5
        u(nx,ny)=(u(1,2)+u(2,1))*0.5
        v(nx,ny)=(v(1,2)+v(2,1))*0.5
     end select

  ! replace new values to old values

     do j=1,ny
        do i=1,nx
           mid_h(i,j)=old_h(i,j)
           mid_u(i,j)=old_u(i,j)
           mid_v(i,j)=old_v(i,j)
        end do
     end do

     do j=1,ny
        do i=1,nx
           old_h(i,j)=h(i,j)
           old_u(i,j)=u(i,j)
           old_v(i,j)=v(i,j)
        end do
     end do

  ! output file

     if(mod((it),dmpstep)==0)then
        counter=counter+1
        do j=1,ny
           do i=1,nx
              dval_h(i,j,counter)=h(i,j)
              dval_u(i,j,counter)=u(i,j)
              dval_v(i,j,counter)=v(i,j)
              dval_pe(i,j,counter)=0.5*g*h(i,j)*h(i,j)
              dval_ke(i,j,counter)=0.5*(u(i,j)*u(i,j)+v(i,j)*v(i,j))
              dval_ae(i,j,counter)=dval_pe(i,j,counter)+dval_ke(i,j,counter)
           end do
        end do
        write(*,*) "*******************************************"
        write(*,*) "File damp (time =", real(it)*dt, "[s])."
        write(*,*) "*******************************************"

     end if
  end do

  call HistoryPut( 'h', dval_h )
  call HistoryPut( 'u', dval_u )
  call HistoryPut( 'v', dval_v )
  call HistoryPut( 'pe', dval_pe )
  call HistoryPut( 'ke', dval_ke )
  call HistoryPut( 'ae', dval_ae )

  if(steady_flag(1:1)=='o')then
     do j=1,ny
        do i=1,nx
           dmp_tmp(i,j,:)=height(i,j)
        end do
     end do
     call HistoryPut( 'ht', dmp_tmp )
  end if

  if(steady_flag(2:2)=='o')then
     do j=1,ny
        do i=1,nx
           dmp_tmp(i,j,:)=ubar(i,j)
        end do
     end do
     call HistoryPut( 'ub', dmp_tmp )
  end if

  if(steady_flag(3:3)=='o')then
     do j=1,ny
        do i=1,nx
           dmp_tmp(i,j,:)=vbar(i,j)
        end do
     end do
     call HistoryPut( 'vb', dmp_tmp )
  end if

end program
