program sound_2d
! 複数のゾンデデータから各高度についてスペクトルを計算し, 各波数についての
! スペクトル時系列図を作成するプログラム.
! リストデータのフォーマットは
! YYYYMMDDHH file_name
! FFT 計算を用いるため, 必ずデータが埋められていること.
! 結果は 1/day 単位で表示する. ゾンデの観測間隔が 6 時間程度であるため.

  use dcl
  use Dcl_Automatic
  use file_operate
  use basis
  use Statistics
  use Thermo_Function
  use Math_Const
  use ffts

  implicit none

!-- integer
  integer :: nt       ! 時間方向の格子数
  integer :: nz       ! z 方向の格子数
  integer :: nf       ! conv_dat 用データ数
  integer :: IWS      ! 描画デバイス
  integer :: dt       ! データの時間間隔 [s]
  integer :: period   ! データのトータル時間 [s]
  integer :: i_counter    ! 一時カウンター
  integer :: i_counter_c  ! 一時カウンター
  integer :: snum, cnum  ! カラーおよび等値線の数
  integer :: cont_flag, shade_flag  ! カラー, 等値線用フラグ変数
  integer :: prim(5)
  integer, allocatable, dimension(:) :: year_v    ! 年
  integer, allocatable, dimension(:) :: month_v   ! 月
  integer, allocatable, dimension(:) :: day_v     ! 日
  integer, allocatable, dimension(:) :: hour_v    ! 時間
  integer, allocatable, dimension(:) :: inter_z ! 内挿用高度配列
  integer, allocatable, dimension(:) :: i_flag  ! 一時配列
  integer, allocatable, dimension(:) :: o_flag  ! 一時配列
  integer, allocatable, dimension(:) :: c_flag  ! 一時配列
!-- tmp
  integer :: nttmp, i, j, k, i_undef
!-- parameter
  integer, parameter :: col_num=6
  integer, parameter :: skip_num=2

!-- real
  real :: z_bot       ! 描画の下端
  real :: z_top       ! 描画の上端
  real :: dz          ! 鉛直方向の描画間隔 [m]
  real :: dvy         ! vy の間隔
  real :: smax, smin, cmax, cmin  ! カラー, 等値線の上限
  real, allocatable, dimension(:) :: freq  ! 振動数
  real, allocatable, dimension(:) :: z     ! 高度座標
  real, allocatable, dimension(:) :: vx     ! viewport 配列
  real, allocatable, dimension(:,:) :: vy     ! viewport 配列
  real, allocatable, dimension(:,:) :: val_f        ! 計算用配列
  real, allocatable, dimension(:,:) :: val_inter    ! 計算用変数配列
  real, allocatable, dimension(:,:) :: tline        ! 計算用変数配列
  real, allocatable, dimension(:,:) :: yline        ! 計算用変数配列
  real, allocatable, dimension(:,:,:) :: val_tmp    ! 計算用変数配列
  real, allocatable, dimension(:,:,:) :: val_draw   ! 描画用変数配列
  real :: undef
  real :: conv_undef
!-- tmp
  real, dimension(2) :: pt_tmp, ept_tmp, sept_tmp, rh_tmp, s_int, c_int

!-- complex part
  complex, allocatable, dimension(:) :: val_fft   ! 計算用変数配列
  complex, allocatable, dimension(:,:) :: omegar    ! 計算用変数配列
  complex, allocatable, dimension(:,:) :: omeganr   ! 計算用変数配列

!-- character
  character(1000) :: flist         ! ファイルリスト名
  character(1000) :: conv_dat      ! sound_1d で出力した対流パラメータ時系列データ
  character(1000) :: title_txt     ! タイトル名
  character(10) :: cont_val      ! 等値線描画用変数名
  character(10) :: shade_val     ! カラー描画用変数名
  character(10) :: conv_list     ! カラムデータフラグ
  character(1000), allocatable, dimension(:,:) :: fname  ! ファイル名
  character(1000), allocatable, dimension(:,:) :: val_w  ! テキストデータ
  character(70) :: sysfont
!-- tmp
  character(1000) :: tmp_c

!-- logical
  logical :: mean_flag       ! 平均場 (FFT 第 0 成分) 描画フラグ
  logical, allocatable, dimension(:) :: fflag  ! 各高度に undef が 1 つでも入っているか.

!-- namelist reading
  namelist /input /flist, conv_dat, conv_list, conv_undef, dz, z_bot,  &
  &                z_top, IWS, title_txt,  &
  &                cmin, cmax, smin, smax, cont_val, shade_val,  &
  &                mean_flag, cnum, snum, undef, sysfont
  read(5,nml=input)

  nz=int((z_top-z_bot)/dz)+1
  nt=line_number_counter( trim(flist) )
  if(mod(nt,2)/=0)then
     nt=nt-1  ! 奇数の場合, 偶数に置き換え.
  end if
  period=dt*(nt-1)
  s_int=(/smin, smax/)
  c_int=(/cmin, cmax/)

  allocate(fname(2,nt))
  allocate(freq(nt/2))
  allocate(z(nz))
  allocate(inter_z(nz))
  allocate(val_inter(col_num+4,nz))
  allocate(val_draw(col_num+4,nt,nz))
  allocate(val_tmp(col_num+4,nt,nz))
  allocate(val_fft(nt/2))
  allocate(year_v(nt))
  allocate(month_v(nt))
  allocate(day_v(nt))
  allocate(hour_v(nt))
  allocate(fflag(nz))
  allocate(omegar(nt/2,nt/2))
  allocate(omeganr(nt/2,nt/2))

  fflag=.false.

!-- DCL set
  if(len_trim(adjustl(sysfont))/=0)then
     call SGISET( 'IFONT', 1 )
     call SWLSET( 'LSYSFNT', .true. )
     write(*,*) "This drawing mode is sysfont."
  else
     call SGISET( 'IFONT', 2 )
     write(*,*) "This drawing mode is dclfont."
  end if

  call UZFACT(0.8)
  call DclSetParm( 'ENABLE_CONTOUR_MESSAGE', .false. )
  CALL GLRSET( 'RMISS', undef )
  CALL GLLSET( 'LMISS', .TRUE. )

  call read_file_text( trim(flist), 2, nt, fname )

  do i=1,nt
     tmp_c=trim(fname(2,i))
     year_v(i)=c2i_convert( trim(tmp_c(1:4)) )
     month_v(i)=c2i_convert( trim(tmp_c(5:6)) )
     day_v(i)=c2i_convert( trim(tmp_c(7:8)) )
     hour_v(i)=c2i_convert( trim(tmp_c(9:10)) )
  end do

  do i=1,nz
     z(i)=z_bot+dz*real(i-1)
  end do
  do i=1,nt/2
     freq(i)=86400.0*real(i-1)/real(period)
  end do

!-- fft の前処理
!-- データ数の素因数分解
  call prim_calc( nt, prim(1:4), prim(5) )
  write(*,*) "prime factor", prim(:)

!-- 回転行列の計算
  call rotate_array_f()
  call rotate_calc( nt/2, 'r', (/prim(1)-1,prim(2),prim(3),prim(4),prim(5)/),  &
  &                 omegar(1:prim(5),1:prim(5)),  &
  &                 omeganr(1:nt/2,1:nt/2) )  ! 実データの FFT なので, データ数は 1/2 となる. それに伴い, prim_fact(1) も -1 となる.

!-- time loop
!-- 1. reading values from text file
!-- 2. interpolating to grid point
!-- 3. calculating each value

  do i=1,nt
     nttmp=line_number_counter( trim(fname(1,i)) ) -2
     allocate(val_w(col_num,nttmp))
     allocate(val_f(col_num,nttmp))
     call read_file_text( trim(fname(1,i)), col_num, nttmp, val_w,  &
  &                       skip=skip_num )

  !-- convert type from character to float
     do k=1,nttmp
        do j=1,col_num
           val_f(j,k)=c2r_convert( val_w(j,k) )
        end do
     end do

  !-- calculating the interpolating points
     do k=1,nz
        call interpo_search_1d( val_f(1,:), z(k), inter_z(k), undeff=int(undef) )
     end do

  !-- calculating each value (temp, u, v, rh)
     do k=1,nz
        do j=2,col_num
           if(inter_z(k)/=int(undef).and.inter_z(k)/=nttmp)then
              if(val_f(j,inter_z(k))==undef.or.  &
  &              val_f(j,inter_z(k)+1)==undef)then
                 val_inter(:,k)=undef
                 fflag(k)=.true.
                 exit
              end if

              call interpolation_1d( val_f(1,inter_z(k):inter_z(k)+1),  &
  &                                  val_f(j,inter_z(k):inter_z(k)+1),  &
  &                                  z(k), val_inter(j,k) )
           else
              val_inter(j,k)=undef
              fflag(k)=.true.
           end if
        end do
     end do

  !-- calculating each value (pt, ept, sept)
     do k=1,nz
        if(val_inter(2,k)==undef)then
           val_inter(col_num+1,k)=undef
           val_inter(col_num+2,k)=undef
           val_inter(col_num+3,k)=undef
           val_inter(col_num+4,k)=undef
           fflag(k)=.true.
        else
           pt_tmp(1)=theta_dry( val_f(3,inter_z(k)), val_f(2,inter_z(k)) )
           ept_tmp(1)=thetae_Bolton( val_f(3,inter_z(k)),  &
  &                                  val_f(4,inter_z(k)),  &
  &                                  val_f(2,inter_z(k)) )
           sept_tmp(1)=thetaes_Bolton( val_f(3,inter_z(k)),  &
  &                                    val_f(2,inter_z(k)) )
           rh_tmp(1)=qvTP_2_RH( val_f(4,inter_z(k)),  &
  &                             val_f(3,inter_z(k)),  &
  &                             val_f(2,inter_z(k)) )
           pt_tmp(2)=theta_dry( val_f(3,inter_z(k)), val_f(2,inter_z(k)) )
           ept_tmp(2)=thetae_Bolton( val_f(3,inter_z(k)+1),  &
  &                                  val_f(4,inter_z(k)+1),  &
  &                                  val_f(2,inter_z(k)+1) )
           sept_tmp(2)=thetaes_Bolton( val_f(3,inter_z(k)+1),  &
  &                                    val_f(2,inter_z(k)+1) )
           rh_tmp(2)=qvTP_2_RH( val_f(4,inter_z(k)+1),  &
  &                             val_f(3,inter_z(k)+1),  &
  &                             val_f(2,inter_z(k)+1) )

           call interpolation_1d( val_f(1,inter_z(k):inter_z(k)+1),  &
  &                               pt_tmp, z(k), val_inter(col_num+1,k) )
           call interpolation_1d( val_f(1,inter_z(k):inter_z(k)+1),  &
  &                               ept_tmp, z(k), val_inter(col_num+2,k) )
           call interpolation_1d( val_f(1,inter_z(k):inter_z(k)+1),  &
  &                               sept_tmp, z(k), val_inter(col_num+3,k) )
           call interpolation_1d( val_f(1,inter_z(k):inter_z(k)+1),  &
  &                               rh_tmp, z(k), val_inter(col_num+4,k) )

        end if
     end do

     deallocate(val_w)
     deallocate(val_f)

     do j=1,col_num+4
        do k=1,nz
           val_tmp(j,i,k)=val_inter(j,k)
        end do
     end do

  end do

!-- 時間データをスペクトルデータに変換.

  do k=1,nz
     if(fflag(k).eqv..false.)then
        do j=1,col_num+4
           call r2c_ffttp_1d( nt, val_tmp(k,j,1:nt), val_fft(1:nt/2),  &
  &                           prim='x', prim_fact=prim,  &
  &                           omega_fix=omegar, omegan_fix=omeganr )
           do i=1,nt/2
              val_draw(j,i,k)=abs(val_fft(i))
           end do
        end do
     else
        do j=1,col_num+4
           do i=1,nt/2
              val_draw(j,i,k)=undef
           end do
        end do
     end if
  end do

!-- draw value separate
!-- 全て val_inter に落としておく.(上まだ未完全)

  select case (cont_val)
  case ('temp')
     cont_flag=3
  case ('qv')
     cont_flag=4
  case ('pt')
     cont_flag=col_num+1
  case ('ept')
     cont_flag=col_num+2
  case ('sept')
     cont_flag=col_num+3
  case ('rh')
     cont_flag=col_num+4
  case ('east')
     cont_flag=5
  case ('north')
     cont_flag=6
  end select

  select case (shade_val)
  case ('temp')
     shade_flag=3
  case ('qv')
     shade_flag=4
  case ('pt')
     shade_flag=col_num+1
  case ('ept')
     shade_flag=col_num+2
  case ('sept')
     shade_flag=col_num+3
  case ('rh')
     shade_flag=col_num+4
  case ('east')
     shade_flag=5
  case ('north')
     shade_flag=6
  end select

!-- DCL part
!-- まず, 高度-時系列断面
  call color_setting( snum, s_int, col_min=15, col_max=85,  &
  &                   min_tab=10999, max_tab=99999 )

  call DclOpenGraphics(IWS)

  if(len_trim(adjustl(sysfont))/=0)then
!     CALL SWSLFT(sysfont)
     CALL SWCSET('FONTNAME', sysfont)
  end if

  if(mean_flag.eqv..true.)then
     call Dcl_2D_Cont_Shade( trim(title_txt),  &
  &       freq, z,  &
  &       val_draw(cont_flag,:,:), val_draw(shade_flag,:,:),  &
  &       c_int, s_int,  &
  &       (/'Frequency (1/day)', 'altitude (m)'/),  &
  &       (/'(f6.1)', '(f6.1)'/),  &
  &       c_num=(/cnum, snum/), trigleg='a' )
  else
     call Dcl_2D_Cont_Shade( trim(title_txt),  &
  &       freq(2:nt/2), z,  &
  &       val_draw(cont_flag,2:nt/2,:), val_draw(shade_flag,2:nt/2,:),  &
  &       c_int, s_int,  &
  &       (/'Frequency (1/day)', 'altitude (m)'/),  &
  &       (/'(f6.1)', '(f6.1)'/),  &
  &       c_num=(/cnum, snum/), trigleg='a' )
  end if

  call DclCloseGraphics

end program
