!---------------------------------------------------------------
! Copyright (C) 2009-2015 GFD Dennou Club. All rights reserved.
!---------------------------------------------------------------

module ffts
! fft 関連のサブルーチン集
  use Math_Const

  public :: ffttp_1d, ffttp_2d, r2c_ffttp_1d

interface ffttp_1d

  module procedure ffttp_1df, ffttp_1dd

end interface ffttp_1d

interface ffttp_2d

  module procedure ffttp_2df, ffttp_2dd

end interface ffttp_2d

interface r2c_ffttp_1d

  module procedure r2c_ffttp_1df, r2c_ffttp_1dd

end interface r2c_ffttp_1d

interface c2r_ffttp_1d

  module procedure c2r_ffttp_1df, c2r_ffttp_1dd

end interface c2r_ffttp_1d

interface rotate_calc

  module procedure rotate_calc_f, rotate_calc_d

end interface rotate_calc

interface phase_velocity_binning

  module procedure phase_velocity_binning_f, phase_velocity_binning_d

end interface phase_velocity_binning


contains


subroutine ffttp_1df( nx, a, b, csign, prim, prim_fact, omega_fix, omegan_fix )
!  Temperton's FFT
!
!  1d の fft を csign をもとに, 正変換, 逆変換するルーチン
  use Math_Const
  implicit none
  integer, intent(in) :: nx  ! 入力配列の要素数
  complex, intent(in), dimension(0:nx-1) :: a  ! 入力配列
  complex, intent(inout), dimension(0:nx-1) :: b  ! 出力配列
  character(1), intent(in) :: csign  ! 正逆変換判定 [r=正変換, i=逆変換]
  character(1), intent(in), optional :: prim  ! 素因数分解をするかどうか
  ! [o=分解する, x=分解しない] default=分解しない. その場合は, 通常の DFT.
  ! 素因数分解する場合, nx=2^a*3^b*5^c*7^d までしか分解しないようにする.
  ! ただし, 'x' のときでも, prim_fact が設定されていれば, そのべきで FFT する.
  integer, dimension(5), intent(in), optional :: prim_fact  ! prim = x のとき設定すると
  ! そのべきで FFT を行う. prim_fact=(/a,b,c,d,e/) : 2^a*3^b*5^c*7^d*e
  complex, dimension(0:nx-1,0:nx-1), intent(in), optional :: omega_fix
                       ! 余素因数に対応する回転行列
  complex, dimension(0:nx-1,0:nx-1), intent(in), optional :: omegan_fix  ! 回転行列
  integer, allocatable, dimension(:) :: l, m, n  ! 要素等の作業用配列
  complex :: ctmp
  integer :: stat, counter, base
  integer :: i, j, k, id, jd, kd  ! do loop 用配列
  integer, parameter, dimension(4) :: prim_dim=(/2, 3, 5, 7/)  ! 素因数
  integer, dimension(4) :: prim_num  ! 各素因数のべき数
  complex, allocatable, dimension(:,:) :: omega
  complex, dimension(0:nx-1,0:nx-1) :: omegan
  complex, dimension(0:nx-1) :: c  ! tmp array
  complex, dimension(0:1,0:1) :: omega2  ! 波数 2 の回転行列
  complex, dimension(0:2,0:2) :: omega3  ! 波数 3 の回転行列
  complex, dimension(0:4,0:4) :: omega5  ! 波数 5 の回転行列
  complex, dimension(0:6,0:6) :: omega7  ! 波数 7 の回転行列


  base=nx
  prim_num=0
  counter=0

  do i=0,nx-1
     b(i)=a(i)
  end do

  if(real(romega2(1,1))/=-1.0.or.aimag(romega2(1,1))/=1.0.or.  &
  &  real(iomega2(1,1))/=-1.0.or.aimag(iomega2(1,1))/=1.0)then
     ! math_const の rotate_array ルーチンが計算されていない場合, これを作動させる.
     call rotate_array_f()
  end if

!-- 素因数分解する処理
  if(present(prim))then
     if(prim=='o')then
        if(present(prim_fact))then
           do i=1,4
!              if(prim_fact(i)==0)then
!                 prim_num(i)=1
!              else
                 prim_num(i)=prim_fact(i)
!              end if
              counter=counter+prim_fact(i)
           end do
           base=prim_fact(5)
        else
           call prim_calc( nx, prim_num, base )
           do i=1,4
              counter=counter+prim_num(i)
           end do
        end if
 
        if(base==1)then
           counter=counter-1
        end if
 
        if(counter/=0)then  ! prim=='o' であっても, 素因数分解できなければ DFT に送る.
           allocate(l(counter+1))
           allocate(m(counter+1))
           allocate(n(counter+1))
           stat=0
           l=0
           m=0
           n=0
   
           do i=1,4
              if(prim_num(i)/=0)then
                 select case(prim_dim(i))
                 case(2)
                    n(stat+1:stat+prim_num(i))=2
                 case(3)
                    n(stat+1:stat+prim_num(i))=3
                 case(5)
                    n(stat+1:stat+prim_num(i))=5
                 case(7)
                    n(stat+1:stat+prim_num(i))=7
                 end select
                 stat=stat+prim_num(i)
              end if
           end do
           if(base/=1)then
              n(counter+1)=base
           end if
        end if
     end if
  end if

!-- 回転行列を定義

  if(counter/=0)then
     select case (csign)
     case ('r')
        do j=0,1
           do i=0,1
              omega2(i,j)=romega2(i,j)
           end do
        end do
        do j=0,2
           do i=0,2
              omega3(i,j)=romega3(i,j)
           end do
        end do
        do j=0,4
           do i=0,4
              omega5(i,j)=romega5(i,j)
           end do
        end do
        do j=0,6
           do i=0,6
              omega7(i,j)=romega7(i,j)
           end do
        end do
     case ('i')
        do j=0,1
           do i=0,1
              omega2(i,j)=iomega2(i,j)
           end do
        end do
        do j=0,2
           do i=0,2
              omega3(i,j)=iomega3(i,j)
           end do
        end do
        do j=0,4
           do i=0,4
              omega5(i,j)=iomega5(i,j)
           end do
        end do
        do j=0,6
           do i=0,6
              omega7(i,j)=iomega7(i,j)
           end do
        end do
     end select
  end if

  if(present(omegan_fix))then
     allocate(omega(0:base-1,0:base-1))
     do j=0,base-1
        do i=0,base-1
           omega(i,j)=omega_fix(i,j)
        end do
     end do
     do j=0,nx-1
        do i=0,nx-1
           omegan(i,j)=omegan_fix(i,j)
        end do
     end do
  else
     allocate(omega(0:base-1,0:base-1))
     call rotate_calc_f( nx, csign, (/prim_num(1), prim_num(2), prim_num(3),  &
  &                      prim_num(4), base/), omega, omegan )
  end if

!-- FFT 開始

  if(counter/=0)then
!-- 係数行列定義
     m(1)=1
     l(1)=nx/(n(1)*m(1))
     do i=2,counter+1
        m(i)=m(i-1)*n(i-1)
        l(i)=nx/(n(i)*m(i))
     end do

!-- 変換行列 W の定義

     do kd=1,counter+1
        do jd=0,l(kd)-1
           do id=0,n(kd)-1
              do k=0,m(kd)-1
                 ctmp=b(jd*m(kd)+k)
                 do j=1,n(kd)-1
                    select case(n(kd))
                    case(2)
                       ctmp=ctmp+omega2(id,j)*b(j*l(kd)*m(kd)+jd*m(kd)+k)

                    case(3)
                       ctmp=ctmp+omega3(id,j)*b(j*l(kd)*m(kd)+jd*m(kd)+k)

                    case(5)
                       ctmp=ctmp+omega5(id,j)*b(j*l(kd)*m(kd)+jd*m(kd)+k)

                    case(7)
                       ctmp=ctmp+omega7(id,j)*b(j*l(kd)*m(kd)+jd*m(kd)+k)

                    case default
                       ctmp=ctmp+omega(id,j)*b(j*l(kd)*m(kd)+jd*m(kd)+k)

                    end select
                 end do
                 c(jd*n(kd)*m(kd)+id*m(kd)+k)=ctmp*omegan(m(kd),(id*jd))
              end do
           end do
        end do
        do id=0,nx-1
           b(id)=c(id)
        end do
     end do

  else

     do j=0,nx-1
        b(j)=a(0)
        do i=1,nx-1
           b(j)=b(j)+a(i)*omegan(i,j)
        end do
     end do

  end if

  if(csign=='r')then
     do j=0,nx-1
        b(j)=b(j)/real(nx)
     end do
  end if

end subroutine

!-----------------------------------------------

subroutine ffttp_1dd( nx, a, b, csign, prim, prim_fact, omega_fix, omegan_fix )
!  Temperton's FFT
!
!  1d の fft を csign をもとに, 正変換, 逆変換するルーチン
  use Math_Const
  implicit none
  integer, intent(in) :: nx  ! 入力配列の要素数
  complex(kind(0d0)), intent(in), dimension(0:nx-1) :: a  ! 入力配列
  complex(kind(0d0)), intent(inout), dimension(0:nx-1) :: b  ! 出力配列
  character(1), intent(in) :: csign  ! 正逆変換判定 [r=正変換, i=逆変換]
  character(1), intent(in), optional :: prim  ! 素因数分解をするかどうか
  ! [o=分解する, x=分解しない] default=分解しない. その場合は, 通常の DFT.
  ! 素因数分解する場合, nx=2^a*3^b*5^c*7^d までしか分解しないようにする.
  ! ただし, 'x' のときでも, prim_fact が設定されていれば, そのべきで FFT する.
  integer, dimension(5), intent(in), optional :: prim_fact  ! prim = x のとき設定すると
  ! そのべきで FFT を行う. prim_fact=(/a,b,c,d,e/) : 2^a*3^b*5^c*7^d*e
  complex(kind(0d0)), dimension(0:nx-1,0:nx-1), intent(in), optional :: omega_fix
                       ! 余素因数に対応する回転行列
  complex(kind(0d0)), dimension(0:nx-1,0:nx-1), intent(in), optional :: omegan_fix  ! 回転行列
  integer, allocatable, dimension(:) :: l, m, n  ! 要素等の作業用配列
  complex :: ctmp
  integer :: stat, counter, base
  integer :: i, j, k, id, jd, kd  ! do loop 用配列
  integer, parameter, dimension(4) :: prim_dim=(/2, 3, 5, 7/)  ! 素因数
  integer, dimension(4) :: prim_num  ! 各素因数のべき数
  complex(kind(0d0)), allocatable, dimension(:,:) :: omega
  complex(kind(0d0)), dimension(0:nx-1,0:nx-1) :: omegan
  complex(kind(0d0)), dimension(0:nx-1) :: c  ! tmp array
  complex(kind(0d0)), dimension(0:1,0:1) :: omega2  ! 波数 2 の回転行列
  complex(kind(0d0)), dimension(0:2,0:2) :: omega3  ! 波数 3 の回転行列
  complex(kind(0d0)), dimension(0:4,0:4) :: omega5  ! 波数 5 の回転行列
  complex(kind(0d0)), dimension(0:6,0:6) :: omega7  ! 波数 7 の回転行列


  base=nx
  prim_num=0
  counter=0

  do i=0,nx-1
     b(i)=a(i)
  end do

  if(dble(romega2_cdp(1,1))/=-1.0d0.or.dimag(romega2_cdp(1,1))/=1.0d0.or.  &
  &  dble(iomega2_cdp(1,1))/=-1.0d0.or.dimag(iomega2_cdp(1,1))/=1.0d0)then
     ! math_const の rotate_array ルーチンが計算されていない場合, これを作動させる.
     call rotate_array_d()
  end if

!-- 素因数分解する処理
  if(present(prim))then
     if(prim=='o')then
        if(present(prim_fact))then
           do i=1,4
!              if(prim_fact(i)==0)then
!                 prim_num(i)=1
!              else
                 prim_num(i)=prim_fact(i)
!              end if
              counter=counter+prim_fact(i)
           end do
           base=prim_fact(5)
        else
           call prim_calc( nx, prim_num, base )
           do i=1,4
              counter=counter+prim_num(i)
           end do
        end if
 
        if(base==1)then
           counter=counter-1
        end if
 
        if(counter/=0)then  ! prim=='o' であっても, 素因数分解できなければ DFT に送る.
           allocate(l(counter+1))
           allocate(m(counter+1))
           allocate(n(counter+1))
           stat=0
           l=0
           m=0
           n=0
   
           do i=1,4
              if(prim_num(i)/=0)then
                 select case(prim_dim(i))
                 case(2)
                    n(stat+1:stat+prim_num(i))=2
                 case(3)
                    n(stat+1:stat+prim_num(i))=3
                 case(5)
                    n(stat+1:stat+prim_num(i))=5
                 case(7)
                    n(stat+1:stat+prim_num(i))=7
                 end select
                 stat=stat+prim_num(i)
              end if
           end do
           if(base/=1)then
              n(counter+1)=base
           end if
        end if
     end if
  end if

!-- 回転行列を定義

  if(counter/=0)then
     select case (csign)
     case ('r')
        do j=0,1
           do i=0,1
              omega2(i,j)=romega2_cdp(i,j)
           end do
        end do
        do j=0,2
           do i=0,2
              omega3(i,j)=romega3_cdp(i,j)
           end do
        end do
        do j=0,4
           do i=0,4
              omega5(i,j)=romega5_cdp(i,j)
           end do
        end do
        do j=0,6
           do i=0,6
              omega7(i,j)=romega7_cdp(i,j)
           end do
        end do
     case ('i')
        do j=0,1
           do i=0,1
              omega2(i,j)=iomega2_cdp(i,j)
           end do
        end do
        do j=0,2
           do i=0,2
              omega3(i,j)=iomega3_cdp(i,j)
           end do
        end do
        do j=0,4
           do i=0,4
              omega5(i,j)=iomega5_cdp(i,j)
           end do
        end do
        do j=0,6
           do i=0,6
              omega7(i,j)=iomega7_cdp(i,j)
           end do
        end do
     end select
  end if

  if(present(omegan_fix))then
     allocate(omega(0:base-1,0:base-1))
     do j=0,base-1
        do i=0,base-1
           omega(i,j)=omega_fix(i,j)
        end do
     end do
     do j=0,nx-1
        do i=0,nx-1
           omegan(i,j)=omegan_fix(i,j)
        end do
     end do
  else
     allocate(omega(0:base-1,0:base-1))
     call rotate_calc_d( nx, csign, (/prim_num(1), prim_num(2), prim_num(3),  &
  &                      prim_num(4), base/), omega, omegan )
  end if

!-- FFT 開始

  if(counter/=0)then
!-- 係数行列定義
     m(1)=1
     l(1)=nx/(n(1)*m(1))
     do i=2,counter+1
        m(i)=m(i-1)*n(i-1)
        l(i)=nx/(n(i)*m(i))
     end do

!-- 変換行列 W の定義

     do kd=1,counter+1
        do jd=0,l(kd)-1
           do id=0,n(kd)-1
              do k=0,m(kd)-1
                 ctmp=b(jd*m(kd)+k)
                 do j=1,n(kd)-1
                    select case(n(kd))
                    case(2)
                       ctmp=ctmp+omega2(id,j)*b(j*l(kd)*m(kd)+jd*m(kd)+k)

                    case(3)
                       ctmp=ctmp+omega3(id,j)*b(j*l(kd)*m(kd)+jd*m(kd)+k)

                    case(5)
                       ctmp=ctmp+omega5(id,j)*b(j*l(kd)*m(kd)+jd*m(kd)+k)

                    case(7)
                       ctmp=ctmp+omega7(id,j)*b(j*l(kd)*m(kd)+jd*m(kd)+k)

                    case default
                       ctmp=ctmp+omega(id,j)*b(j*l(kd)*m(kd)+jd*m(kd)+k)

                    end select
                 end do
                 c(jd*n(kd)*m(kd)+id*m(kd)+k)=ctmp*omegan(m(kd),(id*jd))
              end do
           end do
        end do
        do id=0,nx-1
           b(id)=c(id)
        end do
     end do

  else

     do j=0,nx-1
        b(j)=a(0)
        do i=1,nx-1
           b(j)=b(j)+a(i)*omegan(i,j)
        end do
     end do

  end if

  if(csign=='r')then
     do j=0,nx-1
        b(j)=b(j)/dble(nx)
     end do
  end if

end subroutine

!-----------------------------------------------

subroutine ffttp_2df( nx, ny, a, b, csign, prim, prim_factx, prim_facty,  &
  &                   omegax_fix, omegaxn_fix, omegay_fix, omegayn_fix )
!  Temperton's FFT (2d ver)
!
!  2d の fft を csign をもとに, 正変換, 逆変換するルーチン
  use Math_Const
  implicit none
  integer, intent(in) :: nx  ! 入力配列の要素数 1
  integer, intent(in) :: ny  ! 入力配列の要素数 2
  complex, intent(in), dimension(0:nx-1,0:ny-1) :: a  ! 入力配列
  complex, intent(inout), dimension(0:nx-1,0:ny-1) :: b  ! 出力配列
  character(1), intent(in) :: csign  ! 正逆変換判定 [r=正変換, i=逆変換]
  character(1), optional, intent(in) :: prim  ! 素因数分解をするかどうか
  ! [o=分解する, x=分解しない] default=分解しない. その場合は, 通常の DFT.
  ! 素因数分解する場合, nx=2^a*3^b*5^c*7^d までしか分解しないようにする.
  ! ただし, 'x' のときでも, prim_fact が設定されていれば, そのべきで FFT する.
  integer, dimension(5), optional :: prim_factx  ! prim = x のとき設定すると
  ! そのべきで FFT を行う. prim_fact=(/a,b,c,d/) : 2^a*3^b*5^c*7^d
  ! ここで, prim_fact で設定するべき数は nx 方向のべき数であることに注意.
  integer, dimension(5), optional :: prim_facty  ! prim_factx と同じ.
  complex, dimension(0:nx-1,0:nx-1), intent(in), optional :: omegax_fix   ! 余素因数に対応する回転行列
  complex, dimension(0:nx-1,0:nx-1), intent(in), optional :: omegaxn_fix  ! 回転行列
  complex, dimension(0:ny-1,0:ny-1), intent(in), optional :: omegay_fix   ! 余素因数に対応する回転行列
  complex, dimension(0:ny-1,0:ny-1), intent(in), optional :: omegayn_fix  ! 回転行列
  complex, dimension(0:nx-1,0:ny-1) :: c
  integer :: i

  if(present(prim))then
     if(present(prim_factx))then
        if(present(omegaxn_fix))then

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i)

           do i=0,ny-1
              call ffttp_1df( nx, a(0:nx-1,i), c(0:nx-1,i), csign, prim,  &
  &                           prim_factx, omegax_fix(0:nx-1,0:nx-1),  &
  &                           omegaxn_fix(0:nx-1,0:nx-1) )
           end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i)

           do i=0,nx-1
              call ffttp_1df( ny, c(i,0:ny-1), b(i,0:ny-1), csign, prim,  &
  &                           prim_facty, omegay_fix(0:ny-1,0:ny-1),  &
  &                           omegayn_fix(0:ny-1,0:ny-1) )
           end do

!$omp end do
!$omp end parallel

        else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i)

           do i=0,ny-1
              call ffttp_1df( nx, a(0:nx-1,i), c(0:nx-1,i), csign, prim, prim_factx )
           end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i)

           do i=0,nx-1
              call ffttp_1df( ny, c(i,0:ny-1), b(i,0:ny-1), csign, prim, prim_facty )
           end do

!$omp end do
!$omp end parallel

        end if
     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i)

        do i=0,ny-1
           call ffttp_1df( nx, a(0:nx-1,i), c(0:nx-1,i), csign, prim )
        end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i)

        do i=0,nx-1
           call ffttp_1df( ny, c(i,0:ny-1), b(i,0:ny-1), csign, prim )
        end do

!$omp end do
!$omp end parallel

     end if
  else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i)

     do i=0,ny-1
        call ffttp_1df( nx, a(0:nx-1,i), c(0:nx-1,i), csign )
     end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i)

     do i=0,nx-1
        call ffttp_1df( ny, c(i,0:ny-1), b(i,0:ny-1), csign )
     end do

!$omp end do
!$omp end parallel

  end if

end subroutine

!-----------------------------------------------

subroutine ffttp_2dd( nx, ny, a, b, csign, prim, prim_factx, prim_facty,  &
  &                   omegax_fix, omegaxn_fix, omegay_fix, omegayn_fix )
!  Temperton's FFT (2d ver)
!
!  2d の fft を csign をもとに, 正変換, 逆変換するルーチン
  use Math_Const
  implicit none
  integer, intent(in) :: nx  ! 入力配列の要素数 1
  integer, intent(in) :: ny  ! 入力配列の要素数 2
  complex(kind(0d0)), intent(in), dimension(0:nx-1,0:ny-1) :: a  ! 入力配列
  complex(kind(0d0)), intent(inout), dimension(0:nx-1,0:ny-1) :: b  ! 出力配列
  character(1), intent(in) :: csign  ! 正逆変換判定 [r=正変換, i=逆変換]
  character(1), optional, intent(in) :: prim  ! 素因数分解をするかどうか
  ! [o=分解する, x=分解しない] default=分解しない. その場合は, 通常の DFT.
  ! 素因数分解する場合, nx=2^a*3^b*5^c*7^d までしか分解しないようにする.
  ! ただし, 'x' のときでも, prim_fact が設定されていれば, そのべきで FFT する.
  integer, dimension(5), optional :: prim_factx  ! prim = x のとき設定すると
  ! そのべきで FFT を行う. prim_fact=(/a,b,c,d/) : 2^a*3^b*5^c*7^d
  ! ここで, prim_fact で設定するべき数は nx 方向のべき数であることに注意.
  integer, dimension(5), optional :: prim_facty  ! prim_factx と同じ.
  complex(kind(0d0)), dimension(0:nx-1,0:nx-1), intent(in), optional :: omegax_fix   ! 余素因数に対応する回転行列
  complex(kind(0d0)), dimension(0:nx-1,0:nx-1), intent(in), optional :: omegaxn_fix  ! 回転行列
  complex(kind(0d0)), dimension(0:ny-1,0:ny-1), intent(in), optional :: omegay_fix   ! 余素因数に対応する回転行列
  complex(kind(0d0)), dimension(0:ny-1,0:ny-1), intent(in), optional :: omegayn_fix  ! 回転行列
  complex(kind(0d0)), dimension(0:nx-1,0:ny-1) :: c
  integer :: i

  if(present(prim))then
     if(present(prim_factx))then
        if(present(omegaxn_fix))then

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i)

           do i=0,ny-1
              call ffttp_1dd( nx, a(0:nx-1,i), c(0:nx-1,i), csign, prim,  &
  &                          prim_factx, omegax_fix(0:nx-1,0:nx-1),  &
  &                          omegaxn_fix(0:nx-1,0:nx-1) )
           end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i)

           do i=0,nx-1
              call ffttp_1dd( ny, c(i,0:ny-1), b(i,0:ny-1), csign, prim,  &
  &                          prim_facty, omegay_fix(0:ny-1,0:ny-1),  &
  &                          omegayn_fix(0:ny-1,0:ny-1) )
           end do

!$omp end do
!$omp end parallel

        else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i)

           do i=0,ny-1
              call ffttp_1dd( nx, a(0:nx-1,i), c(0:nx-1,i), csign, prim, prim_factx )
           end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i)

           do i=0,nx-1
              call ffttp_1dd( ny, c(i,0:ny-1), b(i,0:ny-1), csign, prim, prim_facty )
           end do

!$omp end do
!$omp end parallel

        end if
     else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i)

        do i=0,ny-1
           call ffttp_1dd( nx, a(0:nx-1,i), c(0:nx-1,i), csign, prim )
        end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i)

        do i=0,nx-1
           call ffttp_1dd( ny, c(i,0:ny-1), b(i,0:ny-1), csign, prim )
        end do

!$omp end do
!$omp end parallel

     end if
  else

!$omp parallel default(shared)
!$omp do schedule(runtime) private(i)

     do i=0,ny-1
        call ffttp_1dd( nx, a(0:nx-1,i), c(0:nx-1,i), csign )
     end do

!$omp end do
!$omp barrier
!$omp do schedule(runtime) private(i)

     do i=0,nx-1
        call ffttp_1dd( ny, c(i,0:ny-1), b(i,0:ny-1), csign )
     end do

!$omp end do
!$omp end parallel

  end if

end subroutine

!-----------------------------------------------

subroutine r2c_ffttp_1df( nx, a, b, prim, prim_fact, omega_fix, omegan_fix )
! 1 次元実 FFT 計算ルーチン
!
! 入力配列は実数で行い, 複素数配列を返す.
  use Math_Const
  implicit none
  integer, intent(in) :: nx  ! 1 次元データ要素数
  real, intent(in), dimension(0:nx-1) :: a  ! 入力実数データ配列
  complex, intent(inout), dimension(0:nx/2-1) :: b  ! 出力複素数データ配列
  character(1), optional, intent(in) :: prim  ! 素因数分解をするかどうか
  ! [o=分解する, x=分解しない] default=分解しない. その場合は, 通常の DFT.
  ! 素因数分解する場合, nx=2^a*3^b*5^c*7^d までしか分解しないようにする.
  ! なお, 実 FFT 変換は現在, データ要素数が偶数の変換しか行わないように
  ! 実装されていることに注意 (奇数データで変換しようとするとエラーを返す).
  ! また, inout 属性の引数は, データ数が半分になっていることに注意.
  ! 仕様として, nx/2 番目の実数データは, n=0 番目の虚部に格納されて
  ! 出力配列に渡されることに注意.
  ! ただし, 'x' のときでも, prim_fact が設定されていれば, そのべきで FFT する.
  integer, dimension(5), intent(in), optional :: prim_fact  ! prim = x のとき設定すると
  ! そのべきで FFT を行う. prim_fact=(/a,b,c,d/) : 2^a*3^b*5^c*7^d
  ! ここで入力するべき数は nx のべき数である.
  complex, dimension(0:nx/2-1,0:nx/2-1), intent(in), optional :: omega_fix  ! 余素因数に対応する回転行列
  complex, dimension(0:nx/2-1,0:nx/2-1), intent(in), optional :: omegan_fix  ! 回転行列
  complex, dimension(0:nx/2-1) :: c, d
  integer :: i
  integer, dimension(5) :: prim_num

  if(mod(nx,2)/=0)then
     write(*,*) "*** ERROR ***"
     write(*,*) "nx must be even number. Stop."
     stop
  end if

!-- nx 個の実数データを nx/2 個の複素数データに置き換える.
  do i=0,nx/2-1
     c(i)=a(2*i)+img*a(2*i+1)
  end do

!-- FFT 開始

  if(present(prim))then
     if(present(prim_fact))then
        prim_num=prim_fact
        prim_num(1)=prim_num(1)-1  ! データの要素数が半分になっているため,
                                   ! 2 のべき数は 1 つ減っている.
        if(present(omegan_fix))then
           call ffttp_1df( nx/2, c, d, 'r', prim, prim_num,  &
  &                       omega_fix(0:nx/2-1,0:nx/2-1),  &
  &                       omegan_fix(0:nx/2-1,0:nx/2-1) )
        else
           call ffttp_1df( nx/2, c, d, 'r', prim, prim_num )
        end if
     else
        call ffttp_1df( nx/2, c, d, 'r', prim )
     end if
  else
     call ffttp_1df( nx/2, c, d, 'r' )
  end if

!-- 変換後の配列を整理
!-- b(N) の実部は b(0) の虚部に組み込むことにする.
!-- b(k) の計算で 0.25 をかけるのは, 上の fft で 2/N で規格化しており,
!-- もとの計算では, 1/2N で規格化しなければならないので,
!-- 1/4 をかけることで, 2/N -> 1/2N で規格化したことになる.
!-- b(0) にかかっている係数もその類.
  b(0)=0.5*((real(d(0))+aimag(d(0)))+img*(real(d(0))-aimag(d(0))))
  do i=1,nx/2-1
     b(i)=0.25*((conjg(d(nx/2-i))+d(i))  &
  &       -(sin(2.0*pi*real(i)/real(nx))+img*cos(2.0*pi*real(i)/real(nx)))  &
  &       *(d(i)-conjg(d(nx/2-i))))
  end do

end subroutine

!-----------------------------------------------

subroutine r2c_ffttp_1dd( nx, a, b, prim, prim_fact, omega_fix, omegan_fix )
! 1 次元実 FFT 計算ルーチン
!
! 入力配列は実数で行い, 複素数配列を返す.
  use Math_Const
  implicit none
  integer, intent(in) :: nx  ! 1 次元データ要素数
  double precision, intent(in), dimension(0:nx-1) :: a  ! 入力実数データ配列
  complex(kind(0d0)), intent(inout), dimension(0:nx/2-1) :: b  ! 出力複素数データ配列
  character(1), optional, intent(in) :: prim  ! 素因数分解をするかどうか
  ! [o=分解する, x=分解しない] default=分解しない. その場合は, 通常の DFT.
  ! 素因数分解する場合, nx=2^a*3^b*5^c*7^d までしか分解しないようにする.
  ! なお, 実 FFT 変換は現在, データ要素数が偶数の変換しか行わないように
  ! 実装されていることに注意 (奇数データで変換しようとするとエラーを返す).
  ! また, inout 属性の引数は, データ数が半分になっていることに注意.
  ! 仕様として, nx/2 番目の実数データは, n=0 番目の虚部に格納されて
  ! 出力配列に渡されることに注意.
  ! ただし, 'x' のときでも, prim_fact が設定されていれば, そのべきで FFT する.
  integer, dimension(5), intent(in), optional :: prim_fact  ! prim = x のとき設定すると
  ! そのべきで FFT を行う. prim_fact=(/a,b,c,d/) : 2^a*3^b*5^c*7^d
  ! ここで入力するべき数は nx のべき数である.
  complex(kind(0d0)), dimension(0:nx/2-1,0:nx/2-1), intent(in), optional :: omega_fix  ! 余素因数に対応する回転行列
  complex(kind(0d0)), dimension(0:nx/2-1,0:nx/2-1), intent(in), optional :: omegan_fix  ! 回転行列
  complex(kind(0d0)), dimension(0:nx/2-1) :: c, d
  integer :: i
  integer, dimension(5) :: prim_num

  if(mod(nx,2)/=0)then
     write(*,*) "*** ERROR ***"
     write(*,*) "nx must be even number. Stop."
     stop
  end if

!-- nx 個の実数データを nx/2 個の複素数データに置き換える.
  do i=0,nx/2-1
     c(i)=a(2*i)+img_cdp*a(2*i+1)
  end do

!-- FFT 開始

  if(present(prim))then
     if(present(prim_fact))then
        prim_num=prim_fact
        prim_num(1)=prim_num(1)-1  ! データの要素数が半分になっているため,
                                   ! 2 のべき数は 1 つ減っている.
        if(present(omegan_fix))then
           call ffttp_1dd( nx/2, c, d, 'r', prim, prim_num,  &
  &                       omega_fix(0:nx/2-1,0:nx/2-1),  &
  &                       omegan_fix(0:nx/2-1,0:nx/2-1) )
        else
           call ffttp_1dd( nx/2, c, d, 'r', prim, prim_num )
        end if
     else
        call ffttp_1dd( nx/2, c, d, 'r', prim )
     end if
  else
     call ffttp_1dd( nx/2, c, d, 'r' )
  end if

!-- 変換後の配列を整理
!-- b(N) の実部は b(0) の虚部に組み込むことにする.
!-- b(k) の計算で 0.25 をかけるのは, 上の fft で 2/N で規格化しており,
!-- もとの計算では, 1/2N で規格化しなければならないので,
!-- 1/4 をかけることで, 2/N -> 1/2N で規格化したことになる.
!-- b(0) にかかっている係数もその類.
  b(0)=0.5d0*((dble(d(0))+dimag(d(0)))+img_cdp*(dble(d(0))-dimag(d(0))))
  do i=1,nx/2-1
     b(i)=0.25d0*((dconjg(d(nx/2-i))+d(i))  &
  &       -(dsin(2.0d0*pi_dp*dble(i)/dble(nx))  &
  &       +img_cdp*dcos(2.0d0*pi_dp*dble(i)/dble(nx)))  &
  &       *(d(i)-dconjg(d(nx/2-i))))
  end do

end subroutine

!-----------------------------------------------

subroutine c2r_ffttp_1df( nx, a, b, prim, prim_fact, omega_fix, omegan_fix )
! 1 次元実 FFT 逆変換計算ルーチン
!
! 入力配列は複素数数で行い,実数配列を返す.
  use Math_Const
  implicit none
  integer, intent(in) :: nx  ! 1 次元データ要素数
  complex, intent(in), dimension(0:nx/2-1) :: a  ! 入力複素数データ配列
  real, intent(inout), dimension(0:nx-1) :: b  ! 出力実数データ配列
  character(1), optional, intent(in) :: prim  ! 素因数分解をするかどうか
  ! [o=分解する, x=分解しない] default=分解しない. その場合は, 通常の DFT.
  ! 素因数分解する場合, nx=2^a*3^b*5^c*7^d までしか分解しないようにする.
  ! なお, 実 FFT 変換は現在, データ要素数が偶数の変換しか行わないように
  ! 実装されていることに注意 (奇数データで変換しようとするとエラーを返す).
  ! また, in 属性の引数は, データ数が半分になっていることに注意.
  ! 仕様として, 入力引数の n=0 虚部に nx/2 番目の実数データが入っているように
  ! データを渡す.
  ! ただし, 'x' のときでも, prim_fact が設定されていれば, そのべきで FFT する.
  integer, intent(in), optional, dimension(5) :: prim_fact  ! prim = x のとき設定すると
  ! そのべきで FFT を行う. prim_fact=(/a,b,c,d/) : 2^a*3^b*5^c*7^d
  complex, dimension(0:nx/2-1,0:nx/2-1), intent(in), optional :: omega_fix  ! 余素因数に対応する回転行列
  complex, dimension(0:nx/2-1,0:nx/2-1), intent(in), optional :: omegan_fix  ! 回転行列
  complex, dimension(0:nx/2-1) :: c, d
  integer :: i
  integer, dimension(5) :: prim_num

  if(mod(nx,2)/=0)then
     write(*,*) "*** ERROR ***"
     write(*,*) "nx must be even number. Stop."
     stop
  end if

!-- nx/2 個の複素数データ（独立なフーリエ係数）を同数の複素数データに置き換える.
  c(0)=a(0)
  do i=1,nx/2-1
     c(i)=(a(i)+conjg(a(nx/2-i)))  &
  &       +(img*cos(2.0*pi*i/real(nx))-sin(2.0*pi*i/real(nx)))  &
  &       *(a(i)-conjg(a(nx/2-i)))
  end do

!-- FFT 開始

  if(present(prim))then
     if(present(prim_fact))then
        prim_num=prim_fact
        prim_num(1)=prim_num(1)-1    ! データの要素数が半分になっているため,
                                     ! 2 のべき数は 1 つ減っている.
        if(present(omegan_fix))then
           call ffttp_1df( nx/2, c, d, 'i', prim, prim_num,  &
  &                       omega_fix(0:nx/2-1,0:nx/2-1),  &
  &                       omegan_fix(0:nx/2-1,0:nx/2-1) )
        else
           call ffttp_1df( nx/2, c, d, 'i', prim, prim_num )
        end if
     else
        call ffttp_1df( nx/2, c, d, 'i', prim )
     end if
  else
     call ffttp_1df( nx/2, c, d, 'i' )
  end if

!-- 変換後の配列を整理
  do i=0,nx/2-1
     b(2*i)=real(d(i))
     b(2*i+1)=aimag(d(i))
  end do

end subroutine

!-----------------------------------------------

subroutine c2r_ffttp_1dd( nx, a, b, prim, prim_fact, omega_fix, omegan_fix )
! 1 次元実 FFT 逆変換計算ルーチン
!
! 入力配列は複素数数で行い,実数配列を返す.
  use Math_Const
  implicit none
  integer, intent(in) :: nx  ! 1 次元データ要素数
  complex(kind(0d0)), intent(in), dimension(0:nx/2-1) :: a  ! 入力複素数データ配列
  double precision, intent(inout), dimension(0:nx-1) :: b  ! 出力実数データ配列
  character(1), optional, intent(in) :: prim  ! 素因数分解をするかどうか
  ! [o=分解する, x=分解しない] default=分解しない. その場合は, 通常の DFT.
  ! 素因数分解する場合, nx=2^a*3^b*5^c*7^d までしか分解しないようにする.
  ! なお, 実 FFT 変換は現在, データ要素数が偶数の変換しか行わないように
  ! 実装されていることに注意 (奇数データで変換しようとするとエラーを返す).
  ! また, in 属性の引数は, データ数が半分になっていることに注意.
  ! 仕様として, 入力引数の n=0 虚部に nx/2 番目の実数データが入っているように
  ! データを渡す.
  ! ただし, 'x' のときでも, prim_fact が設定されていれば, そのべきで FFT する.
  integer, intent(in), optional, dimension(5) :: prim_fact  ! prim = x のとき設定すると
  ! そのべきで FFT を行う. prim_fact=(/a,b,c,d/) : 2^a*3^b*5^c*7^d
  complex(kind(0d0)), dimension(0:nx/2-1,0:nx/2-1), intent(in), optional :: omega_fix  ! 余素因数に対応する回転行列
  complex(kind(0d0)), dimension(0:nx/2-1,0:nx/2-1), intent(in), optional :: omegan_fix  ! 回転行列
  complex(kind(0d0)), dimension(0:nx/2-1) :: c, d
  integer :: i
  integer, dimension(5) :: prim_num

  if(mod(nx,2)/=0)then
     write(*,*) "*** ERROR ***"
     write(*,*) "nx must be even number. Stop."
     stop
  end if

!-- nx/2 個の複素数データ（独立なフーリエ係数）を同数の複素数データに置き換える.
  c(0)=a(0)
  do i=1,nx/2-1
     c(i)=(a(i)+dconjg(a(nx/2-i)))  &
  &       +(img_cdp*dcos(2.0d0*pi_dp*dble(i)/dble(nx))  &
  &       -dsin(2.0d0*pi_dp*dble(i)/dble(nx)))  &
  &       *(a(i)-dconjg(a(nx/2-i)))
  end do

!-- FFT 開始

  if(present(prim))then
     if(present(prim_fact))then
        prim_num=prim_fact
        prim_num(1)=prim_num(1)-1    ! データの要素数が半分になっているため,
                                     ! 2 のべき数は 1 つ減っている.
        if(present(omegan_fix))then
           call ffttp_1dd( nx/2, c, d, 'i', prim, prim_num,  &
  &                        omega_fix(0:nx/2-1,0:nx/2-1),  &
  &                        omegan_fix(0:nx/2-1,0:nx/2-1) )
        else
           call ffttp_1dd( nx/2, c, d, 'i', prim, prim_num )
        end if
     else
        call ffttp_1dd( nx/2, c, d, 'i', prim )
     end if
  else
     call ffttp_1dd( nx/2, c, d, 'i' )
  end if

!-- 変換後の配列を整理
  do i=0,nx/2-1
     b(2*i)=dble(d(i))
     b(2*i+1)=dimag(d(i))
  end do

end subroutine

!-----------------------------------------------

subroutine phase_velocity_binning_f( km, freq, ival, vbin, bin_val, undef,  &
  &                                  mean_flag, krange, frange )
! 時系列 FFT を行ったスペクトルデータ ival に対して, 
! vbin で区切られた速度ごとに, 分散関係に基づく位相速度の
! ビンに分ける. 
! 
! 手法としては, スペクトルデータは波数方向 1 次元 (k),
! 周波数方向 1 次元 (f) で構成されているので,
! ある波数 (K), 周波数 (F) におけるスペクトルがもつ
! 位相速度は F/K となる. よって, 全スペクトルデータについて
! 位相速度を計算し, その位相速度が vbin_ のどこに入るかを
! ソートする. ソート時にそのスペクトルデータで重みづけ
! してソートするので, スペクトル強度が大きい位相速度を持つ
! グループ (ビン) は頻度が大きくなる. 
!
! 位相速度の単位は km, freq の単位で決まる.
! もし, km が [1/m] で freq が [1/s] の単位を持っていれば,
! 位相速度 vbin の値は m/s でソートされる.
!
! krange, frange が指定された場合, これらの値を長軸・短軸として
! k-f 空間上で楕円方程式を用い, この範囲内に存在する k,f 点のみ
! ビン分けを行う. 
!
! mean_flag が指定された場合, ビン分けしたスペクトルは総和ではなく
! 平均値で返される. 

  implicit none

  real, intent(in) :: km(:)  ! 波数の値
  real, intent(in) :: freq(:)  ! 周波数の値
  real, intent(in) :: ival(size(km),size(freq))  ! 元データ
  real, intent(in) :: vbin(:)
                    ! 位相速度のビン [freq/km]
  real, intent(inout) :: bin_val(size(vbin)+1)
                    ! 位相速度のビンで分けられたスペクトル強度 [ival と同じ]
  real, intent(in), optional :: undef  ! 未定義値
  logical, intent(in), optional :: mean_flag  ! 平均値として返す.
                                              ! デフォルト: .false.
  real, intent(in), optional :: krange  ! ビン分けする閾値波数
  real, intent(in), optional :: frange  ! ビン分けする閾値周波数

  integer :: ii, jj, kk, nk, nf, nbin
  integer :: ncount(size(vbin)+1)
  real :: dundef
  real :: velval(size(km),size(freq))
  logical :: m_flag, range_flag

  nk=size(km)
  nf=size(freq)
  nbin=size(vbin)
  ncount=0
  bin_val=0.0

  if(present(undef))then
     dundef=undef
  else
     dundef=-999.0
  end if

  if(present(mean_flag))then
     m_flag=mean_flag
  else
     m_flag=.false.
  end if

  range_flag=.false.
  if(present(krange))then
     range_flag=.true.
     if(present(frange).eqv..false.)then
        write(*,*) "*** ERROR (phase_velocity_binning) ***: krange is set, but frange is not set."
        stop
     end if
  end if

!-- 各 km, freq に対応する位相速度の計算

  if(range_flag.eqv..false.)then
     do jj=1,nf
        do ii=1,nk
           if(km(ii)==0.0)then
              velval(ii,jj)=dundef
           else
              velval(ii,jj)=freq(jj)/km(ii)
           end if
        end do
     end do
  else
     do jj=1,nf
        do ii=1,nk
           if(km(ii)==0.0)then
              velval(ii,jj)=dundef
           else if((km(ii)/krange)**2+(freq(jj)/frange)**2<=1.0)then
              velval(ii,jj)=freq(jj)/km(ii)
           else
              velval(ii,jj)=dundef
           end if
        end do
     end do
  end if

!-- 各 km, freq におけるスペクトル強度で重みをつけて
!-- 位相速度をビンに分ける.

  do jj=1,nf
     do ii=1,nk
        if(ival(ii,jj)/=dundef.and.ival(ii,jj)/=0.0)then
           if(vbin(1)>velval(ii,jj))then
              bin_val(1)=bin_val(1)+ival(ii,jj)
              ncount(1)=ncount(1)+1
           else if(vbin(nbin)<velval(ii,jj))then
              bin_val(nbin+1)=bin_val(nbin+1)+ival(ii,jj)
              ncount(nbin+1)=ncount(nbin+1)+1
           else
              do kk=1,nbin-1
                 if(vbin(kk)<velval(ii,jj).and.vbin(kk+1)>velval(ii,jj))then
                    bin_val(kk+1)=bin_val(kk+1)+ival(ii,jj)
                    ncount(kk+1)=ncount(kk+1)+1
                    exit
                 else if(vbin(kk)==velval(ii,jj))then
                    bin_val(kk)=bin_val(kk)+0.5*ival(ii,jj)
                    bin_val(kk+1)=bin_val(kk+1)+0.5*ival(ii,jj)
                    ncount(kk)=ncount(kk)+1
                    ncount(kk+1)=ncount(kk+1)+1
                    exit
                 end if
              end do
              if(vbin(nbin)==velval(ii,jj))then
                 bin_val(nbin)=bin_val(nbin)+0.5*ival(ii,jj)
                 bin_val(nbin+1)=bin_val(nbin+1)+0.5*ival(ii,jj)
                 ncount(nbin)=ncount(nbin)+1
                 ncount(nbin+1)=ncount(nbin+1)+1
              end if
           end if
        end if
     end do
  end do

!-- 各位相速度ビンの総和数で割って平均として返す.

  if(m_flag.eqv..true.)then
     do kk=1,nbin+1
        if(ncount(kk)>0)then
           bin_val(kk)=bin_val(kk)/real(ncount(kk))
        else
           bin_val(kk)=0.0
        end if
     end do
  end if

end subroutine phase_velocity_binning_f

!-----------------------------------------------

subroutine phase_velocity_binning_d( km, freq, ival, vbin, bin_val, undef,  &
  &                                  mean_flag, krange, frange )
! 時系列 FFT を行ったスペクトルデータ ival に対して, 
! vbin で区切られた速度ごとに, 分散関係に基づく位相速度の
! ビンに分ける. 
! 
! 手法としては, スペクトルデータは波数方向 1 次元 (k),
! 周波数方向 1 次元 (f) で構成されているので,
! ある波数 (K), 周波数 (F) におけるスペクトルがもつ
! 位相速度は F/K となる. よって, 全スペクトルデータについて
! 位相速度を計算し, その位相速度が vbin_ のどこに入るかを
! ソートする. ソート時にそのスペクトルデータで重みづけ
! してソートするので, スペクトル強度が大きい位相速度を持つ
! グループ (ビン) は頻度が大きくなる. 
!
! 位相速度の単位は km, freq の単位で決まる.
! もし, km が [1/m] で freq が [1/s] の単位を持っていれば,
! 位相速度 vbin の値は m/s でソートされる.
!
! krange, frange が指定された場合, これらの値を長軸・短軸として
! k-f 空間上で楕円方程式を用い, この範囲内に存在する k,f 点のみ
! ビン分けを行う. 
!
! mean_flag が指定された場合, ビン分けしたスペクトルは総和ではなく
! 平均値で返される. 

  implicit none

  double precision, intent(in) :: km(:)  ! 波数の値
  double precision, intent(in) :: freq(:)  ! 周波数の値
  double precision, intent(in) :: ival(size(km),size(freq))  ! 元データ
  double precision, intent(in) :: vbin(:)
                    ! 位相速度のビン [freq/km]
  double precision, intent(inout) :: bin_val(size(vbin)+1)
                    ! 位相速度のビンで分けられたスペクトル強度 [ival と同じ]
  double precision, intent(in), optional :: undef  ! 未定義値
  logical, intent(in), optional :: mean_flag  ! 平均値として返す.
                                              ! デフォルト: .false.
  double precision, intent(in), optional :: krange  ! ビン分けする閾値波数
  double precision, intent(in), optional :: frange  ! ビン分けする閾値周波数

  integer :: ii, jj, kk, nk, nf, nbin
  integer :: ncount(size(vbin)+1)
  double precision :: dundef
  double precision :: velval(size(km),size(freq))
  logical :: m_flag, range_flag

  nk=size(km)
  nf=size(freq)
  nbin=size(vbin)
  ncount=0
  bin_val=0.0d0

  if(present(undef))then
     dundef=undef
  else
     dundef=-999.0d0
  end if

  if(present(mean_flag))then
     m_flag=mean_flag
  else
     m_flag=.false.
  end if

  range_flag=.false.
  if(present(krange))then
     range_flag=.true.
     if(present(frange).eqv..false.)then
        write(*,*) "*** ERROR (phase_velocity_binning) ***: krange is set, but frange is not set."
        stop
     end if
  end if

!-- 各 km, freq に対応する位相速度の計算

  if(range_flag.eqv..false.)then
     do jj=1,nf
        do ii=1,nk
           if(km(ii)==0.0d0)then
              velval(ii,jj)=dundef
           else
              velval(ii,jj)=freq(jj)/km(ii)
           end if
        end do
     end do
  else
     do jj=1,nf
        do ii=1,nk
           if(km(ii)==0.0d0)then
              velval(ii,jj)=dundef
           else if((km(ii)/krange)**2+(freq(jj)/frange)**2<=1.0d0)then
              velval(ii,jj)=freq(jj)/km(ii)
           else
              velval(ii,jj)=dundef
           end if
        end do
     end do
  end if

!-- 各 km, freq におけるスペクトル強度で重みをつけて
!-- 位相速度をビンに分ける.

  do jj=1,nf
     do ii=1,nk
        if(ival(ii,jj)/=dundef.and.ival(ii,jj)/=0.0d0)then
           if(vbin(1)>velval(ii,jj))then
              bin_val(1)=bin_val(1)+ival(ii,jj)
              ncount(1)=ncount(1)+1
           else if(vbin(nbin)<velval(ii,jj))then
              bin_val(nbin+1)=bin_val(nbin+1)+ival(ii,jj)
              ncount(nbin+1)=ncount(nbin+1)+1
           else
              do kk=1,nbin-1
                 if(vbin(kk)<velval(ii,jj).and.vbin(kk+1)>velval(ii,jj))then
                    bin_val(kk+1)=bin_val(kk+1)+ival(ii,jj)
                    ncount(kk+1)=ncount(kk+1)+1
!              write(*,*) "check", vbin(kk+1), bin_val(kk+1), ival(ii,jj),  &
!  &                               ii, jj, velval(ii,jj), km(ii), freq(jj)
                    exit
                 else if(vbin(kk)==velval(ii,jj))then
                    bin_val(kk)=bin_val(kk)+0.5d0*ival(ii,jj)
                    bin_val(kk+1)=bin_val(kk+1)+0.5d0*ival(ii,jj)
                    ncount(kk)=ncount(kk)+1
                    ncount(kk+1)=ncount(kk+1)+1
!              write(*,*) "check2", vbin(kk+1), bin_val(kk+1), bin_val(kk),  &
!  &                                ival(ii,jj), ii, jj, velval(ii,jj),  &
!  &                                km(ii), freq(jj)
                    exit
                 end if
              end do
              if(vbin(nbin)==velval(ii,jj))then
                 bin_val(nbin)=bin_val(nbin)+0.5d0*ival(ii,jj)
                 bin_val(nbin+1)=bin_val(nbin+1)+0.5d0*ival(ii,jj)
                 ncount(nbin)=ncount(nbin)+1
                 ncount(nbin+1)=ncount(nbin+1)+1
!              write(*,*) "check3", vbin(nbin+1), bin_val(nbin+1),  &
!  &                                bin_val(nbin), ival(ii,jj), ii, jj,  &
!  &                                velval(ii,jj), km(ii), freq(jj)
              end if
           end if
        end if
     end do
  end do

!-- 各位相速度ビンの総和数で割って平均として返す.

  if(m_flag.eqv..true.)then
     do kk=1,nbin+1
        if(ncount(kk)>0)then
           bin_val(kk)=bin_val(kk)/dble(ncount(kk))
        else
           bin_val(kk)=0.0d0
        end if
     end do
  end if

end subroutine phase_velocity_binning_d

!-----------------------------------------------

subroutine rotate_calc_f( nx, csign, prim_fact, omega, omegan )
!  FFT に使用する回転行列を計算する.
  use Math_Const
  implicit none
  integer, intent(in) :: nx  ! 入力配列の要素数
  character(1), intent(in) :: csign  ! 正逆変換判定 [r=正変換, i=逆変換]
  integer, dimension(5) :: prim_fact  ! prim = x のとき設定すると
  ! そのべきで FFT を行う. prim_fact=(/a,b,c,d,e/) : 2^a*3^b*5^c*7^d*e
  complex, dimension(0:prim_fact(5)-1,0:prim_fact(5)-1), intent(inout) :: omega
  ! 余因数に対応する回転行列
  complex, dimension(0:nx-1,0:nx-1), intent(inout) :: omegan ! 回転行列
  integer :: counter, base
  integer :: i, j  ! do loop 用配列

  base=prim_fact(5)

  counter=0
  do i=1,4
     counter=counter+prim_fact(i)
  end do

!-- 回転行列を定義

  select case(csign)
  case('r')
     if(counter/=0)then
        if(base/=1)then
           do j=0,base-1
              do i=0,base-1
                 omega(i,j)=cos(2.0*pi*i*j/real(base))-img*sin(2.0*pi*i*j/real(base))
              end do
           end do
        end if
        do j=0,nx-1
           do i=0,nx-1
              omegan(i,j)=cos(2.0*pi*i*j/real(nx))-img*sin(2.0*pi*i*j/real(nx))
           end do
        end do

     else
        do j=0,nx-1
           do i=0,nx-1
              omega(i,j)=cos(2.0*pi*i*j/real(nx))-img*sin(2.0*pi*i*j/real(nx))
              omegan(i,j)=omega(i,j)
           end do
        end do
     end if


  case('i')
     if(counter/=0)then
        if(base/=1)then
           do j=0,base-1
              do i=0,base-1
                 omega(i,j)=cos(2.0*pi*i*j/real(base))+img*sin(2.0*pi*i*j/real(base))
              end do
           end do
        end if
        do j=0,nx-1
           do i=0,nx-1
              omegan(i,j)=cos(2.0*pi*i*j/real(nx))+img*sin(2.0*pi*i*j/real(nx))
           end do
        end do
     else
        do j=0,nx-1
           do i=0,nx-1
              omega(i,j)=cos(2.0*pi*i*j/real(nx))+img*sin(2.0*pi*i*j/real(nx))
              omegan(i,j)=omega(i,j)
           end do
        end do
     end if

  case default
     write(*,*) "******** ERROR : csign is bad. **********"
     write(*,*) "Stop!"
     stop
  end select

end subroutine

!-----------------------------------------------

subroutine rotate_calc_d( nx, csign, prim_fact, omega, omegan )
!  FFT に使用する回転行列を計算する.
  use Math_Const
  implicit none
  integer, intent(in) :: nx  ! 入力配列の要素数
  character(1), intent(in) :: csign  ! 正逆変換判定 [r=正変換, i=逆変換]
  integer, dimension(5) :: prim_fact  ! prim = x のとき設定すると
  ! そのべきで FFT を行う. prim_fact=(/a,b,c,d,e/) : 2^a*3^b*5^c*7^d*e
  complex(kind(0d0)), dimension(0:prim_fact(5)-1,0:prim_fact(5)-1), intent(inout) :: omega
  ! 余因数に対応する回転行列
  complex(kind(0d0)), dimension(0:nx-1,0:nx-1), intent(inout) :: omegan ! 回転行列
  integer :: counter, base
  integer :: i, j  ! do loop 用配列

  base=prim_fact(5)

  counter=0
  do i=1,4
     counter=counter+prim_fact(i)
  end do

!-- 回転行列を定義

  select case(csign)
  case('r')
     if(counter/=0)then
        if(base/=1)then
           do j=0,base-1
              do i=0,base-1
                 omega(i,j)=dcos(2.0d0*pi_dp*dble(i)*dble(j)/dble(base))  &
  &                         -img_cdp*dsin(2.0d0*pi_dp*dble(i)*dble(j)/dble(base))
              end do
           end do
        end if
        do j=0,nx-1
           do i=0,nx-1
              omegan(i,j)=dcos(2.0d0*pi_dp*dble(i)*dble(j)/dble(nx))  &
  &                       -img_cdp*dsin(2.0d0*pi_dp*dble(i)*dble(j)/dble(nx))
           end do
        end do

     else
        do j=0,nx-1
           do i=0,nx-1
              omega(i,j)=dcos(2.0d0*pi_dp*dble(i)*dble(j)/dble(nx))  &
  &                      -img_cdp*dsin(2.0d0*pi_dp*dble(i)*dble(j)/dble(nx))
              omegan(i,j)=omega(i,j)
           end do
        end do
     end if


  case('i')
     if(counter/=0)then
        if(base/=1)then
           do j=0,base-1
              do i=0,base-1
                 omega(i,j)=dcos(2.0d0*pi_dp*dble(i)*dble(j)/dble(base))  &
  &                         +img_cdp*dsin(2.0d0*pi_dp*dble(i)*dble(j)/dble(base))
              end do
           end do
        end if
        do j=0,nx-1
           do i=0,nx-1
              omegan(i,j)=dcos(2.0d0*pi_dp*dble(i)*dble(j)/dble(nx))  &
  &                       +img_cdp*dsin(2.0d0*pi_dp*dble(i)*dble(j)/dble(nx))
           end do
        end do
     else
        do j=0,nx-1
           do i=0,nx-1
              omega(i,j)=dcos(2.0d0*pi_dp*dble(i)*dble(j)/dble(nx))  &
  &                      +img_cdp*dsin(2.0d0*pi_dp*dble(i)*dble(j)/dble(nx))
              omegan(i,j)=omega(i,j)
           end do
        end do
     end if

  case default
     write(*,*) "******** ERROR : csign is bad. **********"
     write(*,*) "Stop!"
     stop
  end select

end subroutine

!-----------------------------------------------

subroutine prim_calc( n, factor, resid )
  ! 2,3,5,7 についての素因数分解を行う.
  implicit none
  integer, intent(in) :: n  ! 分解する数値
  integer, intent(inout), dimension(4) :: factor  ! 分解したときの各素因数
  ! このとき, factor(1:4)=(/a,b,c,d/) であり, factor(5)=e とすると,
  ! n=2^a*3^b*5^c*7^d*e という式になっている.
  integer, intent(inout) :: resid  ! residual prim factor
  integer :: i, base
  integer, dimension(4) :: prim_dim

  prim_dim=(/2,3,5,7/)
  base=n
  factor=(/0,0,0,0/)
  do i=1,4
     do while(mod(base,prim_dim(i))==0)
        base=base/prim_dim(i)
        factor(i)=factor(i)+1
     end do
  end do

  resid=base

end subroutine

end module
