!---------------------------------------------------------------
! Copyright (C) 2009-2015 GFD Dennou Club. All rights reserved.
!---------------------------------------------------------------

module Poly_Function  ! 直交多項式を計算するサブルーチン集

interface CHEBYSHEV

  module procedure CHEBYSHEV_f, CHEBYSHEV_d

end interface

interface GEGENBAUER

  module procedure GEGENBAUER_f, GEGENBAUER_d

end interface

interface HERMITE

  module procedure HERMITE_f, HERMITE_d

end interface

interface JACOBI_POLY

  module procedure JACOBI_POLY_f, JACOBI_POLY_d

end interface

interface LAGUERRE

  module procedure LAGUERRE_f, LAGUERRE_d

end interface

interface LEGENDRE

  module procedure LEGENDRE_f, LEGENDRE_d

end interface

interface AS_LEGENDRE

  module procedure AS_LEGENDRE_f, AS_LEGENDRE_d

end interface

interface SONINE

  module procedure SONINE_f, SONINE_d

end interface


contains
subroutine CHEBYSHEV_f(n, x, che)
!****************************************
!*** チェビシェフ漸化式のサブルーチン ***
!****************************************
  implicit none
  integer, intent(in) :: n  ! 計算するチェビシェフの最高次数
  real, intent(in) :: x(:)  ! チェビシェフ多項式の引数
  real, intent(inout) :: che(0:n,size(x))  ! 計算するチェビシェフ多項式
  integer :: i, j
  integer :: nmax  ! 引数配列 x の要素数

  nmax=size(x)

!-- 初項の設定 ---
  do i=1,nmax
     che(0,i)=1.0
  end do

  if(n > 0)then

     do i=1,nmax
        che(1,i)=x(i)
     end do

     if(n > 1)then
!-- 漸化式の計算 ---
        do j=1,n-1
           do i=1,nmax
              che(j+1,i)=2.0*che(1,i)*che(j,i)-che(j-1,i)
           end do
        end do
     end if
  end if

end  subroutine

subroutine CHEBYSHEV_d(n, x, che)
!****************************************
!*** チェビシェフ漸化式のサブルーチン ***
!****************************************
  implicit none
  integer, intent(in) :: n  ! 計算するチェビシェフの最高次数
  double precision, intent(in) :: x(:)  ! チェビシェフ多項式の引数
  double precision, intent(inout) :: che(0:n,size(x))
  integer :: i, j
  integer :: nmax  ! 引数配列 x の要素数

  nmax=size(x)

!-- 初項の設定 ---
  do i=1,nmax
     che(0,i)=1.0d0
  end do

  if(n > 0)then

     do i=1,nmax
        che(1,i)=x(i)
     end do

     if(n > 1)then
!-- 漸化式の計算 ---
        do j=1,n-1
           do i=1,nmax
              che(j+1,i)=2.0d0*che(1,i)*che(j,i)-che(j-1,i)
           end do
        end do
     end if
  end if

end subroutine


subroutine GEGENBAUER_f(n, x, p, lambda)
!************************************
!*  ゲーゲンバウアー 多項式計算サブルーチン  *
!************************************
!* 使い方
!* n=次数(0次から指定可能)
!************************************
  implicit none
  integer, intent(in) :: n  ! 計算するゲーゲンバウアー多項式の最高次数
  real, intent(in) :: x(:)  ! 引数
  real, intent(in) :: lambda  ! ゲーゲンバウアー係数
  real, intent(inout) :: p(0:n,size(x))  ! 計算するゲーゲンバウアー多項式
  integer :: i, j
  integer :: nmax  ! 引数配列 x の要素数

  nmax=size(x)

!-- 初項の設定 ---
  do i=1,nmax
     p(0,i)=1.0
  end do

  if(n > 0)then

     do i=1,nmax
        p(1,i)=2.0*lambda*x(i)
     end do

     if(n > 1)then
!-- 漸化式の計算 ---
        do j=1,n-1
           do i=1,nmax
              p(j+1,i)=(1.0/real(j+1))*(2.0*(lambda+real(j))*x(i)*p(j,i)  &
     &               -(2.0*lambda+real(j-1))*p(j-1,i))
           end do
        end do
     end if
  end if

end subroutine

subroutine GEGENBAUER_d(n, x, p, lambda)
!************************************
!*  ゲーゲンバウアー 多項式計算サブルーチン  *
!************************************
!* 使い方
!* n=次数(0次から指定可能)
!************************************
  implicit none
  integer, intent(in) :: n  ! 計算するゲーゲンバウアー多項式の最高次数
  double precision, intent(in) :: x(:)  ! 引数
  double precision, intent(in) :: lambda  ! ゲーゲンバウアー係数
  double precision, intent(inout) :: p(0:n,size(x))  ! 計算するゲーゲンバウアー多項式
  integer :: i, j
  integer :: nmax  ! 引数配列 x の要素数

  nmax=size(x)

!-- 初項の設定 ---
  do i=1,nmax
     p(0,i)=1.0d0
  end do

  if(n > 0)then

     do i=1,nmax
        p(1,i)=2.0d0*lambda*x(i)
     end do

     if(n > 1)then
!-- 漸化式の計算 ---
        do j=1,n-1
           do i=1,nmax
              p(j+1,i)=(1.0d0/dble(j+1))*(2.0d0*(lambda+dble(j))*x(i)*p(j,i)  &
     &               -(2.0d0*lambda+dble(j-1))*p(j-1,i))
           end do
        end do
     end if
  end if

end subroutine


subroutine HERMITE_f(n, x, p)
!************************************
!*  Hermite 多項式計算サブルーチン  *
!************************************
!* 使い方
!* n=次数(0次から指定可能)
!************************************
  implicit none
  integer, intent(in) :: n  ! 計算する Hermite 多項式の最高次数
  real, intent(in) :: x(:)  ! 引数
  real, intent(inout) :: p(0:n,size(x))  ! 計算する Hermite 多項式
  integer :: i, j
  integer :: nmax  ! 引数配列 x の要素数

  nmax=size(x)

!-- 初項の設定 ---
  do i=1,nmax
     p(0,i)=1.0
  end do

  if(n > 0)then

     do i=1,nmax
        p(1,i)=2.0*x(i)
     end do

     if(n > 1)then
!-- 漸化式の計算 ---
        do j=1,n-1
           do i=1,nmax
              p(j+1,i)=2.0*(x(i)*p(j,i)-real(j)*p(j-1,i))
           end do
        end do
     end if
  end if

end subroutine


subroutine HERMITE_d(n, x, p)
!************************************
!*  Hermite 多項式計算サブルーチン  *
!************************************
!* 使い方
!* n=次数(0次から指定可能)
!************************************
  implicit none
  integer, intent(in) :: n  ! 計算する Hermit 多項式の最高次数
  double precision, intent(in) :: x(:)  ! 引数
  double precision, intent(inout) :: p(0:n,size(x))  ! 計算する Hermit 多項式
  integer :: i, j
  integer :: nmax  ! 引数配列 x の要素数

  nmax=size(x)

!-- 初項の設定 ---
  do i=1,nmax
     p(0,i)=1.0d0
  end do

  if(n > 0)then

     do i=1,nmax
        p(1,i)=2.0d0*x(i)
     end do

     if(n > 1)then
!-- 漸化式の計算 ---
        do j=1,n-1
           do i=1,nmax
              p(j+1,i)=2.0d0*(x(i)*p(j,i)-dble(j)*p(j-1,i))
           end do
        end do
     end if
  end if

end subroutine


subroutine JACOBI_POLY_f(n, x, p, alpha, beta)
!***********************************
!  Jacobi 多項式計算サブルーチン   *
!***********************************
! 使い方
! n=次数(0次から指定可能)
! nmax=空間格子点数
! p=p(0:n,nmax) の2次元配列
!***********************************
  implicit none
  integer, intent(in) :: n  ! 計算する jacobi 多項式の最高次数
  real, intent(in) :: x(:)  ! 引数
  real, intent(inout) :: p(0:n,size(x))  ! 計算する Jacobi 多項式
  real, intent(in) :: alpha  ! 第一引数
  real, intent(in) :: beta  ! 第二引数
  real :: gamma, omega
  integer :: i, j
  integer :: nmax  ! 引数配列 x の要素数

  nmax=size(x)

!-- 係数の設定 ---
  gamma=alpha+beta
  omega=alpha-beta

!-- 初項の設定 ---
  do i=1,nmax
     p(0,i)=1.0
  end do

  if(n > 0)then

     do i=1,nmax
        p(1,i)=0.5*((gamma+2.0)*x(i)+omega)
     end do

     if(n > 1)then
!-- 漸化式の計算 ---
        do j=1,n-1
           do i=1,nmax
              p(j+1,i)=(0.5/(real(j+1)*real(j+1+gamma)*real(2.0*j+gamma)))  &
     &                 *((2.0*j+gamma+1.0)  &
     &                 *(gamma*omega+(2.0*j+gamma)*(2.0*(j+1)+gamma)*x(i))  &
     &                 *p(j,i)  &
     &                 -2.0*(j+alpha)*(j+beta)*(2.0*(j+1)+gamma)*p(j-1,i))
           end do
        end do
     end if
  end if

end subroutine



subroutine JACOBI_POLY_d(n, x, p, alpha, beta)
!***********************************
!  Jacobi 多項式計算サブルーチン   *
!***********************************
! 使い方
! n=次数(0次から指定可能)
! nmax=空間格子点数
! p=p(0:n,nmax) の2次元配列
!***********************************
  implicit none
  integer, intent(in) :: n  ! 計算する jacobi 多項式の最高次数
  double precision, intent(in) :: x(:)  ! 引数
  double precision, intent(inout) :: p(0:n,size(x))  ! 計算する Jacobi 多項式
  double precision, intent(in) :: alpha  ! 第一引数
  double precision, intent(in) :: beta  ! 第二引数
  double precision :: gamma, omega
  integer :: i, j
  integer :: nmax  ! 引数配列 x の要素数

  nmax=size(x)

!-- 係数の設定 ---
  gamma=alpha+beta
  omega=alpha-beta

!-- 初項の設定 ---
  do i=1,nmax
     p(0,i)=1.0d0
  end do

  if(n > 0)then

     do i=1,nmax
        p(1,i)=0.5d0*((gamma+2.0d0)*x(i)+omega)
     end do

     if(n > 1)then
!-- 漸化式の計算 ---
        do j=1,n-1
           do i=1,nmax
              p(j+1,i)=(0.5d0/(dble(j+1)*dble(j+1+gamma)*dble(2.0*j+gamma)))  &
     &                 *((2.0d0*dble(j)+gamma+1.0d0)  &
     &                 *(gamma*omega+(2.0d0*dble(j)+gamma)  &
     &                 *(2.0d0*dble(j+1)+gamma)*x(i))*p(j,i)  &
     &                 -2.0d0*dble(j+alpha)*dble(j+beta)  &
     &                 *(2.0d0*dble(j+1)+gamma)*p(j-1,i))
           end do
        end do
     end if
  end if

end subroutine


subroutine LAGUERRE_f(n, x, p)
!***********************************
!  Laguerre 多項式計算サブルーチン *
!***********************************
! 使い方
! n=次数(0次から指定可能)
! nmax=空間格子点数
! p=p(0:n,nmax) の2次元配列
!***********************************
  implicit none
  integer, intent(in) :: n  ! 計算する jacobi 多項式の最高次数
  real, intent(in) :: x(:)  ! 引数
  real, intent(inout) :: p(0:n,size(x))  ! 計算する Jacobi 多項式
  integer :: i, j
  integer :: nmax  ! 引数配列 x の要素数

  nmax=size(x)

!-- 初項の設定 ---
  do i=1,nmax
     p(0,i)=1.0
  end do

  if(n > 0)then

     do i=1,nmax
        p(1,i)=1.0-x(i)
     end do

     if(n > 1)then
!-- 漸化式の計算 ---
        do j=1,n-1
           do i=1,nmax
              p(j+1,i)=(2.0*real(j)+1.0-x(i))*p(j,i)  &
     &                 -((real(j))**2)*p(j-1,i)
           end do
        end do
     end if
  end if

end subroutine


subroutine LAGUERRE_d(n, x, p)
!***********************************
!  Laguerre 多項式計算サブルーチン *
!***********************************
! 使い方
! n=次数(0次から指定可能)
! nmax=空間格子点数
! p=p(0:n,nmax) の2次元配列
!***********************************
  implicit none
  integer, intent(in) :: n  ! 計算する jacobi 多項式の最高次数
  double precision, intent(in) :: x(:)  ! 引数
  double precision, intent(inout) :: p(0:n,size(x))  ! 計算する Jacobi 多項式
  integer :: i, j
  integer :: nmax  ! 引数配列 x の要素数

  nmax=size(x)

!-- 初項の設定 ---
  do i=1,nmax
     p(0,i)=1.0d0
  end do

  if(n > 0)then

     do i=1,nmax
        p(1,i)=1.0d0-x(i)
     end do

     if(n > 1)then
!-- 漸化式の計算 ---
        do j=1,n-1
           do i=1,nmax
              p(j+1,i)=(2.0d0*dble(j)+1.0d0-x(i))*p(j,i)  &
     &                 -((dble(j))**2)*p(j-1,i)
           end do
        end do
     end if
  end if

end subroutine


subroutine LEGENDRE_f(n, x, p)
!***********************************
!  Legendre 多項式計算サブルーチン *
!***********************************
! 使い方
! n=次数(0次から指定可能)
! nmax=空間格子点数
! p=p(0:n,nmax) の2次元配列
!***********************************
  implicit none
  integer, intent(in) :: n  ! 計算する jacobi 多項式の最高次数
  real, intent(in) :: x(:)  ! 引数
  real, intent(inout) :: p(0:n,size(x))  ! 計算する Jacobi 多項式
  integer :: i, j
  integer :: nmax  ! 引数配列 x の要素数

  nmax=size(x)

!-- 初項の設定 ---
  do i=1,nmax
     p(0,i)=1.0
  end do

  if(n > 0)then

     do i=1,nmax
        p(1,i)=x(i)
     end do

     if(n > 1)then
!-- 漸化式の計算 ---
        do j=1,n-1
           do i=1,nmax
              p(j+1,i)=(1.0/real(j+1))*(p(j,i)*(2.0*real(j)+1.0)  &
     &                 *(p(1,i))-real(j)*p(j-1,i))
           end do
        end do
     end if
  end if

end subroutine


subroutine LEGENDRE_d(n, x, p)
!***********************************
!  Legendre 多項式計算サブルーチン *
!***********************************
! 使い方
! n=次数(0次から指定可能)
! nmax=空間格子点数
! p=p(0:n,nmax) の2次元配列
!***********************************
  implicit none
  integer, intent(in) :: n  ! 計算する jacobi 多項式の最高次数
  double precision, intent(in) :: x(:)  ! 引数
  double precision, intent(inout) :: p(0:n,size(x))  ! 計算する Jacobi 多項式
  integer :: i, j
  integer :: nmax  ! 引数配列 x の要素数

  nmax=size(x)

!-- 初項の設定 ---
  do i=1,nmax
     p(0,i)=1.0d0
  end do

  if(n > 0)then

     do i=1,nmax
        p(1,i)=x(i)
     end do

     if(n > 1)then
!-- 漸化式の計算 ---
        do j=1,n-1
           do i=1,nmax
              p(j+1,i)=(1.0d0/dble(j+1))*(p(j,i)*(2.0d0*dble(j)+1.0d0)  &
     &                 *(p(1,i))-dble(j)*p(j-1,i))
           end do
        end do
     end if
  end if

end subroutine

subroutine AS_LEGENDRE_f(n, m, x, p)
!***********************************
!  Legendre 陪関数計算サブルーチン *
!***********************************
! 使い方
! n=次数(0次から指定可能)
! m=随伴次数(n>=m)
! nmax=空間格子点数
! p=p(0:n,nmax) の2次元配列
!***********************************
  implicit none
  integer, intent(in) :: n  ! 計算する legendre 陪関数の最高次数
  integer, intent(in) :: m  ! 計算する legendre 陪関数の随伴次数
  real, intent(in) :: x(:)  ! 引数
  real, intent(inout) :: p(0:n,-m:m,size(x))  ! 計算する legendre 陪関数
  integer :: i, j, k, abm
  integer :: nmax  ! 引数配列 x の要素数

  nmax=size(x)

  if(m>n)then
  end if

!-- 初項の設定 ---
  do i=1,nmax
     p(0,0,i)=1.0
  end do

  if(n > 0)then

     do k=-m,m
        do i=1,nmax
           p(1,k,i)=x(i)
        end do

        if(n > 1)then
!-- 漸化式の計算 ---
           do j=1,n-1
              do i=1,nmax
                 p(j+1,k,i)=(1.0/real(j-k+1))*(p(j,k,i)*(2.0*real(j)+1.0)  &
        &                 *(p(1,k,i))-real(j+k)*p(j-1,k,i))
              end do
           end do
        end if
     end do
  end if

end subroutine


subroutine LEGENDRE_d(n, x, p)
!***********************************
!  Legendre 多項式計算サブルーチン *
!***********************************
! 使い方
! n=次数(0次から指定可能)
! nmax=空間格子点数
! p=p(0:n,nmax) の2次元配列
!***********************************
  implicit none
  integer, intent(in) :: n  ! 計算する jacobi 多項式の最高次数
  double precision, intent(in) :: x(:)  ! 引数
  double precision, intent(inout) :: p(0:n,size(x))  ! 計算する Jacobi 多項式
  integer :: i, j
  integer :: nmax  ! 引数配列 x の要素数

  nmax=size(x)

!-- 初項の設定 ---
  do i=1,nmax
     p(0,i)=1.0d0
  end do

  if(n > 0)then

     do i=1,nmax
        p(1,i)=x(i)
     end do

     if(n > 1)then
!-- 漸化式の計算 ---
        do j=1,n-1
           do i=1,nmax
              p(j+1,i)=(1.0d0/dble(j+1))*(p(j,i)*(2.0d0*dble(j)+1.0d0)  &
     &                 *(p(1,i))-dble(j)*p(j-1,i))
           end do
        end do
     end if
  end if

end subroutine


subroutine SONINE_f(n, x, p, lambda)
!***********************************
!  Sonine 多項式計算サブルーチン  *
!***********************************
! 使い方
! n=次数(0次から指定可能)
! nmax=空間格子点数
! p=p(0:n,nmax) の2次元配列
!***********************************
  implicit none
  integer, intent(in) :: n  ! 計算する jacobi 多項式の最高次数
  real, intent(in) :: x(:)  ! 引数
  real, intent(inout) :: p(0:n,size(x))  ! 計算する Jacobi 多項式
  real, intent(in) :: lambda  ! 第一引数
  integer :: i, j
  integer :: nmax  ! 引数配列 x の要素数

  nmax=size(x)

!-- 初項の設定 ---
  do i=1,nmax
     p(0,i)=1.0
  end do

  if(n > 0)then

     do i=1,nmax
         p(1,i)=lambda+1.0-x(i)
     end do

     if(n > 1)then
!-- 漸化式の計算 ---
        do j=1,n-1
           do i=1,nmax
              p(j+1,i)=(1.0/real(j+1))  &
     &                 *((lambda+2.0*j+1.0-x(i))*p(j,i)  &
     &                 -(j+lambda)*p(j-1,i))
           end do
        end do
     end if
  end if

end subroutine


subroutine SONINE_d(n, x, p, lambda)
!***********************************
!  Sonine 多項式計算サブルーチン  *
!***********************************
! 使い方
! n=次数(0次から指定可能)
! nmax=空間格子点数
! p=p(0:n,nmax) の2次元配列
!***********************************
  implicit none
  integer, intent(in) :: n  ! 計算する jacobi 多項式の最高次数
  double precision, intent(in) :: x(:)  ! 引数
  double precision, intent(inout) :: p(0:n,size(x))  ! 計算する Jacobi 多項式
  double precision, intent(in) :: lambda  ! 第一引数
  integer :: i, j
  integer :: nmax  ! 引数配列 x の要素数

  nmax=size(x)

!-- 初項の設定 ---
  do i=1,nmax
     p(0,i)=1.0d0
  end do

  if(n > 0)then

     do i=1,nmax
         p(1,i)=lambda+1.0d0-x(i)
     end do

     if(n > 1)then
!-- 漸化式の計算 ---
        do j=1,n-1
           do i=1,nmax
              p(j+1,i)=(1.0d0/dble(j+1))  &
     &                 *((lambda+2.0d0*dble(j)+1.0d0-x(i))*p(j,i)  &
     &                 -(dble(j)+lambda)*p(j-1,i))
           end do
        end do
     end if
  end if

end subroutine


end module
