module force_solv
! 強制項を計算するモジュール
! ここで用いられるスタッガード格子は,
! 物理領域左下端について, j=1, k=1 として定義されているので,
! スタッガード格子による微分を行う際は, スカラー点での微分に
! grad_for_1d を, その他のベクトル点での微分には grad_back_1d を用いること.
  use val_define
  use read_namelist
  use val_alloc
  use val_init
  use val_coord
  use real_initialize
  use Derivation
  use sub_calc
  use Thermo_Function
  use Cloud_Warm

contains

subroutine force( mode )

  implicit none

  character(1), intent(in) :: mode   ! time splitting mode
  integer :: i, j, k

  !-- 強制項の初期化

  if(mode(1:1)=='b')then
     call real_init( force_v )
     call real_init( force_t )
     call real_init( force_qv )
     call real_init( force_qc )
     call real_init( force_ql )
  else
     call real_init( force_u )
     call real_init( force_w )
     call real_init( force_p )
  end if

  !-- temporary terms

  if(mode(1:1)=='b')then
     call real_init( ADV_v )
     call real_init( ADV_t )
     call real_init( ADV_qv )
     call real_init( ADV_qc )
     call real_init( ADV_ql )
     call real_init( DIFF_v )
     call real_init( DIFF_t )
     call real_init( DIFF_qv )
     call real_init( DIFF_qc )
     call real_init( DIFF_ql )
     call real_init( NUMD_v )
     call real_init( NUMD_t )
     call real_init( NUMD_qv )
     call real_init( NUMD_qc )
     call real_init( NUMD_ql )
  else
     call real_init( ADV_u )
     call real_init( ADV_w )
     call real_init( DIFF_u )
     call real_init( DIFF_w )
     call real_init( NUMD_u )
     call real_init( NUMD_w )
  end if

  !-- grid 各点への値の格納
  !-- 更新された予報変数を作業用変数へ格納.
  call reset_val()

  !-- u -> v, w, v -> u, w, w -> u, v
  call auto_staggered_interp()

  !-- 各強制項の計算
  !-- ADVECTION
  call ADV_calc( mode )
  call DIFF_calc( mode )
  call CC_calc( mode )
!  call GRADP_calc()
!  call FLUX_calc()
  call SPG_calc( mode )

  if(mode(1:1)=='b')then
     call CLOUD_calc()
     call RAD_calc()
  else if(mode(1:1)=='s')then
     call BUOY_calc()
  end if

  if(nunh==0.0.and.nunv==0.0)then
     call NUM_DIFF_calc( mode )
  end if

  !-- 方程式の各項を計算, 強制項へ代入

  if(mode(1:1)=='b')then
!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)

     do k=1,nz
        do j=1,nr
           force_v(j,k)=ADV_v(j,k)+DIFF_v(j,k)+CC_v(j,k)+SPG_v(j,k)+NUMD_v(j,k)
           force_t(j,k)=ADV_t(j,k)+DIFF_t(j,k)+RAD_t(j,k)+SPG_t(j,k)  &
  &                    +CLOUD_pt(j,k)+NUMD_t(j,k)
           force_qv(j,k)=ADV_qv(j,k)+DIFF_qv(j,k)+SPG_qv(j,k)+CLOUD_qv(j,k)+NUMD_qv(j,k)
           force_qc(j,k)=ADV_qc(j,k)+DIFF_qc(j,k)+SPG_qc(j,k)+CLOUD_qc(j,k)+NUMD_qc(j,k)!+FALL_qc(j,k)
           force_ql(j,k)=ADV_ql(j,k)+DIFF_ql(j,k)+SPG_ql(j,k)+CLOUD_ql(j,k)+NUMD_ql(j,k)
if(k==1)then
!write(*,*) "DIFF check", force_qv(j,k), DIFF_t(j,k), DIFF_qv(j,k), j, k
!write(*,*) "force t check", j, k, force_t(j,k), ADV_t(j,k), +DIFF_t(j,k), RAD_t(j,k), SPG_t(j,k)
end if
!write(*,*) "forcew check", force_w(j,k), j, k, ADV_w(j,k), DIFF_w(j,k), GRADP_w(j,k), BUOY_w(j,k), SPG_w(j,k)
        end do
     end do

!$omp end do
!$omp end parallel

  else if(mode(1:1)=='s')then

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)

     do k=1,nz
        do j=1,nr
           force_u(j,k)=ADV_u(j,k)+DIFF_u(j,k)+GRADP_u(j,k)+CC_u(j,k)+SPG_u(j,k)
           force_w(j,k)=ADV_w(j,k)+DIFF_w(j,k)+GRADP_w(j,k)+BUOY_w(j,k)+SPG_w(j,k)
if(abs(force_w(j,k))>1.0.or.force_w(j,k)/=force_w(j,k))then
!write(*,*) "force u check", j, k, force_u(j,k), ADV_u(j,k), +DIFF_u(j,k), +GRADP_u(j,k), +CC_u(j,k), +SPG_u(j,k)
!write(*,*) "forcew check", force_w(j,k), j, k, ADV_w(j,k), DIFF_w(j,k), GRADP_w(j,k), BUOY_w(j,k), SPG_w(j,k)
end if
        end do
     end do

!$omp end do
!$omp end parallel

!-- outer boundary

     do k=1,nz+1
        if(u_u(nr+1,k)+cg<0.0)then
           force_u(nr+1,k)=-(u_u(nr+1,k)+cg)  &
  &                         *((u_u(nr+1,k)-u_u(nr,k))/(r_u(nr+1)-r_u(nr)))  &
  &                        +(coril+v_u(nr+1,k)/r_u(nr+1))*v_u(nr+1,k)
        else
           force_u(nr+1,k)=0.0
        end if
     end do

  end if

end subroutine


subroutine ADV_calc( mode )
!-- 移流計算を行う.
  use val_define
  implicit none
  character(1), intent(in) :: mode
  integer :: j, k
  real, dimension(nr+1,nz+1) :: dudr_u, dvdr_s, dwdr_w, dtdr_s, dqvdr_s, dqcdr_s
  real, dimension(nr+1,nz+1) :: dudz_u, dvdz_s, dwdz_w, dtdz_s, dqvdz_s, dqcdz_s
  real, dimension(nr+1,nz+1) :: dqldr_s, dqldz_s

!$omp parallel default(shared)
!$omp do schedule(runtime) private(k)
  do k=1,nz+1
     !-- (dudr)_u
     call grad_back_1d( r_s, u_s(:,k), dudr_u(:,k) )
     !-- (dwdr)_w
     call grad_for_1d( r_u, w_u(:,k), dwdr_w(:,k) )
     !-- (dphidr)_s
     call grad_for_1d( r_u, v_u(:,k), dvdr_s(:,k) )
     call grad_for_1d( r_u, t_u(:,k), dtdr_s(:,k) )
     call grad_for_1d( r_u, qv_u(:,k), dqvdr_s(:,k) )
     call grad_for_1d( r_u, qc_u(:,k), dqcdr_s(:,k) )
     call grad_for_1d( r_u, ql_u(:,k), dqldr_s(:,k) )
  end do
!$omp end do

!$omp do schedule(runtime) private(j)
  do j=1,nr+1
     !-- (dudz)_u
     call grad_for_1d( z_w, u_w(j,:), dudz_u(j,:) )
     !-- (dwdz)_w
     call grad_back_1d( z_s, w_s(j,:), dwdz_w(j,:) )
     !-- (dphidz)_s
     call grad_for_1d( z_w, v_w(j,:), dvdz_s(j,:) )
     call grad_for_1d( z_w, t_w(j,:), dtdz_s(j,:) )
     call grad_for_1d( z_w, qv_w(j,:), dqvdz_s(j,:) )
     call grad_for_1d( z_w, qc_w(j,:), dqcdz_s(j,:) )
     call grad_for_1d( z_w, ql_w(j,:), dqldz_s(j,:) )
  end do
!$omp end do
!$omp end parallel

  if(mode(1:1)=='b')then

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
  do k=1,nz+1
     do j=1,nr+1
        ADV_v(j,k)=-u_s(j,k)*dvdr_s(j,k)-w_s(j,k)*dvdz_s(j,k)
        ADV_t(j,k)=-u_s(j,k)*dtdr_s(j,k)-w_s(j,k)*dtdz_s(j,k)
        ADV_qv(j,k)=-u_s(j,k)*dqvdr_s(j,k)-w_s(j,k)*dqvdz_s(j,k)
        ADV_qc(j,k)=-u_s(j,k)*dqcdr_s(j,k)-w_s(j,k)*dqcdz_s(j,k)
        ADV_ql(j,k)=-u_s(j,k)*dqldr_s(j,k)-w_s(j,k)*dqldz_s(j,k)
     end do
  end do
!$omp end do
!$omp end parallel

  else if(mode(1:1)=='s')then

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
  do k=1,nz+1
     do j=1,nr+1
        ADV_u(j,k)=-u_u(j,k)*dudr_u(j,k)-w_u(j,k)*dudz_u(j,k)
        ADV_w(j,k)=-u_w(j,k)*dwdr_w(j,k)-w_w(j,k)*dwdz_w(j,k)
     end do
  end do
!$omp end do
!$omp end parallel

  end if

end subroutine ADV_calc


subroutine CC_calc( mode )
!-- 遠心力とコリオリを計算.
  use val_define
  implicit none
  character(1), intent(in) :: mode
  integer :: j, k

  if(mode(1:1)=='s')then

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(j,k)
  do k=1,nz+1
     do j=1,nr+1
        if(r_u(j)/=0.0)then
           CC_u(j,k)=(coril+v_u(j,k)/r_u(j))*v_u(j,k)
        else
           CC_u(j,k)=0.0
        end if
     end do
  end do
!$omp end do
!$omp end parallel

  else if(mode(1:1)=='b')then

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(j,k)
  do k=1,nz+1
     do j=1,nr+1
        if(r_s(j)/=0.0)then
           CC_v(j,k)=-(coril+v_s(j,k)/r_s(j))*u_s(j,k)
        else
           CC_v(j,k)=0.0
        end if
     end do
  end do
!$omp end do
!$omp end parallel

  end if

end subroutine CC_calc


subroutine BUOY_calc()
!-- 浮力を計算する.
  use val_define
  use Phys_Const
  implicit none
  integer :: j, k

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
  do k=1,nz+1
     do j=1,nr+1
        BUOY_w(j,k)=g*((t_w(j,k)-ptb_w(j,k))/ptb_w(j,k)  &
  &                    +0.61*(qv_w(j,k)-qvb_w(j,k))-(qc_w(j,k)+ql_w(j,k)))
     end do
  end do
!$omp end do
!$omp end parallel

end subroutine BUOY_calc


subroutine FLUX_calc()
!-- 圧力方程式のフラックス項を計算する.
  use val_define
  use Thermo_Const
  implicit none
  integer :: j, k
  real, dimension(nr+1,nz+1) :: fu_u, fw_w, dfudr_s, dfwdz_s

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
  do k=1,nz+1
     do j=1,nr+1
        fu_u(j,k)=r_u(j)*u_u(j,k)
        fw_w(j,k)=w_w(j,k)*rhob_w(j,k)*ptvb_w(j,k)
     end do
  end do
!$omp end do
!$omp end parallel

  !-- (dfudr)_s
!$omp parallel default(shared)
!$omp do schedule(runtime) private(k)
  do k=1,nz+1
     call grad_for_1d( r_u, fu_u(:,k), dfudr_s(:,k) )
  end do
!$omp end do

!$omp do schedule(runtime) private(j)
  !-- (dfwdr)_s
  do j=1,nr+1
     call grad_for_1d( z_w, fw_w(j,:), dfwdz_s(j,:) )
  end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(j,k)
  do k=1,nz+1
     do j=1,nr+1
        if(r_s(j)/=0.0)then
           FLUX_s(j,k)=(Rd*tempb_s(j,k)/(Cvd*ptvb_s(j,k)))*(dfudr_s(j,k)/r_s(j)  &
  &                    +dfwdz_s(j,k)/(rhob_s(j,k)*ptvb_s(j,k)))
!           FLUX_s(j,k)=((Cpd/Cvd)*Rd*tempb_s(j,k)/(Cpd*rhob_s(j,k)*ptvb_s(j,k)*ptvb_s(j,k)))  &
!  &                    *(dfudr_s(j,k)/r_s(j)+dfwdz_s(j,k))
        else
           FLUX_s(j,k)=0.0
        end if
     end do
  end do
!$omp end do
!$omp end parallel

end subroutine FLUX_calc


subroutine GRADP_calc()
!-- 圧力傾度計算を行う.
  use val_define
  use Thermo_Const
  implicit none
  integer :: j, k
  real, dimension(nr+1,nz+1) :: dpdr_u, dpdz_w

  !-- (dpdr)_u
!$omp parallel default(shared)
!$omp do schedule(runtime) private(k)
  do k=1,nz+1
     call grad_back_1d( r_s, p_s(:,k), dpdr_u(:,k) )
  end do
!$omp end do

!$omp do schedule(runtime) private(j)
  !-- (dpdz)_w
  do j=1,nr+1
     call grad_back_1d( z_s, p_s(j,:), dpdz_w(j,:) )
  end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
  do k=1,nz+1
     do j=1,nr+1
        GRADP_u(j,k)=-Cpd*ptvb_u(j,k)*dpdr_u(j,k)
        GRADP_w(j,k)=-Cpd*ptvb_w(j,k)*dpdz_w(j,k)
     end do
  end do
!$omp end do
!$omp end parallel

end subroutine GRADP_calc


subroutine RAD_calc()
  use val_define
  implicit none
  integer :: j, k

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
  do k=1,nz+1
     do j=1,nr+1
        RAD_t(j,k)=-(t_s(j,k)-ptb_s(j,k))/tau_R
     end do
  end do
!$omp end do
!$omp end parallel

end subroutine RAD_calc


subroutine DIFF_calc( mode )
!-- 乱流拡散を計算する.
  use val_define
  implicit none
  character(1), intent(in) :: mode
  integer :: j, k
  real :: tmpa, tmpb
  real, dimension(nr+1,nz+1) :: dvrdr_u, dvrdr_s
  real, dimension(nr+1,nz+1) :: dwdr_u, dwdr_w, dwdr_s
  real, dimension(nr+1,nz+1) :: dudr_u, dudr_w, dudr_s
  real, dimension(nr+1,nz+1) :: dvdr_u, dvdr_w, dvdr_s
  real, dimension(nr+1,nz+1) :: dwdz_u, dwdz_w, dwdz_s
  real, dimension(nr+1,nz+1) :: dudz_u, dudz_w, dudz_s
  real, dimension(nr+1,nz+1) :: dvdz_u, dvdz_w, dvdz_s
  real, dimension(nr+1,nz+1) :: dptvbdz_u, dptvbdz_w, dptvbdz_s
  real, dimension(nr+1,nz+1) :: dptvdz_u, dptvdz_w, dptvdz_s
  real, dimension(nr+1,nz+1) :: dptedz_u, dptedz_s, dptedz_w
  real, dimension(nr+1,nz+1) :: ftr_u, fqvr_u, fqcr_u, fqlr_u
  real, dimension(nr+1,nz+1) :: ftz_w, fqvz_w, fqcz_w, fqlz_w
  real, dimension(nr+1,nz+1) :: fqvz_s, fqvz_u, fqcz_s, fqcz_u, fqlz_s, fqlz_u
  real, dimension(nr+1,nz+1) :: taurr_s, taurr_u, taurt_u, taurt_s, taurz_u
  real, dimension(nr+1,nz+1) :: taurz_w, tauzt_w, tauzz_s
  real, dimension(nr+1,nz+1) :: tautt_u
  real, dimension(nr+1,nz+1) :: S_u, S_w, S_s
  real, dimension(nr+1,nz+1) :: Ri_u, Ri_w, Ri_s
  real, dimension(nr+1,nz+1) :: nu_u, nu_w, nu_s
  real, dimension(nr+1,nz+1) :: dtrrdr_u, dtrtdr_s, dtrzdr_w
  real, dimension(nr+1,nz+1) :: dftrdr_s, dfqvrdr_s, dfqcrdr_s, dfqlrdr_s
  real, dimension(nr+1,nz+1) :: dtrzdz_u, dtztdz_s, dtzzdz_w
  real, dimension(nr+1,nz+1) :: dftzdz_s, dfqvzdz_s, dfqczdz_s, dfqlzdz_s
  real, dimension(nr+1,nz+1) :: tmpv_s, tmpv_u
  real, dimension(nr+1) :: CD_u, CD_s, CE_s

  !-- preparing calc
!$omp parallel default(shared)
!$omp do schedule(dynamic) private(j,k)
  do k=1,nz+1
     do j=1,nr+1
        if(r_s(j)/=0.0)then
           tmpv_s(j,k)=v_s(j,k)/r_s(j)
        else
           tmpv_s(j,k)=0.0
        end if
        if(r_u(j)/=0.0)then
           tmpv_u(j,k)=v_u(j,k)/r_u(j)
        else
           tmpv_u(j,k)=0.0
        end if
     end do
  end do
!$omp end do
!$omp end parallel

!$omp parallel default(shared)
!$omp do schedule(runtime) private(k)
  do k=1,nz+1
     call grad_back_1d( r_s, w_s(:,k), dwdr_u(:,k) )
     call grad_for_1d( r_u, w_u(:,k), dwdr_w(:,k) )
     call grad_for_1d( r_u, w_c(:,k), dwdr_s(:,k) )
     call grad_back_1d( r_s, u_s(:,k), dudr_u(:,k) )
     call grad_for_1d( r_u, u_u(:,k), dudr_s(:,k) )
     call grad_for_1d( r_u, u_w(:,k), dudr_w(:,k) )
     call grad_back_1d( r_s, tmpv_s(:,k), dvrdr_u(:,k) )
     call grad_for_1d( r_u, tmpv_u(:,k), dvrdr_s(:,k) )
     call grad_for_1d( r_u, v_u(:,k), dvdr_s(:,k) )
     call grad_back_1d( r_s, v_s(:,k), dvdr_u(:,k) )
     call grad_for_1d( r_u, v_c(:,k), dvdr_w(:,k) )
     call grad_back_1d( r_s, t_s(:,k), ftr_u(:,k) )
     call grad_back_1d( r_s, qv_s(:,k), fqvr_u(:,k) )
     call grad_back_1d( r_s, qc_s(:,k), fqcr_u(:,k) )
     call grad_back_1d( r_s, ql_s(:,k), fqlr_u(:,k) )
  end do
!$omp end do

!$omp do schedule(runtime) private(j)
  do j=1,nr+1
     call grad_for_1d( z_w, u_w(j,:), dudz_u(j,:) )
     call grad_back_1d( z_s, u_s(j,:), dudz_w(j,:) )
     call grad_for_1d( z_w, u_c(j,:), dudz_s(j,:) )
     call grad_for_1d( z_w, v_c(j,:), dvdz_u(j,:) )
     call grad_for_1d( z_w, v_w(j,:), dvdz_s(j,:) )
     call grad_back_1d( z_s, v_s(j,:), dvdz_w(j,:) )
     call grad_for_1d( z_w, w_u(j,:), dwdz_u(j,:) )
     call grad_for_1d( z_w, w_w(j,:), dwdz_s(j,:) )
     call grad_back_1d( z_s, w_s(j,:), dwdz_w(j,:) )
     call grad_back_1d( z_s, t_s(j,:), ftz_w(j,:) )
     call grad_back_1d( z_s, qv_s(j,:), fqvz_w(j,:) )
     call grad_back_1d( z_s, qc_s(j,:), fqcz_w(j,:) )
     call grad_back_1d( z_s, ql_s(j,:), fqlz_w(j,:) )
     call grad_for_1d( z_w, qv_w(j,:), fqvz_s(j,:) )
     call grad_for_1d( z_w, qc_w(j,:), fqcz_s(j,:) )
     call grad_for_1d( z_w, ql_w(j,:), fqlz_s(j,:) )
     call grad_for_1d( z_w, qv_c(j,:), fqvz_u(j,:) )
     call grad_for_1d( z_w, qc_c(j,:), fqcz_u(j,:) )
     call grad_for_1d( z_w, ql_c(j,:), fqlz_u(j,:) )
     call grad_for_1d( z_w, pte_c(j,:), dptedz_u(j,:) )
     call grad_for_1d( z_w, pte_w(j,:), dptedz_s(j,:) )
     call grad_back_1d( z_s, pte_s(j,:), dptedz_w(j,:) )
     call grad_for_1d( z_w, ptvb_w(j,:), dptvbdz_u(j,:) )
     call grad_for_1d( z_w, ptvb_w(j,:), dptvbdz_s(j,:) )
     call grad_back_1d( z_s, ptvb_s(j,:), dptvbdz_w(j,:) )
     call grad_for_1d( z_w, ptv_c(j,:), dptvdz_u(j,:) )
     call grad_for_1d( z_w, ptv_w(j,:), dptvdz_s(j,:) )
     call grad_back_1d( z_s, ptv_s(j,:), dptvdz_w(j,:) )
  end do
!$omp end do
!$omp end parallel

  !-- calculating nu
  !-- 1st. calculating S
!$omp parallel default(shared)
!$omp do schedule(dynamic) private(j,k)
  do k=1,nz+1
     do j=1,nr+1
        if(r_u(j)/=0.0)then
           S_u(j,k)=2.0*(dudr_u(j,k)**2+(u_u(j,k)/r_u(j))**2+dwdz_u(j,k)**2)  &
  &                 +(dudz_u(j,k)+dwdr_u(j,k))**2  &
  &                 +(dvdr_u(j,k)-v_u(j,k)/r_u(j))**2  &
  &                 +dvdz_u(j,k)**2
        else
           S_u(j,k)=0.0
        end if
if(S_u(j,k)>1.0e5)then
!write(*,*) "SU check", dudr_u(j,k), u_u(j,k), dwdz_u(j,k), dudz_u(j,k), dwdr_u(j,k), dvdr_u(j,k), v_u(j,k), dvdz_u(j,k), S_u(j,k)
end if
        if(r_s(j)/=0.0)then
           S_s(j,k)=2.0*(dudr_s(j,k)**2+(u_s(j,k)/r_s(j))**2+dwdz_s(j,k)**2)  &
  &                 +(dudz_s(j,k)+dwdr_s(j,k))**2  &
  &                 +(dvdr_s(j,k)-v_s(j,k)/r_s(j))**2  &
  &                 +dvdz_s(j,k)**2
           S_w(j,k)=2.0*(dudr_w(j,k)**2+(u_w(j,k)/r_s(j))**2+dwdz_w(j,k)**2)  &
  &                 +(dudz_w(j,k)+dwdr_w(j,k))**2  &
  &                 +(dvdr_w(j,k)-v_w(j,k)/r_s(j))**2  &
  &                 +dvdz_w(j,k)**2
        else
           S_s(j,k)=0.0
           S_w(j,k)=0.0
        end if
     end do
  end do
!$omp end do
!$omp end parallel

  !-- 2nd. calculating Ri
!$omp parallel default(shared)
!$omp do schedule(dynamic) private(j,k,tmpa,tmpb)
  do k=1,nz+1
     do j=1,nr+1
        if(qc_u(j,k)+ql_u(j,k)>0.0)then
           tmpa=1.0+LH(temp_u(j,k))*qv_u(j,k)/(Rd*temp_u(j,k))
           tmpb=1.0+0.622*(LH(temp_u(j,k))**2)*qv_u(j,k)/(Cpd*Rd*(temp_u(j,k)**2))
           Ri_u(j,k)=g*((tmpa/(tmpb*ptvb_u(j,k)))*dptedz_u(j,k)  &
  &                     -(fqvz_u(j,k)+fqcz_u(j,k)+fqlz_u(j,k)))

           tmpa=1.0+LH(temp_s(j,k))*qv_s(j,k)/(Rd*temp_s(j,k))
           tmpb=1.0+0.622*(LH(temp_s(j,k))**2)*qv_s(j,k)/(Cpd*Rd*(temp_s(j,k)**2))
           Ri_s(j,k)=g*((tmpa/(tmpb*ptvb_s(j,k)))*dptedz_s(j,k)  &
  &                     -(fqvz_s(j,k)+fqcz_s(j,k)+fqlz_s(j,k)))

           tmpa=1.0+LH(temp_w(j,k))*qv_w(j,k)/(Rd*temp_w(j,k))
           tmpb=1.0+0.622*(LH(temp_w(j,k))**2)*qv_w(j,k)/(Cpd*Rd*(temp_w(j,k)**2))
           Ri_w(j,k)=g*((tmpa/(tmpb*ptvb_w(j,k)))*dptedz_w(j,k)  &
  &                     -(fqvz_w(j,k)+fqcz_w(j,k)+fqlz_w(j,k)))
        else
           Ri_u(j,k)=g*dptvdz_u(j,k)/ptvb_u(j,k)
           Ri_s(j,k)=g*dptvdz_s(j,k)/ptvb_s(j,k)
           Ri_w(j,k)=g*dptvdz_w(j,k)/ptvb_w(j,k)
        end if
     end do
  end do
!$omp end do
!$omp end parallel

  !-- 3rd. calculating nu
!$omp parallel default(shared)
!$omp do schedule(dynamic) private(j,k)
  do k=1,nz+1
     do j=1,nr+1
        if(Ri_u(j,k)>S_u(j,k)*S_u(j,k))then
           nu_u(j,k)=0.0
        else
           nu_u(j,k)=lo*lo*sqrt(S_u(j,k)*S_u(j,k)-Ri_u(j,k))
        end if
        if(Ri_s(j,k)>S_s(j,k)*S_s(j,k))then
           nu_s(j,k)=0.0
        else
           nu_s(j,k)=lo*lo*sqrt(S_s(j,k)*S_s(j,k)-Ri_s(j,k))
        end if
        if(Ri_w(j,k)>S_w(j,k)*S_w(j,k))then
           nu_w(j,k)=0.0
        else
           nu_w(j,k)=lo*lo*sqrt(S_w(j,k)*S_w(j,k)-Ri_w(j,k))
        end if
if(k==2)then
!write(*,*) "nu check", nu_w(j,k), lo, sqrt(S_w(j,k)*S_w(j,k)-Ri_w(j,k)), S_w(j,k), Ri_w(j,k), ptv_s(j,1:2)
end if
     end do
  end do
!$omp end do
!$omp end parallel

  !-- calculating tau
  !-- DIFF term 計算に合わせて先に r を必要な箇所にあらかじめかけておく.
!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
  do k=1,nz+1
     do j=1,nr+1
        taurr_s(j,k)=2.0*nu_s(j,k)*dudr_s(j,k)
        taurr_u(j,k)=2.0*nu_u(j,k)*dudr_u(j,k)
        taurt_u(j,k)=nu_u(j,k)*r_u(j)*dvrdr_u(j,k)
        taurt_s(j,k)=nu_s(j,k)*r_s(j)*dvrdr_s(j,k)
        taurz_u(j,k)=nu_u(j,k)*(dudz_u(j,k)+dwdr_u(j,k))
        taurz_w(j,k)=nu_w(j,k)*(dudz_w(j,k)+dwdr_w(j,k))
        tauzt_w(j,k)=nu_w(j,k)*dvdz_w(j,k)
        tauzz_s(j,k)=2.0*nu_s(j,k)*dwdz_s(j,k)
        if(r_u(j)/=0.0)then
           tautt_u(j,k)=2.0*nu_u(j,k)*u_u(j,k)/r_u(j)
        else
           tautt_u(j,k)=0.0
        end if
        ftr_u(j,k)=-r_u(j)*nu_u(j,k)*ftr_u(j,k)
        fqvr_u(j,k)=-r_u(j)*nu_u(j,k)*fqvr_u(j,k)
        fqcr_u(j,k)=-r_u(j)*nu_u(j,k)*fqcr_u(j,k)
        fqlr_u(j,k)=-r_u(j)*nu_u(j,k)*fqlr_u(j,k)
        ftz_w(j,k)=-nu_w(j,k)*ftz_w(j,k)
        fqvz_w(j,k)=-nu_w(j,k)*fqvz_w(j,k)
        fqcz_w(j,k)=-nu_w(j,k)*fqcz_w(j,k)
        fqlz_w(j,k)=-nu_w(j,k)*fqlz_w(j,k)
     end do
  end do
!$omp end do
!$omp end parallel

  !-- domain bottom process
!$omp parallel default(shared)
!$omp do schedule(runtime) private(j)
  do j=1,nr+1
     CD_s(j)=CD0+4.0e-5*sqrt(u_s(j,1)**2+v_s(j,1)**2)
     CE_s(j)=CD_s(j)
     taurz_w(j,1)=CD_s(j)*u_s(j,1)*sqrt(u_s(j,1)**2+v_u(j,1)**2)
     tauzt_w(j,1)=CD_s(j)*v_s(j,1)*sqrt(u_s(j,1)**2+v_s(j,1)**2)
     ftz_w(j,1)=CE_s(j)*sqrt(u_s(j,1)**2+v_s(j,1)**2)*(pts(j)-t_s(j,1))
     fqvz_w(j,1)=CE_s(j)*sqrt(u_s(j,1)**2+v_s(j,1)**2)*(qvs(j)-qv_s(j,1))
     fqcz_w(j,1)=0.0
     fqlz_w(j,1)=0.0
  end do
!$omp end do
!$omp end parallel

  !-- domain top process
  do j=1,nr+1
     ! physical
     taurz_w(j,nz+1)=0.0
     tauzt_w(j,nz+1)=0.0
     ftz_w(j,nz+1)=0.0
     fqvz_w(j,nz+1)=0.0
     fqcz_w(j,nz+1)=0.0
     fqlz_w(j,nz+1)=0.0
     ! computational
     taurr_s(j,nz+1)=0.0
     taurr_u(j,nz+1)=0.0
     taurt_u(j,nz+1)=0.0
     taurt_s(j,nz+1)=0.0
     taurz_u(j,nz+1)=0.0
     tauzz_s(j,nz+1)=0.0
     ftr_u(j,nz+1)=0.0
     fqvr_u(j,nz+1)=0.0
  end do

  !-- domain inner latetal process
  do k=1,nz+1
     ! physical
     taurz_u(1,k)=0.0
     taurt_u(1,k)=0.0
     taurt_s(1,k)=0.0
     ftr_u(1,k)=0.0
     fqvr_u(1,k)=0.0
  end do

  !-- domain outer latetal process
  do k=1,nz+1
     ! computational
     taurr_s(nr+1,k)=0.0
     taurr_u(nr+1,k)=0.0
     taurt_u(nr+1,k)=0.0
     taurt_s(nr+1,k)=0.0
     taurz_u(nr+1,k)=0.0
     taurz_w(nr+1,k)=0.0
     tauzt_w(nr+1,k)=0.0
     tautt_u(nr+1,k)=0.0
     tauzz_s(nr+1,k)=0.0
     ftr_u(nr+1,k)=0.0
     fqvr_u(nr+1,k)=0.0
     fqcr_u(nr+1,k)=0.0
     fqlr_u(nr+1,k)=0.0
     ftz_w(nr+1,k)=0.0
     fqvz_w(nr+1,k)=0.0
     fqcz_w(nr+1,k)=0.0
     fqlz_w(nr+1,k)=0.0
  end do

  !-- gradient tau, F
!$omp parallel default(shared)
!$omp do schedule(runtime) private(k)
  do k=1,nz+1
     call grad_back_1d( r_s, taurr_s(:,k), dtrrdr_u(:,k) )
     call grad_for_1d( r_u, taurt_u(:,k), dtrtdr_s(:,k) )
     call grad_for_1d( r_u, taurz_u(:,k), dtrzdr_w(:,k) )
     call grad_for_1d( r_u, ftr_u(:,k), dftrdr_s(:,k) )
     call grad_for_1d( r_u, fqvr_u(:,k), dfqvrdr_s(:,k) )
     call grad_for_1d( r_u, fqcr_u(:,k), dfqcrdr_s(:,k) )
     call grad_for_1d( r_u, fqlr_u(:,k), dfqlrdr_s(:,k) )
  end do
!$omp end do

!$omp do schedule(runtime) private(j)
  do j=1,nr+1
     call grad_for_1d( z_w, taurz_w(j,:), dtrzdz_u(j,:) )
     call grad_for_1d( z_w, tauzt_w(j,:), dtztdz_s(j,:) )
     call grad_back_1d( z_s, tauzz_s(j,:), dtzzdz_w(j,:) )
     call grad_for_1d( z_w, ftz_w(j,:), dftzdz_s(j,:) )
     call grad_for_1d( z_w, fqvz_w(j,:), dfqvzdz_s(j,:) )
     call grad_for_1d( z_w, fqcz_w(j,:), dfqczdz_s(j,:) )
     call grad_for_1d( z_w, fqlz_w(j,:), dfqlzdz_s(j,:) )
  end do
!$omp end do
!$omp end parallel

  !-- calculating DIFF term

  if(mode(1:1)=='b')then

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(j,k)
     do k=1,nz+1
        do j=1,nr+1
           if(r_s(j)/=0.0)then
              DIFF_v(j,k)=dtrtdr_s(j,k)+2.0*taurt_s(j,k)/r_s(j)+dtztdz_s(j,k)
              DIFF_t(j,k)=-dftrdr_s(j,k)/r_s(j)-dftzdz_s(j,k)
              DIFF_qv(j,k)=-dfqvrdr_s(j,k)/r_s(j)-dfqvzdz_s(j,k)
              DIFF_qc(j,k)=-dfqcrdr_s(j,k)/r_s(j)-dfqczdz_s(j,k)
              DIFF_ql(j,k)=-dfqlrdr_s(j,k)/r_s(j)-dfqlzdz_s(j,k)
           else
              DIFF_v(j,k)=0.0
              DIFF_t(j,k)=0.0
              DIFF_qv(j,k)=0.0
              DIFF_qc(j,k)=0.0
              DIFF_ql(j,k)=0.0
           end if
if(k==1)then
!write(*,*) "DIFF t checeh", DIFF_qv(j,k), -dfqvrdr_s(j,k)/r_s(j), -dfqvzdz_s(j,k), fqvz_w(j,1:2), fqvr_u(j,k)
!write(*,*) "DIFF uw", DIFF_w(j,k), DIFF_u(j,k), j, k, dtrrdr_u(j,k), -tautt_u(j,k), dtrzdz_u(j,k), taurz_w(j,k)
end if
        end do
     end do
!$omp end do
!$omp end parallel

  else if(mode(1:1)=='s')then

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(j,k)
     do k=1,nz+1
        do j=1,nr+1
           if(r_u(j)/=0.0)then
              DIFF_u(j,k)=(taurr_u(j,k)-tautt_u(j,k))/r_u(j)  &
  &                       +dtrrdr_u(j,k)+dtrzdz_u(j,k)
           else
              DIFF_u(j,k)=0.0
           end if
           if(r_s(j)/=0.0)then
              DIFF_w(j,k)=dtrzdr_w(j,k)+taurz_w(j,k)/r_s(j)+dtzzdz_w(j,k)
           else
              DIFF_w(j,k)=0.0
           end if
        end do
     end do
!$omp end do
!$omp end parallel

  end if

end subroutine DIFF_calc


subroutine SPG_calc( mode )
!-- スポンジ層を計算する.
  use val_define
  implicit none
  character(1), intent(in) :: mode
  integer :: j, k
  real :: coe

  if(mode(1:1)=='b')then

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(j,k,coe)
     do k=1,nz+1
        do j=1,nr+1
           if( z_s(k)>zsp )then
              coe=(z_s(k)-zsp)/(z_w(nz+1)-zsp)
              SPG_v(j,k)=-SPG_coe*coe*(v_s(j,k)-vb_s(j,k))
              SPG_t(j,k)=-SPG_coe*coe*(t_s(j,k)-ptb_s(j,k))
              SPG_qv(j,k)=-SPG_coe*coe*(qv_s(j,k)-qvb_s(j,k))
              SPG_qc(j,k)=-SPG_coe*coe*qc_s(j,k)
              SPG_ql(j,k)=-SPG_coe*coe*ql_s(j,k)
if(abs(SPG_t(j,k))>1.0)then
write(*,*) "SPG check", SPG_t(j,k), j, k, SPG_coe, z_s(k), zsp, t_s(j,k), ptb_s(j,k)
end if
           else
              SPG_v(j,k)=0.0
              SPG_t(j,k)=0.0
              SPG_qv(j,k)=0.0
              SPG_qc(j,k)=0.0
              SPG_ql(j,k)=0.0
           end if
        end do
     end do
!$omp end do
!$omp end parallel

  else if(mode(1:1)=='s')then

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(j,k,coe)
     do k=1,nz+1
        do j=1,nr+1
           if( z_s(k)>zsp )then
              coe=(z_s(k)-zsp)/(z_w(nz+1)-zsp)
              SPG_u(j,k)=-SPG_coe*coe*(u_u(j,k)-ub_u(j,k))
           else
              SPG_u(j,k)=0.0
           end if
           if( z_w(k)>zsp )then
              coe=(z_w(k)-zsp)/(z_w(nz+1)-zsp)
              SPG_w(j,k)=-SPG_coe*coe*(w_w(j,k)-wb_w(j,k))
           else
              SPG_w(j,k)=0.0
           end if
        end do
     end do
!$omp end do
!$omp end parallel

  end if

end subroutine SPG_calc


subroutine NUM_DIFF_calc( mode )
!-- スポンジ層を計算する.
  use val_define
  implicit none
  character(1), intent(in) :: mode
  integer :: j, k

  if(mode(1:1)=='b')then

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
     do k=2,nz
        do j=2,nr
           NUMD_v(j,k)=(nunh*(v_u(j+1,k)+v_u(j-1,k)-2.0*v_s(j,k))  &
  &                    +nunv*(v_w(j,k+1)+v_w(j,k-1)-2.0*v_s(j,k)))/dtb
           NUMD_t(j,k)=(nunh*(t_u(j+1,k)+t_u(j-1,k)-2.0*t_s(j,k))  &
  &                    +nunv*(t_w(j,k+1)+t_w(j,k-1)-2.0*t_s(j,k)))/dtb
           NUMD_qv(j,k)=(nunh*(qv_u(j+1,k)+qv_u(j-1,k)-2.0*qv_s(j,k))  &
  &                     +nunv*(qv_w(j,k+1)+qv_w(j,k-1)-2.0*qv_s(j,k)))/dtb
           NUMD_qc(j,k)=(nunh*(qc_u(j+1,k)+qc_u(j-1,k)-2.0*qc_s(j,k))  &
  &                     +nunv*(qc_w(j,k+1)+qc_w(j,k-1)-2.0*qc_s(j,k)))/dtb
           NUMD_ql(j,k)=(nunh*(ql_u(j+1,k)+ql_u(j-1,k)-2.0*ql_s(j,k))  &
  &                     +nunv*(ql_w(j,k+1)+ql_w(j,k-1)-2.0*ql_s(j,k)))/dtb
if(abs(NUMD_t(j,k))>1.0)then
write(*,*) "NUMD check", NUMD_t(j,k), j, k
end if
        end do
     end do
!$omp end do
!$omp end parallel

  else if(mode(1:1)=='s')then

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
     do k=2,nz
        do j=2,nr
           NUMD_u(j,k)=(nunh*(u_s(j+1,k)+u_s(j-1,k)-2.0*u_u(j,k))  &
  &                    +nunv*(u_w(j,k+1)+u_w(j,k-1)-2.0*u_u(j,k)))/dtb
           NUMD_w(j,k)=(nunh*(w_u(j+1,k)+w_u(j-1,k)-2.0*w_w(j,k))  &
  &                    +nunv*(w_s(j,k+1)+w_s(j,k-1)-2.0*w_w(j,k)))/dtb
        end do
     end do
!$omp end do
!$omp end parallel

  end if

end subroutine NUM_DIFF_calc


subroutine CLOUD_calc()
  use val_define
  implicit none
  integer :: j, k
  real, dimension(nr+1,nz+1) :: termv, qtfall, pres, FALL_qt

!-- setting zero

  call set_zero( qv_s )
  call set_zero( qc_s )
  call set_zero( ql_s )

!$omp parallel default(shared)
!$omp do schedule(runtime) private(j,k)
  do k=1,nz+1
     do j=1,nr+1
        qtfall(j,k)=rhob_s(j,k)*ql_s(j,k)  &
  &                *term_v_warm( rhob_s(j,k), ql_s(j,k), rho0 )
     end do
  end do
!$omp end do

!$omp do schedule(runtime) private(j)
  do j=1,nr+1
     call grad_1d( z_s, qtfall(j,:), FALL_qt(j,:) )
  end do
!$omp end do

!$omp do schedule(runtime) private(j,k)
  do k=1,nz+1
     do j=1,nr+1
         pres(j,k)=p0*(p_s(j,k)+pb_s(j,k))**(Cpd/Rd)
         CLOUD_qv(j,k)=EVrv( temp_s(j,k), qv_s(j,k), ql_s(j,k), pres(j,k) )
         CLOUD_qc(j,k)=-CLcr( qc_s(j,k), ql_s(j,k) )-CNcr( qc_s(j,k) )
         CLOUD_ql(j,k)=-CLOUD_qv(j,k)+CLOUD_qc(j,k)+FALL_qt(j,k)/rhob_s(j,k)
         CLOUD_pt(j,k)=-rhob_s(j,k)*Lv( temp_s(j,k) )*CLOUD_qv(j,k)  &
  &                    /(Cpd*pb_s(j,k))
!        if(qt_s(j,k)>=qt_thres)then
!           termv(j,k)=fallv
!        else
!           termv(j,k)=0.0
!        end if
!        qtfall(j,k)=rhob_s(j,k)*qt_s(j,k)*termv(j,k)
     end do
  end do
!$omp end do
!$omp end parallel

end subroutine CLOUD_calc


subroutine auto_staggered_interp()
!-- staggered 格子の各点への内挿を行う.
  use val_define
  implicit none
  integer :: j, k

  !-- u_u -> u_s, u_w, u_c
  !-- w_w -> w_s, w_u, w_c
  !-- s_s -> s_u, s_w, s_c
!$omp parallel default(shared)
!$omp do schedule(dynamic) private(k)
  do k=1,nz+1
     call auto_interpolation_1d( r_u, r_s, u_u(:,k), u_s(:,k),  &
  &                              undef=undef, undefr=undef, stdopt=.true. )
     call auto_interpolation_1d( r_s, r_u, v_s(:,k), v_u(:,k),  &
  &                              undef=undef, undefr=undef, stdopt=.true. )
     call auto_interpolation_1d( r_s, r_u, w_w(:,k), w_u(:,k),  &
  &                              undef=undef, undefr=undef, stdopt=.true. )
     call auto_interpolation_1d( r_s, r_u, p_s(:,k), p_u(:,k),  &
  &                              undef=undef, undefr=undef, stdopt=.true. )
     call auto_interpolation_1d( r_s, r_u, t_s(:,k), t_u(:,k),  &
  &                              undef=undef, undefr=undef, stdopt=.true. )
     call auto_interpolation_1d( r_s, r_u, temp_s(:,k), temp_u(:,k),  &
  &                              undef=undef, undefr=undef, stdopt=.true. )
     call auto_interpolation_1d( r_s, r_u, ptv_s(:,k), ptv_u(:,k),  &
  &                              undef=undef, undefr=undef, stdopt=.true. )
     call auto_interpolation_1d( r_s, r_u, qv_s(:,k), qv_u(:,k),  &
  &                              undef=undef, undefr=undef, stdopt=.true. )
     call auto_interpolation_1d( r_s, r_u, qc_s(:,k), qc_u(:,k),  &
  &                              undef=undef, undefr=undef, stdopt=.true. )
     call auto_interpolation_1d( r_s, r_u, ql_s(:,k), ql_u(:,k),  &
  &                              undef=undef, undefr=undef, stdopt=.true. )
     v_u(1,k)=v_s(1,k)
     w_u(1,k)=w_w(1,k)
     p_u(1,k)=p_s(1,k)
     t_u(1,k)=t_s(1,k)
     temp_u(1,k)=temp_s(1,k)
     ptv_u(1,k)=ptv_s(1,k)
     qv_u(1,k)=qv_s(1,k)
     qc_u(1,k)=qc_s(1,k)
     ql_u(1,k)=ql_s(1,k)
  end do
!$omp end do

!$omp do schedule(dynamic) private(j)
  do j=1,nr+1
     call auto_interpolation_1d( z_s, z_w, u_u(j,:), u_w(j,:),  &
  &                              undef=undef, undefr=undef, stdopt=.true. )
     call auto_interpolation_1d( z_s, z_w, v_s(j,:), v_w(j,:),  &
  &                              undef=undef, undefr=undef, stdopt=.true. )
     call auto_interpolation_1d( z_w, z_s, w_w(j,:), w_s(j,:),  &
  &                              undef=undef, undefr=undef, stdopt=.true. )
     call auto_interpolation_1d( z_s, z_w, p_s(j,:), p_w(j,:),  &
  &                              undef=undef, undefr=undef, stdopt=.true. )
     call auto_interpolation_1d( z_s, z_w, t_s(j,:), t_w(j,:),  &
  &                              undef=undef, undefr=undef, stdopt=.true. )
     call auto_interpolation_1d( z_s, z_w, temp_s(j,:), temp_w(j,:),  &
  &                              undef=undef, undefr=undef, stdopt=.true. )
     call auto_interpolation_1d( z_s, z_w, ptv_s(j,:), ptv_w(j,:),  &
  &                              undef=undef, undefr=undef, stdopt=.true. )
     call auto_interpolation_1d( z_s, z_w, qv_s(j,:), qv_w(j,:),  &
  &                              undef=undef, undefr=undef, stdopt=.true. )
     call auto_interpolation_1d( z_s, z_w, qc_s(j,:), qc_w(j,:),  &
  &                              undef=undef, undefr=undef, stdopt=.true. )
     call auto_interpolation_1d( z_s, z_w, ql_s(j,:), ql_w(j,:),  &
  &                              undef=undef, undefr=undef, stdopt=.true. )
     u_w(j,1)=u_u(j,1)
     v_w(j,1)=v_s(j,1)
     w_s(j,nz+1)=0.0
     p_w(j,1)=p_s(j,1)
     t_w(j,1)=t_s(j,1)
     temp_w(j,1)=temp_s(j,1)
     ptv_w(j,1)=ptv_s(j,1)
     qv_w(j,1)=qv_s(j,1)
     qc_w(j,1)=qc_s(j,1)
     ql_w(j,1)=ql_s(j,1)
  end do
!$omp end do

! 半格子の中心点を計算する.
! この処理は auto_interpolation_2d でも可能であるが, 
! 並列処理されていないため, 直書きする.

!$omp do schedule(runtime) private(j,k)

  do k=1,nz
     do j=1,nr
        u_c(j,k+1)=0.25*(u_u(j,k+1)+u_u(j+1,k+1)+u_u(j+1,k)+u_u(j,k))

        w_c(j+1,k)=0.25*(w_w(j,k+1)+w_w(j+1,k+1)+w_w(j+1,k)+w_w(j,k))

        v_c(j+1,k+1)=0.25*(v_s(j,k+1)+v_s(j+1,k+1)+v_s(j+1,k)+v_s(j,k))
        p_c(j+1,k+1)=0.25*(p_s(j,k+1)+p_s(j+1,k+1)+p_s(j+1,k)+p_s(j,k))
        t_c(j+1,k+1)=0.25*(t_s(j,k+1)+t_s(j+1,k+1)+t_s(j+1,k)+t_s(j,k))
        temp_c(j+1,k+1)=0.25*(temp_s(j,k+1)+temp_s(j+1,k+1)+temp_s(j+1,k)+temp_s(j,k))
        ptv_c(j+1,k+1)=0.25*(ptv_s(j,k+1)+ptv_s(j+1,k+1)+ptv_s(j+1,k)+ptv_s(j,k))
        qv_c(j+1,k+1)=0.25*(qv_s(j,k+1)+qv_s(j+1,k+1)+qv_s(j+1,k)+qv_s(j,k))
        qc_c(j+1,k+1)=0.25*(qc_s(j,k+1)+qc_s(j+1,k+1)+qc_s(j+1,k)+qc_s(j,k))
        ql_c(j+1,k+1)=0.25*(ql_s(j,k+1)+ql_s(j+1,k+1)+ql_s(j+1,k)+ql_s(j,k))
     end do

     u_c(nr+1,k)=u_w(nr,k)

     w_c(1,k)=0.5*(w_u(1,k)+w_u(1,k+1))

     v_c(1,k+1)=0.5*(v_u(1,k)+v_u(1,k+1))
     p_c(1,k+1)=0.5*(p_u(1,k)+p_u(1,k+1))
     t_c(1,k+1)=0.5*(t_u(1,k)+t_u(1,k+1))
     temp_c(1,k+1)=0.5*(temp_u(1,k)+temp_u(1,k+1))
     ptv_c(1,k+1)=0.5*(ptv_u(1,k)+ptv_u(1,k+1))
     qv_c(1,k+1)=0.5*(qv_u(1,k)+qv_u(1,k+1))
     qc_c(1,k+1)=0.5*(qc_u(1,k)+qc_u(1,k+1))
     ql_c(1,k+1)=0.5*(ql_u(1,k)+ql_u(1,k+1))

  end do

!$omp end do

!$omp do schedule(runtime) private(j)

  do j=1,nr
     u_c(j,1)=0.5*(u_w(j,1)+u_w(j+1,1))

     w_c(j,nz+1)=w_u(j,nz)

     v_c(j+1,1)=0.5*(v_w(j,1)+v_w(j+1,1))
     p_c(j+1,1)=0.5*(p_w(j,1)+p_w(j+1,1))
     t_c(j+1,1)=0.5*(t_w(j,1)+t_w(j+1,1))
     temp_c(j+1,1)=0.5*(temp_w(j,1)+temp_w(j+1,1))
     ptv_c(j+1,1)=0.5*(ptv_w(j,1)+ptv_w(j+1,1))
     qv_c(j+1,1)=0.5*(qv_w(j,1)+qv_w(j+1,1))
     qc_c(j+1,1)=0.5*(qc_w(j,1)+qc_w(j+1,1))
     ql_c(j+1,1)=0.5*(ql_w(j,1)+ql_w(j+1,1))
  end do

!$omp end do
!$omp end parallel

  u_c(nr+1,nz+1)=u_w(nr,nz+1)

  w_c(nr+1,nz+1)=w_u(nr+1,nz)

  v_c(1,1)=0.5*(v_u(1,1)+v_w(1,1))
  p_c(1,1)=0.5*(p_u(1,1)+p_w(1,1))
  t_c(1,1)=0.5*(t_u(1,1)+t_w(1,1))
  temp_c(1,1)=0.5*(temp_u(1,1)+temp_w(1,1))
  ptv_c(1,1)=0.5*(ptv_u(1,1)+ptv_w(1,1))
  qv_c(1,1)=0.5*(qv_u(1,1)+qv_w(1,1))
  qc_c(1,1)=0.5*(qc_u(1,1)+qc_w(1,1))
  ql_c(1,1)=0.5*(ql_u(1,1)+ql_w(1,1))

end subroutine auto_staggered_interp


subroutine reset_val()
!-- 予報変数の作業用変数への更新を行う.
  use val_define
  use Thermo_Const
  implicit none
  integer :: j, k
  real, dimension(nr+1,nz+1) :: tmp_p
  real :: tmpv

!$omp parallel default(shared)
!$omp do schedule(dynamic) private(j,k)
  do k=1,nz+1
     do j=1,nr+1
        u_u(j,k)=u_old(j,k)
        v_s(j,k)=v_old(j,k)
        w_w(j,k)=w_old(j,k)
        p_s(j,k)=p_old(j,k)
        t_s(j,k)=t_old(j,k)
        qv_s(j,k)=qv_old(j,k)
        qc_s(j,k)=qc_old(j,k)
        ql_s(j,k)=ql_old(j,k)
        tmp_p(j,k)=p0*(pb_s(j,k)+p_old(j,k))**(Cpd/Rd)
        temp_s(j,k)=thetaP_2_T( t_old(j,k), tmp_p(j,k) )
        ptv_s(j,k)=TqvP_2_thetav( temp_s(j,k), qv_s(j,k), tmp_p(j,k) )
!write(*,*) "temp", temp_s(j,k), qv_s(j,k), tmp_p(j,k), pb_s(j,k), p_old(j,k)
     end do
  end do
!$omp end do
!$omp end parallel

!-- 海面における変数を更新する.
  do j=1,nr+1
     tmpv=hydro_calc( z_s(1), temp_s(j,1), tmp_p(j,1),  &
  &                   z_s(2), temp_s(j,2), tmp_p(j,2), 0.0 )
     pts(j)=theta_dry( sst_s(j), tmpv )
     qvs(j)=TP_2_qvs( sst_s(j), tmpv )
  end do

end subroutine reset_val


end module
